#!/bin/sh
#  Copyright (c) 2023 Qualcomm Technologies, Inc.
#  All Rights Reserved.
#  Confidential and Proprietary - Qualcomm Technologies, Inc.

#source log file
# Usage to use log function
# log  $(basename "$0") $LINENO "Log message"
. /etc/data/wlanConfig_common.sh

if [[ "$#" -le 0 ]]; then
    log  $(basename "$0") $LINENO ": Usage " "wlanConfig_hmt.sh set_wlan_mode mode"
    return 1
elif [ \( "$1" == "set_wlan_mode" \) -a \( "$#" -ne 2 \) ]; then
    log  $(basename "$0") $LINENO ": Usage " "wlanConfig_hmt.sh set_wlan_mode mode"
    return 1
fi

function checkFileExist() {
    #it will check if dst_file_name exists if not will make copy from src_file_name
    local src_file_name=$1
    local dst_file_name=$2
    #check if tmp config file not present
    [ -f $dst_file_name ] || {
        #make a copy
        cp $src_file_name $dst_file_name
    }
}

# Get Backhaul WAN type: get_wan_type <arg1>
# $1 : QCMAP backhaul WAN type.
function set_wlan_mode() {

  local wlan_mode=$1
  checkFileExist /etc/config/qcmap_wlan /etc/config/qcmap_wlan_current
  util_execute_uci set qcmap_wlan_current.@wlanconfig[0].mode=$wlan_mode
  util_execute_uci set qcmap_wlan_current.@wlanconfig[0].wlanconfigex='0'
  util_execute_uci commit qcmap_wlan_current

  add_wanwlan_iface_on_wan_all_zone $1
  log  $(basename "$0") "set_wlan_mode" $LINENO "set_wlan_mode as $wlan_mode done !!!"
}

# Get Backhaul WAN type: get_wan_type <arg1>
# $1 : QCMAP backhaul WAN type.
function set_mld_wlan_mode() {

  local mld_wlan_mode=$1
  checkFileExist /etc/config/qcmap_wlan /etc/config/qcmap_wlan_current
  util_execute_uci set qcmap_wlan_current.@mldwlanconfig[0].mode=$mld_wlan_mode
  util_execute_uci set qcmap_wlan_current.@wlanconfig[0].wlanconfigex='1'
  util_execute_uci commit qcmap_wlan_current

  add_wanwlan_iface_on_wan_all_zone $1
  log  $(basename "$0") "set_mld_wlan_mode" $LINENO "set_wlan_mode as $wlan_mode done !!!"
}

#only single MLD SAP is supported for HMT
function set_mld_ap_config() {
  local mld_ap_count=$1
  local no_of_mld_link=$2
  local accessprofile=$3

  util_execute_uci set qcmap_wlan_current.@mldapconfig[$mld_ap_count].no_of_mld_link=$no_of_mld_link
  util_execute_uci set qcmap_wlan_current.@mldapconfig[$mld_ap_count].accessprofile=$accessprofile
  util_execute_uci set qcmap_wlan_current.@mldapconfig[$mld_ap_count].ifname="mld0"
  util_execute_uci commit qcmap_wlan_current

  log  $(basename "$0") "set_ap_config" $LINENO "set_mld_ap_config done !!!"
}

function set_primary_config() {

  local band=$1

  util_execute_uci set qcmap_wlan_current.@primaryap[0].band=$band
  util_execute_uci commit qcmap_wlan_current

  log  $(basename "$0") "set_primary_config" $LINENO "set_primary_config on band $band done !!!"
}

function set_guest_config() {

local guest1_band=$1
local guest1_accessprofile=$2
local guest2_band=$3
local guest2_accessprofile=$4
local guest3_band=$5
local guest3_accessprofile=$6

#Now at most 4 AP support, so at most 3 guest AP.

  util_execute_uci set qcmap_wlan_current.@guestap[0].band=$guest1_band
  util_execute_uci set qcmap_wlan_current.@guestap[0].accessprofile=$guest1_accessprofile
  util_execute_uci set qcmap_wlan_current.@guestaptwo[0].band=$guest2_band
  util_execute_uci set qcmap_wlan_current.@guestaptwo[0].accessprofile=$guest2_accessprofile
  util_execute_uci set qcmap_wlan_current.@guestapthree[0].band=$guest3_band
  util_execute_uci set qcmap_wlan_current.@guestapthree[0].accessprofile=$guest3_accessprofile
  util_execute_uci commit qcmap_wlan_current


  log  $(basename "$0") "set_guest_config" $LINENO "set_guest_config on band $1 done !!!"
}

function set_sta_config() {
  local band=$1
  local bridge_mode=$2
  local sta_mode_conn_type=$3
  local ipaddr=$4
  local gateway=$5
  local netmask=$6
  local dnsserver=$7
  local wlan_mode


  wlan_mode=$(util_execute_uci get qcmap_wlan_current.@wlanconfig[0].mode)
  util_execute_uci set qcmap_wlan_current.@stamodeconfig[0].band=$band
  util_execute_uci set qcmap_wlan_current.@stamodeconfig[0].bridge_mode=$bridge_mode
  util_execute_uci set qcmap_wlan_current.@stamodeconfig[0].sta_mode_conn_type=$sta_mode_conn_type
  util_execute_uci set qcmap_wlan_current.@stamodeconfig[0].ipaddr=$ipaddr
  util_execute_uci set qcmap_wlan_current.@stamodeconfig[0].gateway=$gateway
  util_execute_uci set qcmap_wlan_current.@stamodeconfig[0].netmask=$netmask
  util_execute_uci set qcmap_wlan_current.@stamodeconfig[0].dnsserver=$dnsserver

  if [ $wlan_mode == $QCMAP_WLAN_MODE_AP_STA ] || [ $wlan_mode == $QCMAP_WLAN_MODE_AP_AP_STA ] || [ $wlan_mode == $QCMAP_WLAN_MODE_STA_ONLY ]; then
  #delete exiting static type uci entry before reconfiguration
    util_execute_uci delete network.wanwlan.ipaddr 2> /dev/null
    util_execute_uci delete network.wanwlan.netmask 2> /dev/null
    util_execute_uci delete network.wanwlan.dns 2> /dev/null

    if [ $sta_mode_conn_type == $QCMAP_STA_CONNECTION_DYNAMIC ] ; then
      log  $(basename "$0") "set_sta_config" $LINENO "STA dynamic mode"
      if [ $(util_execute_uci get network.wanwlan.proto) != "none" ] ; then
        util_execute_uci_set "network.wanwlan.proto" "dhcp"
      fi

    elif [ $sta_mode_conn_type == $QCMAP_STA_CONNECTION_STATIC ] ; then
      log  $(basename "$0") "set_sta_config" $LINENO "Add STA static mode in network"
      util_execute_uci set network.wanwlan="interface"
      util_execute_uci set network.wanwlan.device="wlan0"
      util_execute_uci set network.wanwlan.proto="static"
      util_execute_uci set network.wanwlan.ipaddr=$ipaddr
      util_execute_uci set network.wanwlan.netmask="255.255.255.0"
      util_execute_uci set network.wanwlan.dns=$dnsserver
    fi
  fi

  util_execute_uci commit

  log  $(basename "$0") "set_sta_config" $LINENO "set_sta_config done !!!"
}

function get_wlan_iface_name() {
  log  $(basename "$0") "get_Iface_name" $LINENO " begin !!!"
  local wlan_mode=$(util_execute_uci get qcmap_wlan.@wlanconfig[0].mode)
  local device_type=$1
  local lan_Iface=""

  if [ $device_type == $QCMAP_MSGR_DEVICE_TYPE_PRIMARY_AP ] ; then

    if [ $wlan_mode == $QCMAP_WLAN_MODE_AP_STA ] || [ $wlan_mode == $QCMAP_WLAN_MODE_AP_AP_STA ] ; then
      lan_Iface=$(util_execute_uci get qcmap_wlan.@primaryap[0].staifname)
    fi

    if  [ $wlan_mode == $QCMAP_WLAN_MODE_AP ] || [ $wlan_mode == $QCMAP_WLAN_MODE_AP_AP ] || [ $wlan_mode == $QCMAP_WLAN_MODE_AP_AP_AP ] || [ $wlan_mode == $QCMAP_WLAN_MODE_AP_AP_AP_AP ] ; then
      lan_Iface=$(util_execute_uci get qcmap_wlan.@primaryap[0].apifname)
    fi

  elif [ $device_type == $QCMAP_MSGR_DEVICE_TYPE_GUEST_AP ] ; then

    if [ $wlan_mode == $QCMAP_WLAN_MODE_AP_AP_STA ] ; then
      lan_Iface=$(util_execute_uci get qcmap_wlan.@guestap[0].staifname)
    fi

    if [ $wlan_mode == $QCMAP_WLAN_MODE_AP_AP ] || [ $wlan_mode == $QCMAP_WLAN_MODE_AP_AP_AP ] || [ $wlan_mode == $QCMAP_WLAN_MODE_AP_AP_AP_AP ] ; then
      lan_Iface=$(util_execute_uci get qcmap_wlan.@guestap[0].apifname)
    fi

  elif [ $device_type == $QCMAP_MSGR_DEVICE_TYPE_GUEST_AP_2 ] ; then

    if [ $wlan_mode == $QCMAP_WLAN_MODE_AP_AP_AP ] || [ $wlan_mode == $QCMAP_WLAN_MODE_AP_AP_AP_AP ] ; then
      lan_Iface=$(util_execute_uci get qcmap_wlan.@guestaptwo[0].ifname)
    fi

  elif [ $device_type == $QCMAP_MSGR_DEVICE_TYPE_GUEST_AP_3 ] ; then

    if [ $wlan_mode == $QCMAP_WLAN_MODE_AP_AP_AP_AP ] ; then
      lan_Iface=$(util_execute_uci get qcmap_wlan.@guestapthree[0].ifname)
    fi

  fi
  log $(basename "$0") "get_Iface_name :" $LINENO ": lan iface gets successfully"
  echo $lan_Iface

}

function utility_network_wlan_interface() {
  local ap_number=$1
  local ipaddr=$2
  local ifname_list
  log  $(basename "$0") "utility_network_wlan_interface" $LINENO " utility_network_wlan_interface for wlan$ap_number $ipaddr"

  ifname_list=$(util_execute_uci get network.lan.ifname)
  ifname_list=$(echo "$ifname_list" | sed "s/^ *//g" | sed "s/ *$//g" | tr -s ' ')    #Remove first/last and trailing spaces if any
  ifname_list="${ifname_list} wlan$ap_number"
  util_execute_uci_set "network.lan.ifname" "$ifname_list"
  util_execute_uci set network.wlan"$ap_number"="interface"
  util_execute_uci set network.wlan"$ap_number".device="wlan$ap_number"
  util_execute_uci set network.wlan"$ap_number".proto="static"
  util_execute_uci set network.wlan"$ap_number".ipv6="1"
  util_execute_uci set network.wlan"$ap_number".ipaddr=$ipaddr
  util_execute_uci set network.wlan"$ap_number".netmask="255.255.255.0"
}
#update 2g/5g/6g_wireless file
#usage: utility_update_single_wireless <band-no> <ap-no-in-this-band>
#it depends on global variable station_band
function utility_update_single_wireless() {
  local band_no=$1
  local ap_no=$2
  local tmp_delete_ap_number=0

  log  $(basename "$0") "utility_update_single_wireless" $LINENO " Now the next AP number is $tmp_next_ap_number"
  log  $(basename "$0") "utility_update_single_wireless" $LINENO " handle $band_no no:$ap_no"

  cp /etc/misc/wifi/"$band_no"g_wireless /etc/config/
  sed -i "s/wifiX/wifi$pdev_number/g" /etc/config/"$band_no"g_wireless


  #set station firstly.
  if [ $station_band == $band_no ] ; then
    log  $(basename "$0") "utility_update_single_wireless" $LINENO "add station wlan$tmp_next_ap_number in network"

    util_execute_uci set "$band_no"g_wireless.wifi"$pdev_number"_STA.ifname="wlan$tmp_next_ap_number"
    util_execute_uci set network.wanwlan.device="wlan$tmp_next_ap_number"

    local sta_mode_bridge=$(util_execute_uci get qcmap_wlan_current.@stamodeconfig[0].bridge_mode)
    if [ $sta_mode_bridge == 1 ]; then
      util_execute_uci set "$band_no"g_wireless.wifi"$pdev_number"_STA.network="lan"
      ifname_list=$(util_execute_uci get network.lan.ifname)
      ifname_list=$(echo "$ifname_list" | sed "s/^ *//g" | sed "s/ *$//g" | tr -s ' ')
      ifname_list="${ifname_list} wlan$tmp_next_ap_number"
      util_execute_uci_set "network.lan.ifname" "$ifname_list"
      #delete wanwlan_v6 setting since we let br-lan apply address
      util_execute_uci delete network.wanwlan_v6 2> /dev/null
      #set wanwlan without dhcp and no IPv6, to ensure it don't apply IPv6 link local address
      util_execute_uci_set network.wanwlan.proto   "static"
      util_execute_uci_set network.wanwlan.ipv6    "0"
      util_execute_uci_set network.wanwlan.ipaddr  "169.254.25.1"
      util_execute_uci_set network.wanwlan.netmask "255.255.255.0"
    else
      #add wanwlan_v6 setting with dhcpv6
      util_execute_uci_set network.wanwlan_v6.device     "wlan$tmp_next_ap_number"
      util_execute_uci_set network.wanwlan_v6            "interface"
      util_execute_uci_set network.wanwlan_v6.proto      "dhcpv6"
      util_execute_uci_set network.wanwlan_v6.ipv6       "1"
      util_execute_uci_set network.wanwlan_v6.type       "internet"
      util_execute_uci_set network.wanwlan_v6.device     "wlan0"
      util_execute_uci_set network.wanwlan_v6.downstream "lan"

      ifname_list=$(util_execute_uci get network.lan.ifname)
      ifname_list=$(echo "$ifname_list" | sed s/wlan$i//)
      util_execute_uci_set "network.lan.ifname" "$ifname_list"
    fi

    util_execute_uci_set network.wanwlan_v6.disabled   "0"
    util_execute_uci commit network

    tmp_next_ap_number=$((tmp_next_ap_number + 1))
  else
    util_execute_uci delete "$band_no"g_wireless.wifi"$pdev_number"_STA
  fi


  #enable the AP and set ifname for HMT
  for i in `seq 1 $ap_no`
    do
      util_execute_uci set "$band_no"g_wireless.wifi"$pdev_number"_AP$i.ifname="wlan""$tmp_next_ap_number"
      utility_network_wlan_interface $tmp_next_ap_number $(eval "echo \${static_ip_wlan$tmp_next_ap_number}")

      tmp_next_ap_number=$((tmp_next_ap_number + 1))
      log  $(basename "$0") "utility_update_single_wireless" $LINENO " update the next AP number is $tmp_next_ap_number"
    done

  tmp_delete_ap_number=$((ap_no + 1))

  #delete the AP not used, to let wireless file more clear
  log $(basename "$0") "utility_update_single_wireless" $LINENO "delete un-used AP $tmp_delete_ap_number"
  for i in `seq $tmp_delete_ap_number 2`
    do
      util_execute_uci delete "$band_no"g_wireless.wifi"$pdev_number"_AP$i
    done

  #commit changes
  log  $(basename "$0") "utility_update_single_wireless" $LINENO "uci commit $band_no""gwireless"
  util_execute_uci commit "$band_no"g_wireless

  cat /etc/config/"$band_no"g_wireless >> /etc/config/tmp_wireless
  rm /etc/config/"$band_no"g_wireless

  pdev_number=$((pdev_number + 1))
  log  $(basename "$0") "utility_update_single_wireless" $LINENO " update next pdev number as $pdev_number"

}

function utility_update_wireless_for_different_band() {
  log  $(basename "$0") "utility_update_wireless_for_different_band" $LINENO " begin !!!"
  local wlan_mode=$(util_execute_uci get qcmap_wlan_current.@wlanconfig[0].mode)
  local primary_band=$(util_execute_uci get qcmap_wlan_current.@primaryap[0].band)
  local guest_band1=$(util_execute_uci get qcmap_wlan_current.@guestap[0].band)
  local guest_band2=$(util_execute_uci get qcmap_wlan_current.@guestaptwo[0].band)
  local guest_band3=$(util_execute_uci get qcmap_wlan_current.@guestapthree[0].band)
  local station_band=$(util_execute_uci get qcmap_wlan_current.@stamodeconfig[0].band)

  if [ $wlan_mode == $QCMAP_WLAN_MODE_AP_AP_STA ]; then
    if [ $primary_band != $station_band ] && [ $guest_band1 == $station_band ]; then
      util_execute_uci set wireless.wifi0_AP1.ifname="wlan2"
      util_execute_uci set wireless.wifi1_AP1.ifname="wlan1"
    fi
  elif [ $wlan_mode == $QCMAP_WLAN_MODE_AP_AP_AP ]; then
    if [ $primary_band != $guest_band1 ] && [ $primary_band == $guest_band2 ]; then
      util_execute_uci set wireless.wifi0_AP2.ifname="wlan2"
      util_execute_uci set wireless.wifi1_AP1.ifname="wlan1"
    fi
  elif [ $wlan_mode == $QCMAP_WLAN_MODE_AP_AP_AP_AP ]; then
    if [ $primary_band != $guest_band1 ] && [ $primary_band != $guest_band2 ] && [ $primary_band == $guest_band3 ]; then
      util_execute_uci set wireless.wifi0_AP2.ifname="wlan3"
      util_execute_uci set wireless.wifi1_AP1.ifname="wlan1"
      util_execute_uci set wireless.wifi1_AP2.ifname="wlan2"
    elif [ $primary_band != $guest_band1 ] && [ $primary_band != $guest_band3 ] && [ $primary_band == $guest_band2 ]; then
      util_execute_uci set wireless.wifi0_AP2.ifname="wlan2"
      util_execute_uci set wireless.wifi1_AP1.ifname="wlan1"
      util_execute_uci set wireless.wifi1_AP2.ifname="wlan3"
    fi
  fi

  util_execute_uci commit wireless
  log  $(basename "$0") "utility_update_wireless_for_different_band" $LINENO " end !!!"
}

function HMT_get_prefer_primary_band()
{
  #  local bIsPreferHighBand=$(util_execute_uci get qcmap_wlan.@prefer[0].prefer_highband)
  #  local bIsPrefer6GBand=$(util_execute_uci get qcmap_wlan.@prefer[0].prefer_6gband)

  # by default use high-band for now
  local bIsPreferHighBand=1
  # by default use 5G band instead of 6G in the high-bands
  local bIsPrefer6GBand=0

  if [ $bIsPreferHighBand != 0 ] ; then
    if [ $bIsPrefer6GBand != 0 ] ; then
      prefer_primary_band=6
    else
      prefer_primary_band=5
    fi
  prefer_second_band=2
  else
    if [ $bIsPrefer6GBand != 0 ] ; then
      prefer_second_band=6
    else
      prefer_second_band=5
    fi

    prefer_primary_band=2
  fi

  log  $(basename "$0") "HMT_get_prefer_primary_band" $LINENO " prefer_primary is band$prefer_primary_band, prefer_second is band$prefer_second_band"

}

function utility_generate_wireless_file() {
  local tmp_ap_no_2g=$1
  local tmp_ap_no_5g=$2
  local tmp_ap_no_6g=$3
  local tmp_primary_band=$4

  if [ $tmp_ap_no_2g != 0 ] && [ $tmp_primary_band != 2 ] ; then
    log  $(basename "$0") "utility_generate_wireless_file" $LINENO " handle 2g_wireless no:$tmp_ap_no_2g"
    utility_update_single_wireless 2 $tmp_ap_no_2g
  fi

  if [ $tmp_ap_no_5g != 0 ] && [ $tmp_primary_band != 5 ] ; then
    log  $(basename "$0") "utility_generate_wireless_file" $LINENO " handle 5g_wireless no:$tmp_ap_no_5g"
    utility_update_single_wireless 5 $tmp_ap_no_5g
  fi

  if [ $tmp_ap_no_6g != 0 ]  && [ $tmp_primary_band != 6 ] ; then
    log  $(basename "$0") "utility_generate_wireless_file" $LINENO " handle 6g_wireless no:$tmp_ap_no_6g"
    utility_update_single_wireless 6 $tmp_ap_no_6g
  fi
}

function get_current_total_ap_number() {
  log  $(basename "$0") "get_current_total_ap_number" $LINENO " begin !!!"
  local wlan_mode=$(util_execute_uci get qcmap_wlan_current.@wlanconfig[0].mode)

  if [ $wlan_mode != $QCMAP_WLAN_MODE_7AP ]; then
    if [ $wlan_mode == $QCMAP_WLAN_MODE_AP ] || [ $wlan_mode == $QCMAP_WLAN_MODE_AP_STA ]; then
      current_total_ap_number=1
    elif [ $wlan_mode == $QCMAP_WLAN_MODE_AP_AP ] || [ $wlan_mode == $QCMAP_WLAN_MODE_AP_AP_STA ]; then
      current_total_ap_number=2
    elif [ $wlan_mode == $QCMAP_WLAN_MODE_AP_AP_AP ] ; then
      current_total_ap_number=3
    elif [ $wlan_mode == $QCMAP_WLAN_MODE_AP_AP_AP_AP ] ; then
      current_total_ap_number=4  # only WKK support 4AP
    else
      current_total_ap_number=0
    fi
  else
    current_total_ap_number=$(util_execute_uci get qcmap_wlan_current.@wlanconfig[0].totalapcount)
  fi
  log  $(basename "$0") "get_current_total_ap_number" $LINENO " total $current_total_ap_number !!!"
}

function get_ap_no_per_band() {

  ap_no_of_2g=0
  ap_no_of_5g=0
  ap_no_of_6g=0

  #get number of AP that used in each band
  if [ $primary_band == 2 ]; then
    ap_no_of_2g=$((ap_no_of_2g + 1))
  elif [ $primary_band == 5 ]; then
    ap_no_of_5g=$((ap_no_of_5g + 1))
  elif [ $primary_band == 6 ]; then
    ap_no_of_6g=$((ap_no_of_6g + 1))
  fi

  if [ $guest_band1 == 2 ]; then
    ap_no_of_2g=$((ap_no_of_2g + 1))
  elif [ $guest_band1 == 5 ]; then
    ap_no_of_5g=$((ap_no_of_5g + 1))
  elif [ $guest_band1 == 6 ]; then
    ap_no_of_6g=$((ap_no_of_6g + 1))
  fi

  if [ $guest_band2 == 2 ]; then
    ap_no_of_2g=$((ap_no_of_2g + 1))
  elif [ $guest_band2 == 5 ]; then
    ap_no_of_5g=$((ap_no_of_5g + 1))
  elif [ $guest_band2 == 6 ]; then
    ap_no_of_6g=$((ap_no_of_6g + 1))
  fi

  if [ $guest_band3 == 2 ]; then
    ap_no_of_2g=$((ap_no_of_2g + 1))
  elif [ $guest_band3 == 5 ]; then
    ap_no_of_5g=$((ap_no_of_5g + 1))
  elif [ $guest_band3 == 6 ]; then
    ap_no_of_6g=$((ap_no_of_6g + 1))
  fi
 }

function is_wifi_up(){
   if [ -d "/sys/module/cnss2/" ]; then
     echo 1
   else
     echo 0
     log  $(basename "$0") "is_wifi_up" $LINENO "wifi not yet attached in pcie"
   fi
}

function update_wireless_network() {
  log  $(basename "$0") "update_wireless_network" $LINENO " begin !!!"
  local wlan_mode=$(util_execute_uci get qcmap_wlan_current.@wlanconfig[0].mode)
  local primary_band=$(util_execute_uci get qcmap_wlan_current.@primaryap[0].band)
  local guest_band1=$(util_execute_uci get qcmap_wlan_current.@guestap[0].band)
  local guest_band2=$(util_execute_uci get qcmap_wlan_current.@guestaptwo[0].band)
  local guest_band3=$(util_execute_uci get qcmap_wlan_current.@guestapthree[0].band)
  local station_band=$(util_execute_uci get qcmap_wlan_current.@stamodeconfig[0].band)

  #TODO: WKK have different requirement on wireless per description in update_wireless_wkk func
  #      Need make sure the common part and different part and update the function
  #      Now the scripts work for HMT only

  #current LOGIC (need update if WKK have different ):
  #  1. Get  band informaion of primary and guest1/2/3 and station.
  #     then Get the number of AP in each radio
  #  2. each radio will have different pDev number
  #     Primary AP will have pDev0, and Guest AP with pDev1/2
  #         a. At most 2 pDev will be present in HMT, ie. 2G/5G DBS or 2G/6G DBS
  #         b. At most 3 pDev will be present in WKK, ie. 2G/5G/6G. (?)
  #  3. update the template wireless files
  #     for WKK, WKK require read wlanX number from /sys/class/net
  #          But per HMT, /sys/class/net will not present until wifi up, how WKK read it before wifi up
  #     for HMT, the wlanX number we can update as below HMT logic

  #
  #  for HMT, need update the template with:
  #  1. replace wifiX with actuall pDev Number
  #  2. add wireless.wifiX_[AP,STA].ifname, i.e wlan0/1/..
  #     Now the wlan number sequece will be as following:
  #    1) PrimaryAP
  #    2) GuestAP that same band with PrimaryAP
  #    3) Station if station band is same with PrimaryAP or STA mode only
  #    4) GuestAP that different band with PrimaryAP, with order from 2G/5G/6G
  #    4.1) Station that under this GuestAP
  #  3. update disabled option to be 0
  #  4. delete those disabled sections(option, we now do it to make wireless files clear)
  #  then we :
  #  1. merged the 2g/5g/6g_wireless files into wireless files
  #  2. update network files

  #here if use bash instead of sh, we can use array then can let code more clean
  local ap_no_of_2g=0
  local ap_no_of_5g=0
  local ap_no_of_6g=0
  local tmp_next_ap_number=0
  local wlan_port_list=""
  local ifname_list

  get_current_total_ap_number
  HMT_get_prefer_primary_band
  if [ $primary_band == 0 ] && [ $current_total_ap_number -ge 1 ] ; then
    primary_band=$prefer_primary_band
    log  $(basename "$0") "update_wireless_network" $LINENO " primary_band is reset to $primary_band"
  fi
  if [ $wlan_mode == $QCMAP_WLAN_MODE_AP_STA ] ||  [ $wlan_mode == $QCMAP_WLAN_MODE_STA_ONLY ] || [ $wlan_mode == $QCMAP_WLAN_MODE_AP_AP_STA ] ; then
    if [ $station_band == 0 ] ; then
      station_band=$prefer_primary_band
      log  $(basename "$0") "update_wireless_network" $LINENO " station_band is reset to $station_band"
    fi
  fi
  log  $(basename "$0") "update_wireless_network" $LINENO " current_total_ap_number is $current_total_ap_number"
  if [ $guest_band1 == 0 ] && [ $current_total_ap_number -gt 1 ]; then
    guest_band1=$primary_band
    log  $(basename "$0") "update_wireless_network" $LINENO " gustband1 is reset to $guest_band1"
  fi
  if [ $guest_band2 == 0 ] && [ $current_total_ap_number  -gt  2 ]; then
    guest_band2=$prefer_second_band
    #check that:  if ap_no_2g -ge 3 then set guest_band2 as 5
    get_ap_no_per_band
    if [ $ap_no_of_2g -ge 3 ]; then
      guest_band2=5
    fi
    log  $(basename "$0") "update_wireless_network" $LINENO " gustband2 is reset to $guest_band2"
  fi

  #if all primary_band/ guest_band1/ guest_band2 are setted mannually then
  #check that:  if ap_no_5g -eq 3 then set guest_band2 as 2
  #check that:  if ap_no_2g -eq 3 then set guest_band2 as 5
  get_ap_no_per_band
  if [ $ap_no_of_5g -ge 3 ]; then
    guest_band2=2
    echo "three APs with same band is not allowed, modify automatically!"
  elif [ $ap_no_of_2g -ge 3 ]; then
    guest_band2=5
    echo "three APs with same band is not allowed, modify automatically!"
  fi

  if [ $guest_band3 == 0 ] && [ $current_total_ap_number  -gt  3 ]; then
    guest_band3=$prefer_second_band
    #check that:  if ap_no_2g -ge 3 then set guest_band2 as 5
    get_ap_no_per_band
    if [ $ap_no_of_2g -ge 3 ]; then
      guest_band3=5
    fi
    log  $(basename "$0") "update_wireless_network" $LINENO " gustband3 is reset to $guest_band3"
  fi

   #if all primary_band/ guest_band1/ guest_band2/ guest_band3 are setted mannually then
  #check that:  if ap_no_5g -eq 3 then set guest_band3 as 2
  #check that:  if ap_no_2g -eq 3 then set guest_band3 as 5
  get_ap_no_per_band
  if [ $ap_no_of_5g -ge 3 ]; then
    guest_band3=2
    echo "three APs with same band is not allowed, modify automatically!"
  elif [ $ap_no_of_2g -ge 3 ]; then
    guest_band3=5
    echo "three APs with same band is not allowed, modify automatically!"
  fi

  util_execute_uci set qcmap_wlan_current.@primaryap[0].band=$primary_band
  util_execute_uci set qcmap_wlan_current.@stamodeconfig[0].band=$station_band
  util_execute_uci set qcmap_wlan_current.@guestap[0].band=$guest_band1
  util_execute_uci set qcmap_wlan_current.@guestaptwo[0].band=$guest_band2
  util_execute_uci set qcmap_wlan_current.@guestapthree[0].band=$guest_band3

  get_ap_no_per_band
  util_execute_uci set qcmap_wlan_current.@wlanconfig[0].guestapcount2g=$ap_no_of_2g
  util_execute_uci set qcmap_wlan_current.@wlanconfig[0].guestapcount5g=$ap_no_of_5g
  util_execute_uci set qcmap_wlan_current.@wlanconfig[0].guestapcount6g=$ap_no_of_6g

  log  $(basename "$0") "update_wireless_network" $LINENO " change wireless files"

  #update wlan port list in  network.device by delete then add
  #delete all, at most 4 AP here, will WKK support for ?
  log  $(basename "$0") "update_wireless_network" $LINENO " clean wlan ports !!!"
  for i in `seq 0 3`
  do
      ifname_list=$(util_execute_uci get network.lan.ifname)
      ifname_list=$(echo "$ifname_list" | sed s/wlan$i//)
      util_execute_uci_set "network.lan.ifname" "$ifname_list"
      util_execute_uci delete network.wlan$i  2>/dev/null #ignore error since it is expected that interfaces may not present in network
  done

  util_execute_uci delete network.wanwlan.device 2>/dev/null

  #Create empty /etc/config/tmp_wireless   file
  cat /dev/null > /etc/config/tmp_wireless

  #Copy [x]g_wireless file to /etc/config, then we can use uci command easily to change the files
  #handle primary band firstly, then primary band will come from wlan0
  if [ $primary_band != 0 ] ; then

    local ap_no_of_primary_band=0
    if [ $primary_band == 2 ] ; then
      ap_no_of_primary_band=$ap_no_of_2g
    elif [ $primary_band == 5 ] ; then
      ap_no_of_primary_band=$ap_no_of_5g
    elif [ $primary_band == 6 ] ; then
      ap_no_of_primary_band=$ap_no_of_6g
    fi
  fi

  if [ $station_band != 0 ] ; then

    if [ $primary_band == 0 ] ; then
      #STA only mode
      utility_update_single_wireless $station_band  0

    elif [ $station_band == $primary_band ] ; then
      #AP-STA or AP-AP-STA mode
      utility_update_single_wireless $primary_band $ap_no_of_primary_band
      utility_generate_wireless_file $ap_no_of_2g $ap_no_of_5g $ap_no_of_6g $primary_band

    elif [ $station_band != $primary_band ] ; then
      if [  $station_band != $guest_band1 ] ; then
        #AP-STA mode or AP-AP-STA mode and guest AP band != station band
        utility_update_single_wireless $station_band 0
      fi

      utility_generate_wireless_file $ap_no_of_2g $ap_no_of_5g $ap_no_of_6g $primary_band
      utility_update_single_wireless $primary_band $ap_no_of_primary_band
    fi
  elif [ $primary_band != 0 ] ; then
    util_execute_uci_set network.wanwlan_v6.disabled   "1"
    utility_update_single_wireless $primary_band $ap_no_of_primary_band
    utility_generate_wireless_file $ap_no_of_2g $ap_no_of_5g $ap_no_of_6g $primary_band
  fi

  log  $(basename "$0") "update_wireless_network" $LINENO " uci commit wireless/network"
  #restore wifi side configruation first.
  ucitool restore_default_config
  mv /etc/config/tmp_wireless /etc/config/wireless

  #set different band manually may cause wlan number incorrect, this function used to correct wlan number
  utility_update_wireless_for_different_band

  util_execute_uci commit wireless
  util_execute_uci commit network

}

function get_mld_link_count() {
  local band=$1
  local count=0
  local link_band

  local no_of_mld_link=$(uci get qcmap_wlan_current.@mldapconfig[0].no_of_mld_link)
  for i in `seq 0 $((no_of_mld_link-1))`
  do
    link_band=$(uci get qcmap_wlan_current.@mldapconfig[0].link_band$i)
    if [ $link_band -eq $band ]; then
      count=$((count+1))
    fi
  done

  echo $count
}

function utility_update_single_wireless_mld() {
  local band_no=$1
  local link_no=$2
  local tmp_delete_link_number=0
  local link_no_of_6g=$(get_mld_link_count 6)
  log  $(basename "$0") "utility_update_single_wireless_mld" $LINENO "handle $band_no no:$link_no"

  cp /etc/misc/wifi/"$band_no"g_wireless /etc/config/
  sed -i "s/wifiX/wifi$pdev_number/g" /etc/config/"$band_no"g_wireless

  if [ $band_no -eq 2 ]; then
    sed -i "s/11axg/11beg/g" /etc/config/2g_wireless
  elif [ $band_no -eq 5 ]; then
    sed -i "s/11axa/11bea/g" /etc/config/5g_wireless
  fi

  #Ensure wlan security setting the same when 6g link exist
  if [ $link_no_of_6g -gt 0 ] && [ $band_no -eq 2 -o $band_no -eq 5 ]; then
    sed -i "s/psk2+//g" /etc/config/"$band_no"g_wireless
    util_execute_uci del "$band_no"g_wireless.@wifi-iface[0].key
    util_execute_uci del "$band_no"g_wireless.@wifi-iface[1].key

    util_execute_uci add_list "$band_no"g_wireless.@wifi-iface[0].sae_password="1234567890"
    util_execute_uci add_list "$band_no"g_wireless.@wifi-iface[1].sae_password="1234567890"
    util_execute_uci set "$band_no"g_wireless.@wifi-iface[0].sae_pwe="1"
    util_execute_uci set "$band_no"g_wireless.@wifi-iface[1].sae_pwe="1"
  fi

  for i in `seq 1 $link_no`
  do
    util_execute_uci set "$band_no"g_wireless.wifi"$pdev_number"_AP$i.ifname="wlan0"
    util_execute_uci set "$band_no"g_wireless.wifi"$pdev_number"_AP$i.mld="mld0"
    util_execute_uci set "$band_no"g_wireless.wifi"$pdev_number"_AP$i.sae="1"
    util_execute_uci set "$band_no"g_wireless.wifi"$pdev_number"_AP$i.ssid="QSoftAP_MLD0"
  done

  #delete the AP/STA config not used, to let wireless file more clear
  tmp_delete_link_number=$((link_no + 1))
  for i in `seq $tmp_delete_link_number 2`
  do
    util_execute_uci delete "$band_no"g_wireless.wifi"$pdev_number"_AP$i
  done
  util_execute_uci delete "$band_no"g_wireless.wifi"$pdev_number"_STA
  util_execute_uci commit "$band_no"g_wireless

  cat /etc/config/"$band_no"g_wireless >> /etc/config/tmp_wireless
  rm /etc/config/"$band_no"g_wireless

  pdev_number=$((pdev_number + 1))
  log  $(basename "$0") "utility_update_single_wireless_mld" $LINENO " update next pdev number as $pdev_number"
}

# Only support single MLD SAP for HMT, max 2 links
function update_wireless_network_ex() {
  local wlanconfigex=$(uci get qcmap_wlan_current.@wlanconfig[0].wlanconfigex)
  local totalmldapcount=$(uci get qcmap_wlan_current.@mldwlanconfig[0].totalmldapcount)

  local link_band0=$(uci get qcmap_wlan_current.@mldapconfig[0].link_band0)
  local link_band1=$(uci get qcmap_wlan_current.@mldapconfig[0].link_band1)
  local link_band2=$(uci get qcmap_wlan_current.@mldapconfig[0].link_band2)
  local link_no_of_2g=$(get_mld_link_count 2)
  local link_no_of_5g=$(get_mld_link_count 5)
  local link_no_of_6g=$(get_mld_link_count 6)
  local ifname_list=""

  log  $(basename "$0") "update_wireless_network_ex" $LINENO " totalmldapcount:$totalmldapcount"
  if [ $wlanconfigex -eq 1 ] && [ "$totalmldapcount" == "1" ]; then
    #delete all wlan port list in network.device, at most 4 AP here
    log  $(basename "$0") "update_wireless_network_ex" $LINENO " clean wlan ports !!!"
    for i in `seq 0 3`
    do
      ifname_list=$(util_execute_uci get network.lan.ifname)
      ifname_list=$(echo "$ifname_list" | sed s/wlan$i//)
      util_execute_uci_set "network.lan.ifname" "$ifname_list"
      #ignore error since it is expected that interfaces may not present in network
      util_execute_uci delete network.wlan$i  2>/dev/null
    done
    util_execute_uci delete network.wanwlan.device 2>/dev/null

    #Create empty /etc/config/tmp_wireless file
    cat /dev/null > /etc/config/tmp_wireless

    if [ $link_no_of_2g != 0 ]; then
      log  $(basename "$0") "update_wireless_network_ex" $LINENO " handle 2g_wireless no:$link_no_of_2g"
      utility_update_single_wireless_mld 2 $link_no_of_2g
    fi

    if [ $link_no_of_5g != 0 ]; then
      log  $(basename "$0") "update_wireless_network_ex" $LINENO " handle 5g_wireless no:$link_no_of_5g"
      utility_update_single_wireless_mld 5 $link_no_of_5g
    fi

    if [ $link_no_of_6g != 0 ] ; then
      log  $(basename "$0") "update_wireless_network_ex" $LINENO " handle 6g_wireless no:$link_no_of_6g"
      utility_update_single_wireless_mld 6 $link_no_of_6g
    fi

    #add wlan0 in network.lan.ifname
    ifname_list=$(util_execute_uci get network.lan.ifname)
    ifname_list=$(echo "$ifname_list" | sed "s/^ *//g" | sed "s/ *$//g" | tr -s ' ')
    util_execute_uci_set "network.lan.ifname" "$ifname_list wlan0"

    #add wlan0 interface in network
    util_execute_uci set network.wlan0="interface"
    util_execute_uci set network.wlan0.device="wlan0"
    util_execute_uci set network.wlan0.proto="static"
    util_execute_uci set network.wlan0.ipv6="1"
    util_execute_uci set network.wlan0.ipaddr=$(eval "echo \${static_ip_wlan0}")
    util_execute_uci set network.wlan0.netmask="255.255.255.0"
    util_execute_uci commit network

    #restore wifi side configruation first.
    ucitool restore_default_config
    mv /etc/config/tmp_wireless /etc/config/wireless

    #Add MLD AP config
    #Note: mld_macaddr is congugurale by user, we are setting dummy mac for single MLD AP
    util_execute_uci_set wireless.mld0 wifi-mld
    util_execute_uci_set wireless.mld0.mld_macaddr "aa:bb:cc:dd:ee:ff"
    util_execute_uci_set wireless.mld0.mld_ssid "QSoftAP_MLD0"
    util_execute_uci_set wireless.mld0.mld_ref "0"
    util_execute_uci commit wireless

    log $(basename "$0") "update_wireless_network_ex" $LINENO "/etc/config/wireless updated"
  fi
}

function start_hostapd_cli_service_sta() {
  log  $(basename "$0") "start_hostapd_cli_service_sta" $LINENO " begin !!!"
  local wlan_mode=$(util_execute_uci get qcmap_wlan.@wlanconfig[0].mode)
  local wlan_interface_start
  local wlan_interface_end
  get_total_ap_number

  if [ $wlan_mode == $QCMAP_WLAN_MODE_AP_STA ] || [ $wlan_mode == $QCMAP_WLAN_MODE_AP_AP_STA ] ; then
    wlan_interface_start=1
    wlan_interface_end=$total_ap_number
  elif [ $wlan_mode == $QCMAP_WLAN_MODE_AP ] || [ $wlan_mode == $QCMAP_WLAN_MODE_AP_AP ] || [ $wlan_mode == $QCMAP_WLAN_MODE_AP_AP_AP ] || [ $wlan_mode == $QCMAP_WLAN_MODE_AP_AP_AP_AP ]; then
    wlan_interface_start=0
    wlan_interface_end=$((total_ap_number-1))
  fi

  for ap_no in `seq $wlan_interface_start $wlan_interface_end`
  do
    for line in $(/usr/sbin/hostapd_cli -i wlan$ap_no -s /data/vendor/wifi -p /data/vendor/wifi/hostapd list_sta)
    do
      /usr/bin/QCMAP_StaInterface "wlan$ap_no" "AP-STA-CONNECTED" "${line}"
      log $(basename "$0") "start_hostapd_cli_service_sta:" $LINENO " notify QCMAP on wlan$ap_no AP-STA-CONNECTED ${line}"
    done

    /usr/sbin/hostapd_cli -i wlan$ap_no -s /data/vendor/wifi -p /data/vendor/wifi/hostapd -B -a /usr/bin/QCMAP_StaInterface
  done


  log $(basename "$0") "start_hostapd_cli_service_sta:" $LINENO " hostapd_cli service starts successfully"
}

function enable_wlan_hmt() {
  cp /etc/config/qcmap_wlan_current /etc/config/qcmap_wlan
  util_execute_uci commit qcmap_wlan

  #enable wireless.wifi-ifaces firstly then call wifi up
  for i in `seq 0 3`
    do
      util_execute_uci -q set wireless.@wifi-iface[$i].disabled=0
    done

  util_execute_uci set wireless.wifi0.disabled=0  2>/dev/null
  util_execute_uci set wireless.wifi1.disabled=0  2>/dev/null
  util_execute_uci set wireless.wifi2.disabled=0  2>/dev/null

  util_execute_uci commit wireless

#  backhaulCommConfig will add wanwlan/wanwlan_v6 to wan zone after get IP address
#  update_wanwlan_iface_on_wan_zone 1 #add wanwlan in wan zone

  /sbin/wifi up
}

function disable_wlan_hmt() {
  #wifi down firstly then disable wireless.wifi-ifaces
  /sbin/wifi down
  reset_wlan_atbootup_hmt
}

# WLAN reset in case SSR scenario
function reset_wlan_atbootup_hmt() {
  local wlanmode=$(uci get qcmap_wlan.@wlanconfig[0].mode)
  local mldwlanmode=$(uci get qcmap_wlan.@mldwlanconfig[0].mode)
  local ezmesh_enable=$(uci get qcmap_ezmesh.@ezmesh[0].enable_state)
  local ezmesh_ap_len=$(uci get qcmap_ezmesh.@ezmesh[0].ap_config_len)

  update_wanwlan_iface_on_wan_zone 0 #remove wanwlan in wan zone

  if [ $wlanmode -ne 0 ] || [ $mldwlanmode -ne 0 ] || { [ $ezmesh_enable -eq 1 ] && [ $ezmesh_ap_len -ne 0 ]; }; then
    for i in `seq 0 3`
      do
        util_execute_uci -q set wireless.@wifi-iface[$i].disabled=1
      done

    util_execute_uci set wireless.wifi0.disabled=1  2>/dev/null
    util_execute_uci set wireless.wifi1.disabled=1  2>/dev/null
    util_execute_uci set wireless.wifi2.disabled=1  2>/dev/null
    util_execute_uci commit wireless
  fi

  util_execute_uci set qcmap_lan.@runstatus[0].sap='0'
  util_execute_uci set qcmap_lan.@runstatus[0].station='0'
  util_execute_uci set qcmap_lan.@runstatus[0].activatesupplicant='0'
  util_execute_uci set qcmap_lan.@runstatus[0].activatehostapd='0'
  util_execute_uci commit qcmap_lan

  local sta_mode_conn_type=$(util_execute_uci get qcmap_wlan.@stamodeconfig[0].sta_mode_conn_type)
  local bridge_mode=$(util_execute_uci get qcmap_wlan.@stamodeconfig[0].bridge_flag)
  if [ $sta_mode_conn_type -eq $QCMAP_STA_CONNECTION_STATIC ] || [ $bridge_mode == 1 ]; then
    ap_sta_mode_disconnect 1
  fi
  util_execute_uci set qcmap_wlan.@stamodeconfig[0].bridge_flag="0"
  uci commit qcmap_wlan
}

function get_mldap_single_link_status() {
  local link_number=$1
  local retry=0
  local mld_ap_pid=0
  local hostapd_mld_pid_file=/data/vendor/wifi/hostapd-global.pid

  while [ $retry -lt $hostapd_start_poll_max_count ]
  do
    if [ -f $hostapd_mld_pid_file ]; then
      mld_ap_pid=`cat $hostapd_mld_pid_file`
      log  $(basename "$0") "get_mldap_link_status" $LINENO " Get MLD AP pid:$mld_ap_pid"
      if [ $mld_ap_pid -ne 0 ]; then
        if [ $link_number -eq 0 ]; then
          sap_link_state=`hostapd_cli -i wlan0 status | grep -i state | awk -F '=' '{print $2}'`
        else
          sap_link_state=`hostapd_cli -p /data/vendor/wifi/hostapd-1 -i wlan0 status | grep -i state | awk -F '=' '{print $2}'`
        fi

        if [ \( "$sap_link_state" == "ENABLED" \) -o \( "$sap_link_state" == "DFS" \) ]; then
          log  $(basename "$0") "get_mldap_link_status" $LINENO " Get MLD AP link$link_number status:$sap_link_state"
          return
        fi
      fi
    fi

    retry=$(($retry+1))
    sleep $hostapd_poll_delay_S
  done
}

function get_single_sap_status() {
  local ap_number=$1
  local single_sap_state=""
  local retry=0
  local current_ap_pid_file=/data/vendor/wifi/hostapd-global.pid

  if [ ! -f $current_ap_pid_file ]; then
    current_ap_pid_file=/data/vendor/wifi/hostapd-wlan$ap_number.pid
  fi

  while [ $retry -lt $hostapd_start_poll_max_count ]
  do
    if [ -f $current_ap_pid_file ]; then
      current_ap_pid=`cat $current_ap_pid_file`
      log  $(basename "$0") "get_single_sap_status" $LINENO " Get wlan$ap_number pid:$current_ap_pid"
      if [ $current_ap_pid -ne 0 ]; then
        single_sap_state=`hostapd_cli -i wlan$ap_number status | grep -i state | awk -F '=' '{print $2}'`
        if [ \( "$single_sap_state" == "ENABLED" \) -o \( "$single_sap_state" == "DFS" \) ]; then
          log  $(basename "$0") "get_single_sap_status" $LINENO " Get wlan$ap_number sap status:$single_sap_state"
          #save the correct status
          sap_state=$single_sap_state
          log  $(basename "$0") "get_single_sap_status" $LINENO " Retry number: $retry"
          return
        fi
      fi
    fi

    #br_state=`brctl show | grep wlan$ap_number`
    #if [ \( "$br_state" == "" \) ]; then
    #  log  $(basename "$0") "get_single_sap_status" $LINENO "bridge add failure"
    #  return
    #else
    #  log  $(basename "$0") "get_single_sap_status" $LINENO "$br_state"
    #fi

    retry=$(($retry+1))
    sleep $hostapd_poll_delay_S
  done

  #save the fail status
  sap_state=$single_sap_state
}

#GET hostapd state info to check if really AP mode is started or not
function get_sap_status() {
  local wlan_mode=$(util_execute_uci get qcmap_wlan.@wlanconfig[0].mode)
  local wlanconfigex=$(util_execute_uci get qcmap_wlan.@wlanconfig[0].wlanconfigex)
  local totalmldapcount=$(util_execute_uci get qcmap_wlan.@mldwlanconfig[0].totalmldapcount)
  local mldaplink=$(util_execute_uci get qcmap_wlan.@mldapconfig[$i].no_of_mld_link)
  local ap_start_number=0
  local isWirelessDisabled=$(util_execute_uci get wireless.wifi0.disabled)
  local supplicant_pid_file

  if [ $isWirelessDisabled == 1 ]; then
    # if wireless is disabled, don't need check status
    log $(basename "$0") "get_sap_status" $LINENO "WLAN is Diabled"
    util_execute_uci set qcmap_lan.@runstatus[0].sap='0'
    util_execute_uci commit qcmap_lan
    return
  fi

  #single MLD AP status
  if [ $wlanconfigex -eq 1 ] && [ "$totalmldapcount" == "1" ]; then
    for i in `seq 0 $((mldaplink-1))`
    do
      sap_link_state=""
      get_mldap_single_link_status $i

      if [ \( "$sap_link_state" != "ENABLED" \) -a \( "$sap_link_state" != "DFS" \) ]; then
        log  $(basename "$0") "get_sap_status" $LINENO " Hostapd start failed, mld0 sap link$i status:$sap_link_state"
        util_execute_uci set qcmap_lan.@runstatus[0].sap='0'
        util_execute_uci commit qcmap_lan
        return
      fi
    done

    util_execute_uci set qcmap_lan.@runstatus[0].sap='1'
    util_execute_uci commit qcmap_lan
    log $(basename "$0") "get_sap_status" $LINENO "Get mld0 last link status:$sap_link_state"
    echo Hostapd start success, Get mld0 status:$sap_link_state > /dev/kmsg
    return
  fi

  get_total_ap_number
  if [ $total_ap_number == 0 ]; then
    # check STA is enabled or not
    supplicant_pid_file=/data/vendor/wifi/wpa_supplicant-wlan0.pid
    if [ -f $supplicant_pid_file ]; then
      log $(basename "$0") "get_sap_status" $LINENO "WLAN STA Only mode is enabled already"
    fi
    return
  fi

  if [ $wlan_mode == $QCMAP_WLAN_MODE_AP_STA ] ||  [ $wlan_mode == $QCMAP_WLAN_MODE_STA_ONLY ] || [ $wlan_mode == $QCMAP_WLAN_MODE_AP_AP_STA ] ; then
    ap_start_number=1
  else
    total_ap_number=$((total_ap_number-1))
  fi

  for i in $(seq $ap_start_number $total_ap_number)
  do
    sap_state=""
    get_single_sap_status $i

    if [ \( "$sap_state" != "ENABLED" \) -a \( "$sap_state" != "DFS" \) ]; then
      log  $(basename "$0") "get_sap_status" $LINENO " Hostapd start failed, Get wlan$i status:$sap_state"
      util_execute_uci set qcmap_lan.@runstatus[0].sap='0'
      util_execute_uci commit qcmap_lan

      return
    fi
  done

  #print the last ap status info
  util_execute_uci set qcmap_lan.@runstatus[0].sap='1'
  util_execute_uci commit qcmap_lan
  log $(basename "$0") "get_sap_status" $LINENO "Hostapd start success, Get wlan$i status:$sap_state"
  echo Hostapd start success, Get wlan$i status:$sap_state > /dev/kmsg
}

#GET supplicant state info to check if STA/AP+STA/AP+AP+STA mode is started or not
function get_supplicant_status_hmt() {
  sta_iface="wlan0" #by default, HMT sta interface
  get_supplicant_status
}

function install_single_guestap_access_rules() {
  local guest_ap1_profile=$(util_execute_uci get qcmap_wlan.@guestap[0].accessprofile)
  local guest_ap2_profile=$(util_execute_uci get qcmap_wlan.@guestaptwo[0].accessprofile)
  local guest_ap3_profile=$(util_execute_uci get qcmap_wlan.@guestapthree[0].accessprofile)
  local guest_ap_profile=0
  local guest_ap_number=$1

  if [ $guest_ap_number == 1 ] ; then
    guest_ap_profile=$guest_ap1_profile
  elif [ $guest_ap_number == 2 ] ; then
    guest_ap_profile=$guest_ap2_profile
  elif [ $guest_ap_number == 3 ] ; then
    guest_ap_profile=$guest_ap3_profile
  else
    log  $(basename "$0") "install_single_guestap_access_rules" $LINENO "invalid Guest AP number: $guest_ap_number"
    return
  fi

  #run this funtion first before install guestap access rules.
  utility_update_guest_ap_public_ip

  #Install access restrictions for the Guest SSID if the profile is INTERNETONLY
  if [ $guest_ap_profile == 1 ] ; then
    utility_install_single_guestap_access_rules wlan$1
    log  $(basename "$0") "install_single_guestap_access_rules" $LINENO "guestap_$1 access is Limited, install rules success"

    #Set wlan$1 mode to "internet" in "/etc/data/ipa/IPACM_cfg.xml"
    ipa xml wlanmode wlan$1 internet
    log  $(basename "$0") "install_single_guestap_access_rules" $LINENO "Set wlan$1 mode to internet in /etc/data/ipa/IPACM_cfg.xml"
  else
    log  $(basename "$0") "install_single_guestap_access_rules" $LINENO "guestap_$1 access is Full"
  fi
}

#Install access restrictions for the Guest SSID if the profile is INTERNETONLY.
function install_guest_ap_access_rules() {
  get_total_ap_number
  local wlan_mode=$(util_execute_uci get qcmap_wlan.@wlanconfig[0].mode)
  local guest_ap_start_number=1

  #return while no GUEST AP found
  if [ $total_ap_number -le 1 ]; then
    log $(basename "$0") "install_guest_ap_access_rules" $LINENO " no Guest AP found."
    return
  fi

  if [ $wlan_mode == $QCMAP_WLAN_MODE_AP_AP_STA ]; then
    guest_ap_start_number=2
  else
    total_ap_number=$((total_ap_number-1))
  fi

  #clean ebTables filter tables
  util_run_command ebtables -t filter --flush
  /etc/data/qcmap_ebtables.sh
  for i in $(seq $guest_ap_start_number $total_ap_number)
  do
    #Set wlan$i mode to default "full"
    ipa xml wlanmode wlan$i full

    #Install ebtables rules for Guest AP access
    install_single_guestap_access_rules $i
  done
}

function get_wlan_disable_status() {
  for i in `seq 0 3`
  do
    wlan_disable_status=$(util_execute_uci -q get wireless.@wifi-iface[$i].disabled)
    if [ "$wlan_disable_status" != "" ] && [ $wlan_disable_status == 0 ] ; then
      log  $(basename "$0") "get_wlan_disable_status" $LINENO " WLAN is enable!!!"
      return
    fi
  done

  wlan_disable_status=1
  log  $(basename "$0") "get_wlan_disable_status" $LINENO " WLAN is disable!!!"
}

function get_active_ap_pid() {
  local hostapd_wlan0_pid_file=/data/vendor/wifi/hostapd-wlan0.pid
  local hostapd_wlan1_pid_file=/data/vendor/wifi/hostapd-wlan1.pid
  local hostapd_wlan2_pid_file=/data/vendor/wifi/hostapd-wlan2.pid
  local hostapd_wlan3_pid_file=/data/vendor/wifi/hostapd-wlan3.pid
  local wlan_mode=$(util_execute_uci get qcmap_wlan.@wlanconfig[0].mode)

  log  $(basename "$0") "get_active_ap_pid" $LINENO " begin !!!"

  #set the pid to default 0
  ap_pid=0
  guest_ap_pid=0
  guest_ap2_pid=0
  guest_ap3_pid=0

  #if wifi mode include sta, return after get pid
  if [ $wlan_mode -eq $QCMAP_WLAN_MODE_AP_STA ] || [ $wlan_mode -eq $QCMAP_WLAN_MODE_AP_AP_STA ]; then
    #Get the Primary AP PID
    if [ -f $hostapd_wlan1_pid_file ]; then
      ap_pid=`cat $hostapd_wlan1_pid_file`
      log $(basename "$0") "get_active_ap_pid" $LINENO "wlan_mode: $wlan_mode, Primary AP PID: $ap_pid"
    fi

    #Get the Guest AP PID
    if [ -f $hostapd_wlan2_pid_file ]; then
      guest_ap_pid=`cat $hostapd_wlan2_pid_file`
      log $(basename "$0") "get_active_ap_pid" $LINENO "wlan_mode: $wlan_mode, Guest AP2 PID: $guest_ap_pid"
    fi
    return
  fi

  #Get the Primary AP PID
  if [ -f $hostapd_wlan0_pid_file ]; then
    ap_pid=`cat $hostapd_wlan0_pid_file`
    log $(basename "$0") "get_active_ap_pid" $LINENO "wlan_mode: $wlan_mode, Primary AP PID: $ap_pid"
  fi

  #Get the Guest AP PID
  if [ -f $hostapd_wlan1_pid_file ]; then
    guest_ap_pid=`cat $hostapd_wlan1_pid_file`
    log $(basename "$0") "get_active_ap_pid" $LINENO "wlan_mode: $wlan_mode, Guest AP PID: $guest_ap_pid"
  fi

  #Get the Guest AP2 PID
  if [ -f $hostapd_wlan2_pid_file ]; then
    guest_ap2_pid=`cat $hostapd_wlan2_pid_file`
    log $(basename "$0") "get_active_ap_pid" $LINENO "wlan_mode: $wlan_mode, Guest AP2 PID: $guest_ap2_pid"
  fi

  #Get the Guest AP3 PID
  if [ -f $hostapd_wlan3_pid_file ]; then
    guest_ap3_pid=`cat $hostapd_wlan3_pid_file`
    log $(basename "$0") "get_active_ap_pid" $LINENO "wlan_mode: $wlan_mode, Guest AP3 PID: $guest_ap3_pid"
  fi

  log  $(basename "$0") "get_active_ap_pid" $LINENO " end !!!"
}

function get_activate_ap_number() {
  local wlan_mode=$(util_execute_uci get qcmap_wlan.@wlanconfig[0].mode)
  local ap_type=$1

  if [ $wlan_mode -eq $QCMAP_WLAN_MODE_AP_STA ] || [ $wlan_mode -eq $QCMAP_WLAN_MODE_AP_AP_STA ]; then
    activate_ap_number=$ap_type
  else
    activate_ap_number=$((ap_type-1))
  fi

  log  $(basename "$0") "get_activate_ap_number" $LINENO " ap_type:$ap_type, wlan_mode:$wlan_mode, activate_ap_number:$activate_ap_number"
}

function get_wlan_pdev() {
  local wlan_ifname=""
  local wlan_number=$1
  local i j

  log $(basename "$0") "get_wlan_pdev" $LINENO "Start get_wlan_pdev"

  for i in `seq 0 1`
  do
    for j in `seq 1 2`
    do
      wlan_ifname=$(uci get wireless.wifi$i\_AP$j.ifname)
      if [ "$wlan_ifname" != "" ] && [ "$wlan_ifname" == "wlan$wlan_number" ]; then
        wlan_pdev=$(uci get wireless.wifi$i\_AP$j.device)
        log $(basename "$0") "get_wlan_pdev" $LINENO "get wlan$wlan_number pdev: $wlan_pdev"
        return
      fi
    done
  done

  log $(basename "$0") "get_wlan_pdev" $LINENO "get wlan$wlan_number pdev fail"
}

function StartHostAPD() {
  local wlan_mode=$(util_execute_uci get qcmap_wlan.@wlanconfig[0].mode)
  local ap_type=$1

  log $(basename "$0") "StartHostAPD" $LINENO "Start Hostapd, ap_type:$ap_type, wlan_mode:$wlan_mode"

  get_activate_ap_number $ap_type
  get_wlan_pdev $activate_ap_number

  wifi multi_up $wlan_pdev wlan$activate_ap_number >> /dev/null
  #sleep 1 second wait pid file update
  sleep 1

  #CheckAPStatus
  sap_state=""
  get_single_sap_status $activate_ap_number
  if [ \( "$sap_state" != "ENABLED" \) -a \( "$sap_state" != "DFS" \) ]; then
    log  $(basename "$0") "StartHostAPD" $LINENO " Hostapd start failed, Get wlan$activate_ap_number status:$sap_state"
    hostapd_activate_status=0
    return
  fi

  #start hostapd_cli service process after hostapd start
  /usr/sbin/hostapd_cli -i wlan$activate_ap_number -s /data/vendor/wifi -p /data/vendor/wifi/hostapd -B -a /usr/bin/QCMAP_StaInterface

  log $(basename "$0") "StartHostAPD" $LINENO "End StartHostAPD"
}

function activate_hostapd_action_start() {
  local ap_type=$1
  get_active_ap_pid
  get_total_ap_number

  if [ $ap_type -eq $QCMAP_PRIMARY_AP ] && [ $ap_pid -ne 0 ]; then
    log $(basename "$0") "activate_hostapd_action_start" $LINENO "Hostapd already running for Primary AP. PID: $ap_pid"
    hostapd_activate_status=0
    return
  elif [ $ap_type -eq $QCMAP_GUEST_AP ] && [ $guest_ap_pid -ne 0 ]; then
    log $(basename "$0") "activate_hostapd_action_start" $LINENO "Hostapd already running for Guest AP. PID: $guest_ap_pid"
    hostapd_activate_status=0
    return
  elif [ $ap_type -eq $QCMAP_GUEST_AP_2 ] && [ $guest_ap2_pid -ne 0 ]; then
    log $(basename "$0") "activate_hostapd_action_start" $LINENO "Hostapd already running for Guest AP2. PID: $guest_ap2_pid"
    hostapd_activate_status=0
    return
  elif [ $ap_type -eq $QCMAP_GUEST_AP_3 ] && [ $guest_ap3_pid -ne 0 ]; then
    log $(basename "$0") "activate_hostapd_action_start" $LINENO "Hostapd already running for Guest AP3. PID: $guest_ap3_pid"
    hostapd_activate_status=0
    return
  fi

  #Start Hostapd for ALL AP
  if [ $ap_type -eq $QCMAP_ALL_AP ]; then
    if [ $ap_pid -ne 0 ] && [ $guest_ap_pid -ne 0 ] && [ $guest_ap2_pid -ne 0 ] && [ $guest_ap3_pid -ne 0 ]; then
      log $(basename "$0") "activate_hostapd_action_start" $LINENO "Hostapd are running for all AP."
      hostapd_activate_status=0
      return
    fi

    for i in `seq $QCMAP_PRIMARY_AP $total_ap_number`
    do
      if [ $i -eq $QCMAP_PRIMARY_AP ] && [ $ap_pid -ne 0 ]; then
        log $(basename "$0") "activate_hostapd_action_start" $LINENO "Hostapd already running for Primary AP. PID: $ap_pid"
        continue
      elif [ $i -eq $QCMAP_GUEST_AP ] && [ $guest_ap_pid -ne 0 ]; then
        log $(basename "$0") "activate_hostapd_action_start" $LINENO "Hostapd already running for Guest AP. PID: $guest_ap_pid"
        continue
      elif [ $i -eq $QCMAP_GUEST_AP_2 ] && [ $guest_ap2_pid -ne 0 ]; then
        log $(basename "$0") "activate_hostapd_action_start" $LINENO "Hostapd already running for Guest AP2. PID: $guest_ap2_pid"
        continue
      elif [ $i -eq $QCMAP_GUEST_AP_3 ] && [ $guest_ap3_pid -ne 0 ]; then
        log $(basename "$0") "activate_hostapd_action_start" $LINENO "Hostapd already running for Guest AP3. PID: $guest_ap3_pid"
        continue
      fi

      StartHostAPD $i
      if [ $hostapd_activate_status -ne 1 ]; then
        log $(basename "$0") "activate_hostapd_action_start" $LINENO "Activate Hostapd Start Fail, ap_type: $ap_type"
        return
      fi
    done
  #Start Hostapd for Primary_AP/Guest_AP/Guest_AP2/Guest_AP3
  else
    StartHostAPD $ap_type
    if [ $hostapd_activate_status -ne 1 ]; then
      log $(basename "$0") "activate_hostapd_action_start" $LINENO "Activate Hostapd Start Fail, ap_type: $ap_type"
      return
    fi
  fi
}

function StopHostAPD() {
  local wlan_mode=$(util_execute_uci get qcmap_wlan.@wlanconfig[0].mode)
  local ap_type=$1

  log $(basename "$0") "StopHostAPD" $LINENO "Stop Hostapd, ap_type:$ap_type, wlan_mode:$wlan_mode"

  get_activate_ap_number $ap_type
  get_wlan_pdev $activate_ap_number

  wifi multi_down $wlan_pdev wlan$activate_ap_number >> /dev/null
  #sleep 1 second wait pid file update
  sleep 1

  #get the pid after stop hostapd
  get_active_ap_pid

  #CheckHostapdStatus
  if [ $ap_type -eq $QCMAP_PRIMARY_AP ] && [ $ap_pid -ne 0 ]; then
    log $(basename "$0") "StopHostAPD" $LINENO "Hostapd stop failed, wlan_mode:$wlan_mode, ap_type:$ap_type"
    hostapd_activate_status=0
    return
  elif [ $ap_type -eq $QCMAP_GUEST_AP ] && [ $guest_ap_pid -ne 0 ]; then
    log $(basename "$0") "StopHostAPD" $LINENO "Hostapd stop failed, wlan_mode:$wlan_mode, ap_type:$ap_type"
    hostapd_activate_status=0
    return
  elif [ $ap_type -eq $QCMAP_GUEST_AP_2 ] && [ $guest_ap2_pid -ne 0 ]; then
    log $(basename "$0") "StopHostAPD" $LINENO "Hostapd stop failed, wlan_mode:$wlan_mode, ap_type:$ap_type"
    hostapd_activate_status=0
    return
  elif [ $ap_type -eq $QCMAP_GUEST_AP_3 ] && [ $guest_ap3_pid -ne 0 ]; then
    log $(basename "$0") "StopHostAPD" $LINENO "Hostapd stop failed, wlan_mode:$wlan_mode, ap_type:$ap_type"
    hostapd_activate_status=0
    return
  fi

  log $(basename "$0") "StopHostAPD" $LINENO "End StopHostAPD"
}

function activate_hostapd_action_stop() {
  local ap_type=$1
  get_active_ap_pid
  get_total_ap_number

  if [ $ap_type -eq $QCMAP_PRIMARY_AP ] && [ $ap_pid -eq 0 ]; then
    log $(basename "$0") "activate_hostapd_action_stop" $LINENO "Hostapd already stopped for Primary AP"
    hostapd_activate_status=0
    return
  elif [ $ap_type -eq $QCMAP_GUEST_AP ] && [ $guest_ap_pid -eq 0 ]; then
    log $(basename "$0") "activate_hostapd_action_stop" $LINENO "Hostapd already stopped for Guest AP"
    hostapd_activate_status=0
    return
  elif [ $ap_type -eq $QCMAP_GUEST_AP_2 ] && [ $guest_ap2_pid -eq 0 ]; then
    log $(basename "$0") "activate_hostapd_action_stop" $LINENO "Hostapd already stopped for Guest AP2"
    hostapd_activate_status=0
    return
  elif [ $ap_type -eq $QCMAP_GUEST_AP_3 ] && [ $guest_ap3_pid -eq 0 ]; then
    log $(basename "$0") "activate_hostapd_action_stop" $LINENO "Hostapd already stopped for Guest AP3"
    hostapd_activate_status=0
    return
  fi

  #Stop Hostapd for ALL AP
  if [ $ap_type -eq $QCMAP_ALL_AP ]; then
    if [ $ap_pid -eq 0 ] && [ $guest_ap_pid -eq 0 ] && [ $guest_ap2_pid -eq 0 ] && [ $guest_ap3_pid -eq 0 ]; then
      log $(basename "$0") "activate_hostapd_action_stop" $LINENO "Hostapd is stopped for all AP."
      hostapd_activate_status=0
      return
    fi

    for i in `seq $QCMAP_PRIMARY_AP $total_ap_number`
    do
      if [ $i -eq $QCMAP_PRIMARY_AP ] && [ $ap_pid -eq 0 ]; then
        log $(basename "$0") "activate_hostapd_action_stop" $LINENO "Hostapd already stopped for Primary AP"
        continue
      elif [ $i -eq $QCMAP_GUEST_AP ] && [ $guest_ap_pid -eq 0 ]; then
        log $(basename "$0") "activate_hostapd_action_stop" $LINENO "Hostapd already stopped for Guest AP"
        continue
      elif [ $i -eq $QCMAP_GUEST_AP_2 ] && [ $guest_ap2_pid -eq 0 ]; then
        log $(basename "$0") "activate_hostapd_action_stop" $LINENO "Hostapd already stopped for Guest AP2"
        continue
      elif [ $i -eq $QCMAP_GUEST_AP_3 ] && [ $guest_ap3_pid -eq 0 ]; then
        log $(basename "$0") "activate_hostapd_action_stop" $LINENO "Hostapd already stopped for Guest AP3"
        continue
      fi

      StopHostAPD $i
      if [ $hostapd_activate_status -ne 1 ]; then
        log $(basename "$0") "activate_hostapd_action_stop" $LINENO "Activate Hostapd Stop Fail, ap_type: $ap_type"
        return
      fi
    done
  #Stop Hostapd for Primary_AP/Guest_AP/Guest_AP2/Guest_AP3
  else
    StopHostAPD $ap_type
    if [ $hostapd_activate_status -ne 1 ]; then
      log $(basename "$0") "activate_hostapd_action_stop" $LINENO "Activate Hostapd Stop Fail, ap_type: $ap_type"
      return
    fi
  fi
}

function RestartHostAPD() {
  local wlan_mode=$(util_execute_uci get qcmap_wlan.@wlanconfig[0].mode)
  local ap_type=$1

  log $(basename "$0") "RestartHostAPD" $LINENO "Retart Hostapd, ap_type:$ap_type, wlan_mode:$wlan_mode"

  sleep 0.5
  #save the pid before restart
  get_active_ap_pid
  local old_ap_pid=$ap_pid
  local old_guest_ap_pid=$guest_ap_pid
  local old_guest_ap2_pid=$guest_ap2_pid
  local old_guest_ap3_pid=$guest_ap3_pid

  get_activate_ap_number $ap_type
  get_wlan_pdev $activate_ap_number

  wifi multi_down $wlan_pdev wlan$activate_ap_number >> /dev/null
  #sleep 1 second to bring down iface.
  sleep 1

  wifi multi_up $wlan_pdev wlan$activate_ap_number >> /dev/null
  #sleep 1 second wait pid file update
  sleep 1

  #get the pid after restart
  get_active_ap_pid

  #Check Hostapd Restart or not
  if [ $ap_type -eq $QCMAP_PRIMARY_AP ] && [ $ap_pid -eq $old_ap_pid ]; then
    log $(basename "$0") "RestartHostAPD" $LINENO "Retart Hostapd for Primary AP fail, ap_type:$ap_type, ap_pid:$ap_pid"
    hostapd_activate_status=0
    return
  elif [ $ap_type -eq $QCMAP_GUEST_AP ] && [ $guest_ap_pid -eq $old_guest_ap_pid ]; then
    log $(basename "$0") "RestartHostAPD" $LINENO "Retart Hostapd for Guest AP fail, ap_type:$ap_type, guest_ap_pid:$guest_ap_pid"
    hostapd_activate_status=0
    return
  elif [ $ap_type -eq $QCMAP_GUEST_AP_2 ] && [ $guest_ap2_pid -eq $old_guest_ap2_pid ]; then
    log $(basename "$0") "RestartHostAPD" $LINENO "Retart Hostapd for Guest AP2 fail, ap_type:$ap_type, guest_ap2_pid:$guest_ap2_pid"
    hostapd_activate_status=0
    return
  elif [ $ap_type -eq $QCMAP_GUEST_AP_3 ] && [ $guest_ap3_pid -eq $old_guest_ap3_pid ]; then
    log $(basename "$0") "RestartHostAPD" $LINENO "Retart Hostapd for Guest AP3 fail, ap_type:$ap_type, guest_ap3_pid:$guest_ap3_pid"
    hostapd_activate_status=0
    return
  fi

  #CheckAPStatus
  sap_state=""
  get_single_sap_status $activate_ap_number
  if [ \( "$sap_state" != "ENABLED" \) -a \( "$sap_state" != "DFS" \) ]; then
    log  $(basename "$0") "RestartHostAPD" $LINENO " Hostapd start failed, Get wlan$activate_ap_number status:$sap_state"
    hostapd_activate_status=0
    return
  fi

  #start hostapd_cli service process after hostapd restart
  /usr/sbin/hostapd_cli -i wlan$activate_ap_number -s /data/vendor/wifi -p /data/vendor/wifi/hostapd -B -a /usr/bin/QCMAP_StaInterface
  log $(basename "$0") "RestartHostAPD" $LINENO "End RestartHostAPD"
}

function activate_hostapd_action_restart() {
  local ap_type=$1

  get_total_ap_number

  #Restart Hostapd for ALL AP
  if [ $ap_type -eq $QCMAP_ALL_AP ]; then
    for i in `seq $QCMAP_PRIMARY_AP $total_ap_number`
    do
      RestartHostAPD $i
    done
  #Restart Hostapd for Primary_AP/Guest_AP/Guest_AP2/Guest_AP3
  else
    RestartHostAPD $ap_type
  fi
}

function activate_hostapd_config() {
  local ap_type=$1
  local action_type=$2
  local wlan_mode=$(util_execute_uci get qcmap_wlan.@wlanconfig[0].mode)

  log $(basename "$0") "activate_hostapd_config" $LINENO "activate hostapd config begin"

  util_execute_uci set qcmap_lan.@runstatus[0].activatehostapd='0'
  util_execute_uci commit qcmap_lan

  #check wlan is enable or disable
  get_wlan_disable_status
  if [ $wlan_disable_status -ne 0 ]; then
    log $(basename "$0") "activate_hostapd_config" $LINENO "Cannot activate hostapd when WLAN is disabled"
    return
  fi

  if [ $wlan_mode -eq $QCMAP_WLAN_MODE_STA_ONLY ]; then
    log $(basename "$0") "activate_hostapd_config" $LINENO "Cannot activate hostapd when WLAN is in sta only mode"
    return
  fi

  if [ $ap_type -lt $QCMAP_PRIMARY_AP ] || [ $ap_type -gt $QCMAP_ALL_AP ]; then
    log $(basename "$0") "activate_hostapd_config" $LINENO "Cannot activate hostapd incorrect ap_type: $ap_type"
    return
  fi
  if [ $action_type -lt $QCMAP_HOSTAPD_START ] || [ $action_type -gt $QCMAP_HOSTAPD_RESTART ]; then
    log $(basename "$0") "activate_hostapd_config" $LINENO "Cannot activate hostapd incorrect action_type: $action_type"
    return
  fi

  #Can not activate hostapd when STA_ONLY and ap_type is ALL_AP enabled
  if [ \( $ap_type -eq $QCMAP_ALL_AP \) ] && \
     [    \( $wlan_mode -eq $QCMAP_WLAN_MODE_STA_ONLY \) \
      -o \( $wlan_mode -eq $QCMAP_WLAN_MODE_STA_ONLY_BRIDGE \) ]; then
    log $(basename "$0") "activate_hostapd_config" $LINENO "Cannot activate hostapd. ap_type: $ap_type, wlan_mode:$wlan_mode"
    return
  fi

  #Cannot activate hostapd when /AP-AP/AP-AP-STA/AP-AP-AP/AP-AP-AP-AP is not enabled
  if [    \( $ap_type -eq $QCMAP_GUEST_AP \) ] && \
     [   \( $wlan_mode -ne $QCMAP_WLAN_MODE_AP_AP \) \
      -a \( $wlan_mode -ne $QCMAP_WLAN_MODE_AP_AP_STA \) \
      -a \( $wlan_mode -ne $QCMAP_WLAN_MODE_AP_AP_STA_BRIDGE \) \
      -a \( $wlan_mode -ne $QCMAP_WLAN_MODE_AP_AP_AP \) \
      -a \( $wlan_mode -ne $QCMAP_WLAN_MODE_AP_AP_AP_AP \) ]; then
    log $(basename "$0") "activate_hostapd_config" $LINENO "Cannot activate hostapd. ap_type: $ap_type, wlan_mode:$wlan_mode"
    return
  fi
  if [ $ap_type -eq $QCMAP_GUEST_AP_2 ] && \
     [ \( $wlan_mode -ne $QCMAP_WLAN_MODE_AP_AP_AP \) -a \( $wlan_mode -ne $QCMAP_WLAN_MODE_AP_AP_AP_AP \) ]; then
    log $(basename "$0") "activate_hostapd_config" $LINENO "Cannot activate hostapd. ap_type: $ap_type, wlan_mode:$wlan_mode"
    return
  fi
  if [ $ap_type -eq $QCMAP_GUEST_AP_3 ] && [ $wlan_mode -ne $QCMAP_WLAN_MODE_AP_AP_AP_AP ]; then
    log $(basename "$0") "activate_hostapd_config" $LINENO "Cannot activate hostapd. ap_type: $ap_type, wlan_mode:$wlan_mode"
    return
  fi

  #activate hostapd
  if [ $action_type -eq 1 ]; then
    activate_hostapd_action_start $ap_type
  elif [ $action_type -eq 2 ]; then
    activate_hostapd_action_stop $ap_type
  elif [ $action_type -eq 3 ]; then
    activate_hostapd_action_restart $ap_type
  fi

  #qcmap client will get the echo command result
  if [ $hostapd_activate_status -eq 1 ]; then
    util_execute_uci set qcmap_lan.@runstatus[0].activatehostapd='1'
    util_execute_uci commit qcmap_lan
    echo "Activate Hostapd Success"
  fi

  log $(basename "$0") "activate_hostapd_config" $LINENO "Activate Hostapd end"
}

function activate_supplicant_config() {
  local wlan_mode=$(util_execute_uci get qcmap_wlan.@wlanconfig[0].mode)
  local supplicant_pid_file=/data/vendor/wifi/wpa_supplicant-wlan0.pid
  local supplicant_pid=0
  local old_supplicant_pid=0


  log $(basename "$0") "activate_supplicant_config" $LINENO "activate supplicant config begin"
  util_execute_uci set qcmap_lan.@runstatus[0].activatesupplicant='0'
  util_execute_uci commit qcmap_lan

  #check wlan is enable or disable
  get_wlan_disable_status
  if [ $wlan_disable_status -ne 0 ]; then
    log $(basename "$0") "activate_supplicant_config" $LINENO "Cannot activate supplicant when WLAN is disabled"
    return
  fi

  if [ $wlan_mode -ne $QCMAP_WLAN_MODE_STA_ONLY ] && [ $wlan_mode -ne $QCMAP_WLAN_MODE_AP_STA ] && [ $wlan_mode -ne $QCMAP_WLAN_MODE_AP_AP_STA ]; then
    log $(basename "$0") "activate_supplicant_config" $LINENO "WLAN is in incorrect Mode: $wlan_mode"
    return
  fi

  #Get the wpa_supplicant pid before restart
  if [ -f $supplicant_pid_file ]; then
    supplicant_pid=`cat $supplicant_pid_file`
    log $(basename "$0") "activate_supplicant_config" $LINENO "wlan_mode: $wlan_mode, wpa supplicant pid: $supplicant_pid"
  fi

  old_supplicant_pid=$supplicant_pid

  wifi multi_down wifi0 wlan0 >> /dev/null
  #sleep 1 second to bring down iface.
  sleep 1

  wifi multi_up wifi0 wlan0 >> /dev/null
  #sleep 1 second wait pid file update
  sleep 1

  #Get the wpa_supplicant pid after restart
  if [ -f $supplicant_pid_file ]; then
    supplicant_pid=`cat $supplicant_pid_file`
    log $(basename "$0") "activate_supplicant_config" $LINENO "wlan_mode: $wlan_mode, restart wpa supplicant pid: $supplicant_pid"
  fi

  if [ $old_supplicant_pid -eq $supplicant_pid ] ; then
    log $(basename "$0") "activate_supplicant_config" $LINENO "Retart wpa supplicant fail, ap_type:$ap_type, supplicant_pid:$supplicant_pid"
    return
  fi

  util_execute_uci set qcmap_lan.@runstatus[0].activatesupplicant='1'
  util_execute_uci commit qcmap_lan
  #Restart wpa_cli
  start_wpa_cli_service

  #qcmap client will get the echo command result
  echo "Activate wpa Supplicant Success"

  log $(basename "$0") "activate_supplicant_config" $LINENO "activate supplicant config end"
}

function restart_tethered_wlan_client() {
  activate_hostapd_action_restart $QCMAP_ALL_AP

  log $(basename "$0") "restart_tethered_wlan_client" $LINENO "restart hostapd and start hostapd_cli process after ippt enable/disable"
}

function get_active_wlan_info() {
  #Get active wlan device info, including ifname, AP type, card type and enable state
  local wlan_mode=$(util_execute_uci get qcmap_wlan.@wlanconfig[0].mode)
  #Store AP type enabled state
  local primary_ap_enabled=0
  local guest_ap_enabled=0
  local guest_ap2_enabled=0
  local guest_ap3_enabled=0
  local sta_enabled=0
  local supplicant_pid_file=/data/vendor/wifi/wpa_supplicant-wlan0.pid

  #reset AP/STA state 0 before getting actual value
  uci set qcmap_wlan.@primaryap[0].state=0
  uci set qcmap_wlan.@guestap[0].state=0
  uci set qcmap_wlan.@guestaptwo[0].state=0
  uci set qcmap_wlan.@guestapthree[0].state=0
  uci set qcmap_wlan.@stamodeconfig[0].state=0
  uci commit

  get_active_ap_pid

  if [ $ap_pid -ne 0 ]; then
    uci set qcmap_wlan.@primaryap[0].state=1
  fi
  if [ $guest_ap_pid -ne 0 ]; then
    uci set qcmap_wlan.@guestap[0].state=1
  fi
  if [ $guest_ap2_pid -ne 0 ]; then
    uci set qcmap_wlan.@guestaptwo[0].state=1
  fi
  if [ $guest_ap3_pid -ne 0 ]; then
    uci set qcmap_wlan.@guestapthree[0].state=1
  fi
  if [ -f $supplicant_pid_file ]; then
    uci set qcmap_wlan.@stamodeconfig[0].state=1
  fi
  uci commit
}

function get_active_mld_wlan_info() {
  local hostapd_mld_pid_file=/data/vendor/wifi/hostapd-global.pid
  local wlanconfigex=$(util_execute_uci get qcmap_wlan.@wlanconfig[0].wlanconfigex)
  local mld_ap_pid=0

  if [ $wlanconfigex -eq 1 ] && [ -f $hostapd_mld_pid_file ]; then
    mld_ap_pid=`cat $hostapd_mld_pid_file`
    log $(basename "$0") "get_active_ap_pid" $LINENO "MLD AP PID: $mld_ap_pid"

    if [ $mld_ap_pid -ne 0 ]; then
      util_execute_uci set qcmap_wlan.@mldapconfig[0].state=1
      util_execute_uci commit qcmap_wlan
    fi
  fi
}

function get_wlan_band_enable_info() {
  #Get SAP and STA band enable info
  local wlan_mode=$(util_execute_uci get qcmap_wlan.@wlanconfig[0].mode)
  local ap_start_number=0
  local single_sap_state=""
  local wpa_state=""
  local sap_5G_freq=0
  local sap_2G_freq=0
  local sta_5G_freq=0
  local sta_2G_freq=0
  local wlan_AP_5G_enabled=0
  local wlan_AP_2G_enabled=0
  local wlan_STA_5G_enabled=0
  local wlan_STA_2G_enabled=0
  get_total_ap_number

  if [ $wlan_mode == $QCMAP_WLAN_MODE_AP_STA ] ||  [ $wlan_mode == $QCMAP_WLAN_MODE_STA_ONLY ] || [ $wlan_mode == $QCMAP_WLAN_MODE_AP_AP_STA ] ; then
    wpa_state=`wpa_cli status | grep -i wpa_state | awk -F '=' '{print $2}'`

    if [ "$wpa_state" == "SCANNING" ]; then
      log  $(basename "$0") "get_wlan_band_enable_info" $LINENO "sta enabled but not connect to router"
    elif [ "$wpa_state" == "COMPLETED" ]; then
      sta_5G_freq=`wpa_cli status | grep -i 'freq=5' | awk -F '=' '{print $2}'`
      sta_2G_freq=`wpa_cli status | grep -i 'freq=2' | awk -F '=' '{print $2}'`

      if [ "$sta_5G_freq" != "" ] && [ $sta_5G_freq -ne 0 ]; then
        wlan_STA_5G_enabled=1;
      elif [ "$sta_2G_freq" != "" ] && [ $sta_2G_freq -ne 0 ]; then
        wlan_STA_2G_enabled=1;
      fi

      log  $(basename "$0") "get_wlan_band_enable_info" $LINENO "sta_5G_freq:$sta_5G_freq, sta_2G_freq:$sta_2G_freq"
    fi

    ap_start_number=1
  else
    total_ap_number=$((total_ap_number-1))
  fi

  for i in $(seq $ap_start_number $total_ap_number)
  do
    single_sap_state=""
    single_sap_state=`hostapd_cli -i wlan$i status | grep -i state | awk -F '=' '{print $2}'`

    if [ \( "$single_sap_state" == "ENABLED" \) -o \( "$single_sap_state" == "DFS" \) ]; then
      sap_5G_freq=`hostapd_cli -i wlan$i status | grep -i 'freq=5' | awk -F '=' '{print $2}'`
      sap_2G_freq=`hostapd_cli -i wlan$i status | grep -i 'freq=2' | awk -F '=' '{print $2}'`

      if [ "$sap_5G_freq" != "" ] && [ $sap_5G_freq -ne 0 ]; then
        wlan_AP_5G_enabled=1;
        log  $(basename "$0") "get_wlan_band_enable_info" $LINENO "wlan$i, sap_5G_freq:$sap_5G_freq"
      elif [ "$sap_2G_freq" != "" ] && [ $sap_2G_freq -ne 0 ]; then
        wlan_AP_2G_enabled=1;
        log  $(basename "$0") "get_wlan_band_enable_info" $LINENO "wlan$i, sap_2G_freq:$sap_2G_freq"
      fi
    else
      log  $(basename "$0") "get_wlan_band_enable_info" $LINENO "wlan$i not enabled"
    fi
  done

  echo "$wlan_AP_5G_enabled $wlan_AP_2G_enabled $wlan_STA_5G_enabled $wlan_STA_2G_enabled"
}

function switch_wlan_enable_band() {
  #$1, 1:sap enable 5g, 0:sap disable 5g
  #$2, 1:sta enable 5g, 0:sta disable 5g
  local wlan_mode=$(util_execute_uci get qcmap_wlan.@wlanconfig[0].mode)
  local ap_start_number=0
  local single_sap_state=""
  local wpa_state=""
  local sap_5G_freq=0
  local sap_2G_freq=0
  local sta_5G_freq=0
  local sta_2G_freq=0
  local sap_5G_enable_status=$1
  local sta_5G_enable_status=$2
  get_total_ap_number

  if [ $wlan_mode == $QCMAP_WLAN_MODE_AP_STA ] ||  [ $wlan_mode == $QCMAP_WLAN_MODE_STA_ONLY ] || [ $wlan_mode == $QCMAP_WLAN_MODE_AP_AP_STA ] ; then
    wpa_state=`wpa_cli status | grep -i wpa_state | awk -F '=' '{print $2}'`

    if [ "$wpa_state" == "SCANNING" ]; then
      log  $(basename "$0") "switch_wlan_enable_band" $LINENO "sta enabled but not connect to router"

      if [ $sta_5G_enable_status == $WLAN_5G_ENABLE ]; then
        wpa_cli -i wlan0 SET setband AUTO
        log  $(basename "$0") "switch_wlan_enable_band" $LINENO "wpa_state=SCANNING, set wlan0 to 5G band"
      fi
    elif [ "$wpa_state" == "COMPLETED" ]; then
      sta_5G_freq=`wpa_cli status | grep -i 'freq=5' | awk -F '=' '{print $2}'`
      sta_2G_freq=`wpa_cli status | grep -i 'freq=2' | awk -F '=' '{print $2}'`

      if [ $sta_5G_enable_status == $WLAN_5G_ENABLE ] && [ "$sta_5G_freq" == "" ]; then
        wpa_cli -i wlan0 SET setband AUTO
        log  $(basename "$0") "switch_wlan_enable_band" $LINENO "set wlan0 to 5G success"
      elif [ $sta_5G_enable_status == $WLAN_5G_DISABLE ] && [ "$sta_2G_freq" == "" ] ; then
        wpa_cli -i wlan0 SET setband 2G
        log  $(basename "$0") "switch_wlan_enable_band" $LINENO "set wlan0 to 2G success"
      fi
    fi

    ap_start_number=1
  else
    total_ap_number=$((total_ap_number-1))
  fi

  for i in $(seq $ap_start_number $total_ap_number)
  do
    if [ $sap_5G_enable_status == $WLAN_5G_ENABLE ] ; then
      #restart wlan$i if function disable_wlan_5g_ap() already disable wlan$i
      single_sap_state=""
      single_sap_state=`hostapd_cli -i wlan$i status | grep -i state | awk -F '=' '{print $2}'`

      if [ \( "$single_sap_state" != "ENABLED" \) -a \( "$single_sap_state" != "DFS" \) ]; then
        if [ $wlan_mode == $QCMAP_WLAN_MODE_AP_AP_AP_AP ] ; then
          hostapd_cli -i wlan$((i % total_ap_number + 2)) SET setband AUTO
          #sleep 2 second, so that host driver can operate complete after receiving this command and allow wlan restart.
          sleep 2
        fi

        get_wlan_pdev $i
        wifi multi_up $wlan_pdev wlan$i >> /dev/null

        ifconfig wlan$i up
        #sleep 2s wait wlan$i start success
        sleep 2

        #restart hostapd_cli service
        /usr/sbin/hostapd_cli -i wlan$i -s /data/vendor/wifi -p /data/vendor/wifi/hostapd -B -a /usr/bin/QCMAP_StaInterface

        log  $(basename "$0") "switch_wlan_enable_band" $LINENO "HostApd is terminated for wlan$i, Re-starting it"
      fi
    fi

    #switch wlan$i band if wlan$i enabled
    single_sap_state=""
    single_sap_state=`hostapd_cli -i wlan$i status | grep -i state | awk -F '=' '{print $2}'`
    if [ \( "$single_sap_state" == "ENABLED" \) -o \( "$single_sap_state" == "DFS" \) ]; then
      sap_5G_freq=`hostapd_cli -i wlan$i status | grep -i 'freq=5' | awk -F '=' '{print $2}'`
      sap_2G_freq=`hostapd_cli -i wlan$i status | grep -i 'freq=2' | awk -F '=' '{print $2}'`

      if [ $sap_5G_enable_status == $WLAN_5G_ENABLE ] && [ "$sap_5G_freq" == "" ]; then
        hostapd_cli -i wlan$i SET setband AUTO
        log  $(basename "$0") "switch_wlan_enable_band" $LINENO "set wlan$i to 5G success"
      elif [ $sap_5G_enable_status == $WLAN_5G_DISABLE ] && [ "$sap_2G_freq" == "" ]; then
        hostapd_cli -i wlan$i SET setband 2G
        log  $(basename "$0") "switch_wlan_enable_band" $LINENO "set wlan$i to 2G success"
      fi
    fi
  done

  log  $(basename "$0") "switch_wlan_enable_band" $LINENO "enable status, sap:$sap_5G_enable_status, sta:$sta_5G_enable_status"
}

function disable_wlan_5g_ap() {
  #while switch 5g to 2g, disable 5g AP if 2g band is full
  local wlan_mode=$(util_execute_uci get qcmap_wlan.@wlanconfig[0].mode)
  local ap_start_number=0
  local single_sap_state=""
  local sap_5G_freq=0
  local sap_2G_freq=0
  get_total_ap_number

  if [ $wlan_mode == $QCMAP_WLAN_MODE_AP_STA ] ||  [ $wlan_mode == $QCMAP_WLAN_MODE_STA_ONLY ] || [ $wlan_mode == $QCMAP_WLAN_MODE_AP_AP_STA ] ; then
    ap_start_number=1
  else
    total_ap_number=$((total_ap_number-1))
  fi

  for i in $(seq $ap_start_number $total_ap_number)
  do
    single_sap_state=""
    single_sap_state=`hostapd_cli -i wlan$i status | grep -i state | awk -F '=' '{print $2}'`

    if [ \( "$single_sap_state" == "ENABLED" \) -o \( "$single_sap_state" == "DFS" \) ]; then
      sap_5G_freq=`hostapd_cli -i wlan$i status | grep -i 'freq=5' | awk -F '=' '{print $2}'`
      sap_2G_freq=`hostapd_cli -i wlan$i status | grep -i 'freq=2' | awk -F '=' '{print $2}'`

      if [ "$sap_5G_freq" != "" ] && [ $sap_5G_freq -ne 0 ]; then
        get_wlan_pdev $i

        ifconfig wlan$i down
        wifi multi_down $wlan_pdev wlan$i >> /dev/null
        log  $(basename "$0") "disable_wlan_5g_ap" $LINENO "2g band is full, disable wlan$i"
      elif [ "$sap_2G_freq" != "" ] && [ $sap_2G_freq -ne 0 ]; then
        log  $(basename "$0") "disable_wlan_5g_ap" $LINENO "wlan$i already switch to 2G."
      fi
    else
      log  $(basename "$0") "disable_wlan_5g_ap" $LINENO "wlan$i disabled"
    fi
  done
}

function set_mld_ap_config_len() {
  local apcount2g=$1
  local apcount5g=$2
  local apcount5g_low=$3
  local apcount5g_high=$4
  local apcount6g=$5
  local totalapcount=$6
  local totalmldapcount=$7

  util_execute_uci_set qcmap_wlan_current.@wlanconfig[0].apcount2g "$apcount2g"
  util_execute_uci_set qcmap_wlan_current.@wlanconfig[0].apcount5g "$apcount5g"
  util_execute_uci_set qcmap_wlan_current.@wlanconfig[0].apcount5g_low "$apcount5g_low"
  util_execute_uci_set qcmap_wlan_current.@wlanconfig[0].apcount5g_high "$apcount5g_high"
  util_execute_uci_set qcmap_wlan_current.@wlanconfig[0].apcount6g "$apcount6g"
  util_execute_uci_set qcmap_wlan_current.@wlanconfig[0].totalapcount "$totalapcount"
  util_execute_uci_set qcmap_wlan_current.@mldwlanconfig[0].totalmldapcount "$totalmldapcount"
  uci commit
}

case $1 in
  set_wlan_mode)
    log  $(basename "$0") "case:set_wlan_mode" $LINENO "set_wlan_mode"
    set_wlan_mode $2
    ;;

  set_mld_wlan_mode)
    log  $(basename "$0") "case:set_mld_wlan_mode" $LINENO "set_mld_wlan_mode"
    set_mld_wlan_mode $2
    ;;

  set_mld_ap_config)
    log  $(basename "$0") "case:set_mld_ap_config" $LINENO "set_mld_ap_config"
    set_mld_ap_config $2 $3 $4
    ;;

  set_primary_config)
    log  $(basename "$0") "case:set_primary_config" $LINENO "set_primary_config"
    set_primary_config $2
    ;;

  set_guest_config)
    log  $(basename "$0") "case:set_guest_config" $LINENO "set_guest_config"
    set_guest_config $2 $3 $4 $5 $6 $7
    ;;

  set_sta_config)
    log  $(basename "$0") "case:set_sta_config" $LINENO "set_sta_config"
    set_sta_config $2 $3 $4 $5 $6 $7 $8
    ;;

  update_wireless_network)
    log  $(basename "$0") "case:update_wireless_network" $LINENO "update_wireless_network"
    update_wireless_network
    ;;

  update_wireless_network_ex)
    log  $(basename "$0") "case:update_wireless_network_ex" $LINENO "update_wireless_network_ex"
    update_wireless_network_ex
    ;;

  is_wifi_up)
    log  $(basename "$0") "case:is_wifi_up" $LINENO "is_wifi_up"
    is_wifi_up
    ;;

  configure_mwan3)
    log  $(basename "$0") "case:configure_mwan3" $LINENO "configure_mwan3_hmt"
    configure_mwan3 $2
    ;;

  enable_wlan)
    log  $(basename "$0") "case:enable_wlan" $LINENO "enable_wlan"
    echo QCMAP:Start WLAN Enable > /dev/kmsg
    enable_wlan_hmt
    ;;

  disable_wlan)
    log  $(basename "$0") "case:disable_wlan" $LINENO "disable_wlan"
    echo QCMAP:Disable WLAN > /dev/kmsg
    disable_wlan_hmt
    ;;

  reset_wlan_atbootup)
    log  $(basename "$0") "case:reset_wlan_atbootup" $LINENO "reset_wlan_atbootup"
    reset_wlan_atbootup_hmt
    ;;

  get_sap_status)
    log  $(basename "$0") "case:get_sap_status" $LINENO "get_sap_status"
    get_sap_status
    ;;

  get_supplicant_status)
    log  $(basename "$0") "case:get_supplicant_status" $LINENO "get_supplicant_status"
    get_supplicant_status_hmt
    ;;

  install_guest_ap_access_rules)
    log  $(basename "$0") "case:install_guest_ap_access_rules" $LINENO "install_guest_ap_access_rules"
    install_guest_ap_access_rules
    ;;

  delete_guest_ap_access_rules)
    log  $(basename "$0") "case:delete_guest_ap_access_rules" $LINENO "delete_guest_ap_access_rules"
    delete_guest_ap_access_rules
    ;;

  set_wlan_bootup_enable)
    log  $(basename "$0") "case:set_wlan_bootup_enable" $LINENO "set_wlan_bootup_enable"
    set_wlan_bootup_enable $2
    ;;

  get_sta_status)
    log  $(basename "$0") "case:get_sta_status" $LINENO "get_sta_status"
    get_sta_status wlan0
    ;;

  activate_hostapd_config)
    log  $(basename "$0") "case:activate_hostapd_config" $LINENO "activate_hostapd_config"
    activate_hostapd_config $2 $3
    ;;

  activate_supplicant_config)
    log  $(basename "$0") "case:activate_supplicant_config" $LINENO "activate_supplicant_config"
    activate_supplicant_config
    ;;

  get_active_wlan_info)
    log  $(basename "$0") "case:get_active_wlan_info" $LINENO "get_active_wlan_info"
    get_active_wlan_info
    ;;

  get_active_mld_wlan_info)
    log  $(basename "$0") "case:get_active_mld_wlan_info" $LINENO "get_active_mld_wlan_info"
    get_active_mld_wlan_info
    ;;

  get_wlan_band_enable_info)
    log  $(basename "$0") "case:get_wlan_band_enable_info" $LINENO "get_wlan_band_enable_info"
    get_wlan_band_enable_info
    ;;

  switch_wlan_enable_band)
    log  $(basename "$0") "case:switch_wlan_enable_band" $LINENO "switch_wlan_enable_band"
    switch_wlan_enable_band $2 $3
    ;;

  disable_wlan_5g_ap)
    log  $(basename "$0") "case:disable_wlan_5g_ap" $LINENO "disable_wlan_5g_ap"
    disable_wlan_5g_ap
    ;;

  get_wlan_iface_name)
    log  $(basename "$0") "case:get_Iface_name" $LINENO "get_Iface_name"
    get_wlan_iface_name $2
    ;;

  start_hostapd_cli_service_sta)
    log  $(basename "$0") "case:start_hostapd_cli_service_sta" $LINENO "start_hostapd_cli_service_sta"
    start_hostapd_cli_service_sta
    ;;

  restart_tethered_wlan_client)
    log  $(basename "$0") "case:restart_tethered_wlan_client" $LINENO "restart_tethered_wlan_client"
    restart_tethered_wlan_client
    ;;

  add_delete_rules_for_bridge_mode)
    log  $(basename "$0") "case: add_delete_rules_for_bridge_mode" $LINENO "add_delete_rules_for_bridge_mode"
    add_delete_rules_for_bridge_mode $2
    ;;

  start_wpa_cli_service)
    log  $(basename "$0") "case:start_wpa_cli_service" $LINENO "start_wpa_cli_service"
    start_wpa_cli_service
    ;;

  ap_sta_mode_connect)
    log  $(basename "$0") "case:ap_sta_mode_connect" $LINENO "ap_sta_mode_connect"
    ap_sta_mode_connect $2
    ;;

  ap_sta_mode_disconnect)
    log  $(basename "$0") "case:ap_sta_mode_disconnect" $LINENO "ap_sta_mode_disconnect"
    ap_sta_mode_disconnect $2
    ;;

  get_sta_associcated_status)
    log  $(basename "$0") "case:get_sta_associcated_status" $LINENO "get_sta_associcated_status"
    get_sta_associcated_status
    ;;

  set_mld_ap_config_len)
    log  $(basename "$0") "case:set_mld_ap_config_len" $LINENO "set_mld_ap_config_len"
    set_mld_ap_config_len $2 $3 $4 $5 $6 $7 $8
    ;;

  *)
    log  $(basename "$0") "case: Invalid" $LINENO "Invalid option"
    ;;

esac
