#!/bin/sh
#  Copyright (c) 2023 Qualcomm Technologies, Inc.
#  All Rights Reserved.
#  Confidential and Proprietary - Qualcomm Technologies, Inc.

#source log file
# Usage to use log function
# log  $(basename "$0") $LINENO "Log message"
. /etc/data/mbbUtils.sh

#source common functions

#QCMAP Server have no permission to on /lib/functions.sh which include by lanUtils.sh
#. /etc/data/lanUtils.sh

#Function to execute uci set cmd for string value
#$1 - uci options
#$2 - uci value
function util_execute_uci_set() {
  uci set "$1=$2"
  log $(basename "$0") "util_execute_uci" $LINENO " uci set $1=$2"
}

#Function to execute uci cmd and print to log
#$1 - uci cmd
function util_execute_uci() {
  uci $*
  log  $(basename "$0") "util_execute_uci" $LINENO " uci $*"
}

#function to get fetch band info corresponding to athX iface
function util_get_band_info_from_ath_iface() {
  local link=$1
  local link_index=$(echo $link | cut -c 4)
  local wifi_iface="wifi$link_index"
  local wifi_iface_band=$(cfg80211tool $wifi_iface g_oper_reg_info | awk '{ print $4 }' | cut -c 6)

  log $(basename "$0") "util_get_band_info_from_ath_iface" $LINENO "link:$link link_index=$link_index wifi_iface:$wifi_iface wifi_iface_band=$wifi_iface_band"

  echo $wifi_iface_band
}

#function to get index of all wifi-iface config having band passed
#$1 band info
#$2 is the index of config corresponding to mldX iface
function util_get_wifi_iface_count_from_wifi_config()
{
  local band=$1
  local index=$2
  local index_string=""
  local count=0

  #get all wifi index
  wifi_iface_idx_list=$(uci show wireless | grep -i wireless.@wifi-iface | grep -i =wifi-iface | awk -F'[][]' '{print $2}')

  for i in $wifi_iface_idx_list
  do
    wifi=$(uci get wireless.@wifi-iface[$i].device)
    wifi_band=$(cfg80211tool $wifi g_oper_reg_info | awk '{ print $4 }' | cut -c 6)
    if [ $wifi_band -eq $band ]; then
      index_string="$index_string$i "
    fi
  done

  for j in $index_string;
  do
    if [ $j -ne $index ];then
      count=$((count+1))
    else
      echo $count
      return
    fi
  done
}

eval static_ip_wlan0="169.254.24.1"
eval static_ip_wlan1="169.254.25.1"
eval static_ip_wlan2="169.254.26.1"
eval static_ip_wlan3="169.254.27.1"
ip_octet=28
pdev_number=0
QCMAP_WLAN_MODE_AP=1
QCMAP_WLAN_MODE_AP_AP=2
QCMAP_WLAN_MODE_AP_STA=3
QCMAP_WLAN_MODE_AP_AP_AP=4
QCMAP_WLAN_MODE_AP_AP_STA=5
QCMAP_WLAN_MODE_STA_ONLY=6
QCMAP_WLAN_MODE_AP_AP_AP_AP=7
QCMAP_WLAN_MODE_AP_STA_BRIDGE=8
QCMAP_WLAN_MODE_AP_AP_STA_BRIDGE=9
QCMAP_WLAN_MODE_STA_ONLY_BRIDGE=10
QCMAP_WLAN_MODE_7AP=13
total_ap_number=0
total_ap_link=0
total_sta_link=0
prefer_primary_band=0
prefer_second_band=0
station_band=0
QCMAP_STA_CONNECTION_DYNAMIC=1
QCMAP_STA_CONNECTION_STATIC=2
QCMAP_STA_CONNECTED=1
QCMAP_STA_DISCONNECTED=2
INTERNET_ONLY=1
config_count=0
wifi0_athlist=""
wifi1_athlist=""
wifi2_athlist=""
mld_list=""
QCMAP_WLAN_2GHz_BAND=2
QCMAP_WLAN_5GHz_BAND=5
QCMAP_WLAN_6GHz_BAND=6
QCMAP_WLAN_5GHz_LOW_BAND=20
QCMAP_WLAN_5GHz_HIGH_BAND=21
#worst case 12s, one time 150ms
hostapd_start_poll_max_count=80
hostapd_poll_delay_S=0.15

#sap_state used to save every single sap status
sap_state=""
sap_link_state=""

#parameter need to be updated in function utility_update_guest_ap_public_ip
lan_ip=""
netmask=""
lipv4_public_ip=""
prefix_v6=""
prefix_len=0
ipv4_wan_connected=0
ipv6_wan_connected=0

#used to support activate hostapd
QCMAP_PRIMARY_AP=1
QCMAP_GUEST_AP=2
QCMAP_GUEST_AP_2=3
QCMAP_GUEST_AP_3=4
QCMAP_ALL_AP=5
QCMAP_GUEST_AP_4=6
QCMAP_GUEST_AP_5=7
QCMAP_GUEST_AP_6=8
QCMAP_GUEST_AP_7=9
QCMAP_GUEST_AP_8=10
QCMAP_GUEST_AP_9=11
QCMAP_GUEST_AP_10=12
QCMAP_GUEST_AP_11=13
QCMAP_GUEST_AP_12=14
QCMAP_GUEST_AP_13=15
QCMAP_GUEST_AP_14=16
QCMAP_GUEST_AP_15=17
QCMAP_GUEST_AP_16=18
QCMAP_GUEST_AP_17=19
QCMAP_GUEST_AP_18=20
QCMAP_GUEST_AP_19=21
QCMAP_GUEST_AP_20=22

#QCMAP MLD AP type
QCMAP_MLD_AP_1=1
QCMAP_MLD_AP_2=2
QCMAP_MLD_AP_3=3
QCMAP_MLD_AP_4=4
QCMAP_MLD_AP_5=5
QCMAP_MLD_AP_6=6
QCMAP_MLD_AP_7=7
QCMAP_MLD_AP_8=8
QCMAP_MLD_AP_9=9
QCMAP_MLD_AP_10=10
QCMAP_MLD_AP_11=11
QCMAP_MLD_AP_12=12
QCMAP_MLD_AP_13=13
QCMAP_MLD_AP_14=14
QCMAP_MLD_AP_15=15
QCMAP_MLD_AP_16=16
QCMAP_MLD_AP_17=17
QCMAP_MLD_AP_18=18
QCMAP_MLD_AP_19=19
QCMAP_MLD_AP_20=20
QCMAP_MLD_AP_21=21
QCMAP_MLD_AP_ALL=22

#Hostapd action type
QCMAP_HOSTAPD_START=1
QCMAP_HOSTAPD_STOP=2
QCMAP_HOSTAPD_RESTART=3
ap_pid=0
guest_ap_pid=0
guest_ap2_pid=0
guest_ap3_pid=0
wlan_disable_status=1
hostapd_activate_status=1
activate_ap_number=0
wlan_pdev=wifi0

#check sap/sta 5g enable status
WLAN_5G_ENABLE=1
WLAN_5G_DISABLE=0

QCMAP_MSGR_DEVICE_TYPE_PRIMARY_AP=1
QCMAP_MSGR_DEVICE_TYPE_GUEST_AP=2
QCMAP_MSGR_DEVICE_TYPE_GUEST_AP_2=5
QCMAP_MSGR_DEVICE_TYPE_GUEST_AP_3=6

CDT_WLAN_WKK_ATTACH=3
CDT_WLAN_HMT_ATTACH=2
CDT_RCM=0
CDT_LGA=5
CDT_TYPE=$(cat /sys/devices/soc0/platform_subtype_id)
PLATFORM_VERSION=$(cat /sys/devices/soc0/platform_version)

if [ $CDT_LGA -eq $CDT_TYPE ]; then
  if [ $((${PLATFORM_VERSION}>>16)) -eq 2 ]; then
    CDT_TYPE=$CDT_WLAN_WKK_ATTACH
  elif [ $((${PLATFORM_VERSION}>>16)) -eq 1 ]; then
    CDT_TYPE=$CDT_WLAN_HMT_ATTACH
  fi
elif [ $CDT_RCM -eq $CDT_TYPE ]; then
  CDT_TYPE=$CDT_WLAN_HMT_ATTACH
fi

QCMAP_WLAN_MAX_AP_IDX_COUNT=32

#WlAN IFACE Info IDX
QCMAP_MSGR_WLAN_IFACE_PRIMARY_AP=0
QCMAP_MSGR_WLAN_IFACE_GUEST_AP_ONE=1
QCMAP_MSGR_WLAN_IFACE_GUEST_AP_TWO=2
QCMAP_MSGR_WLAN_IFACE_GUEST_AP_THREE=4
QCMAP_MSGR_WLAN_IFACE_GUEST_AP_FOUR=5
QCMAP_MSGR_WLAN_IFACE_GUEST_AP_FIVE=6
QCMAP_MSGR_WLAN_IFACE_GUEST_AP_SIX=7
QCMAP_MSGR_WLAN_IFACE_GUEST_AP_SEVEN=8
QCMAP_MSGR_WLAN_IFACE_GUEST_AP_EIGHT=9
QCMAP_MSGR_WLAN_IFACE_GUEST_AP_NINE=10
QCMAP_MSGR_WLAN_IFACE_GUEST_AP_TEN=11
QCMAP_MSGR_WLAN_IFACE_GUEST_AP_ELEVEN=12
QCMAP_MSGR_WLAN_IFACE_GUEST_AP_TWELVE=13
QCMAP_MSGR_WLAN_IFACE_GUEST_AP_THIRTEEN=14
QCMAP_MSGR_WLAN_IFACE_GUEST_AP_FOURTEEN=17
QCMAP_MSGR_WLAN_IFACE_GUEST_AP_FIFTEEN=18
QCMAP_MSGR_WLAN_IFACE_GUEST_AP_SIXTEEN=19
QCMAP_MSGR_WLAN_IFACE_GUEST_AP_SEVENTEEN=20
QCMAP_MSGR_WLAN_IFACE_GUEST_AP_EIGHTTEEN=21
QCMAP_MSGR_WLAN_IFACE_GUEST_AP_NINETEEN=22
QCMAP_MSGR_WLAN_IFACE_GUEST_AP_TWENTY=23
QCMAP_MSGR_WLAN_IFACE_MLD_AP_ONE=25
QCMAP_MSGR_WLAN_IFACE_MLD_AP_TWO=26
QCMAP_MSGR_WLAN_IFACE_MLD_AP_THREE=27
QCMAP_MSGR_WLAN_IFACE_MLD_AP_FOUR=28
QCMAP_MSGR_WLAN_IFACE_MLD_AP_FIVE=29
QCMAP_MSGR_WLAN_IFACE_MLD_AP_SIX=30
QCMAP_MSGR_WLAN_IFACE_MLD_AP_SEVEN=31

#Function to get wlan config index from wlan type
#$1 - wlan_ap_type
function util_convert_wlan_ap_type_to_index() {
  local wlan_ap_type="$1"

  case "$wlan_ap_type" in
    $QCMAP_MSGR_WLAN_IFACE_PRIMARY_AP) echo '0'
    ;;
    $QCMAP_MSGR_WLAN_IFACE_GUEST_AP_ONE) echo '1'
    ;;
    $QCMAP_MSGR_WLAN_IFACE_GUEST_AP_TWO) echo '2'
    ;;
    $QCMAP_MSGR_WLAN_IFACE_GUEST_AP_THREE) echo '3'
    ;;
    $QCMAP_MSGR_WLAN_IFACE_GUEST_AP_FOUR) echo '4'
    ;;
    $QCMAP_MSGR_WLAN_IFACE_GUEST_AP_FIVE) echo '5'
    ;;
    $QCMAP_MSGR_WLAN_IFACE_GUEST_AP_SIX) echo '6'
    ;;
    $QCMAP_MSGR_WLAN_IFACE_GUEST_AP_SEVEN) echo '7'
    ;;
    $QCMAP_MSGR_WLAN_IFACE_GUEST_AP_EIGHT) echo '8'
    ;;
    $QCMAP_MSGR_WLAN_IFACE_GUEST_AP_NINE) echo '9'
    ;;
    $QCMAP_MSGR_WLAN_IFACE_GUEST_AP_TEN) echo '10'
    ;;
    $QCMAP_MSGR_WLAN_IFACE_GUEST_AP_ELEVEN) echo '11'
    ;;
    $QCMAP_MSGR_WLAN_IFACE_GUEST_AP_TWELVE) echo '12'
    ;;
    $QCMAP_MSGR_WLAN_IFACE_GUEST_AP_THIRTEEN) echo '13'
    ;;
    $QCMAP_MSGR_WLAN_IFACE_GUEST_AP_FOURTEEN) echo '14'
    ;;
    $QCMAP_MSGR_WLAN_IFACE_GUEST_AP_FIFTEEN) echo '15'
    ;;
    $QCMAP_MSGR_WLAN_IFACE_GUEST_AP_SIXTEEN) echo '16'
    ;;
    $QCMAP_MSGR_WLAN_IFACE_GUEST_AP_SEVENTEEN) echo '17'
    ;;
    $QCMAP_MSGR_WLAN_IFACE_GUEST_AP_EIGHTTEEN) echo '18'
    ;;
    $QCMAP_MSGR_WLAN_IFACE_GUEST_AP_NINETEEN) echo '19'
    ;;
    $QCMAP_MSGR_WLAN_IFACE_GUEST_AP_TWENTY) echo '20'
    ;;
    $QCMAP_MSGR_WLAN_IFACE_MLD_AP_ONE) echo '0'
    ;;
    $QCMAP_MSGR_WLAN_IFACE_MLD_AP_TWO) echo '1'
    ;;
    $QCMAP_MSGR_WLAN_IFACE_MLD_AP_THREE) echo '2'
    ;;
    $QCMAP_MSGR_WLAN_IFACE_MLD_AP_FOUR) echo '3'
    ;;
    $QCMAP_MSGR_WLAN_IFACE_MLD_AP_FIVE) echo '4'
    ;;
    $QCMAP_MSGR_WLAN_IFACE_MLD_AP_SIX) echo '5'
    ;;
    $QCMAP_MSGR_WLAN_IFACE_MLD_AP_SEVEN) echo '6'
    ;;
    *) echo '-1'
    ;;
  esac
}

function ap_sta_mode_connect()
{
  log $(basename "$0") "ap_sta_mode_connect" $LINENO "Enter ap_sta_mode_connect"
  local bh_switch_request=$1
  local bridge_mode=$(util_execute_uci get qcmap_wlan_current.@stamodeconfig[0].bridge_mode)
  local sta_mode_conn_type=$(util_execute_uci get qcmap_wlan.@stamodeconfig[0].sta_mode_conn_type)

  if [ $bridge_mode -eq 1 ]; then
    ap_sta_bridge_mode_connect
  else
    if [ $sta_mode_conn_type -eq $QCMAP_STA_CONNECTION_STATIC ]; then
      load_static_station_setting $bh_switch_request
    fi
  fi
}

function ap_sta_mode_disconnect()
{
  log $(basename "$0") "ap_sta_mode_disconnect" $LINENO "Enter ap_sta_mode_disconnect"
  local bh_switch_request=$1
  local bridge_iface=$(util_execute_uci get network.lan_v6.device)
  local bridge_mode=$(util_execute_uci get qcmap_wlan_current.@stamodeconfig[0].bridge_mode)
  local sta_mode_conn_type=$(util_execute_uci get qcmap_wlan.@stamodeconfig[0].sta_mode_conn_type)

  # if change wlan mode first, then activate wlan directly.
  # bridge mode config will not clean. here add bridge_iface condition
  if [ $bridge_mode -eq 1 ] || [ -n $bridge_iface ]; then
    ap_sta_bridge_mode_disconnect
    util_execute_uci set qcmap_wlan.@stamodeconfig[0].bridge_flag="0"
    util_execute_uci commit qcmap_wlan
  else
    if [ $sta_mode_conn_type -eq $QCMAP_STA_CONNECTION_STATIC ]; then
      unload_static_station_setting $bh_switch_request
    fi
  fi
}

function ap_sta_bridge_mode_connect(){

  /etc/init.d/firewall stop

  local sta_mode=$(util_execute_uci get qcmap_wlan.@stamodeconfig[0].sta_mode_conn_type)
  local bridge_iface=$(util_execute_uci get network.lan.device)
  local lease_file=$(util_execute_uci get dhcp.lan_dns.leasefile)
  local sta_iface=$(util_execute_uci get qcmap_wlan.@stamodeconfig[0].ifname)

  util_execute_uci set qcmap_wlan.@stamodeconfig[0].bridge_flag=$(util_execute_uci get qcmap_wlan.@stamodeconfig[0].bridge_mode)
  uci commit qcmap_wlan

  add_delete_rules_for_bridge_mode 1


  ap_sta_bridge_mac=`ifconfig $sta_iface | grep HWaddr | awk -F' ' '{print $5}'`
  if [ $sta_mode == $QCMAP_STA_CONNECTION_DYNAMIC ] ; then
    util_execute_uci delete network.lan_bind4

    util_execute_uci delete network.lan.ipaddr
    util_execute_uci delete network.lan.netmask
    util_execute_uci set network.lan.proto="dhcp"
    util_execute_uci set network.lan.macaddr="$ap_sta_bridge_mac"

    util_execute_uci delete network.wanwlan_v6
    util_execute_uci set network.lan_v6="interface"
    util_execute_uci set network.lan_v6.device=$bridge_iface
    util_execute_uci set network.lan_v6.proto="dhcpv6"
    util_execute_uci set network.lan_v6.ipv6="1"
    util_execute_uci set network.lan_v6.type="internet"

    rm -rf $lease_file

    util_execute_uci commit network
    ubus call network reload
  else
    local ip=$(util_execute_uci get qcmap_wlan.@stamodeconfig[0].ipaddr)
    local netmask=$(util_execute_uci get qcmap_wlan.@stamodeconfig[0].netmask)
    local dns=$(util_execute_uci get qcmap_wlan.@stamodeconfig[0].dnsserver)
    local gw=$(util_execute_uci get qcmap_wlan.@stamodeconfig[0].gateway)
    local resolv_file=$(util_execute_uci get dhcp.lan_dns.resolvfile)

    util_execute_uci delete network.wanwlan.dns

    util_execute_uci set network.lan.macaddr="$ap_sta_bridge_mac"
    util_execute_uci set network.lan.ipaddr=$ip
    util_execute_uci set network.lan.netmask=$netmask
    util_execute_uci set network.lan.dns=$dns

    util_execute_uci set network.lan_v6="interface"
    util_execute_uci set network.lan_v6.device=$bridge_iface
    util_execute_uci set network.lan_v6.proto="dhcpv6"
    util_execute_uci set network.lan_v6.ipv6="1"
    util_execute_uci set network.lan_v6.type="internet"

    util_execute_uci commit network
    ubus call network reload

    route add default gw $gw dev $bridge_iface

    /etc/data/dhcpRelay.sh enable_dhcp_proxy lan
  fi

  log  $(basename "$0") "ap_sta_bridge_mode_connect" $LINENO "station bridge mode setting done !!!"
}
function ap_sta_bridge_mode_disconnect(){

  local sta_mode=$(util_execute_uci get qcmap_wlan.@stamodeconfig[0].sta_mode_conn_type)
  /etc/init.d/firewall start

  util_execute_uci set network.lan.proto="static"
  util_execute_uci set network.lan.ipaddr="192.168.224.1"
  util_execute_uci set network.lan.netmask="255.255.254.0"
  util_execute_uci delete network.lan.dns
  util_execute_uci delete network.lan.macaddr

  util_execute_uci delete network.lan_v6

  util_execute_uci commit network
  ubus call network reload

  add_delete_rules_for_bridge_mode 0

  /etc/data/dhcpRelay.sh disable_dhcp_proxy

  log  $(basename "$0") "ap_sta_bridge_mode_disconnect" $LINENO "station bridge mode revert done !!!"
}

function setup_static_sta_backhaul()
{
  log  $(basename "$0") "setup_static_sta_backhaul" $LINENO "Enter setup_static_sta_backhaul"
  local interface=$(util_execute_uci get qcmap_wlan_current.@stamodeconfig[0].ifname)
  local ip=$(util_execute_uci get qcmap_wlan_current.@stamodeconfig[0].ipaddr)
  local subnet=$(util_execute_uci get qcmap_wlan_current.@stamodeconfig[0].netmask)
  local dns=$(util_execute_uci get qcmap_wlan_current.@stamodeconfig[0].dnsserver)
  local gateway=$(util_execute_uci get qcmap_wlan_current.@stamodeconfig[0].gateway)

  echo "export IFNAME=\"$interface\""   > /tmp/ipv4config.wanwlan
  echo "export PUBLIC_IP=\"$ip\""      >> /tmp/ipv4config.wanwlan
  echo "export NETMASK=\"$subnet\""    >> /tmp/ipv4config.wanwlan
  echo "export DNSSERVERS=\"$dns\""    >> /tmp/ipv4config.wanwlan
  echo "export GATEWAY=\"$gateway\""   >> /tmp/ipv4config.wanwlan

  log  $(basename "$0") "setup_static_sta_backhaul" $LINENO "setup_static_sta_backhaul for /tmp/ipv4config.wanwlan"

}

function load_static_station_setting()
{
  log  $(basename "$0") "load_static_station_setting" $LINENO "Enter load_static_station_setting"
  local bh_switch_request=$1
  local interface=$(util_execute_uci get qcmap_wlan_current.@stamodeconfig[0].ifname)
  local gateway=$(util_execute_uci get qcmap_wlan_current.@stamodeconfig[0].gateway)

  if [ $sta_mode_conn_type == $QCMAP_STA_CONNECTION_STATIC ] ; then

    if [ ! -f /tmp/ipv4config.wanwlan ]; then
      setup_static_sta_backhaul
    fi

    log  $(basename "$0") "load_static_station_setting" $LINENO "route add default gw $gateway dev $interface"
    route add default gw $gateway dev $interface

    log  $(basename "$0") "load_static_station_setting" $LINENO "bh_switch_request is $bh_switch_request"
    if [ "$bh_switch_request" == 1 ]; then
      log  $(basename "$0") "switch_bh wanwlan 1" $LINENO "switch_bh wanwlan 1"
      /etc/data/backhaulCommonConfig.sh switch_bh wanwlan 1
    fi

  fi
}

function unload_static_station_setting()
{
  log  $(basename "$0") "unload_static_station_setting" $LINENO "Enter unload_static_station_setting"
  local bh_switch_request=$1
  local interface=$(util_execute_uci get qcmap_wlan_current.@stamodeconfig[0].ifname)
  local gateway=$(util_execute_uci get qcmap_wlan_current.@stamodeconfig[0].gateway)

  if [ $sta_mode_conn_type == $QCMAP_STA_CONNECTION_STATIC ] ; then

    log  $(basename "$0") "unload_static_station_setting" $LINENO "route del default gw $gateway dev $interface"
    route del default gw $gateway dev $interface

    log  $(basename "$0") "unload_static_station_setting" $LINENO "bh_switch_request is $bh_switch_request"
    if [ "$bh_switch_request" == 1 ]; then
      log  $(basename "$0") "switch_bh wanwlan 0" $LINENO "switch_bh wanwlan 0"
      /etc/data/backhaulCommonConfig.sh switch_bh wanwlan 0
    fi
  fi
}

function add_wanwlan_iface_on_wan_all_zone() {
  local wlan_mode=$1

  #Getting wan_all zone index
  zone_wan_all_idx=`util_execute_uci show firewall | grep -i name | grep -i "wan_all" | awk -F'[][]' '{print $2}'`

  #checking if wanwlan iface is already there in the list; if yes, delete it firstly
  all_iface_wan_listed=`util_execute_uci -q get firewall.@zone[$zone_wan_all_idx].network`
  if [ ! -z "$all_iface_wan_listed" ]; then
    for iface in $all_iface_wan_listed
    do
      if [ "$iface" = "wanwlan" ]; then
        log $(basename "$0") "add_wan_iface_on_wan_all_zone :" $LINENO ": wan iface is already added on list network of wan_all zone, need to delete it firstly"
        util_execute_uci del_list firewall.@zone[$zone_wan_all_idx].network="wanwlan"
        util_execute_uci commit firewall
      fi
    done
  fi

}

function update_wanwlan_iface_on_wan_zone() {
  local wlan_mode=$(util_execute_uci get qcmap_wlan.@wlanconfig[0].mode)
  local isAdd=$1

  if [ $wlan_mode == $QCMAP_WLAN_MODE_AP_STA ] || [ $wlan_mode == $QCMAP_WLAN_MODE_AP_AP_STA ] || [ $wlan_mode == $QCMAP_WLAN_MODE_STA_ONLY ]; then
    #Getting wan_all zone index
    zone_wan_idx=`util_execute_uci show firewall | grep -i name | grep -i "'wan'" | awk -F'[][]' '{print $2}'`

    #checking if wanwlan/wanwlan_v6 iface is already there in the list; if yes, delete it
    util_execute_uci -q del_list firewall.@zone[$zone_wan_idx].network="wanwlan"
    util_execute_uci -q del_list firewall.@zone[$zone_wan_idx].network="wanwlan_v6"
    util_execute_uci commit firewall

    if [ $isAdd -eq 1 ]; then
      #checking wlan mode. If wlan is STA mode, add wanwlan into wan_all firewall zone,
      util_execute_uci add_list firewall.@zone[$zone_wan_idx].network=wanwlan
      util_execute_uci add_list firewall.@zone[$zone_wan_idx].network=wanwlan_v6
      log $(basename "$0") "update_wanwlan_iface_on_wan_zone :" $LINENO ": wanwlan iface is successfully added on list network of wan_all zone"
      util_execute_uci commit firewall
    fi

    #don't need firewall reload here, when IP address allocated/deallocated, it will reload firewall from backhaul switch

  fi
}

# check qcmap_lan mwan3track default cfg, enable/disable mwan3track for WLAN
function configure_mwan3() {
  local action=$1
  local enable_wlan=1
  local disable_wlan=0
  local wlan_mode=$(util_execute_uci get qcmap_wlan.@wlanconfig[0].mode)
  local wanwlan_status=`util_execute_uci -q get qcmap_lan.@mwan3track[0].wanwlan`

  if [ $action -eq $enable_wlan ]; then
    if [ $wlan_mode == $QCMAP_WLAN_MODE_AP_STA ] ||  [ $wlan_mode == $QCMAP_WLAN_MODE_STA_ONLY ] || [ $wlan_mode == $QCMAP_WLAN_MODE_AP_AP_STA ] ; then
      if [ $wanwlan_status -eq 0 ] ; then
        util_execute_uci set mwan3.wanwlan.enabled='1'
        util_execute_uci set mwan3.wanwlan_v6.enabled='1'
        util_execute_uci commit mwan3
        qcmap_reload_mwan3
      elif [ $wanwlan_status -eq 1 ] ; then
        return
      fi
    else
      return
    fi
  elif [ $action -eq $disable_wlan ]; then
    if [ $wlan_mode == $QCMAP_WLAN_MODE_AP_STA ] ||  [ $wlan_mode == $QCMAP_WLAN_MODE_STA_ONLY ] || [ $wlan_mode == $QCMAP_WLAN_MODE_AP_AP_STA ] ; then
      if [ $wanwlan_status -eq 0 ] ; then
        util_execute_uci set mwan3.wanwlan.enabled='0'
        util_execute_uci set mwan3.wanwlan_v6.enabled='0'
        util_execute_uci commit mwan3
        qcmap_reload_mwan3
      elif [ $wanwlan_status -eq 1 ] ; then
        return
      fi
    else
      return
    fi
  fi
}

function get_total_ap_number() {
  log  $(basename "$0") "get_total_ap_number" $LINENO " begin !!!"
  local wlan_mode=$(util_execute_uci get qcmap_wlan.@wlanconfig[0].mode)

  if [ $wlan_mode != $QCMAP_WLAN_MODE_7AP ]; then
    if [ $wlan_mode == $QCMAP_WLAN_MODE_AP ] || [ $wlan_mode == $QCMAP_WLAN_MODE_AP_STA ]; then
      total_ap_number=1
    elif [ $wlan_mode == $QCMAP_WLAN_MODE_AP_AP ] || [ $wlan_mode == $QCMAP_WLAN_MODE_AP_AP_STA ]; then
      total_ap_number=2
    elif [ $wlan_mode == $QCMAP_WLAN_MODE_AP_AP_AP ] ; then
      total_ap_number=3
    elif [ $wlan_mode == $QCMAP_WLAN_MODE_AP_AP_AP_AP ] ; then
      total_ap_number=4  # only WKK support 4AP
    else
      total_ap_number=0
    fi
  else
    total_ap_number=$(util_execute_uci get qcmap_wlan.@wlanconfig[0].totalapcount)
  fi
  log  $(basename "$0") "get_total_ap_number" $LINENO " total $total_ap_number !!!"
}
function start_wpa_cli_service() {
  log  $(basename "$0") "function start_wpa_cli_service" $LINENO " begin !!!"
  local wlan_mode=$(util_execute_uci get qcmap_wlan.@wlanconfig[0].mode)
  local sta_iface=$(util_execute_uci get qcmap_wlan.@stamodeconfig[0].ifname)
  local wpa_state

  if [ $wlan_mode == $QCMAP_WLAN_MODE_AP_STA ] || [ $wlan_mode == $QCMAP_WLAN_MODE_AP_AP_STA ] || [ $wlan_mode == $QCMAP_WLAN_MODE_STA_ONLY ]; then
     wpa_state=$(/usr/sbin/wpa_cli -i wlan0 -p /data/vendor/wifi/wpa_supplicant -s /data/vendor/wifi status | grep wpa_state | awk -F'=' '{print $2}')
     if [ -n $wpa_state -a "$wpa_state" == "COMPLETED" ]; then
        /usr/bin/QCMAP_StaInterface "wlan0" "CONNECTED"
        log  $(basename "$0") "start_wpa_cli_service" $LINENO " notify wlan0 CONNECTED"
     fi
     /usr/sbin/wpa_cli -i $sta_iface -p /data/vendor/wifi/wpa_supplicant -s /data/vendor/wifi  -B -a /usr/bin/QCMAP_StaInterface
  fi
}
function add_delete_rules_for_bridge_mode() {

  local add=$1
  local sta_iface=$(util_execute_uci get qcmap_wlan.@stamodeconfig[0].ifname)
  local ap_iface=$(util_execute_uci get qcmap_wlan.@primaryap[0].staifname)
  local ap_ap_iface=$(util_execute_uci get qcmap_wlan.@guestap[0].staifname)

  if [ $add == 1 ]; then
    ap_sta_station_mac=`ifconfig $sta_iface | grep HWaddr | awk -F' ' '{print $5}'`
    ap_sta_ap_mac=`ifconfig $ap_iface | grep HWaddr | awk -F' ' '{print $5}'`

    ebtables -t nat -A POSTROUTING -o $sta_iface -j snat --to-source $ap_sta_station_mac
    ebtables -t nat -A POSTROUTING -o $ap_iface -j snat --to-source $ap_sta_ap_mac
    if [ $wlan_mode == $QCMAP_WLAN_MODE_AP_AP_STA ] ; then
      ap_sta_ap_ap_mac=`ifconfig $ap_ap_iface | grep HWaddr | awk -F' ' '{print $5}'`
      ebtables -t nat -A POSTROUTING -o $ap_ap_iface -j snat --to-source $ap_sta_ap_ap_mac
    fi
    log  $(basename "$0") "add_delete_rules_for_bridge_mode" $LINENO " add ebtables rules succssfully"
  else
    util_run_command ebtables -t nat --flush
    util_run_command ebtables -t filter --flush
    util_run_command ebtables -t broute --flush
    /etc/data/qcmap_ebtables.sh

    ip6tables -t filter -D FORWARD -o $sta_iface -p ipv6-icmp -m icmp6 --icmpv6-type 135 -j DROP
    ip6tables -t filter -D FORWARD -i $sta_iface -p ipv6-icmp -m icmp6 --icmpv6-type 135 -j DROP
    ip6tables -t filter -D FORWARD -o $sta_iface -p ipv6-icmp -m icmp6 --icmpv6-type 136 -j DROP
    ip6tables -t filter -D FORWARD -i $sta_iface -p ipv6-icmp -m icmp6 --icmpv6-type 136 -j DROP
    ip6tables -t filter -D OUTPUT -p ipv6-icmp -m icmp6 --icmpv6-type 137 -j DROP
    log  $(basename "$0") "add_delete_rules_for_bridge_mode" $LINENO " delete ip6tables rules succssfully"
  fi
}

function utility_update_guest_ap_public_ip() {
  #default PDN config file
  local current_bh=""
  local default_pdn
  local ipv4config_file
  local ipv6config_file

  lan_ip=$(util_execute_uci get network.lan.ipaddr)
  netmask=$(util_execute_uci get network.lan.netmask)
  current_bh=$(uci -q get qcmap_lan.@profile[0].bh_present)
  if [ "$current_bh" == "wan" ]; then
    default_pdn=$(uci -q get qcmap_lan.@no_of_configs[0].default_pdn)
    ipv4config_file=/tmp/ipv4config$default_pdn
    ipv6config_file=/tmp/ipv6config$default_pdn
    log  $(basename "$0") "utility_update_guest_ap_public_ip" $LINENO "WWAN BH found with default_pdn:$default_pdn"
  elif [ "$current_bh" != "wan" ]; then
    log  $(basename "$0") "utility_update_guest_ap_public_ip" $LINENO "non-WWAN BH found: $current_bh"
    ipv4config_file=/tmp/ipv4config.$current_bh
    ipv6config_file=/tmp/ipv6config.$current_bh
  fi

  if [ -f $ipv4config_file ]; then
    ipv4_wan_connected=1
    ipv4_public_ip=`cat $ipv4config_file | grep PUBLIC_IP | sed -r 's/.*"(.+)".*/\1/'`
    log  $(basename "$0") "utility_update_guest_ap_public_ip" $LINENO "IPV4 WWAN backhual is connected."
  else
    log  $(basename "$0") "utility_update_guest_ap_public_ip" $LINENO "file $ipv4config_file doesn't exist."
  fi

  if [ -f $ipv6config_file ]; then
    ipv6_wan_connected=1
    prefix_v6=`cat $ipv6config_file | grep PUBLIC_IP6 | sed -r 's/.*"(.+)".*/\1/'`
    prefix_len=`cat $ipv6config_file | grep NETMASK6 | sed -r 's/.*"(.+)".*/\1/'`
    log  $(basename "$0") "utility_update_guest_ap_public_ip" $LINENO "IPV6 WWAN backhual is connected."
  else
    log  $(basename "$0") "utility_update_guest_ap_public_ip" $LINENO "file $ipv6config_file doesn't exist."
  fi
}

#install single guest ap access rules
#need a input argument, for example: utility_install_single_guestap_access_rules wlan1
#need get the variable value before this function used. $lan_ip, $netmask, $ipv4_public_ip, $prefix_v6, $prefix_len
function utility_install_single_guestap_access_rules() {
  local if_name=$1

  #* ------------------ IPv4 Rules ----------------------- *
  #* Rules to allow DNS/DHCP and block the rest of the traffic from Guest AP Client
  #(interface via which frame is received)  to Apps local subnet *
  ebtables -I INPUT -i $if_name -p IPv4 --ip-proto udp --ip-dport 53 -j ACCEPT
  ebtables -I INPUT -i $if_name -p IPv4 --ip-proto tcp --ip-dport 53 -j ACCEPT
  ebtables -I INPUT -i $if_name -p IPv4 --ip-proto udp --ip-dport 67 -j ACCEPT
  ebtables -A INPUT -i $if_name -p IPv4 --ip-destination $lan_ip/$netmask -j DROP
  ebtables -I INPUT -i $if_name -p IPv4 --ip-proto tcp --ip-dport 8888 -j ACCEPT

  #* Rules to allow DNS/DHCP and block the rest of the traffic from APSS/local subnet
  #to GuestAPClient(interface via which a frame is going to be sent ) *
  ebtables -I OUTPUT -o $if_name -p IPv4 --ip-proto udp --ip-sport 53 -j ACCEPT
  ebtables -I OUTPUT -o $if_name -p IPv4 --ip-proto tcp --ip-sport 53 -j ACCEPT
  ebtables -I OUTPUT -o $if_name -p IPv4 --ip-proto udp --ip-sport 67 -j ACCEPT
  ebtables -A OUTPUT -o $if_name -p IPv4 --ip-source $lan_ip/$netmask -j DROP

  if [ $ipv4_wan_connected == 1 ] && [ \( "$ipv4_public_ip" != "" \) ]; then
    #* Rules to block traffic to pubilc ip on APSS *
    ebtables -D OUTPUT -o $if_name -p IPv4 --ip-source $ipv4_public_ip -j DROP
    ebtables -A OUTPUT -o $if_name -p IPv4 --ip-source $ipv4_public_ip -j DROP
    ebtables -D INPUT -i $if_name -p IPv4 --ip-destination $ipv4_public_ip -j DROP
    ebtables -A INPUT -i $if_name -p IPv4 --ip-destination $ipv4_public_ip -j DROP
  else
    ebtables -A INPUT -i $if_name -p IPv4 -j DROP
    ebtables -A OUTPUT -o $if_name -p IPv4 -j DROP
  fi

  #* ------------------ IPv6 Rules ----------------------- *
  #* Rule to allow ICMP/DNS messages and block rest of the traffic to APSS or local LAN *
  ebtables -I INPUT -i $if_name -p IPv6 --ip6-proto ipv6-icmp --ip6-icmp-type 133:136/0 -j ACCEPT
  ebtables -I INPUT -i $if_name -p IPv6 --ip6-proto udp --ip6-dport 53 -j ACCEPT
  ebtables -I INPUT -i $if_name -p IPv6 --ip6-proto tcp --ip6-dport 53 -j ACCEPT
  ebtables -A INPUT -i $if_name -p IPv6 --ip6-destination fe80::/64 -j DROP

  if [ $ipv6_wan_connected == 1 ] && [ \( "$prefix_v6" != "" \) ]; then
    #* If a global IPv6 prefix is available, also add prefix based input rules *
    ebtables -A INPUT -i $if_name -p IPv6 --ip6-destination $prefix_v6/$prefix_len -j DROP
  fi

  #* Rule to Drop traffic routed from local subnet to guest AP. Allow DNS/ICMP responses *
  ebtables -I OUTPUT -o $if_name -p IPv6 --ip6-proto ipv6-icmp --ip6-icmp-type 133:136/0 -j ACCEPT
  ebtables -I OUTPUT -o $if_name -p IPv6 --ip6-proto udp --ip6-sport 53 -j ACCEPT
  ebtables -I OUTPUT -o $if_name -p IPv6 --ip6-proto tcp --ip6-sport 53 -j ACCEPT
  ebtables -A OUTPUT -o $if_name -p IPv6 --ip6-source fe80::/64 -j DROP

  if [ $ipv6_wan_connected == 1 ] && [ \( "$prefix_v6" != "" \) ]; then
    #* If a global IPv6 prefix is available, also add prefix based output rules *
    ebtables -A OUTPUT -o $if_name -p IPv6 --ip6-source $prefix_v6/$prefix_len -j DROP
  fi

  if [ $ipv6_wan_connected == 0 ]; then
    #* Add Drop rules if there is no backhaul avaulable/connected *
    ebtables -A OUTPUT -o $if_name -p IPv6 -j DROP
    ebtables -A INPUT -i $if_name -p IPv6 -j DROP
  fi

  #* Rules to drop briged traffic to/from LAN from/to guest AP *
  ebtables -A FORWARD -o $if_name -j DROP
  ebtables -A FORWARD -i $if_name -j DROP
}


#clean ebTables filter tables
function delete_guest_ap_access_rules() {
  util_run_command ebtables -t filter --flush
  /etc/data/qcmap_ebtables.sh

  for i in $(seq 0 3)
  do
    #Set wlan$i mode to "full" in "/etc/data/ipa/IPACM_cfg.xml"
    ipa xml wlanmode wlan$i full
  done
}

function set_wlan_bootup_enable {
  local set_wlan_bootup=$1

  util_execute_uci set qcmap_lan.@no_of_configs[0].wlan_bootup_enable=$set_wlan_bootup
  util_execute_uci commit qcmap_lan

  log  $(basename "$0") "set_wlan_bootup_enable" $LINENO "set_wlan_bootup_enable as $set_wlan_bootup done !!!"
}

function get_sta_status() {
  #Check if STA is connected to AP successuly under either dynamic or static mode
  local device=$1
  local sta_mode_conn_type=$(util_execute_uci get qcmap_wlan.@stamodeconfig[0].sta_mode_conn_type)
  local wlan_mode=$(util_execute_uci get qcmap_wlan.@wlanconfig[0].mode)

  if [ $wlan_mode == $QCMAP_WLAN_MODE_AP_STA ] || [ $wlan_mode == $QCMAP_WLAN_MODE_AP_AP_STA ] || [ $wlan_mode == $QCMAP_WLAN_MODE_STA_ONLY ]; then
    if [ $sta_mode_conn_type == $QCMAP_STA_CONNECTION_DYNAMIC ] ; then
      #For dynamic STA mode, check ifconfig v4/v6 addr
      dhcp_sta_v4addr=`ifconfig $device | grep "inet addr"`
      dhcp_sta_v6addr=`ifconfig $device | grep "inet6 addr" | grep -v Link`
      if [ "$dhcp_sta_v4addr" != "" ] || [ "$dhcp_sta_v6addr" != "" ] ; then
        log  $(basename "$0") "get_sta_status" $LINENO "Get DYNAMIC STA Status Connected"
        util_execute_uci set qcmap_wlan.@stamodeconfig[0].status=$QCMAP_STA_CONNECTED
      else
        log  $(basename "$0") "get_sta_status" $LINENO "Get DYNAMIC STA Status Disconnected"
        util_execute_uci set qcmap_wlan.@stamodeconfig[0].status=$QCMAP_STA_DISCONNECTED
      fi
    elif [ $sta_mode_conn_type == $QCMAP_STA_CONNECTION_STATIC ] ; then
      #For static sta mode, check iw ssid info
      static_sta_ssid=`iw $device info | grep ssid`
      if [ "$static_sta_ssid" != "" ] ; then
        log  $(basename "$0") "get_sta_status" $LINENO "Get STATIC STA Status Connected"
        util_execute_uci set qcmap_wlan.@stamodeconfig[0].status=$QCMAP_STA_CONNECTED
      else
        log  $(basename "$0") "get_sta_status" $LINENO "Get STATIC STA Status Disconnected"
        util_execute_uci set qcmap_wlan.@stamodeconfig[0].status=$QCMAP_STA_DISCONNECTED
      fi
    fi
  else
    #NO STA configured
    log  $(basename "$0") "get_sta_status" $LINENO "Not under STA mode"
    util_execute_uci set qcmap_wlan.@stamodeconfig[0].status=$QCMAP_STA_DISCONNECTED
  fi

  util_execute_uci commit qcmap_wlan
}

function get_sta_associcated_status()
{
  local ifname=$(util_execute_uci get wireless.wifi0_STA.ifname)
  local if_status=""
  local static_sta_ssid=""

  if [ -n "$ifname" ]; then
    if_status=$(ifconfig | grep $ifname)
    static_sta_ssid=$(iw $ifname info | grep ssid)
  fi

  if [ -n "$if_status" ]; then
    if [ -n "$static_sta_ssid" ]; then
       echo "1"
    else
       echo "0"
    fi
  else
    echo "0"
  fi
}

#GET supplicant state info to check if STA/AP+STA/AP+AP+STA mode is started or not
function get_supplicant_status() {
  #worst case 5s, one time 100ms
  local supplicant_start_max_retry=50
  local supplicant_start_delay_S=0.1
  local supplicant_pid_file
  local supplicant_pid=0
  local retry=0
  local is_sta_link_found=0
  local ath_iface

  while [ $retry -lt $supplicant_start_max_retry ]
  do
    if [ $CDT_WLAN_WKK_ATTACH -eq $CDT_TYPE ]; then
      ath_iface=$(ls /sys/class/net | grep ath)
      sta_iface=$(get_sta_iface_wkk "$ath_iface")
      is_sta_link_found=$(echo $ath_iface | grep -w $sta_iface -c)
      log  $(basename "$0") "get_supplicant_status" $LINENO " is_sta_link_found:$is_sta_link_found in retry:$retry athlist:$ath_iface sta:$sta_iface"
    else
      supplicant_pid_file=/data/vendor/wifi/wpa_supplicant-wlan0.pid
    fi
    if [ -f $supplicant_pid_file ] && [ $CDT_WLAN_HMT_ATTACH -eq $CDT_TYPE ]; then
      supplicant_pid=`cat $supplicant_pid_file`
      echo "Supplicant running normal, Get wpa_supplicant.pid: $supplicant_pid" > /dev/kmsg
      log  $(basename "$0") "get_supplicant_status" $LINENO " Supplicant start success, wpa_supplicant.pid: $supplicant_pid"
      util_execute_uci set qcmap_lan.@runstatus[0].station='1'
      util_execute_uci commit qcmap_lan
      ifconfig $sta_iface allmulti
      return
    elif [ $is_sta_link_found -eq 1 ] && [ $CDT_WLAN_WKK_ATTACH -eq $CDT_TYPE ]; then
      echo "Supplicant running normal, Get wpa_supplicant.pid: $supplicant_pid" > /dev/kmsg
      log  $(basename "$0") "get_supplicant_status" $LINENO " Supplicant start success, wpa_supplicant.pid: $supplicant_pid"
      util_execute_uci set qcmap_lan.@runstatus[0].station='1'
      util_execute_uci commit qcmap_lan
      ifconfig $sta_iface allmulti
      return
    fi

    retry=$(($retry+1))
    sleep $supplicant_start_delay_S
  done

  log  $(basename "$0") "get_supplicant_status" $LINENO " Supplicant start failed"
  util_execute_uci set qcmap_lan.@runstatus[0].station='0'
  util_execute_uci commit qcmap_lan
}
