#!/bin/sh
#  Copyright (c) 2022 Qualcomm Technologies, Inc.
#  All Rights Reserved.
#  Confidential and Proprietary - Qualcomm Technologies, Inc.

#  Usage: tethering.sh set_eth_config mode no_of_nics nic_number
#  Usage: tethering.sh set_eth_type eth_iface_name eth_nic_nw_type nic_number
#  Usage: tethering.sh start_macsec nic_number
#  Usage: tethering.sh enable_macsec macsec_iface_name mode nic_number
#  Usage: tethering.sh add_macsec nic_number
#  Usage: tethering.sh stop_macsec nic_number
#  Usage: tethering.sh disable_macsec nic_number
#  Usage: tethering.sh delete_macsec nic_number

#source log file.
. /etc/data/mbbUtils.sh

#Source lanUtils.sh file
. /etc/data/lanUtils.sh

if [[ "$#" -le 0 ]]; then
    log  $(basename "$0") $LINENO ": Usage " "tethering.sh set_eth_config mode no_of_nics"
    log  $(basename "$0") $LINENO ": Usage " "tethering.sh set_eth_type eth_iface_name eth_nic_nw_type nic_number"
    return 1
elif [ \( "$1" == "set_eth_config" \) -a \( "$#" -ne 4 \) ]; then
    log  $(basename "$0") $LINENO "Usage: tethering.sh set_eth_config mode no_of_nics nic_number"
    return 1
elif [ \( "$1" == "set_eth_type" \) -a \( "$#" -ne 4 \) ]; then
    log  $(basename "$0") $LINENO "Usage tethering.sh set_eth_type eth_iface_name eth_nic_nw_type nic_number"
    return 1
elif [ \( "$1" == "start_macsec" \) -a \( "$#" -ne 2 \) ]; then
    log  $(basename "$0") $LINENO "Usage tethering.sh start_macsec nic_number"
    return 1
elif [ \( "$1" == "enable_macsec" \) -a \( "$#" -ne 4 \) ]; then
    log  $(basename "$0") $LINENO "Usage tethering.sh enable_macsec macsec_iface_name mode nic_number"
    return 1
elif [ \( "$1" == "add_macsec" \) -a \( "$#" -ne 2 \) ]; then
    log  $(basename "$0") $LINENO "Usage tethering.sh add_macsec nic_number"
    return 1
elif [ \( "$1" == "stop_macsec" \) -a \( "$#" -ne 2 \) ]; then
    log  $(basename "$0") $LINENO "Usage tethering.sh stop_macsec nic_number"
    return 1
elif [ \( "$1" == "disable_macsec" \) -a \( "$#" -ne 2 \) ]; then
    log  $(basename "$0") $LINENO "Usage tethering.sh disable_macsec nic_number"
    return 1
elif [ \( "$1" == "delete_macsec" \) -a \( "$#" -ne 2 \) ]; then
    log  $(basename "$0") $LINENO "Usage tethering.sh delete_macsec nic_number"
    return 1
fi

QCMAP_ETH_MODE_LAN=0
QCMAP_ETH_MODE_WAN=1
QCMAP_MACSEC_MODE_SUPPLICANT=1
QCMAP_MACSEC_MODE_AUTHENTICATOR=2

function add_waneth_iface_on_wan_all_zone() {
  local eth_mode=$1

  #Getting wan_all zone index
  zone_wan_all_idx=`uci show firewall | grep -i name | grep -i "wan_all" | awk -F'[][]' '{print $2}'`

  #checking if waneth iface is already there in the list; if yes, delete it firstly
  all_iface_wan_listed=`uci get firewall.@zone[$zone_wan_all_idx].network`
  if [ ! -z "$all_iface_wan_listed" ]; then
    for iface in $all_iface_wan_listed
    do
      if [ "$iface" = "waneth" ]; then
        log $(basename "$0") "add_wan_iface_on_wan_all_zone :" $LINENO ": $iface iface is already added on list network of wan_all zone, need to delete it firstly"
        util_execute_uci del_list firewall.@zone[$zone_wan_all_idx].network="waneth"
        util_execute_uci commit firewall
      fi
      if [ "$iface" = "waneth_v6" ]; then
        log $(basename "$0") "add_wan_iface_on_wan_all_zone :" $LINENO ": $iface iface is already added on list network of wan_all zone, need to delete it firstly"
        util_execute_uci del_list firewall.@zone[$zone_wan_all_idx].network="waneth_v6"
        util_execute_uci commit firewall
      fi
    done
  fi

  #checking eth mode. If eth is wan mode, add waneth into wan_all firewall zone,
  if [ $eth_mode == $QCMAP_ETH_MODE_WAN ]; then
    util_execute_uci add_list firewall.@zone[$zone_wan_all_idx].network=waneth
    util_execute_uci add_list firewall.@zone[$zone_wan_all_idx].network=waneth_v6
    log $(basename "$0") "add_waneth_iface_on_wan_all_zone :" $LINENO ": waneth iface is successfully added on list network of wan_all zone"
    util_execute_uci commit firewall
  fi
}

#set ethernet config to qcmap_lan.
#set_ethernet_config <mode> <no_of_nics> <nic_number>
#nic_number param is only for single nic configuration (corresponds to enabled nic)
#$1 : mode [0-LAN 1-WAN 2-WAN_LAN]
#$2 : no_of_nics [2]
#$3 : nic_number [1,2]
function set_ethernet_config() {
  local mode=$1
  local no_of_nics=$2
  local nic_number=$3
  local iface_name
  local disabled_nic_number disabled_nic
  local macsec_enabled macsec_active
  local ifname_list
  ifname_list=$(uci get network.lan.ifname)
  disabled_nic=$(uci get qcmap_lan.@eth_config[0].disabled)
  #Enable any nic that was previosuly disabled
  if [ $disabled_nic -eq $QCMAP_INTERFACE_TYPE_ETH ]; then
    uci delete network.eth.disabled
    iface_name=$(util_get_interface_name $QCMAP_INTERFACE_TYPE_ETH)
    disabled_nic_number=0
  elif [ $disabled_nic -eq $QCMAP_INTERFACE_TYPE_ETH_NIC2 ]; then
    uci delete network.eth_nic2.disabled
    iface_name=$(util_get_interface_name $QCMAP_INTERFACE_TYPE_ETH_NIC2)
    disabled_nic_number=1
  elif [ $disabled_nic -eq $QCMAP_INTERFACE_TYPE_MACSEC ]; then
    uci delete network.macsec_nic.disabled
    iface_name=$(util_get_interface_name $QCMAP_INTERFACE_TYPE_ETH)
    disabled_nic_number=0
  elif [ $disabled_nic -eq $QCMAP_INTERFACE_TYPE_MACSEC_NIC2 ]; then
    uci delete network.macsec_nic2.disabled
    iface_name=$(util_get_interface_name $QCMAP_INTERFACE_TYPE_ETH_NIC2)
    disabled_nic_number=1
  fi

  uci set qcmap_lan.@eth_nic_config[$disabled_nic_number].enabled="1"
  macsec_enabled=$(uci get qcmap_lan.@eth_nic_config[$disabled_nic_number].macsec_enabled)
  macsec_active=$(uci get qcmap_lan.@eth_nic_config[$disabled_nic_number].macsec_active)
  if [ $macsec_enabled -eq 0 ]; then
    ifname_list="${ifname_list} ${iface_name}"
    ifname_list=$(echo "$ifname_list" | sed "s/^ *//g" | sed "s/ *$//g" | tr -s ' ')    #Remove first/last and trailing spaces if any
  elif [ $macsec_active -eq 1 ]; then
    ifname_list="${ifname_list} ${iface_name}"
    ifname_list=$(echo "$ifname_list" | sed "s/^ *//g" | sed "s/ *$//g" | tr -s ' ')    #Remove first/last and trailing spaces if any
    add_macsec_to_bridge $disabled_nic_number
  fi
  disabled_nic=0

  #If you are switching to single nic config, disable the unused nic
  if [ $no_of_nics -eq 1 ]; then
    if [ $nic_number -eq 2 ]; then
      uci set qcmap_lan.@eth_nic_config[0].ifname=""
      uci set qcmap_lan.@eth_nic_config[0].type="0"
      uci set network.eth.disabled='1'
      uci set qcmap_lan.@eth_nic_config[0].enabled="0"
      iface_name=$(util_get_interface_name $QCMAP_INTERFACE_TYPE_ETH)
      macsec_enabled=$(uci get qcmap_lan.@eth_nic_config[0].macsec_enabled)
      macsec_active=$(uci get qcmap_lan.@eth_nic_config[0].macsec_active)
      if [ $macsec_enabled -eq 1 ]; then
        disabled_nic=$QCMAP_INTERFACE_TYPE_MACSEC
      else
        disabled_nic=$QCMAP_INTERFACE_TYPE_ETH
      fi
    elif [ $nic_number -eq 1 ]; then
      uci set qcmap_lan.@eth_nic_config[1].ifname=""
      uci set qcmap_lan.@eth_nic_config[1].type="0"
      uci set network.eth_nic2.disabled='1'
      uci set qcmap_lan.@eth_nic_config[1].enabled="0"
      iface_name=$(util_get_interface_name $QCMAP_INTERFACE_TYPE_ETH_NIC2)
      macsec_enabled=$(uci get qcmap_lan.@eth_nic_config[1].macsec_enabled)
      macsec_active=$(uci get qcmap_lan.@eth_nic_config[1].macsec_active)
      if [ $macsec_enabled -eq 1 ]; then
        disabled_nic=$QCMAP_INTERFACE_TYPE_MACSEC_NIC2
      else
        disabled_nic=$QCMAP_INTERFACE_TYPE_ETH_NIC2
      fi
    fi

    if [ $macsec_enabled -eq 0 ]; then
      ifname_list=$(echo "$ifname_list" | sed s/$iface_name//)
      ifname_list=$(echo "$ifname_list" | sed "s/^ *//g" | sed "s/ *$//g" | tr -s ' ')    #Remove first/last and trailing spaces if any
    elif [ $macsec_active -eq 1 ]; then
      ifname_list=$(echo "$ifname_list" | sed s/$iface_name//)
      ifname_list=$(echo "$ifname_list" | sed "s/^ *//g" | sed "s/ *$//g" | tr -s ' ')    #Remove first/last and trailing spaces if any
      delete_macsec_from_bridge $nic_number
    fi
  fi

  if [ "$mode" -ne "0" ];then
    add_waneth_iface_on_wan_all_zone $QCMAP_ETH_MODE_WAN
  else
    add_waneth_iface_on_wan_all_zone $QCMAP_ETH_MODE_LAN
  fi

  uci set network.lan.ifname="$ifname_list"
  uci set qcmap_lan.@eth_config[0].disabled=$disabled_nic
  uci set qcmap_lan.@eth_config[0].mode="$mode"
  uci set qcmap_lan.@eth_config[0].no_of_nics="$no_of_nics"
  uci commit
  util_run_command ubus call network reload

  util_run_command /etc/init.d/firewall reload
}

#Update VLAN info when mode is changed from wan to lan.
#update_vlan_interface <phy_interface>
#$1 action[add, delete]
#$2 phy_interface
function update_vlan_interface() {
  local action=$1
  local eth_iface_name=$2
  local no_of_vlans vlan_id phy_interface res ifname
  no_of_vlans=$(uci get qcmap_lan.@no_of_configs[0].no_of_vlans)
  i=0
  while [ $i -lt $no_of_vlans ]
  do
    vlan_id=$(uci get qcmap_lan.@vlan[$i].vlan_id)
    phy_interface=$(uci get qcmap_lan.@vlan[$i].phy_interface)
    #check if given eth interface is under vlan, we need to add vlan interface.
    res=`echo "$phy_interface" | grep -w -c "$eth_iface_name"`
    if [ $res -eq 1 ]; then
      ifname=$(uci get network.lan$vlan_id.ifname)
      if [ "$action" = "add" ]; then
        #Add vlan interface to respective bridge.
        ifname="${ifname}$eth_iface_name.$vlan_id "
      elif [ "$action" = "delete" ]; then
        #Remove vlan interface name from respective bridge
        ifname=${ifname//"$eth_iface_name.$vlan_id "/}
      else
        log $(basename "$0") $LINENO "update_vlan_interface" "Invalid action: $action"
      fi
      uci set network.lan$vlan_id.ifname="$ifname"
    fi
    i=$((i+1))
  done
  log $(basename "$0") $LINENO "update_vlan_interface" "success "
}

#Update ETH type in qcmap_lan and network config.
#$1 nic_number[1,2]
#$2 ifname : ETH interface name.
#$3 eth_nic_type: [0-LAN, 1-WAN]
#$4 diabled : [1-disabled, 0-enabled]
function  update_eth_type() {
  local nic_number=$1
  local ifname=$2
  local eth_nic_type=$3
  local default_ethwan_profile=$(util_execute_uci get qcmap_lan.@lan[0].default_waneth_profile_num)
  local ethwan_proto=$(util_execute_uci get qcmap_lan.profile_$default_ethwan_profile.proto)
  local vlan_id=$(util_execute_uci get qcmap_lan.profile_$default_ethwan_profile.vlan_id)
  local need_enable_eth_static_ip=0
  local eth_type

  if [ $nic_number -eq 1 ]; then
    eth_type="eth"
  else
    eth_type="eth_nic2"
  fi

  # iface will alway have link local ip in the begining
  # if dhcp bakchual up, will move to public ip
  util_execute_uci set network.$eth_type.disabled="0"
  util_execute_uci set network.$eth_type.device="$ifname"
  util_execute_uci set qcmap_lan.@eth_nic_config[$((nic_number-1))].type="$eth_nic_type"
}

#Set Ethernet type to qcmap_lan
#$1 : eth_iface_name
#$2 : type[0-LAN, 1-WAN]
#$3 : NIC number[1,2]
function set_ethernet_type() {
  local eth_iface_name=$1
  local eth_nic_type=$2
  local nic_number=$3
  local type ifname ifname_list
  local disabled waneth_iface_name
  local isEthMasterAlreadySet
  local isOdhcpdNeedRestart=0
  local waneth_status=`util_execute_uci -q get qcmap_lan.@mwan3track[0].waneth`
  local default_profile=$(util_execute_uci get qcmap_lan.@lan[0].default_waneth_profile_num)
  local no_of_wan_profiles=$(util_execute_uci get qcmap_lan.@no_of_configs[0].no_of_profiles)
  local profile_id vlan_id vlan_ids proto
  local device
  local brwan_enabled=$(util_execute_uci get qcmap_lan.@global[0].brwan)

  util_execute_uci set qcmap_lan.@eth_nic_config[$((nic_number-1))].ifname="$eth_iface_name"
  util_execute_uci set qcmap_lan.@eth_nic_config[$((nic_number-1))].name="$eth_iface_name"
  ifname=$(util_execute_uci get qcmap_lan.@eth_nic_config[$((nic_number-1))].ifname)
  type=$(util_execute_uci get qcmap_lan.@eth_nic_config[$((nic_number-1))].type)
  if [ $ifname = $eth_iface_name ] && [ $type = $eth_nic_type ]; then
    log $(basename "$0") $LINENO "set_ethernet_type" ": IFNAME: $ifname is already set to type: $type"
    exit 1
  fi

  isEthMasterAlreadySet=$(util_execute_uci get dhcp.waneth_v6.master)
  if [ -z "$isEthMasterAlreadySet" ]; then
    isEthMasterAlreadySet=0
  fi

  #change lan->wan or wan->lan
  if [ $eth_nic_type -eq $QCMAP_ETH_MODE_LAN ]; then
    log $(basename "$0") $LINENO "set_ethernet_type" "setting $eth_iface_name as lan"
    # set eth to lan
    update_eth_type $nic_number $ifname $eth_nic_type
    ifname_list=$(uci get network.lan.ifname)
    ifname_list=$(echo "$ifname_list" | sed "s/^ *//g" | sed "s/ *$//g" | tr -s ' ')    #Remove first/last and trailing spaces if any
    ifname_list="${ifname_list} $eth_iface_name"
    util_execute_uci_set network.lan.ifname "$ifname_list"
    update_vlan_interface "add" $eth_iface_name
    waneth_iface_name=$(uci get network.waneth.device)
    if [ -n "$default_profile" ]; then
      waneth_iface_name=${waneth_iface_name:0:4}
    fi
    log $(basename "$0") $LINENO "set_ethernet_type" "waneth_iface_name=$waneth_iface_name: eth_iface_name=$eth_iface_name"
    if [ "$waneth_iface_name" = "$eth_iface_name" ]; then
      for i in $(seq $no_of_wan_profiles)
      do
        #for on-demand waneth profile
        profile_id=$(util_execute_uci get qcmap_lan.@profile[$((i-1))].profile_id)
        vlan_ids=$(util_execute_uci get qcmap_lan.@profile[$((i-1))].vlan_ids)
        if [ $profile_id -ge 100 -a $profile_id -ne $default_profile ]; then
          util_execute_uci set network.waneth$profile_id.disabled='1'
          util_execute_uci set network.waneth${profile_id}_v6.disabled='1'
          util_execute_uci -q delete dhcp.waneth${profile_id}_v6.dhcpv6
          util_execute_uci -q delete dhcp.waneth${profile_id}_v6.ra
          util_execute_uci -q delete dhcp.waneth${profile_id}_v6.ndp
          util_execute_uci -q delete dhcp.waneth${profile_id}_v6.master
          util_execute_uci -q delete dhcp.waneth${profile_id}_v6.ndproxy_routing
          util_execute_uci -q delete dhcp.lan$vlan_ids.master_dhcp
        fi
      done

      util_execute_uci set network.waneth.disabled='1'
      util_execute_uci set network.waneth_v6.disabled='1'
      #remove master setting when disable waneth_v6,
      #this is to let odhcpd send deprecated RA to slave interface immediatly when waneth_v6 down.
      util_execute_uci delete dhcp.waneth_v6.master
      util_execute_uci delete dhcp.waneth_v6.ra
      util_execute_uci delete dhcp.waneth_v6.ndp
      util_execute_uci delete dhcp.waneth_v6.dhcpv6
      util_execute_uci delete dhcp.lan.master_dhcp
      util_run_command /etc/init.d/odhcpd reload
      util_run_command sleep 0.5

      if [ $isEthMasterAlreadySet -eq 1 ];then
        #odhcpd not support interface change device name,
        #odhcpd reload don't help !
        #so we need restart odhcpd
        isOdhcpdNeedRestart=1
        util_run_command /etc/init.d/odhcpd stop
      fi
    fi

    if [ $waneth_status -eq 0 ] ; then
      log $(basename "$0") "set_ethernet_type" $LINENO "usr default cfg waneth disabled, need to disable waneth mwan3track"
      util_execute_uci set mwan3.waneth.enabled='0'
      util_execute_uci set mwan3.waneth_v6.enabled='0'
      util_execute_uci commit mwan3

      local mwan3status=$(/etc/init.d/mwan3 status)
      if [ $mwan3status == "running" ]; then
        /etc/init.d/mwan3 reload
      fi

    elif [ $waneth_status -eq 1 ] ; then
      log $(basename "$0") "set_ethernet_type" $LINENO "usr default cfg waneth enabled, no need to disable waneth mwan3track"
    fi

    # update to ipa
    ipa xml category $eth_iface_name LAN > /dev/null
  elif [ $eth_nic_type -eq $QCMAP_ETH_MODE_WAN ]; then
    log $(basename "$0") $LINENO "set_ethernet_type" "setting $eth_iface_name as wan"
    #set eth to wan
    update_eth_type $nic_number $ifname $eth_nic_type
    ifname_list=$(util_execute_uci get network.lan.ifname)
    ifname_list=$(echo "$ifname_list" | sed s/$eth_iface_name//)
    util_execute_uci_set network.lan.ifname "$ifname_list"
    update_vlan_interface "delete"  $eth_iface_name

    if [ $isEthMasterAlreadySet -eq 1 ];then
      #in case switch from eth0/wan,eth1/lan to eth0/lan eth1/wan, or vice versa
      #we need let odhcpd send deprecated RA to slave interface immediatly for the old wan eth;
      util_execute_uci delete dhcp.waneth_v6.master
      util_execute_uci delete dhcp.waneth_v6.ra
      util_execute_uci delete dhcp.waneth_v6.ndp
      util_execute_uci delete dhcp.waneth_v6.dhcpv6
      util_run_command /etc/init.d/odhcpd reload
      util_run_command sleep 0.5

      #odhcpd not support interface change device name,
      #odhcpd reload don't help !
      #so we need restart odhcpd
      isOdhcpdNeedRestart=1
      util_run_command /etc/init.d/odhcpd stop
    fi

    #for pppoe/dhcp
    if [ -n "$default_profile" ]; then
      vlan_id=$(util_execute_uci get qcmap_lan.profile_$default_profile.vlan_id)
      proto=$(util_execute_uci get qcmap_lan.profile_$default_profile.proto)
      log $(basename "$0") "set_ethernet_type" $LINENO "wan$default_profile profile exist, vlan_id:$vlan_id, proto:$proto"
    fi

    if [ $vlan_id -gt 0 ]; then
      device="$eth_iface_name.$vlan_id"
    else
      device=$eth_iface_name
    fi

    # if not in Deco mode, then need to update network.waneth.device
    if [ "$brwan_enabled" != "1" ]; then
      util_execute_uci set network.waneth.device="$device"
    fi

    if [ "$proto" == "dhcp" ]; then
      util_execute_uci del network.waneth_v6.ifname
      util_execute_uci set network.waneth_v6.device=$device
    elif [ "$proto" == "pppoe" ]; then
      util_execute_uci del network.waneth_v6.device
      util_execute_uci set network.waneth_v6.ifname="@waneth"
    fi

    util_execute_uci set network.waneth.disabled='0'
    util_execute_uci set network.waneth_v6.disabled='0'
    for i in $(seq $no_of_wan_profiles)
    do
      profile_id=$(util_execute_uci get qcmap_lan.@profile[$((i-1))].profile_id)
      if [ $profile_id -ge 100 -a $profile_id -ne $default_profile ]; then
        util_execute_uci set network.waneth$profile_id.disabled='0'
        util_execute_uci set network.waneth${profile_id}_v6.disabled='0'
      fi
    done

    if [ $waneth_status -eq 0 ] ; then
      log $(basename "$0") "set_ethernet_type" $LINENO "usr default cfg waneth disabled, need to enable waneth mwan3track"
      util_execute_uci set mwan3.waneth.enabled='1'
      util_execute_uci set mwan3.waneth_v6.enabled='1'
      util_execute_uci commit mwan3

      local mwan3status=$(/etc/init.d/mwan3 status)
      if [ $mwan3status == "running" ]; then
        /etc/init.d/mwan3 reload
      fi

    elif [ $waneth_status -eq 1 ] ; then
      log $(basename "$0") "set_ethernet_type" $LINENO "usr default cfg waneth enabled, no need to enable waneth mwan3track"
    fi

    # update to ipa
    ipa xml category $eth_iface_name WAN > /dev/null
  fi

  util_execute_uci commit
  log $(basename "$0") $LINENO "set_ethernet_type" "success"

  util_run_command ubus call network reload

  if [ $isOdhcpdNeedRestart -eq 1 ];then
    util_run_command /etc/init.d/odhcpd start
  fi
}

#$1 : NIC number[1,2]
#$2 : action[to_macsec, to_inactive, to_eth]
function update_vlan_macsec() {
  local nic_number=$1
  local action=$2
  local eth_iface_name
  local macsec_iface_name
  local macsec_enabled macsec_active
  local no_of_vlans
  local ifname phyname offload_phyname
  local ipa_offload is_iface_offloaded

  eth_iface_name=$(uci get qcmap_lan.@eth_nic_config[$((nic_number-1))].name)
  macsec_iface_name=$(uci get qcmap_lan.@eth_nic_config[$((nic_number-1))].macsec_name)
  macsec_enabled=$(uci get qcmap_lan.@eth_nic_config[$((nic_number-1))].macsec_enabled)
  macsec_active=$(uci get qcmap_lan.@eth_nic_config[$((nic_number-1))].macsec_active)
  no_of_vlans=$(uci get qcmap_lan.@no_of_configs[0].no_of_vlans)

  for i in $( seq 0 $no_of_vlans )
  do
    vlan_id=$(uci get qcmap_lan.@vlan[$i].vlan_id)
    phy_interface=$(uci get qcmap_lan.@vlan[$i].phy_interface)
    ifname=$(uci get network.lan$vlan_id.ifname)
    phyname=$(uci get qcmap_lan.@vlan[$i].phy_interface)
    ipa_offload=$(uci get qcmap_lan.@vlan[$i].ipa_offload)

    if [ "$action" = "to_macsec" ]; then
      #check if vlan is under eth iface
      res=`echo "$phy_interface" | grep -w -c "$eth_iface_name"`
      if [ $res -eq 1 ]; then
        ifname=${ifname//"$eth_iface_name.$vlan_id "/}
        ifname="${ifname}$macsec_iface_name.$vlan_id "
        phyname=${phyname//"$eth_iface_name "/}
        phyname="${phyname}$macsec_iface_name "
        if [ $ipa_offload -eq 1 ]; then
          is_iface_offloaded=`echo $offloaded_phy_ifaces | grep -c $eth_iface_name`
          if [ $is_iface_offloaded -eq 1 ]; then
            offload_phyname=$(uci get qcmap_lan.@vlan[$i].offload_phy_interfaces)
            offload_phyname=${phyname//"$eth_iface_name "/}
            offload_phyname="${phyname}$macsec_iface_name "
          fi
        fi
      fi
    elif [ "$action" = "to_inactive" ]; then
      #check if vlan is under macsec iface
      res=`echo "$phy_interface" | grep -w -c "$macsec_iface_name"`
      if [ $res -eq 1 ]; then
        ifname=${ifname//"$macsec_iface_name.$vlan_id "/}
        phyname=${phyname//"$macsec_iface_name "/}
        phyname="${phyname}$eth_iface_name "
        if [ $ipa_offload -eq 1 ]; then
          is_iface_offloaded=`echo $offloaded_phy_ifaces | grep -c $macsec_iface_name`
          if [ $is_iface_offloaded -eq 1 ]; then
            offload_phyname=$(uci get qcmap_lan.@vlan[$i].offload_phy_interfaces)
            offload_phyname=${phyname//"$macsec_iface_name "/}
            offload_phyname="${phyname}$eth_iface_name "
          fi
        fi
      fi
    elif [ "$action" = "to_eth" ]; then
      if [ $macsec_active -eq 1 ]; then
         #check if vlan is under macsec iface
         res=`echo "$phy_interface" | grep -w -c "$macsec_iface_name"`
         if [ $res -eq 1 ]; then
           ifname=${ifname//"$macsec_iface_name.$vlan_id "/}
           ifname="${ifname}$eth_iface_name.$vlan_id "
           phyname=${phyname//"$macsec_iface_name "/}
           phyname="${phyname}$eth_iface_name "
           if [ $ipa_offload -eq 1 ]; then
             is_iface_offloaded=`echo $offloaded_phy_ifaces | grep -c $macsec_iface_name`
             if [ $is_iface_offloaded -eq 1 ]; then
               offload_phyname=$(uci get qcmap_lan.@vlan[$i].offload_phy_interfaces)
               offload_phyname=${phyname//"$macsec_iface_name "/}
               offload_phyname="${phyname}$eth_iface_name "
             fi
           fi
         fi
      else
        #check if vlan is under eth iface
        res=`echo "$phy_interface" | grep -w -c "$eth_iface_name"`
        if [ $res -eq 1 ]; then
          ifname="${ifname}$eth_iface_name.$vlan_id "
        fi
      fi
    fi

    uci set network.lan$vlan_id.ifname="$ifname"
    uci set qcmap_lan.@vlan[$i].phy_interface="$phyname"
    if [ $ipa_offload -eq 1 ]; then
      if [ $is_iface_offloaded -eq 1 ]; then
        uci set qcmap_lan.@vlan[$i].offload_phy_interfaces="$offload_phyname"
      fi
    fi
  done
  uci commit
  ubus call network reload
}

# start_macsec: called during reboot and when macsec is enabled.
# starts supplicant/authenticator script which brings macsec iface up
#$1 : NIC number[1,2]
function start_macsec() {
  local nic_number=$1
  local mode eth_iface
  local macsec_enabled

  macsec_enabled=$(uci get qcmap_lan.@eth_nic_config[$((nic_number-1))].macsec_enabled)
  if [ "$macsec_enabled" -eq 0 ]; then
    log $(basename "$0") $LINENO "start_macsec" "macsec not enabled"
    exit 1
  fi
  mode=$(uci get qcmap_lan.@eth_nic_config[$((nic_number-1))].macsec_type)
  eth_iface=$(uci get qcmap_lan.@eth_nic_config[$((nic_number-1))].name)
  #adding sleep as WA, to accomodate delay in eth iface in bootup
  util_run_command sleep 5
  if [ $mode -eq $QCMAP_MACSEC_MODE_SUPPLICANT ]; then
    /etc/init.d/mka-supplicant.init start $eth_iface
    log $(basename "$0") $LINENO "start_macsec" "setting $eth_iface as supplicant"
  elif [ $mode -eq $QCMAP_MACSEC_MODE_AUTHENTICATOR ]; then
    /etc/init.d/mka-authenticator.init start $eth_iface
    log $(basename "$0") $LINENO "start_macsec" "setting $eth_iface as authenticator"
  fi
  log $(basename "$0") $LINENO "start_macsec" "complete"
  uci commit
}

# enable_macsec: called when client sets macsec config in setethernetnicconfig function.
# sets macsec configuration, removes eth iface from bridge, and starts suppl/auth
#Set Ethernet type to qcmap_lan
#$1 : macsec_iface_name
#$2 : mode[1-Supplicant, 2-Authenticator]
#$3 : NIC number[1,2]
function enable_macsec() {
  local macsec_iface_name=$1
  local mode=$2
  local nic_number=$3
  local type ifname eth_iface_name
  local no_of_vlans phy_interface res
  local macsec_enabled
  local disabled_nic

  macsec_enabled=$(uci get qcmap_lan.@eth_nic_config[$((nic_number-1))].macsec_enabled)
  disabled_nic=$(uci get qcmap_lan.@eth_config[0].disabled)
  if [ "$macsec_enabled" -eq 1 ]; then
    log $(basename "$0") $LINENO "enable_macsec" "macsec already enabled"
    exit 1
  fi

  if [ $nic_number -eq 1 ]; then
    if [ $disabled_nic -eq $QCMAP_INTERFACE_TYPE_ETH ]; then
      log $(basename "$0") $LINENO "enable_macsec" "ethernet nic is disabled, cannot enable macsec"
      exit 1
    fi
    uci set network.eth.proto="none"
    uci set network.macsec_nic="interface"
  elif [ $nic_number -eq 2 ]; then
    if [ $disabled_nic -eq $QCMAP_INTERFACE_TYPE_ETH_NIC2 ]; then
      log $(basename "$0") $LINENO "enable_macsec" "ethernet nic2 is disabled, cannot enable macsec"
      exit 1
    fi
    uci set network.eth_nic2.proto="none"
    uci set network.macsec_nic2="interface"
  fi

  uci set qcmap_lan.@eth_nic_config[$((nic_number-1))].macsec_name="$macsec_iface_name"
  uci set qcmap_lan.@eth_nic_config[$((nic_number-1))].macsec_enabled=1
  uci set qcmap_lan.@eth_nic_config[$((nic_number-1))].macsec_type=$mode

  #delete eth from bridge
  eth_iface_name=$(uci get qcmap_lan.@eth_nic_config[$((nic_number-1))].name)
  ifname_list=$(uci get network.lan.ifname)
  ifname_list=$(echo "$ifname_list" | sed s/$eth_iface_name//)
  ifname_list=$(echo "$ifname_list" | sed "s/^ *//g" | sed "s/ *$//g" | tr -s ' ')    #Remove first/last and trailing spaces if any
  uci set network.lan.ifname="$ifname_list"
  uci commit
  log $(basename "$0") $LINENO "enable_macsec" "complete"
  ubus call network reload
}

# add_macsec: called when macsec link is recieved by ref_daemon
# adds macsec iface to network config and bridge
#$1 : NIC number[1,2]
function add_macsec_to_bridge() {
  local nic_number=$1
  local type ifname ifname_list
  local macsec_name=$(uci get qcmap_lan.@eth_nic_config[$((nic_number-1))].macsec_name)
  local ignore_macsec_down=$(uci get qcmap_lan.@eth_nic_config[$((nic_number-1))].ignore_macsec_down)
  local macsec_enabled
  macsec_enabled=$(uci get qcmap_lan.@eth_nic_config[$((nic_number-1))].macsec_enabled)
  if [ "$macsec_enabled" -eq 0 ]; then
    log $(basename "$0") $LINENO "add_macsec_to_bridge" "macsec already enabled"
    exit 1
  fi
  ifname_list=$(uci get network.lan.ifname)
  if [[ "$ifname_list" == *"$macsec_name"* ]]; then
    log $(basename "$0") $LINENO "add_macsec_to_bridge" "macsec already added to bridge"
    exit 1
  fi
  if [ $nic_number -eq 1 ]; then
    uci set network.macsec_nic.device="$macsec_name"
    uci set network.macsec_nic.proto="static"
    uci set network.macsec_nic.ipaddr="169.254.25.1"
    uci set network.macsec_nic.netmask="255.255.255.0"
  elif [ $nic_number -eq 2 ]; then
    uci set network.macsec_nic2.device="$macsec_name"
    uci set network.macsec_nic2.proto="static"
    uci set network.macsec_nic2.ipaddr="169.254.25.1"
    uci set network.macsec_nic2.netmask="255.255.255.0"
  fi
  ifname_list="${ifname_list} ${macsec_name}"
  ifname_list=$(echo "$ifname_list" | sed "s/^ *//g" | sed "s/ *$//g" | tr -s ' ')    #Remove first/last and trailing spaces if any
  uci set network.lan.ifname="$ifname_list"
  uci set qcmap_lan.@eth_nic_config[$((nic_number-1))].ignore_macsec_down=1
  uci set qcmap_lan.@eth_nic_config[$((nic_number-1))].macsec_active=1
  uci commit
  ubus call network reload
  #this is a workaround, actual fix needs to be in netifd; check why ipv6_disabled flag is set to 1
  sleep 5
  echo '0' > /proc/sys/net/ipv6/conf/$macsec_name/disable_ipv6
  update_vlan_macsec $nic_number "to_macsec"
}

# stop_macsec: called when macsec is disabled
# stops supplicant/authetnicator script
#$1 : NIC number[1,2]
function stop_macsec() {
  local nic_number=$1
  local mode=$(uci get qcmap_lan.@eth_nic_config[$((nic_number-1))].macsec_type)
  local eth_iface=$(uci get qcmap_lan.@eth_nic_config[$((nic_number-1))].name)

  if [ $mode -eq $QCMAP_MACSEC_MODE_SUPPLICANT ]; then
    /etc/init.d/mka-supplicant.init stop $eth_iface
    log $(basename "$0") $LINENO "stop_macsec" "setting $eth_iface as supplicant"
  elif [ $mode -eq $QCMAP_MACSEC_MODE_AUTHENTICATOR ]; then
    /etc/init.d/mka-authenticator.init stop $eth_iface
    log $(basename "$0") $LINENO "stop_macsec" "setting $eth_iface as authenticator"
  fi
  delete_macsec_from_bridge $nic_number
  log $(basename "$0") $LINENO "stop_macsec" "complete"
}

# disable_macsec: called by client when updating setethernetnicconfig
# deletes macsec from bridge, stops suppl/auth, and adds eth back to the bridge
#$1 : NIC number[1,2]
function disable_macsec() {
  local nic_number=$1
  local eth_iface_name
  local macsec_iface_name=$(uci get qcmap_lan.@eth_nic_config[$((nic_number-1))].macsec_name)

  update_vlan_macsec $nic_number "to_eth"
  delete_macsec_from_bridge $nic_number
  stop_macsec $nic_number

  eth_iface_name=$(uci get qcmap_lan.@eth_nic_config[$((nic_number-1))].name)
  uci set qcmap_lan.@eth_nic_config[$((nic_number-1))].ifname="$eth_iface_name"
  uci delete qcmap_lan.@eth_nic_config[$((nic_number-1))].macsec_name
  uci set qcmap_lan.@eth_nic_config[$((nic_number-1))].macsec_enabled=0
  uci delete qcmap_lan.@eth_nic_config[$((nic_number-1))].macsec_type

  if [ $nic_number -eq 1 ]; then
    uci set network.eth.proto="static"
    uci delete network.macsec_nic
  elif [ $nic_number -eq 2 ]; then
    uci set network.eth_nic2.proto="static"
    uci delete network.macsec_nic2
  fi
  #add eth back to bridge
  ifname_list=$(uci get network.lan.ifname)
  ifname_list="${ifname_list} ${eth_iface_name}"
  ifname_list=$(echo "$ifname_list" | sed "s/^ *//g" | sed "s/ *$//g" | tr -s ' ')    #Remove first/last and trailing spaces if any
  uci set network.lan.ifname="$ifname_list"
  log $(basename "$0") $LINENO "disable_macsec" "complete"
  uci commit
  ubus call network reload
  sleep 3
  ifconfig $eth_iface_name up
}

# delete_macsec: called when client disables macsec, macsec link goes down, and on reboot
# removes macsec iface from bridge/network config
#$1 : NIC number[1,2]
function delete_macsec_from_bridge() {
  local nic_number=$1
  local type ifname ifname_list
  local macsec_name=$(uci get qcmap_lan.@eth_nic_config[$((nic_number-1))].macsec_name)
  local eth_name=$(uci get qcmap_lan.@eth_nic_config[$((nic_number-1))].name)
  local ignore_macsec_down=$(uci get qcmap_lan.@eth_nic_config[$((nic_number-1))].ignore_macsec_down)
  local macsec_enabled=$(uci get qcmap_lan.@eth_nic_config[$((nic_number-1))].macsec_enabled)
  if [ "$macsec_enabled" -eq 0 ]; then
    log $(basename "$0") $LINENO "delete_macsec_from_bridge" "macsec not enabled"
    exit 1
  fi
  if [ -z $macsec_name ]; then
    log $(basename "$0") $LINENO "delete_macsec_from_bridge" "macsec name not set"
    exit 1
  fi
  if [ $ignore_macsec_down -eq 1 ]; then
    uci delete qcmap_lan.@eth_nic_config[$((nic_number-1))].ignore_macsec_down
    uci commit
    exit 1
  fi
  update_vlan_macsec $nic_number "to_inactive"
  if [ $nic_number -eq 1 ]; then
    uci delete network.macsec_nic.device
  elif [ $nic_number -eq 2 ]; then
    uci delete network.macsec_nic2.device
  fi
  ifname_list=$(uci get network.lan.ifname)
  ifname_list=$(echo "$ifname_list" | sed s/$macsec_name//)
  ifname_list=$(echo "$ifname_list" | sed "s/^ *//g" | sed "s/ *$//g" | tr -s ' ')    #Remove first/last and trailing spaces if any
  uci set network.lan.ifname="$ifname_list"
  uci set qcmap_lan.@eth_nic_config[$((nic_number-1))].macsec_active=0
  uci commit
  ubus call network reload
}

case $1 in
  set_eth_config)
    set_ethernet_config $2 $3 $4
    ;;
  set_eth_type)
    set_ethernet_type $2 $3 $4
    ;;
  start_macsec)
    start_macsec $2
    ;;
  enable_macsec)
    enable_macsec $2 $3 $4
    ;;
  add_macsec)
    add_macsec_to_bridge $2
    ;;
  stop_macsec)
    stop_macsec $2
    ;;
  disable_macsec)
    disable_macsec $2
    ;;
  delete_macsec)
    delete_macsec_from_bridge $2
    ;;
  *)
    log $(basename "$0") $LINENO "case: " "Invalid option"
    ;;
esac

