#!/bin/sh
# :Copyright (c) 2024 Qualcomm Technologies, Inc.
#  All Rights Reserved.
#  Confidential and Proprietary - Qualcomm Technologies, Inc.

#Usage: For logging
source /etc/data/mbbUtils.sh

#source wlan common file
[ -f /etc/data/wlanConfig_common.sh ] && . /etc/data/wlanConfig_common.sh



# Usage: ./qcmapPbrConfig.sh start_pbr
# Usage: ./qcmapPbrConfig.sh stop_pbr
# Usage: ./qcmapPbrConfig.sh add_pbr_policy
# Usage: ./qcmapPbrConfig.sh del_pbr_policy
# Usage: ./qcmapPbrConfig.sh add_map_policy_to_pdn
# Usage: ./qcmapPbrConfig.sh delete_map_policy_to_pdn
# Usage: ./qcmapPbrConfig.sh get_no_of_policies
# Usage: ./qcmapPbrConfig.sh pbr_data_call_connected_event
# Usage: ./qcmapPbrConfig.sh update_pbr_policies_on_wlan_event
# Usage: ./qcmapPbrConfig.sh delete_conntrack_on_wan_event_pbr
# Usage: ./qcmapPbrConfig.sh update_all_pbr_policies_on_wlan_event
# Usage: ./qcmapPbrConfig.sh update_firewall_on_topology_change




if [[ "$#" -le 0 ]]; then
  log  $(basename "$0") $LINENO ": Usage " "qcmapPbrConfig.sh"
  return 1
elif [[ "$1" == "start_pbr" && $# -ne 1 ]]; then
  echo "Check usage ./qcmapPbrConfig.sh start_pbr"
  return 1
elif [[ "$1" == "stop_pbr" && $# -ne 1 ]]; then
  echo "Check usage ./qcmapPbrConfig.sh stop_pbr"
  return 1
elif [[ "$1" == "add_pbr_policy" && $# -ne 14 ]]; then
  echo "Check usage ./qcmapPbrConfig.sh add_pbr_policy"
  return 1
elif [[ "$1" == "del_pbr_policy" && $# -ne 2 ]]; then
  echo "Check usage ./qcmapPbrConfig.sh del_pbr_policy"
  return 1
elif [[ "$1" == "add_map_policy_to_pdn" && $# -ne 3 ]]; then
  echo "Check usage ./qcmapPbrConfig.sh add_map_policy_to_pdn"
  return 1
elif [[ "$1" == "delete_map_policy_to_pdn" && $# -ne 2 ]]; then
  echo "Check usage ./qcmapPbrConfig.sh delete_map_policy_to_pdn"
  return 1
elif [[ "$1" == "get_no_of_policies" && $# -ne 1 ]]; then
  echo "Check usage ./qcmapPbrConfig.sh get_no_of_policies"
elif [[ "$1" == "pbr_data_call_connected_event" && $# -ne 4 ]]; then
  echo "Check usage ./qcmapPbrConfig.sh pbr_data_call_connected_event"
  return 1
elif [[ "$1" == "update_pbr_policies_on_wlan_event" && $# -ne 4 ]]; then
  echo "Check usage ./qcmapPbrConfig.sh update_pbr_policies_on_wlan_event"
  return 1
elif [[ "$1" == "delete_conntrack_on_wan_event_pbr" && $# -ne 2 ]]; then
  echo "Check usage ./qcmapPbrConfig.sh delete_conntrack_on_wan_event_pbr"
  return 1
elif [[ "$1" == "update_all_pbr_policies_on_wlan_event" && $# -ne 1 ]]; then
  echo "Check usage ./qcmapPbrConfig.sh update_all_pbr_policies_on_wlan_event"
  return 1
elif [[ "$1" == "update_firewall_on_topology_change" && $# -ne 2 ]]; then
  echo "Check usage ./qcmapPbrConfig.sh update_firewall_on_topology_change"
  return 1
fi


QCMAP_NONE="none"

QCMAP_PBR_SRC_TYPE_IP_ADDR=0
QCMAP_PBR_SRC_TYPE_IP6_ADDR=1
QCMAP_PBR_SRC_TYPE_CIDR_ADDR=2
QCMAP_PBR_SRC_TYPE_HOST_NAME=3
QCMAP_PBR_SRC_TYPE_MAC_ADDR=4
QCMAP_PBR_SRC_TYPE_LAN_IFACE_NAME=5
QCMAP_PBR_SRC_TYPE_WLAN_AP_IDX=6

QCMAP_PBR_DEST_TYPE_IP_ADDR=0
QCMAP_PBR_DEST_TYPE_IP6_ADDR=1
QCMAP_PBR_DEST_TYPE_CIDR_ADDR=2
QCMAP_PBR_DEST_TYPE_HOST_NAME=3
QCMAP_PBR_DEST_TYPE_DOMAIN_NAME=4

QCMAP_PBR_PORT_TYPE_LIST=0
QCMAP_PBR_PORT_TYPE_RANGE=1

QCMAP_PBR_IPPROTO_ICMP=1
QCMAP_PBR_IPPROTO_TCP=6
QCMAP_PBR_IPPROTO_UDP=17
QCMAP_PBR_IPPROTO_ICMP6=58
QCMAP_PBR_IPPROTO_TCP_UDP=253

IP_V4=4
IP_V6=6
IP_V4V6=10

QCMAP_PBR_SUCCESS=0
QCMAP_PBR_ERROR=1

MAX_METRIC=1024


#Function to convert ipv4 to hex
function ip_to_hex() {
  printf '%02X' ${1//./ }
}

#Function to retrieve the number of policies
function util_get_num_of_policies() {
  echo $(cat /etc/config/pbr | grep policy -c)
}

#Function to get policy index from pbr corresponding to the policy name
#$1 - policy_name
function util_get_policy_index_from_policy_name() {
  local policy_name="$1"
  local policy_idx

  local no_of_policies=$(util_get_num_of_policies)
  #iterate through the policy list and find out the index matching policy name
  for i in $(seq 0 $((no_of_policies-1)))
  do
    name=$(uci get pbr.@policy[$i].name)
    #check if policy_id is same as policy
    if [ "$policy_name" = "$name" ]; then
      policy_idx=$i
      break
    fi
  done

  if [ -z "$policy_idx" ]; then
    logger $(basename "$0")":util_get_policy_index:"$LINENO":policy index not found in pbr db. exiting!"
  else
    # In case of error if we echo,value is -1 which is a valid value
    echo $policy_idx
  fi
}

#Function to store resolved fqdn ip
#$1 - policy_idx
function store_fqdn_policy_resolved_ip() {
  local policy_idx="$1"
  local ip_list nslookup_ip_list
  local ula parsed_ula
  local bridge_addr
  local wan_up
  local tmpfile profile_handle

  dest_addr_type=$(uci get pbr.@policy[$policy_idx].dest_addr_type)
  if [ -z $dest_addr_type ] || [ $dest_addr_type != $QCMAP_PBR_DEST_TYPE_DOMAIN_NAME ]; then
    return $QCMAP_PBR_SUCCESS
  fi

  domain=$(uci get pbr.@policy[$policy_idx].dest_addr)
  mapped_interface=$(uci get pbr.@policy[$policy_idx].interface)

  if [ -z "$mapped_interface" ]; then
    uci set pbr.@policy[$policy_idx].fqdn_ip=""
    uci commit pbr
    return $QCMAP_PBR_SUCCESS
  fi

  if [[ "$mapped_interface" == *"_v6"* ]]; then
    wan_up=$(is_any_wwan_backhaul_up 6)
  else
    wan_up=$(is_any_wwan_backhaul_up 4)
  fi

  if [ $wan_up -eq 1 ]; then
    #Using nslookup to send dns query to dnsmasq in order to fill ipsets
    if [[ "$mapped_interface" == *"_v6"* ]]; then
      ula=$(uci get qcmap_current_lan_config.lan.ula_prefix)
      parsed_ula=$(echo $ula | awk -F'::' '{print $1}')
      bridge_addr=$(ip addr show br-lan | grep "$parsed_ula" | sed -E 's/.*inet6 ([^/]+)\/.*/\1/')
      nslookup_ip_list=$(nslookup -type=AAAA $domain $bridge_addr | awk '/^Name:/ {c=2;N=$4} !--c {print N,$2}' | tr -d ' ' | tr '\n' ' ')
    else
      bridge_addr=$(uci get network.lan.ipaddr)
      nslookup_ip_list=$(nslookup -type=A $domain $bridge_addr | awk '/^Name:/ {c=2;N=$4} !--c {print N,$2}' | tr -d ' ' | tr '\n' ' ')
    fi

    # Extract IP set names
    ipset_names=$(cat /var/dnsmasq.d/pbr | grep $domain | awk -F'/' '{print $3}' | awk '{print $1}' | tr -s ',' ' ')
    # Find ipset corresponding to mapped iface
    for set in $ipset_names; do
      if [[ "$set" == *"_${mapped_interface}"* ]] && [[ "$set" != *"_${mapped_interface}_v6"* ]]; then
        ipset=$set
      fi
    done

    # Extract resolved IP
    members_index=$(ipset list $ipset | grep -n "Members:" | cut -d: -f1)
    last_line=$(ipset list $ipset | tail -1)
    last_index=$(ipset list $ipset | grep -n $last_line | cut -d: -f1)
    first_index=$((members_index+1))
    ip_list=$(ipset list $ipset | sed -n "${first_index},${last_index}p" | tr -s '\n' ' ')

    if [ -z $ip_list ] && [ -n $nslookup_ip_list ]; then
      ip_list=$nslookup_ip_list
    fi
    uci set pbr.@policy[$policy_idx].fqdn_ip="$ip_list"
    uci commit pbr
  fi
}

#Function to restore resolved fqdn ip post pbr start for all fqdn policies
function restore_fqdn_policies_post_reload()
{
  local no_of_policies=$(util_get_num_of_policies)

  for i in $(seq 0 $((no_of_policies-1)))
  do
    store_fqdn_policy_resolved_ip $i
  done
}

#$1: proto
#$2: src_ip
#$3: src_subnet_mask
#$4: dst_ip
#$5: dst_subnet_mask
#$6: sport or min_sport
#$7: max_sport
#$8: dport or min_dport
#$9: max_dport
#$10: version
function delete_conntrack_for_all_proto_types() {
  local proto="$1"
  local src_ip=$2
  local src_subnet_mask=$3
  local dst_ip=$4
  local dst_subnet_mask=$5
  local src_port=$6
  local max_sport=$7
  local dest_port=$8
  local max_dport=$9
  shift 1
  local version=$9

  if [ $version = $IP_V6 ]; then
    if [ "$proto" == "icmp" ]; then
      delete_conntrack_entry_utility_ipv6 "icmp" "$src_ip" "$src_subnet_mask" "$dst_ip" "$dst_subnet_mask" "$src_port" "$max_sport" "$dest_port" "$max_dport"
    elif [ "$proto" == "tcp" ]; then
      $delete_conntrack_entry_utility_ipv6 "tcp" "$src_ip" "$src_subnet_mask" "$dst_ip" "$dst_subnet_mask" "$src_port" "$max_sport" "$dest_port" "$max_dport"
    elif [ "$proto" == "udp" ]; then
      delete_conntrack_entry_utility_ipv6 "udp" "$src_ip" "$src_subnet_mask" "$dst_ip" "$dst_subnet_mask" "$src_port" "$max_sport" "$dest_port" "$max_dport"
    elif [ "$proto" == "tcp udp" ]; then
      delete_conntrack_entry_utility_ipv6 "tcp" "$src_ip" "$src_subnet_mask" "$dst_ip" "$dst_subnet_mask" "$src_port" "$max_sport" "$dest_port" "$max_dport"
      delete_conntrack_entry_utility_ipv6 "udp" "$src_ip" "$src_subnet_mask" "$dst_ip" "$dst_subnet_mask" "$src_port" "$max_sport" "$dest_port" "$max_dport"
    else
      delete_conntrack_entry_utility_ipv6 "icmp" "$src_ip" "$src_subnet_mask" "$dst_ip" "$dst_subnet_mask" "$src_port" "$max_sport" "$dest_port" "$max_dport"
      delete_conntrack_entry_utility_ipv6 "tcp" "$src_ip" "$src_subnet_mask" "$dst_ip" "$dst_subnet_mask" "$src_port" "$max_sport" "$dest_port" "$max_dport"
      delete_conntrack_entry_utility_ipv6 "udp" "$src_ip" "$src_subnet_mask" "$dst_ip" "$dst_subnet_mask" "$src_port" "$max_sport" "$dest_port" "$max_dport"
    fi
  else
    if [ "$proto" == "icmp" ]; then
      delete_conntrack_entry_utility "icmp" "$src_ip" "$src_subnet_mask" "$dst_ip" "$dst_subnet_mask" "$src_port" "$max_sport" "$dest_port" "$max_dport"
    elif [ "$proto" == "tcp" ]; then
      delete_conntrack_entry_utility "tcp" "$src_ip" "$src_subnet_mask" "$dst_ip" "$dst_subnet_mask" "$src_port" "$max_sport" "$dest_port" "$max_dport"
    elif [ "$proto" == "udp" ]; then
      delete_conntrack_entry_utility "udp" "$src_ip" "$src_subnet_mask" "$dst_ip" "$dst_subnet_mask" "$src_port" "$max_sport" "$dest_port" "$max_dport"
    elif [ "$proto" == "tcp udp" ]; then
      delete_conntrack_entry_utility "tcp" "$src_ip" "$src_subnet_mask" "$dst_ip" "$dst_subnet_mask" "$src_port" "$max_sport" "$dest_port" "$max_dport"
      delete_conntrack_entry_utility "udp" "$src_ip" "$src_subnet_mask" "$dst_ip" "$dst_subnet_mask" "$src_port" "$max_sport" "$dest_port" "$max_dport"
    else
      delete_conntrack_entry_utility "icmp" "$src_ip" "$src_subnet_mask" "$dst_ip" "$dst_subnet_mask" "$src_port" "$max_sport" "$dest_port" "$max_dport"
      delete_conntrack_entry_utility "tcp" "$src_ip" "$src_subnet_mask" "$dst_ip" "$dst_subnet_mask" "$src_port" "$max_sport" "$dest_port" "$max_dport"
      delete_conntrack_entry_utility "udp" "$src_ip" "$src_subnet_mask" "$dst_ip" "$dst_subnet_mask" "$src_port" "$max_sport" "$dest_port" "$max_dport"
    fi
  fi
}


#$1 - policy_name
function delete_conntrack_for_pbr() {
  local policy_name="$1"
  local policy_idx
  local proto
  local src_addr_type src_addr_list src_subnet_mask src_prefix_v6
  local dest_addr_type dest_addr_list dest_subnet_mask dest_prefix_v6
  local src_port_type src_port max_sport
  local dest_port_type dest_port max_dport
  local wan_dscp
  local isV6=0
  local check_sport_list=0
  local check_dport_list=0
  local flush_conntrack=0

  policy_idx=$(util_get_policy_index_from_policy_name $policy_name)
  if [ -z "$policy_idx" ]; then
    logger $(basename "$0")":util_get_policy_index:"$LINENO":policy index not found in pbr db. exiting!"
    return $QCMAP_PBR_ERROR
  fi
  mapped_interface=$(uci get pbr.@policy[$policy_idx].interface)
  if [[ "$mapped_interface" == *"_v6"* ]]; then
    isV6=1
  fi
  wan_dscp=$(uci get pbr.@policy[$policy_idx].qcmap_dscp)
  if [ -n "$wan_dscp" ]; then
    flush_conntrack=1
  fi
  proto=$(uci get pbr.@policy[$policy_idx].proto)
  if [ "$proto" == "ipv6-icmp" ]; then
    proto="icmp"
  fi
  src_addr_type=$(uci get pbr.@policy[$policy_idx].src_addr_type)
  src_addr_list=$(uci get pbr.@policy[$policy_idx].src_addr)
  src_subnet_mask="255.255.255.255"
  if [ $src_addr_type = $QCMAP_PBR_SRC_TYPE_CIDR_ADDR ]; then
    src_subnet_mask="$(echo $src_addr_list | cut -d'/' -f2)"
    src_addr_list="$(echo $src_addr_list | cut -d'/' -f1)"
    tmp=$(( 0xffffffff ^ ((1 << (32 - $src_subnet_mask)) - 1) ))
    src_subnet_mask="$(( (tmp >> 24) & 0xff )).$(( (tmp >> 16) & 0xff )).$(( (tmp >> 8) & 0xff )).$(( tmp & 0xff ))"
  elif [ $src_addr_type = $QCMAP_PBR_SRC_TYPE_IP6_ADDR ]; then
    src_prefix_lengthv6="$(echo $src_addr_list | cut -d'/' -f2)"
    src_addr_list="$(echo $src_addr_list | cut -d'/' -f1)"
    isV6=1
  elif [ $src_addr_type = $QCMAP_PBR_SRC_TYPE_MAC_ADDR ]; then
    macaddr=$src_addr_list
    if [ $isV6 -eq 0 ]; then
      src_addr_list=$(ip n s | grep $macaddr | grep -oE "\b([0-9]{1,3}\.){3}[0-9]{1,3}\b")
    else
      src_addr_list=$(ip -6 n s | grep $macaddr | grep -vE 'fe80|host' | awk '{print $1}')
      src_prefix_lengthv6="128"
    fi
  elif [ $src_addr_type = $QCMAP_PBR_SRC_TYPE_LAN_IFACE_NAME ]; then
    iface_name="$(echo $src_addr_list | cut -d'@' -f2)"
    is_wifi_iface=$(echo $iface_name | grep 'wlan\|ath')
    if [ $is_wifi_iface ]; then
      flush_conntrack=1
    else
      macaddr=$(bridge fdb show | grep $iface_name | grep -v -e permanent | grep -oE '([[:xdigit:]]{1,2}:){5}[[:xdigit:]]{1,2}')
      if [ $isV6 -eq 0 ]; then
        src_addr_list=$(ip n s | grep $macaddr | grep -oE "\b([0-9]{1,3}\.){3}[0-9]{1,3}\b")
       else
        src_addr_list=$(ip -6 n s | grep $macaddr | grep -vE 'fe80|host' | awk '{print $1}')
        src_prefix_lengthv6="128"
      fi
    fi
  elif [ $src_addr_type = $QCMAP_PBR_SRC_TYPE_WLAN_AP_IDX ]; then
    flush_conntrack=1
  fi

  if [ $flush_conntrack -eq 0 ]; then
      dest_addr_type=$(uci get pbr.@policy[$policy_idx].dest_addr_type)
      dest_addr_list=$(uci get pbr.@policy[$policy_idx].dest_addr)
      dest_subnet_mask="255.255.255.255"
      if [ $dest_addr_type = $QCMAP_PBR_DEST_TYPE_CIDR_ADDR ]; then
        dest_subnet_mask="$(echo $dest_addr | cut -d'/' -f2)"
        dest_addr_list="$(echo $dest_addr | cut -d'/' -f1)"
        tmp=$(( 0xffffffff ^ ((1 << (32 - $dest_subnet_mask)) - 1) ))
        dest_subnet_mask="$(( (tmp >> 24) & 0xff )).$(( (tmp >> 16) & 0xff )).$(( (tmp >> 8) & 0xff )).$(( tmp & 0xff ))"
      elif [ $dest_addr_type = $QCMAP_PBR_DEST_TYPE_IP6_ADDR ]; then
        dest_prefix_lengthv6="$(echo $dest_addr | cut -d'/' -f2)"
        dest_addr_list="$(echo $dest_addr_list | cut -d'/' -f1)"
        isV6=1
      elif [ $dest_addr_type = $QCMAP_PBR_DEST_TYPE_DOMAIN_NAME ]; then
        dest_addr_list=$(uci get pbr.@policy[$policy_idx].fqdn_ip)
        dest_prefix_lengthv6="128"
      fi

      src_port_type=$(uci get pbr.@policy[$policy_idx].src_port_type)
      src_port=$(uci get pbr.@policy[$policy_idx].src_port)
      if [ $src_port_type = $QCMAP_PBR_PORT_TYPE_LIST ]; then
        check_sport_list=1
        max_sport=0
      elif [ $src_port_type = $QCMAP_PBR_PORT_TYPE_RANGE ]; then
        max_sport="$(echo $src_port | cut -d'-' -f2)"
        src_port="$(echo $src_port | cut -d'-' -f1)"
      fi

      dest_port_type=$(uci get pbr.@policy[$policy_idx].dest_port_type)
      dest_port=$(uci get pbr.@policy[$policy_idx].dest_port)
      if [ $dest_port_type = $QCMAP_PBR_PORT_TYPE_LIST ]; then
        check_dport_list=1
        max_dport=0
      elif [ $dest_port_type = $QCMAP_PBR_PORT_TYPE_RANGE ]; then
        max_dport="$(echo $dest_port | cut -d'-' -f2)"
        dest_port="$(echo $dest_port | cut -d'-' -f1)"
      fi

      #Adding value to empty lists in order to ensure loop runs once
      if [ -z "$src_addr_list" ]; then
        src_addr_list="$QCMAP_NONE"
      fi
      if [ -z "$dest_addr_list" ]; then
        dest_addr_list="$QCMAP_NONE"
      fi
      for src_addr in $src_addr_list; do
        for dest_addr in $dest_addr_list; do
          if [ "$src_addr_list" = "$QCMAP_NONE" ]; then
            src_addr=""
            src_addr_list=""
          fi
          if [ "$dest_addr_list" = "$QCMAP_NONE" ]; then
            dest_addr=""
            dest_addr_list=""
          fi
          if [ $check_sport_list -eq 1 ] && [ $check_dport_list -eq 1 ]; then
            for sport in $src_port; do
              src_port=$sport
              for dport in $dest_port; do
                dest_port=$dport
                if [ $isV6 = 0 ]; then
                  delete_conntrack_for_all_proto_types "$proto" "$src_addr" "$src_subnet_mask" "$dest_addr" "$dest_subnet_mask" "$src_port" "$max_sport" "$dest_port" "$max_dport" "$IP_V4"
                else
                  delete_conntrack_for_all_proto_types "$proto" "$src_addr" "$src_prefix_lengthv6" "$dest_addr" "$dest_prefix_lengthv6" "$src_port" "$max_sport" "$dest_port" "$max_dport" "$IP_V6"
                fi
              done
            done
          elif [ $check_sport_list -eq 1 ]; then
            for port in $src_port; do
              src_port=$port
              if [ $isV6 = 0 ]; then
                delete_conntrack_for_all_proto_types "$proto" "$src_addr" "$src_subnet_mask" "$dest_addr" "$dest_subnet_mask" "$src_port" "$max_sport" "$dest_port" "$max_dport" "$IP_V4"
              else
                delete_conntrack_for_all_proto_types "$proto" "$src_addr" "$src_prefix_lengthv6" "$dest_addr" "$dest_prefix_lengthv6" "$src_port" "$max_sport" "$dest_port" "$max_dport" "$IP_V6"
              fi
            done
          elif [ $check_dport_list -eq 1 ]; then
            for port in $dest_port; do
              dest_port=$port
              if [ $isV6 = 0 ]; then
                delete_conntrack_for_all_proto_types "$proto" "$src_addr" "$src_subnet_mask" "$dest_addr" "$dest_subnet_mask" "$src_port" "$max_sport" "$dest_port" "$max_dport" "$IP_V4"
              else
                delete_conntrack_for_all_proto_types "$proto" "$src_addr" "$src_prefix_lengthv6" "$dest_addr" "$dest_prefix_lengthv6" "$src_port" "$max_sport" "$dest_port" "$max_dport" "$IP_V6"
              fi
            done
          else
            if [ $isV6 = 0 ]; then
              delete_conntrack_for_all_proto_types "$proto" "$src_addr" "$src_subnet_mask" "$dest_addr" "$dest_subnet_mask" "$src_port" "$max_sport" "$dest_port" "$max_dport" "$IP_V4"
            else
              delete_conntrack_for_all_proto_types "$proto" "$src_addr" "$src_prefix_lengthv6" "$dest_addr" "$dest_prefix_lengthv6" "$src_port" "$max_sport" "$dest_port" "$max_dport" "$IP_V6"
            fi
          fi
        done
      done
  else
    conntrack -F
  fi
}

#$1:wan_iface
function delete_conntrack_on_wan_event_pbr() {
  local wan_iface=$1
  local dscp
  local src_addr src_addr_type
  local interface enabled
  local policy_list=""
  local no_of_policies=$(util_get_num_of_policies)

  for i in $(seq 0 $((no_of_policies-1)))
  do
    interface=$(uci get pbr.@policy[$i].interface)
    enabled=$(uci get pbr.@policy[$i].enabled)
    if [ "$enabled" -eq 1 ] && [ "$wan_iface" = "$interface" ]; then
      name=$(uci get pbr.@policy[$i].name)
      dscp=$(uci get pbr.@policy[$i].qcmap_dscp)
      if [ -n "$dscp" ]; then
        conntrack -F
        return $QCMAP_PBR_SUCCESS
      fi
      src_addr_type=$(uci get pbr.@policy[$i].src_addr_type)
      src_addr=$(uci get pbr.@policy[$i].src_addr)
      if [ $src_addr_type = $QCMAP_PBR_SRC_TYPE_LAN_IFACE_NAME ]; then
        iface_name="$(echo $src_addr | cut -d'@' -f2)"
        is_wifi_iface=$(echo $iface_name | grep 'wlan\|ath')
        if [ $is_wifi_iface ]; then
          conntrack -F
          return $QCMAP_PBR_SUCCESS
        fi
      elif [ $src_addr_type = $QCMAP_PBR_SRC_TYPE_WLAN_AP_IDX ]; then
        conntrack -F
        return $QCMAP_PBR_SUCCESS
      fi
      policy_list="${policy_list} ${name}"
    fi
  done

  for policy in $policy_list; do
    policy_idx=$(util_get_policy_index_from_policy_name $policy)
    store_fqdn_policy_resolved_ip $policy_idx
    delete_conntrack_for_pbr $policy
  done
}

#Utility function to Delete the client conntrack
#$1: protocol
#$2: src_ip
#$3: src_subnet_mask
#$4: dst_ip
#$5: dst_subnet_mask
#$6: sport or min_sport
#$7: max_sport
#$8: dport or min_dport
#$9: max_dport
function delete_conntrack_entry_utility() {
  local protocol=$1
  local src_ip=$2
  local src_subnet_mask=$3
  local dst_ip=$4
  local dst_subnet_mask=$5
  local src_port=$6
  local max_sport=$7
  local dest_port=$8
  local max_dport=$9
  local check_src_ip=false
  local check_dst_ip=false
  local check_sport=false
  local check_dport=false

  local conntrack_cmd1=""
  local conntrack_cmd2=""
  local conntrack_cmd3=""
  local conntrack_cmd4=""
  local conntrack_cmd5=""
  local conntrack_cmd6=""

  local dcmd1=""
  local dcmd2=""
  local dcmd3=""
  local dcmd4=""
  local dcmd5=""
  local dcmd6=""

  log  $(basename "$0") "delete_conntrack_entry_for_drop_ipv4_firewall_entries_utility" $LINENO "Entering delete_conntrack_entry_for_drop_ipv4_firewall_entries_utility function"

  if [ $protocol = "udp" ];then
    conntrack_cmd1="conntrack -L | cut -f1,9,10,11,12 -d ' ' "
  elif [ $protocol = "tcp" ];then
    conntrack_cmd1="conntrack -L | cut -f1,10,11,12,13 -d ' ' "
  elif [ $protocol = "icmp" ];then
    conntrack_cmd1="conntrack -L | cut -f1,8,9 -d ' ' "
  fi

  if [ $src_ip ]; then
    first=${src_ip%%.*}
    last3=${src_ip#*.}
    second=${last3%%.*}
    last2=${last3#*.}
    third=${last2%.*}
    fourth=${last2#*.}
    first_part=`echo $first.`
    second_part=`echo $first.$second.`
    third_part=`echo $first.$second.$third.`
    fouth_part=`echo $first.$second.$third.$fourth`

    IP1=`ip_to_hex 255.0.0.0`
    IP2=`ip_to_hex 255.255.0.0`
    IP3=`ip_to_hex 255.255.255.0`
    IP4=`ip_to_hex 255.255.255.255`
    subnet_value=`ip_to_hex $src_subnet_mask`

    hex_IP1="0x$IP1"
    hex_IP2="0x$IP2"
    hex_IP3="0x$IP3"
    hex_IP4="0x$IP4"
    hex_subnet_value="0x$subnet_value"

    hex_subnet_value_num=$(printf "%d" "$hex_subnet_value")
    hex_IP1_num=$(printf "%d" "$hex_IP1")
    hex_IP2_num=$(printf "%d" "$hex_IP2")
    hex_IP3_num=$(printf "%d" "$hex_IP3")
    hex_IP4_num=$(printf "%d" "$hex_IP4")

    if [ $hex_subnet_value_num -lt $hex_IP1_num ]; then
      check_src_ip=true

    # We need to grep according to the subnet mask. If subnet mask
    # is 255.0.0.0 / 255.255.0.0 / 255.255.255.0 / 255.255.255.255 ,
    # then we can grep straight-away for the match. Otherwise, we grep for
    # the minimum pattern that matches perfectly and check later whether
    # we need to delete that conntrack entry or not

    elif [ $hex_subnet_value_num -ge $hex_IP1_num ] && [ $hex_subnet_value_num -lt $hex_IP2_num ]; then
      conntrack_cmd2="| grep src=$first_part"

      if [ $hex_subnet_value_num != $hex_IP1_num ]; then
        check_src_ip=true
      fi

    elif [ $hex_subnet_value_num -ge $hex_IP2_num ] && [ $hex_subnet_value_num -lt $hex_IP3_num ]; then
      conntrack_cmd2="| grep src=$second_part"

      if [ $hex_subnet_value_num != $hex_IP2_num ]; then
        check_src_ip=true
      fi

    elif [ $hex_subnet_value_num -ge $hex_IP3_num ] && [ $hex_subnet_value_num -lt $hex_IP4_num ]; then
      conntrack_cmd2="| grep src=$third_part"

      if [ $hex_subnet_value_num != $hex_IP3_num ]; then
        check_src_ip=true
      fi

    elif [ $hex_subnet_value_num -eq $hex_IP4_num ]; then
      conntrack_cmd2="| grep src=$src_ip"

      if [ $hex_subnet_value_num != $hex_IP4_num ]; then
        check_src_ip=true
      fi
    fi
  fi

  if [ $dst_ip ]; then
    first=${dst_ip%%.*}
    last3=${dst_ip#*.}
    second=${last3%%.*}
    last2=${last3#*.}
    third=${last2%.*}
    fourth=${last2#*.}
    first_part=`echo $first.`
    second_part=`echo $first.$second.`
    third_part=`echo $first.$second.$third.`
    fouth_part=`echo $first.$second.$third.$fourth`

    IP1=`ip_to_hex 255.0.0.0`
    IP2=`ip_to_hex 255.255.0.0`
    IP3=`ip_to_hex 255.255.255.0`
    IP4=`ip_to_hex 255.255.255.255`
    subnet_value=`ip_to_hex $dst_subnet_mask`

    hex_IP1="0x$IP1"
    hex_IP2="0x$IP2"
    hex_IP3="0x$IP3"
    hex_IP4="0x$IP4"
    hex_subnet_value="0x$subnet_value"

    hex_subnet_value_num=$(printf "%d" "$hex_subnet_value")
    hex_IP1_num=$(printf "%d" "$hex_IP1")
    hex_IP2_num=$(printf "%d" "$hex_IP2")
    hex_IP3_num=$(printf "%d" "$hex_IP3")
    hex_IP4_num=$(printf "%d" "$hex_IP4")

    if [ $hex_subnet_value_num -lt $hex_IP1_num ]; then
      check_dst_ip=true

    # We need to grep according to the subnet mask. If subnet mask
    # is 255.0.0.0 / 255.255.0.0 / 255.255.255.0 / 255.255.255.255 ,
    # then we can grep straight-away for the match. Otherwise, we grep for
    # the minimum pattern that matches perfectly and check later whether
    # we need to delete that conntrack entry or not

    elif [ $hex_subnet_value_num -ge $hex_IP1_num ] && [ $hex_subnet_value_num -lt $hex_IP2_num ]; then
      conntrack_cmd6="| grep dst=$first_part"

      if [ $hex_subnet_value_num != $hex_IP1_num ]; then
        check_dst_ip=true
      fi

    elif [ $hex_subnet_value_num -ge $hex_IP2_num ] && [ $hex_subnet_value_num -lt $hex_IP3_num ]; then
      conntrack_cmd6="| grep dst=$second_part"

      if [ $hex_subnet_value_num != $hex_IP2_num ]; then
        check_dst_ip=true
      fi

    elif [ $hex_subnet_value_num -ge $hex_IP3_num ] && [ $hex_subnet_value_num -lt $hex_IP4_num ]; then
      conntrack_cmd6="| grep dst=$third_part"

      if [ $hex_subnet_value_num != $hex_IP3_num ]; then
        check_dst_ip=true
      fi

    elif [ $hex_subnet_value_num -eq $hex_IP4_num ]; then
      conntrack_cmd6="| grep dst=$dst_ip"

      if [ $hex_subnet_value_num != $hex_IP4_num ]; then
        check_dst_ip=true
      fi
    fi
  fi

  case "$protocol" in
    "tcp")
      conntrack_cmd3="| grep tcp "
      if [ "$src_port" ] && [ "$src_port" != "Any" ]; then
        if [ $max_sport -eq 0 ]; then
          conntrack_cmd4="| grep sport=$src_port "
        else
          min_sport=$src_port
          max_sport=$max_sport
          check_sport=true
        fi
      fi
      if [ "$dest_port" ] && [ "$dest_port" != "Any" ]; then
        if [ $max_dport -eq 0 ]; then
          conntrack_cmd5="| grep dport=$dest_port "
        else
          min_dport=$dest_port
          max_dport=$max_dport
          check_dport=true
        fi
      fi
      ;;
    "udp")
      conntrack_cmd3="| grep udp "
      if [ "$src_port" ] && [ "$src_port" != "Any" ]; then
        if [ $max_sport -eq 0 ]; then
          conntrack_cmd4="| grep sport=$src_port "
        else
          min_sport=$src_port
          max_sport=$max_sport
          check_sport=true
        fi
      fi
      if [ "$dest_port" ] && [ "$dest_port" != "Any" ]; then
        if [ $max_dport -eq 0 ]; then
          conntrack_cmd5="| grep dport=$dest_port "
        else
          min_dport=$dest_port
          max_dport=$max_dport
          check_dport=true
        fi
      fi
      ;;
    "icmp")
      conntrack_cmd3="| grep icmp "
      ;;
  esac

  command1=$conntrack_cmd1$conntrack_cmd2$conntrack_cmd6$conntrack_cmd3$conntrack_cmd4$conntrack_cmd5
  log $(basename "$0") "conntrack display ipv4 drop cmd : " $LINENO "$command1"
  touch /tmp/data/conntrack_entries.txt
  eval $command1 > /tmp/data/conntrack_entries.txt

  file="/tmp/data/conntrack_entries.txt"

  if [ -f ${file} ]
  then
      if [ -s ${file} ]
      then
        echo "conntrack_entries.txt File exists and not empty"
        #reading conntrack_entries.txt file line-by-line
        while read -r line; do
          readed_line=$(echo $line)
          #s_ipaddr is the ip address that we get by reading conntrack_entries.txt file line-by-line
          #proto is the protocol that we get by reading conntrack_entries.txt file line-by-line
          #sport is the source port that we get by reading conntrack_entries.txt file line-by-line
          #dport is the destination port that we get by reading conntrack_entries.txt file line-by-line

          proto=$(printf '%s\n' "$readed_line" |  awk '{print $1}')
          s_ipaddr=$(printf '%s\n' "$readed_line" |  awk '{print $2}' | cut -c 5-)
          #need to change based on adding dst to cut in the beginning
          #d_ipaddr=$(printf '%s\n' "$readed_line" |  awk '{print $3}' | cut -c 5-)
          if [ $protocol != "icmp" ]; then
            sport=$(printf '%s\n' "$readed_line" |  awk '{print $4}' | cut -c 7-)
            dport=$(printf '%s\n' "$readed_line" |  awk '{print $5}' | cut -c 7-)
          fi

          #Check for subnet mask
          if [ "$check_src_ip" = "true" ]; then
            src_ip_int=$(string_to_int_ip $src_ip)
            s_ipaddr_int=$(string_to_int_ip $s_ipaddr)
            subnet_mask_int=$(string_to_int_ip $src_subnet_mask)
            res1=$(( src_ip_int&subnet_mask_int ))
            res2=$(( s_ipaddr_int&subnet_mask_int ))
            if [ $res1 -ne $res2 ]; then
              continue
            fi
          fi

          #Check for subnet mask
          if [ "$check_dst_ip" = "true" ]; then
            dst_ip_int=$(string_to_int_ip $dst_ip)
            d_ipaddr_int=$(string_to_int_ip $d_ipaddr)
            subnet_mask_int=$(string_to_int_ip $dst_subnet_mask)
            res1=$(( dst_ip_int&subnet_mask_int ))
            res2=$(( d_ipaddr_int&subnet_mask_int ))
            if [ $res1 -ne $res2 ]; then
              continue
            fi
          fi

          if [ $protocol != "icmp" ];then
            #Check for source port range
            if [ $check_sport = "true" ] && [ "$sport" ];then
               if [ $sport -lt $min_sport ] || [ $sport -gt $max_sport ];then
                 continue
               fi
            fi

            #Check for destination port range
            if [ $check_dport = "true" ] && [ "$dport" ];then
               if [ $dport -lt $min_dport ] || [ $dport -gt $max_dport ];then
                 continue
               fi
            fi
          fi

          #creation of deletion command
          dcmd1="conntrack -D"
          if [ "$proto" ];then
            dcmd2=" -p $proto"
          fi
          if [ "$s_ipaddr" ];then
            dcmd3="  --orig-src $s_ipaddr"
          fi
          if [ "$d_ipaddr" ];then
            dcmd6="  --orig-dst $d_ipaddr"
          fi
          if [ "$sport" ];then
            dcmd4=" --sport $sport"
          fi
          if [ "$dport" ];then
            dcmd5=" --dport $dport"
          fi

          deletion_final_command=$dcmd1$dcmd2$dcmd3$dcmd6$dcmd4$dcmd5

          echo "Final Deletion cmd is:" $deletion_final_command

          eval $deletion_final_command

        done <$file

        rm -rf /tmp/data/conntrack_entries.txt
      else
        echo "conntrack_entries.txt File exists but empty"
      fi
  else
      echo "conntrack_entries.txt File not exists"
  fi
}

#Utility function to Delete the client conntrack when we are adding a DROP firewall entry for
#that IPv6 address and port combination
#$1 is index in qcmap_firewall
#$2 is protocol
function delete_conntrack_entry_utility_ipv6() {
  local protocol=$1
  local src_ipv6=$2
  local src_prefix_lengthv6=$3
  local dst_ipv6=$4
  local dst_prefix_lengthv6=$5
  local src_port=$6
  local max_sport=$7
  local dest_port=$8
  local max_dport=$9
  local check_src_ip=false
  local check_dst_ip=false
  local check_sport=false
  local check_dport=false
  local processV6srcaddr=false
  local processV6dstaddr=false

  local conntrack_cmd1=""
  local conntrack_cmd2=""
  local conntrack_cmd3=""
  local conntrack_cmd4=""
  local conntrack_cmd5=""
  local conntrack_cmd6=""

  local dcmd1=""
  local dcmd2=""
  local dcmd3=""
  local dcmd4=""
  local dcmd5=""
  local dcmd6=""


  log  $(basename "$0") "delete_conntrack_entry_utility_ipv6 function"

  if [ $protocol = "tcp" ]; then
    conntrack_cmd1="conntrack -L -f ipv6 | cut -f1,10,11,12,13 -d ' ' | grep -v dport=53 "
  elif [ $protocol = "udp" ]; then
    conntrack_cmd1="conntrack -L -f ipv6 | cut -f1,9,10,11,12 -d ' ' | grep -v dport=53  "
  elif [ $protocol = "icmp" ]; then
    conntrack_cmd1="conntrack -L -f ipv6 | cut -f1,6,7 -d ' ' "
  fi

  if [ $src_ipv6 ]; then
    prefixWordsSrc=$((src_prefix_lengthv6/16))
    prefixBitsSrc=$((src_prefix_lengthv6%16))

    if [ $prefixBitsSrc -eq 0 ] && [ $prefixWordsSrc -eq 8 ];then
      conntrack_cmd2="| grep src=$src_ipv6"
    else
      processV6srcaddr=true
    fi
  fi

  if [ $dst_ipv6 ]; then
    prefixWordsDst=$((dst_prefix_lengthv6/16))
    prefixBitsDst=$((dst_prefix_lengthv6%16))

    if [ $prefixBitsDst -eq 0 ] && [ $prefixWordsDst -eq 8 ];then
      conntrack_cmd6="| grep dst=$dst_ipv6"
    else
      processV6dstaddr=true
    fi
  fi

  case "$protocol" in
    "tcp")
      conntrack_cmd3="| grep tcp "
      if [ "$src_port" ] && [ "$src_port" != "Any" ]; then
        if [ $max_sport -eq 0 ]; then
          conntrack_cmd4="| grep sport=$src_port "
        else
          min_sport=$src_port
          max_sport=$max_sport
          check_sport=true;
        fi
      fi
      if [ "$dest_port" ] && [ "$dest_port" != "Any" ]; then
        if [ $max_dport -eq 0 ]; then
          conntrack_cmd5="| grep dport=$dest_port "
        else
          min_dport=$dest_port
          max_dport=$max_dport
          check_dport=true;
        fi
      fi
      ;;
    "udp")
      conntrack_cmd3="| grep udp "
      if [ "$src_port" ] && [ "$src_port" != "Any" ]; then
        if [ $max_sport -eq 0 ]; then
          conntrack_cmd4="| grep sport=$src_port "
        else
          min_sport=$src_port
          max_sport=$max_sport
          check_sport=true;
        fi
      fi
      if [ "$dest_port" ] && [ "$dest_port" != "Any" ]; then
        if [ $dest_port_range -eq 0 ]; then
          conntrack_cmd5="| grep dport=$dest_port "
        else
          min_dport=$dest_port
          max_dport=$max_dport
          check_dport=true;
        fi
      fi
      ;;
    "icmp")
      conntrack_cmd3="| grep icmpv6 "
      ;;
  esac

  command1=$conntrack_cmd1$conntrack_cmd2$conntrack_cmd6$conntrack_cmd3$conntrack_cmd4$conntrack_cmd5
  log $(basename "$0") "conntrack display ipv6 drop cmd : " $LINENO "$command1"
  touch /tmp/data/v6conntrack.txt
  eval $command1 > /tmp/data/v6conntrack.txt

  file="/tmp/data/v6conntrack.txt"

  if [ -f ${file} ]
  then
      if [ -s ${file} ]
      then
        echo "v6conntrack.txt File exists and not empty"
        #reading v6conntrack.txt file line-by-line
        while read -r line; do
          readed_line=$(echo $line)
          #s_ipaddr is the ip address that we get by v6conntrack.txt file line-by-line
          #proto is the protocol that we get by v6conntrack.txt file line-by-line
          #sport is the source port that we get by v6conntrack.txt file line-by-line
          #dport is the destination port that we get by v6conntrack.txt file line-by-line

          proto=$(printf '%s\n' "$readed_line" |  awk '{print $1}')
          s_ipaddr=$(printf '%s\n' "$readed_line" |  awk '{print $2}' | cut -c 5-)
          d_ipaddr=$(printf '%s\n' "$readed_line" |  awk '{print $3}' | cut -c 5-)
          if [ $protocol != "icmp" ]; then
            dport=$(printf '%s\n' "$readed_line" |  awk '{print $5}' | cut -c 7-)
            sport=$(printf '%s\n' "$readed_line" |  awk '{print $4}' | cut -c 7-)
          fi

          #Check for subnet mask
          if [ "$processV6srcaddr" = "true" ];then
            if [ "$s_ipaddr" ] && [ "$src_ipv6" != "Any" ];then
              cmp_len=$(( $((prefixWordsSrc-1)) + ( $prefixWordsSrc * 4 ) ))
              expand_s_ipaddr=`expand_ipv6 $s_ipaddr`
              cmp_s_ipaddr=`echo $expand_s_ipaddr | cut -c 1-$cmp_len`
              expand_src_ipv6=`expand_ipv6 $src_ipv6`
              cmp_src_ipv6=`echo $expand_src_ipv6 | cut -c 1-$cmp_len`
              if [ "$cmp_s_ipaddr" != "$cmp_src_ipv6" ];then
                skipPrefixBitProcessSrc=true
              fi
            fi
            if [ $skipPrefixBitProcessSrc = "true" ];then
              continue
            else
              hex1_after_prefixWords=`echo $expand_s_ipaddr | cut -c $((cmp_len+2))-$((cmp_len+5))`
              hex2_after_prefixWords=`echo $expand_src_ipv6 | cut -c $((cmp_len+2))-$((cmp_len+5))`
              binary1=`hex2binary $hex1_after_prefixWords`
              binary2=`hex2binary $hex2_after_prefixWords`
              binary1_prefixBits=`echo $binary1 | cut -c 1-$prefixBits`
              binary2_prefixBits=`echo $binary2 | cut -c 1-$prefixBits`
              if [ "$binary1_prefixBits" != "$binary2_prefixBits" ];then
                continue;
              fi
            fi
          fi
		  
		  #Check for subnet mask
          if [ "$processV6dstaddr" = "true" ];then
            if [ "$d_ipaddr" ] && [ "$dst_ipv6" != "Any" ];then
              cmp_len=$(( $((prefixWordsDst-1)) + ( $prefixWordsDst * 4 ) ))
              expand_d_ipaddr=`expand_ipv6 $d_ipaddr`
              cmp_d_ipaddr=`echo $expand_d_ipaddr | cut -c 1-$cmp_len`
              expand_dst_ipv6=`expand_ipv6 $dst_ipv6`
              cmp_dst_ipv6=`echo $expand_dst_ipv6 | cut -c 1-$cmp_len`
              if [ "$cmp_d_ipaddr" != "$cmp_dst_ipv6" ];then
                skipPrefixBitProcessDst=true
              fi
            fi
            if [ $skipPrefixBitProcessDst = "true" ];then
              continue
            else
              hex1_after_prefixWords=`echo $expand_d_ipaddr | cut -c $((cmp_len+2))-$((cmp_len+5))`
              hex2_after_prefixWords=`echo $expand_dst_ipv6 | cut -c $((cmp_len+2))-$((cmp_len+5))`
              binary1=`hex2binary $hex1_after_prefixWords`
              binary2=`hex2binary $hex2_after_prefixWords`
              binary1_prefixBits=`echo $binary1 | cut -c 1-$prefixBits`
              binary2_prefixBits=`echo $binary2 | cut -c 1-$prefixBits`
              if [ "$binary1_prefixBits" != "$binary2_prefixBits" ];then
                continue;
              fi
            fi
          fi

          if [ $protocol != "icmp" ];then
            #Check for source port range
            if [ $check_sport = "true" ];then
               if [ $sport -lt $min_sport ] || [ $sport -gt $max_sport ];then
                 continue
               fi
            fi

            #Check for destination port range
            if [ $check_dport = "true" ];then
               if [ $dport -lt $min_dport ] || [ $dport -gt $max_dport ];then
                 continue
               fi
            fi
          fi

          #creation of deletion command
          dcmd1="conntrack -D -f ipv6"
          if [ "$proto" ];then
            dcmd2=" -p $proto"
          fi
          if [ "$s_ipaddr" ];then
            dcmd3="  --orig-src $s_ipaddr"
          fi
		  if [ "$d_ipaddr" ];then
            dcmd6="  --orig-dst $d_ipaddr"
          fi
          if [ "$sport" ];then
            dcmd4=" --sport $sport"
          fi
          if [ "$dport" ];then
            dcmd5=" --dport $dport"
          fi

          deletion_final_command=$dcmd1$dcmd2$dcmd3$dcmd6$dcmd4$dcmd5

          echo "Final Deletion cmd is:" $deletion_final_command

          eval $deletion_final_command

        done <$file

        rm -rf /tmp/data/v6conntrack.txt
      else
        echo "v6conntrack.txt file exists but empty"
      fi
  else
      echo "v6conntrack.txt file not exists"
  fi
}

#Function to start pbr
function start_pbr() {
  #uci set firewall.@forwarding[0].dest=wan_all
  #uci commit firewall
  #/etc/init.d/firewall reload
  sysctl -w net.bridge.bridge-nf-call-iptables=1
  sysctl -w net.bridge.bridge-nf-call-ip6tables=1
  /etc/init.d/pbr start
  restore_fqdn_policies_post_reload
}

#Function to stop pbr
function stop_pbr() {
  #uci set firewall.@forwarding[0].dest=wan
  #uci commit firewall
  #/etc/init.d/firewall reload
  /etc/init.d/pbr stop
  sysctl -w net.bridge.bridge-nf-call-iptables=0
  sysctl -w net.bridge.bridge-nf-call-ip6tables=0
  uci set pbr.config.metric_counter4=1
  uci set pbr.config.metric_counter6=2
  uci commit pbr
}

#Function to check and/or mark if policy is dscp only
#$1 - policy_index
function is_dscp_only(){
  local index=$1
  policy_attr="src_addr src_port dest_addr dest_port proto"
  for attr in $policy_attr; do
    out=$(uci show pbr.@policy[$index] | grep -i "$attr")
    if [ -n "$out" ]; then
      return $QCMAP_PBR_ERROR
    fi
  done
  uci set pbr.@policy[$index].is_dscp_only=1
}

# Function to add pbr policy
#$1  - name
#$2  - src_addr_type
#$3  - src_addr
#$4  - src_port_type
#$5  - src_port
#$6  - dest_addr_type
#$7  - dest_addr
#$8  - dest_port_type
#$9  - dest_port
#$10 - proto
#$11 - wan_dscp
#$12 - dl_dscp
#$13 - ip_family
function add_pbr_policy() {
  local name enabled src_addr src_port dest_addr dest_port proto dscp wlan_ap_list ip_family
  local index

  name=$1
  src_addr_type=$2
  src_addr=$3
  src_port_type=$4
  src_port="$5"
  dest_addr_type=$6
  dest_addr=$7
  dest_port_type=$8
  dest_port="$9"
  shift 1
  proto="$9"
  shift 1
  dscp="$9"
  shift 1
  dl_dscp="$9"
  shift 1
  ip_family="$9"

  index=$(util_get_num_of_policies)

  uci add pbr policy
  uci set pbr.@policy[$index].name="$name"
  uci set pbr.@policy[$index].enabled="0"

  if [ "$dl_dscp" != $QCMAP_NONE ]; then
    uci set pbr.@policy[$index].qcmap_dl_dscp="$dl_dscp"
    is_dscp_only "$index"
  else
    if [ "$src_addr_type" != $QCMAP_NONE ]; then
      uci set pbr.@policy[$index].src_addr_type="$src_addr_type"
    fi

    if [ "$src_addr" != $QCMAP_NONE ]; then
      if [ "$src_addr_type" != $QCMAP_PBR_SRC_TYPE_WLAN_AP_IDX ]; then
        uci set pbr.@policy[$index].src_addr="$src_addr"
      else
        uci set pbr.@policy[$index].qcmap_wlan_ap_list="$src_addr"
      fi
    fi

    if [ "$src_port_type" != $QCMAP_NONE ]; then
      uci set pbr.@policy[$index].src_port_type="$src_port_type"
    fi

    if [ "$src_port" != $QCMAP_NONE ]; then
      uci set pbr.@policy[-1].src_port="$src_port"
    fi

    if [ "$dest_addr_type" != $QCMAP_NONE ]; then
      uci set pbr.@policy[$index].dest_addr_type="$dest_addr_type"
    fi

    if [ "$dest_addr" != $QCMAP_NONE ]; then
      uci set pbr.@policy[$index].dest_addr="$dest_addr"
    fi

    if [ "$dest_port_type" != $QCMAP_NONE ]; then
      uci set pbr.@policy[$index].dest_port_type="$dest_port_type"
    fi

    if [ "$dest_port" != $QCMAP_NONE ]; then
      uci set pbr.@policy[$index].dest_port="$dest_port"
    fi

    if [ "$proto" != $QCMAP_NONE ]; then
      if [ "$proto" -eq "$QCMAP_PBR_IPPROTO_ICMP" ]; then
        if [ $ip_family -eq $IP_V6 ]; then
          uci set pbr.@policy[$index].proto="ipv6-icmp"
        else
          uci set pbr.@policy[$index].proto="icmp"
        fi
      elif [ "$proto" -eq "$QCMAP_PBR_IPPROTO_TCP" ]; then
        uci set pbr.@policy[$index].proto="tcp"
      elif [ "$proto" -eq "$QCMAP_PBR_IPPROTO_UDP" ]; then
        uci set pbr.@policy[$index].proto="udp"
      elif [ "$proto" -eq "$QCMAP_PBR_IPPROTO_TCP_UDP" ]; then
        uci set pbr.@policy[$index].proto="tcp udp"
      fi
    fi

    if [ "$dscp" != $QCMAP_NONE ]; then
      uci set pbr.@policy[$index].qcmap_dscp="$dscp"
      is_dscp_only "$index"
    fi
  fi

  uci set pbr.@policy[$index].ip_family="$ip_family"

  #Do not tag internal policy created for v4v6 call with qcmap
  if  [ "${name%_v6}" = "$name" ]; then
    #Add qcmap tag to policy in order to identify it is managed by qcmap
    uci set pbr.@policy[$index].qcmap="1"
  else
    if [ "$proto" != $QCMAP_NONE ]; then
      if [ "$proto" -eq "$QCMAP_PBR_IPPROTO_ICMP" ]; then
        uci set pbr.@policy[$index].proto="ipv6-icmp"
      fi
    fi
  fi

  uci commit pbr
}


#Function to update wlan ap pbr policies when wlan is enabled/disabled
#$1 - enabled(1)/disabled(0)
#$2 - policy_idx
#$3 - wlan_ap_ifname_list
function update_pbr_policies_on_wlan_event() {
  local enabling_wlan="$1"
  local policy_idx="$2"
  local wlan_ap_ifname_list="$3"
  local src_addr_type
  local is_wlan_enabled
  local wan_interface
  local policy_handle
  local policy_idx_v6
  local ip_family

  is_wlan_enabled=$(uci get qcmap_lan.@runstatus[0].sap)
  wan_interface=$(uci get pbr.@policy[$policy_idx].interface)
  src_addr_type=$(uci get pbr.@policy[$policy_idx].src_addr_type)
  ip_family=$(uci get pbr.@policy[$policy_idx].ip_family)
  policy_handle=$(uci get pbr.@policy[$policy_idx].name)
  if [ $ip_family -eq $IP_V4V6 ]; then
    policy_handle_v6="${policy_handle}_v6"
    policy_idx_v6=$(util_get_policy_index_from_policy_name $policy_handle_v6)
    if [ -z "$policy_idx_v6" ]; then
      logger $(basename "$0")":util_get_policy_index:"$LINENO":policy index v6 not found in pbr db. exiting!"
      return $QCMAP_PBR_ERROR
    fi
  fi

  if [ $src_addr_type = $QCMAP_PBR_SRC_TYPE_WLAN_AP_IDX ]; then
    if [ "$enabling_wlan" -eq 1 ] && [ "$is_wlan_enabled" = 1 ]; then
      if [ "$wlan_ap_ifname_list" != $QCMAP_NONE ]; then
        uci set pbr.@policy[$policy_idx].src_addr="$wlan_ap_ifname_list"
        if [ -n "$wan_interface" ]; then
          uci set pbr.@policy[$policy_idx].enabled="1"
        fi
        if [ $ip_family -eq $IP_V4V6 ]; then
          uci set pbr.@policy[$policy_idx_v6].src_addr="$wlan_ap_ifname_list"
          if [ -n "$wan_interface" ]; then
            uci set pbr.@policy[$policy_idx_v6].enabled="1"
          fi
        fi
      fi
    else
      uci set pbr.@policy[$policy_idx].enabled="0"
      uci del pbr.@policy[$policy_idx].src_addr
      if [ $ip_family -eq $IP_V4V6 ]; then
        uci set pbr.@policy[$policy_idx_v6].enabled="0"
        uci del pbr.@policy[$policy_idx_v6].src_addr
      fi
    fi
  fi

  uci commit pbr
}

function update_all_pbr_policies_on_wlan_event()
{
  local is_wlan_enabled
  local src_addr_str
  local wlan_mode
  local mld_enabled
  local no_of_policies=$(util_get_num_of_policies)

  is_wlan_enabled=$(uci get qcmap_lan.@runstatus[0].sap)
  if [ $is_wlan_enabled -eq 1 ]; then
    wlan_mode=$(uci get qcmap_wlan.@wlanconfig[0].mode)
    for i in $(seq 0 $((no_of_policies-1)))
    do
      is_qcmap_policy=$(uci get pbr.@policy[$i].qcmap)
      if [ -n "$is_qcmap_policy" ]; then
        src_addr_type=$(uci get pbr.@policy[$i].src_addr_type)
        if [ $src_addr_type = $QCMAP_PBR_SRC_TYPE_WLAN_AP_IDX ]; then
          src_addr_str=""
          wlan_ap_list=$(uci get pbr.@policy[$i].qcmap_wlan_ap_list)
          mld_enabled=$(uci get qcmap_wlan.@wlanconfig[0].wlanconfigex)
          for ap_idx in $wlan_ap_list; do
            ifname=$QCMAP_NONE
            if [ -f /etc/data/wlanConfig_common.sh ] && [ "$ap_idx" -lt "$QCMAP_WLAN_MAX_AP_IDX_COUNT" ]; then
              idx=$(util_convert_wlan_ap_type_to_index $ap_idx)
              if [ $idx -eq -1 ]; then
                continue
              fi
              if [ $mld_enabled -eq 0 ] && [ $ap_idx -lt $QCMAP_MSGR_WLAN_IFACE_MLD_AP_ONE ]; then
                ifname=$(uci get qcmap_wlan.@apconfig[$idx].ifname)
              else
                ifname=$(uci get qcmap_wlan.@mldapconfig[$idx].ifname)
              fi
            fi
            if [ -n "$ifname" ] && [ "$ifname" != "$QCMAP_NONE" ]; then
              src_addr_str="${src_addr_str} @${ifname}"
            fi
          done

          wan_interface=$(uci get pbr.@policy[$i].interface)
          if [ -n "$wan_interface" ]; then
            pbr_reload_required=1
          fi

          if [ "$src_addr_str" != "" ]; then
            update_pbr_policies_on_wlan_event 1 "$i" "$src_addr_str"
          fi
        fi
      fi
    done
  fi

  if [ "$pbr_reload_required" -eq 1 ]; then
    /etc/init.d/pbr start
    restore_fqdn_policies_post_reload
  fi
}


#Function to enable pbr policy
#$1 - policy_name
function enable_pbr_policy() {
  local policy_name="$1"
  local policy_idx
  local is_dscp_only
  local src_addr_type

  policy_idx=$(util_get_policy_index_from_policy_name $policy_name)
  if [ -z "$policy_idx" ]; then
    logger $(basename "$0")":util_get_policy_index:"$LINENO":policy index not found in pbr db. exiting!"
    return $QCMAP_PBR_ERROR
  fi

  src_addr_type=$(uci get pbr.@policy[$policy_idx].src_addr_type)
  if [ "$src_addr_type" = $QCMAP_PBR_SRC_TYPE_WLAN_AP_IDX ]; then
    is_wlan_enabled=$(uci get qcmap_lan.@runstatus[0].sap)
    if [ $is_wlan_enabled = 1 ]; then
      uci set pbr.@policy[$policy_idx].enabled="1"
      uci commit pbr
    fi
  else
    is_dscp_only=$(uci get pbr.@policy[$policy_idx].is_dscp_only)
    if [ -n "$policy_idx" ] && [ "$is_dscp_only" != "1" ]; then
      uci set pbr.@policy[$policy_idx].enabled="1"
      uci commit pbr
    fi
  fi
}

#Function to disable pbr policy
#$1 - policy_name
function disable_pbr_policy() {
  local policy_name="$1"
  local policy_idx

  policy_idx=$(util_get_policy_index_from_policy_name $policy_name)
  if [ -n "$policy_idx" ]; then
    uci set pbr.@policy[$policy_idx].enabled="0"
  fi

  uci commit pbr
}


#Function to delete pbr policy
#$1 - policy_handle
function del_pbr_policy() {
  local policy_handle=$1
  local no_of_policies

  no_of_policies=$(util_get_num_of_policies)

  policy_idx=$(util_get_policy_index_from_policy_name $policy_handle)

  if [ -n "$policy_idx" ]; then
    ip_family=$(uci get pbr.@policy[$policy_idx].ip_family)
    uci delete pbr.@policy[$policy_idx]

    if [ $ip_family -eq $IP_V4V6 ]; then
      policy_handle_v6="${policy_handle}_v6"
      policy_idx=$(util_get_policy_index_from_policy_name $policy_handle_v6)
      if [ -n "$policy_idx" ]; then
        uci delete pbr.@policy[$policy_idx]
      fi
    fi
    uci commit pbr
  fi
}

#Function to map policy to pdn
#$1 - policy_handle
#$2 - profile_handle
function add_map_policy_to_pdn ()
{
  local policy_handle="$1"
  local profile_handle="$2"
  local ip_family
  local policy_idx wan_ifname wan_dscp dl_dscp
  local policy_handle_v6

  policy_idx=$(util_get_policy_index_from_policy_name $policy_handle)

  if [ -n "$policy_idx" ]; then
    ip_family=$(uci get pbr.@policy[$policy_idx].ip_family)

    if [ $profile_handle -eq 1 ]; then
      wan_ifname="wan"
    else
      wan_ifname="wan$profile_handle"
    fi
    if [ $ip_family -eq $IP_V6 ]; then
      wan_ifname="${wan_ifname}_v6"
    fi

    uci set pbr.@policy[$policy_idx].interface=$wan_ifname
    wan_dscp=$(uci get pbr.@policy[$policy_idx].qcmap_dscp)
    dl_dscp=$(uci get pbr.@policy[$policy_idx].qcmap_dl_dscp)

    #set pbr config wan_dscp
    if [ -n "$wan_dscp" ] && [ -z "$dl_dscp" ]; then
      uci set pbr.config."$wan_ifname"_dscp="$wan_dscp"
    fi

    #set pbr config dl_dscp
    if [ -z "$wan_dscp" ] && [ -n "$dl_dscp" ]; then
      uci set pbr.config."$wan_ifname"_dl_dscp="$dl_dscp"
    fi

    # enable the policy
    enable_pbr_policy $policy_handle

    if [ $ip_family -eq $IP_V4V6 ]; then
      policy_handle_v6="${policy_handle}_v6"
      policy_idx=$(util_get_policy_index_from_policy_name $policy_handle_v6)
      if [ -n "$policy_idx" ]; then
        if [ $profile_handle -eq 1 ]; then
          wan_ifname="wan_v6"
        else
          wan_ifname="wan${profile_handle}_v6"
        fi
        uci set pbr.@policy[$policy_idx].interface=$wan_ifname
        wan_dscp=$(uci get pbr.@policy[$policy_idx].qcmap_dscp)
        if [ -n "$wan_dscp" ] && [ -z "$dl_dscp" ]; then
          uci set pbr.config."$wan_ifname"_dscp="$wan_dscp"
        fi

        #set pbr config dl_dscp
        if [ -z "$wan_dscp" ] && [ -n "$dl_dscp" ]; then
          uci set pbr.config."$wan_ifname"_dl_dscp="$dl_dscp"
        fi

        # enable the policy
        enable_pbr_policy $policy_handle_v6
      fi
    fi

    uci commit pbr

    #If it is wlan policy and wlan is not enabled do not do pbr reload
    is_wlan_enabled=$(uci get qcmap_lan.@runstatus[0].sap)
    src_addr_type=$(uci get pbr.@policy[$policy_idx].src_addr_type)
    if [ "$src_addr_type" = $QCMAP_PBR_SRC_TYPE_WLAN_AP_IDX ] && [ $is_wlan_enabled != 1 ]; then
      return $QCMAP_PBR_SUCCESS
    fi

    /etc/init.d/pbr start
    restore_fqdn_policies_post_reload
    delete_conntrack_for_pbr $policy_handle
    if [ -n "$policy_handle_v6" ]; then
      delete_conntrack_for_pbr $policy_handle_v6
    fi

    if [ -f /tmp/ipv4config$profile_handle ] || [ -f /tmp/ipv6config$profile_handle ]; then
      if [ $profile_handle -eq 1 ]; then
        wan_ifname="wan"
      else
        wan_ifname="wan$profile_handle"
      fi
      if [ -f /tmp/ipv6config$profile_handle ]; then
        wan_ifname="${wan_ifname}_v6"
      fi
      if [ -n "$dl_dscp" ]; then
        rmnet_iface="$(ifstatus ${wan_ifname} | grep "\"l3_device\":" | awk -F: '{print $2}' | awk -F\" '{print $2'})"
        ipa dscp add pdn $rmnet_iface $dl_dscp
      fi
    fi
  fi
}

#Function to delete pdn to policy mapping
#$1 - policy_handle
function delete_map_policy_to_pdn () {
  local policy_handle="$1"
  local policy_idx
  local policy_idx6
  local wan_ifname
  local profile_handle
  local wan_dscp
  local policy_handle_v6
  local dl_dscp

  policy_idx=$(util_get_policy_index_from_policy_name $policy_handle)
  wan_ifname=$(uci get pbr.@policy[$policy_idx].interface)

  if [ -n "$policy_idx" ]; then
    wan_dscp=$(uci get pbr.@policy[$policy_idx].qcmap_dscp)
    dl_dscp=$(uci get pbr.@policy[$policy_idx].qcmap_dl_dscp)

    #delete pbr config wan dscp
    if [ -n "$wan_dscp" ]; then
      uci del pbr.config."$wan_ifname"_dscp="$wan_dscp"
    fi

    #delete pbr config dl dscp
    if [ -n "$dl_dscp" ]; then
      uci del pbr.config."$wan_ifname"_dl_dscp="$dl_dscp"
    fi

    # disable the policy
    disable_pbr_policy $policy_handle

    if [[ "$wan_ifname" =~ ^wan* ]]; then
      wan_ifname=$(echo "$wan_ifname" | cut -d'_' -f1)
      profile_handle=${wan_ifname#wan}
    fi

    # empty profile handle if wan_ifname=wan -> profile handle=1
    if [[ -z $profile_handle ]]; then
      profile_handle=1
    fi

    ip_family=$(uci get pbr.@policy[$policy_idx].ip_family)
    if [ $ip_family -eq $IP_V4V6 ]; then
      policy_handle_v6="${policy_handle}_v6"
      policy_idx6=$(util_get_policy_index_from_policy_name $policy_handle_v6)
      if [ -n "$policy_idx6" ]; then
        wan_ifname=$(uci get pbr.@policy[$policy_idx6].interface)

        wan_dscp=$(uci get pbr.@policy[$policy_idx6].qcmap_dscp)
        if [ -n "$wan_dscp" ]; then
          uci del pbr.config."$wan_ifname"_dscp="$wan_dscp"
        fi

        #delete pbr config dl dscp
        if [ -n "$dl_dscp" ]; then
          uci del pbr.config."$wan_ifname"_dl_dscp="$dl_dscp"
        fi

        # disable the policy
        disable_pbr_policy $policy_handle_v6
      fi
    fi

    /etc/init.d/pbr start
    restore_fqdn_policies_post_reload
    delete_conntrack_for_pbr $policy_handle
    if [ -n "$policy_handle_v6" ]; then
      delete_conntrack_for_pbr $policy_handle_v6
    fi

    if [ -f /tmp/ipv4config$profile_handle ] || [ -f /tmp/ipv6config$profile_handle ]; then
      if [ $profile_handle -eq 1 ]; then
        wan_ifname="wan"
      else
        wan_ifname="wan$profile_handle"
      fi
      if [ -f /tmp/ipv6config$profile_handle ]; then
        wan_ifname="${wan_ifname}_v6"
      fi

      # delete ipa path for dl dscp
      if [ -n "$dl_dscp" ]; then
        rmnet_iface="$(ifstatus ${wan_ifname} | grep "\"l3_device\":" | awk -F: '{print $2}' | awk -F\" '{print $2'})"
        ipa dscp del pdn $rmnet_iface
      fi
    fi

    #delete interface mapping post conntrack deletion
    uci del pbr.@policy[$policy_idx].interface
    if [ $ip_family -eq $IP_V4V6 ]; then
      uci del pbr.@policy[$policy_idx6].interface
    fi
    uci commit pbr
  fi
}

#$1 - profile
#$2 - call type
#$3 - rmnet_iface
function pbr_data_call_connected_event () {
  local profile="$1"
  local call_type="$2"
  local rmnet_iface="$3"
  local wan_ifname

  if [ $profile -eq 1 ]; then
    wan_ifname="wan"
  else
    wan_ifname="wan${profile}"
  fi

  if [ "$call_type" = "$IP_V6" ]; then
    wan_ifname="${wan_ifname}_v6"
  fi

  #dscp
  dl_dscp=$(uci get pbr.config."$wan_ifname"_dl_dscp)
  if [ -n "$dl_dscp" ]; then
    ipa dscp add pdn $rmnet_iface $dl_dscp
  fi
}

#$1 - config_file
#$2 - profile
#$3 - call type
function teardown_lan_pbr () {
  local config="$1"
  local profile="$2"
  local call_type="$3"
  local wan_ifname
  local default_pdn
  local v4_or_v6_call_up=0

  if [ -f /tmp/ipv4config${profile} ] || [ -f /tmp/ipv6config${profile} ]; then
    v4_or_v6_call_up=1
  fi

  default_pdn=$(/etc/data/lanUtils.sh util_get_default_pdn)
  if [ $profile -eq 1 ]; then
    wan_ifname="wan"
    if [ "$call_type" = "$IP_V6" ]; then
      wan_ifname="${wan_ifname}_v6"
    fi

    dl_dscp=$(uci get pbr.config.${wan_ifname}_dl_dscp)
    if [ $v4_or_v6_call_up -eq 0 ] && [ -n "$dl_dscp" ]; then
      rmnet_iface=$(cat /tmp/state/network | grep network.${wan_ifname}.ifname | awk -F "'" '{print $2}')
      ipa dscp del pdn $rmnet_iface
    fi
  else
    if [ "$call_type" = "$IP_V4" ]; then
      wan_ifname="wan${profile}"

      #Delete DNS entries
      util_del_dnsv4 $profile

      #Delete MTU options
      util_delete_mtu_options $profile $call_type

      #Perform dnsmasq reload
      /etc/init.d/dnsmasq reload

      #Get IPV4NATConfig status
      ipv4_nat_status=$(util_get_ipv4_nat_status $profile)

      if [ "$ipv4_nat_status" -eq 0 ]; then
          #Disabling Firewall on v4 BH down
          /etc/data/firewallConfig.sh disable_firewall $profile 1
       fi

      #Perform firewall reload
      /etc/init.d/firewall reload

      dl_dscp=$(uci get pbr.config."$wan_ifname"_dl_dscp)
      if [ $v4_or_v6_call_up -eq 0 ] && [ -n "$dl_dscp" ]; then
        rmnet_iface=$(cat /tmp/state/network | grep network.${wan_ifname}.ifname | awk -F "'" '{print $2}')
        ipa dscp del pdn $rmnet_iface
      fi

      uci_revert_state network $config
    else
      wan_ifname="wan${profile}_v6"

      #Delete DNS search list
      if [ $default_pdn -eq $profile ]; then
          util_delete_dns_search_list
      fi
      #Delete DNSv6 entries
      util_del_dnsv6 $profile

      #Delete MTU options
      util_delete_mtu_options $profile 6

      #Delete V6 lifetime
      util_del_v6_lifetime $profile

      #Delete routes and rules of lan side
      util_delete_lan_route_rule_v6 $profile

      #Delete v6 dhcp config entry
      util_rmnet_dhcp_delete_v6

      #Final commit
      uci commit

      #Perform network reload
      ubus call network reload

      #Perform odhcpd reload
      /etc/init.d/odhcpd reload

      #Get IPV4NATConfig status
      ipv4_nat_status=$(util_get_ipv4_nat_status $profile)
      if [ "$ipv4_nat_status" -eq 0 ]; then
          #Disabling Firewall on v6 BH down
          /etc/data/firewallConfig.sh disable_firewall $profile 2
      fi

      #Perform firewall reload
      /etc/init.d/firewall reload

      dl_dscp=$(uci get pbr.config."$wan_ifname"_dl_dscp)
      rmnet_iface=$(cat /tmp/state/network | grep network.${wan_ifname}.ifname | awk -F "'" '{print $2}')
      if [ $v4_or_v6_call_up -eq 0 ] && [ -n "$dl_dscp" ]; then
        ipa dscp del pdn $rmnet_iface
      fi

      /etc/data/lanUtils.sh util_remove_nptv6_rules $profile

      #Delete all upstream related network rules from state
      uci_revert_state network $cfg
    fi
  fi
}

#Function to Inform NetIfD of DNS information and add DNS information to resolv file
#$1 - Profile ID
#$2 - DNS Server IP address
function util_add_dnsv4_pbr() {
    local profile="$1"
    local DNS="$2"
    local downstream isPresent
    local default_pdn
    local metric_counter
    local iface_name

    downstream="lan"
    default_pdn=$(/etc/data/lanUtils.sh util_get_default_pdn)
    if [ $profile -eq 1 ]; then
      iface_name="wan"
    else
      iface_name="wan${profile}"
    fi

    #Metric is added so that duplicate dns servers for different PDN can be handled
    metric_counter=$(uci get pbr.config.metric_counter4)
    if [ $metric_counter != $MAX_METRIC ]; then
      metric_counter=$((metric_counter+1))
    fi
    resolve_file=$(uci get dhcp."$downstream"_dns.resolvfile)
    for dns in $DNS; do
        proto_add_dns_server "$dns"
        isPresent=$(/etc/data/lanUtils.sh util_check_single_dns_present "$dns" $resolve_file)
        if [ $default_pdn -eq $profile ]; then
          proto_add_ipv4_route "$dns" 32 "" "" ""
        elif [ -n "$isPresent" -a $isPresent -eq 1 ]; then
          proto_add_ipv4_route "$dns" 32 "" "" "$metric_counter"
          uci set pbr.config.metric_counter4="$metric_counter"
          uci commit pbr
        else
          proto_add_ipv4_route "$dns" 32 "" "" "1"
        fi
    done

    #Reversing DNS addresses for default pdn so they get added to dnsmasq resolv file in correct order
    if [ $default_pdn -eq $profile ]; then
      DNS=$(echo "$DNS"| awk '{ for (i=NF; i>1; i--) printf("%s ",$i); print $1; }')
    fi

    for dns in $DNS; do
            for i in $downstream; do
                resolve_file=$(uci get dhcp."$i"_dns.resolvfile)

                #NOTE: check if the DNS entries are already present in resolv file
                isPresent=$(/etc/data/lanUtils.sh util_check_single_dns_present "$dns #${iface_name}" $resolve_file)
                if [ -n "$isPresent" -a $isPresent -eq 1 ]; then
                    #Indicates dnsserver information is already present in file
                    logger $(basename "$0")":util_add_dnsv4:"$LINENO": dnsserver information is already present in file"
                    continue
                fi

                #Else, add dns entry to resolve file
                #append to top if default pdn
                if [ $default_pdn -eq $profile ]; then
                    echo "#test" >>  $resolve_file
                    sed -i "1inameserver $dns #${iface_name}" $resolve_file
                    sed -i '/#test/d' $resolve_file
                else
                    echo "nameserver $dns #${iface_name}" >> $resolve_file
                fi
            done
    done

    #Save dns information in /tmp/state/network
    for i in $downstream; do
        uci_set_state network ${i} "dns4_${profile}" "$DNS"
    done

}

#Function to Inform NetIfD of DNS information
#$1 - Profile ID
#$2 - DNS Server IP address
function util_add_dnsv6_pbr() {
  local profile="$1"
  local DNS6="$2"
  local downstream proxydnsv6 resolve_file
  local default_pdn isPresent
  local bridge_addr6
  local ula parsed_ula
  local metric_counter

  #Get downstream values
  downstream="lan"
  default_pdn=$(/etc/data/lanUtils.sh util_get_default_pdn)
  if [ $profile -eq 1 ]; then
    iface_name="wan_v6"
  else
    iface_name="wan${profile}_v6"
  fi

  metric_counter=$(uci get pbr.config.metric_counter6)
  if [ $metric_counter != $MAX_METRIC ]; then
    metric_counter=$((metric_counter+1))
  fi
  resolve_file=$(uci get dhcp."$downstream"_dns.resolvfile)
  for dns in $DNS6; do
    proto_add_dns_server "$dns"
    isPresent=$(/etc/data/lanUtils.sh util_check_single_dns_present "$dns" $resolve_file)
    if [ $default_pdn -eq $profile ]; then
      proto_add_ipv6_route "$dns" 128 "" "1" "" "" ""
    elif [ -n "$isPresent" -a $isPresent -eq 1 ]; then
      proto_add_ipv6_route "$dns" 128 "" "$metric_counter" "" "" ""
      uci set pbr.config.metric_counter6="$metric_counter"
      uci commit pbr
    else
      proto_add_ipv6_route "$dns" 128 "" "2" "" "" ""
    fi
  done

  if [ $default_pdn -eq $profile ]; then
    DNS6=$(echo "$DNS6"| awk '{ for (i=NF; i>1; i--) printf("%s ",$i); print $1; }')
  fi

  for dns in $DNS6; do
    logger $(basename "$0")":util_add_dnsv6:"$LINENO": proxydnsv6:$proxydnsv6"
    for i in $downstream; do
      #Condition to check for NULL DNS entry
      if [ "$dns" = "::" ]; then
        logger $(basename "$0")":util_add_dnsv6:"$LINENO":NULL dns entry found"
      else
        resolve_file=$(uci get dhcp."$i"_dns.resolvfile)
        #NOTE: check if the DNS entries are already present in resolv file
        isPresent=$(/etc/data/lanUtils.sh util_check_single_dns_present "$dns #${iface_name}" $resolve_file)
        if [ -n "$isPresent" -a $isPresent -eq 1 ]; then
          #Indicates dnsserver information is already present in file
          logger $(basename "$0")":util_add_dnsv6:"$LINENO": dnsserver information is already present in file"
          continue
        fi

        if [ $default_pdn -eq $profile ]; then
          echo "#test" >>  $resolve_file
          sed -i "1inameserver $dns #${iface_name}" $resolve_file
          sed -i '/#test/d' $resolve_file
        else
          echo "nameserver $dns #${iface_name}" >> $resolve_file
        fi
        ula=$(uci get qcmap_current_lan_config.lan.ula_prefix)
        parsed_ula=$(echo $ula | awk -F'::' '{print $1}')
        bridge_addr6=$(ip addr show br-lan | grep "$parsed_ula" | sed -E 's/.*inet6 ([^/]+)\/.*/\1/')
        prev_dns=$(uci get dhcp.$i.dns)
        if [ -z $prev_dns ] || [ $prev_dns != $bridge_addr6 ]; then
          uci del_list dhcp.$i.dns="$prev_dns"
          uci add_list dhcp.$i.dns="$bridge_addr6"
        fi
      fi
    done
    uci commit dhcp
  done

  #Save dns information in /tmp/state/network
  for i in $downstream; do
      uci_set_state network ${i} "dns6_${profile}" "$DNS6"
  done
}

#Function to add MTU options on LAN side
# $1 - Profile id
# $2 - IPtype
function util_add_mtu_options_pbr() {
    local profile=$1
    local iptype=$2
    local wan


    if [ -n "$profile" ] && [ -n "$iptype" ];
    then
        logger $(basename "$0")":util_add_mtu_options_pbr:"$LINENO": $profile $iptype"
    else
        logger $(basename "$0")":util_add_mtu_options_pbr"$LINENO": missing args, exit"
        exit 1
    fi

    if [ "$iptype" -ne "4" ] && [ "$iptype" -ne "6" ];
    then
        logger $(basename "$0")":util_add_mtu_options_pbr:"$LINENO": invalid ip_type:$iptype "
        exit 1
    fi

    logger $(basename "$0")":util_add_mtu_options_pbr:"$LINENO": profile :$profile iptype: $iptype"

    if [ $profile -eq 1 ]; then
        wan="wan"
    else
        wan="wan$profile"
    fi

    # source the config file
    [ -f /tmp/ipv4config$profile ] && . /tmp/ipv4config$profile
    [ -f /tmp/ipv6config$profile ] && . /tmp/ipv6config$profile
    ip4mtu=$IPV4MTU
    ip6mtu=$IPV6MTU

    if [ "$iptype" -eq "4" -a -n "$ip4mtu" ] || [ "$iptype" -eq "6" -a -n "$ip6mtu" ]; then

        #Update MTU for v4 and v6 on WAN side
        if [ "$iptype" -eq "4" ]; then
             uci set network."$wan".mtu=$ip4mtu
        else
             uci set network."$wan"_v6.mtu=$ip6mtu
        fi

        local downstream
        downstream="lan"
        least_mtu=$(/etc/data/lanUtils.sh util_get_min_pdu_mtu $iptype)
        if [ "$iptype" -eq "4" ]; then
            prev_ip4mtu=$(uci get dhcp.lan.dhcp_option_force | grep -oE '[^ ]*' | grep -o 26,.* | cut -d "," -f2)
            if [ -n "$prev_ip4mtu" ]; then
                uci del_list dhcp.lan.dhcp_option_force='26,'$prev_ip4mtu''
            fi
            if [ -n "$least_mtu" ]; then
                ip4mtu=$least_mtu
            fi
        fi

        [ -n "$downstream" ] && {
            logger $(basename "$0")":util_add_mtu_options_pbr:"$LINENO": adding MTU:$ip4mtu/$ip6mtu for $downstream"
            for i in $downstream; do
                    if [ "$iptype" -eq "4" ]; then
                         uci add_list dhcp.$i.dhcp_option_force='26,'$ip4mtu''
                    fi
                    if [ "$iptype" -eq "6" ]; then
                         if [ -n "$least_mtu" ]; then
                             uci set dhcp.$i.ra_mtu=$least_mtu
                         else
                             uci set dhcp.$i.ra_mtu=$ip6mtu
                         fi
                    fi
            done
            uci commit dhcp
            }
        uci commit network
    else
        logger $(basename "$0")"util_add_mtu_options_pbr:"$LINENO": NO MTU found in file"
    fi
}

#Function to delete MTU options on LAN side
# $1 - Profile id
# $2 - IPtype
function util_delete_mtu_options_pbr() {
    local profile=$1
    local iptype=$2
    local wan

    if [ -n "$profile" ] && [ -n "$iptype" ];
    then
         logger $(basename "$0")":util_delete_mtu_options_pbr $1 $2 "
    else
        logger $(basename "$0")":util_delete_mtu_options_pbr missing args, exit"
        exit 1
    fi

    if [ "$iptype" -ne "4" ] && [ "$iptype" -ne "6" ];
    then
        logger $(basename "$0")":util_delete_mtu_options_pbr invalid ip_type:$iptype "
        exit 1
    fi

    if [ $profile -eq 1 ]; then
        wan="wan"
    else
        wan="wan$profile"
    fi

    #load MTU from WAN options set
    ip4mtu=$(uci get network."$wan".mtu)
    ip6mtu=$(uci get network."$wan"_v6.mtu)

    if [ "$iptype" -eq "4"  -a -n "$ip4mtu" ] || [ "$iptype" -eq "6" -a -n "$ip6mtu" ]; then

        #Delete MTU for v4 and v6 on WAN side
        if [ "$iptype" -eq "4" ]; then
            uci del network."$wan".mtu=$ip4mtu
        else
            uci del network."$wan"_v6.mtu=$ip6mtu
        fi

        local downstream
        downstream="lan"
        least_mtu=$(/etc/data/lanUtils.sh util_get_min_pdu_mtu $iptype)
        if [ "$iptype" -eq "4" ]; then
            prev_ip4mtu=$(uci get dhcp.lan.dhcp_option_force | grep -oE '[^ ]*' | grep -o 26,.* | cut -d "," -f2)
            if [ -n "$prev_ip4mtu" ]; then
                ip4mtu=$prev_ip4mtu
            fi
        fi

        #Delete MTU for v4 & V6
        [ -n "$downstream" ] && {
         for i in $downstream; do
            if [ "$iptype" -eq "4" ]; then
                uci del_list dhcp.$i.dhcp_option_force='26,'$ip4mtu''
                if [ -n "$least_mtu" ]; then
                    uci add_list dhcp.$i.dhcp_option_force='26,'$least_mtu''
                fi
            fi
            if [ "$iptype" -eq "6" ]; then
                uci del dhcp.$i.ra_mtu=$ip6mtu
                if [ -n "$least_mtu" ]; then
                    uci set dhcp.$i.ra_mtu=$least_mtu
                fi
            fi
            done
            uci commit dhcp
        }
        uci commit network
    else
        logger $(basename "$0")": NO MTU found in file"
    fi
}

#Function to setup routes and rules for Wan side network
#$1 - Wan side section name
#$2 - rmnet_dataX interface name
#$3 - Dedicated Routing Table ID
#$4 - Profile ID
#$5 - MTU
function util_setup_wan_route_rule_v6_pbr() {
	local cfg="$1"
	local interface="$2"
	local dedicated_rt="$3"
	local profile="$4"
	local mtu="$5"
	local ip6

	#Source config file
	[ -f /tmp/ipv6config$profile ] && . /tmp/ipv6config$profile
	ip6=$PUBLIC_IP6

	ip "-6" rule del oif $interface
	ip "-6" rule add oif $interface table $dedicated_rt prio 32000
	if [ -n "$mtu" ]; then
		ip "-6" route add default dev $interface table $dedicated_rt mtu lock $mtu
	else
		ip "-6" route add default dev $interface table $dedicated_rt
	fi

	#Save the Generated IP address to /tmp/state/network
	uci_set_state network $cfg upip6 $ip6

}


#$1 - qcmap_topology
function update_firewall_on_topology_change()
{
  mode=$1
  no_of_profiles=$(uci get qcmap_lan.@no_of_configs[0].no_of_profiles)
  for profile_id in $(seq 1 $((no_of_profiles)))
  do
    if [ $profile_id -eq 1 ]; then
      lan_zone_idx=$(uci show firewall | grep -i "zone" | grep -w -i "lan_wan" | awk -F'[][]' '{print $2}')
    else
      lan_zone_idx=$(uci show firewall | grep -i "zone" | grep -w -i "lan_wan$profile_id" | awk -F'[][]' '{print $2}')
    fi

    if [ $profile_id -ne 1 ]; then
      lan_iface_list=$(uci get firewall.@zone[$lan_zone_idx].network)
      is_lan_in_list=$(echo "$lan_iface_list" | grep -w -c lan)
      if [ $mode = $QCMAP_SINGLE_BRIDGE_TOPOLOGY ] && [ $is_lan_in_list -eq 0 ]; then
        uci add_list firewall.@zone[$lan_zone_idx].network=lan
      elif [ $mode != $QCMAP_SINGLE_BRIDGE_TOPOLOGY ] && [ $is_lan_in_list -eq 1 ]; then
        uci del_list firewall.@zone[$lan_zone_idx].network=lan
      fi
    fi
  done
}


case $1 in
  start_pbr)
    echo "Start Pbr"
    start_pbr
    ;;
  stop_pbr)
    echo "Stop Pbr"
    stop_pbr
    ;;
  add_pbr_policy)
    echo "Add Pbr policy"
    saved_param="$2"
    saved_param2="$3"
    saved_param3="$4"
    saved_param4="$5"
    saved_param5="$6"
    shift 5
    add_pbr_policy "$saved_param" "$saved_param2" "$saved_param3" "$saved_param4" "$saved_param5" "$2" "$3" "$4" "$5" "$6" "$7" "$8" "$9"
    ;;
  del_pbr_policy)
    del_pbr_policy $2 $3
    ;;
  add_map_policy_to_pdn)
    add_map_policy_to_pdn $2 $3
    ;;
  delete_map_policy_to_pdn)
    delete_map_policy_to_pdn $2
    ;;
  get_no_of_policies)
    util_get_num_of_policies
    ;;
  pbr_data_call_connected_event)
    pbr_data_call_connected_event $2 $3 $4
    ;;
  update_pbr_policies_on_wlan_event)
    update_pbr_policies_on_wlan_event $2 $3 "$4"
    ;;
  delete_conntrack_on_wan_event_pbr)
    delete_conntrack_on_wan_event_pbr $2
    ;;
  update_all_pbr_policies_on_wlan_event)
    update_all_pbr_policies_on_wlan_event
    ;;
  update_firewall_on_topology_change)
    update_firewall_on_topology_change $2
    ;;
  *)
    logger "Invalid option"
    ;;
esac

