#!/bin/sh
#  Copyright (c) 2022 Qualcomm Technologies, Inc.
#  All Rights Reserved.
#  Confidential and Proprietary - Qualcomm Technologies, Inc.
#  Usage: Add Firewall Config on OpenWRT

# Usage: firewallConfig.sh SET_FIREWALL <enable_firewall> <pkts_allowed> <profile_handle>
# Usage: firewallConfig.sh ENABLE_FIREWALL <profile_handle>
# Usage: firewallConfig.sh DISABLE_FIREWALL <profile_handle>
# Usage: firewallConfig.sh DELETE_FIREWALL <firewall_handle> profile_handle
# Usage: firewallConfig.sh DISPLAY_FIREWALL <ip_version>
# Usage: firewallConfig.sh set_hw_filtering

#Usage: For logging
source /etc/data/mbbUtils.sh

#source common functions
. /etc/data/lanUtils.sh

if [[ "$#" -le 0 ]]; then
  echo "Check usage ./firewallConfig.sh set_firewall enable_firewall pkts_allowed profile_handle"
  echo "Check usage ./firewallConfig.sh enable_firewall profile_handle bh_type"
  echo "Check usage ./firewallConfig.sh disable_firewall profile_handle bh_type"
  echo "Check usage ./firewallConfig.sh delete_firewall firewall_handle profile_handle"
  echo "Check usage ./firewallConfig.sh display_firewall ip_version"
  echo "Check usage ./firewallConfig.sh set_hw_filtering"
  return 1
elif [[ \( "$1" == "set_firewall" \) && \( "$#" -ne 4 \) ]]; then
  echo "Check usage ./firewallConfig.sh set_firewall enable_firewall pkts_allowed profile_handle"
  return 1
elif [[ \( "$1" == "enable_firewall" \) && \( "$#" -ne 3 \) ]]; then
  echo "Check usage ./firewallConfig.sh enable_firewall profile_handle bh_type"
  return 1
elif [[ \( "$1" == "disable_firewall" \) && \( "$#" -ne 3 \) ]]; then
  echo "Check usage ./firewallConfig.sh disable_firewall profile_handle bh_type"
  return 1
elif [[ \( "$1" == "delete_firewall" \) && \( "$#" -ne 3 \) ]]; then
  echo "Check usage ./firewallConfig.sh delete_firewall firewall_handle profile_handle"
  return 1
elif [[ \( "$1" == "display_firewall" \) && \( "$#" -ne 2 \) ]]; then
  echo "Check usage ./firewallConfig.sh display_firewall ip_version"
  return 1
elif [[ \( "$1" == "set_hw_filtering" \) && \( "$#" -ne 1 \) ]]; then
  echo "Check usage ./firewallConfig.sh set_hw_filtering "
  return 1
fi

#Function to Check whether any v4 firewall rule is configured with "Any" as
#a parameter
#$1 -> Profile handle
function check_firewall_rule_with_parameters_as_any(){
  local is_any_configured=0

  local firewall_search_string="FirewallV4-$1"
  v4_firewall_indexes=`uci show qcmap_firewall | grep -i "$firewall_search_string" | awk -F'[][]' '{print $2}'`

  for i in $v4_firewall_indexes
  do
    if [ $is_any_configured -eq 0 ]; then
      src_addr=$(uci get qcmap_firewall.@firewall_rule[$i].src_addr)
      dest_addr=$(uci get qcmap_firewall.@firewall_rule[$i].dest_addr)

      if [ "$src_addr" = "Any" ] && [ "$dest_addr" = "Any" ];then
        is_any_configured=1
      fi

      proto=$(uci get qcmap_firewall.@firewall_rule[$i].proto)

      case "$proto" in
        "tcp")
           src_port=$(uci get qcmap_firewall.@firewall_rule[$i].src_port)
           dest_port=$(uci get qcmap_firewall.@firewall_rule[$i].dest_port)
           if [ "$src_port" = "Any" ] && [ "$dest_port" = "Any" ];then
             is_any_configured=$(($is_any_configured&1))
           fi
           ;;
        "udp")
           src_port=$(uci get qcmap_firewall.@firewall_rule[$i].src_port)
           dest_port=$(uci get qcmap_firewall.@firewall_rule[$i].dest_port)
           if [ "$src_port" = "Any" ] && [ "$dest_port" = "Any" ];then
             is_any_configured=$(($is_any_configured&1))
           fi
           ;;
        "tcpudp")
           src_port=$(uci get qcmap_firewall.@firewall_rule[$i].src_port)
           dest_port=$(uci get qcmap_firewall.@firewall_rule[$i].dest_port)
           if [ "$src_port" = "Any" ] && [ "$dest_port" = "Any" ];then
             is_any_configured=$(($is_any_configured&1))
           fi
           ;;
        "icmp")
           ;;
      esac
    fi
  done

  echo $is_any_configured

}

#Function to Check and delete external v4 conntrack entries if required.
#$1 -> Profile handle
function check_external_ipv4_conntracks(){
  local ul_count=0
  local dl_count=0
  local is_any_configured=0

  firewall_search_string="FirewallV4-$1"
  v4_firewall_indexes=`uci show qcmap_firewall | grep -i "$firewall_search_string" | awk -F'[][]' '{print $2}'`

  for i in $v4_firewall_indexes
  do
    fw_direction=$(uci get qcmap_firewall.@firewall_rule[$i].direction)
    if [ "$fw_direction" = "UL" ];then
      ul_count=$((ul_count+1))
    elif [ "$fw_direction" = "DL" ];then
      dl_count=$((dl_count+1))
    fi
  done

  is_any_configured=`check_firewall_rule_with_parameters_as_any $1`

  if [ $is_any_configured -eq 0 ]; then
    log  $(basename "$0") "check_external_ipv4_conntracks" $LINENO "Firewall entries are not configured to ANY type. exiting.."
    return
  fi

  if [ $ul_count -eq 0 ] && [ $dl_count -ne 0 ]; then
    allowed_fw_dir=QCMAP_MSGR_DL_FIREWALL
  elif [ $dl_count -eq 0 ] && [ $ul_count -ne 0 ]; then
    allowed_fw_dir=QCMAP_MSGR_UL_FIREWALL
  else
    log  $(basename "$0") "check_external_ipv4_conntracks" $LINENO "Firewall IPV4 not configured in single direction. exiting.."
    return
  fi

  #Getting profile index from qcmap_lan
  profile_idx=`uci show qcmap_lan | grep -i "profile_id='$1'" | awk -F'[][]' '{print $2}'`
  if [ "$profile_idx" ]; then
    pkt_allowed=$(uci get qcmap_lan.@profile[$profile_idx].pkts_allowed)
    if [ $pkts_allowed -eq 0 ]; then
      #Reverse allowed firewall direction if packets are not allowed in current direction(Blacklisting case).
      if [ $allowed_fw_dir = "QCMAP_MSGR_DL_FIREWALL" ];then
        allowed_fw_dir=QCMAP_MSGR_UL_FIREWALL
      elif [ $allowed_fw_dir = "QCMAP_MSGR_UL_FIREWALL" ];then
        allowed_fw_dir=QCMAP_MSGR_DL_FIREWALL
      fi
    fi
  fi

  get_ipv4v6config_file $1

  if [ -f "$V4_config_file" ]; then
      PUBLIC_IP_ADDR=`awk -F "=" 'FNR == 2 {print $2}' $V4_config_file | tr -d '"'`
  fi

  if [ $allowed_fw_dir = "QCMAP_MSGR_DL_FIREWALL" ];then
      conntrack -D --reply-dst $PUBLIC_IP_ADDR
  elif [ $allowed_fw_dir = "QCMAP_MSGR_UL_FIREWALL" ];then
      conntrack -D --orig-dst $PUBLIC_IP_ADDR
  fi

}

#Function to convert ipv4 to hex
function ip_to_hex() {
  printf '%02X' ${1//./ }
}

#convert ipv4 ip from string to integer
function string_to_int_ip() {
  if [ "$1" ];then
    result=`valid_ipv4 $1`
    if [ $result = "success" ];then
      local ip=$1
      first_part=$(printf '%s\n' "$ip" |  awk -F'.' '{print $1}')
      second_part=$(printf '%s\n' "$ip" |  awk -F'.' '{print $2}')
      third_part=$(printf '%s\n' "$ip" |  awk -F'.' '{print $3}')
      fourth_part=$(printf '%s\n' "$ip" |  awk -F'.' '{print $4}')

      first_multiplier=$((${first_part}*16777216))
      second_multiplier=$((${second_part}*65536))
      third_multiplier=$((${third_part}*256))
      fourth_multiplier=$((${fourth_part}*1))

      converted_int=$((first_multiplier+second_multiplier+third_multiplier+fourth_multiplier))
      echo $converted_int
    fi
  fi
}

# helper to convert hex to dec
hex2dec(){
    [ "$1" != "" ] && printf "%d" "$(( 0x$1 ))"
}

# expand an ipv6 address
expand_ipv6() {
    ip=$1

    # prepend 0 if we start with :
    echo $ip | grep -qs "^:" && ip="0${ip}"

    # expand ::
    if echo $ip | grep -qs "::"; then
        colons=$(echo $ip | sed 's/[^:]//g')
        missing=$(echo ":::::::::" | sed "s/$colons//")
        expanded=$(echo $missing | sed 's/:/:0/g')
        ip=$(echo $ip | sed "s/::/$expanded/")
    fi

    blocks=$(echo $ip | grep -o "[0-9a-f]\+")
    set $blocks

    printf "%04x:%04x:%04x:%04x:%04x:%04x:%04x:%04x\n" \
        $(hex2dec $1) \
        $(hex2dec $2) \
        $(hex2dec $3) \
        $(hex2dec $4) \
        $(hex2dec $5) \
        $(hex2dec $6) \
        $(hex2dec $7) \
        $(hex2dec $8)
}

#Function hex2binary: Hex to Binary Conversion.
function hex2binary() {
if [ $# -eq 0 ]
then
    echo "Argument(s) not supplied "
    echo "Usage: hex2binary.sh hex_number(s)"
else

    while [ $# -ne 0 ]
    do
        DecNum=`printf "%d" 0x$1`
        Binary=
        Number=$DecNum

        while [ $DecNum -ne 0 ]
        do
            Bit=$(expr $DecNum % 2)
            Binary=$Bit$Binary
            DecNum=$(expr $DecNum / 2)
        done


        printf "%016d\n" \
           ${Binary:-0}

        shift
        # Shifts command line arguments one step.Now $1 holds second argument
        unset Binary
    done

fi
}

#Function to Delete all the conntrack entries for external Ipv4 communication
#$1 is wan profile handle
function delete_external_ipv4_conntracks(){
  get_ipv4v6config_file $1
  if [ -f "$V4_config_file" ];then
    PUBLIC_IP_ADDR=`awk -F "=" 'FNR == 2 {print $2}' $V4_config_file | tr -d '"'`
    conntrack -D --orig-dst $PUBLIC_IP_ADDR
    conntrack -D --reply-dst $PUBLIC_IP_ADDR
  fi
}

#Usage : Converts unit32 to ip addr format
#$1 - ip
function uint_to_ipv4() {
  local ip=$1
  echo "$(( ($ip >> 24) % 256 )).$(( ($ip >> 16) % 256 )).$(( ($ip >> 8) % 256 )).$(( $ip % 256 ))"
}

#$1 - subnet mask in uint32 form
function count_set_bits_in_netmask(){
  local subnet_uint=$1
  local count=0
    while [ $subnet_uint -gt 0 ]
    do
            count=$((count+(subnet_uint&1)))
            subnet_uint=$((subnet_uint >> 1))
    done
    echo $count
}


#Function to Delete the gayteway ip from mwan3_connected_ipv4 ipset.
#mwan3_connected_ipv4 ipset is having a ipset of rmnet network. Because
#of this packets are hitting mwan3_connected_ipv4 hook, but ideally it should
#hit mwan3_backhaul_policy_pref when data tranfer server ip is in rmnet network.
#$1 v4_config_file
function delete_ipset_for_rmnet_gateway(){
  V4_config_file=$1
  if [ -f "$V4_config_file" ];then
    rmnet_addr=`awk -F "=" 'FNR == 2 {print $2}' $V4_config_file | tr -d '"'`
    gateway_ip=`awk -F "=" 'FNR == 5 {print $2}' $V4_config_file | tr -d '"'`
    subnet=`awk -F "=" 'FNR == 3 {print $2}' $V4_config_file | tr -d '"'`

    rmnet_addr_uint=`string_to_int_ip $rmnet_addr`
    subnet_uint=`string_to_int_ip $subnet`
    network_id_uint=$((rmnet_addr_uint&subnet_uint))
    network_id=`uint_to_ipv4 $network_id_uint`
    subnet_in_cidr=`count_set_bits_in_netmask $subnet_uint`

    log  $(basename "$0") "delete_ipset_for_rmnet_gateway" $LINENO "gateway_ip=$gateway_ip  network_id=$network_id subnet_in_cidr=$subnet_in_cidr" 

    #executing ipset deletion commands
    ipset del mwan3_connected_ipv4 $gateway_ip
    ipset del mwan3_connected_ipv4 $network_id/$subnet_in_cidr
  fi
}

#Utility function to Delete the client conntrack when we are adding a DROP firewall entry for
#that IPv6 address and port combination
#$1 is index in qcmap_firewall
#$2 is protocol
function delete_conntrack_entry_for_drop_ipv6_firewall_entries_utility() {
  local index=$1
  local protocol=$2
  local check_ip=false
  local check_sport=false
  local check_dport=false

  local conntrack_cmd1=""
  local conntrack_cmd2=""
  local conntrack_cmd3=""
  local conntrack_cmd4=""
  local conntrack_cmd5=""

  local dcmd1=""
  local dcmd2=""
  local dcmd3=""
  local dcmd4=""
  local dcmd5=""
  local dcmd6=""


  log  $(basename "$0") "delete_conntrack_entry_for_drop_ipv6_firewall_entries_utility" $LINENO "Entering delete_conntrack_entry_for_drop_ipv6_firewall_entries_utility function"

  if [ $protocol = "tcp" ]; then
    conntrack_cmd1="conntrack -L -f ipv6 | cut -f1,10,11,12,13 -d ' ' | grep -v dport=53 "
  elif [ $protocol = "udp" ]; then
    conntrack_cmd1="conntrack -L -f ipv6 | cut -f1,9,10,11,12 -d ' ' | grep -v dport=53  "
  elif [ $protocol = "icmp" ]; then
    conntrack_cmd1="conntrack -L -f ipv6 | cut -f1,6,7 -d ' ' "
  fi

  #src_ipV6 address is the address we have from firewall config
  src_ipv6=$(uci get qcmap_firewall.@firewall_rule[$index].src_addr)

  if [ $src_ipv6 != "Any" ]; then
    src_prefix_lengthv6=$(uci get qcmap_firewall.@firewall_rule[$index].src_prefix_length)
    prefixWords_src_addr=$((src_prefix_lengthv6/16))
    prefixBits_src_addr=$((src_prefix_lengthv6%16))

    if [ $prefixBits_src_addr -eq 0 ] && [ $prefixWords_src_addr -eq 8 ];then
      processV6srcaddr=false
      conntrack_cmd2="| grep src=$src_ipv6"
    else
      processV6srcaddr=true
    fi
  fi

  #dest_ipV6 address is the address we have from firewall config
  dest_ipv6=$(uci get qcmap_firewall.@firewall_rule[$index].dest_addr)

  if [ $dest_ipv6 != "Any" ]; then
    dest_prefix_lengthv6=$(uci get qcmap_firewall.@firewall_rule[$index].dest_prefix_length)
    prefixWords_dest_addr=$((dest_prefix_lengthv6/16))
    prefixBits_dest_addr=$((dest_prefix_lengthv6%16))

    if [ $prefixBits_dest_addr -eq 0 ] && [ $prefixWords_dest_addr -eq 8 ];then
      processV6destaddr=false
      conntrack_cmd2="| grep dst=$dest_ipv6"
    else
      processV6destaddr=true
    fi
  fi

  case "$protocol" in
    "tcp")
      conntrack_cmd3="| grep tcp "
      src_port=$(uci get qcmap_firewall.@firewall_rule[$index].src_port)
      if [ "$src_port" ] && [ "$src_port" != "Any" ]; then
        src_port_range=$(uci get qcmap_firewall.@firewall_rule[$index].src_port_range)
        if [ $src_port_range -eq 0 ]; then
          conntrack_cmd4="| grep sport=$src_port "
        else
          min_sport=${src_port}
          max_sport=$(($src_port + $src_port_range))
          check_sport=true;
        fi
      fi
      dest_port=$(uci get qcmap_firewall.@firewall_rule[$index].dest_port)
      if [ "$dest_port" ] && [ "$dest_port" != "Any" ]; then
        dest_port_range=$(uci get qcmap_firewall.@firewall_rule[$index].dest_port_range)
        if [ $dest_port_range -eq 0 ]; then
          conntrack_cmd5="| grep dport=$dest_port "
        else
          min_dport=${dest_port}
          max_dport=$(($dest_port + $dest_port_range))
          check_dport=true;
        fi
      fi
      ;;
    "udp")
      conntrack_cmd3="| grep udp "
      src_port=$(uci get qcmap_firewall.@firewall_rule[$index].src_port)
      if [ "$src_port" ] && [ "$src_port" != "Any" ]; then
        src_port_range=$(uci get qcmap_firewall.@firewall_rule[$index].src_port_range)
        if [ $src_port_range -eq 0 ]; then
          conntrack_cmd4="| grep sport=$src_port "
        else
          min_sport=${src_port}
          max_sport=$(($src_port + $src_port_range))
          check_sport=true;
        fi
      fi
      dest_port=$(uci get qcmap_firewall.@firewall_rule[$index].dest_port)
      if [ "$dest_port" ] && [ "$dest_port" != "Any" ]; then
        dest_port_range=$(uci get qcmap_firewall.@firewall_rule[$index].dest_port_range)
        if [ $dest_port_range -eq 0 ]; then
          conntrack_cmd5="| grep dport=$dest_port "
        else
          min_dport=${dest_port}
          max_dport=$(($dest_port + $dest_port_range))
          check_dport=true;
        fi
      fi
      ;;
    "icmp")
      conntrack_cmd3="| grep icmpv6 "
      ;;
  esac

  command1=$conntrack_cmd1$conntrack_cmd2$conntrack_cmd3$conntrack_cmd4$conntrack_cmd5
  log $(basename "$0") "conntrack display ipv6 drop cmd : " $LINENO "$command1"
  touch /tmp/data/v6conntrack.txt
  eval $command1 > /tmp/data/v6conntrack.txt

  file="/tmp/data/v6conntrack.txt"

  if [ -f ${file} ]
  then
      if [ -s ${file} ]
      then
        echo "v6conntrack.txt File exists and not empty"
        #reading v6conntrack.txt file line-by-line
        while read -r line; do
          readed_line=$(echo $line)
          #s_ipaddr is the ip address that we get by v6conntrack.txt file line-by-line
          #proto is the protocol that we get by v6conntrack.txt file line-by-line
          #sport is the source port that we get by v6conntrack.txt file line-by-line
          #dport is the destination port that we get by v6conntrack.txt file line-by-line

          proto=$(printf '%s\n' "$readed_line" |  awk '{print $1}')
          s_ipaddr=$(printf '%s\n' "$readed_line" |  awk '{print $2}' | cut -c 5-)
          d_ipaddr=$(printf '%s\n' "$readed_line" |  awk '{print $3}' | cut -c 5-)
          if [ $protocol != "icmp" ]; then
            dport=$(printf '%s\n' "$readed_line" |  awk '{print $5}' | cut -c 7-)
            sport=$(printf '%s\n' "$readed_line" |  awk '{print $4}' | cut -c 7-)
          fi

          #Check for ipv6 src address and prefix length
          if [ "$processV6srcaddr" = "true" ];then
            if [ "$s_ipaddr" ] && [ "$src_ipv6" != "Any" ];then
              cmp_len_src=$(( $((prefixWords_src_addr-1)) + ( $prefixWords_src_addr * 4 ) ))
              expand_s_ipaddr=`expand_ipv6 $s_ipaddr`
              cmp_s_ipaddr=`echo $expand_s_ipaddr | cut -c 1-$cmp_len_src`
              expand_src_ipv6=`expand_ipv6 $src_ipv6`
              cmp_src_ipv6=`echo $expand_src_ipv6 | cut -c 1-$cmp_len_src`
              if [ "$cmp_s_ipaddr" != "$cmp_src_ipv6" ];then
                skipPrefixBitProcessForSrc=true
              fi
            fi
            if [ $skipPrefixBitProcessForSrc = "true" ];then
              continue
            else
              hex1_after_prefixWords_src_addr=`echo $expand_s_ipaddr | cut -c $((cmp_len_src+2))-$((cmp_len_src+5))`
              hex2_after_prefixWords_src_addr=`echo $expand_src_ipv6 | cut -c $((cmp_len_src+2))-$((cmp_len_src+5))`
              binary1_src_addr=`hex2binary $hex1_after_prefixWords_src_addr`
              binary2_src_addr=`hex2binary $hex2_after_prefixWords_src_addr`
              binary1_prefixBits_src_addr=`echo $binary1_src_addr | cut -c 1-$prefixBits_src_addr`
              binary2_prefixBits_src_addr=`echo $binary2_src_addr | cut -c 1-$prefixBits_src_addr`
              if [ "$binary1_prefixBits_src_addr" != "$binary2_prefixBits_src_addr" ];then
                continue;
              fi
            fi

          fi

          #Check for ipv6 dest address and dest address prefix length
          if [ "$processV6destaddr" = "true" ];then
            if [ "$s_ipaddr" ] && [ "$dest_ipv6" != "Any" ];then
              cmp_len_dest=$(( $((prefixWords_dest_addr-1)) + ( $prefixWords_dest_addr * 4 ) ))
              expand_d_ipaddr=`expand_ipv6 $d_ipaddr`
              cmp_d_ipaddr=`echo $expand_d_ipaddr | cut -c 1-$cmp_len_dest`
              expand_dest_ipv6=`expand_ipv6 $dest_ipv6`
              cmp_dest_ipv6=`echo $expand_dest_ipv6 | cut -c 1-$cmp_len_dest`
              if [ "$cmp_d_ipaddr" != "$cmp_dest_ipv6" ];then
                skipPrefixBitProcessForDest=true
              fi
            fi
            if [ $skipPrefixBitProcessForDest = "true" ];then
              continue
            else
              hex1_after_prefixWords_dest_addr=`echo $expand_d_ipaddr | cut -c $((cmp_len_dest+2))-$((cmp_len_dest+5))`
              hex2_after_prefixWords_dest_addr=`echo $expand_dest_ipv6 | cut -c $((cmp_len_dest+2))-$((cmp_len_dest+5))`
              binary1_dest_addr=`hex2binary $hex1_after_prefixWords_dest_addr`
              binary2_dest_addr=`hex2binary $hex2_after_prefixWords_dest_addr`
              binary1_prefixBits_dest_addr=`echo $binary1_dest_addr | cut -c 1-$prefixBits_dest_addr`
              binary2_prefixBits_dest_addr=`echo $binary2_dest_addr | cut -c 1-$prefixBits_dest_addr`
              if [ "$binary1_prefixBits_dest_addr" != "$binary2_prefixBits_dest_addr" ];then
                continue;
              fi
            fi

          fi

          if [ $protocol != "icmp" ];then
            #Check for source port range
            if [ $check_sport = "true" ];then
               if [ $sport -lt $min_sport ] || [ $sport -gt $max_sport ];then
                 continue
               fi
            fi

            #Check for destination port range
            if [ $check_dport = "true" ];then
               if [ $dport -lt $min_dport ] || [ $dport -gt $max_dport ];then
                 continue
               fi
            fi
          fi

          #creation of deletion command
          dcmd1="conntrack -D -f ipv6"
          if [ "$proto" ];then
            dcmd2=" -p $proto"
          fi
          if [ "$s_ipaddr" ];then
            dcmd3="  --orig-src $s_ipaddr"
          fi
          if [ "$d_ipaddr" ];then
            dcmd4="  --orig-dst $d_ipaddr"
          fi
          if [ "$sport" ];then
            dcmd5=" --sport $sport"
          fi
          if [ "$dport" ];then
            dcmd6=" --dport $dport"
          fi

          deletion_final_command=$dcmd1$dcmd2$dcmd3$dcmd4$dcmd5$dcmd6

          echo "Final Deletion cmd is:" $deletion_final_command

          eval $deletion_final_command

        done <$file

        rm -rf /tmp/data/v6conntrack.txt
      else
        echo "v6conntrack.txt file exists but empty"
      fi
  else
      echo "v6conntrack.txt file not exists"
  fi
}

#Function to create & execute the delete command to delete conntrack v6 entries in case of ACCEPT
function delete_conntrack_entryV6() {
  local proto=$1
  local s_ipaddr=$2
  local d_ipaddr=$3
  local sport=$4
  local dport=$5

  local dcmd1=""
  local dcmd2=""
  local dcmd3=""
  local dcmd4=""
  local dcmd5=""
  local dcmd6=""

  #creation of deletion command
  dcmd1="conntrack -D -f ipv6"
  if [ "$proto" ];then
    dcmd2=" -p $proto"
  fi
  if [ "$s_ipaddr" ];then
    dcmd3="  --orig-src $s_ipaddr"
  fi
  if [ "$d_ipaddr" ];then
    dcmd4="  --orig-dst $d_ipaddr"
  fi
  if [ "$sport" ];then
    dcmd5=" --sport $sport"
  fi
  if [ "$dport" ];then
    dcmd6=" --dport $dport"
  fi

  deletion_final_command=$dcmd1$dcmd2$dcmd3$dcmd4$dcmd5$dcmd6

  echo "Final Deletion cmd V6 is:" $deletion_final_command

  eval $deletion_final_command
}

#Utility function to Delete the client conntrack when we are adding a Accept firewall entry for
#that IPv6 address and port combination
#$1 is index in qcmap_firewall
#$2 is protocol
function delete_conntrack_entry_for_accept_ipv6_firewall_entries_utility() {
  local index=$1
  local protocol=$2
  check_sport=false
  check_dport=false

  local conntrack_cmd1=""
  local conntrack_cmd2=""

  log  $(basename "$0") "delete_conntrack_entry_for_accept_ipv6_firewall_entries_utility" $LINENO "Entering delete_conntrack_entry_for_accept_ipv6_firewall_entries_utility function"

  case "$protocol" in
    "tcp")
      conntrack_cmd1="conntrack -L -f ipv6 | cut -f1,10,11,12,13 -d ' ' | grep -v dport=53 "
      conntrack_cmd2="| grep tcp "
      src_port=$(uci get qcmap_firewall.@firewall_rule[$index].src_port)
      if [ "$src_port" ] && [ "$src_port" != "Any" ]; then
        src_port_range=$(uci get qcmap_firewall.@firewall_rule[$index].src_port_range)
        min_sport=${src_port}
        max_sport=$(($src_port + $src_port_range))
        check_sport=true
      fi
      dest_port=$(uci get qcmap_firewall.@firewall_rule[$index].dest_port)
      if [ "$dest_port" ] && [ "$dest_port" != "Any" ]; then
        dest_port_range=$(uci get qcmap_firewall.@firewall_rule[$index].dest_port_range)
        min_dport=${dest_port}
        max_dport=$(($dest_port + $dest_port_range))
        check_dport=true
      fi
      ;;
    "udp")
      conntrack_cmd1="conntrack -L -f ipv6 | cut -f1,9,10,11,12 -d ' ' | grep -v dport=53 "
      conntrack_cmd2="| grep udp "
      src_port=$(uci get qcmap_firewall.@firewall_rule[$index].src_port)
      if [ "$src_port" ] && [ "$src_port" != "Any" ]; then
        src_port_range=$(uci get qcmap_firewall.@firewall_rule[$index].src_port_range)
        min_sport=${src_port}
        max_sport=$(($src_port + $src_port_range))
        check_sport=true
      fi
      dest_port=$(uci get qcmap_firewall.@firewall_rule[$index].dest_port)
      if [ "$dest_port" ] && [ "$dest_port" != "Any" ]; then
        dest_port_range=$(uci get qcmap_firewall.@firewall_rule[$index].dest_port_range)
        min_dport=${dest_port}
        max_dport=$(($dest_port + $dest_port_range))
        check_dport=true
      fi
      ;;
    "icmp")
      conntrack_cmd1="conntrack -L -f ipv6 | cut -f1,6,7 -d ' ' "
      conntrack_cmd2="| grep icmpv6 "
      ;;
  esac

  #src_ipv6 address is the address we have from firewall config
  src_ipv6=$(uci get qcmap_firewall.@firewall_rule[$index].src_addr)

  if [ $src_ipv6 != "Any" ]; then
    src_prefix_lengthv6=$(uci get qcmap_firewall.@firewall_rule[$index].src_prefix_length)
    prefixWords_src_addr=$((src_prefix_lengthv6/16))
    prefixBits_src_addr=$((src_prefix_lengthv6%16))
    processV6srcaddr=true
  fi

  #dest_ipv6 address is the address we have from firewall config
  dest_ipv6=$(uci get qcmap_firewall.@firewall_rule[$index].dest_addr)

  if [ $dest_ipv6 != "Any" ]; then
    dest_prefix_lengthv6=$(uci get qcmap_firewall.@firewall_rule[$index].dest_prefix_length)
    prefixWords_dest_addr=$((dest_prefix_lengthv6/16))
    prefixBits_dest_addr=$((dest_prefix_lengthv6%16))
    processV6destaddr=true
  fi

  command1=$conntrack_cmd1$conntrack_cmd2
  log $(basename "$0") "conntrack display ipv6 accept cmd : " $LINENO "$command1"
  touch /tmp/data/v6conntrack.txt
  eval $command1 > /tmp/data/v6conntrack.txt

  file="/tmp/data/v6conntrack.txt"
  if [ -f ${file} ]
  then
      if [ -s ${file} ]
      then
        echo "v6conntrack.txt File exists and not empty"
        #reading conntrack_entries.txt file line-by-line
        while read -r line; do
          readed_line=$(echo $line)

          #Fetching 5 tuple info from leading file line by line
          proto=$(printf '%s\n' "$readed_line" |  awk '{print $1}')
          s_ipaddr=$(printf '%s\n' "$readed_line" |  awk '{print $2}' | cut -c 5-)
          d_ipaddr=$(printf '%s\n' "$readed_line" |  awk '{print $3}' | cut -c 5-)
          if [ $protocol != "icmp" ]; then
            dport=$(printf '%s\n' "$readed_line" |  awk '{print $5}' | cut -c 7-)
            sport=$(printf '%s\n' "$readed_line" |  awk '{print $4}' | cut -c 7-)
          fi

          #Check for ipv6 src address and src prefix length
          if [ "$processV6srcaddr" = "true" ];then
            if [ "$s_ipaddr" ] && [ "$src_ipv6" != "Any" ];then
              cmp_len_src=$(( $((prefixWords_src_addr-1)) + ( $prefixWords_src_addr * 4 ) ))
              expand_s_ipaddr=`expand_ipv6 $s_ipaddr`
              cmp_s_ipaddr=`echo $expand_s_ipaddr | cut -c 1-$cmp_len_src`
              expand_src_ipv6=`expand_ipv6 $src_ipv6`
              cmp_src_ipv6=`echo $expand_src_ipv6 | cut -c 1-$cmp_len_src`
              if [ "$cmp_s_ipaddr" != "$cmp_src_ipv6" ];then
                delete_conntrack_entryV6 $proto $s_ipaddr $d_ipaddr $sport $dport
              fi
            fi
            if [ $prefixWords_src_addr -lt 8 ];then
              hex1_after_prefixWords_src_addr=`echo $expand_s_ipaddr | cut -c $((cmp_len_src+2))-$((cmp_len_src+5))`
              hex2_after_prefixWords_src_addr=`echo $expand_src_ipv6 | cut -c $((cmp_len_src+2))-$((cmp_len_src+5))`
              binary1_src_addr=`hex2binary $hex1_after_prefixWords_src_addr`
              binary2_src_addr=`hex2binary $hex2_after_prefixWords_src_addr`
              binary1_prefixBits_src_addr=`echo $binary1_src_addr | cut -c 1-$prefixBits_src_addr`
              binary2_prefixBits_src_addr=`echo $binary2_src_addr | cut -c 1-$prefixBits_src_addr`
              if [ "$binary1_prefixBits_src_addr" != "$binary2_prefixBits_src_addr" ];then
                delete_conntrack_entryV6 $proto $s_ipaddr $d_ipaddr $sport $dport
                continue
              fi
            fi
          fi

          #Check for ipv6 dest length and dest prefix length
          if [ "$processV6destaddr" = "true" ];then
            if [ "$d_ipaddr" ] && [ "$dest_ipv6" != "Any" ];then
              cmp_len_dest=$(( $((prefixWords_dest_addr-1)) + ( $prefixWords_dest_addr * 4 ) ))
              expand_d_ipaddr=`expand_ipv6 $d_ipaddr`
              cmp_d_ipaddr=`echo $expand_d_ipaddr | cut -c 1-$cmp_len_dest`
              expand_dest_ipv6=`expand_ipv6 $dest_ipv6`
              cmp_dest_ipv6=`echo $expand_dest_ipv6 | cut -c 1-$cmp_len_dest`
              if [ "$cmp_d_ipaddr" != "$cmp_dest_ipv6" ];then
                delete_conntrack_entryV6 $proto $s_ipaddr $d_ipaddr $sport $dport
              fi
            fi
            if [ $prefixWords_dest_addr -lt 8 ];then
              hex1_after_prefixWords_dest_addr=`echo $expand_d_ipaddr | cut -c $((cmp_len_dest+2))-$((cmp_len_dest+5))`
              hex2_after_prefixWords_dest_addr=`echo $expand_dest_ipv6 | cut -c $((cmp_len_dest+2))-$((cmp_len_dest+5))`
              binary1_dest_addr=`hex2binary $hex1_after_prefixWords_dest_addr`
              binary2_dest_addr=`hex2binary $hex2_after_prefixWords_dest_addr`
              binary1_prefixBits_dest_addr=`echo $binary1_dest_addr | cut -c 1-$prefixBits_dest_addr`
              binary2_prefixBits_dest_addr=`echo $binary2_dest_addr | cut -c 1-$prefixBits_dest_addr`
              if [ "$binary1_prefixBits_dest_addr" != "$binary2_prefixBits_dest_addr" ];then
                delete_conntrack_entryV6 $proto $s_ipaddr $d_ipaddr $sport $dport
                continue
              fi
            fi
          fi

          if [ $protocol != "icmp" ];then
            #Check for source port range
            if [ $check_sport = "true" ] && [ "$sport" ];then
               if [ $sport -lt $min_sport ] || [ $sport -gt $max_sport ];then
                 delete_conntrack_entryV6 $proto $s_ipaddr $d_ipaddr $sport $dport
                 continue
               fi
            fi

            #Check for destination port range
            if [ $check_dport = "true" ] && [ "$dport" ];then
               if [ $dport -lt $min_dport ] || [ $dport -gt $max_dport ];then
                 delete_conntrack_entryV6 $proto $s_ipaddr $d_ipaddr $sport $dport
                 continue
               fi
            fi
          fi

           #check protocol
           if [ $(uci get qcmap_firewall.@firewall_rule[$index].proto) = "tcpudp" ] || [ $(uci get qcmap_firewall.@firewall_rule[$index].proto) = "$protocol" ];then
             continue
           fi

           delete_conntrack_entryV6 $proto $s_ipaddr $d_ipaddr $sport $dport

        done <$file

        rm -rf /tmp/data/conntrack_entries.txt
      else
        echo "v6conntrack.txt File exists but empty"
      fi
  else
      echo "v6conntrack.txt File not exists"
  fi
}

#Utility function to Delete the client conntrack when we are adding a DROP firewall entry for
#that IPv4 address and port combination
#$1 is index in qcmap_firewall
#$2 is protocol
#$3 is profile_id
function delete_conntrack_entry_for_drop_ipv4_firewall_entries_utility() {
  local index=$1
  local protocol=$2
  local profile_id=$3
  local check_ip=false
  local check_sport=false
  local check_dport=false

  local conntrack_cmd1=""
  local conntrack_cmd2=""
  local conntrack_cmd3=""
  local conntrack_cmd4=""
  local conntrack_cmd5=""

  local dcmd1=""
  local dcmd2=""
  local dcmd3=""
  local dcmd4=""
  local dcmd5=""
  local PUBLIC_IP_ADDR=""

  log  $(basename "$0") "delete_conntrack_entry_for_drop_ipv4_firewall_entries_utility" $LINENO "Entering delete_conntrack_entry_for_drop_ipv4_firewall_entries_utility function"

  get_ipv4v6config_file $profile_id
  if [ -f "$V4_config_file" ]; then
    PUBLIC_IP_ADDR=`awk -F "=" 'FNR == 2 {print $2}' $V4_config_file | tr -d '"'`
  fi

  if [ $protocol = "udp" ];then
    conntrack_cmd1="conntrack -L | grep -i '$PUBLIC_IP_ADDR' | cut -f1,9,11,12 -d ' ' "
  elif [ $protocol = "tcp" ];then
    conntrack_cmd1="conntrack -L | grep -i '$PUBLIC_IP_ADDR' | cut -f1,10,12,13 -d ' ' "
  elif [ $protocol = "icmp" ];then
    conntrack_cmd1="conntrack -L | grep -i '$PUBLIC_IP_ADDR' | cut -f1,8,9 -d ' ' "
  fi

  #src_ip address is the address we have from firewall config
  src_ip=$(uci get qcmap_firewall.@firewall_rule[$index].src_addr)

  if [ $src_ip != "Any" ]; then
    subnet_mask=$(uci get qcmap_firewall.@firewall_rule[$index].src_addr_mask)
    first=${src_ip%%.*}
    last3=${src_ip#*.}
    second=${last3%%.*}
    last2=${last3#*.}
    third=${last2%.*}
    fourth=${last2#*.}
    first_part=`echo $first.`
    second_part=`echo $first.$second.`
    third_part=`echo $first.$second.$third.`
    fouth_part=`echo $first.$second.$third.$fourth`

    IP1=`ip_to_hex 255.0.0.0`
    IP2=`ip_to_hex 255.255.0.0`
    IP3=`ip_to_hex 255.255.255.0`
    IP4=`ip_to_hex 255.255.255.255`
    subnet_value=`ip_to_hex $subnet_mask`

    hex_IP1="0x$IP1"
    hex_IP2="0x$IP2"
    hex_IP3="0x$IP3"
    hex_IP4="0x$IP4"
    hex_subnet_value="0x$subnet_value"

    hex_subnet_value_num=$(printf "%d" "$hex_subnet_value")
    hex_IP1_num=$(printf "%d" "$hex_IP1")
    hex_IP2_num=$(printf "%d" "$hex_IP2")
    hex_IP3_num=$(printf "%d" "$hex_IP3")
    hex_IP4_num=$(printf "%d" "$hex_IP4")

    if [ $hex_subnet_value_num -lt $hex_IP1_num ]; then
      check_ip=true

    # We need to grep according to the subnet mask. If subnet mask
    # is 255.0.0.0 / 255.255.0.0 / 255.255.255.0 / 255.255.255.255 ,
    # then we can grep straight-away for the match. Otherwise, we grep for
    # the minimum pattern that matches perfectly and check later whether
    # we need to delete that conntrack entry or not

    elif [ $hex_subnet_value_num -ge $hex_IP1_num ] && [ $hex_subnet_value_num -lt $hex_IP2_num ]; then
      conntrack_cmd2="| grep src=$first_part"

      if [ $hex_subnet_value_num != $hex_IP1_num ]; then
        check_ip=true
      fi

    elif [ $hex_subnet_value_num -ge $hex_IP2_num ] && [ $hex_subnet_value_num -lt $hex_IP3_num ]; then
      conntrack_cmd2="| grep src=$second_part"

      if [ $hex_subnet_value_num != $hex_IP2_num ]; then
        check_ip=true
      fi

    elif [ $hex_subnet_value_num -ge $hex_IP3_num ] && [ $hex_subnet_value_num -lt $hex_IP4_num ]; then
      conntrack_cmd2="| grep src=$third_part"

      if [ $hex_subnet_value_num != $hex_IP3_num ]; then
        check_ip=true
      fi

    elif [ $hex_subnet_value_num -eq $hex_IP4_num ]; then
      conntrack_cmd2="| grep src=$src_ip"

      if [ $hex_subnet_value_num != $hex_IP4_num ]; then
        check_ip=true
      fi
    fi
  fi

  case "$protocol" in
    "tcp")
      conntrack_cmd3="| grep tcp "
      src_port=$(uci get qcmap_firewall.@firewall_rule[$index].src_port)
      if [ "$src_port" ] && [ "$src_port" != "Any" ]; then
        src_port_range=$(uci get qcmap_firewall.@firewall_rule[$index].src_port_range)
        if [ $src_port_range -eq 0 ]; then
          conntrack_cmd4="| grep sport=$src_port "
        else
          min_sport=${src_port}
          max_sport=$(($src_port + $src_port_range))
          check_sport=true
        fi
      fi
      dest_port=$(uci get qcmap_firewall.@firewall_rule[$index].dest_port)
      if [ "$dest_port" ] && [ "$dest_port" != "Any" ]; then
        dest_port_range=$(uci get qcmap_firewall.@firewall_rule[$index].dest_port_range)
        if [ $dest_port_range -eq 0 ]; then
          conntrack_cmd5="| grep dport=$dest_port "
        else
          min_dport=${dest_port}
          max_dport=$(($dest_port + $dest_port_range))
          check_dport=true
        fi
      fi
      ;;
    "udp")
      conntrack_cmd3="| grep udp "
      src_port=$(uci get qcmap_firewall.@firewall_rule[$index].src_port)
      if [ "$src_port" ] && [ "$src_port" != "Any" ]; then
        src_port_range=$(uci get qcmap_firewall.@firewall_rule[$index].src_port_range)
        if [ $src_port_range -eq 0 ]; then
          conntrack_cmd4="| grep sport=$src_port "
        else
          min_sport=${src_port}
          max_sport=$(($src_port + $src_port_range))
          check_sport=true
        fi
      fi
      dest_port=$(uci get qcmap_firewall.@firewall_rule[$index].dest_port)
      if [ "$dest_port" ] && [ "$dest_port" != "Any" ]; then
        dest_port_range=$(uci get qcmap_firewall.@firewall_rule[$index].dest_port_range)
        if [ $dest_port_range -eq 0 ]; then
          conntrack_cmd5="| grep dport=$dest_port "
        else
          min_dport=${dest_port}
          max_dport=$(($dest_port + $dest_port_range))
          check_dport=true
        fi
      fi
      ;;
    "icmp")
      conntrack_cmd3="| grep icmp "
      ;;
  esac

  command1=$conntrack_cmd1$conntrack_cmd2$conntrack_cmd3$conntrack_cmd4$conntrack_cmd5
  log  $(basename "$0") "delete_conntrack_entry_for_drop_ipv4_firewall_entries_utility" $LINENO "command1=$command1"
  log $(basename "$0") "conntrack display ipv4 drop cmd : " $LINENO "$command1"
  touch /tmp/data/conntrack_entries.txt
  eval $command1 > /tmp/data/conntrack_entries.txt

  file="/tmp/data/conntrack_entries.txt"

  if [ -f ${file} ]
  then
      if [ -s ${file} ]
      then
        log  $(basename "$0") "delete_conntrack_entry_for_drop_ipv4_firewall_entries_utility" $LINENO "conntrack_entries.txt File exists and not empty"
        #reading conntrack_entries.txt file line-by-line
        while read -r line; do
          readed_line=$(echo $line)
          #s_ipaddr is the ip address that we get by reading conntrack_entries.txt file line-by-line
          #proto is the protocol that we get by reading conntrack_entries.txt file line-by-line
          #sport is the source port that we get by reading conntrack_entries.txt file line-by-line
          #dport is the destination port that we get by reading conntrack_entries.txt file line-by-line

          proto=$(printf '%s\n' "$readed_line" |  awk '{print $1}')
          s_ipaddr=$(printf '%s\n' "$readed_line" |  awk '{print $2}' | cut -c 5-)
          if [ $protocol != "icmp" ]; then
            sport=$(printf '%s\n' "$readed_line" |  awk '{print $3}' | cut -c 7-)
            dport=$(printf '%s\n' "$readed_line" |  awk '{print $4}' | cut -c 7-)
          fi

          #Check for subnet mask
          if [ "$check_ip" = "true" ]; then
            src_ip_int=$(string_to_int_ip $src_ip)
            s_ipaddr_int=$(string_to_int_ip $s_ipaddr)
            subnet_mask_int=$(string_to_int_ip $subnet_mask)
            res1=$(( src_ip_int&subnet_mask_int ))
            res2=$(( s_ipaddr_int&subnet_mask_int ))
            if [ $res1 -ne $res2 ]; then
              continue
            fi
          fi

          if [ $protocol != "icmp" ];then
            #Check for source port range
            if [ $check_sport = "true" ] && [ "$sport" ];then
               if [ $sport -lt $min_sport ] || [ $sport -gt $max_sport ];then
                 continue
               fi
            fi

            #Check for destination port range
            if [ $check_dport = "true" ] && [ "$dport" ];then
               if [ $dport -lt $min_dport ] || [ $dport -gt $max_dport ];then
                 continue
               fi
            fi
          fi

          #creation of deletion command
          dcmd1="conntrack -D"
          if [ "$proto" ];then
            dcmd2=" -p $proto"
          fi
          if [ "$s_ipaddr" ];then
            dcmd3="  --orig-src $s_ipaddr"
          fi
          if [ "$sport" ];then
            dcmd4=" --sport $sport"
          fi
          if [ "$dport" ];then
            dcmd5=" --dport $dport"
          fi

          deletion_final_command=$dcmd1$dcmd2$dcmd3$dcmd4$dcmd5

          echo "Final Deletion cmd is:" $deletion_final_command

          eval $deletion_final_command

        done <$file

        rm -rf /tmp/data/conntrack_entries.txt
      else
        echo "conntrack_entries.txt File exists but empty"
      fi
  else
      echo "conntrack_entries.txt File not exists"
  fi
}

#Function to Delete the client conntrack when we are adding a DROP firewall entry for
#that IPv4 address and port combination
#$1 is Firewall handle
#$2 is wan_profile_handle
function delete_conntrack_entry_for_drop_ipv4_firewall_entries() {
  local firewall_handle=$1

  no_of_wan_profiles=$(uci get qcmap_lan.@no_of_configs[0].no_of_profiles)
  for i in $(seq $no_of_wan_profiles)
  do
    profile_id=$(uci get qcmap_lan.@profile[$((i-1))].profile_id)
    if [ $profile_id -eq $2 ];
    then
      firewall_enabled=$(uci get qcmap_lan.@profile[$((i-1))].firewall_enabled)
      firewall_pkts_allowed=$(uci get qcmap_lan.@profile[$((i-1))].pkts_allowed)
      break
    fi
  done

  if [ $firewall_enabled -eq 1 ] && [ $firewall_pkts_allowed -eq 1 ]; then
    #index represent the index of the firewall rule in qcmap_firewall w.r.to the firewall handle.
    index=`uci show qcmap_firewall | grep -i "$firewall_handle" | awk -F'[][]' '{print $2}'`
    if [ "$index" ]; then
      local next_hdr_proto=$(uci get qcmap_firewall.@firewall_rule[$index].proto)
    fi
    if [ "$next_hdr_proto" = "Any" ] || [ "$next_hdr_proto" = "tcp" ] || [ "$next_hdr_proto" = "udp" ] || [ "$next_hdr_proto" = "tcpudp" ] || [ "$next_hdr_proto" = "icmp" ]
    then
      if [ "$next_hdr_proto" = "Any" ] || [ "$next_hdr_proto" = "tcpudp" ];then
        delete_conntrack_entry_for_drop_ipv4_firewall_entries_utility $index "tcp" $2
        delete_conntrack_entry_for_drop_ipv4_firewall_entries_utility $index "udp" $2
      else
        delete_conntrack_entry_for_drop_ipv4_firewall_entries_utility $index $next_hdr_proto $2
      fi
    fi
  fi
}

#Function to Delete the client conntrack when we are adding a DROP firewall entry for
#that IPv6 address and port combination
#$1 is Firewall handle
#$2 is wan_profile_handle
function delete_conntrack_entry_for_drop_ipv6_firewall_entries() {
  local firewall_handle=$1

  no_of_wan_profiles=$(uci get qcmap_lan.@no_of_configs[0].no_of_profiles)
  for i in $(seq $no_of_wan_profiles)
  do
    profile_id=$(uci get qcmap_lan.@profile[$((i-1))].profile_id)
    if [ $profile_id -eq $2 ];
    then
      firewall_enabled=$(uci get qcmap_lan.@profile[$((i-1))].firewall_enabled)
      firewall_pkts_allowed=$(uci get qcmap_lan.@profile[$((i-1))].pkts_allowed)
      break
    fi
  done

  if [ $firewall_enabled -eq 1 ] && [ $firewall_pkts_allowed -eq 1 ]; then
    #index represent the index of the firewall rule in qcmap_firewall w.r.to the firewall handle.
    index=`uci show qcmap_firewall | grep -i "$firewall_handle" | awk -F'[][]' '{print $2}'`
    if [ "$index" ]; then
      local next_hdr_proto=$(uci get qcmap_firewall.@firewall_rule[$index].proto)
    fi
    if [ "$next_hdr_proto" = "Any" ] || [ "$next_hdr_proto" = "tcp" ] || [ "$next_hdr_proto" = "udp" ] || [ "$next_hdr_proto" = "tcpudp" ] || [ "$next_hdr_proto" = "icmp" ]
    then
      if [ "$next_hdr_proto" = "Any" ] || [ "$next_hdr_proto" = "tcpudp" ];then
        delete_conntrack_entry_for_drop_ipv6_firewall_entries_utility $index "tcp"
        delete_conntrack_entry_for_drop_ipv6_firewall_entries_utility $index "udp"
      else
        delete_conntrack_entry_for_drop_ipv6_firewall_entries_utility $index $next_hdr_proto
      fi
    fi
  fi
}

#Function to delete the conntrack v4 entry in case of Accept
function delete_conntrackv4_entry(){
  local proto=$1
  local src_ipaddr=$2
  local sport=$3
  local dport=$4

  local dcmd1=""
  local dcmd2=""
  local dcmd3=""
  local dcmd4=""

  #creation of deletion command
  dcmd1="conntrack -D -p $protocol"
  if [ "$src_ipaddr" ];then
    dcmd2="  --orig-src $src_ipaddr"
  fi
  if [ "$sport" ];then
    dcmd3=" --sport $sport"
  fi
  if [ "$dport" ];then
    dcmd4=" --dport $dport"
  fi

  deletion_final_command=$dcmd1$dcmd2$dcmd3$dcmd4

  echo "Final Deletion cmd is:" $deletion_final_command

  eval $deletion_final_command
}

#Utility function to Delete the client conntrack when we are adding a ACCEPT firewall entry for
#that IPv4 address and port combination
#$1 is index in qcmap_firewall
#$2 is protocol
#$3 is profile handle
function delete_conntrack_entry_for_accept_ipv4_firewall_entries_utility() {
  local count=$1
  local protocol=$2
  local profile_id=$3
  check_sport=false;
  check_dport=false;
  check_ip=false;

  local conntrack_cmd1=""
  local conntrack_cmd2=""
  local PUBLIC_IP_ADDR=""
  local default_ethwan_profile=$(util_execute_uci get qcmap_lan.@lan[0].default_waneth_profile_num)

  log  $(basename "$0") "delete_conntrack_entry_for_accept_ipv4_firewall_entries_utility" $LINENO "Entering delete_conntrack_entry_for_accept_ipv4_firewall_entries_utility function"

  get_ipv4v6config_file $profile_id
  if [ -f "$V4_config_file" ]; then
    PUBLIC_IP_ADDR=`awk -F "=" 'FNR == 2 {print $2}' $V4_config_file | tr -d '"'`
  fi

  src_ip=$(uci get qcmap_firewall.@firewall_rule[$count].src_addr)
  if [ $src_ip != "Any" ]; then
    check_ip=true
    subnet_mask=$(uci get qcmap_firewall.@firewall_rule[$count].src_addr_mask)
  fi

  case "$protocol" in
    "tcp")
      conntrack_cmd2="| grep tcp "
      src_port=$(uci get qcmap_firewall.@firewall_rule[$count].src_port)
      if [ "$src_port" ] && [ "$src_port" != "Any" ]; then
        src_port_range=$(uci get qcmap_firewall.@firewall_rule[$count].src_port_range)
        min_sport=${src_port}
        max_sport=$(($src_port + $src_port_range))
        check_sport=true;
      fi
      dest_port=$(uci get qcmap_firewall.@firewall_rule[$count].dest_port)
      if [ "$dest_port" ] && [ "$dest_port" != "Any" ]; then
        dest_port_range=$(uci get qcmap_firewall.@firewall_rule[$count].dest_port_range)
        min_dport=${dest_port}
        max_dport=$(($dest_port + $dest_port_range))
        check_dport=true;
      fi
      ;;
    "udp")
      conntrack_cmd2="| grep udp "
      src_port=$(uci get qcmap_firewall.@firewall_rule[$count].src_port)
      if [ "$src_port" ] && [ "$src_port" != "Any" ]; then
        src_port_range=$(uci get qcmap_firewall.@firewall_rule[$count].src_port_range)
        min_sport=${src_port}
        max_sport=$(($src_port + $src_port_range))
        check_sport=true;
      fi
      dest_port=$(uci get qcmap_firewall.@firewall_rule[$count].dest_port)
      if [ "$dest_port" ] && [ "$dest_port" != "Any" ]; then
        dest_port_range=$(uci get qcmap_firewall.@firewall_rule[$count].dest_port_range)
        min_dport=${dest_port}
        max_dport=$(($dest_port + $dest_port_range))
        check_dport=true;
      fi
      ;;
    "icmp")
      conntrack_cmd2="| grep icmp "
      ;;
  esac

  touch /tmp/data/conntrack_entries.txt

  bridge_ip=$(uci get network.lan.ipaddr)
  #we shouldn't grep the conntracks generated by bridge.
  if [ $protocol = "tcp" ];then
    conntrack_cmd1="conntrack -L | grep -i '$PUBLIC_IP_ADDR' |cut -f1,10,12,13 -d ' ' | grep -v -w src=$bridge_ip "
  elif [ $protocol = "udp" ];then
    conntrack_cmd1="conntrack -L | grep -i '$PUBLIC_IP_ADDR' | cut -f1,9,11,12 -d ' ' | grep -v -w src=$bridge_ip "
  elif [ $protocol = "icmp" ];then
    conntrack_cmd1="conntrack -L | grep -i '$PUBLIC_IP_ADDR' | cut -f1,8,9 -d ' ' | grep -v -w src=$bridge_ip "
  fi

  command1=$conntrack_cmd1$conntrack_cmd2
  log $(basename "$0") "conntrack display ipv4 accept cmd : " $LINENO "$command1"
  eval $command1 > /tmp/data/conntrack_entries.txt

  if [ $profile_id -eq 1 -o  $profile_id -eq $default_ethwan_profile ]; then
    lan_zone_idx=$(uci show firewall | grep -i "zone" | grep -i -w "lan_wan" | awk -F'[][]' '{print $2}')
  else
    lan_zone_idx=$(uci show firewall | grep -i "zone" | grep -i -w "lan_wan$profile_id" | awk -F'[][]' '{print $2}')
  fi

  all_lan_iface_listed=`uci get firewall.@zone[$lan_zone_idx].network`

  for iface in $all_lan_iface_listed
  do
    ip_addr=$(uci get network.$iface.ipaddr)
    if [ $protocol = "tcp" ];then
      conntrack_cmd1="conntrack -L | grep -i '$PUBLIC_IP_ADDR' | cut -f1,10,12,13 -d ' ' | grep -v -w src=$ip_addr "
    elif [ $protocol = "udp" ];then
      conntrack_cmd1="conntrack -L | grep -i '$PUBLIC_IP_ADDR' | cut -f1,9,11,12 -d ' ' | grep -v -w src=$ip_addr "
    elif [ $protocol = "icmp" ];then
      conntrack_cmd1="conntrack -L | grep -i '$PUBLIC_IP_ADDR' | cut -f1,8,9 -d ' ' | grep -v -w src=$ip_addr "
    fi

    command1=$conntrack_cmd1$conntrack_cmd2
    log $(basename "$0") "conntrack display ipv4 accept cmd : " $LINENO "$command1"
    eval $command1 > /tmp/data/conntrack_entries.txt
  done

  file="/tmp/data/conntrack_entries.txt"
  if [ -f ${file} ]
  then
      if [ -s ${file} ]
      then
        echo "conntrack_entries.txt File exists and not empty"
        #reading conntrack_entries.txt file line-by-line
        while read -r line; do
          readed_line=$(echo $line)
          #src_ipaddr is the ip address that we get by conntrack_entries.txt file line-by-line
          #proto is the protocol that we get by conntrack_entries.txt file line-by-line
          #sport is the source port that we get by conntrack_entries.txt file line-by-line
          #dport is the destination port that we get by conntrack_entries.txt file line-by-line
          proto=$(printf '%s\n' "$readed_line" |  awk '{print $1}')
          src_ipaddr=$(printf '%s\n' "$readed_line" |  awk '{print $2}' | cut -c 5-)
          if [ $protocol != "icmp" ]; then
            sport=$(printf '%s\n' "$readed_line" |  awk '{print $3}' | cut -c 7-)
            dport=$(printf '%s\n' "$readed_line" |  awk '{print $4}' | cut -c 7-)
          fi

          #Check for subnet mask
          if [ "$check_ip" = "true" ]; then
            src_ip_int=$(string_to_int_ip $src_ip)
            s_ipaddr_int=$(string_to_int_ip $src_ipaddr)
            subnet_mask_int=$(string_to_int_ip $subnet_mask)
            res1=$(( src_ip_int&subnet_mask_int ))
            res2=$(( s_ipaddr_int&subnet_mask_int ))
            if [ $res1 -ne $res2 ]; then
              delete_conntrackv4_entry $proto $src_ipaddr $sport $dport
              continue
            fi
          fi

          if [ $protocol != "icmp" ];then
            #Check for source port range
            if [ $check_sport = "true" ] && [ "$sport" ];then
               if [ $sport -lt $min_sport ] || [ $sport -gt $max_sport ];then
                 delete_conntrackv4_entry $proto $src_ipaddr $sport $dport
                 continue
               fi
            fi

            #Check for destination port range
            if [ $check_dport = "true" ] && [ "$dport" ];then
               if [ $dport -lt $min_dport ] || [ $dport -gt $max_dport ];then
                 delete_conntrackv4_entry $proto $src_ipaddr $sport $dport
                 continue
               fi
            fi
          fi

           #check protocol
           if [ $(uci get qcmap_firewall.@firewall_rule[$count].proto) = "tcpudp" ] || [ $(uci get qcmap_firewall.@firewall_rule[$count].proto) = "$protocol" ];then
             continue
           fi

           delete_conntrackv4_entry $proto $src_ipaddr $sport $dport

        done <$file

        rm -rf /tmp/data/conntrack_entries.txt
      else
        echo "conntrack_entries.txt File exists but empty"
      fi
  else
      echo "conntrack_entries.txt File not exists"
  fi
}

#Function to Delete the client conntrack when we are adding a ACCEPT firewall entry for
#that IPv4 address and port combination
#$1 is Firewall handle
#$2 is wan_profile_handle
function delete_conntrack_entry_for_accept_ipv4_firewall_entries() {
  local firewall_handle=$1

  no_of_wan_profiles=$(uci get qcmap_lan.@no_of_configs[0].no_of_profiles)
  for i in $(seq $no_of_wan_profiles)
  do
    profile_id=$(uci get qcmap_lan.@profile[$((i-1))].profile_id)
    if [ $profile_id -eq $2 ];
    then
      firewall_pkts_allowed=$(uci get qcmap_lan.@profile[$((i-1))].pkts_allowed)
      break
    fi
  done
  index=`uci show qcmap_firewall | grep -i "$firewall_handle" | awk -F'[][]' '{print $2}'`
  if [ "$index" ]; then
    local next_hdr_proto=$(uci get qcmap_firewall.@firewall_rule[$index].proto)
  fi
  if [ "$next_hdr_proto" = "Any" ] || [ "$next_hdr_proto" = "tcp" ] || [ "$next_hdr_proto" = "udp" ] || [ "$next_hdr_proto" = "tcpudp" ] || [ "$next_hdr_proto" = "icmp" ]
  then
    if [ $firewall_pkts_allowed -eq 1 ]; then
      delete_conntrack_entry_for_accept_ipv4_firewall_entries_utility $index "tcp" $2
      delete_conntrack_entry_for_accept_ipv4_firewall_entries_utility $index "udp" $2
    else
      if [ "$next_hdr_proto" = "Any" ] || [ "$next_hdr_proto" = "tcpudp" ];then
        delete_conntrack_entry_for_drop_ipv4_firewall_entries_utility $index "tcp" $2
        delete_conntrack_entry_for_drop_ipv4_firewall_entries_utility $index "udp" $2
      else
        delete_conntrack_entry_for_drop_ipv4_firewall_entries_utility $index $next_hdr_proto $2
      fi
    fi
  fi
}

#Function to Delete the client conntrack when we are adding a ACCEPT firewall entry for
#that IPv6 address and port combination
#$1 is Firewall handle
#$2 is wan_profile_handle
function delete_conntrack_entry_for_accept_ipv6_firewall_entries() {
  local firewall_handle=$1

  no_of_wan_profiles=$(uci get qcmap_lan.@no_of_configs[0].no_of_profiles)
  for i in $(seq $no_of_wan_profiles)
  do
    profile_id=$(uci get qcmap_lan.@profile[$((i-1))].profile_id)
    if [ $profile_id -eq $2 ];
    then
      firewall_pkts_allowed=$(uci get qcmap_lan.@profile[$((i-1))].pkts_allowed)
      break
    fi
  done
  index=`uci show qcmap_firewall | grep -i "$firewall_handle" | awk -F'[][]' '{print $2}'`
  if [ "$index" ]; then
    local next_hdr_proto=$(uci get qcmap_firewall.@firewall_rule[$index].proto)
  fi
  if [ "$next_hdr_proto" = "Any" ] || [ "$next_hdr_proto" = "tcp" ] || [ "$next_hdr_proto" = "udp" ] || [ "$next_hdr_proto" = "tcpudp" ] || [ "$next_hdr_proto" = "icmp" ]
  then
    if [ $firewall_pkts_allowed -eq 1 ]; then
      delete_conntrack_entry_for_accept_ipv6_firewall_entries_utility $index "tcp" $2
      delete_conntrack_entry_for_accept_ipv6_firewall_entries_utility $index "udp" $2
    else
      if [ "$next_hdr_proto" = "Any" ] || [ "$next_hdr_proto" = "tcpudp" ];then
        delete_conntrack_entry_for_drop_ipv6_firewall_entries_utility $index "tcp"
        delete_conntrack_entry_for_drop_ipv6_firewall_entries_utility $index "udp"
      else
        delete_conntrack_entry_for_drop_ipv6_firewall_entries_utility $index $next_hdr_proto
      fi
    fi
  fi
}


#Deletes all whitelisting rules for a particular wan profile
#$1 is wan_profile_id
function delete_wl_firewall() {

  idx=$(uci get qcmap_lan.@no_of_configs[0].no_of_rules)
  count1=`uci show firewall | grep -i -w "whitelisting_matching_fwd_ul_v4_accept-$1" | awk -F'[][]' '{print $2}'`
  if [ "$count1" ]; then
    uci delete firewall.@rule[$count1]
    uci commit firewall
    idx=$((idx-1))
  fi

  count2=`uci show firewall | grep -i -w "whitelisting_matching_fwd_ul_v6_accept-$1" | awk -F'[][]' '{print $2}'`
  if [ "$count2" ]; then
    uci delete firewall.@rule[$count2]
    uci commit firewall
    idx=$((idx-1))
  fi

  count3=`uci show firewall | grep -i -w "whitelisting_matching_fwd_ul_v4_drop-$1" | awk -F'[][]' '{print $2}'`
  if [ "$count3" ]; then
    uci delete firewall.@rule[$count3]
    uci commit firewall
    idx=$((idx-1))
  fi

  count4=`uci show firewall | grep -i -w "whitelisting_matching_fwd_ul_v6_drop-$1" | awk -F'[][]' '{print $2}'`
  if [ "$count4" ]; then
    uci delete firewall.@rule[$count4]
    uci commit firewall
    idx=$((idx-1))
  fi

  count5=`uci show firewall | grep -i -w "whitelisting_matching_output_ul_v4_accept-$1" | awk -F'[][]' '{print $2}'`
  if [ "$count5" ]; then
    uci delete firewall.@rule[$count5]
    uci commit firewall
    idx=$((idx-1))
  fi

  count6=`uci show firewall | grep -i -w "whitelisting_matching_output_ul_v6_accept-$1" | awk -F'[][]' '{print $2}'`
  if [ "$count6" ]; then
    uci delete firewall.@rule[$count6]
    uci commit firewall
    idx=$((idx-1))
  fi

  count7=`uci show firewall | grep -i -w "whitelisting_matching_output_ul_v4_drop-$1" | awk -F'[][]' '{print $2}'`
  if [ "$count7" ]; then
    uci delete firewall.@rule[$count7]
    uci commit firewall
    idx=$((idx-1))
  fi

  count8=`uci show firewall | grep -i -w "whitelisting_matching_output_ul_v6_drop-$1" | awk -F'[][]' '{print $2}'`
  if [ "$count8" ]; then
    uci delete firewall.@rule[$count8]
    uci commit firewall
    idx=$((idx-1))
  fi

  count9=`uci show firewall | grep -i -w "whitelisting_matching_preroute_dl_v4_accept-$1" | awk -F'[][]' '{print $2}'`
  if [ "$count9" ]; then
    uci delete firewall.@rule[$count9]
    uci commit firewall
    idx=$((idx-1))
  fi

  count10=`uci show firewall | grep -i -w "whitelisting_matching_preroute_dl_v6_accept-$1" | awk -F'[][]' '{print $2}'`
  if [ "$count10" ]; then
    uci delete firewall.@rule[$count10]
    uci commit firewall
    idx=$((idx-1))
  fi

  count11=`uci show firewall | grep -i -w "whitelisting_matching_preroute_dl_v4_drop-$1" | awk -F'[][]' '{print $2}'`
  if [ "$count11" ]; then
    uci delete firewall.@rule[$count11]
    uci commit firewall
    idx=$((idx-1))
  fi

    count12=`uci show firewall | grep -i -w "whitelisting_matching_preroute_dl_v6_drop-$1" | awk -F'[][]' '{print $2}'`
  if [ "$count12" ]; then
    uci delete firewall.@rule[$count12]
    uci commit firewall
    idx=$((idx-1))
  fi

  count13=`uci show firewall | grep -i "wan$1_dl_accept" | awk -F'[][]' '{print $2}'`
  if [ "$count13" ]; then
    uci delete firewall.@rule[$count13]
    uci commit firewall
    idx=$((idx-1))
  fi

  count14=`uci show firewall | grep -i "wan$1_input_dl_accept" | awk -F'[][]' '{print $2}'`
  if [ "$count14" ]; then
    uci delete firewall.@rule[$count14]
    uci commit firewall
    idx=$((idx-1))
  fi

  util_execute_uci set qcmap_lan.@no_of_configs[0].no_of_rules=$idx
  uci commit qcmap_lan
  /etc/init.d/firewall reload
}

#Deletes all blacklisting rules for a particular wan profile
#$1 is wan_profile_id
function delete_bl_firewall() {
  idx=$(uci get qcmap_lan.@no_of_configs[0].no_of_rules)
  count1=`uci show firewall | grep -i -w "blacklisting_matching_fwd_ul_v4_drop-$1" | awk -F'[][]' '{print $2}'`
  if [ "$count1" ]; then
    uci delete firewall.@rule[$count1]
    uci commit firewall
    idx=$((idx-1))
  fi

  count2=`uci show firewall | grep -i -w "blacklisting_matching_fwd_ul_v6_drop-$1" | awk -F'[][]' '{print $2}'`
  if [ "$count2" ]; then
    uci delete firewall.@rule[$count2]
    uci commit firewall
    idx=$((idx-1))
  fi

  count3=`uci show firewall | grep -i -w "blacklisting_matching_output_ul_v4_drop-$1" | awk -F'[][]' '{print $2}'`
  if [ "$count3" ]; then
    uci delete firewall.@rule[$count3]
    uci commit firewall
    idx=$((idx-1))
  fi

  count4=`uci show firewall | grep -i -w "blacklisting_matching_output_ul_v6_drop-$1" | awk -F'[][]' '{print $2}'`
  if [ "$count4" ]; then
    uci delete firewall.@rule[$count4]
    uci commit firewall
    idx=$((idx-1))
  fi

  count5=`uci show firewall | grep -i -w "blacklisting_matching_preroute_dl_v4_drop-$1" | awk -F'[][]' '{print $2}'`
  if [ "$count5" ]; then
    uci delete firewall.@rule[$count5]
    uci commit firewall
    idx=$((idx-1))
  fi

  count6=`uci show firewall | grep -i -w "blacklisting_matching_preroute_dl_v6_drop-$1" | awk -F'[][]' '{print $2}'`
  if [ "$count6" ]; then
    uci delete firewall.@rule[$count6]
    uci commit firewall
    idx=$((idx-1))
  fi

  count7=`uci show firewall | grep -i "wan$1_dl_accept" | awk -F'[][]' '{print $2}'`
  if [ "$count7" ]; then
    uci delete firewall.@rule[$count7]
    uci commit firewall
    idx=$((idx-1))
  fi

  count8=`uci show firewall | grep -i "wan$1_input_dl_accept" | awk -F'[][]' '{print $2}'`
  if [ "$count8" ]; then
    uci delete firewall.@rule[$count8]
    uci commit firewall
    idx=$((idx-1))
  fi

  util_execute_uci set qcmap_lan.@no_of_configs[0].no_of_rules=$idx
  uci commit qcmap_lan
  /etc/init.d/firewall reload
}

#Function to delete conntrack when we are enabling firewall
#$1 is wan_profile_id
#$2 is BH Type
function delete_conntrack_on_enable_firewall(){
  idx=$(uci get qcmap_lan.@no_of_configs[0].no_of_rules)
  no_of_default_firewall_rules=$(uci get qcmap_lan.@no_of_configs[0].no_of_default_firewall_rules)
  if [ $idx -eq $no_of_default_firewall_rules ];
  then
    return
  fi

  get_ipv4v6config_file $1
  IPType=$2

  case $IPType in
    1)
      if [ -f "$V4_config_file" ];then
        rules_index=`uci show firewall | grep -i "name" | grep -i "rule" | grep -i "$1'$" | awk -F'[][]' '{print $2}'`
        for idx in $rules_index
        do
          firewall_handle=`uci get firewall.@rule[$idx].name | awk -F'-' '{print $1}'`
          if [[ $firewall_handle =~ ^[0-9]+$ ]];then
            ip_version=`uci get firewall.@rule[$idx].family`
            if [ $ip_version = "ipv4" ];then
              delete_conntrack_entry_for_accept_ipv4_firewall_entries $firewall_handle $1
            fi
          fi
        done
      else
        log $(basename "$0") "delete_conntrack_on_enable_firewall" $LINENO "V4 Backhaul is not Up."
      fi
      ;;
    2)
      if [ -f "$V6_config_file" ];then
        rules_index=`uci show firewall | grep -i "name" | grep -i "rule" | grep -i "$1'$" | awk -F'[][]' '{print $2}'`
        for idx in $rules_index
        do
          firewall_handle=`uci get firewall.@rule[$idx].name | awk -F'-' '{print $1}'`
          if [[ $firewall_handle =~ ^[0-9]+$ ]];then
            ip_version=`uci get firewall.@rule[$idx].family`
            if [ $ip_version = "ipv6" ];then
              delete_conntrack_entry_for_accept_ipv6_firewall_entries $firewall_handle $1
            fi
          fi
        done
      else
        log $(basename "$0") "delete_conntrack_on_enable_firewall" $LINENO "V6 Backhaul is not Up."
      fi
      ;;
    *)
      echo "INVALID BH Type"
      ;;
  esac
}

#Function to delete conntrack when we are enabling firewall
#$1 is wan_profile_id
#$2 is BH Type
function delete_conntrack_on_disable_firewall(){
  idx=$(uci get qcmap_lan.@no_of_configs[0].no_of_rules)
  no_of_default_firewall_rules=$(uci get qcmap_lan.@no_of_configs[0].no_of_default_firewall_rules)
  if [ $idx -eq $no_of_default_firewall_rules ];
  then
    return
  fi

  get_ipv4v6config_file $1


  if [ ! -f $V4_config_file ] && [ ! -f $V6_config_file ];then
    rules_index=`uci show firewall | grep -i "name" | grep -i "rule" | grep -i "$1'$" | awk -F'[][]' '{print $2}'`
    for idx in $rules_index
    do
      firewall_handle=$(uci get firewall.@rule[$idx].name | awk -F'-' '{print $1}')
      if [[ $firewall_handle =~ ^[0-9]+$ ]]
      then
        ip_version=`uci get firewall.@rule[$idx].family`
        if [ $ip_version = "ipv4" ];then
          delete_conntrack_entry_for_drop_ipv4_firewall_entries $firewall_handle $1
        elif [ $ip_version = "ipv6" ]; then
          delete_conntrack_entry_for_drop_ipv6_firewall_entries $firewall_handle $1
        fi
      fi
    done
  elif [ $2 -eq 1 ];then
      rules_index=`uci show firewall | grep -i "name" | grep -i "rule" | grep -i "$1'$" | awk -F'[][]' '{print $2}'`
      for idx in $rules_index
      do
        firewall_handle=`uci get firewall.@rule[$idx].name | awk -F'-' '{print $1}'`
        if [[ $firewall_handle =~ ^[0-9]+$ ]]
        then
          ip_version=`uci get firewall.@rule[$idx].family`
          if [ $ip_version = "ipv4" ];then
            delete_conntrack_entry_for_drop_ipv4_firewall_entries $firewall_handle $1
          fi
        fi
      done
  elif [ $2 -eq 2 ];then
   rules_index=`uci show firewall | grep -i "name" | grep -i "rule" | grep -i "$1'$" | awk -F'[][]' '{print $2}'`
   for idx in $rules_index
   do
        firewall_handle=`uci get firewall.@rule[$idx].name | awk -F'-' '{print $1}'`
        if [[ $firewall_handle =~ ^[0-9]+$ ]]
        then
          ip_version=`uci get firewall.@rule[$idx].family`
          if [ $ip_version = "ipv6" ];then
            delete_conntrack_entry_for_drop_ipv6_firewall_entries $firewall_handle $1
          fi
        fi
   done
  fi
}

#Enables the firewall rule when backhaul is UP on a PDN
#$1 is wan_profile_id
#$2 is BH Type
function enable_fw() {
  local profile_handle=$1
  idx=$(uci get qcmap_lan.@no_of_configs[0].no_of_rules)
  no_of_default_firewall_rules=$(uci get qcmap_lan.@no_of_configs[0].no_of_default_firewall_rules)
  if [ $idx -eq $no_of_default_firewall_rules ];
  then
    log $(basename "$0") "enable_fw" $LINENO " No Rules Added"
    return
  fi
  profile_idx=`mbb_util_get_uci_profile_index $profile_handle`
  log  $(basename "$0") "enable_fw" $LINENO "profile_handle=$profile_handle profile_idx=$profile_idx"
  enable_firewall_flag=`uci get qcmap_lan.@profile[$profile_idx].firewall_enabled`
  if [ $enable_firewall_flag -eq 0 ]; then
    log  $(basename "$0") "enable_fw" $LINENO "Firewall is Not Enabled for this particular profile_handle"
    return
  fi
  get_ipv4v6config_file $1

  IPType=$2

  case $IPType in
    1)
      #checking the IPPT feature mode and check the variable ippt_wo_nat_firewall_support is set or not
      ippt_fw_support_flag=$(check_ippt_feature_mode_with_firewall_support $3)
      if [ $ippt_fw_support_flag -eq 1 ];then
        return
      fi
      if [ -f "$V4_config_file" ];then
        rules_index=`uci show firewall | grep -i "name" | grep -i "rule" | grep -i -w "$1'$" | awk -F'[][]' '{print $2}'`
        for idx in $rules_index
        do
          firewall_handle=`uci get firewall.@rule[$idx].name | awk -F'-' '{print $1}'`
          ip_version=`uci get firewall.@rule[$idx].family`
          if [ $ip_version = "ipv4" ];then
            uci set firewall.@rule[$idx].enabled='1'
          fi
        done
      else
        log $(basename "$0") "enable_fw " $LINENO "Backhaul is not Up. Configuration Saved"
      fi
      ;;
    2)
      if [ -f "$V6_config_file" ];then
        rules_index=`uci show firewall | grep -i "name" | grep -i "rule" | grep -i -w "$1'$" | awk -F'[][]' '{print $2}'`
        for idx in $rules_index
        do
          firewall_handle=`uci get firewall.@rule[$idx].name | awk -F'-' '{print $1}'`
          ip_version=`uci get firewall.@rule[$idx].family`
          if [ $ip_version = "ipv6" ];then
            uci set firewall.@rule[$idx].enabled='1'
          fi
        done
      else
        log $(basename "$0") "enable_fw " $LINENO "Backhaul is not Up. Configuration Saved"
      fi
      ;;
    *)
      echo "INVALID BH Type"
      ;;
  esac

  uci commit
  /etc/init.d/firewall reload

  delete_ipset_for_rmnet_gateway $V4_config_file

}

#Disables the firewall rules when backhaul is DOWN on a PDN
#$1 is wan_profile_id
#$2 is BH Type
function disable_fw() {
  idx=$(uci get qcmap_lan.@no_of_configs[0].no_of_rules)
  no_of_default_firewall_rules=$(uci get qcmap_lan.@no_of_configs[0].no_of_default_firewall_rules)
  if [ $idx -eq $no_of_default_firewall_rules ];
  then
    log $(basename "$0") "disable_fw : " $LINENO "No Rules Added"
    return
  fi

  get_ipv4v6config_file $1

  if [ ! -f $V4_config_file ] && [ ! -f $V6_config_file ];then
    rules_index=`uci show firewall | grep -i "name" | grep -i "rule" | grep -i -w "$1'$" | awk -F'[][]' '{print $2}'`
    for idx in $rules_index
    do
      uci set firewall.@rule[$idx].enabled='0'
    done
  elif [ $2 -eq 1 ];then
    rules_index=`uci show firewall | grep -i "name" | grep -i "rule" | grep -i -w "$1'$" | awk -F'[][]' '{print $2}'`
    for idx in $rules_index
    do
      firewall_handle=`uci get firewall.@rule[$idx].name | awk -F'-' '{print $1}'`
      ip_version=`uci get firewall.@rule[$idx].family`
      if [ $ip_version = "ipv4" ];then
        uci set firewall.@rule[$idx].enabled='0'
      fi
    done
  elif [ $2 -eq 2 ];then
    rules_index=`uci show firewall | grep -i "name" | grep -i "rule" | grep -i -w "$1'$" | awk -F'[][]' '{print $2}'`
    for idx in $rules_index
    do
      firewall_handle=`uci get firewall.@rule[$idx].name | awk -F'-' '{print $1}'`
      ip_version=`uci get firewall.@rule[$idx].family`
      if [ $ip_version = "ipv6" ];then
        uci set firewall.@rule[$idx].enabled='0'
      fi
    done
  fi
  uci commit
  /etc/init.d/firewall reload

}

function get_ipv4v6config_file()
{
   local defaultProfileId
   local inputProfileId=$1

   if [ $inputProfileId -le 100 ]; then
     log $(basename "$0") "get_ipv4v6config_file" $LINENO "profile_id is less than 100, profile is wwan"
     defaultProfileId=$(uci get qcmap_lan.@no_of_configs[0].default_pdn)
     V4_config_file=/tmp/ipv4config$inputProfileId
     V6_config_file=/tmp/ipv6config$inputProfileId
   fi

   if [ $inputProfileId -ge 100 -a $inputProfileId -le 200 ]; then
     log $(basename "$0") "get_ipv4v6config_file" $LINENO "profile_id is greater than 100, profile is waneth"
     defaultProfileId=$(uci get qcmap_lan.@lan[0].default_waneth_profile_num)
     V4_config_file=/tmp/ipv4config.waneth$inputProfileId
     V6_config_file=/tmp/ipv6config.waneth$inputProfileId
   fi

   no_of_wan_profiles=$(uci get qcmap_lan.@no_of_configs[0].no_of_profiles)

   log $(basename "$0") "get_ipv4v6config_file" $LINENO "inputProfileId=$inputProfileId defaultProfileId=$defaultProfileId"

   if [ $defaultProfileId -eq $inputProfileId ]; then

     profile_index=$(util_get_default_profile_index)

     log $(basename "$0") "get_ipv4v6config_file" $LINENO "profile_index=$profile_index"

     backhaul_name_v4=$(uci get qcmap_lan.@profile[$profile_index].bh_present)
     backhaul_name_v6=$(uci get qcmap_lan.@profile[$profile_index].bh_present_v6)
     log $(basename "$0") "get_ipv4v6config_file" $LINENO "current v4 backhaul=$backhaul_name_v4 v6 backhaul is=$backhaul_name_v6"
     if [ $backhaul_name_v4 == "wan" ] || [ $backhaul_name_v6 == "wan" ]; then
        log $(basename "$0") "get_ipv4v6config_file" $LINENO "current backhaul is WWAN"
     else
        log $(basename "$0") "get_ipv4v6config_file" $LINENO "current v4 backhaul=$backhaul_name_v4 v6 backhaul is=$backhaul_name_v6"
        V4_config_file=/tmp/ipv4config.$backhaul_name_v4
        V6_config_file=/tmp/ipv6config.$backhaul_name_v6
        log $(basename "$0") "get_ipv4v6config_file" $LINENO "V4_config_file=$V4_config_file V6_config_file=$V6_config_file"
     fi
   fi

}

#$1 is profile handle
function check_ippt_feature_mode_with_firewall_support(){
  local profile_handle=$1
  local profile_idx=`mbb_util_get_uci_profile_index $profile_handle`

  #Quectel: Modify for IPPT With NAT Only one Device. By beale.hu
  ippt_feature_mode=$(util_get_with_nat $profile_idx)
  firewall_support_flag=$(uci get qcmap_lan.@no_of_configs[0].ippt_wo_nat_firewall_support)
  log $(basename "$0") "check_ippt_feature_mode_with_firewall_support" $LINENO "ippt_feature_mode=$ippt_feature_mode"

  if [ $ippt_feature_mode -eq 2 -o $ippt_feature_mode -eq 3 ] && [ $firewall_support_flag -eq 0 ]; then
    echo 1
  else
    echo 0
  fi
}

#$1 is wan_profile_handle
#$2 is ip version
function get_wan_iface_name_from_wan_profile_handle() {
  local wan_profile_id=$1
  local ip_version=$2
  defaultProfileId=`uci get qcmap_lan.@no_of_configs[0].default_pdn`

  if [ $wan_profile_id -eq $defaultProfileId ]; then
    local profile_index=`mbb_util_get_uci_profile_index $wan_profile_id`
    present_bh=`uci get qcmap_lan.@profile[$profile_index].bh_present`

    if [ "$present_bh" = "wan" ] || [ -z "$present_bh" ];then
       log $(basename "$0") "add_wan_iface_on_wan_all_zone :" $LINENO ": current bh is wan"
        config_file_name=/tmp/${ip_version}config${wan_profile_id}
    else
        config_file_name=/tmp/${ip_version}config.${present_bh}
    fi
  else
    config_file_name=/tmp/${ip_version}config${wan_profile_id}
  fi

  if [ -f "$config_file_name" ];then
    DEVNAME=`awk -F "=" 'FNR == 1 {print $2}' $config_file_name | tr -d '"'`
  fi

  echo $DEVNAME
}

#$1 is wan iface name
#$2 is wan_profile_handle
#$3 is ip version
function install_iptables_rules_to_drop_all_traffic() {
  local wan_iface=$1
  local wan_profile_handle=$2
  local ip_version=$3

  if [ "$ip_version" = "ipv4" ];then
    #iptables -t mangle -I PREROUTING -i $wan_iface -j DROP //quectel prevent commands fail in race condition
    iptables -w -t mangle -I PREROUTING -i $wan_iface -j DROP
  elif [ "$ip_version" = "ipv6" ];then
    #ip6tables -t mangle -I PREROUTING -i $wan_iface -j DROP //quectel prevent commands fail in race condition
    ip6tables -w -t mangle -I PREROUTING -i $wan_iface -j DROP
  fi

  #Get downstream value
  downstream=$(util_get_downstream_value_v4 $wan_profile_handle)

  for lan_iface in $downstream
  do
    if [ "$ip_version" = "ipv4" ];then
      #iptables -t mangle -I PREROUTING -i br-$lan_iface -j DROP //quectel prevent commands fail in race condition
      iptables -w -t mangle -I PREROUTING -i br-$lan_iface -j DROP
    elif [ "$ip_version" = "ipv6" ];then
      #ip6tables -t mangle -I PREROUTING -i br-$lan_iface -j DROP  //quectel prevent commands fail in race condition
      ip6tables -w -t mangle -I PREROUTING -i br-$lan_iface -j DROP
    fi
  done
}

#$1 is wan iface name
#$2 is wan_profile_handle
#$3 is ip version
function delete_iptables_rules_to_drop_all_traffic() {
  local wan_iface=$1
  local wan_profile_handle=$2
  local ip_version=$3

  if [ "$ip_version" = "ipv4" ];then
    #iptables -t mangle -D PREROUTING -i $wan_iface -j DROP //quectel prevent commands fail in race condition
    iptables -w -t mangle -D PREROUTING -i $wan_iface -j DROP
  elif [ "$ip_version" = "ipv6" ];then
    #ip6tables -t mangle -D PREROUTING -i $wan_iface -j DROP //quectel prevent commands fail in race condition
    ip6tables -w -t mangle -D PREROUTING -i $wan_iface -j DROP
  fi

  #Get downstream value
  downstream=$(util_get_downstream_value_v4 $wan_profile_handle)

  for lan_iface in $downstream
  do
    if [ "$ip_version" = "ipv4" ];then
      #iptables -t mangle -D PREROUTING -i br-$lan_iface -j DROP //quectel prevent commands fail in race condition
      iptables -w -t mangle -D PREROUTING -i br-$lan_iface -j DROP
    elif [ "$ip_version" = "ipv6" ];then
      #ip6tables -t mangle -D PREROUTING -i br-$lan_iface -j DROP //quectel prevent commands fail in race condition
      ip6tables -w -t mangle -D PREROUTING -i br-$lan_iface -j DROP
    fi
  done
}

#Set the firewall config
#$1 is enable_firewall
#$2 is pkts_allowed
#$3 is wan_profile_handle
function set_firewall_config() {
  local default_eth_wan_profile_handle=$(util_execute_uci get qcmap_lan.@lan[0].default_waneth_profile_num)
  local default_wan_profile=$(util_execute_uci get qcmap_lan.@no_of_configs[0].default_pdn)
  local default_wwan_profile_idx=$(util_get_default_profile_index)
  local enable_fw_flag=$1
  local default_policy=$2
  local wan_profile_handle=$3
  local is_default_profile_handle=0

  log $(basename "$0") "set_firewall_config" $LINENO "enable_fw_flag=$enable_fw_flag default_policy=$default_policy wan_profile_handle=$wan_profile_handle"

  if [ $wan_profile_handle -eq $default_wan_profile ] || [ -n $default_eth_wan_profile_handle -a $wan_profile_handle -eq $default_eth_wan_profile_handle ];then
    is_default_profile_handle=1
  fi

  if [ -n $default_eth_wan_profile_handle ]; then 
    local default_waneth_profile_idx=$(util_get_profile_index $default_eth_wan_profile_handle)
  fi

  if [ $is_default_profile_handle -eq 0 ];then
    log $(basename "$0") "set_firewall_config" $LINENO "Profile is non-default"
    uci set qcmap_lan.@no_of_configs[0].is_common_firewall_config_applied=0
    uci commit qcmap_lan
    set_firewall_config_ex $enable_fw_flag $default_policy $wan_profile_handle
    return
  fi

  if [ -z $default_eth_wan_profile_handle ];then
    log $(basename "$0") "set_firewall_config" $LINENO "default waneth profile is not present"
    uci set qcmap_lan.@no_of_configs[0].is_common_firewall_config_applied=0
    uci commit qcmap_lan
    set_firewall_config_ex $enable_fw_flag $default_policy $wan_profile_handle
    return
  fi

  enable_fw_flag_wwan=$(uci get qcmap_lan.@profile[$default_wwan_profile_idx].firewall_enabled)
  default_fw_policy_wwan=$(uci get qcmap_lan.@profile[$default_wwan_profile_idx].pkts_allowed)
  enable_fw_flag_waneth=$(uci get qcmap_lan.@profile[$default_waneth_profile_idx].firewall_enabled)
  default_fw_policy_waneth=$(uci get qcmap_lan.@profile[$default_waneth_profile_idx].pkts_allowed)

  log $(basename "$0") "set_firewall_config" $LINENO "enable_fw_flag_wwan=$enable_fw_flag_wwan default_fw_policy_wwan=$default_fw_policy_wwan enable_fw_flag_waneth=$enable_fw_flag_waneth default_fw_policy_waneth=$default_fw_policy_waneth"

  if [ $enable_fw_flag_wwan -eq 1 -a $enable_fw_flag -eq 1 ]; then
    if [ $default_fw_policy_wwan -eq 1 -a $default_policy -eq 1 ]; then
      log $(basename "$0") "set_firewall_config" $LINENO "Firewall is already enabled with same config"
      return
    fi
  fi

  if [ $enable_fw_flag_wwan -eq 1 -a $enable_fw_flag -eq 1 ]; then
    if [ $default_fw_policy_wwan -eq 0 -a $default_policy -eq 0 ]; then
      log $(basename "$0") "set_firewall_config" $LINENO "Firewall is already enabled with same config"
      return
    fi
  fi

  if [ $enable_fw_flag_waneth -eq 1 -a $enable_fw_flag -eq 1 ]; then
    if [ $default_fw_policy_waneth -eq 1 -a $default_policy -eq 1 ]; then
      log $(basename "$0") "set_firewall_config" $LINENO "Firewall is already enabled with same config for waneth"
      return
    fi
  fi

  if [ $enable_fw_flag_waneth -eq 1 -a $enable_fw_flag -eq 1 ]; then
    if [ $default_fw_policy_waneth -eq 0 -a $default_policy -eq 0 ]; then
      log $(basename "$0") "set_firewall_config" $LINENO "Firewall is already enabled with same config for waneth"
      return
    fi
  fi

  if [ $enable_fw_flag_wwan -eq 0 -a $enable_fw_flag -eq 0 ]; then
    log $(basename "$0") "set_firewall_config" $LINENO "Firewall is already diabled with same config"
    return
  fi

  if [ $enable_fw_flag_waneth -eq 0 -a $enable_fw_flag -eq 0 ]; then
    log $(basename "$0") "set_firewall_config" $LINENO "Firewall is already diabled with same config for waneth"
    return
  fi

  uci set qcmap_lan.@no_of_configs[0].is_common_firewall_config_applied=0
  uci commit qcmap_lan
  set_firewall_config_ex $enable_fw_flag $default_policy $default_wan_profile
  set_firewall_config_ex $enable_fw_flag $default_policy $default_eth_wan_profile_handle

}

#Set the firewall config
#$1 is enable_firewall
#$2 is pkts_allowed
#$3 is wan_profile_handle
function set_firewall_config_ex() {
  topology=$(uci get qcmap_lan.@global[0].topology)
  no_of_wan_profiles=$(uci get qcmap_lan.@no_of_configs[0].no_of_profiles)
  local default_eth_wan_profile_handle=$(util_execute_uci get qcmap_lan.@lan[0].default_waneth_profile_num)
  local default_wan_profile=$(util_execute_uci get qcmap_lan.@no_of_configs[0].default_pdn)
  local default_wwan_profile_idx=$(util_get_default_profile_index)
  local default_waneth_profile_idx=$(util_get_profile_index $default_eth_wan_profile_handle)

  wan_profile_handle=$3

  log $(basename "$0") "set_firewall_config_ex" $LINENO "enable_firewall_flag=$1 pkts_allowed=$2 wan_profile_handle=$3"
  local idx=""

  #Here we are looping through all the profiles to find the config of current profile handle
  for i in $(seq $no_of_wan_profiles)
  do
    profile_id=$(uci get qcmap_lan.@profile[$((i-1))].profile_id)
    if [ $profile_id -eq $3 ];
    then
      prev_firewall_enabled_flag=$(uci get qcmap_lan.@profile[$((i-1))].firewall_enabled)
      prev_pkt_allowed_flag=$(uci get qcmap_lan.@profile[$((i-1))].pkts_allowed)
      util_execute_uci set qcmap_lan.@profile[$((i-1))].firewall_enabled=$1
      util_execute_uci set qcmap_lan.@profile[$((i-1))].pkts_allowed=$2
      uci commit
      enable_flag=$(uci get qcmap_lan.@profile[$((i-1))].firewall_enabled)
      default_policy=$(uci get qcmap_lan.@profile[$((i-1))].pkts_allowed)
      profile_index=$i
      break
    fi
  done

  #checking the IPPT feature mode and check the variable ippt_wo_nat_firewall_support is set or not
  ippt_fw_support_flag=$(check_ippt_feature_mode_with_firewall_support $wan_profile_handle)

  if [ $enable_flag -eq 0 ];
  then
    if [ $prev_firewall_enabled_flag -eq 0 ];then
        log  $(basename "$0") "set_firewall_config" $LINENO "Firewall is already disabled !!!"
        return
    fi

    echo 1 > /proc/sys/net/netfilter/nf_conntrack_tcp_loose
    wan_iface_ipv4=`get_wan_iface_name_from_wan_profile_handle $wan_profile_handle ipv4`
    wan_iface_ipv6=`get_wan_iface_name_from_wan_profile_handle $wan_profile_handle ipv6`
    if [ "$wan_iface_ipv4" ];then
      install_iptables_rules_to_drop_all_traffic $wan_iface_ipv4 $wan_profile_handle "ipv4"
    fi
    if [ "$wan_iface_ipv6" ];then
      install_iptables_rules_to_drop_all_traffic $wan_iface_ipv6 $wan_profile_handle "ipv6"
    fi
    #Whenever we are disabling firewall on a PDN, then we will add that wan iface from list network in wan_all zone
    /etc/data/backhaulWWANConfig.sh add_wan_iface_on_wan_all_zone $3

    util_execute_uci set qcmap_lan.@profile[$((profile_index-1))].pkts_allowed=0
    #removing dl accept rule when firewall is disabled on the PDN.
    idx=$(uci get qcmap_lan.@no_of_configs[0].no_of_rules)
    count=`uci show firewall | grep -i "wan$3_dl_accept" | awk -F'[][]' '{print $2}'`
    if [ "$count" ]; then
      uci delete firewall.@rule[$count]
      uci commit
      idx=$((idx-1))
    fi
    uci set qcmap_lan.@no_of_configs[0].no_of_rules=$idx
    uci commit

    disable_fw $3 1
    disable_fw $3 2
    delete_wl_firewall $3
    delete_bl_firewall $3

    util_delete_iptables_rules_for_gre_packet_marking $3 "ipv4"
    util_delete_iptables_rules_for_gre_packet_marking $3 "ipv6"

    if [ "$wan_iface_ipv4" ];then
      delete_iptables_rules_to_drop_all_traffic $wan_iface_ipv4 $wan_profile_handle "ipv4"
    fi

    if [ "$wan_iface_ipv6" ];then
      delete_iptables_rules_to_drop_all_traffic $wan_iface_ipv6 $wan_profile_handle "ipv6"
    fi
    delete_conntrack_on_disable_firewall $3 1
    delete_conntrack_on_disable_firewall $3 2

    echo $(basename "$0") "set_firewall_config : " $LINENO " Disable firewall done"
  fi

  if [ $enable_flag -eq 1 ] && [ $default_policy -eq 1 ]
  then
    #If previous pkt allowed flag is 1 ang again user is trying to set it to 1, then we are exiting
    if [ "$prev_pkt_allowed_flag" -eq 1 ]
    then
      log  $(basename "$0") "set_firewall_config" $LINENO "Firewall is already enabled with same config enable_flag=$enable_flag default_policy=$default_policy prev_pkt_allowed_flag=$prev_pkt_allowed_flag!!!" 
      return
    fi

    echo 0 > /proc/sys/net/netfilter/nf_conntrack_tcp_loose
    #since we are enabling whitelisting firewall, we are deleting blacklisting rules
    delete_bl_firewall $3

    #Whenever we are enabling firewall on a PDN, then we will remove that wan iface from list network in wan_all zone
    /etc/data/backhaulWWANConfig.sh del_wan_iface_from_wan_all_zone $3

    is_common_firewall_config_applied=$(uci get qcmap_lan.@no_of_configs[0].is_common_firewall_config_applied);
    if [ $is_common_firewall_config_applied -eq 0 ];then
      idx=$(uci get qcmap_lan.@no_of_configs[0].no_of_rules)
      idx=$((idx+1))

      uci add firewall rule
      uci set firewall.@rule[$idx]='rule'
      uci set firewall.@rule[$idx].name=whitelisting_matching_fwd_ul_v4_accept-$3
      if [ $3 -eq 1 ] || [ $topology == $QCMAP_SINGLE_BRIDGE_TOPOLOGY ] || [ $3 -eq $default_eth_wan_profile_handle ]; then
        uci set firewall.@rule[$idx].src='lan_wan'
      else
        uci set firewall.@rule[$idx].src=lan_wan$3
      fi
      if [ $3 -eq 1 ] || [ $3 -eq $default_eth_wan_profile_handle ]; then
        uci set firewall.@rule[$idx].dest='wan'
      else
        uci set firewall.@rule[$idx].dest=wan$3
      fi
      uci set firewall.@rule[$idx].target='ACCEPT'
      uci set firewall.@rule[$idx].proto='all'
      uci set firewall.@rule[$idx].extra='ct mark 0x35'
      uci set firewall.@rule[$idx].family='ipv4'
      uci set firewall.@rule[$idx].enabled='0'

      idx=$((idx+1))

      uci add firewall rule
      uci set firewall.@rule[$idx]='rule'
      uci set firewall.@rule[$idx].name=whitelisting_matching_fwd_ul_v6_accept-$3
      if [ $3 -eq 1 ] || [ $topology == $QCMAP_SINGLE_BRIDGE_TOPOLOGY ] || [ $3 -eq $default_eth_wan_profile_handle ]; then
        uci set firewall.@rule[$idx].src='lan_wan'
      else
        uci set firewall.@rule[$idx].src=lan_wan$3
      fi
      if [ $3 -eq 1 ] || [ $3 -eq $default_eth_wan_profile_handle ]; then
        uci set firewall.@rule[$idx].dest='wan'
      else
        uci set firewall.@rule[$idx].dest=wan$3
      fi
      uci set firewall.@rule[$idx].target='ACCEPT'
      uci set firewall.@rule[$idx].proto='all'
      uci set firewall.@rule[$idx].extra='ct mark 0x35'
      uci set firewall.@rule[$idx].family='ipv6'
      uci set firewall.@rule[$idx].enabled='0'


      idx=$((idx+1))

      uci add firewall rule
      uci set firewall.@rule[$idx]='rule'
      uci set firewall.@rule[$idx].name=whitelisting_matching_fwd_ul_v4_drop-$3
      if [ $3 -eq 1 ] || [ $3 -eq $default_eth_wan_profile_handle ]; then
      uci set firewall.@rule[$idx].src='lan_wan'
      else
        uci set firewall.@rule[$idx].src=lan_wan$3
      fi
      if [ $3 -eq 1 ] || [ $3 -eq $default_eth_wan_profile_handle ]; then
        uci set firewall.@rule[$idx].dest='wan'
      else
        uci set firewall.@rule[$idx].dest=wan$3
      fi
      uci set firewall.@rule[$idx].target='DROP'
      uci set firewall.@rule[$idx].extra='ct mark != 0x35'
      uci set firewall.@rule[$idx].proto='all'
      uci set firewall.@rule[$idx].family='ipv4'
      uci set firewall.@rule[$idx].enabled='0'

      idx=$((idx+1))

      uci add firewall rule
      uci set firewall.@rule[$idx]='rule'
      uci set firewall.@rule[$idx].name=whitelisting_matching_fwd_ul_v6_drop-$3
      if [ $3 -eq 1 ] || [ $3 -eq $default_eth_wan_profile_handle ]; then
      uci set firewall.@rule[$idx].src='lan_wan'
      else
        uci set firewall.@rule[$idx].src=lan_wan$3
      fi
      if [ $3 -eq 1 ] || [ $3 -eq $default_eth_wan_profile_handle ]; then
        uci set firewall.@rule[$idx].dest='wan'
      else
        uci set firewall.@rule[$idx].dest=wan$3
      fi
      uci set firewall.@rule[$idx].target='DROP'
      uci set firewall.@rule[$idx].extra='ct mark != 0x35'
      uci set firewall.@rule[$idx].proto='all'
      uci set firewall.@rule[$idx].family='ipv6'
      uci set firewall.@rule[$idx].enabled='0'

      idx=$((idx+1))

      uci add firewall rule
      uci set firewall.@rule[$idx]='rule'
      uci set firewall.@rule[$idx].name=whitelisting_matching_output_ul_v4_accept-$3
      if [ $3 -eq 1 ] || [ $3 -eq $default_eth_wan_profile_handle ]; then
        uci set firewall.@rule[$idx].dest='wan'
      else
        uci set firewall.@rule[$idx].dest=wan$3
      fi
      uci set firewall.@rule[$idx].target='ACCEPT'
      uci set firewall.@rule[$idx].proto='all'
      uci set firewall.@rule[$idx].extra='ct mark 0x35'
      uci set firewall.@rule[$idx].family='ipv4'
      uci set firewall.@rule[$idx].enabled='0'

      idx=$((idx+1))

      uci add firewall rule
      uci set firewall.@rule[$idx]='rule'
      uci set firewall.@rule[$idx].name=whitelisting_matching_output_ul_v6_accept-$3
      if [ $3 -eq 1 ] || [ $3 -eq $default_eth_wan_profile_handle ]; then
        uci set firewall.@rule[$idx].dest='wan'
      else
        uci set firewall.@rule[$idx].dest=wan$3
      fi
      uci set firewall.@rule[$idx].target='ACCEPT'
      uci set firewall.@rule[$idx].proto='all'
      uci set firewall.@rule[$idx].extra='ct mark 0x35'
      uci set firewall.@rule[$idx].family='ipv6'
      uci set firewall.@rule[$idx].enabled='0'

      idx=$((idx+1))

      uci add firewall rule
      uci set firewall.@rule[$idx]='rule'
      uci set firewall.@rule[$idx].name=whitelisting_matching_output_ul_v4_drop-$3
      if [ $3 -eq 1 ] || [ $3 -eq $default_eth_wan_profile_handle ]; then
        uci set firewall.@rule[$idx].dest='wan'
      else
        uci set firewall.@rule[$idx].dest=wan$3
      fi
      uci set firewall.@rule[$idx].target='DROP'
      uci set firewall.@rule[$idx].extra='ct mark != 0x35'
      uci set firewall.@rule[$idx].proto='all'
      uci set firewall.@rule[$idx].family='ipv4'
      uci set firewall.@rule[$idx].enabled='0'

      idx=$((idx+1))

      uci add firewall rule
      uci set firewall.@rule[$idx]='rule'
      uci set firewall.@rule[$idx].name=whitelisting_matching_output_ul_v6_drop-$3
      if [ $3 -eq 1 ] || [ $3 -eq $default_eth_wan_profile_handle ]; then
        uci set firewall.@rule[$idx].dest='wan'
      else
        uci set firewall.@rule[$idx].dest=wan$3
      fi
      uci set firewall.@rule[$idx].target='DROP'
      uci set firewall.@rule[$idx].extra='ct mark != 0x35'
      uci set firewall.@rule[$idx].proto='all'
      uci set firewall.@rule[$idx].family='ipv6'
      uci set firewall.@rule[$idx].enabled='0'

      idx=$((idx+1))

      uci add firewall rule
      uci set firewall.@rule[$idx]='rule'
      uci set firewall.@rule[$idx].name=whitelisting_matching_preroute_dl_v4_accept-$3
      if [ $3 -eq 1 ] || [ $3 -eq $default_eth_wan_profile_handle ]; then
        uci set firewall.@rule[$idx].src='wan'
      else
        uci set firewall.@rule[$idx].src=wan$3
      fi
      uci set firewall.@rule[$idx].dest='*'
      uci set firewall.@rule[$idx].target='ACCEPT'
      uci set firewall.@rule[$idx].proto='all'
      uci set firewall.@rule[$idx].mangle='1'
      uci set firewall.@rule[$idx].extra='ct mark 0x35'
      uci set firewall.@rule[$idx].family='ipv4'
      uci set firewall.@rule[$idx].enabled='0'

      idx=$((idx+1))

      uci add firewall rule
      uci set firewall.@rule[$idx]='rule'
      uci set firewall.@rule[$idx].name=whitelisting_matching_preroute_dl_v6_accept-$3
      if [ $3 -eq 1 ] || [ $3 -eq $default_eth_wan_profile_handle ]; then
        uci set firewall.@rule[$idx].src='wan'
      else
        uci set firewall.@rule[$idx].src=wan$3
      fi
      uci set firewall.@rule[$idx].dest='*'
      uci set firewall.@rule[$idx].target='ACCEPT'
      uci set firewall.@rule[$idx].proto='all'
      uci set firewall.@rule[$idx].mangle='1'
      uci set firewall.@rule[$idx].extra='ct mark 0x35'
      uci set firewall.@rule[$idx].family='ipv6'
      uci set firewall.@rule[$idx].enabled='0'

      idx=$((idx+1))

      uci add firewall rule
      uci set firewall.@rule[$idx]='rule'
      uci set firewall.@rule[$idx].name=whitelisting_matching_preroute_dl_v4_drop-$3
      if [ $3 -eq 1 ] || [ $3 -eq $default_eth_wan_profile_handle ]; then
        uci set firewall.@rule[$idx].src='wan'
      else
        uci set firewall.@rule[$idx].src=wan$3
      fi
      uci set firewall.@rule[$idx].dest='*'
      uci set firewall.@rule[$idx].target='DROP'
      uci set firewall.@rule[$idx].extra='ct mark != 0x35'
      uci set firewall.@rule[$idx].mangle='1'
      uci set firewall.@rule[$idx].proto='all'
      uci set firewall.@rule[$idx].family='ipv4'
      uci set firewall.@rule[$idx].enabled='0'

      idx=$((idx+1))

      uci add firewall rule
      uci set firewall.@rule[$idx]='rule'
      uci set firewall.@rule[$idx].name=whitelisting_matching_preroute_dl_v6_drop-$3
      if [ $3 -eq 1 ] || [ $3 -eq $default_eth_wan_profile_handle ]; then
        uci set firewall.@rule[$idx].src='wan'
      else
        uci set firewall.@rule[$idx].src=wan$3
      fi
      uci set firewall.@rule[$idx].dest='*'
      uci set firewall.@rule[$idx].target='DROP'
      uci set firewall.@rule[$idx].extra='ct mark != 0x35'
      uci set firewall.@rule[$idx].mangle='1'
      uci set firewall.@rule[$idx].proto='all'
      uci set firewall.@rule[$idx].family='ipv6'
      uci set firewall.@rule[$idx].enabled='0'

      idx=$((idx+1))

      uci add firewall rule
      uci set firewall.@rule[$idx]='rule'
      uci set firewall.@rule[$idx].name=wan$3_dl_accept
      if [ $3 -eq 1 ] || [ $3 -eq $default_eth_wan_profile_handle ]; then
        uci set firewall.@rule[$idx].src='wan'
      else
        uci set firewall.@rule[$idx].src=wan$3
      fi
      uci set firewall.@rule[$idx].dest='*'
      uci set firewall.@rule[$idx].target='ACCEPT'
      uci set firewall.@rule[$idx].proto='all'
      uci set firewall.@rule[$idx].enabled='1'

      uci commit

      idx=$((idx+1))

      uci add firewall rule
      uci set firewall.@rule[$idx]='rule'
      uci set firewall.@rule[$idx].name=wan$3_input_dl_accept
      if [ $3 -eq 1 ] || [ $3 -eq $default_eth_wan_profile_handle ]; then
        uci set firewall.@rule[$idx].src='wan'
      else
        uci set firewall.@rule[$idx].src=wan$3
      fi
      uci set firewall.@rule[$idx].target='ACCEPT'
      uci set firewall.@rule[$idx].proto='all'
      uci set firewall.@rule[$idx].enabled='1'

      uci commit

      uci set qcmap_lan.@no_of_configs[0].no_of_rules=$idx
      uci set qcmap_lan.@no_of_configs[0].is_common_firewall_config_applied=1
      uci commit

    fi
    if [ $ippt_fw_support_flag -eq 0 ];then
      enable_fw $3 1
    fi
    enable_fw $3 2

    util_install_iptables_rules_for_gre_packet_marking $3 "ipv4"
    util_install_iptables_rules_for_gre_packet_marking $3 "ipv6"

    if [ $ippt_fw_support_flag -eq 0 ];then
       check_external_ipv4_conntracks $3
    fi

    #Delete conntrack entries if any embedded conntrack entry generated during ongoing traffic.
    #Deletion of conntrack entries should happen after installing the iptable rules.
    profile_idx=`uci show qcmap_lan | grep -i "profile_id='$3'" | awk -F'[][]' '{print $2}'`
    if [ "$profile_idx" ]; then
      no_of_v4_rules=$(uci get qcmap_lan.@profile[$profile_idx].no_of_v4_fw_rules)
      if [ $no_of_v4_rules -eq 0 ] && [ $ippt_fw_support_flag -eq 0 ];then
         delete_external_ipv4_conntracks $3
      fi
    fi

    if [ -f "$V4_config_file" ] && [ $ippt_fw_support_flag -eq 0 ]
    then
      delete_conntrack_on_enable_firewall $3 1
    fi

    if [ -f "$V6_config_file" ]
    then
      delete_conntrack_on_enable_firewall $3 2
    fi
    log $(basename "$0") "set_firewall_config_ex" $LINENO  "Firewall is Enabled in WL"

elif [ $enable_flag -eq 1 ] && [ $default_policy -eq 0 ]
  then
    #If previous pkt allowed flag is 0 and again user is trying to set it to 0, then we are exiting
    if [ "$prev_pkt_allowed_flag" -eq 0 ] && [ "$prev_firewall_enabled_flag" -eq 1 ]
    then
      log  $(basename "$0") "set_firewall_config" $LINENO "Firewall is already enabled with same config enable_flag=$enable_flag default_policy=$default_policy prev_pkt_allowed_flag=$prev_pkt_allowed_flag!!!" 
      return
    fi

    echo 0 > /proc/sys/net/netfilter/nf_conntrack_tcp_loose
    #since we are enabling blacklisting firewall, we are deleting whitelisting rules
    delete_wl_firewall $3

    #Whenever we are enabling firewall on a PDN, then we will remove that wan iface from list network in wan_all zone
    /etc/data/backhaulWWANConfig.sh del_wan_iface_from_wan_all_zone $3

    is_common_firewall_config_applied=$(uci get qcmap_lan.@no_of_configs[0].is_common_firewall_config_applied);
    if [ $is_common_firewall_config_applied -eq 0 ];then
      idx=$(uci get qcmap_lan.@no_of_configs[0].no_of_rules)
      idx=$((idx+1))

      uci add firewall rule
      uci set firewall.@rule[$idx]='rule'
      uci set firewall.@rule[$idx].name=blacklisting_matching_fwd_ul_v4_drop-$3
      if [ $3 -eq 1 ] || [ $topology == $QCMAP_SINGLE_BRIDGE_TOPOLOGY ] || [ $3 -eq $default_eth_wan_profile_handle ]; then
        uci set firewall.@rule[$idx].src='lan_wan'
      else
        uci set firewall.@rule[$idx].src=lan_wan$3
      fi
      if [ $3 -eq 1 ] || [ $3 -eq $default_eth_wan_profile_handle ]; then
        uci set firewall.@rule[$idx].dest='wan'
      else
        uci set firewall.@rule[$idx].dest=wan$3
      fi
      uci set firewall.@rule[$idx].target='DROP'
      uci set firewall.@rule[$idx].proto='all'
      uci set firewall.@rule[$idx].extra='ct mark 0x35'
      uci set firewall.@rule[$idx].family='ipv4'
      uci set firewall.@rule[$idx].enabled='0'

      idx=$((idx+1))

      uci add firewall rule
      uci set firewall.@rule[$idx]='rule'
      uci set firewall.@rule[$idx].name=blacklisting_matching_fwd_ul_v6_drop-$3
      if [ $3 -eq 1 ] || [ $3 -eq $default_eth_wan_profile_handle ]; then
      uci set firewall.@rule[$idx].src='lan_wan'
      else
        uci set firewall.@rule[$idx].src=lan_wan$3
      fi
      if [ $3 -eq 1 ] || [ $3 -eq $default_eth_wan_profile_handle ]; then
        uci set firewall.@rule[$idx].dest='wan'
      else
        uci set firewall.@rule[$idx].dest=wan$3
      fi
      uci set firewall.@rule[$idx].target='DROP'
      uci set firewall.@rule[$idx].proto='all'
      uci set firewall.@rule[$idx].extra='ct mark 0x35'
      uci set firewall.@rule[$idx].family='ipv6'
      uci set firewall.@rule[$idx].enabled='0'

      idx=$((idx+1))

      uci add firewall rule
      uci set firewall.@rule[$idx]='rule'
      uci set firewall.@rule[$idx].name=blacklisting_matching_output_ul_v4_drop-$3
      if [ $3 -eq 1 ] || [ $3 -eq $default_eth_wan_profile_handle ]; then
        uci set firewall.@rule[$idx].dest='wan'
      else
        uci set firewall.@rule[$idx].dest=wan$3
      fi
      uci set firewall.@rule[$idx].target='DROP'
      uci set firewall.@rule[$idx].proto='all'
      uci set firewall.@rule[$idx].extra='ct mark 0x35'
      uci set firewall.@rule[$idx].family='ipv4'
      uci set firewall.@rule[$idx].enabled='0'

      idx=$((idx+1))

      uci add firewall rule
      uci set firewall.@rule[$idx]='rule'
      uci set firewall.@rule[$idx].name=blacklisting_matching_output_ul_v6_drop-$3
      if [ $3 -eq 1 ] || [ $3 -eq $default_eth_wan_profile_handle ]; then
        uci set firewall.@rule[$idx].dest='wan'
      else
        uci set firewall.@rule[$idx].dest=wan$3
      fi
      uci set firewall.@rule[$idx].target='DROP'
      uci set firewall.@rule[$idx].proto='all'
      uci set firewall.@rule[$idx].extra='ct mark 0x35'
      uci set firewall.@rule[$idx].family='ipv6'
      uci set firewall.@rule[$idx].enabled='0'

      idx=$((idx+1))

      uci add firewall rule
      uci set firewall.@rule[$idx]='rule'
      uci set firewall.@rule[$idx].name=blacklisting_matching_preroute_dl_v4_drop-$3
      if [ $3 -eq 1 ] || [ $3 -eq $default_eth_wan_profile_handle ]; then
        uci set firewall.@rule[$idx].src='wan'
      else
        uci set firewall.@rule[$idx].src=wan$3
      fi
      uci set firewall.@rule[$idx].dest='*'
      uci set firewall.@rule[$idx].target='DROP'
      uci set firewall.@rule[$idx].proto='all'
      uci set firewall.@rule[$idx].mangle='1'
      uci set firewall.@rule[$idx].extra='ct mark 0x35'
      uci set firewall.@rule[$idx].family='ipv4'
      uci set firewall.@rule[$idx].enabled='0'

      idx=$((idx+1))

      uci add firewall rule
      uci set firewall.@rule[$idx]='rule'
      uci set firewall.@rule[$idx].name=blacklisting_matching_preroute_dl_v6_drop-$3
      if [ $3 -eq 1 ] || [ $3 -eq $default_eth_wan_profile_handle ]; then
        uci set firewall.@rule[$idx].src='wan'
      else
        uci set firewall.@rule[$idx].src=wan$3
      fi
      uci set firewall.@rule[$idx].dest='*'
      uci set firewall.@rule[$idx].target='DROP'
      uci set firewall.@rule[$idx].proto='all'
      uci set firewall.@rule[$idx].mangle='1'
      uci set firewall.@rule[$idx].extra='ct mark 0x35'
      uci set firewall.@rule[$idx].family='ipv6'
      uci set firewall.@rule[$idx].enabled='0'

      idx=$((idx+1))

      uci add firewall rule
      uci set firewall.@rule[$idx]='rule'
      uci set firewall.@rule[$idx].name=wan$3_dl_accept
      if [ $3 -eq 1 ] || [ $3 -eq $default_eth_wan_profile_handle ]; then
        uci set firewall.@rule[$idx].src='wan'
      else
        uci set firewall.@rule[$idx].src=wan$3
      fi
      uci set firewall.@rule[$idx].dest='*'
      uci set firewall.@rule[$idx].target='ACCEPT'
      uci set firewall.@rule[$idx].proto='all'
      uci set firewall.@rule[$idx].enabled='1'

      uci commit

      idx=$((idx+1))

      uci add firewall rule
      uci set firewall.@rule[$idx]='rule'
      uci set firewall.@rule[$idx].name=wan$3_input_dl_accept
      if [ $3 -eq 1 ] || [ $3 -eq $default_eth_wan_profile_handle ]; then
        uci set firewall.@rule[$idx].src='wan'
      else
        uci set firewall.@rule[$idx].src=wan$3
      fi
      uci set firewall.@rule[$idx].target='ACCEPT'
      uci set firewall.@rule[$idx].proto='all'
      uci set firewall.@rule[$idx].enabled='1'

      uci commit

      uci set qcmap_lan.@no_of_configs[0].no_of_rules=$idx
      uci set qcmap_lan.@no_of_configs[0].is_common_firewall_config_applied=1
      uci commit
    fi

    if [ $ippt_fw_support_flag -eq 0 ];then
      enable_fw $3 1
    fi

    enable_fw $3 2

    if [ $ippt_fw_support_flag -eq 0 ];then
      check_external_ipv4_conntracks $3
    fi

    util_delete_iptables_rules_for_gre_packet_marking $3 "ipv4"
    util_delete_iptables_rules_for_gre_packet_marking $3 "ipv6"

    if [ -f "$V4_config_file" ] && [ $ippt_fw_support_flag -eq 0 ]
    then
      delete_conntrack_on_enable_firewall $3 1
    fi

    if [ -f "$V6_config_file" ]
    then
      delete_conntrack_on_enable_firewall $3 2
    fi
    log $(basename "$0") "set_firewall_config_ex" $LINENO  "Firewall is Enabled in BL"
  else
    log $(basename "$0") "set_firewall_config_ex" $LINENO  "Firewall is not Enabled"
  fi
}

#Delete the firewall entry on the basis of firewall handle
#$1 is firewall_handle
#$2 is wan_profile_handle
function delete_firewall() {
  log $(basename "$0") "delete_firewall" $LINENO "Entered delete_firewall firewall_handle=$1 wan_profile_handle=$2"
  cnt=0
  no_of_rules_added=$(uci get qcmap_firewall.@firewall[0].no_of_rules)
  idx=$(uci get qcmap_lan.@no_of_configs[0].no_of_rules)

  log $(basename "$0") "delete_firewall" $LINENO "no of rules in qcmap_firewall=$no_of_rules_added, qcmap_lan=$idx"

  #finding the profile index in qcmap_lan
  no_of_wan_profiles=$(uci get qcmap_lan.@no_of_configs[0].no_of_profiles)
  for i in $(seq $no_of_wan_profiles)
  do
    profile_id=$(uci get qcmap_lan.@profile[$((i-1))].profile_id)
    if [ $profile_id -eq $2 ];
    then
      profile_index=$((i-1))
      break
    fi
  done

  log $(basename "$0") "delete_firewall" $LINENO "profile_index=$profile_index"  

  for i in $(seq $no_of_rules_added)
  do
    firewall_handle=$(uci get qcmap_firewall.@firewall_rule[$((i-1))].firewall_handle)
    if [ $firewall_handle -eq $1 ];
    then
       fw_direction=`uci get qcmap_firewall.@firewall_rule[$((i-1))].direction`
       ip_version=`uci get qcmap_firewall.@firewall_rule[$((i-1))].family`
       log $(basename "$0") "delete_firewall" $LINENO "Firewall handle matched:$firewall_handle, dir=$fw_direction, ip_version=$ip_version"
       if [ $ip_version = "ipv4" ]
       then
         prev_v4_fw_count=`uci get qcmap_lan.@profile[$profile_index].no_of_v4_fw_rules`
         if [ $prev_v4_fw_count -gt 0 ]; then
           log $(basename "$0") "delete_firewall" $LINENO "no of v4 fw rules is : $prev_v4_fw_count"
           util_execute_uci set qcmap_lan.@profile[$profile_index].no_of_v4_fw_rules=$((prev_v4_fw_count-1))
           uci commit qcmap_lan
         delete_conntrack_entry_for_drop_ipv4_firewall_entries $1 $2
         else
            log $(basename "$0") "delete_firewall" $LINENO "fw v4 count is less tahn or equals to zero, no of v4 fw rules is : $prev_v4_fw_count"
         fi
       elif [ $ip_version = "ipv6" ]
       then
         prev_v6_fw_count=`uci get qcmap_lan.@profile[$profile_index].no_of_v6_fw_rules`
         if [ $prev_v6_fw_count -gt 0 ]; then
          log $(basename "$0") "delete_firewall" $LINENO "no of v6 fw rules is : $prev_v6_fw_count"
           util_execute_uci set qcmap_lan.@profile[$profile_index].no_of_v6_fw_rules=$((prev_v6_fw_count-1))
           uci commit qcmap_lan
         delete_conntrack_entry_for_drop_ipv6_firewall_entries $1 $2
         else
           log $(basename "$0") "delete_firewall" $LINENO "fw v6 count is less tahn or equals to zero, no of v4 fw rules is : $prev_v6_fw_count"
         fi
       fi
       if [ $fw_direction = "UL" ];then
           count=`uci show firewall | grep -i -w "$firewall_handle-0-$2" | awk -F'[][]' '{print $2}'`
           if [ "$count" ]; then
             util_execute_uci delete firewall.@rule[$count]
             idx=$((idx-1))
             uci commit firewall
           fi
           count=`uci show firewall | grep -i -w "$firewall_handle-1-$2" | awk -F'[][]' '{print $2}'`
           if [ "$count" ]; then
             util_execute_uci delete firewall.@rule[$count]
             idx=$((idx-1))
             uci commit firewall
           fi
       elif [ $fw_direction = "DL" ];then
           count=`uci show firewall | grep -i -w "$firewall_handle-0-$2" | awk -F'[][]' '{print $2}'`
           if [ "$count" ]; then
             util_execute_uci delete firewall.@rule[$count]
             idx=$((idx-1))
             uci commit firewall
           fi
       fi
       util_execute_uci set qcmap_lan.@no_of_configs[0].no_of_rules=$idx
       index_fw=`uci show qcmap_firewall | grep -i "$firewall_handle" | awk -F'[][]' '{print $2}'`
       util_execute_uci delete qcmap_firewall.@firewall_rule[$index_fw]
       util_execute_uci set qcmap_firewall.@firewall[0].no_of_rules=$((no_of_rules_added-1))

       log $(basename "$0") "delete_firewall" $LINENO "index_fw in qcmap_firewall = $index_fw"

       uci commit
       V4_config_file=/tmp/ipv4config$2
       V6_config_file=/tmp/ipv6config$2
       if [ -f "$V4_config_file" ] || [ -f "$V6_config_file" ]
       then
         /etc/init.d/firewall reload
         log $(basename "$0") "delete_firewall" $LINENO "Firewall deletion for handle=$1 done successfully"
       else
         log $(basename "$0") "delete_firewall" $LINENO "Backhaul down. firewall entry deleted from configuration file."
       fi
       break
    else
         cnt=$((cnt+1))
    fi
  done

  if [ $cnt -eq $no_of_rules_added ];
  then
    log $(basename "$0") "delete_firewall" $LINENO "No firewall Entry present for handle=$1"
  fi

}

#Set hardware filtering during reboot
function set_hw_filtering(){

state=$(uci get qcmap_firewall.@firewall[0].hw_filtering_state)
if [ $state -eq 0 ];then
  log  $(basename "$0") "set_hw_filtering" $LINENO "HW Filtering is disabled !!!"
  exit 1
fi

local ipa_swflt_cmd="ipa swflt "
local mac_filter_cmd=""
local mac_filter_params=""
local ip_seg_flt_cmd=""
local ip_seg_flt_params=""
local iface_flt_cmd=""
local iface_flt_params=""

#mac_filter_state equals to 1, states that Hw filtering is enabled based on mac
mac_filter_state=$(uci get qcmap_firewall.@mac_filtering[0].mac_filter_state)
if [ $mac_filter_state -eq 1 ]; then
  mac_filter_cmd="--mac-enable --add-mac "
  no_of_mac_clients=$(uci get qcmap_firewall.@mac_filtering[0].mac_filter_num_of_clients)
  for i in $(seq $no_of_mac_clients)
  do
    mac_addr=$(uci get qcmap_firewall.@mac_filtering[0].HWMACFilterClient$i)
    mac_filter_params="${mac_filter_params}$mac_addr "
  done
  mac_filter_cmd="${mac_filter_cmd}$mac_filter_params"
elif [ $mac_filter_state -eq 0 ]; then
  ipa swflt --mac-disable
fi

#ip_segment_filter_state equals to 1, states that Hw filtering is enabled based on ip segment
ip_segment_filter_state=$(uci get qcmap_firewall.@ip_segment_filtering[0].ip_segment_filter_state)
if [ $ip_segment_filter_state -eq 1 ]; then
  ip_seg_flt_cmd="--ip-enable --add-ip-seg "
  no_of_ip_segments=$(uci get qcmap_firewall.@ip_segment_filtering[0].num_of_ip_segment)
  for i in $(seq $no_of_ip_segments)
  do
    start_ip=$(uci get qcmap_firewall.@ip_segment_filtering[0].IPSegmentStart$i)
    end_ip=$(uci get qcmap_firewall.@ip_segment_filtering[0].IPSegmentEnd$i)
    ip_seg_flt_params="${ip_seg_flt_params}$start_ip $end_ip "
  done
  ip_seg_flt_cmd="${ip_seg_flt_cmd}$ip_seg_flt_params"
elif [ $ip_segment_filter_state -eq 0 ]; then
  ipa swflt --ip-disable
fi

#iface_filter_state equals to 1, states that Hw filtering is enabled based on iface name
iface_filter_state=$(uci get qcmap_firewall.@if_name_filtering[0].iface_filter_state)
if [ $iface_filter_state -eq 1 ]; then
  iface_flt_cmd="--port-enable --add-port "
  no_of_ifaces=$(uci get qcmap_firewall.@if_name_filtering[0].num_of_iface)
  for i in $(seq $no_of_ifaces)
  do
    iface=$(uci get qcmap_firewall.@if_name_filtering[0].InterfaceName$i)
    iface_flt_params="${iface_flt_params}$iface "
  done
  iface_flt_cmd="${iface_flt_cmd}$iface_flt_params"
elif [ $iface_filter_state -eq 0 ]; then
  ipa swflt --port-disable
fi

if [ $mac_filter_state -eq 1 ] || [ $ip_segment_filter_state -eq 1 ] || [ $iface_filter_state -eq 1 ]; then
  if [ ! -z "$mac_filter_cmd" ]; then
    #append MAC based HW filtering cmd params
    ipa_swflt_cmd="${ipa_swflt_cmd}$mac_filter_cmd"
  fi
  if [ ! -z "$ip_seg_flt_cmd" ]; then
    #append IP based HW filtering cmd params
    ipa_swflt_cmd="${ipa_swflt_cmd}$ip_seg_flt_cmd"
  fi
  if [ ! -z "$iface_flt_cmd" ]; then
    #append IFACE based HW filtering cmd params
    ipa_swflt_cmd="${ipa_swflt_cmd}$iface_flt_cmd"
  fi
  #now run the whole swflt command at once
  eval $ipa_swflt_cmd
fi
}

#Function to reset IPPT firewall parameters on bootup
function reset_IPPT_Firewall_params(){

  local is_firewall_reset_done=0
  #Delete firewall PREROUTING and OUTPUT entries
  #Check if PREROUTING rule exists
  local idx=$(uci get qcmap_lan.@no_of_configs[0].no_of_rules)
  local index=$(uci show firewall | grep -i "IPPT" | sort -r | awk -F'[][]' '{print $2}')
  if [ -n "$index" ]; then
    #delete firewall rule
    is_firewall_reset_done=1
    for rule_idx in $index
    do
      uci delete firewall.@rule[$rule_idx]
      idx=$((idx-1))
      uci set qcmap_lan.@no_of_configs[0].no_of_rules=$idx
    done
  fi
  uci commit
  if [ "$is_firewall_reset_done" -eq 1 ]; then
    /etc/init.d/firewall reload
  fi
}

#Function to disable all IPV4 firewall rules when IPPT feature mode is set w/o NAT
#$1 is wan_profile_handle
function disable_ipv4_fw_on_ippt_wo_nat_enablement(){
local profile_handle=$1

rules_index=`uci show firewall | grep -i "name" | grep -i "rule" | grep -i "$1'$" | awk -F'[][]' '{print $2}'`

for idx in $rules_index
do
  ip_version=`uci get firewall.@rule[$idx].family`
  if [ $ip_version = "ipv4" ];then
    uci set firewall.@rule[$idx].enabled='0'
  fi
done

uci commit firewall
/etc/init.d/firewall reload
}

#Function to enable all IPV4 firewall rules when IPPT feature mode is set with NAT
#$1 is wan_profile_handle
function enable_ipv4_fw_on_ippt_wo_nat_disablement(){
local profile_handle=$1

rules_index=`uci show firewall | grep -i "name" | grep -i "rule" | grep -i "$1'$" | awk -F'[][]' '{print $2}'`

for idx in $rules_index
do
  ip_version=`uci get firewall.@rule[$idx].family`
  if [ $ip_version = "ipv4" ];then
    uci set firewall.@rule[$idx].enabled='1'
  fi
done
uci commit firewall
/etc/init.d/firewall reload
}

function reset_firewall_on_ippt_wo_nat_mode_is_set_on_boot_up()
{
  no_of_profiles=$(uci get qcmap_lan.@no_of_configs[0].no_of_profiles)
  firewall_support_flag=$(uci get qcmap_lan.@no_of_configs[0].ippt_wo_nat_firewall_support)

  for i in $(seq $no_of_profiles)
  do
    profile_id=$(uci get qcmap_lan.@profile[$((i-1))].profile_id)
    #Quectel: Modify for IPPT With NAT Only one Device. By beale.hu
    profile_idx=`mbb_util_get_uci_profile_index $profile_id`
    ippt_feature_mode=$(util_get_with_nat $profile_idx)
    log $(basename "$0") "reset_firewall_on_ippt_wo_nat_mode_is_set_on_boot_up" $LINENO "ippt_feature_mode=$ippt_feature_mode"
    if [ $ippt_feature_mode -eq 2 -o $ippt_feature_mode -eq 3 ] && [ $firewall_support_flag -eq 0 ]; then
      disable_ipv4_fw_on_ippt_wo_nat_enablement $profile_id
    else
      enable_ipv4_fw_on_ippt_wo_nat_disablement $profile_id
    fi
  done

}

#function to install ipv4 vlan_to_non_wwan iptables rule
function install_ipv4_rules_to_block_vlan_access_to_non_wwan_backhaul() {
  local current_bh=`/etc/data/backhaulCommonConfig.sh read_current_backhaul`
  local current_bh_ipv4=$(echo $current_bh | awk -F ' ' '{print $1}')
  local current_bh_ipv6=$(echo $current_bh | awk -F ' ' '{print $2}')
  local defaultProfileId=$(util_execute_uci get qcmap_lan.@no_of_configs[0].default_pdn)

  if [ -z "$current_bh" ] || [ -z "${current_bh// }" ];then
    log  $(basename "$0") "install_rules_to_block_vlan_access_to_non_wwan_backhaul" $LINENO "current_backhaul=$current_bh or current_bh_ipv4=$current_bh_ipv4 is empty, returning!!"
    return
  fi

  if [ -z "$current_bh_ipv4" ] && [ ! -z "$current_bh_ipv6" ];then
    current_bh_ipv4=$current_bh_ipv6
  fi

  if [ $current_bh_ipv4 == "wan" ]; then
    log  $(basename "$0") "install_rules_to_block_vlan_access_to_non_wwan_backhaul" $LINENO "current backhaul is WWAN"
    return
  else
    log  $(basename "$0") "install_rules_to_block_vlan_access_to_non_wwan_backhaul" $LINENO "current v4 backhaul is $current_bh_ipv4"
    V4_config_file=/tmp/ipv4config.$current_bh_ipv4

    if [ ! -f $V4_config_file ]; then
      log  $(basename "$0") "install_rules_to_block_vlan_access_to_non_wwan_backhaul" $LINENO "Config file is not present V4_config_file=$V4_config_file"
      return
    fi

      local DEVNAME=`awk -F "=" 'FNR == 1 {print $2}' $V4_config_file | tr -d '"'`

    if [ ! -z $DEVNAME ]; then
       #clearing old rules
       /etc/data/lanUtils.sh util_clear_vlan_to_non_wwan_ipt_rule $DEVNAME

       #UL rule to drop packet originated from VLAN client when non-WWAN BH is active
       iptables -t filter -I FORWARD ! -i br-lan -o $DEVNAME -j DROP

       #DL rule to drop packet originated from non-WWAN BH to VLAN client.
       iptables -t filter -I FORWARD -i $DEVNAME ! -o br-lan -j DROP
    fi
  fi

}

#function to install ipv6 vlan_to_non_wwan iptables rule
function install_ipv6_rules_to_block_vlan_access_to_non_wwan_backhaul() {
  local current_bh=`/etc/data/backhaulCommonConfig.sh read_current_backhaul`
  local current_bh_ipv4=$(echo $current_bh | awk -F ' ' '{print $1}')
  local current_bh_ipv6=$(echo $current_bh | awk -F ' ' '{print $2}')
  local defaultProfileId=$(util_execute_uci get qcmap_lan.@no_of_configs[0].default_pdn)

  if [ -z "$current_bh" ] || [ -z "${current_bh// }" ];then
    log  $(basename "$0") "install_rules_to_block_vlan_access_to_non_wwan_backhaul" $LINENO "current_backhaul=$current_bh is empty, returning!!"
    return
  fi

  if [ -z "$current_bh_ipv6" ] && [ ! -z "$current_bh_ipv4" ];then
    current_bh_ipv6=$current_bh_ipv4
  fi

  if [ $current_bh_ipv6 == "wan" ]; then
    log  $(basename "$0") "install_rules_to_block_vlan_access_to_non_wwan_backhaul" $LINENO "current backhaul is WWAN"
    /etc/data/lanUtils.sh util_clear_vlan_to_non_wwan_ipt_rule
  else
    log  $(basename "$0") "install_rules_to_block_vlan_access_to_non_wwan_backhaul" $LINENO "current v6 backhaul is $current_bh_ipv6"

    V6_config_file=/tmp/ipv6config.$current_bh_ipv6

    if [ ! -f $V6_config_file ]; then
      log  $(basename "$0") "install_rules_to_block_vlan_access_to_non_wwan_backhaul" $LINENO "Config files are not present V4_config_file=$V4_config_file V6_config_file=$V6_config_file "
      return
    fi

      local DEVNAME=`awk -F "=" 'FNR == 1 {print $2}' $V6_config_file | tr -d '"'`

    if [ ! -z $DEVNAME ]; then
       log  $(basename "$0") "install_rules_to_block_vlan_access_to_non_wwan_backhaul" $LINENO "interface_name=$DEVNAME"

       #clearing old rules
       /etc/data/lanUtils.sh util_clear_vlan_to_non_wwan_ipt_rule $DEVNAME

       #UL rule to drop packet originated from VLAN client when non-WWAN BH is active
       ip6tables -t filter -I FORWARD ! -i br-lan -o $DEVNAME -j DROP

       #DL rule to drop packet originated from non-WWAN BH to VLAN client.
       ip6tables -t filter -I FORWARD -i $DEVNAME ! -o br-lan -j DROP
    fi
  fi

}

function update_extra_option() {
  shift 1
  local index=$1
  shift 1
  local extra="$@"
  uci set firewall.@rule[$index].extra="$extra"
  uci commit firewall
}

function install_drop_all_rule_for_disabled_ipv4() {

    idx=$(uci get qcmap_lan.@no_of_configs[0].no_of_rules)
    idx=$((idx+1))

    uci add firewall rule
    uci set firewall.@rule[$idx]='rule'
    uci set firewall.@rule[$idx].name='Drop-WAN-IPv4-Disabled'
    uci set firewall.@rule[$idx].src='wan'
    uci set firewall.@rule[$idx].dest='*'
    uci set firewall.@rule[$idx].proto='all'
    uci set firewall.@rule[$idx].target='DROP'
    uci set firewall.@rule[$idx].family='ipv4'
    uci set firewall.@rule[$idx].enabled='1'
    uci set firewall.@rule[$idx].mangle='1'

    uci reorder firewall.@rule[$idx]='0'

    idx=$((idx+1))

    uci add firewall rule
    uci set firewall.@rule[$idx]='rule'
    uci set firewall.@rule[$idx].name='Drop-WAN-All-IPv4-Disabled'
    uci set firewall.@rule[$idx].src='wan_all'
    uci set firewall.@rule[$idx].dest='*'
    uci set firewall.@rule[$idx].proto='all'
    uci set firewall.@rule[$idx].target='DROP'
    uci set firewall.@rule[$idx].family='ipv4'
    uci set firewall.@rule[$idx].enabled='1'
    uci set firewall.@rule[$idx].mangle='1'

    uci reorder firewall.@rule[$idx]='0'

    util_execute_uci commit firewall
    util_execute_uci set qcmap_lan.@no_of_configs[0].no_of_rules=$idx
    util_execute_uci commit qcmap_lan

    /etc/init.d/firewall reload
}

function install_drop_all_rule_for_disabled_ipv6() {

    idx=$(uci get qcmap_lan.@no_of_configs[0].no_of_rules)
    idx=$((idx+1))

    uci add firewall rule
    uci set firewall.@rule[$idx]='rule'
    uci set firewall.@rule[$idx].name='Drop-WAN-IPv6-Disabled'
    uci set firewall.@rule[$idx].src='wan'
    uci set firewall.@rule[$idx].dest='*'
    uci set firewall.@rule[$idx].proto='all'
    uci set firewall.@rule[$idx].target='DROP'
    uci set firewall.@rule[$idx].family='ipv6'
    uci set firewall.@rule[$idx].enabled='1'
    uci set firewall.@rule[$idx].mangle='1'

    uci reorder firewall.@rule[$idx]='0'

    idx=$((idx+1))

    uci add firewall rule
    uci set firewall.@rule[$idx]='rule'
    uci set firewall.@rule[$idx].name='Drop-WAN-All-IPv6-Disabled'
    uci set firewall.@rule[$idx].src='wan_all'
    uci set firewall.@rule[$idx].dest='*'
    uci set firewall.@rule[$idx].proto='all'
    uci set firewall.@rule[$idx].target='DROP'
    uci set firewall.@rule[$idx].family='ipv6'
    uci set firewall.@rule[$idx].enabled='1'
    uci set firewall.@rule[$idx].mangle='1'

    uci reorder firewall.@rule[$idx]='0'

    util_execute_uci commit firewall
    util_execute_uci set qcmap_lan.@no_of_configs[0].no_of_rules=$idx
    util_execute_uci commit qcmap_lan

    /etc/init.d/firewall reload
}

function delete_drop_all_rule_for_disabled_ipv4() {

  idx=$(uci get qcmap_lan.@no_of_configs[0].no_of_rules)
  count1=`uci show firewall | grep -i -w "Drop-WAN-IPv4-Disabled" | awk -F'[][]' '{print $2}'`
  if [ "$count1" ]; then
    uci delete firewall.@rule[$count1]
    uci commit firewall
    idx=$((idx-1))
  fi

  count2=`uci show firewall | grep -i -w "Drop-WAN-All-IPv4-Disabled" | awk -F'[][]' '{print $2}'`
  if [ "$count2" ]; then
    uci delete firewall.@rule[$count2]
    uci commit firewall
    idx=$((idx-1))
  fi

  util_execute_uci set qcmap_lan.@no_of_configs[0].no_of_rules=$idx
  uci commit qcmap_lan
  /etc/init.d/firewall reload
}

function delete_drop_all_rule_for_disabled_ipv6() {

  idx=$(uci get qcmap_lan.@no_of_configs[0].no_of_rules)
  count1=`uci show firewall | grep -i -w "Drop-WAN-IPv6-Disabled" | awk -F'[][]' '{print $2}'`
  if [ "$count1" ]; then
    uci delete firewall.@rule[$count1]
    uci commit firewall
    idx=$((idx-1))
  fi

  count2=`uci show firewall | grep -i -w "Drop-WAN-All-IPv6-Disabled" | awk -F'[][]' '{print $2}'`
  if [ "$count2" ]; then
    uci delete firewall.@rule[$count2]
    uci commit firewall
    idx=$((idx-1))
  fi

  util_execute_uci set qcmap_lan.@no_of_configs[0].no_of_rules=$idx
  uci commit qcmap_lan
  /etc/init.d/firewall reload
}

case $1 in
  set_firewall)
    set_firewall_config $2 $3 $4
    ;;
  enable_firewall)
    enable_fw $2 $3
    ;;
  disable_firewall)
    disable_fw $2 $3
    ;;
  delete_firewall)
    delete_firewall $2 $3
    ;;
  display_firewall)
    display_firewall $2
    ;;
  set_hw_filtering)
    set_hw_filtering
    ;;
  reset_IPPT_Firewall_params)
    reset_IPPT_Firewall_params
    ;;
  check_external_ipv4_conntracks)
    check_external_ipv4_conntracks $2
    ;;
  delete_ipset_for_rmnet_gateway)
    delete_ipset_for_rmnet_gateway $2
    ;;
  disable_ipv4_fw_on_ippt_wo_nat_enablement)
    disable_ipv4_fw_on_ippt_wo_nat_enablement $2
    ;;
  reset_firewall_on_ippt_wo_nat_mode_is_set_on_boot_up)
    reset_firewall_on_ippt_wo_nat_mode_is_set_on_boot_up
    ;;
  install_ipv4_rules_to_block_vlan_access_to_non_wwan_backhaul)
    install_ipv4_rules_to_block_vlan_access_to_non_wwan_backhaul
    ;;
  install_ipv6_rules_to_block_vlan_access_to_non_wwan_backhaul)
    install_ipv6_rules_to_block_vlan_access_to_non_wwan_backhaul
    ;;
  update_extra_option)
    update_extra_option $*
    ;;
  install_drop_all_rule_for_disabled_ipv6)
    install_drop_all_rule_for_disabled_ipv6
    ;;
  install_drop_all_rule_for_disabled_ipv4)
    install_drop_all_rule_for_disabled_ipv4
    ;;
  delete_drop_all_rule_for_disabled_ipv6)
    delete_drop_all_rule_for_disabled_ipv6
    ;;
  delete_drop_all_rule_for_disabled_ipv4)
    delete_drop_all_rule_for_disabled_ipv4
    ;;
  *)
    log $(basename "$0") "" $LINENO ":Invalid option"
    ;;
esac
