#!/bin/sh

# Copyright (c) 2023 Qualcomm Technologies, Inc.
# All Rights Reserved.
# Confidential and Proprietary - Qualcomm Technologies, Inc.

#source common functions
. /etc/data/mbbUtils.sh
. /etc/data/lanUtils.sh

IPV6_DUMMY_PD_ACTIVATED=2
IPV6_ENABLE_PD_ACTIVATED=1
IPV6_DISABLE_PD_ACTIVATED=0

IPV6_DELEGATED_PREFIX_STATUS_ENABLE=1
IPV6_DELEGATED_PREFIX_STATUS_DISABLE=0

IPV6_ENABLE_EXT_ROUTER_MODE=1
IPV6_DISABLE_EXT_ROUTER_MODE=0

IPV6_ENABLE_PD_MANAGER=1
IPV6_DISABLE_PD_MANAGER=0

IPV6_DHCP_ENABLE_PD_CONFIG=1
IPV6_DHCP_DISABLE_PD_CONFIG=0

# used for dhcp config
IPV6_DHCP_PREFIX_SHARING_MODE=0
IPV6_DHCP_LEGACY_PD_MODE=1
IPV6_DHCP_IDU_PD_MODE=2

IPV6_BR_LAN_ADDRESS_DISABLE=0
IPV6_BR_LAN_ADDRESS_ENABLE=1

#dhcp file have "-rw------" authority, qcmap can not write the file.
#So set the pd_activated config to qcmap_lan first, qcmap can read pd_activated config from qcmap_lan
function qcmap_lan_enable_pd_activated() {
  local enable_status=$1

  if [ "$enable_status" == "$IPV6_ENABLE_PD_ACTIVATED" ] ; then
    util_execute_uci set qcmap_lan.@lan[0].qcmap_prefix_delegation_activated=1
  elif [ "$enable_status" == "$IPV6_DISABLE_PD_ACTIVATED" ] ; then
    util_execute_uci -q delete qcmap_lan.@lan[0].qcmap_prefix_delegation_activated
  elif [ "$enable_status" == "$IPV6_DUMMY_PD_ACTIVATED" ] ; then
    # IDU mode, Disable the PD first, will set this config to 0;
    # then bring down BH, will delete this config
    util_execute_uci set qcmap_lan.@lan[0].qcmap_prefix_delegation_activated=0
  fi

  util_execute_uci commit qcmap_lan

  log  $(basename "$0") "qcmap_lan_enable_pd_activated" $LINENO "qcmap_lan enable pd activated config: $enable_status"
}

function set_delegated_prefix_available_status() {
  local delegated_prefix_available_status=$1

  if [ "$delegated_prefix_available_status" == "$IPV6_DELEGATED_PREFIX_STATUS_ENABLE" ] ; then
    util_execute_uci set qcmap_lan.@lan[0].qcmap_delegated_prefix_available=1
  elif [ "$delegated_prefix_available_status" == "$IPV6_DELEGATED_PREFIX_STATUS_DISABLE" ] ; then
    util_execute_uci set qcmap_lan.@lan[0].qcmap_delegated_prefix_available=0
  fi

  util_execute_uci commit qcmap_lan

  log  $(basename "$0") "set_delegated_prefix_available_status" $LINENO "delegated prefix available status: $delegated_prefix_available_status"
}

function qcmap_lan_enable_ext_router_mode() {
  local enable_status=$1
  local defaultProfileId

  if [ "$enable_status" == "$IPV6_ENABLE_EXT_ROUTER_MODE" ] ; then
    util_execute_uci set qcmap_lan.@lan[0].qcmap_ext_router_mode_enabled=1
    util_execute_uci set dhcp.lan.ra_slaac='0'
  elif [ "$enable_status" == "$IPV6_DISABLE_EXT_ROUTER_MODE" ] ; then
    util_execute_uci -q delete qcmap_lan.@lan[0].qcmap_ext_router_mode_enabled
    util_execute_uci set dhcp.lan.ra_slaac='1'
  fi

  defaultProfileId=`uci get qcmap_lan.@no_of_configs[0].default_pdn`
  util_enable_disable_IPV6_SNAT "$defaultProfileId" "$enable_status"

  util_execute_uci commit qcmap_lan
  util_execute_uci commit dhcp
  util_run_command /etc/init.d/odhcpd reload

  log  $(basename "$0") "qcmap_lan_enable_ext_router_mode" $LINENO "qcmap_lan enable external router mode: $enable_status"
}

function qcmap_lan_enable_pd_manager() {
  local enable_status=$1
  local defaultProfileId
  local profile_idx
  local current_bh_v6
  local qcmap_lan_pd_activated
  local qcmap_lan_ext_router_mode_enabled

  if [ "$enable_status" == "$IPV6_ENABLE_PD_MANAGER" ] ; then
    # make sure dhcp.lan.qcmap_prefix_delegation_moder is empty when enable a new mode, then update dhcp pd config will correct
    util_execute_uci -q delete dhcp.lan.qcmap_prefix_delegation_mode
    util_execute_uci_set "qcmap_lan.@lan[0].pd_manager" "/dev/socket/data/pd_manager"
    util_execute_uci commit qcmap_lan
    util_execute_uci commit dhcp

    defaultProfileId=`uci get qcmap_lan.@no_of_configs[0].default_pdn`
    profile_idx=$(util_get_profile_index $defaultProfileId)
    current_bh_v6=$(uci get qcmap_lan.@profile[$profile_idx].bh_present_v6)
    log $(basename "$0") ":qcmap_lan_enable_pd_manager:" $LINENO "current_bh_v6=$current_bh_v6"

    if [ "$current_bh_v6" == "wan"  ]; then
      # add dhcp pd config when enable ipv6 router mode
      dhcp_enable_pd_config $IPV6_DHCP_ENABLE_PD_CONFIG
    fi
  elif [ "$enable_status" == "$IPV6_DISABLE_PD_MANAGER" ] ; then
    # delete dhcp pd config when disable ipv6 router mode
    dhcp_enable_pd_config $IPV6_DHCP_DISABLE_PD_CONFIG

    qcmap_lan_pd_activated=$(util_execute_uci get qcmap_lan.@lan[0].qcmap_prefix_delegation_activated)
    qcmap_lan_ext_router_mode_enabled=$(util_execute_uci get qcmap_lan.@lan[0].qcmap_ext_router_mode_enabled)
    # clear qcmap_lan pf config when no mode enabled
    if [ -z "$qcmap_lan_pd_activated" ] && [ -z "$qcmap_lan_ext_router_mode_enabled" ]; then
      util_execute_uci -q delete qcmap_lan.@lan[0].pd_manager
      util_execute_uci -q delete qcmap_lan.@lan[0].qcmap_delegated_prefix_available
      util_execute_uci commit qcmap_lan
    fi
  fi

  log  $(basename "$0") "qcmap_lan_enable_pd_manager" $LINENO "set pd_manager status: $enable_status"
}

function enable_br_lan_ipv6_address() {
  local enable_status=$1
  local defaultProfileId
  local downstream
  local prefix6
  local rtTable
  local dhcp_pd_mode

  defaultProfileId=`util_execute_uci get qcmap_lan.@no_of_configs[0].default_pdn`
  if [ ! -f /tmp/ipv6config$defaultProfileId ]; then
    log $(basename "$0") ":enable_br_lan_ipv6_address:" $LINENO "file /tmp/ipv6config$defaultProfileId not exist"
    return
  fi

  if [ "$enable_status" == "$IPV6_BR_LAN_ADDRESS_DISABLE" ] ; then
    util_execute_uci -q delete network.lan_bind6.ip6addr
    util_execute_uci commit network
    util_run_command ubus call network reload
    util_run_command sleep 0.2
    log  $(basename "$0") ":enable_br_lan_ipv6_address:" $LINENO "delete br_lan ipv6 address"
  elif [ "$enable_status" == "$IPV6_BR_LAN_ADDRESS_ENABLE" ] ; then
    util_execute_uci set qcmap_lan.@lan[0].br_lan_addr6_add_flag=1
    util_execute_uci commit qcmap_lan

    . /tmp/ipv6config$defaultProfileId
    prefix6=$PREFIX6
    downstream="lan"
    rtTable=$(util_execute_uci get network.${downstream}_bind6.ip6table)
    util_rmnet_lan_setup_ip6 $downstream $prefix6 $rtTable
    log $(basename "$0") ":enable_br_lan_ipv6_address:" $LINENO "util_rmnet_lan_setup_ip6 $downstream $prefix6 $rtTable"

    # delete the flag after update br-lan ipv6 address
    util_execute_uci -q delete qcmap_lan.@lan[0].br_lan_addr6_add_flag
    util_execute_uci commit qcmap_lan
  fi
}

# Function to perform link toggle
function pd_link_toggle() {
  local downdevice
  network_get_device downdevice "lan"

  util_perform_link_toggle "USB" $downdevice
  util_perform_link_toggle "ETH" $downdevice
  util_perform_link_toggle "ETH_NIC2" $downdevice
  /etc/data/wlanConfig.sh restart_tethered_wlan_client

  log $(basename "$0") ":pd_link_toggle:" $LINENO "Restart tethered clients Success "
}

#we check the expected PD status and current status here,
#if not align, do the setting and reload;
#if align, do nothing.
function dhcp_enable_pd_config() {
  local enable_status=$1
  local dhcp_pd_mode
  local qcmap_lan_pd_manager
  local qcmap_lan_pd_activated
  local qcmap_lan_ext_router_mode_enabled
  local delegated_prefix_available

  dhcp_pd_mode=$(util_execute_uci get dhcp.lan.qcmap_prefix_delegation_mode)
  qcmap_lan_pd_manager=$(util_execute_uci get qcmap_lan.@lan[0].pd_manager)
  qcmap_lan_pd_activated=$(util_execute_uci get qcmap_lan.@lan[0].qcmap_prefix_delegation_activated)
  qcmap_lan_ext_router_mode_enabled=$(util_execute_uci get qcmap_lan.@lan[0].qcmap_ext_router_mode_enabled)
  delegated_prefix_available=$(util_execute_uci get qcmap_lan.@lan[0].qcmap_delegated_prefix_available)

  if [ "$enable_status" == "$IPV6_DHCP_ENABLE_PD_CONFIG" ] && [ -z "$dhcp_pd_mode" ] && [ -n "$qcmap_lan_pd_manager" ]; then
    if [ -n "$qcmap_lan_pd_activated" ] || [ -n "$qcmap_lan_ext_router_mode_enabled" ] ; then

      # Legacy or IDU mode
      # qcmap_lan_pd_activated(1) && qcmap_lan_ext_router_mode_enabled(1)     && delegated_prefix_available(1)  -->IDU PD mode
      # qcmap_lan_pd_activated(1) && qcmap_lan_ext_router_mode_enabled(empty) && delegated_prefix_available(1)  -->legacy PD mode
      if [ -n "$qcmap_lan_pd_activated" ] && [ "$delegated_prefix_available" == "$IPV6_DELEGATED_PREFIX_STATUS_ENABLE" ]; then
        if [ -n "$qcmap_lan_ext_router_mode_enabled" ]; then
          util_execute_uci del_list dhcp.lan.ra_flags='managed-config'
          util_execute_uci add_list dhcp.lan.ra_flags='managed-config'
          util_execute_uci set dhcp.lan.qcmap_prefix_delegation_mode=$IPV6_DHCP_IDU_PD_MODE
        else
          util_execute_uci del_list dhcp.lan.ra_flags='managed-config'
          util_execute_uci set dhcp.lan.qcmap_prefix_delegation_mode=$IPV6_DHCP_LEGACY_PD_MODE
        fi

      # Sharing mode
      # qcmap_lan_pd_activated(1)     && qcmap_lan_ext_router_mode_enabled(1) && delegated_prefix_available(0)  -->Prefix sharing mode
      # qcmap_lan_pd_activated(empty) && qcmap_lan_ext_router_mode_enabled(1) && delegated_prefix_available(0)  -->Prefix sharing mode
      # qcmap_lan_pd_activated(empty) && qcmap_lan_ext_router_mode_enabled(1) && delegated_prefix_available(1)  -->Prefix sharing mode
      elif [ -n "$qcmap_lan_ext_router_mode_enabled" ]; then
          util_execute_uci del_list dhcp.lan.ra_flags='managed-config'
          util_execute_uci add_list dhcp.lan.ra_flags='managed-config'
          util_execute_uci set dhcp.lan.qcmap_prefix_delegation_mode=$IPV6_DHCP_PREFIX_SHARING_MODE

      # qcmap_lan_pd_activated(1) && qcmap_lan_ext_router_mode_enabled(empty) && delegated_prefix_available(0)  -->as normal mode
      # for legacy mode, need update br_lan ipv6 address even prefix not available
      else
        if [ -n "$qcmap_lan_pd_activated" ] && [ -z "$qcmap_lan_ext_router_mode_enabled" ]; then
          enable_br_lan_ipv6_address $IPV6_BR_LAN_ADDRESS_ENABLE
        fi

        log $(basename "$0") ":dhcp_enable_pd_config:" $LINENO "prefix not available, as normal mode"
        return
      fi

      util_execute_uci_set "dhcp.lan.pd_manager" "$qcmap_lan_pd_manager"
      util_execute_uci commit dhcp
      error=$(chmod 666 /dev/socket/data/pd_manager 2>&1) ||
            log  $(basename "$0") "dhcp_enable_pd_config" $LINENO "failed to chmod :$error"

      util_run_command /etc/init.d/odhcpd reload

      log $(basename "$0") ":dhcp_enable_pd_config:" $LINENO "Add dhcp pd config and reload odhcpd"
    fi
  elif [ "$enable_status" == "$IPV6_DHCP_DISABLE_PD_CONFIG" ] && [ -n "$dhcp_pd_mode" ]; then
    util_execute_uci -q delete dhcp.lan.pd_manager
    util_execute_uci -q delete dhcp.lan.qcmap_prefix_delegation_mode
    util_execute_uci del_list dhcp.lan.ra_flags='managed-config'

    # IDU mode, Disable the PD first, then bring down BH
    if [ "$qcmap_lan_pd_activated" == "0" ] ; then
      util_execute_uci -q delete qcmap_lan.@lan[0].qcmap_prefix_delegation_activated
      util_execute_uci commit qcmap_lan
      log  $(basename "$0") ":dhcp_enable_pd_config:" $LINENO "delete qcmap qcmap_prefix_delegation_activated config"
    fi

    util_execute_uci commit dhcp
    # Legacy and IDU mode receive different info from client, restart odhcpd when a mode disabled
    util_run_command /etc/init.d/odhcpd restart

    # toggle tethered link after mode disabled
    pd_link_toggle

    log $(basename "$0") ":dhcp_enable_pd_config:" $LINENO "Delete dhcp pd config and toggle link success"
  fi

  if [ -n "$qcmap_lan_pd_manager" ]; then
    update_network_ipv6addr_ps
  fi

  # for legacy mode, need update br_lan ipv6 address to normal
  if [ -n "$qcmap_lan_pd_activated" ] && [ -z "$qcmap_lan_ext_router_mode_enabled" ]; then
    enable_br_lan_ipv6_address $IPV6_BR_LAN_ADDRESS_ENABLE
  fi

  # toggle tethered link after mode enabled
  dhcp_pd_mode=$(util_execute_uci get dhcp.lan.qcmap_prefix_delegation_mode)
  if [ -n "$dhcp_pd_mode" ]; then
    pd_link_toggle
  fi
}

# update br-lan ipv6addr when mode is prefix sharing;
# update br-lan to normal prefix when no mode enabled
function update_network_ipv6addr_ps() {
  local defaultProfileId
  local downstream
  local prefix6
  local rtTable
  local dhcp_pd_mode

  defaultProfileId=`util_execute_uci get qcmap_lan.@no_of_configs[0].default_pdn`
  if [ ! -f /tmp/ipv6config$defaultProfileId ]; then
    log $(basename "$0") ":update_network_ipv6addr_ps:" $LINENO "file /tmp/ipv6config$defaultProfileId not exist"
    return
  fi

  dhcp_pd_mode=$(util_execute_uci get dhcp.lan.qcmap_prefix_delegation_mode)
  if [ -z "$dhcp_pd_mode" ] || [ "$dhcp_pd_mode" == "$IPV6_DHCP_PREFIX_SHARING_MODE" ]; then
    . /tmp/ipv6config$defaultProfileId
    prefix6=$PREFIX6
    downstream="lan"
    rtTable=$(util_execute_uci get network.${downstream}_bind6.ip6table)
    util_rmnet_lan_setup_ip6 $downstream $prefix6 $rtTable
    log $(basename "$0") ":update_network_ipv6addr_ps:" $LINENO "util_rmnet_lan_setup_ip6 $downstream $prefix6 $rtTable"
  fi

  log $(basename "$0") ":update_network_ipv6addr_ps:" $LINENO "END!"
}

case $1 in
    qcmap_lan_enable_pd_activated)
      log  $(basename "$0") "case:qcmap_lan_enable_pd_activated" $LINENO "qcmap_lan_enable_pd_activated"
      qcmap_lan_enable_pd_activated $2
      ;;
    set_delegated_prefix_available_status)
      log  $(basename "$0") "case:set_delegated_prefix_available_status" $LINENO "set_delegated_prefix_available_status"
      set_delegated_prefix_available_status $2
      ;;
    qcmap_lan_enable_ext_router_mode)
      log  $(basename "$0") "case:qcmap_lan_enable_ext_router_mode" $LINENO "qcmap_lan_enable_ext_router_mode"
      qcmap_lan_enable_ext_router_mode $2
      ;;
    qcmap_lan_enable_pd_manager)
      log  $(basename "$0") "case:qcmap_lan_enable_pd_manager" $LINENO "qcmap_lan_enable_pd_manager"
      qcmap_lan_enable_pd_manager $2
      ;;
    dhcp_enable_pd_config)
      log  $(basename "$0") "case:dhcp_enable_pd_config" $LINENO "dhcp_enable_pd_config"
      dhcp_enable_pd_config $2
      ;;
    update_network_ipv6addr_ps)
      log  $(basename "$0") "case:update_network_ipv6addr_ps" $LINENO "update_network_ipv6addr_ps"
      update_network_ipv6addr_ps
      ;;
    enable_br_lan_ipv6_address)
      log  $(basename "$0") "case:enable_br_lan_ipv6_address" $LINENO "enable_br_lan_ipv6_address"
      enable_br_lan_ipv6_address $2
      ;;
    *)
    echo $(basename "$0")":"$LINENO":Invalid option"
     ;;
esac