#!/bin/sh
#  Copyright (c) 2023 Qualcomm Technologies, Inc.
#  All Rights Reserved.
#  Confidential and Proprietary - Qualcomm Technologies, Inc.

#  Usage: backhaulCommonConfig.sh set_bh_pref wan1 wan2 wan3 wan4

. /etc/data/mbbUtils.sh

#source common functions
. /etc/data/lanUtils.sh

#source ip collision file
. /etc/data/ip_collision.sh

#source tinyproxy
[ -f /etc/data/tinyproxyConfig.sh ] && . /etc/data/tinyproxyConfig.sh

. /etc/data/firewallConfig.sh

QCMAP_WLAN_MODE_AP_STA=3
QCMAP_WLAN_MODE_AP_AP_STA=5

QCMAP_STA_CONNECTION_DYNAMIC=1
QCMAP_STA_CONNECTION_STATIC=2

IPV6_DHCP_ENABLE_PD_CONFIG=1
IPV6_DHCP_DISABLE_PD_CONFIG=0

QCMAP_BH_LOAD_BALANCE_DISABLE=0
QCMAP_BH_LOAD_BALANCE_ENABLE=1

qcmap_bh_load_balance_enabled=0

if [[ "$#" -le 0 ]]; then
    echo "Usage: backhaulCommonConfig.sh set_bh_pref wan1 wan2 wan3 wan4 wan5"
    echo "       backhaulCommonConfig.sh switch_bh"
    return 1
elif [ \( "$1" == "set_bh_pref" \) -a \( "$#" -ne 6 \) ]; then
    echo "Usage: backhaulCommonConfig.sh set_bh_pref wan1 wan2 wan3 wan4 wan5"
    return 1
elif [ \( "$1" == "switch_bh" \) -a \( "$#" -lt 3 \) ]; then
    echo "Usage: backhaulCommonConfig.sh set_bh_pref"
    return 1
fi

#Function to update IPV6 and Address and prefix Information
#to /tmp/ipv6config.<wan> file. incase of of non WAN case.
function update_prefix_info_to_tmp_file() {
	local filename
	local ifname var quartet_count bit_count tmp
	local last_quartet start end BIN current_bh_v6
    local prefix_info="" prefix_len ipv6_address

    #Get File name.
	current_bh_v6=$(util_execute_uci get qcmap_lan.@profile[$profile_idx].bh_present_v6)
	if [ -z $current_bh_v6 ] || [ $current_bh_v6 == "wan" ]; then
	  log $(basename "$0") "update_prefix_info_to_tmp_file" $LINENO "current_bh_v6 is empty or its WAN BH"
	  return
	fi
	#Get IFNAME from tmp file.
	filename="/tmp/ipv6config.${current_bh_v6}"
    ifname=$(cat $filename | grep IFNAME | awk -F "=" '{print $2}' | tr -d '"')
	if [ -z $ifname ]; then
	  log $(basename "$0") "update_prefix_info_to_tmp_file" $LINENO "ifname[$ifname] is empty"
	  return
	fi

	#Get Prefix and prefix len from interface
	prefix_len=$(ifconfig $ifname | grep inet6 | grep Scope:Global | grep -v "::" | tail -n1 | awk -F '/' '{print $2}' | awk '{print $1}')
	ipv6_address=$(ifconfig $ifname | grep inet6 | grep Scope:Global | grep -v "::" | tail -n1 | awk -F ':' '{print $2":"$3":"$4":"$5":"$6":"$7":"$8":"$9}' |  sed "s/^ *//g" | sed "s/ *$//g" | tr -s ' ' | cut -f 1 -d '/')

	if [ -z $prefix_len ] || [ -z $ipv6_address ]; then
	  log $(basename "$0") "update_prefix_info_to_tmp_file" $LINENO " prefix_len[$prefix_len] or ipv6_address[$ipv6_address] is empty"
	  return
	fi
	log $(basename "$0") "update_prefix_info_to_tmp_file" $LINENO " prefix_len[$prefix_len] ipv6_address[$ipv6_address]"

	quartet_count=$((${prefix_len}/16))
	bit_count=$((${prefix_len}%16))
	for i in `seq 1 $quartet_count`
	do
	    var="{print \$$i}"
		echo "var:$var"
		tmp="echo \"$ipv6_address\" | awk -F ':' '${var}'"
		tmp=$(eval $tmp)
		echo "tmp: $tmp"
		echo "$i $tmp $ipv6_address"
		prefix_info="${prefix_info}${tmp}:"
	done
	#add remaing bits in case of any reminder remians.
	if [ $bit_count -ne 0 ]; then
		var="{print \$$((quartet_count+1))}"
		tmp="echo \"$ipv6_address\" | awk -F ':' '${var}'"
		tmp=$(eval $tmp)
		#convert last quartel to binary
		BIN=$(echo "obase=2; ibase=16; ${tmp^^}" | bc )
		#covert to 16 digit bin format
		BIN=$(printf '%016d' $BIN)
		start=1
		end=4
		last_quartet=""
		for count in `seq 1 4`
		do
		    if [ $end -ge $bit_count ]; then
				tmp_bin=$(echo $BIN | cut -c$start-$bit_count)
				hex=$(echo "obase=16; ibase=2; ${tmp_bin^^}" | bc)
				if [ $hex -ne 0 ]; then
					last_quartet="${last_quatet}${hex}"
				fi
				break
			else
				tmp_bin=$(echo $BIN | cut -c$start-$end)
				hex=$(echo "obase=16; ibase=2; ${tmp_bin^^}" | bc)
				if [ $hex -ne 0 ]; then
					last_quartet="${last_quatet}${hex}"
				fi
				start=$((start+4))
				end=$((end+4))
			fi
		done
		#Append last quartet to prefix.
		if [ ! -z $last_quartet ]; then
			prefix_info="${prefix_info}${last_quartet}:"
		fi
	fi
	log $(basename "$0") "update_prefix_info_to_tmp_file" $LINENO " prefix_info[$prefix_info]"
	#Update prefix and prefix len in tmp file.
	cmd="sed -i 's/.*NETMASK6.*/export NETMASK6=\"${prefix_len}\"/' $filename"
	log $(basename "$0") "update_prefix_info_to_tmp_file" $LINENO " executing command [$cmd]"
	eval $cmd
	cmd="sed -i 's/.*PREFIX6.*/export PREFIX6=\"${prefix_info}:0\"/' $filename"
	eval $cmd
	sync
	log $(basename "$0") "update_prefix_info_to_tmp_file" $LINENO " executing command [$cmd]"
	log $(basename "$0") "update_prefix_info_to_tmp_file" $LINENO " Prefix info updated"
}


# Get Backhaul WAN type: get_wan_type <arg1>
# $1 : QCMAP backhaul WAN type.
function get_wan_type() {
  local wan_type=$1
  QCMAP_WWAN_BACKHAUL=1
  QCMAP_USB_CRADLE_BACKHAUL=2
  QCMAP_WLAN_BACKHAUL=3
  QCMAP_ETHERNET_BACKHAUL=4
  QCMAP_BT_BACKHAUL=5

  if [ $wan_type -eq $QCMAP_WWAN_BACKHAUL ]; then
    echo "wan"
  elif [ $wan_type -eq $QCMAP_USB_CRADLE_BACKHAUL ]; then
    echo "wanusb"
  elif [ $wan_type -eq $QCMAP_ETHERNET_BACKHAUL ]; then
    echo "waneth"
  elif [ $wan_type -eq $QCMAP_WLAN_BACKHAUL ]; then
    echo "wanwlan"
  elif [ $wan_type -eq $QCMAP_BT_BACKHAUL ]; then
    echo "wanbt"
  fi
}

function update_v4_mwan3_default_route() {
  local backhaul_name=$1
  local linkstate=$2
  local defaultroute
  local file=$3
  log $(basename "$0") "update_v4_mwan3_default_route" $LINENO "backhaul_name=$backhaul_name linkstate=$linkstate"

  if [ ! -f "$file" ]; then
    return
  fi

  local default_pdn=$(util_get_default_pdn)
  local default_pdn_index=$(util_get_profile_index $default_pdn)
  local active_ippt=$(uci get qcmap_lan.@profile[$default_pdn_index].active_ippt)
  #can't notify mwan3 if it is IPPT case since mwan3 is stopped
  if [ $active_ippt -eq 1 ] && [ $backhaul_name == "wan" ] ; then
        return
  fi

  . $file

  if [ $linkstate == 0 ]; then

    #delete the ipv4config;
    #if it is rmnet, need check if the IPv4 is still up since IPPT enable case will call rmnet.script down even iface is up
    if [[ $backhaul_name == "wan" ]]; then
      log $(basename "$0") "update_v4_mwan3_default_route" $LINENO " $file will be deleted by lan client"
    else
      log $(basename "$0") "update_v4_mwan3_default_route" $LINENO " delete file $file"
      rm $file
    fi
  fi
}

function update_v6_mwan3_default_route() {
  local backhaul_name=$1
  local linkstate=$2
  local file=$3
  local defaultroute

  log $(basename "$0") "update_v6_mwan3_default_route" $LINENO "backhaul_name=$backhaul_name linkstate=$linkstate file=$file"

  if [ ! -f "$file" ]; then
    log $(basename "$0") "update_v6_mwan3_default_route" $LINENO "file $file is not present"
    return
  fi

  local default_pdn=$(util_get_default_pdn)
  local default_pdn_index=$(util_get_profile_index $default_pdn)
  local active_ippt=$(uci get qcmap_lan.@profile[$default_pdn_index].active_ippt)
  #can't notify mwan3 if it is IPPT case since mwan3 is stopped
  if [ -n "$active_ippt" ]; then
    if [ $active_ippt -eq 1 ] && [ $backhaul_name == "wan" ] ; then
        return
    fi
  fi

  . $file

  if [ $linkstate == 0 ]; then
    if [[ $backhaul_name == "wan" ]]; then
      log $(basename "$0") "update_v6_mwan3_default_route" $LINENO " $file will be deleted by lan client"
    else
      log $(basename "$0") "update_v6_mwan3_default_route" $LINENO " delete file $file"
      rm $file
    fi
  fi

}

#Function to check if any BH config file is present
#$1 - IP type
function check_if_config_file_present() {
  local ip_type="$1"
  local isPresent=0
  local backhaullist defaultProfileId file

  log $(basename "$0") "check_if_config_file_present:" $LINENO "ip_type=$ip_type"

  backhaullist=$(uci get mwan3.backhaul_pref.use_member)
  defaultProfileId=`uci get qcmap_lan.@no_of_configs[0].default_pdn`

  if [ ! -z "$backhaullist" ]; then
    if [ -n "$ip_type" -a "$ip_type" = "ipv4" ]; then
      #Indicates IPV4 config file
      for backhaulmember in $backhaullist
      do
        backhaul=$(uci get mwan3.$backhaulmember.interface)

        get_backhaul_file "ipv4" $backhaul

        if [ -f "$file" ]; then
          log $(basename "$0") ":check_if_config_file_present:" $LINENO ":Config file v4: $file found"
          isPresent=1
          break
        fi
      done
    elif [ -n "$ip_type" -a "$ip_type" = "ipv6" ]; then
      #Indicates IPV6 config file
      for backhaulmember6 in $backhaullist
      do
        backhaul6=$(uci get mwan3.$backhaulmember6.interface)
        get_backhaul_file "ipv6" $backhaul6

        if [ -f "$file" ]; then
          log $(basename "$0") ":check_if_config_file_present:" $LINENO ":Config file v6: $file found"
          isPresent=1
          break
        fi
      done
    else
      log $(basename "$0") ":check_if_config_file_present:" $LINENO ":Incorrect ip_type passed"
    fi
  fi

  echo $isPresent

}

#Function to delete DNSV4 Information of previous BH and add new DNS Information of current Bh
#$1 - Default Profile ID
function update_dnsv4() {
  local defaultProfileId="$1"
  local linkstate="$2"
  local current_bh previous_bh profile_idx
  local DNS isFilePresent

  log $(basename "$0") ":update_dnsv4:" $LINENO "Updating dnsv4 at Bh Switch"

  #Get current_bh and previous_bh information
  profile_idx=$(util_get_profile_index $defaultProfileId)

  current_bh=$(uci get qcmap_lan.@profile[$profile_idx].bh_present)
  previous_bh=$(uci get qcmap_lan.@profile[$profile_idx].bh_previous)

  #Check if linkstate is 0
  if [ $linkstate -eq 0 ]; then
    #Indicates disconnect BH
    #Check if any config file is present
    #NOTE: if any /tmp/ipv4config file is present, indicates its a part of BHSwitch
    isFilePresent=$(check_if_config_file_present "ipv4")
    if [ -n "$isFilePresent" -a $isFilePresent -eq 0 ]; then
      #Indicates this disconnect BH event is the last of the BH's
      #Delete bh_present and bh_previous entries from qcmap_lan db
      [ -n "$current_bh" ]  && util_execute_uci del qcmap_lan.@profile[$profile_idx].bh_present
      [ -n "$previous_bh" ] && util_execute_uci del qcmap_lan.@profile[$profile_idx].bh_previous
      uci commit qcmap_lan
      #Check if current_bh is a secondary BH.
      if [ -n "$current_bh" -a "$current_bh" != "wan" -a $linkstate -eq 0 ]; then
        #Indicates secondary backhaul in disconnect stage
        util_del_non_wwan_dnsv4 $current_bh
        #We do not want to execute the next if condition. Hence returning
        return
      fi
    #If isFilePresent is 1, then it indicates the disconnect BH trigger event
    #is part of BH Switch process. We continue with regular processing.
    fi
  fi

  #Delete DNSSERVER information at BH switch
  if [ -n "$previous_bh" ]; then
    if [ "$previous_bh" = "wan" ]; then
      #Indicates rmnet(wwan) backhaul
      util_del_dnsv4 $defaultProfileId
    else
      #indicates non_wwan backhaul
      util_del_non_wwan_dnsv4 $previous_bh
    fi
  else
    #Check if current_bh is not rmnet BH and linkstate is 0(down)
    #NOTE: reason for having this check is to address a corner case where
    #secondary BH is brought up on a fresh boot and is brought down. Since
    #there will be no bh_previous, we rely on bh_present to delete the dnsserver information
    if [ -n "$current_bh" -a "$current_bh" != "wan" -a $linkstate -eq 0 ]; then
      #Indicates secondary backhaul in disconnect stage
      util_del_non_wwan_dnsv4 $current_bh
      #We do not want to execute the next if condition. Hence returning
      return
    fi
  fi

  #Update DNSSERVER information at BH switch
  if [ -n "$current_bh" ]; then
    if [ "$current_bh" = "wan" ]; then
      #Indicates rmnet(wwan) backhaul
      if [ -f /tmp/ipv4config$defaultProfileId ]; then
        #get DNSSERVER information
        . /tmp/ipv4config$defaultProfileId && DNS=$DNSSERVERS

        #Call util function to add dnsv4 information
        [ -n "$DNS" ] && util_add_dnsv4 $defaultProfileId "$DNS"
      fi
    else
      #indicates non_wwan backhaul
      util_add_non_wwan_dnsv4 $current_bh
    fi
  fi

}

#Function to delete DNSV6 Information of previous BH and add new DNS Information of current Bh
#$1 - Default Profile ID
function update_dnsv6() {
  local defaultProfileId="$1"
  local linkstate="$2"
  local current_bh_v6 previous_bh_v6 profile_idx
  local isFilePresent

  log $(basename "$0") ":update_dnsv6:" $LINENO "Updating dnsv6 at Bh Switch"

  #Get current_bh and previous_bh information
  profile_idx=$(util_get_profile_index $defaultProfileId)

  current_bh_v6=$(uci get qcmap_lan.@profile[$profile_idx].bh_present_v6)
  previous_bh_v6=$(uci get qcmap_lan.@profile[$profile_idx].bh_previous_v6)

  #Check if linkstate is 0
  if [ $linkstate -eq 0 ]; then
    #Indicates disconnect BH
    #Check if any config file is present
    #NOTE: if any /tmp/ipv6config file is present, indicates its a part of BHSwitch
    isFilePresent=$(check_if_config_file_present "ipv6")
    if [ -n "$isFilePresent" -a $isFilePresent -eq 0 ]; then
      #Indicates this disconnect BH event is the last of the BH's
      #Delete bh_present and bh_previous entries from qcmap_lan db
      [ -n "$current_bh_v6" ] &&  util_execute_uci  del qcmap_lan.@profile[$profile_idx].bh_present_v6
      [ -n "$previous_bh_v6" ] && util_execute_uci  del qcmap_lan.@profile[$profile_idx].bh_previous_v6
      uci commit qcmap_lan
      #Check if current_bh is a secondary BH.
      if [ -n "$current_bh_v6" -a "$current_bh_v6" != "wan" -a $linkstate -eq 0 ]; then
        #Indicates secondary backhaul in disconnect stage
        util_del_non_wwan_dnsv6 $current_bh_v6
        #We do not want to execute the next if condition. Hence returning
        return
      fi
    #NOTE:If isFilePresent is 1, then it indicates the disconnect BH trigger event
    #is part of BH Switch process. We continue with regular processing.
    fi
  fi

  #Delete DNSSERVER information at BH switch
  if [ -n "$previous_bh_v6" ]; then
    if [ "$previous_bh_v6" = "wan" ]; then
      #Indicates rmnet(wwan) backhaul
      util_del_dnsv6 $defaultProfileId
    else
      #indicates non_wwan backhaul
      util_del_non_wwan_dnsv6 $previous_bh_v6
    fi
  else
    #Check if current_bh_v6 is not rmnet BH and linkstate is 0(down)
    #NOTE: reason for having this check is to address a corner case where
    #secondary BH is brought up on a fresh boot and is brought down. Since
    #there will be no bh_previous, we rely on bh_present to delete the dnsserver information
    if [ -n "$current_bh_v6" -a "$current_bh_v6" != "wan" -a $linkstate -eq 0 ]; then
      #Indicates secondary backhaul in disconnect stage
      util_del_non_wwan_dnsv6 $current_bh_v6
      #We do not want to execute the next if condition. Hence returning
      return
    fi
  fi

  #Update DNSSERVER information at BH switch
  if [ -n "$current_bh_v6" ]; then
    if [ "$current_bh_v6" = "wan" ]; then
      #Indicates rmnet(wwan) backhaul
      if [ -f /tmp/ipv6config$defaultProfileId ]; then
        #get DNSSERVER6 information
        . /tmp/ipv6config$defaultProfileId && DNS6=$DNSSERVERS6

        #Call util function to add dnsv4 information
        [ -n "$DNS6" ] && util_add_dnsv6 $defaultProfileId "$DNS6"
      fi
    else
      #indicates non_wwan backhaul
      util_add_non_wwan_dnsv6 $current_bh_v6
    fi
  fi

}

function set_brlan_wwan_ipv6prefix_after_switchBH() {
  local profile=$1
  local downstream=$2
  local enable_status
  local prefix6
  local current_bh_v6 ip6addrPresent
  local profile_idx=$(util_get_profile_index $profile)
  log $(basename "$0") ":set_brlan_wwan_ipv6prefix_after_switchBH:" $LINENO "check prefix status"

  if [ ! -f /tmp/ipv6config$profile ]; then
    log $(basename "$0") ":set_brlan_wwan_ipv6prefix_after_switchBH:" $LINENO "file /tmp/ipv6config$profile not exist"
    return
  fi

  current_bh_v6=$(util_execute_uci get qcmap_lan.@profile[$profile_idx].bh_present_v6)
  ip6addrPresent=$(util_execute_uci get network.${downstream}_bind6.ip6addr)
  log $(basename "$0") ":enable_brlan_wwan_ipv6prefix:" $LINENO "current_bh_v6=$current_bh_v6,ip6addrPresent=$ip6addrPresent"
  if [ "$current_bh_v6" == "wan" ]; then
    if [ -z "$ip6addrPresent" ]; then
      #br-lan don't have IPv6 address and current is wwan backhaul
      enable_status=1
    else
      return
    fi
  else
    if [ ! -z "$ip6addrPresent" ]; then
      #br-lan have IPv6 address and current is not wwan backhaul
      enable_status=0
    else
      return
    fi
  fi

  log $(basename "$0") ":set_brlan_wwan_ipv6prefix_after_switchBH:" $LINENO "enable_status=$enable_status"
  if [ $enable_status -eq 0 ] ; then

    util_execute_uci -q delete network.${downstream}_bind6.ip6addr
    util_execute_uci    commit network
    util_run_command ubus call network reload
    util_run_command sleep 0.2
    log  $(basename "$0") ":set_brlan_wwan_ipv6prefix_after_switchBH:" $LINENO "delete br_lan ipv6 address"

  elif [ $enable_status -eq 1 ] ; then

    util_execute_uci set qcmap_lan.@lan[0].br_lan_addr6_add_flag=1
    util_execute_uci commit qcmap_lan

    . /tmp/ipv6config$profile
    prefix6=$PREFIX6
    rtTable=$(util_execute_uci get network.${downstream}_bind6.ip6table)
    util_rmnet_lan_setup_ip6 $downstream $prefix6 $rtTable
    log $(basename "$0") ":set_brlan_wwan_ipv6prefix_after_switchBH:" $LINENO "util_rmnet_lan_setup_ip6 $downstream $prefix6 $rtTable"

    # delete the flag after update br-lan ipv6 address
    util_execute_uci -q delete qcmap_lan.@lan[0].br_lan_addr6_add_flag
    util_execute_uci commit qcmap_lan
  fi
}

function is_firewall_up_on_default_pdn() {

  local defaultProfileId=`uci get qcmap_lan.@no_of_configs[0].default_pdn`
  local profile_idx=$(util_get_default_profile_index)

  local firewall_enable_flag=$(util_execute_uci get qcmap_lan.@profile[$profile_idx].firewall_enabled)

  log $(basename "$0") "is_firewall_up_on_default_pdn" $LINENO "firewall_enable_flag:$firewall_enable_flag, defaultProfileId:$defaultProfileId , profile_idx: $profile_idx "

  echo $firewall_enable_flag
}
#Funtion to cleanup wan_all section from firewall
#config upon Backhaul switch.
#$1 - previous_bh
#$2 - current_bh
function firewall_cleanup() {
  local previous_bh=$1
  local current_bh=$2
  local firewall_change=0
  local wan_all_idx wan_idx lan_wan_idx
  local found wan_listed wan_all_listed lan_wan_listed
  local wan_interface_index wan_interface_name
  local firewall_enable_flag
  local downstream downstream_member

  log $(basename "$0") "firewall_cleanup" $LINENO "previous_bh:$previous_bh, current_bh:$current_bh"

  #Update wan_all with highest prority BH
  #remove previous BH and add current BH
  wan_all_idx=`uci show firewall | grep -i name | grep -i -w "wan_all" | awk -F'[][]' '{print $2}'`
  wan_interface_index=$(util_get_default_pdn)
  log $(basename "$0") "firewall_cleanup" $LINENO "wan_interface_index: $wan_interface_index"
  if [ $wan_interface_index -eq 1 ]; then
    wan_interface_name="wan"
  else
    wan_interface_name="wan$wan_interface_index"
  fi

  wan_idx=`uci show firewall | grep -i name | grep -i -w "$wan_interface_name" | awk -F'[][]' '{print $2}'`

  lan_wan_idx=`util_execute_uci show firewall | grep zone | grep -w "lan_$wan_interface_name" | awk -F'[][]' '{print $2}'`
  lan_wan_listed=$(util_execute_uci get firewall.@zone[$lan_wan_idx].network)

  if [ -n "$previous_bh" ]; then
    #clean lan_wan zone first
    for temp_network in $lan_wan_listed
    do
      if  [ "$temp_network" != "lan" ]; then
        util_execute_uci del_list firewall.@zone[$lan_wan_idx].network="$temp_network"
      fi
    done

    util_execute_uci del_list firewall.@zone[$wan_all_idx].network="$previous_bh"
    util_execute_uci del_list firewall.@zone[$wan_all_idx].network="${previous_bh}_v6"
    util_execute_uci del_list firewall.@zone[$wan_idx].network="$previous_bh"
    util_execute_uci del_list firewall.@zone[$wan_idx].network="${previous_bh}_v6"
    util_execute_uci commit firewall
    firewall_change=1
  fi

  if [ -n "$current_bh" ]; then
    #add only those lan ifaces which are mapped to highest priority active BH to lan_wan zone
    downstream=$(util_execute_uci get network.$current_bh.downstream)
    for downstream_member in $downstream
    do
      if  [ "$downstream_member" != "lan" ]; then
        util_execute_uci add_list firewall.@zone[$lan_wan_idx].network="$downstream_member"
      fi
    done

    wan_all_listed=`util_execute_uci get firewall.@zone[$wan_all_idx].network`
    wan_listed=`util_execute_uci get firewall.@zone[$wan_idx].network`

    firewall_enable_flag=$(is_firewall_up_on_default_pdn)
    found=$(echo $wan_all_listed | grep -w -c $current_bh)
    if [ $firewall_enable_flag -eq 0 ]; then
      #if firewall is disabled then only add current_bh in wan_all
      log $(basename "$0") "firewall_cleanup" $LINENO "firewall_enable_flag:$firewall_enable_flag"
      if [ $found -eq 0 ]; then
        log $(basename "$0") "firewall_cleanup" $LINENO "current_bh:$current_bh not listed to wan_all"
        util_execute_uci add_list firewall.@zone[$wan_all_idx].network="$current_bh"
        util_execute_uci commit firewall
        firewall_change=1
      fi
      found=$(echo $wan_all_listed | grep -w -c ${current_bh}_v6)
      if [ $found -eq 0 ]; then
        log $(basename "$0") "firewall_cleanup" $LINENO "current_bh:${current_bh}_v6 not listed to wan_all"
        util_execute_uci add_list firewall.@zone[$wan_all_idx].network="${current_bh}_v6"
        util_execute_uci commit firewall
        firewall_change=1
      fi
    fi

    found=$(echo $wan_listed | grep -w -c $current_bh)
    if [ $found -eq 0 ]; then
      log $(basename "$0") "firewall_cleanup" $LINENO "current_bh:$current_bh not listed to wan"
      util_execute_uci add_list firewall.@zone[$wan_idx].network="$current_bh"
      util_execute_uci commit firewall
      firewall_change=1
    fi
    found=$(echo $wan_listed | grep -w -c ${current_bh}_v6)
    if [ $found -eq 0 ]; then
      log $(basename "$0") "firewall_cleanup" $LINENO "current_bh:${current_bh}_v6 not listed to wan"
      util_execute_uci add_list firewall.@zone[$wan_idx].network="${current_bh}_v6"
      util_execute_uci commit firewall
      firewall_change=1
    fi
  fi

  if [ $firewall_change -eq 1 ]; then
    /etc/init.d/firewall reload
  fi

}

#compare two backhaul priority, echo the one with higher priority
function compare_backhaul_priority()
{
  local bh1=$1
  local bh2=$2
  local backhaullist
  local backhaulmember

  log $(basename "$0") "compare_backhaul_priority" $LINENO "bh1=$bh1 bh2=$bh2"
  backhaullist=$(util_execute_uci get mwan3.backhaul_pref.use_member)
  for backhaulmember in $backhaullist
  do
    backhaulname=${backhaulmember#*m_}
    if [ "$backhaulname" == "$bh1" ]; then
      echo $bh1
      return
    elif [ "$backhaulname" == "$bh2" ]; then
      echo $bh2
      return
    fi
  done

}

function reload_for_sta_mode()
{
   local bridge_mode
   local wlan_proto
   local ipv4_current_bh
   local ipv4_higher_bh

   bridge_mode=$(util_execute_uci get qcmap_wlan.@stamodeconfig[0].bridge_mode)

   sta_associated_status=$(/etc/data/wlanConfig.sh get_sta_associcated_status)
   if [ "$sta_associated_status" == "0" ]; then
     return
   fi

   log $(basename "$0") "reload_for_sta_mode" $LINENO "STA is associcated, bridge mode = $bridge_mode "

   #use wanwlan.proto to check whether bridge mode is enabled in br-lan
   wlan_proto=$(util_execute_uci get network.wanwlan.proto)

   ipv4_current_bh=$(util_get_bh_present)
   #get the expected high priority backhaul by compared with current and wanwlan
   ipv4_higher_bh=$(compare_backhaul_priority "$ipv4_current_bh" "wanwlan")

   if [ "$wlan_proto" == "static" -a "$ipv4_higher_bh" != "wanwlan" ];then
    #in case wlan_proto is static but current backhual is not wanwlan, then we need disconnect ap-sta mode
    log $(basename "$0") "reload_for_sta_mode" $LINENO "wlan_proto is static but not higest priority backhaul, disconnect sta"
    /etc/data/wlanConfig.sh ap_sta_mode_disconnect 0
   elif [ "$wlan_proto" == "static" -a "$ipv4_higher_bh" == "wanwlan" ];then
    #in case wlan_proto is static and current backhual is wanwlan, then check ipv4 route
    log $(basename "$0") "reload_for_sta_mode" $LINENO "bh is wanwlan with static config, check ipv4 route and add it"
    local ip_r_default=$(ip r s | grep "default via")
    log $(basename "$0") "reload_for_sta_mode" $LINENO "ip r s | grep default via: $ip_r_default"
    #add related ip route and update firewall
    if [ -z $ip_r_default ]; then
      log $(basename "$0") "reload_for_sta_mode" $LINENO "no default ip route"
      /etc/data/wlanConfig.sh ap_sta_mode_connect 0
    fi
   elif [ "$wlan_proto" == "dhcp" -a "$ipv4_higher_bh" == "wanwlan" ];then
    #in case wlan_proto is dhcp and wanwlan should be high priority backhual, then connect ap-sta mode
    log $(basename "$0") "reload_for_sta_mode" $LINENO "call  ap_sta_mode_connect since wanwlan is higest priority"
    /etc/data/wlanConfig.sh ap_sta_mode_connect 0
   fi
}

function update_firewall_zone_for_load_balance()
{
  local file_wan file_waneth
  local defaultProfileId
  local wan_all_idx wan_idx
  local wan_all_listed wan_listed
  local found

  log $(basename "$0") "update_firewall_zone_for_load_balance" $LINENO "Enter update_firewall_zone_for_load_balance"

  get_bh_load_balance_status
  if [ "$qcmap_bh_load_balance_enabled" == "$QCMAP_BH_LOAD_BALANCE_ENABLE" ]; then
    defaultProfileId=`util_execute_uci get qcmap_lan.@no_of_configs[0].default_pdn`

    file_wan=/tmp/ipv4config$defaultProfileId
    file_waneth=/tmp/ipv4config.waneth

    #both wan and waneth need up
    if [ ! -f "$file_wan" ] || [ ! -f "$file_waneth" ]; then
      log $(basename "$0") "update_firewall_zone_for_load_balance" $LINENO "wan and waneth backhaul not all up"
      return
    fi

    wan_all_idx=`util_execute_uci show firewall | grep -i name | grep -i -w "wan_all" | awk -F'[][]' '{print $2}'`
    wan_idx=`util_execute_uci show firewall | grep -i name | grep -i -w "wan" | awk -F'[][]' '{print $2}'`
    wan_all_listed=`util_execute_uci get firewall.@zone[$wan_all_idx].network`
    wan_listed=`util_execute_uci get firewall.@zone[$wan_idx].network`

    #wan_all zone
    found=$(echo $wan_all_listed | grep -w -c "wan")
    if [ $found -eq 0 ]; then
      log $(basename "$0") "update_firewall_zone_for_load_balance" $LINENO "wan not listed to zone wan_all"
      util_execute_uci add_list firewall.@zone[$wan_all_idx].network="wan"
      util_execute_uci commit firewall
    fi
    found=$(echo $wan_all_listed | grep -w -c "waneth")
    if [ $found -eq 0 ]; then
      log $(basename "$0") "update_firewall_zone_for_load_balance" $LINENO "waneth not listed to zone wan_all"
      util_execute_uci add_list firewall.@zone[$wan_all_idx].network="waneth"
      util_execute_uci commit firewall
    fi

    #wan zone
    found=$(echo $wan_listed | grep -w -c "wan")
    if [ $found -eq 0 ]; then
      log $(basename "$0") "update_firewall_zone_for_load_balance" $LINENO "wan not listed to zone wan"
      util_execute_uci add_list firewall.@zone[$wan_idx].network="wan"
      util_execute_uci commit firewall
    fi
    found=$(echo $wan_listed | grep -w -c "waneth")
    if [ $found -eq 0 ]; then
      log $(basename "$0") "update_firewall_zone_for_load_balance" $LINENO "waneth not listed to zone wan"
      util_execute_uci add_list firewall.@zone[$wan_idx].network="waneth"
      util_execute_uci commit firewall
    fi

    /etc/init.d/firewall reload

    log $(basename "$0") "update_firewall_zone_for_load_balance" $LINENO "success"
  fi
}

function reload_for_tinyproxy()
{
   local tinyproxy_enable_state=$(util_execute_uci get qcmap_lan.@no_of_configs[0].enable_tinyproxy)

   if [ $tinyproxy_enable_state -eq 1 ] ; then
     ipv4_current_bh=$(util_get_bh_present)
     #get the expected high priority backhaul by compared with current and wanwlan
     ipv4_higher_bh=$(compare_backhaul_priority "$ipv4_current_bh" "wan")
     log $(basename "$0") "reload_for_tinyproxy" $LINENO "current bh: $ipv4_current_bh higher BH: $ipv4_higher_bh"

     if [ "$ipv4_higher_bh" == "wan" ] ; then
       stop_tinyproxy
       setup_tinyproxy
       log $(basename "$0") "reload_for_tinyproxy" $LINENO "retart tinyproxy for wan bh"
     elif [ "$ipv4_higher_bh" != "wan" ] ; then
       stop_tinyproxy
       log $(basename "$0") "reload_for_tinyproxy" $LINENO "stop tinyproxy for non-wan bh"
     fi
   fi
}

# check backhaul is default or not
# backhaul is default: return 0
# backhaul is not default: return 1
function is_default_backhaul() {
  local backhaul_name=$1
  local wwan_profile
  local wwan_default_profile=$(util_execute_uci get qcmap_lan.@no_of_configs[0].default_pdn)

  # for wanwlan, wanbt, wanusb
  if [ "$backhaul_name" == "wanwlan" -o "$backhaul_name" == "wanbt" -o "$backhaul_name" == "wanusb" ]; then
    log $(basename "$0") "is_default_backhaul" $LINENO "$backhaul_name backhaul is default!"
    return 0
  fi

  #for waneth and wan
  if [ "$backhaul_name" == "waneth" -o "$backhaul_name" == "wan" ]; then
    log $(basename "$0") "is_default_backhaul" $LINENO "$backhaul_name backhaul is default!"
    return 0
  elif [[ "$backhaul_name" =~ ^rmnet.* ]]; then
    wwan_profile=${backhaul_name#rmnet}
    if [ "$wwan_profile" == "$wwan_default_profile" ]; then
      log $(basename "$0") "is_default_backhaul" $LINENO "$backhaul_name backhaul is default!"
      return 0
    fi
  elif [[ "$backhaul_name" =~ ^wan.* ]]; then
    wwan_profile=${backhaul_name#wan}
    if [ "$wwan_profile" == "$wwan_default_profile" ]; then
      log $(basename "$0") "is_default_backhaul" $LINENO "$backhaul_name backhaul is default!"
      return 0
    fi
  fi

  log $(basename "$0") "is_default_backhaul" $LINENO "$backhaul_name backhaul is not default!"
  return 1
}

#reconfigure pppoe dhcpv6 relay setting
function update_vlan_dhcpv6_relay_mode()
{
  local vlan_idx=$1
  local new_backhaul=$2
  local vlan_id=$(util_execute_uci get qcmap_lan.@vlan[$vlan_idx].vlan_id)
  local wwan_profile_num=$(util_execute_uci get qcmap_lan.@vlan[$vlan_idx].wwan_profile_num)
  local waneth_profile_num=$(util_execute_uci get qcmap_lan.@vlan[$vlan_idx].waneth_profile_num)
  local backhaul_name=""
  local ismaster=""
  local need_set_dhcp_server_mode=0
  local ipv6_bh=$(util_execute_uci get qcmap_lan.@vlan[$vlan_idx].bh_present_v6)
  local bridge_mode=$(util_execute_uci get qcmap_wlan.@stamodeconfig[0].bridge_mode)
  local brwan_enabled=$(util_execute_uci get qcmap_lan.@global[0].brwan)
  local dhcp_lan_ndp

  log $(basename "$0") "update_vlan_dhcpv6_relay_mode" $LINENO "Enter update_vlan_dhcpv6_relay_mode, new_backhaul:$2"

  if [ -z "$bridge_mode" ]; then
    bridge_mode=0
  fi

  if [ -z "$ipv6_bh" ]; then
    dhcp_lan_ndp=$(util_execute_uci get dhcp.lan$vlan_id.ndp)
    if [ -z "$dhcp_lan_ndp" ]; then
      log $(basename "$0") "update_vlan_dhcpv6_relay_mode" $LINENO "dhcpv6 relay was not set and no need set(ipv6 backhaul not up), break"
      return
    fi
    #IPv6 not ready on current backhaul but was in other backhaul, handle it as no IPv6 backhaul
    new_backhaul="no"
  fi

  backhaullist=$(util_execute_uci get mwan3.backhaul_pref.use_member)
  for backhaulmember in $backhaullist
  do
    if [ "$backhaulmember" == "m_wan" -a -n "$wwan_profile_num" ]; then
      backhaul_name="wan$wwan_profile_num"
    elif [ "$backhaulmember" == "m_waneth" -a -n "$waneth_profile_num" ]; then
      backhaul_name="waneth$waneth_profile_num"
    else
      log $(basename "$0") "update_vlan_dhcpv6_relay_mode" $LINENO ":not support $backhaulmember switch or bh not exist"
      continue
    fi

    ismaster=$(util_execute_uci -q get dhcp.${backhaul_name}_v6.master)
    master_dhcp=$(util_execute_uci get dhcp.lan$vlan_id.master_dhcp)
    if [ -z "$ismaster" -o -z "$master_dhcp" ]; then
      if [ "$backhaul_name" == "$new_backhaul" ]; then
        if [ "$backhaul_name" == "wan$wwan_profile_num" ]; then
          log $(basename "$0") "update_vlan_dhcpv6_relay_mode" $LINENO "dhcpv6 relay no need set since wwan is default backhaul"
          #set dhcp.lan as server mode later,
          #this is to allow odhcpd know which is slave interface to send deprecated RA.
          need_set_dhcp_server_mode=1
        else
          if [ $bridge_mode -eq 0 ]; then
            log $(basename "$0") "update_vlan_dhcpv6_relay_mode" $LINENO "dhcpv6 master not on default backhaul, $backhaul_name: set it"

            if [ -n "$wwan_profile_num" ]; then
              #before set dhcp to relay mode, remove IPv6 on br-lan which is from wwan
              set_brlan_wwan_ipv6prefix_after_switchBH $wwan_profile_num "lan$vlan_id"

              #before switch to relay mode, let odhcpd send deprecated RA for server mode
              util_execute_uci_set dhcp.lan$vlan_id.ra     "disabled"
              util_execute_uci commit dhcp
              util_run_command /etc/init.d/odhcpd reload
              util_run_command sleep 0.2
            fi

            util_execute_uci_set dhcp.${backhaul_name}_v6.dhcpv6 "relay"
            util_execute_uci_set dhcp.${backhaul_name}_v6.ra     "relay"
            util_execute_uci_set dhcp.${backhaul_name}_v6.ndp    "relay"
            util_execute_uci_set dhcp.${backhaul_name}_v6.master "1"
            util_execute_uci_set dhcp.${backhaul_name}_v6.ndproxy_routing "0"
            util_execute_uci_set dhcp.lan$vlan_id.dhcpv6 "relay"
            util_execute_uci_set dhcp.lan$vlan_id.ra     "relay"
            util_execute_uci_set dhcp.lan$vlan_id.ndp    "relay"
            util_execute_uci_set dhcp.lan$vlan_id.master_dhcp "${backhaul_name}_v6"
          elif [ $bridge_mode -eq 1 ]; then
            log $(basename "$0") "update_vlan_dhcpv6_relay_mode" $LINENO "STA bridge mode: handle in dhcpRelay.sh"
          fi
        fi
      fi
    else
      if [ "$backhaul_name" != "$new_backhaul" ]; then
        log $(basename "$0") "update_vlan_dhcpv6_relay_mode" $LINENO "dhcpv6 master was on $backhaul_name but default backhaul on $new_backhaul: delete it"
        util_execute_uci -q delete dhcp.${backhaul_name}_v6.dhcpv6
        util_execute_uci -q delete dhcp.${backhaul_name}_v6.ra
        util_execute_uci -q delete dhcp.${backhaul_name}_v6.ndp
        util_execute_uci -q delete dhcp.${backhaul_name}_v6.master
        util_execute_uci -q delete dhcp.${backhaul_name}_v6.ndproxy_routing
        need_set_dhcp_server_mode=1

        #remove the master firstly and trigger reload, then odhcpd able to send deprecated RA for deprecated wan interface.
        util_execute_uci commit dhcp
        /etc/init.d/odhcpd reload
        #wait odhcpd finish send deprecated RA.
        sleep 0.5
      else
        log $(basename "$0") "update_vlan_dhcpv6_relay_mode" $LINENO "dhcpv6 master already on $backhaul_name: no action"
      fi
    fi
  done

  if [ $need_set_dhcp_server_mode -eq 1 ]; then
     util_execute_uci_set dhcp.lan$vlan_id.dhcpv6 "server"
     util_execute_uci_set dhcp.lan$vlan_id.ra     "server"
     util_execute_uci -q delete dhcp.lan$vlan_id.master_dhcp
  fi

  util_execute_uci commit dhcp
  /etc/init.d/odhcpd reload

  if [ -n "$wwan_profile_num" ]; then
    #after set dhcp mode, ensure br-lan$vlan_id have IPv6 addr back
    set_brlan_wwan_ipv6prefix_after_switchBH $wwan_profile_num "lan$vlan_id"
  fi
}

#we have seperated backhaul with IPv4 IPv6,
#then it is possbile highest backhaul have only IP type active, such as IPv4, but other backhaul may IPv6, or vise versa
#then IPv4 go one backhaul, IPv6 go another,
function combine_vlan_backhaul_ipv4ipv6()
{
  local vlan_idx=$1
  local wwan_profile_num=$(util_execute_uci get qcmap_lan.@vlan[$vlan_idx].wwan_profile_num)
  local waneth_profile_num=$(util_execute_uci get qcmap_lan.@vlan[$vlan_idx].waneth_profile_num)
  local checkIP
  local bh_previous
  local backhaul_file=""
  local backhaul6_file=""
  local backhaul_name=""
  local new_backhaul=""
  local ipv4_bh=$(util_execute_uci get qcmap_lan.@vlan[$vlan_idx].bh_present)
  local ipv6_bh=$(util_execute_uci get qcmap_lan.@vlan[$vlan_idx].bh_present_v6)

  log $(basename "$0") "combine_vlan_backhaul_ipv4ipv6" $LINENO "Enter combine_vlan_backhaul_ipv4ipv6 vlan_idx=$vlan_idx"

  #if no value, set it to invalid value to easy handle.
  if [ -z "$ipv4_bh" ]; then
    ipv4_bh="no"
  fi
  if [ -z "$ipv6_bh" ]; then
    ipv6_bh="no"
  fi

  backhaullist=$(util_execute_uci get mwan3.backhaul_pref.use_member)
  for backhaulmember in $backhaullist
  do
    if [ "$backhaulmember" == "m_wan" -a -n "$wwan_profile_num" ]; then
      backhaul_name="wan$wwan_profile_num"
      backhaul6_file="/tmp/ipv6config$wwan_profile_num"
    elif [ "$backhaulmember" == "m_waneth" -a -n "$waneth_profile_num" ]; then
      backhaul_name="waneth$waneth_profile_num"
      backhaul6_file="/tmp/ipv6config.waneth$waneth_profile_num"
    else
      log $(basename "$0") "combine_vlan_backhaul_ipv4ipv6" $LINENO ":not support $backhaulmember switch or bh not exist"
      continue
    fi

    #temp workaround for ipv6: switch_bh_v6 is not called from netifd/odhcp6c when ipv6 interface down;
    #delete the file and don't consider the backhaul as current backhaul
    if [[ "$backhaul_name" =~ ^waneth[0-9] ]] && [ -f $backhaul6_file ]; then
      . $backhaul6_file
      checkIP=$(ip -6 addr show $IFNAME | grep $PUBLIC_IP6)
      if [ -z "$checkIP" ]; then
        log $(basename "$0") "combine_vlan_backhaul_ipv4ipv6" $LINENO "${backhaul_name}_v6 not up since IPv6 addr not preset"
        rm $backhaul6_file
        if [ "$backhaul_name" == "$ipv6_bh" ]; then
           ipv6_bh="no"
        fi
      fi
    fi
  done

  for backhaulmember in $backhaullist
  do
    if [ "$backhaulmember" == "m_wan" -a -n "$wwan_profile_num" ]; then
      backhaul_name="wan$wwan_profile_num"
      backhaul_file="/tmp/ipv4config$wwan_profile_num"
      backhaul6_file="/tmp/ipv6config$wwan_profile_num"
    elif [ "$backhaulmember" == "m_waneth" -a -n "$waneth_profile_num" ]; then
      backhaul_name="waneth$waneth_profile_num"
      backhaul_file="/tmp/ipv4config.waneth$waneth_profile_num"
      backhaul6_file="/tmp/ipv6config.waneth$waneth_profile_num"
    else
      log $(basename "$0") "combine_vlan_backhaul_ipv4ipv6" $LINENO ":not support $backhaulmember switch or bh not exist"
      continue
    fi

    log $(basename "$0") "combine_vlan_backhaul_ipv4ipv6" $LINENO "ipv4_bh=$ipv4_bh, ipv6_bh=$ipv6_bh, loop on $backhaul_name"

    if [ "$backhaul_name" == "$ipv4_bh" -o "$backhaul_name" == "$ipv6_bh" ]; then
      #either Ipv4 or IPv6 is up;
      log $(basename "$0") "combine_vlan_backhaul_ipv4ipv6" $LINENO "$backhaul_name is highest priority"
      new_backhaul=$backhaul_name

      if [ "$backhaul_name" != "$ipv6_bh" ]; then
        if [ -f $backhaul6_file ]; then
          #IPv6 present
         util_execute_uci_set qcmap_lan.@vlan[$vlan_idx].bh_present_v6 "$backhaul_name"

          bh_previous=$(util_execute_uci get qcmap_lan.@vlan[$vlan_idx].bh_previous)
          util_execute_uci_set qcmap_lan.@vlan[$vlan_idx].bh_previous_v6 "$bh_previous"
        else
          #IPv6 is not on this highest prioity backhaul, disable default route for IPv6 on the low priority backhaul
          util_execute_uci_set qcmap_lan.@vlan[$vlan_idx].bh_present_v6 ""
        fi
      fi

      if [ "$backhaul_name" != "$ipv4_bh" ]; then
        if [ -f $backhaul_file ]; then
          #IPv4 present
          util_execute_uci_set qcmap_lan.@vlan[$vlan_idx].bh_present "$backhaul_name"
        else
          #IPv4 is not on this highest prioity backhaul, disable  default route for IPv4 on the low priority backhaul
          util_execute_uci_set qcmap_lan.@vlan[$vlan_idx].bh_present ""
        fi
      fi
      break
    fi
  done

  util_execute_uci commit qcmap_lan
  util_execute_uci commit network
  ubus call network reload

  update_vlan_dhcpv6_relay_mode $vlan_idx $new_backhaul
}

#IPv4 backhaul
#update preset backhaul on vlan
function update_vlan_present_backhaul() {
  local vlan_idx=$1
  local input_backhaul=$2
  local linkstate=$3
  local wwan_profile_num=$(util_execute_uci get qcmap_lan.@vlan[$vlan_idx].wwan_profile_num)
  local waneth_profile_num=$(util_execute_uci get qcmap_lan.@vlan[$vlan_idx].waneth_profile_num)
  local waneth_file
  local backhaul_name=""
  local backhaul_file=""
  local checkIP
  local current_backhaul
  local new_backhaul=""
  local new_backhaul_file=""

  log $(basename "$0") "update_vlan_present_backhaul" $LINENO "Enter update_vlan_present_backhaul vlan_idx=$vlan_idx input_backhaul=$input_backhaul linkstate=$linkstate"


  #run loop to get bh_present
  backhaullist=$(util_execute_uci get mwan3.backhaul_pref.use_member)
  if [ -n "$backhaullist" ]; then
    #IPv4 backhaul
    current_backhaul=$(util_execute_uci get qcmap_lan.@vlan[$vlan_idx].bh_present)
    log $(basename "$0") "update_vlan_present_backhaul" $LINENO "current_IPv4_backhaul:$current_backhaul"

    #initialize the bh_present be empty.
    util_execute_uci_set qcmap_lan.@vlan[$vlan_idx].bh_present ""

    for backhaulmember in $backhaullist
    do
      if [ "$backhaulmember" == "m_waneth" ] && [ "$linkstate" == "0" ]; then
        log $(basename "$0") "update_vlan_present_backhaul" $LINENO " ignore $backhaul since already handled"
        continue
      fi

      if [ "$backhaulmember" == "m_wan" -a -n "$wwan_profile_num" ]; then
        backhaul_name="wan$wwan_profile_num"
        backhaul_file="/tmp/ipv4config$wwan_profile_num"
      elif [ "$backhaulmember" == "m_waneth" -a -n "$waneth_profile_num" ]; then
        backhaul_name="waneth$waneth_profile_num"
        backhaul_file="/tmp/ipv4config.waneth$waneth_profile_num"
      else
        log $(basename "$0") "update_vlan_present_backhaul" $LINENO ":not support $backhaulmember switch or bh not exist"
        continue
      fi

      if [ ! -f "$backhaul_file" ]; then
        log $(basename "$0") "update_vlan_present_backhaul" $LINENO ":Backhaul not up since $backhaul_file not present"
        continue
      fi
      log $(basename "$0") "update_vlan_present_backhaul" $LINENO ":file present: $backhaul_file"

      . $backhaul_file

      #not use PUBLIC_IP check since when IPPT case rmnet don't have publicIP
      checkIP=`ip addr show $IFNAME | grep inet`
      if [ -z "$checkIP" ]; then
        log $(basename "$0") "update_vlan_present_backhaul" $LINENO ":Backhaul not up since addr $PUBLIC_IP not valid"
        continue
      fi
      log $(basename "$0") "update_vlan_present_backhaul" $LINENO ":$backhaul_name is up with devname=$IFNAME, addr=$PUBLIC_IP"

      if [ -z "$new_backhaul" ] ; then
        new_backhaul=$backhaul_name
        new_backhaul_file=$backhaul_file

        if [ -n "$current_backhaul" -a "$current_backhaul" != "$backhaul_name" ]; then
          util_execute_uci_set qcmap_lan.@vlan[$vlan_idx].bh_previous "$current_backhaul"
        fi
        util_execute_uci_set qcmap_lan.@vlan[$vlan_idx].bh_present "$backhaul_name"
        util_execute_uci commit qcmap_lan
      fi
    done

    combine_vlan_backhaul_ipv4ipv6 $vlan_idx
  else
    log $(basename "$0")":update_vlan_present_backhaul:"$LINENO":get backhaul_pref.use_member failed:$backhaullist"
  fi
}

#Add/del waneth network iface to wan_all zone when Backhaul is Up/down in that backhaul.
function add_del_on_demand_waneth_iface_on_wan_all_zone(){
  local backhaul_name=$1
  local link_state=$2
  local waneth_iface waneth6_iface wan_all_listed
  local wan_all_idx=`util_execute_uci show firewall | grep -i name | grep -i "wan_all" | awk -F'[][]' '{print $2}'`

  log $(basename "$0") "add_del_on_demand_waneth_iface_on_wan_all_zone:" $LINENO "backhaul_name:$backhaul_name link_state:$link_state"
  waneth_iface=$backhaul_name
  waneth6_iface=${backhaul_name}_v6
  wan_all_listed=`util_execute_uci get firewall.@zone[$wan_all_idx].network`

  #check if waneth V4 interface already there in the list
  found=$(echo $wan_all_listed | grep -w -c $waneth_iface)
  if [ "$link_state" == "1" -a $found -eq 0 ]; then
    util_execute_uci add_list firewall.@zone[$wan_all_idx].network=$waneth_iface
  elif [ "$link_state" == "0" -a $found -eq 1 ]; then
    util_execute_uci del_list firewall.@zone[$wan_all_idx].network=$waneth_iface
  fi

  #check if waneth V6 interface already there in the list
  found=$(echo $wan_all_listed | grep -w -c $waneth6_iface)
  if [ "$link_state" == "1" -a $found -eq 0 ]; then
    util_execute_uci add_list firewall.@zone[$wan_all_idx].network=$waneth6_iface
  elif [ "$link_state" == "0" -a $found -eq 1 ]; then
    util_execute_uci del_list firewall.@zone[$wan_all_idx].network=$waneth6_iface
  fi

  util_execute_uci commit firewall
  /etc/init.d/firewall reload
  log $(basename "$0") "add_del_on_demand_waneth_iface_on_wan_all_zone:" $LINENO ":$backhaul_name iface update on wan_all zone, state:$link_state"
}

function switch_on_demand_backhaul() {
  local backhaul_name=$1
  local link_state=$2
  local vlan_id vlan_idx
  local current_bh new_bh
  local input_profile
  local temp_profile_num profile_num
  local ethwan_profile_id
  local ethwan_profile_vlan_id
  local ethwan_profile_iface
  local wwan_profile
  local no_of_vlans=$(util_execute_uci get qcmap_lan.@no_of_configs[0].no_of_vlans)
  local profile_id downdevice dedicated_rt
  local eth_type ethwan_profile_proto
  local ipv6_waneth_config_file
  local brwan_enabled=$(util_execute_uci get qcmap_lan.@global[0].brwan)

  log $(basename "$0") ":switch_on_demand_backhaul:" $LINENO "Enter switch_on_demand_backhaul backhaul_name=$backhaul_name link_state=$link_state"

  #add/del backhaul to firewall wan_all zone
  if [[ "$backhaul_name" =~ ^waneth.* ]]; then
    add_del_on_demand_waneth_iface_on_wan_all_zone $backhaul_name $link_state
  fi

  if [[ "$backhaul_name" =~ ^waneth.* ]]; then
    input_profile=${backhaul_name#waneth}
    profile_num="waneth_profile_num"

    ethwan_profile_id=$input_profile
    ethwan_profile_vlan_id=$(util_execute_uci get qcmap_lan.profile_$ethwan_profile_id.vlan_id)
    ethwan_profile_iface=$(util_execute_uci get qcmap_lan.profile_$ethwan_profile_id.device)
    ethwan_profile_proto=$(util_execute_uci get qcmap_lan.profile_$ethwan_profile_id.proto)

    if [ "$ethwan_profile_iface" == "eth0" ]; then
       eth_type="eth"
    elif [ "$ethwan_profile_iface" == "eth1" ]; then
       eth_type="eth_nic2"
    fi

    #update ipa
    if [ "$link_state" == "0" ]; then
      #Delete v4 conntrack entries on teardown
      util_del_conntrack $ethwan_profile_id "ipv4" 0
      if [ "$ethwan_profile_proto" == "pppoe" ]; then
        ipa pppoe del_mapping $ethwan_profile_iface $ethwan_profile_vlan_id pppoe-$backhaul_name > /dev/null
        log $(basename "$0") ":switch_on_demand_backhaul:" $LINENO "ipa pppoe del_mapping $ethwan_profile_iface $ethwan_profile_vlan_id pppoe-$backhaul_name"
      fi

      # if dhcp backhaul delete dhcp ipv4 address, add the static ip
      if [ "$ethwan_profile_proto" == "dhcp" -a $ethwan_profile_vlan_id -eq 0 ]; then
        util_execute_uci set network.$eth_type.disabled="0"

        ipv4_waneth_config_file=/tmp/ipv4config.$backhaul_name
        if [ -f "$ipv4_waneth_config_file" ]; then
          log $(basename "$0") "switch_on_demand_backhaul" $LINENO "delete dhcp v4 file $ipv4_waneth_config_file"
          rm $ipv4_waneth_config_file
        fi
      fi
    elif [ "$link_state" == "1" ]; then
      if [ "$ethwan_profile_proto" == "pppoe" ]; then
        ipa pppoe add_mapping $ethwan_profile_iface $ethwan_profile_vlan_id pppoe-$backhaul_name > /dev/null
        log $(basename "$0") ":switch_on_demand_backhaul:" $LINENO "ipa pppoe add_mapping $ethwan_profile_iface $ethwan_profile_vlan_id pppoe-$backhaul_name"
      fi

      # if dhcp backhaul get dhcp ipv4 address, delete the static ip
      # in Deco mode, keep the static ip
      if [ "$brwan_enabled" != "1" -a "$ethwan_profile_proto" == "dhcp" -a $ethwan_profile_vlan_id -eq 0 ]; then
        util_execute_uci set network.$eth_type.disabled="1"
      fi
    fi

    if [ "$ethwan_profile_proto" == "dhcp" -a $ethwan_profile_vlan_id -eq 0 ]; then
      util_execute_uci commit network
      ubus call network reload
    fi
  elif [[ "$backhaul_name" =~ ^rmnet.* ]]; then
    input_profile=${backhaul_name#rmnet}
    profile_num="wwan_profile_num"
    backhaul_name="wan$input_profile"
  elif [[ "$backhaul_name" =~ ^wan.* ]]; then
    input_profile=${backhaul_name#wan}
    profile_num="wwan_profile_num"
  fi

  #delete waneth backhaul file when link down
  if [[ "$backhaul_name" =~ ^waneth.* ]] && [ "$link_state" == "0" ]; then
    waneth_file=/tmp/ipv4config.$backhaul_name
    if [ -f "$waneth_file" ]; then
      log $(basename "$0") "update_vlan_present_backhaul" $LINENO "Ayush delete file $waneth_file"
      rm $waneth_file
    fi
  fi

  #check if vlan is exist
  if [ $no_of_vlans -eq 0 ]; then
    log $(basename "$0") "switch_on_demand_backhaul" $LINENO "No VLAN is configured"
    return
  fi

  #loop through the number of vlans
  for vlan_idx in $(seq 0 $((no_of_vlans-1)))
  do
    temp_profile_num=$(util_execute_uci get qcmap_lan.@vlan[$vlan_idx].$profile_num)
    if [ -n "$temp_profile_num" -a $temp_profile_num -eq $input_profile ]; then
      vlan_id=$(util_execute_uci get qcmap_lan.@vlan[$vlan_idx].vlan_id)
      log $(basename "$0") ":switch_on_demand_backhaul:" $LINENO "vlan_id:$vlan_id"

      #Get current BH enabled before updating priority BH
      current_bh=$(util_execute_uci get qcmap_lan.@vlan[$vlan_idx].bh_present)

      update_vlan_present_backhaul $vlan_idx $backhaul_name $link_state

      #Get present BH after updating priority BH
      new_bh=$(util_execute_uci get qcmap_lan.@vlan[$vlan_idx].bh_present)
      log $(basename "$0") ":switch_on_demand_backhaul:" $LINENO "current_bh: $current_bh; new_bh: $new_bh"

      wwan_profile=$(util_execute_uci get qcmap_lan.@vlan[$vlan_idx].wwan_profile_num)
      if [ -n "$wwan_profile" ] && [ -n "$current_bh" -a "$current_bh" != "$new_bh" ]; then
        #IPPT setting need to be updated only when wan(rmnet) on-demand BH is up
        #current_bh=wan - Indicates other priority BH is brought up, call rmnet.script to disable IPPT.
        #new_bh=wan - Indicates other priority BH is brought down, call rmnet.script to enable IPPT
        if [ "$current_bh" == "wan$wwan_profile" -o "$new_bh" == "wan$wwan_profile" ]; then
          log $(basename "$0") ":switch_on_demand_backhaul:" $LINENO "Update IPPT as wan$wwan_profile is up"
          /etc/data/ippt.sh ippt_update_vlan_BHSwitch $vlan_idx
        fi
      fi

      # need update rule according to backhaul priority
      if [ -n "$new_bh" ]; then
        if [[ "$new_bh" =~ ^waneth.* ]]; then
          profile_id=${new_bh#waneth}
        elif [[ "$new_bh" =~ ^wan.* ]]; then
          profile_id=${new_bh#wan}
        fi
        log $(basename "$0") ":switch_on_demand_backhaul:" $LINENO "create dedicated routing table"
        util_create_dedicated_rt_non_wwan $profile_id
        downdevice="br-lan$vlan_id"
        dedicated_rt="custom_bind_${profile_id}"
        util_run_command ip "-4" rule del iif $downdevice
        util_run_command ip "-4" rule add iif $downdevice table $dedicated_rt prio 10
        log $(basename "$0") ":switch_on_demand_backhaul:" $LINENO "add route util_setup_nonwwan_route_rule_v4"
        util_setup_nonwwan_route_rule_v4 $new_bh
      fi
    fi
  done

  delete_conntrack_on_switch_bh_on_demand "ipv4"

}

function run_switch_backhaul_with_lock() {
  local backhaul_name=$1
  local PPP_IPPARAM=$1
  local link_state=$2

  (
    flock 300
    log $(basename "$0") "run_switch_backhaul_with_lock_backhaul_with_lock" $LINENO "Acquired lock, running switch_backhaul"
    local profile_proto=$(util_execute_uci get network.$backhaul_name.proto)
    if [[ "$backhaul_name" =~ ^waneth.* ]] && [ $link_state -eq 1 ] && [ "$profile_proto" = "pppoe" ]; then
	    IFNAME=$3
	    IPLOCAL=$4
	    NETMASK=$5
	    IPREMOTE=$6
	    MTU=$7
	    DNS1=$8
	    DNS2=$9
	    log $(basename "$0") "run_switch_backhaul_with_lock_backhaul_with_lock" $LINENO "IFNAME=$IFNAME IPLOCAL=$IPLOCAL NETMASK=$NETMASK IPREMOTE=$IPREMOTE MTU=$MTU DNS1=$DNS1 DNS2=$DNS2"
      echo "export IFNAME=\"$IFNAME\""                >  /tmp/ipv4config.$PPP_IPPARAM
      echo "export PUBLIC_IP=\"$IPLOCAL\""            >> /tmp/ipv4config.$PPP_IPPARAM
      echo "export NETMASK=\"$NETMASK\""              >> /tmp/ipv4config.$PPP_IPPARAM
      if [ -n "$DNS2" -a "$DNS1" != "$DNS2" ]; then
          echo "export DNSSERVERS=\"$DNS1 $DNS2\"" >> /tmp/ipv4config.$PPP_IPPARAM
      else
          echo "export DNSSERVERS=\"$DNS1\"" >> /tmp/ipv4config.$PPP_IPPARAM
      fi
      echo "export GATEWAY=\"$IPREMOTE\""             >> /tmp/ipv4config.$PPP_IPPARAM
      if [  -f /sys/class/net/$IFNAME/mtu ]; then
              MTU=$(cat /sys/class/net/$IFNAME/mtu)
      fi
      echo "export IPV4MTU=\"$MTU\""                  >> /tmp/ipv4config.$PPP_IPPARAM

      local profileid=$(uci get network.$PPP_IPPARAM.profile)
      /etc/data/lanUtils.sh util_create_dedicated_rt_non_wwan $profileid
      /etc/data/lanUtils.sh util_setup_nonwwan_route_rule_v4 $PPP_IPPARAM
    fi

    if [ -f /tmp/ipv4config.$PPP_IPPARAM ]; then
      log $(basename "$0") "run_switch_backhaul_with_lock_backhaul_with_lock" $LINENO "/tmp/ipv4config.$PPP_IPPARAM is present"
    fi
    switch_backhaul "$backhaul_name" "$link_state"
    log $(basename "$0") "run_switch_backhaul_with_lock_backhaul_with_lock" $LINENO "Releasing lock, after function execution"
  ) 300>/var/lock/switch_backhaul.lock
}

#Function to do neccessary updates on switch backhaul
#$1 - Backhaul name
#$2 - link_state(setup/teardown) BH
function switch_backhaul() {
  local defaultProfileId
  local current_bh profile_idx previous_bh
  local DNS
  local backhaul_name=$1
  local link_state=$2
  local bridge_mode=$(util_execute_uci get qcmap_wlan.@stamodeconfig[0].bridge_mode)
  local combined_current_bh
  local default_ethwan_profile=$(util_execute_uci get qcmap_lan.@lan[0].default_waneth_profile_num)
  local default_ethwan_vlan_id default_ethwan_iface
  local eth_type default_ethwan_proto
  local brwan_enabled=$(util_execute_uci get qcmap_lan.@global[0].brwan)

  log $(basename "$0") ":switch_backhaul:" $LINENO "Enter switch_backhaul backhaul_name=$backhaul_name link_state=$link_state"

  #Check input parament is null or not
  if [ -z "$1" -o -z "$2"  ]; then
    log $(basename "$0") "switch_backhaul" $LINENO "input is null"
    return
  fi

  if [ -n "$default_ethwan_profile" ]; then
    default_ethwan_vlan_id=$(util_execute_uci get qcmap_lan.profile_$default_ethwan_profile.vlan_id)
    default_ethwan_iface=$(util_execute_uci get qcmap_lan.profile_$default_ethwan_profile.device)
    default_ethwan_proto=$(util_execute_uci get qcmap_lan.profile_$default_ethwan_profile.proto)

    #Check if backhaul is default or not
    if is_default_backhaul $1; then
      log $(basename "$0") "switch_backhaul" $LINENO "backhaul is default"
    else
      log $(basename "$0") "switch_backhaul" $LINENO "$backhaul_name backhaul is not default, go switch_on_demand_backhaul"
      switch_on_demand_backhaul $1 $2
      return
    fi

    if [ "$default_ethwan_iface" == "eth0" ]; then
       eth_type="eth"
    elif [ "$default_ethwan_iface" == "eth1" ]; then
       eth_type="eth_nic2"
    fi

    #Update IPA for default ETHWAN profile
    if [ "$backhaul_name" == "waneth" -a "$link_state" == "1" ]; then
      ipa pppoe add_mapping $default_ethwan_iface $default_ethwan_vlan_id pppoe-$backhaul_name > /dev/null
      log $(basename "$0") "switch_backhaul" $LINENO "ipa pppoe add_mapping $default_ethwan_iface $default_ethwan_vlan_id pppoe-$backhaul_name"

      # if dhcp backhaul get dhcp ipv4 address, delete the static ip
      # in Deco mode, keep the static ip
      if [ "$brwan_enabled" != "1" -a "$default_ethwan_proto" == "dhcp" -a $default_ethwan_vlan_id -eq 0 ]; then
        util_execute_uci set network.$eth_type.disabled="1"
      fi
    elif [ "$backhaul_name" == "waneth" -a "$link_state" == "0" ]; then
      #Delete v4 conntrack entries on teardown
      util_del_conntrack $default_ethwan_profile "ipv4" 0
      ipa pppoe del_mapping $default_ethwan_iface $default_ethwan_vlan_id pppoe-$backhaul_name > /dev/null
      log $(basename "$0") "switch_backhaul" $LINENO "ipa pppoe del_mapping $default_ethwan_iface $default_ethwan_vlan_id pppoe-$backhaul_name"

      # if dhcp backhaul delete dhcp ipv4 address, add the static ip
      if [ "$default_ethwan_proto" == "dhcp" -a $default_ethwan_vlan_id -eq 0 ]; then
        util_execute_uci set network.$eth_type.disabled="0"
      fi
    fi
  else
    log $(basename "$0") "switch_backhaul" $LINENO "pppoe feature not enabled"
  fi

  defaultProfileId=`util_execute_uci get qcmap_lan.@no_of_configs[0].default_pdn`
  profile_idx=$(util_get_profile_index $defaultProfileId)

  #Get current BH enabled before updating priority BH
  current_bh=$(util_execute_uci get qcmap_lan.@profile[$profile_idx].bh_present)

  #remove iptables rules to block vlan to non wwan traffic in case of teardown
  if [ "$link_state" == "0" ]; then
    util_clear_ipv4_vlan_to_non_wwan_ipt_rule "$backhaul_name"
  fi
  update_present_backhaul "$backhaul_name" "$link_state" "ipv4"

  #Get previous BH after updating priority BH
  previous_bh=$(util_execute_uci get qcmap_lan.@profile[$profile_idx].bh_previous)

  #Get present BH after updating priority BH
  new_bh=$(util_execute_uci get qcmap_lan.@profile[$profile_idx].bh_present)
  log $(basename "$0") ":switch_backhaul:" $LINENO "current_bh:$current_bh;previous_bh:$previous_bh;new_bh:$new_bh"

  #Cleanup firewall upon BH switch in case if DL path keep only current BH i.e, highest priority BH
  combined_current_bh=$new_bh
  if [ -z "$new_bh" ]; then
    combined_current_bh=$(util_execute_uci get qcmap_lan.@profile[$profile_idx].bh_present_v6)
  fi
  firewall_cleanup "$current_bh" "$combined_current_bh"

  #Condition to avoid BHSwitch loop during IPPT reconfiguration
  if [ -z "$new_bh" -a "$current_bh" = "wan" -a "$previous_bh" = "wan" ]; then
    #We will enter this condition when:
    #*. WWAN BH is brought up on a non-ippt mode
    #*. During the process of configuring the above rmnet_update system call, user triggers IPPT reconfiguration.
    #*. This leads to a loop where PREVIOUS_BH will be updated to WAN and NEW_BH will be
    #   updated to NULL, while CURRENT_BH was WAN before calling UPDATE_PRESENT_BACKHAUL()
    #*. In this situation, we need to avoid calling IPPT_UPDATE_ATBHSWITCH()
    log $(basename "$0") ":switch_backhaul:" $LINENO "current_bh:$current_bh;previous_bh:$previous_bh;new_bh:$new_bh"

  elif [ -n "$current_bh" -a "$current_bh" != "$new_bh" ]; then

    #IPPT setting need to be updated only when wan(rmnet) BH is up
    #current_bh=wan - Indicates other priority BH is brought up, call rmnet.script to disable IPPT.
    #new_bh=wan - Indicates other priority BH is brought down, call rmnet.script to enable IPPT
    if [ "$current_bh" = "wan" -o "$new_bh" = "wan" ]; then
      log $(basename "$0") ":switch_backhaul:" $LINENO "Update IPPT as wan is up"
      /etc/data/ippt.sh ippt_update_atBHSwitch
    fi

    if [ -n "$new_bh" ]; then
      /etc/data/nat_alg_vpn_config.sh ENABLE_NAT -1

      #Need to stop firewall for bridge mode
      if [ $bridge_mode -eq 1 -a "$new_bh" == "wanwlan" ]; then
        /etc/data/firewallConfig.sh disable_firewall $defaultProfileId 1
        log $(basename "$0") ":switch_backhaul:" $LINENO "stop firewall since bridge mode is enabled"
      else
        /etc/data/firewallConfig.sh enable_firewall $defaultProfileId 1
      fi
    fi
  fi

  #enable/disable firewall when backhaul up/down
  if [ "$backhaul_name" == "waneth" -a "$link_state" == "1" -a "$bridge_mode" != "1" ]; then
    /etc/data/firewallConfig.sh enable_firewall $default_ethwan_profile 1
  elif [ "$backhaul_name" == "waneth" -a "$link_state" == "0" ]; then
    /etc/data/firewallConfig.sh disable_firewall $default_ethwan_profile 1
  fi

  #Check ip_collision.sh file is present
  if [ -f /etc/data/ip_collision.sh ]; then
    if [ -n "$current_bh" -a "$current_bh" != "$new_bh" ]; then
      log $(basename "$0") ":switch_backhaul:" $LINENO "previous_bh=$previous_bh:current_bh=$current_bh:new_bh=$new_bh:backhaul_name=$backhaul_name"
      #Update ip_collision if current/new bh is wwan
      if [ "$current_bh" = "wan" -o "$new_bh" = "wan" ]; then
        log $(basename "$0") ":switch_backhaul:" $LINENO "previous_bh=$previous_bh:current_bh=$current_bh:new_bh=$new_bh:backhaul_name=$backhaul_name"
        util_execute_uci set  qcmap_lan.@profile[$profile_idx].ip_collision_enable_in_progress="1"
        reload_ip_collision_as_need $defaultProfileId $new_bh
      fi
    fi
  fi

  reload_for_sta_mode
  update_dhcp_relay_for_bridge_mode
  reload_for_tinyproxy

  #Update DNSSERVERS information at BH switch
  update_dnsv4 $defaultProfileId $2

  #Update DNSSERVERS information at BH switch
  update_dnsv6 $defaultProfileId 1

  # update firewall.zone for dual backhaul
  update_firewall_zone_for_load_balance

  #After BH Switch Update public IP for guest AP rule.
  /etc/data/wlanConfig.sh utility_update_guest_ap_public_ip

  #Install iptables rules to block vlan to non wwan traffic for ipv4
  if [ "$new_bh" == "wanwlan" -a "$link_state" == "1" ]; then
    /etc/data/firewallConfig.sh install_ipv4_rules_to_block_vlan_access_to_non_wwan_backhaul
  fi

  #Uninstall and Install Guest AP rules.
  /etc/data/wlanConfig.sh delete_guest_ap_access_rules
  /etc/data/wlanConfig.sh install_guest_ap_access_rules

  delete_conntrack_on_switch_bh "ipv4"
}

function update_dhcp_pd_config()
{
  local defaultProfileId
  local profile_idx
  local current_bh_v6

  defaultProfileId=`uci get qcmap_lan.@no_of_configs[0].default_pdn`
  profile_idx=$(util_get_profile_index $defaultProfileId)
  current_bh_v6=$(uci get qcmap_lan.@profile[$profile_idx].bh_present_v6)
  log $(basename "$0") ":update_dhcp_pd_config:" $LINENO "current_bh_v6=$current_bh_v6"

  #If current_bh_v6 is wan and qcmap_lan pd_manager enabled, need add dhcp pd config
  #If current_bh_v6 is not wan or wan backhaul down, need delete dhcp pd config
  if [ -n "$current_bh_v6" -a "$current_bh_v6" == "wan"  ]; then
    /etc/data/dhcp_reload_pd_option.sh dhcp_enable_pd_config $IPV6_DHCP_ENABLE_PD_CONFIG
    log $(basename "$0") ":update_dhcp_pd_config:" $LINENO "/etc/data/dhcp_reload_pd_option.sh dhcp_enable_pd_config $IPV6_DHCP_ENABLE_PD_CONFIG"
  elif [ -z "$current_bh_v6" ] || [ "$current_bh_v6" != "wan" ]; then
    /etc/data/dhcp_reload_pd_option.sh dhcp_enable_pd_config $IPV6_DHCP_DISABLE_PD_CONFIG
    log $(basename "$0") ":update_dhcp_pd_config:" $LINENO "/etc/data/dhcp_reload_pd_option.sh dhcp_enable_pd_config $IPV6_DHCP_DISABLE_PD_CONFIG"
  fi
}

#update preset on-demand backhaul and set flag according to priority
function update_vlan_present_backhaul_v6() {
  local vlan_idx=$1
  local input_backhaul=$2
  local linkstate=$3
  local wwan_profile_num=$(util_execute_uci get qcmap_lan.@vlan[$vlan_idx].wwan_profile_num)
  local waneth_profile_num=$(util_execute_uci get qcmap_lan.@vlan[$vlan_idx].waneth_profile_num)
  local waneth_file6
  local backhaul6_name=""
  local backhaul6_file=""
  local checkIP
  local current_backhaul6
  local new_backhaul6=""
  local new_backhaul6_file=""

  log $(basename "$0") "update_vlan_present_backhaul_v6" $LINENO "Enter update_vlan_present_backhaul_v6 vlan_idx=$vlan_idx input_backhaul=$input_backhaul linkstate=$linkstate"

  #run loop to get bh_present_v6
  backhaullist=$(util_execute_uci get mwan3.backhaul_pref.use_member)
  if [ -n "$backhaullist" ]; then
    #IPv6 backhaul
    current_backhaul6=$(util_execute_uci get qcmap_lan.@vlan[$vlan_idx].bh_present_v6)
    log $(basename "$0") "update_vlan_present_backhaul_v6" $LINENO "current_IPv6_backhaul:$current_backhaul6"

    #initialize the bh_present be empty.
    util_execute_uci_set qcmap_lan.@vlan[$vlan_idx].bh_present_v6 ""

    for backhaulmember6 in $backhaullist
    do
      if [ "$backhaulmember6" == "m_waneth" ] && [ "$linkstate" == "0" ]; then
        log $(basename "$0") "update_vlan_present_backhaul_v6" $LINENO " ignore $input_backhaul since already handled"
        continue
      fi

      if [ "$backhaulmember6" == "m_wan" -a -n "$wwan_profile_num" ]; then
        backhaul6_name="wan$wwan_profile_num"
        backhaul6_file="/tmp/ipv6config$wwan_profile_num"
      elif [ "$backhaulmember6" == "m_waneth" -a -n "$waneth_profile_num" ]; then
        backhaul6_name="waneth$waneth_profile_num"
        backhaul6_file="/tmp/ipv6config.waneth$waneth_profile_num"
      else
        log $(basename "$0") "update_vlan_present_backhaul_v6" $LINENO ":not support $backhaulmember6 switch"
        continue
      fi

      if [ ! -f "$backhaul6_file" ]; then
        log $(basename "$0") "update_vlan_present_backhaul_v6" $LINENO ":Backhaul not up since file $backhaul6_file not present"
        continue
      fi
      log $(basename "$0") "update_vlan_present_backhaul_v6" $LINENO ":file present: $backhaul_file6"

      . $backhaul6_file

      #not use PUBLIC_IP check since when IPPT case rmnet don't have publicIP
      checkIP=`ip -6 addr show $IFNAME | grep $PUBLIC_IP6`
      if [ -z "$checkIP" ]; then
        log $(basename "$0") "update_vlan_present_backhaul_v6" $LINENO ":Backhaul not up since addr $PUBLIC_IP6 not valid"
        continue
      fi
      log $(basename "$0") "update_vlan_present_backhaul_v6" $LINENO ":$backhaul6_name v6 is up with devname=$IFNAME, addr=$PUBLIC_IP6"

      if [ -z "$new_backhaul6" ]; then
        new_backhaul6=$backhaul6_name
        new_backhaul6_file=$backhaul6_file

        if [ -n "$current_backhaul6" -a "$current_backhaul6" != "$backhaul6_name" ]; then
          util_execute_uci_set qcmap_lan.@vlan[$vlan_idx].bh_previous_v6 "$current_backhaul6"
        fi
        util_execute_uci_set qcmap_lan.@vlan[$vlan_idx].bh_present_v6 "$backhaul6_name"
        util_execute_uci commit qcmap_lan
      fi
    done

    combine_vlan_backhaul_ipv4ipv6 $vlan_idx
  else
    log $(basename "$0")":update_vlan_present_backhaul_v6:"$LINENO":get backhaul_pref.use_member failed:$backhaullist"
  fi
}

function switch_on_demand_backhaul_v6() {
  local backhaul6_name=$1
  local link_state=$2
  local vlan_id vlan_idx
  local current_bh_v6 new_bh_v6
  local input_profile
  local temp_profile_num profile_num
  local ethwan_profile_id
  local ethwan_profile_vlan_id
  local ethwan_profile_iface
  local no_of_vlans=$(util_execute_uci get qcmap_lan.@no_of_configs[0].no_of_vlans)
  local profile_id downdevice dedicated_rt

  log $(basename "$0") ":switch_on_demand_backhaul_v6:" $LINENO "Enter switch_on_demand_backhaul_v6 backhaul6_name=$backhaul6_name link_state=$link_state"

  #add/del backhaul to firewall wan_all zone
  if [[ "$backhaul_name" =~ ^waneth.* ]]; then
    add_del_on_demand_waneth_iface_on_wan_all_zone $backhaul_name $link_state
  fi

  if [[ "$backhaul6_name" =~ ^waneth.* ]]; then
    input_profile=${backhaul6_name#waneth}
    profile_num="waneth_profile_num"

    ethwan_profile_id=$input_profile
    ethwan_profile_vlan_id=$(util_execute_uci get qcmap_lan.profile_$ethwan_profile_id.vlan_id)
    ethwan_profile_iface=$(util_execute_uci get qcmap_lan.profile_$ethwan_profile_id.device)

    local ethwan_profile_proto=$(util_execute_uci get qcmap_lan.profile_$ethwan_profile_id.proto)
    #update ipa
    if [ "$link_state" == "0" ]; then
      #Delete v6 conntrack entries on teardown
      util_del_conntrack $ethwan_profile_id "ipv6" 0
      if [ "$ethwan_profile_proto" == "pppoe" ]; then
        ipa pppoe del_mapping $ethwan_profile_iface $ethwan_profile_vlan_id pppoe-$backhaul6_name > /dev/null
        log $(basename "$0") ":switch_on_demand_backhaul_v6:" $LINENO "ipa pppoe del_mapping $ethwan_profile_iface $ethwan_profile_vlan_id pppoe-$backhaul6_name"
      fi
    elif [ "$link_state" == "1" ]; then
    if [ "$ethwan_profile_proto" == "pppoe" ]; then
       ipa pppoe add_mapping $ethwan_profile_iface $ethwan_profile_vlan_id pppoe-$backhaul6_name > /dev/null
       log $(basename "$0") ":switch_on_demand_backhaul_v6:" $LINENO "ipa pppoe add_mapping $ethwan_profile_iface $ethwan_profile_vlan_id pppoe-$backhaul6_name"
    fi
    fi
  elif [[ "$backhaul6_name" =~ ^rmnet.* ]]; then
    input_profile=${backhaul6_name#rmnet}
    profile_num="wwan_profile_num"
    backhaul6_name="wan$input_profile"
  elif [[ "$backhaul6_name" =~ ^wan.* ]]; then
    input_profile=${backhaul6_name#wan}
    profile_num="wwan_profile_num"
  fi

    #delete waneth backhaul6 file when link down
  if [[ "$backhaul6_name" =~ ^waneth.* ]] && [ "$link_state" == "0" ]; then
    waneth_file6=/tmp/ipv6config.$backhaul6_name
    if [ -f "$waneth_file6" ]; then
      log $(basename "$0") "update_vlan_present_backhaul_v6" $LINENO "Ayush delete file $waneth_file6"
      rm $waneth_file6
    fi
  fi

   #check if vlan is exist
  if [ $no_of_vlans -eq 0 ]; then
    log $(basename "$0") "switch_on_demand_backhaul_v6" $LINENO "No VLAN is configured"
    return
  fi

  #loop through the number of vlans
  for vlan_idx in $(seq 0 $((no_of_vlans-1)))
  do
    temp_profile_num=$(util_execute_uci get qcmap_lan.@vlan[$vlan_idx].$profile_num)
    if [ -n "$temp_profile_num" -a $temp_profile_num -eq $input_profile ]; then
      vlan_id=$(util_execute_uci get qcmap_lan.@vlan[$vlan_idx].vlan_id)
      log $(basename "$0") ":switch_on_demand_backhaul_v6:" $LINENO "vlan_id:$vlan_id"

      #Get current BH enabled before updating priority BH
      current_bh_v6=$(util_execute_uci get qcmap_lan.@vlan[$vlan_idx].bh_present_v6)

      update_vlan_present_backhaul_v6 $vlan_idx $backhaul6_name $link_state

      #Get present v6 BH after updating priority BH
      new_bh_v6=$(util_execute_uci get qcmap_lan.@vlan[$vlan_idx].bh_present_v6)
      log $(basename "$0") ":switch_on_demand_backhaul_v6:" $LINENO "current_bh_v6:$current_bh_v6; new_bh_v6:$new_bh_v6"

      # need update rule according to backhaul6 priority
      if [ -n "$new_bh_v6" ]; then
        if [[ "$new_bh_v6" =~ ^waneth.* ]]; then
          profile_id=${new_bh_v6#waneth}
        elif [[ "$new_bh_v6" =~ ^wan.* ]]; then
          profile_id=${new_bh_v6#wan}
        fi
        log $(basename "$0") ":switch_on_demand_backhaul_v6:" $LINENO "create dedicated routing table"
        util_create_dedicated_rt_non_wwan $profile_id
        downdevice="br-lan$vlan_id"
        dedicated_rt="custom_bind_${profile_id}"
        util_run_command ip "-6" rule del iif $downdevice
        util_run_command ip "-6" rule add iif $downdevice table $dedicated_rt prio 10
        log $(basename "$0") ":switch_on_demand_backhaul_v6:" $LINENO "add route util_setup_nonwwan_route_rule_v6"
        util_setup_nonwwan_route_rule_v6 $new_bh_v6
      fi
    fi
  done
  delete_conntrack_on_switch_bh_on_demand "ipv6"
}

#Function to do neccessary updates on switch backhaul
#$1 - Backhaul name
#$2 - link_state(setup/teardown) BH
function switch_backhaul_v6() {
  local defaultProfileId
  local current_bh_v6 profile_idx previous_bh_v6
  local backhaul_name=$1
  local link_state=$2
  local bridge_mode=$(util_execute_uci get qcmap_wlan.@stamodeconfig[0].bridge_mode)
  local combined_current_bh
  local default_ethwan_profile=$(util_execute_uci get qcmap_lan.@lan[0].default_waneth_profile_num)
  local default_ethwan_vlan_id default_ethwan_iface

  log $(basename "$0") ":switch_backhaul_v6:" $LINENO "Enter switch_backhaul_v6 backhaul_name=$backhaul_name link_state=$link_state"

  #Check input parament is null or not
  if [ -z "$1" -o -z "$2"  ]; then
    log $(basename "$0") "switch_backhaul_v6" $LINENO "input is null"
    return
  fi

  if [ -n "$default_ethwan_profile" ]; then
    default_ethwan_vlan_id=$(util_execute_uci get qcmap_lan.profile_$default_ethwan_profile.vlan_id)
    default_ethwan_iface=$(util_execute_uci get qcmap_lan.profile_$default_ethwan_profile.device)

    #Check if backhaul is default or not
    if is_default_backhaul $1; then
      log $(basename "$0") "switch_backhaul_v6" $LINENO "backhaul is default"
    else
        log $(basename "$0") "switch_backhaul_v6" $LINENO "backhaul is not default, go switch_on_demand_backhaul_v6"
        switch_on_demand_backhaul_v6 $1 $2
      return
    fi

    #Update IPA for default ETHWAN profile
    if [ "$backhaul_name" == "waneth" -a "$link_state" == "1" ]; then
      ipa pppoe add_mapping $default_ethwan_iface $default_ethwan_vlan_id pppoe-$backhaul_name > /dev/null
      log $(basename "$0") "switch_backhaul" $LINENO "ipa pppoe add_mapping $default_ethwan_iface $default_ethwan_vlan_id pppoe-$backhaul_name"
    elif [ "$backhaul_name" == "waneth" -a "$link_state" == "0" ]; then
      #Delete v6 conntrack entries on teardown
      util_del_conntrack $default_ethwan_profile "ipv6" 0
      ipa pppoe del_mapping $default_ethwan_iface $default_ethwan_vlan_id pppoe-$backhaul_name > /dev/null
      log $(basename "$0") "switch_backhaul" $LINENO "ipa pppoe del_mapping $default_ethwan_iface $default_ethwan_vlan_id pppoe-$backhaul_name"
    fi
  else
    log $(basename "$0") "switch_backhaul_v6" $LINENO "pppoe feature not enabled"
  fi

  defaultProfileId=`util_execute_uci get qcmap_lan.@no_of_configs[0].default_pdn`
  profile_idx=$(util_get_profile_index $defaultProfileId)

  #Get previous BH enabled before updating priority BH
  previous_bh_v6=$(util_execute_uci get qcmap_lan.@profile[$profile_idx].bh_present_v6)

  #remove ip6tables rules to block vlan to non wwan traffic in case of teardown
  if [ "$link_state" == "0" ]; then
    util_clear_ipv6_vlan_to_non_wwan_ipt_rule "$backhaul_name"
  fi

  update_present_backhaul_v6 "$backhaul_name" "$link_state" "ipv6"

  current_bh_v6=$(util_execute_uci get qcmap_lan.@profile[$profile_idx].bh_present_v6)

  #Cleanup firewall upon BH switch in case if DL path keep only current BH i.e, highest priority BH
  combined_current_bh=$current_bh_v6
  if [ -z "$current_bh_v6" ]; then
    combined_current_bh=$(util_execute_uci get qcmap_lan.@profile[$profile_idx].bh_present)
  fi
  firewall_cleanup "$previous_bh_v6" "$combined_current_bh"

  if [ -n "$current_bh_v6" ]; then
    #Need to stop firewall for bridge mode
    if [ $bridge_mode -eq 1 -a "$current_bh_v6" == "wanwlan" ]; then
      /etc/data/firewallConfig.sh disable_firewall $defaultProfileId 2
      log $(basename "$0") ":switch_backhaul_v6:" $LINENO "stop firewall since bridge mode is enabled"
    else
      /etc/data/firewallConfig.sh enable_firewall $defaultProfileId 2
    fi
  fi

  #enable/disable firewall when backhaul up/down
  if [ "$backhaul_name" == "waneth" -a "$link_state" == "1" -a "$bridge_mode" != "1" ]; then
    /etc/data/firewallConfig.sh enable_firewall $default_ethwan_profile 2
  elif [ "$backhaul_name" == "waneth" -a "$link_state" == "0" ]; then
    /etc/data/firewallConfig.sh disable_firewall $default_ethwan_profile 2
  fi

  reload_for_sta_mode
  update_dhcp_relay_for_bridge_mode
  reload_for_tinyproxy

  #Update DNSSERVERS6 information at BH switch
  update_dnsv6 $defaultProfileId $2

  #Update DNSSERVERS information at BH switch
  update_dnsv4 $defaultProfileId 1

  #Update pd_manager and qcmap_prefix_delegation_mode in dhcp
  if [ -f /etc/data/dhcp_reload_pd_option.sh ]; then
    update_dhcp_pd_config
  fi
  #After BH Switch Update public IP for guest AP rule.
  /etc/data/wlanConfig.sh utility_update_guest_ap_public_ip

  #Install ip6tables rules to block vlan to non wwan traffic for ipv4
  if [ "$current_bh_v6" == "wanwlan" -a "$link_state" == "1" ]; then
    /etc/data/firewallConfig.sh install_ipv6_rules_to_block_vlan_access_to_non_wwan_backhaul
  fi

  #Uninstall and Install Guest AP rules.
  /etc/data/wlanConfig.sh delete_guest_ap_access_rules
  /etc/data/wlanConfig.sh install_guest_ap_access_rules

  delete_conntrack_on_switch_bh "ipv6"

}

function get_backhaul_file()
{
   local iptype=$1
   local backhaul=$2
   defaultProfileId=`uci get qcmap_lan.@no_of_configs[0].default_pdn`

   bridge_mode=$(util_execute_uci -q get qcmap_wlan.@stamodeconfig[0].bridge_mode)

   if [[ $backhaul =~ ^[wan0-9]+$ ]] ; then
     file="/tmp/"$iptype"config"$defaultProfileId
   else
     if [ $bridge_mode -eq 1 -a "$backhaul" == "wanwlan" ];then
       #sta bridge mode, tmp file is /tmp/ipv4config.lan
       file="/tmp/"$iptype"config.lan"
     else
       file="/tmp/"$iptype"config."$backhaul
     fi
   fi
}

#check whether wwan will be prefer backhaul
#  update_present_backhaul: will be called to update prefered backhaul and also default route in UCI(bh_present) when new backhaul active,
#      it has to be called when the inteface is updated done by netifd.
#  is_wwan_prefer_backhaul: can be used when UCI(bh_present) not ready.
function is_wwan_prefer_backhaul()
{
  local iptype="ipv4"
  local isWWANAPrefered="0"

  backhaullist=$(uci get mwan3.backhaul_pref.use_member)

  for backhaulmember in $backhaullist
  do
    backhaul=$(uci get mwan3.$backhaulmember.interface)
    get_backhaul_file "$iptype" "$backhaul"

    if [ ! -f "$file" ]; then
      log $(basename "$0") "is_wwan_prefer_backhaul" $LINENO "Backhaul not up since $file not present"
      continue
    fi
    log $(basename "$0") "is_wwan_prefer_backhaul" $LINENO "file present: $file"

    . $file

    checkIP=`ip addr show $IFNAME | grep inet`

    if [ ! -n "$checkIP" ]; then
      log $(basename "$0") "is_wwan_prefer_backhaul" $LINENO "Backhaul not up since addr $PUBLIC_IP not valid"
      continue
    fi

    log $(basename "$0") "is_wwan_prefer_backhaul" $LINENO "$backhaul is up with devname=$IFNAME, addr=$PUBLIC_IP"
    if [[ $backhaul =~ ^[wan0-9]+$ ]] ; then
      log $(basename "$0") "is_wwan_prefer_backhaul" $LINENO "wwan is prefered active backhaul"
      isWWANAPrefered="1"
      break
    else
      log $(basename "$0") "is_wwan_prefer_backhaul" $LINENO "non-wwan is not prefered active backhaul"
      isWWANAPrefered="0"
      break
    fi

  done

  echo "$isWWANAPrefered"
}

#check whether on-demand wwan will be prefer backhaul
#for pppoe feature, only compare wwan and waneth
#if wwan not mapped to a vlan, isWWANAPrefered=1
function is_on_demand_wwan_prefer_backhaul()
{
  local profile=$1
  local isWWANAPrefered="1"
  local profile_idx=$(util_get_profile_index $profile)
  local no_of_vlans=$(util_execute_uci get qcmap_lan.@no_of_configs[0].no_of_vlans)
  local vlan_idx
  local temp_profile_num
  local found=0
  local ipv4_bh

  #Get profile_index.

  #check if vlan is exist
  if [ $no_of_vlans -eq 0 ]; then
    log $(basename "$0") "is_on_demand_wwan_prefer_backhaul" $LINENO "No VLAN is configured"
    echo "1"
    return
  fi

  #loop through the number of vlans
  for vlan_idx in $(seq 0 $((no_of_vlans-1)))
  do
    temp_profile_num=$(util_execute_uci get qcmap_lan.@vlan[$vlan_idx].wwan_profile_num)
    if [ -n "$temp_profile_num" -a $temp_profile_num -eq $profile ]; then
      found=1
      vlan_id=$(util_execute_uci get qcmap_lan.@vlan[$vlan_idx].vlan_id)
      log $(basename "$0") ":is_on_demand_wwan_prefer_backhaul:" $LINENO "wan$profile mapped to vlan$vlan_id"

      ipv4_bh=$(util_execute_uci get qcmap_lan.@vlan[$vlan_idx].bh_present)
      if [[ "$ipv4_bh" =~ ^wan[0-9] ]]; then
        isWWANAPrefered="1"
        break
      else
        isWWANAPrefered="0"
        break
      fi
    fi
  done

  if [ "$found" -eq 0 ]; then
    log $(basename "$0") ":is_on_demand_wwan_prefer_backhaul:" $LINENO "wan$profile not mapped to any vlan"
    isWWANAPrefered="1"
  fi

  log $(basename "$0") ":is_on_demand_wwan_prefer_backhaul:" $LINENO "isWWANAPrefered: $isWWANAPrefered"
  echo "$isWWANAPrefered"
}

##reconfigure dhcpv6 relay setting
function update_dhcpv6_relay_mode()
{
  local new_backhaul=$1
  local need_reload_dhcp=0
  local need_set_dhcp_server_mode=0
  local ipv6_bh
  local eogre_enabled
  ipv6_bh=$(util_execute_uci -q get qcmap_lan.@profile[$profile_idx].bh_present_v6)
  local bridge_mode=$(util_execute_uci get qcmap_wlan.@stamodeconfig[0].bridge_mode)
  local dhcp_lan_ndp dhcp_waneth_ndp
  local defaultProfileId=$(util_execute_uci get qcmap_lan.@no_of_configs[0].default_pdn)

  log $(basename "$0") "update_dhcpv6_relay_mode" $LINENO "enter update_dhcpv6_relay_mode, new_backhaul:$new_backhaul"
  if [ -z "$bridge_mode" ]; then
    bridge_mode=0
  fi

  if [ -z "$ipv6_bh" ]; then
    dhcp_lan_ndp=$(util_execute_uci get dhcp.lan.ndp)
    dhcp_waneth_ndp=$(util_execute_uci get dhcp.waneth_v6.ndp)
    if [ -z "$dhcp_lan_ndp" ]; then
      #switch_backhaul and switch_backhaul_v6 is called almost the same time
      #due to race condition, waneth_v6.ndp sometimes may not be deleted before
      if [ -n "$dhcp_waneth_ndp" ]; then
        log $(basename "$0") "update_dhcpv6_relay_mode" $LINENO "delete dhcp.waneth_v6.ndp"
        util_execute_uci -q delete dhcp.waneth_v6.master
        util_execute_uci -q delete dhcp.waneth_v6.ra
        util_execute_uci -q delete dhcp.waneth_v6.ndp
        util_execute_uci -q delete dhcp.waneth_v6.dhcpv6
        util_execute_uci -q delete dhcp.waneth_v6.ndproxy_routing
        util_execute_uci commit dhcp
        /etc/init.d/odhcpd reload
      fi

      log $(basename "$0") "update_dhcpv6_relay_mode" $LINENO "dhcpv6 relay was not set and no need set(ipv6 backhaul not up), break"
      return
    fi
    #IPv6 not ready on current backhaul but was in other backhaul, handle it as no IPv6 backhaul
    new_backhaul="no"
  fi

  backhaullist=$(uci get mwan3.backhaul_pref.use_member)
  for backhaulmember in $backhaullist
  do
    backhaulname=${backhaulmember#*m_}

    ismaster=$(util_execute_uci -q get dhcp.${backhaulname}_v6.master)
    master_dhcp=$(util_execute_uci get dhcp.lan.master_dhcp)
    if [ -z "$ismaster" -o -z "$master_dhcp" ]; then
      if [ "$backhaulname" == "$new_backhaul" ]; then
        if [ "$backhaulname" == "wan" ]; then
          log $(basename "$0") "update_dhcpv6_relay_mode" $LINENO "dhcpv6 relay no need set since wwan is default backhaul"
          #set dhcp.lan as server mode later,
          #this is to allow odhcpd know which is slave interface to send deprecated RA.
          need_set_dhcp_server_mode=1
          need_reload_dhcp=1
        else
          if [ $bridge_mode -eq 0 ]; then
            log $(basename "$0") "update_dhcpv6_relay_mode" $LINENO "dhcpv6 master not on default backhaul $backhaulname: set it"

            #before set dhcp to relay mode, remove IPv6 on br-lan which is from wwan
            set_brlan_wwan_ipv6prefix_after_switchBH $defaultProfileId "lan"

            #before switch to relay mode, let odhcpd send deprecated RA for server mode
            util_execute_uci_set dhcp.lan.ra     "disabled"
            util_execute_uci commit dhcp
            util_run_command /etc/init.d/odhcpd reload
            util_run_command sleep 0.2

            util_execute_uci_set dhcp.${backhaulname}_v6.dhcpv6 "relay"
            util_execute_uci_set dhcp.${backhaulname}_v6.ra     "relay"
            util_execute_uci_set dhcp.${backhaulname}_v6.ndp    "relay"
            util_execute_uci_set dhcp.${backhaulname}_v6.master "1"
            util_execute_uci_set dhcp.${backhaulname}_v6.ndproxy_routing "0"
            util_execute_uci_set dhcp.lan.dhcpv6 "relay"
            util_execute_uci_set dhcp.lan.ra     "relay"
            util_execute_uci_set dhcp.lan.ndp    "relay"
            util_execute_uci_set dhcp.lan.master_dhcp "${backhaulname}_v6"
            need_reload_dhcp=1
          elif [ $bridge_mode -eq 1 ]; then
            log $(basename "$0") "update_dhcpv6_relay_mode" $LINENO "STA bridge mode: handle in dhcpRelay.sh"
          fi
        fi
      fi
    else
      if [ "$backhaulname" != "$new_backhaul" ]; then
        log $(basename "$0") "update_dhcpv6_relay_mode" $LINENO "dhcpv6 master was on $backhaulname but default backhaul on $new_backhaul: delete it"
        util_execute_uci -q delete dhcp.${backhaulname}_v6.master
        util_execute_uci -q delete dhcp.${backhaulname}_v6.ra
        util_execute_uci -q delete dhcp.${backhaulname}_v6.ndp
        util_execute_uci -q delete dhcp.${backhaulname}_v6.dhcpv6
        util_execute_uci -q delete dhcp.${backhaulname}_v6.ndproxy_routing
        if [ "$backhaulname" == "waneth" ]; then
          need_set_dhcp_server_mode=1
        fi

        #remove the master firstly and trigger reload, then odhcpd able to send deprecated RA for deprecated wan interface.
        util_execute_uci commit dhcp
        /etc/init.d/odhcpd reload
        #wait odhcpd finish send deprecated RA.
        sleep 0.5

        if [ "$new_backhaul" == "no" ]; then
          eogre_enabled=$(util_execute_uci get qcmap_lan.@no_of_configs[0].eogre_enabled)
          if [ $eogre_enabled -ne 1 ]; then
            util_execute_uci_set    dhcp.lan.dhcpv6 "server"
            util_execute_uci_set    dhcp.lan.ra     "server"
          fi
        fi
        need_reload_dhcp=1
      else
        log $(basename "$0") "update_dhcpv6_relay_mode" $LINENO "dhcpv6 master already on $backhaulname: no action"
      fi
    fi
  done

  if [ $need_set_dhcp_server_mode -eq 1 ]; then
     eogre_enabled=$(util_execute_uci get qcmap_lan.@no_of_configs[0].eogre_enabled)
     if [ $eogre_enabled -ne 1 ]; then
       util_execute_uci_set    dhcp.lan.dhcpv6 "server"
       util_execute_uci_set    dhcp.lan.ra     "server"
       util_execute_uci delete dhcp.lan.master_dhcp
     fi
  fi

  if [ $need_reload_dhcp -eq 1  ]; then
     util_execute_uci commit dhcp
     /etc/init.d/odhcpd reload
  fi

  #after set dhcp mode, ensure br-lan have IPv6 addr back
  set_brlan_wwan_ipv6prefix_after_switchBH $defaultProfileId "lan"

}

function update_vlan_default_backhaul_status() {
  local vlan_idx vlan_id
  local vlan_wan_profile vlan_ethwan_profile
  local ismaster master_dhcp
  local default_wan_profile=$(util_execute_uci get qcmap_lan.@no_of_configs[0].default_pdn)
  local profile_idx=$(util_get_default_profile_index)
  local default_ethwan_profile=$(util_execute_uci get qcmap_lan.@lan[0].default_waneth_profile_num)
  local ipv4_bh=$(util_execute_uci get qcmap_lan.@profile[$profile_idx].bh_present)
  local ipv6_bh=$(util_execute_uci get qcmap_lan.@profile[$profile_idx].bh_present_v6)
  local no_of_vlans=$(util_execute_uci get qcmap_lan.@no_of_configs[0].no_of_vlans)
  local bridge_mode=$(util_execute_uci get qcmap_wlan.@stamodeconfig[0].bridge_mode)

  if [ -z "$bridge_mode" ]; then
    bridge_mode=0
  fi

  log $(basename "$0") "update_vlan_default_backhaul_status" $LINENO "Enter update_vlan_default_backhaul_status"

  #check if vlan is exist
  if [ $no_of_vlans -eq 0 ]; then
    log $(basename "$0") "update_vlan_default_backhaul_status" $LINENO "No VLAN is configured"
    return
  fi

  #loop through the number of vlans
  for vlan_idx in $(seq 0 $((no_of_vlans-1)))
  do
    vlan_wan_profile=$(util_execute_uci get qcmap_lan.@vlan[$vlan_idx].wwan_profile_num)
    vlan_ethwan_profile=$(util_execute_uci get qcmap_lan.@vlan[$vlan_idx].waneth_profile_num)
    if [ "$vlan_wan_profile" == "$default_wan_profile" ] || [ "$vlan_ethwan_profile" == "$default_ethwan_profile" ]; then
      util_execute_uci_set qcmap_lan.@vlan[$vlan_idx].bh_present "$ipv4_bh"
      util_execute_uci_set qcmap_lan.@vlan[$vlan_idx].bh_present_v6 "$ipv6_bh"

      vlan_id=$(util_execute_uci get qcmap_lan.@vlan[$vlan_idx].vlan_id)
      ismaster=$(util_execute_uci get dhcp.waneth_v6.master)
      master_dhcp=$(util_execute_uci get dhcp.lan$vlan_id.master_dhcp)
      if [ $bridge_mode -eq 0 ] && [ -n "$ismaster" -a -z "$master_dhcp" ]; then
        #before set dhcp to relay mode, remove IPv6 on br-lan which is from wwan
        set_brlan_wwan_ipv6prefix_after_switchBH $defaultProfileId "lan$vlan_id"

        #before switch to relay mode, let odhcpd send deprecated RA for server mode
        util_execute_uci_set dhcp.lan$vlan_id.ra     "disabled"
        util_execute_uci commit dhcp
        util_run_command /etc/init.d/odhcpd reload
        util_run_command sleep 0.2

        log $(basename "$0") "update_vlan_default_backhaul_status" $LINENO "set relay mode for dhcp lan$vlan_id"
        util_execute_uci_set dhcp.lan$vlan_id.dhcpv6 "relay"
        util_execute_uci_set dhcp.lan$vlan_id.ra     "relay"
        util_execute_uci_set dhcp.lan$vlan_id.ndp    "relay"
        util_execute_uci_set dhcp.lan$vlan_id.master_dhcp "waneth_v6"
      elif [ -z "$ismaster" -a -n "$master_dhcp" ]; then
        log $(basename "$0") "update_vlan_default_backhaul_status" $LINENO "set server mode for dhcp lan$vlan_id"
        util_execute_uci_set dhcp.lan$vlan_id.dhcpv6 "server"
        util_execute_uci_set dhcp.lan$vlan_id.ra     "server"
        util_execute_uci -q delete dhcp.lan$vlan_id.master_dhcp
      else
        log $(basename "$0") "update_vlan_default_backhaul_status" $LINENO "mode already set"
        continue
      fi

      util_execute_uci commit dhcp
      /etc/init.d/odhcpd reload

      #after set dhcp mode, ensure br-lan$vlan_id have IPv6 addr back
      set_brlan_wwan_ipv6prefix_after_switchBH $defaultProfileId "lan$vlan_id"
    fi
  done
}

#we have seperated backhaul with IPv4 IPv6,
#then it is possbile highest backhaul have only IP type active, such as IPv4, but other backhaul may IPv6, or vise versa
#then IPv4 go one backhaul, IPv6 go another,
#to avoid this, we disable the defaultroute of IPv6 to disable the data path, and also set bh_present value
function combine_backhaul_ipv4ipv6()
{
  local profile_idx=$1
  local previous_backhaul
  local new_backhaul=""
  local bh_previous
  local wanwlan_proto

  log $(basename "$0") "combine_backhaul_ipv4ipv6" $LINENO "combine_backhaul_ipv4ipv6 enter profile_idx=$profile_idx"

  ipv4_bh=$(util_execute_uci get qcmap_lan.@profile[$profile_idx].bh_present)
  ipv6_bh=$(util_execute_uci get qcmap_lan.@profile[$profile_idx].bh_present_v6)
  wanwlan_proto=$(util_execute_uci get network.wanwlan.proto)

  #if no value, set it to invalid value to easy handle.
  if [ ! -n "$ipv4_bh" ];then
    ipv4_bh="no"
  fi
  if [ ! -n "$ipv6_bh" ];then
    ipv6_bh="no"
  fi

  backhaullist=$(uci get mwan3.backhaul_pref.use_member)
  for backhaulmember in $backhaullist
  do
    backhaulname=${backhaulmember#*m_}
    defaultroute_value=$(util_execute_uci get network.${backhaulname}.defaultroute)
    if [ $defaultroute_value -eq 1 ]; then
      previous_backhaul=$backhaulname
    fi
    util_execute_uci_set network.${backhaulname}_v6.defaultroute "0"
    util_execute_uci_set network.${backhaulname}.defaultroute "0"

    #temp workaround for ipv6: switch_bh_v6 is not called from netifd/odhcp6c when ipv6 interface down;
    #delete the file and don't consider the backhaul as current backhaul
    get_backhaul_file "ipv6" $backhaulname
    if [ "$backhaulname" != "wan" -a -f $file ]; then
      . $file
      checkIP=$(ip -6 addr show $IFNAME | grep $PUBLIC_IP6)
      if [ ! -n "$checkIP" ]; then
        log $(basename "$0") "combine_backhaul_ipv4ipv6" $LINENO "${backhaulname}_v6 not up since IPv6 addr not preset"
        rm $file
        if [ "$backhaulname" == "$ipv6_bh" ];then
           ipv6_bh="no"
        fi
      fi
    fi

  done

  for backhaulmember in $backhaullist
  do
    backhaulname=${backhaulmember#*m_}

    log $(basename "$0") "combine_backhaul_ipv4ipv6" $LINENO "ipv4_bh=$ipv4_bh, ipv6_bh=$ipv6_bh, loop on $backhaulname"

    if [ "$ipv4_bh" == "no" -a "$ipv6_bh" == "no" ]; then
      #this can happen in below scenario for example:
      #wwan have IPv4IPv6 and is higest priority, waneth is low prority and with v4 only, then,
      #1. trigger wwan IPv4 disconnect firstly, then bh_present update to empty, bh_present_v6 keep "wan"
      #2. few seconds later, trigger wwan IPv6 disconnect, then bh_present_v6 update to empty,
      #and so, both ipv4_bh and ipv6_bh are empty,
      #then we need check again whether any other backhaul is active and set defaultroute and bh_present for them.
      get_backhaul_file "ipv4" $backhaulname
      ipv4file=$file
      get_backhaul_file "ipv6" $backhaulname
      ipv6file=$file
      if [ -f $ipv4file -o -f $ipv6file ];then
        if [ -f $ipv4file ];then
          util_execute_uci_set network.${backhaulname}.defaultroute    "1"
          util_execute_uci_set qcmap_lan.@profile[$profile_idx].bh_present "$backhaulname"
        fi
        if [ -f $ipv6file ];then
          util_execute_uci_set network.${backhaulname}_v6.defaultroute "1"
          util_execute_uci_set qcmap_lan.@profile[$profile_idx].bh_present_v6 "$backhaulname"
        fi
        break;
      fi
    elif [ "$backhaulname" == "$ipv4_bh" -o "$backhaulname" == "$ipv6_bh" ]; then
      #either Ipv4 or IPv6 is up;
      log $(basename "$0") "combine_backhaul_ipv4ipv6" $LINENO "$backhaulname is highest priority"

      #logic to ensure delete route happen before add route
      if [ -n "$previous_backhaul" ]; then
        if [ "$previous_backhaul" != "$backhaulname" ]; then
          #default route change from lowest priority to high priority backhaul
          #trigger network reload for delete-route and wait 200ms, this is to ensure delete route happen firstly;
          util_execute_uci commit network
          ubus call network reload
          log $(basename "$0") "combine_backhaul_ipv4ipv6" $LINENO "wait 200ms to allow netifd delete default route of $previous_backhaul"
          sleep 0.2
        fi
      fi

      util_execute_uci_set network.${backhaulname}_v6.defaultroute "1"
      if [ "$wanwlan_proto" == "static" ] && [ "$backhaulname" == "wanwlan" ]; then
        log $(basename "$0") "combine_backhaul_ipv4ipv6" $LINENO "backhaul is wanwlan and proto is static"
        util_execute_uci_set network.${backhaulname}.defaultroute    "0"
      else
        util_execute_uci_set network.${backhaulname}.defaultroute    "1"
      fi
      new_backhaul=$backhaulname

      if [ "$backhaulname" != "$ipv6_bh" ]; then
        get_backhaul_file "ipv6" $backhaulname
        if [ -f $file ]; then
          #IPv6 present
          util_execute_uci_set qcmap_lan.@profile[$profile_idx].bh_present_v6 "$backhaulname"

          bh_previous=$(util_execute_uci get qcmap_lan.@profile[$profile_idx].bh_previous)
          util_execute_uci_set qcmap_lan.@profile[$profile_idx].bh_previous_v6 "$bh_previous"
        else
          #IPv6 is not on this highest prioity backhaul, disable default route for IPv6 on the low priority backhaul
          util_execute_uci_set qcmap_lan.@profile[$profile_idx].bh_present_v6 ""
        fi
      fi

      if [ "$backhaulname" != "$ipv4_bh" ]; then
        get_backhaul_file "ipv4" $backhaulname
        if [ -f $file ]; then
          #IPv4 present
          util_execute_uci_set qcmap_lan.@profile[$profile_idx].bh_present "$backhaulname"
        else
          #IPv4 is not on this highest prioity backhaul, disable  default route for IPv4 on the low priority backhaul
          util_execute_uci_set qcmap_lan.@profile[$profile_idx].bh_present ""
        fi
      fi
      break
    fi

    if [ "$backhaulname" == "$previous_backhaul" ]; then
      #go to here means the previous_backhaul priority is high than new highest active backhaul,
      #then, it indicate the previous_backhaul is down;
      #then, the default route will be deleted automatic when iface down, we don't need add sleep to wait route delete
      log $(basename "$0") "combine_backhaul_ipv4ipv6" $LINENO "$previous_backhaul is down, no need sleep/wait for delete default route for $previous_backhaul"
      previous_backhaul="";
    fi

  done

  util_execute_uci commit qcmap_lan
  util_execute_uci commit network
  ubus call network reload

  update_dhcpv6_relay_mode "$new_backhaul"

  update_vlan_default_backhaul_status
}

function get_bh_load_balance_status() {
  local wan_metric
  local waneth_metric

  wan_metric=$(util_execute_uci get mwan3.m_wan.metric)
  waneth_metric=$(util_execute_uci get mwan3.m_waneth.metric)

  if [ "$wan_metric" == "$waneth_metric" ]; then
    qcmap_bh_load_balance_enabled=1
    echo "$qcmap_bh_load_balance_enabled"
  else
    qcmap_bh_load_balance_enabled=0
    echo "$qcmap_bh_load_balance_enabled"
  fi

  log $(basename "$0") "get_bh_load_balance_status" $LINENO " backhaul load balance status: $qcmap_bh_load_balance_enabled"
}

function add_second_default_route_for_load_balance()
{
  local backhaulmember backhaullist backhaulname
  local file_wan file_waneth
  local defaultProfileId
  local defaultroute_value
  local table_num

  log $(basename "$0") "add_second_default_route_for_load_balance" $LINENO "Enter add_second_default_route_for_load_balance"

  get_bh_load_balance_status
  if [ "$qcmap_bh_load_balance_enabled" == "$QCMAP_BH_LOAD_BALANCE_ENABLE" ]; then
    defaultProfileId=`util_execute_uci get qcmap_lan.@no_of_configs[0].default_pdn`

    file_wan=/tmp/ipv4config$defaultProfileId
    file_waneth=/tmp/ipv4config.waneth

    #both wan and waneth need up
    if [ ! -f "$file_wan" ] || [ ! -f "$file_waneth" ]; then
      log $(basename "$0") "add_second_default_route_for_load_balance" $LINENO "wan and waneth backhaul not all up"
      return
    fi

    backhaullist=$(util_execute_uci get mwan3.backhaul_pref.use_member)
    for backhaulmember in $backhaullist
    do
      backhaulname=${backhaulmember#*m_}
      defaultroute_value=$(util_execute_uci get network.${backhaulname}.defaultroute)
      if [ $defaultroute_value -eq 1 ]; then
        log $(basename "$0") "add_second_default_route_for_load_balance" $LINENO "default route backhaul is $backhaulname"
        continue
      #add router for the second backhaul
      elif [ $defaultroute_value -eq 0 ]; then
        if [ $backhaulname == "wan" ]; then
          . $file_wan
          table_num=`uci show mwan3 | grep -i "\.interface" | grep -n -w "wan" | head -1 | cut -d ":" -f 1`
        elif [ $backhaulname == "waneth" ]; then
          . $file_waneth
          table_num=`uci show mwan3 | grep -i "\.interface" | grep -n -w "waneth" | head -1 | cut -d  ":" -f 1`
        fi

        ip route add table $table_num default via $GATEWAY dev $IFNAME proto static src $PUBLIC_IP metric 256 mtu lock 1400
        log $(basename "$0") "ip router cmd:" $LINENO "cmd: ip route add table $table_num default via $GATEWAY dev $IFNAME proto static src $PUBLIC_IP metric 256 mtu lock 1400"
      fi
    done

    log $(basename "$0") "add_second_default_route_for_load_balance" $LINENO "success"
  fi
}

function run_combine_backhaul_with_lock() {
  local profile_idx=$1
  (
    flock 200
    log $(basename "$0") "run_combine_backhaul_with_lock" $LINENO "Acquired lock, running combine_backhaul_ipv4ipv6"
    combine_backhaul_ipv4ipv6 "$profile_idx"
    log $(basename "$0") "run_combine_backhaul_with_lock" $LINENO "Releasing lock, after function execution"
  ) 200>/var/lock/combine_backhaul.lock
}

#update preset backhaul and set flag according to priority
function update_present_backhaul() {
  local checkIP
  local current_backhaul
  local new_backhaul
  local new_backhaul_file
  local backhaulname=$1
  local linkstate=$2
  local iptype=$3
  local default_ethwan_profile=$(util_execute_uci get qcmap_lan.@lan[0].default_waneth_profile_num)

  log $(basename "$0") "update_present_backhaul" $LINENO "Enter update_present_backhaul backhaulname=$backhaulname linkstate=$linkstate iptype=$iptype"

  backhaullist=$(uci get mwan3.backhaul_pref.use_member)
  defaultProfileId=`uci get qcmap_lan.@no_of_configs[0].default_pdn`

  profile_idx=$(util_get_default_profile_index)
  log $(basename "$0") ":update_present_backhaul:" $LINENO ":profile_idx: $profile_idx"

  if [[ "$backhaulname" =~ ^rmnet.* ]]; then
    input_profile=${backhaulname#rmnet}
    if [ $input_profile != $defaultProfileId ]; then
      log $(basename "$0") "update_present_backhaul" $LINENO " no need backhaul-switch for ondemaond PDP $input_profile !"
      return
    else
      backhaulname="wan"
    fi
  fi

  #remove default route when link down
  if [ "$linkstate" == "0" ]; then
    util_remove_mtu_option_in_firewall "4" $defaultProfileId
    get_backhaul_file $iptype $backhaulname
    update_v4_mwan3_default_route $backhaulname $linkstate $file
  fi

  #run loop to get bh_present
  if [ ! -z "$backhaullist" ]; then
    #IPv4 backhaul
    current_backhaul=$(uci get qcmap_lan.@profile[$profile_idx].bh_present)
    log $(basename "$0") "update_present_backhaul" $LINENO "current_IPv4_backhaul:$current_backhaul"

    #initialize the bh_present be empty.
    util_execute_uci_set qcmap_lan.@profile[$profile_idx].bh_present ""

    for backhaulmember in $backhaullist
    do
      backhaul=$(uci get mwan3.$backhaulmember.interface)
      if [ "$backhaul" == "$backhaulname" ] && [ "$linkstate" == "0" ] ; then
        #the scenario happen when IPPT case, rmnet.script down is called but the ipv4config still need be present
        log $(basename "$0") "update_present_backhaul" $LINENO " ignore $backhaul since already handled"
        continue
      fi
      get_backhaul_file $iptype $backhaul

      if [ ! -f "$file" ]; then
        log $(basename "$0") "update_present_backhaul" $LINENO ":Backhaul not up since $file not present"
        continue
      fi
      log $(basename "$0") "update_present_backhaul" $LINENO ":file present: $file"

      . $file

      #not use PUBLIC_IP check since when IPPT case rmnet don't have publicIP
      checkIP=`ip addr show $IFNAME | grep inet`

      if [ ! -n "$checkIP" ]; then
        log $(basename "$0") "update_present_backhaul" $LINENO ":Backhaul not up since addr $PUBLIC_IP not valid"
        continue
      fi

      log $(basename "$0") "update_present_backhaul" $LINENO ":$backhaul is up with devname=$IFNAME, addr=$PUBLIC_IP"

      if [ -z "$new_backhaul" ] ; then
        new_backhaul=$backhaul
        new_backhaul_file=$file

        if [ -n "$current_backhaul" -a "$current_backhaul" != "$backhaul" ]; then
          util_execute_uci_set qcmap_lan.@profile[$profile_idx].bh_previous "$current_backhaul"
        fi
        util_execute_uci_set qcmap_lan.@profile[$profile_idx].bh_present "$backhaul"
        util_execute_uci commit qcmap_lan
      fi
    done

      if [ -f $new_backhaul_file ]; then
        unset IPV4MTU
        . $new_backhaul_file
        if [ ! -z $IPV4MTU ] && [ -n $IPV4MTU ]; then
          util_add_mtu_option_in_firewall "4" "$IFNAME" "$IPV4MTU" $defaultProfileId
        fi
      fi

    #combine_backhaul_ipv4ipv6 $profile_idx
    run_combine_backhaul_with_lock $profile_idx

    # if enable load balance first, then both wan and waneth backhaul up,
    # will add default router for another backhaul
    add_second_default_route_for_load_balance

    if [ -z $current_backhaul ] || [ "$current_backhaul" != "$new_backhaul" ]; then
      #send sig to QCMAP for sending backhaul status indication
      echo "$new_backhaul" > /var/run/data/backhaul/current_backhaul
    fi

    if [ -n $default_ethwan_profile ]; then
      util_execute_uci_set qcmap_lan.profile_$default_ethwan_profile.bh_present ""

      # save deafult waneth backaul status to waneth profile
      if [ "$new_backhaul" != "wan" ]; then
        util_execute_uci_set qcmap_lan.profile_$default_ethwan_profile.bh_present "$new_backhaul"
      fi
    fi
  else
    log $(basename "$0")":update_present_backhaul:"$LINENO":get backhaul_pref.use_member failed:$backhaullist"
  fi

}

function update_present_backhaul_v6() {
  local checkIP
  local current_backhaul6
  local new_backhaul6
  local new_backhaul6_file
  local backhaulname=$1
  local linkstate=$2
  local iptype=$3
  local ipv4_bh
  local default_ethwan_profile=$(util_execute_uci get qcmap_lan.@lan[0].default_waneth_profile_num)

  log $(basename "$0") "update_present_backhaul_v6" $LINENO "Enter update_present_backhaul_v6 backhaulname=$backhaulname linkstate=$linkstate iptype=$iptype"

  backhaullist=$(uci get mwan3.backhaul_pref.use_member)
  defaultProfileId=`uci get qcmap_lan.@no_of_configs[0].default_pdn`

  profile_idx=$(util_get_profile_index $defaultProfileId)
  log $(basename "$0") "update_present_backhaul_v6" $LINENO ":profile_idx: $profile_idx"

  if [[ "$backhaulname" =~ ^rmnet.* ]]; then
    input_profile=${backhaul#rmnet}
    if [ $input_profile != $defaultProfileId ]; then
      log $(basename "$0") "update_present_backhaul_v6" $LINENO " no need backhaul-switch for ondemaond PDP $input_profile !"
      return
    else
      backhaulname="wan"
    fi
  fi

  #remove default route when link down
  if [ "$linkstate" == "0" ]; then
    util_remove_mtu_option_in_firewall "6" $defaultProfileId
    get_backhaul_file $iptype $backhaulname
    update_v6_mwan3_default_route $backhaulname $linkstate $file
  fi

  #run loop to get bh_present
  if [ ! -z "$backhaullist" ]; then
    #IPv6 backhaul
    current_backhaul6=$(uci get qcmap_lan.@profile[$profile_idx].bh_present_v6)
    log $(basename "$0") "update_present_backhaul_v6" $LINENO "current_IPv6_backhaul:$current_backhaul6"

    #initialize the bh_present be empty.
    util_execute_uci_set qcmap_lan.@profile[$profile_idx].bh_present_v6 ""

    for backhaulmember6 in $backhaullist
    do
      backhaul6=$(uci get mwan3.$backhaulmember6.interface)
      get_backhaul_file $iptype $backhaul6
      if [ ! -f "$file" ]; then
        log $(basename "$0") "update_present_backhaul_v6"  $LINENO ":Backhaul not up since $file not present"
        continue
      fi
      log $(basename "$0") "update_present_backhaul_v6" $LINENO ":file present: $file"

      . $file

      checkIP=`ip -6 addr show $IFNAME | grep $PUBLIC_IP6`

      if [ ! -n "$checkIP" ]; then
        log $(basename "$0") "update_present_backhaul_v6" $LINENO ":Backhaul not up since addr $PUBLIC_IP6 not valid"
        continue
      fi

      log $(basename "$0") "update_present_backhaul_v6" $LINENO ":$backhaul6 v6 is up with devname=$IFNAME, addr=$PUBLIC_IP6"

      if [ -z "$new_backhaul6" ] ; then
        new_backhaul6=$backhaul6
        new_backhaul6_file=$file

        if [ -n "$current_backhaul6" -a "$current_backhaul6" != "$backhaul6" ]; then
          log $(basename "$0") "update_present_backhaul_v6" $LINENO "previous/current_backhaul6:$current_backhaul6 new_bh_v6=$backhaul6"
          util_execute_uci_set qcmap_lan.@profile[$profile_idx].bh_previous_v6 "$current_backhaul6"
        fi
        util_execute_uci_set qcmap_lan.@profile[$profile_idx].bh_present_v6 "$backhaul6"
        util_execute_uci commit qcmap_lan
      fi
    done

      if [ -f $new_backhaul6_file ]; then
        unset IPV6MTU
        . $new_backhaul6_file
        if [ ! -z $IPV6MTU ] && [ -n $IPV6MTU ]; then
          util_add_mtu_option_in_firewall "6" "$IFNAME" "$IPV6MTU" $defaultProfileId
        fi
      fi
    #combine_backhaul_ipv4ipv6 $profile_idx
    run_combine_backhaul_with_lock $profile_idx

    if [ -z $current_backhaul6 ] || [ "$current_backhaul6" != "$new_backhaul6" ]; then
      #send sig to QCMAP for sending backhaul status indication
      echo "$new_backhaul6" > /var/run/data/backhaul/current_backhaul
    fi

    if [ -n $default_ethwan_profile ]; then
      util_execute_uci_set qcmap_lan.profile_$default_ethwan_profile.bh_present_v6 ""

      # save deafult waneth backaul status to wan profile
      if [ "$new_backhaul6" != "wan" ]; then
        ipv4_bh=$(util_execute_uci get qcmap_lan.@profile[$profile_idx].bh_present)
        if [ "$ipv4_bh" != "wan" ]; then
          util_execute_uci_set qcmap_lan.profile_$default_ethwan_profile.bh_present "$ipv4_bh"
        fi
        util_execute_uci_set qcmap_lan.profile_$default_ethwan_profile.bh_present_v6 "$new_backhaul6"
      fi
    fi
  else
    log $(basename "$0") "update_present_backhaul_v6" $LINENO ":get backhaul_pref.use_member failed:$backhaullist"
  fi
}

#Get preset backhaul and set flag according to priority
function read_current_backhaul() {
  defaultProfileId=$(util_execute_uci get qcmap_lan.@no_of_configs[0].default_pdn)
  no_of_profiles=$(util_execute_uci get qcmap_lan.@no_of_configs[0].no_of_profiles)

  for i in $(seq 0 $((no_of_profiles-1)))
  do
    profile_id=$(util_execute_uci get qcmap_lan.@profile[$i].profile_id)

    #check if profile_id is same as profile
    if [ $profile_id -eq $defaultProfileId ]; then
      profile_idx=$i
      break
    fi
  done

  ipv4_backhaul=$(util_execute_uci get qcmap_lan.@profile[$profile_idx].bh_present)
  ipv6_backhaul=$(util_execute_uci get qcmap_lan.@profile[$profile_idx].bh_present_v6)

  echo "$ipv4_backhaul $ipv6_backhaul"
}


#enable or disable dhcp relay for ap sta bridge mode
function update_dhcp_relay_for_bridge_mode() {

  local wlan_mode=$(util_execute_uci get qcmap_wlan.@wlanconfig[0].mode)
  local bridge_mode=$(util_execute_uci get qcmap_wlan.@stamodeconfig[0].bridge_mode)
  local sta_mode=$(util_execute_uci get qcmap_wlan.@stamodeconfig[0].sta_mode_conn_type)
  local ipv4_bh=$(util_get_bh_present)

  if [ $sta_mode == $QCMAP_STA_CONNECTION_DYNAMIC ] ; then

    file=/tmp/ipv4config.lan
    if [ -f "$file" ] && [ $wlan_mode -eq $QCMAP_WLAN_MODE_AP_STA -o $wlan_mode -eq $QCMAP_WLAN_MODE_AP_AP_STA ] && [ $bridge_mode == 1 ] && [ "$ipv4_bh" == "wanwlan" ]; then
      /etc/data/dhcpRelay.sh enable_dhcp_proxy lan
    else
      /etc/data/dhcpRelay.sh disable_dhcp_proxy
    fi
  fi
}

#Loop vlan, call switch_bh if backhaul present
function qcmap_reload_lan_backhaul()
{
  local no_of_vlans
  local ipv4_backhaul ipv6_backhaul
  local temp_ipv4_bh temp_ipv6_bh

  log $(basename "$0") "qcmap_reload_lan_backhaul" $LINENO "Enter qcmap_reload_lan_backhaul"

  #check if vlan is exist
  no_of_vlans=$(util_execute_uci get qcmap_lan.@no_of_configs[0].no_of_vlans)
  if [ $no_of_vlans -eq 0 ]; then
    log $(basename "$0") "qcmap_reload_lan_backhaul" $LINENO "No VLAN is configured"
    return
  fi

  #loop through the number of vlans
  #if several vlan mapping to one same wan, not need call same switch_bh again
  for i in $(seq $no_of_vlans)
  do
    ipv4_backhaul=$(util_execute_uci -q get qcmap_lan.@vlan[$((i-1))].bh_present)
    ipv6_backhaul=$(util_execute_uci -q get qcmap_lan.@vlan[$((i-1))].bh_present_v6)

    if is_default_backhaul $ipv4_backhaul; then
      log $(basename "$0") "qcmap_reload_lan_backhaul" $LINENO "$ipv4_backhaul is default"
    elif [ -n "$ipv4_backhaul" -a "$ipv4_backhaul" != "$temp_ipv4_bh" ]; then
      switch_backhaul $ipv4_backhaul 1
      temp_ipv4_bh=$(util_execute_uci -q get qcmap_lan.@vlan[$((i-1))].bh_present)
      log $(basename "$0") "qcmap_reload_lan_backhaul" $LINENO "$ipv4_backhaul is not default"
    fi

    if is_default_backhaul $ipv6_backhaul; then
      log $(basename "$0") "qcmap_reload_lan_backhaul" $LINENO "$ipv6_backhaul is default"
    elif [ -n "$ipv6_backhaul" -a "$ipv6_backhaul" != "$temp_ipv6_bh" ]; then
      switch_backhaul_v6 $ipv6_backhaul 1
      temp_ipv6_bh=$(util_execute_uci -q get qcmap_lan.@vlan[$((i-1))].bh_present_v6)
      log $(basename "$0") "qcmap_reload_lan_backhaul" $LINENO "$ipv6_backhaul is not default"
    fi
  done
}

#$1 is ip version
function delete_conntrack_on_switch_bh()
{
  local ip_version=$1
  profile_idx=$(mbb_util_get_uci_profile_index)
  if [ "$ip_version" = "ipv4" ]; then
     #Get previous BH after updating priority BH
     previous_bh=$(util_execute_uci get qcmap_lan.@profile[$profile_idx].bh_previous)
     #Get present BH after updating priority BH
     new_bh=$(util_execute_uci get qcmap_lan.@profile[$profile_idx].bh_present)
     log  $(basename "$0") "delete_conntrack_on_switch_bh" $LINENO "previous_bh=$previous_bh new_bh=$new_bh"
     local default_ethwan_profile=$(util_execute_uci get qcmap_lan.@lan[0].default_waneth_profile_num)
     local defaultProfileId=`util_execute_uci get qcmap_lan.@no_of_configs[0].default_pdn`
     if [ -n "$previous_bh" -a "$previous_bh" != "$new_bh" ]; then
       if [ -n "$previous_bh" -a "$previous_bh" = "waneth" ]; then
         log  $(basename "$0") "delete_conntrack_on_switch_bh" $LINENO " eth_profile=$default_ethwan_profile wwan_Profile=$defaultProfileId deleting conntrack for ipv4 waneth"
         util_del_conntrack $default_ethwan_profile "ipv4" 1
       else
         log  $(basename "$0") "delete_conntrack_on_switch_bh" $LINENO " eth_profile=$default_ethwan_profile wwan_Profile=$defaultProfileId deleting conntrack for ipv4 wwan"
         util_del_conntrack $defaultProfileId "ipv4" 1
       fi
     fi
  else
     #Get previous BH after updating priority BH
     previous_bh_v6=$(util_execute_uci get qcmap_lan.@profile[$profile_idx].bh_previous_v6)
     #Get present BH after updating priority BH
     new_bh_v6=$(util_execute_uci get qcmap_lan.@profile[$profile_idx].bh_present_v6)
     log  $(basename "$0") "delete_conntrack_on_switch_bh" $LINENO "previous_bh=$previous_bh_v6 new_bh=$new_bh_v6"
     local default_ethwan_profile=$(util_execute_uci get qcmap_lan.@lan[0].default_waneth_profile_num)
     local defaultProfileId=`util_execute_uci get qcmap_lan.@no_of_configs[0].default_pdn`
     if [  -n "$previous_bh_v6" -a "$previous_bh_v6" != "$new_bh_v6" ]; then
       if [ -n "$previous_bh_v6" -a "$previous_bh_v6" = "waneth" ]; then
         log  $(basename "$0") "delete_conntrack_on_switch_bh" $LINENO " eth_profile=$default_ethwan_profile wwan_Profile=$defaultProfileId deleting conntrack for ipv6 waneth"
         util_del_conntrack $default_ethwan_profile "ipv6" 1
       else
         log  $(basename "$0") "delete_conntrack_on_switch_bh" $LINENO " eth_profile=$default_ethwan_profile wwan_Profile=$defaultProfileId deleting conntrack for ipv6 waneth"
         util_del_conntrack $defaultProfileId "ipv6" 1
       fi
     fi
  fi
}

#$1 is ip version
function delete_conntrack_on_switch_bh_on_demand()
{
  local ip_version=$1
  local no_of_vlans=$(util_execute_uci get qcmap_lan.@no_of_configs[0].no_of_vlans)
  #check if vlan is exist
  if [ $no_of_vlans -eq 0 ]; then
    log $(basename "$0") "delete_conntrack_on_switch_bh_on_demand" $LINENO "No VLAN is configured"
    return
  fi

  #loop through the number of vlans
  for vlan_idx in $(seq 0 $((no_of_vlans-1)))
  do
    temp_profile_num=$(util_execute_uci get qcmap_lan.@vlan[$vlan_idx].$profile_num)
    if [ -n "$temp_profile_num" ]; then
      vlan_id=$(util_execute_uci get qcmap_lan.@vlan[$vlan_idx].vlan_id)

      log $(basename "$0") ":delete_conntrack_on_switch_bh_on_demand:" $LINENO "vlan_id:$vlan_id"

      if [ "$ip_version" = "ipv4" ]; then
        new_bh=$(util_execute_uci get qcmap_lan.@vlan[$vlan_idx].bh_present)
        previous_bh=$(util_execute_uci get qcmap_lan.@vlan[$vlan_idx].bh_previous)
        log  $(basename "$0") "delete_conntrack_on_switch_bh" $LINENO "ip_version=$ip_version previous_bh=$previous_bh new_bh=$new_bh"

        if [ -n "$previous_bh" -a "$previous_bh" != "$new_bh" ]; then
          if [[ "$previous_bh" =~ ^waneth.* ]]; then
            profile_id=${previous_bh#waneth}
          elif [[ "$previous_bh" =~ ^wan.* ]]; then
            profile_id=${previous_bh#wan}
          fi
          log $(basename "$0") ":delete_conntrack_on_switch_bh_on_demand:" $LINENO "profile_id:$profile_id"
          util_del_conntrack $profile_id "ipv4" 1
        fi
      else
        new_bh_v6=$(util_execute_uci get qcmap_lan.@vlan[$vlan_idx].bh_present_v6)
        previous_bh_v6=$(util_execute_uci get qcmap_lan.@vlan[$vlan_idx].bh_previous_v6)
        log  $(basename "$0") "delete_conntrack_on_switch_bh_on_demand" $LINENO "ip_version=$ip_version previous_bh=$previous_bh_v6 new_bh=$new_bh_v6"

        if [ -n "$previous_bh_v6" -a "$previous_bh_v6" != "$new_bh_v6" ]; then
          if [[ "$previous_bh_v6" =~ ^waneth.* ]]; then
            profile_id_v6=${previous_bh_v6#waneth}
          elif [[ "$previous_bh_v6" =~ ^wan.* ]]; then
            profile_id_v6=${previous_bh_v6#wan}
          fi
          log $(basename "$0") ":delete_conntrack_on_switch_bh_on_demand:" $LINENO "profile_id_v6:$profile_id_v6"
          util_del_conntrack ${profile_id_v6} "ipv6" 1
        fi
      fi
    fi

  done

}

#Set backhaul prefrence in mwan3 config.
#set_backhaul_pref <wan_pref1> <wan_pref2> ...
#$1 : first BH pref
#$2 : second BH pref
#$3 : third BH pref
#$4 : fourth BH pref
function set_backhaul_pref() {
  local wan1=$1
  local wan2=$2
  local wan3=$3
  local wan4=$4
  local wan5=$5

  w1=$(get_wan_type $wan1)
  w2=$(get_wan_type $wan2)
  w3=$(get_wan_type $wan3)
  w4=$(get_wan_type $wan4)
  w5=$(get_wan_type $wan5)

  uci delete mwan3.backhaul_pref 2>/dev/null
  uci delete mwan3.backhaul_pref6 2> /dev/null

  uci set mwan3.backhaul_pref=policy
  uci set mwan3.backhaul_pref6=policy

  i=1
  for wan_type in $w1 $w2 $w3 $w4 $w5
    do
      uci set mwan3.m_$wan_type.metric="$i"
      uci set mwan3.m_$wan_type.weight="$((i+1))"

      wan6_type=$wan_type"_v6"
      uci set mwan3.m_$wan6_type.metric="$i"
      uci set mwan3.m_$wan6_type.weight="$((i+1))"


      #create policy for backhaul pref
      uci add_list mwan3.backhaul_pref.use_member="m_$wan_type"
      uci add_list mwan3.backhaul_pref6.use_member="m_$wan6_type"

      i=$((i+1))
    done
  log $(basename "$0") "set_backhaul_pref" $LINENO ": set_backhaul_pref success "
  echo "set_backhaul_pref success"

  uci commit mwan3
  qcmap_reload_mwan3

  qcmap_reload_lan_backhaul
  delete_conntrack_on_switch_bh "ipv4"
  delete_conntrack_on_switch_bh "ipv6"

  delete_conntrack_on_switch_bh_on_demand "ipv4"
  delete_conntrack_on_switch_bh_on_demand "ipv6"

  log $(basename "$0") "set_backhaul_pref" $LINENO ": End set_backhaul_pref "
}

# disable ipv4 state for wlan/eth/bt
function disable_ipv4() {
  local proto=$(util_execute_uci get network.waneth.proto)

  [ -f /tmp/ipv4config.waneth ] && . /tmp/ipv4config.waneth
  ip_eth=$PUBLIC_IP

  [ -f /tmp/ipv4config.wanwlan ] && . /tmp/ipv4config.wanwlan
  ip_wlan=$PUBLIC_IP

  [ -f /tmp/ipv4config.wanbt ] && . /tmp/ipv4config.wanbt
  ip_bt=$PUBLIC_IP

  #Install drop all rule for disable IPv4
  delete_drop_all_rule_for_disabled_ipv4

  install_drop_all_rule_for_disabled_ipv4

  if [ "$proto" == "pppoe" ]; then
    util_execute_uci_set "network.waneth.pppd_options" "noip"
  else
    util_execute_uci_set "network.waneth.proto" "none"
  fi

  util_execute_uci_set "network.wanwlan.proto" "none"
  util_execute_uci_set "network.wanbt.proto" "none"

  util_execute_uci commit network
  ubus call network reload

  #Clear all IPv4 conntrack rules for active backhauls
  util_del_conntrack_for_disabled_ipv4 "$ip_eth"
  util_del_conntrack_for_disabled_ipv4 "$ip_wlan"
  util_del_conntrack_for_disabled_ipv4 "$ip_bt"

  log  $(basename "$0") "disable_ipv4" $LINENO "disabled IPV4."
}

# disable ipv6 state for wlan/eth/bt
function disable_ipv6() {
  local profile=$(util_execute_uci get qcmap_lan.@lan[0].default_waneth_profile_num)
  Default_PREFIX6="::/0"

  [ -f /tmp/ipv6config.waneth ] && . /tmp/ipv6config.waneth
  ipv6_eth_prefix=$PREFIX6
  ipv6_eth=$PUBLIC_IP6
  ipv6_eth_prefix_len=$NETMASK6

  if [[ "$ipv6_eth_prefix" == "$Default_PREFIX6" || $profile -gt 100 ]]; then
    log $(basename "$0") "disable_ipv6" $LINENO "this is waneth profile, no PREFIX6 in $v6_config_file"
    netaddr6=$(util_get_IPv6_Prefix $ipv6_eth $ipv6_eth_prefix_len)
    ipv6_eth_prefix="$netaddr6::/$ipv6_eth_prefix_len"
  fi

  [ -f /tmp/ipv6config.wanwlan ] && . /tmp/ipv6config.wanwlan
  ipv6_wlan_prefix=$PREFIX6
  ipv6_wlan=$PUBLIC_IP6
  ipv6_wlan_prefix_len=$NETMASK6

  if [[ "$ipv6_wlan_prefix" == "$Default_PREFIX6" ]]; then
    log $(basename "$0") "disable_ipv6" $LINENO "this is wanwlan profile, no PREFIX6 in $v6_config_file"
    netaddr6=$(util_get_IPv6_Prefix $ipv6_wlan $ipv6_wlan_prefix_len)
    ipv6_wlan_prefix="$netaddr6::/$ipv6_wlan_prefix_len"
  fi

  [ -f /tmp/ipv6config.wanbt ] && . /tmp/ipv6config.wanbt
  ipv6_bt_prefix=$PREFIX6

  #Install drop all rule for disable IPv6
  delete_drop_all_rule_for_disabled_ipv6

  install_drop_all_rule_for_disabled_ipv6

  util_execute_uci_set "network.wanwlan_v6.proto" "none"
  util_execute_uci_set "network.wanbt_v6.proto" "none"
  util_execute_uci_set "network.waneth_v6.proto" "none"

  util_execute_uci commit network
  ubus call network reload

  util_del_conntrack_for_disabled_v6 "$ipv6_eth_prefix"
  util_del_conntrack_for_disabled_v6 "$ipv6_wlan_prefix"
  util_del_conntrack_for_disabled_v6 "$ipv6_bt_prefix"

  log  $(basename "$0") "disable_ipv6" $LINENO "disabled IPV6."
}

# enable ipv4 state for wlan/eth/bt
function enable_ipv4() {
  local proto=$(util_execute_uci get network.waneth.proto)
  local pppd_options=$(util_execute_uci get network.waneth.pppd_options)

  if [ "$proto" == "pppoe" -a "$pppd_options" == "noip" ]; then
    util_execute_uci del network.waneth.pppd_options
  else
    util_execute_uci_set "network.waneth.proto" "dhcp"
  fi

  util_execute_uci_set "network.wanwlan.proto" "dhcp"
  util_execute_uci_set "network.wanbt.proto" "dhcp"

  #Delete drop all rule for disable IPv4
  delete_drop_all_rule_for_disabled_ipv4

  util_execute_uci commit network
  ubus call network reload

  log  $(basename "$0") "enable_ipv4" $LINENO "enabled IPV4."
}

# enable ipv6 state for wlan/eth/bt
function enable_ipv6() {
  util_execute_uci_set "network.wanwlan_v6.proto" "dhcpv6"
  util_execute_uci_set "network.wanbt_v6.proto" "dhcpv6"
  util_execute_uci_set "network.waneth_v6.proto" "dhcpv6"

  #Delete drop all rule for disable IPv6
  delete_drop_all_rule_for_disabled_ipv6

  util_execute_uci commit network
  ubus call network reload

  log  $(basename "$0") "enable_ipv6" $LINENO "enabled IPV6."
}

# enable/disable load balance feature will call this function
# ETH up/down will call this function
function set_ipa_load_balance() {
  get_bh_load_balance_status

  local sw_path_wan=$(util_execute_uci get qcmap_lan.@sw_path_enabled[0].waneth)

  if [ $sw_path_wan -eq 1 ]; then
    log $(basename "$0") "set_ipa_load_balance" $LINENO "SW path preference is set. "
    return
  fi

  if [ "$qcmap_bh_load_balance_enabled" == "$QCMAP_BH_LOAD_BALANCE_ENABLE" ]; then
    ipa -X dualbackhaul enable
    log $(basename "$0") "set_ipa_load_balance" $LINENO "enable ipa dualbackhaul cmd: ipa -X dualbackhaul enable"
  elif [ "$qcmap_bh_load_balance_enabled" == "$QCMAP_BH_LOAD_BALANCE_DISABLE" ]; then
    ipa -X dualbackhaul disable
    log $(basename "$0") "set_ipa_load_balance" $LINENO "disable ipa dualbackhaul cmd: ipa -X dualbackhaul disable"
  fi

  log $(basename "$0") "set_ipa_load_balance" $LINENO " End!"
}

#Set backhaul load balance in mwan3 config
#Only Support Wan and Ethernet
function set_bh_load_balance() {
  local status=$1
  local wan_weight=$2
  local waneth_weight=$3
  local metric_default=1

  if [ "$status" == "" ]; then
    log $(basename "$0") "set_bh_load_balance" $LINENO "input status error"
    return
  fi

  if [ "$status" == "$QCMAP_BH_LOAD_BALANCE_ENABLE" ] && [ "$wan_weight" != "" ] && [ "$waneth_weight" != "" ]; then
    util_execute_uci delete mwan3.backhaul_pref 2>/dev/null
    util_execute_uci set mwan3.backhaul_pref=policy

    #WWAN
    util_execute_uci set mwan3.m_wan.metric=$metric_default
    util_execute_uci set mwan3.m_wan.weight=$wan_weight

    #Ethernet
    util_execute_uci set mwan3.m_waneth.metric=$metric_default
    util_execute_uci set mwan3.m_waneth.weight=$waneth_weight

    #create policy for backhaul
    #add wan member first, then router main table defalut router interface will be wan
    util_execute_uci add_list mwan3.backhaul_pref.use_member="m_wan"
    util_execute_uci add_list mwan3.backhaul_pref.use_member="m_waneth"

    util_execute_uci set mwan3.https=rule
    util_execute_uci set mwan3.https.sticky='1'
    util_execute_uci set mwan3.https.dest_port='443'
    util_execute_uci set mwan3.https.proto='tcp'
    util_execute_uci set mwan3.https.use_policy='backhaul_pref'

    util_execute_uci commit mwan3
    #/etc/init.d/mwan3 reload
    qcmap_reload_mwan3

    # if wan and waneth bakchaul up first, then enable load balance
    # call add_second_default_route_for_load_balance to add router
    # call update_firewall_zone_for_load_balance to update firewall zone
    add_second_default_route_for_load_balance
    update_firewall_zone_for_load_balance

    log $(basename "$0") "set_bh_load_balance" $LINENO ": enable load balance success "
    echo "enable load balance success"
  elif [ "$status" == "$QCMAP_BH_LOAD_BALANCE_DISABLE" ] ; then
    get_bh_load_balance_status

    if [ "$qcmap_bh_load_balance_enabled" == "$QCMAP_BH_LOAD_BALANCE_DISABLE" ]; then
      log $(basename "$0") "set_bh_load_balance" $LINENO "load balance already disabled"
      return
    fi

    # set backhaul priority to default
    set_backhaul_pref 1 2 3 4 5

    util_execute_uci -q delete mwan3.https 2>/dev/null

    log $(basename "$0") "set_bh_load_balance" $LINENO ": disable load balance success "
    echo "disable load balance success"
  fi

  #call ipa dualbackhaul command
  set_ipa_load_balance
}

#Function to enable or disable IPA pipes for ETH peripheral.
function set_ipa_offload_config()
{
  local sw_path_wan
  local sw_path_lan
  local sw_path_wan_name
  local sw_path_lan_name
  local no_of_configs=$(uci get qcmap_lan.@sw_path_enabled[0].no_of_configs)

  local event=$(uci get qcmap_lan.@no_of_configs[0].ioss_event)

  if [ $event -eq 0 ]; then
    log $(basename "$0") "set_ipa_offload_config" $LINENO ": IOSS event not received yet. "
    return
  fi

  sw_path_wan=$(util_execute_uci get qcmap_lan.@sw_path_enabled[0].waneth)
  sw_path_lan=$(util_execute_uci get qcmap_lan.@sw_path_enabled[0].laneth)
  sw_path_lan_name=$(util_execute_uci get qcmap_lan.@sw_path_enabled[0].laneth_name)

  if [ $sw_path_wan -eq 1 ]; then
    sw_path_wan_name=$(util_execute_uci get qcmap_lan.@sw_path_enabled[0].waneth_name)
  elif [ $sw_path_wan -eq 0 -a  $sw_path_lan -eq 0 ]; then
    sw_path_wan_name=$(util_execute_uci get qcmap_lan.@sw_path_enabled[0].waneth_name)
    if [ $no_of_configs -eq 0 ]; then
      eth-cfg configure $sw_path_wan_name --ipa-offload enable
      eth-cfg configure $sw_path_lan_name --ipa-offload enable
      log $(basename "$0") "set_ipa_offload_config" $LINENO ": Disabling SW path feature. "
    fi
    return
  else
    log $(basename "$0") "set_ipa_offload_config" $LINENO ": SW path WAN is not configured. "
    return
  fi

  eth-cfg configure $sw_path_wan_name --ipa-offload disable
  eth-cfg configure $sw_path_lan_name --ipa-offload disable

  log $(basename "$0") "set_ipa_offload_config" $LINENO ": SW path WAN/LAN is configured successfully. "
  return
}

function add_br_ethwan_iface() {
  local ethwan_iface=$1
  local ifname=$2
  local profile=$3
  local proto=$4
  local br_ethwan_device
  log $(basename "$0") "add_br_ethwan_iface" $LINENO "$proto"

  # add br-ethwan network config
  util_execute_uci set network.ethwan$profile=interface
  util_execute_uci set network.ethwan$profile.device="br-ethwan$profile"
  util_execute_uci set network.ethwan$profile.netmask='255.255.255.0'
  util_execute_uci set network.ethwan$profile.type='bridge'
  util_execute_uci set network.ethwan$profile.ieee1905managed='1'
  util_execute_uci set network.ethwan$profile.defaultroute='0'
  util_execute_uci set network.ethwan$profile.ifname=$ifname
  if [ "$proto" == "dhcp" ]; then
    util_execute_uci set network.ethwan$profile.proto="dhcp"
  elif [ "$proto" == "pppoe" ]; then
    util_execute_uci set network.ethwan$profile.proto="static"
    util_execute_uci set network.ethwan$profile.ipaddr="169.254.$profile.1"
  fi

  # modify ethwan_iface.ifname
  util_execute_uci set network.$ethwan_iface.device="br-ethwan$profile"
  util_execute_uci commit network
}

# Add required configs when ethwan profile is created
# <profile-id>, <iface_type>, <vlanid>, <proto>, <ip_type>, <isDefault>, <username>, <password>
# for pppoe/dhcp, iface is always eth1
function create_wan_profile() {
  local profile=$1
  local iface_type=$2
  local vlan_id=$3
  local proto=$4
  local ip_type=$5
  local is_default=$6
  local username=$7
  local password=$8
  local iface=$(util_get_interface_name $iface_type)
  local ethwan_iface ethwan_zone_name
  local profile_id temp_vlan_id
  local idx idx_dhcp idx_pppoe
  local device=$iface
  local ifname=$iface
  local no_of_wan_profiles
  local default_profile=$(util_execute_uci get qcmap_lan.@lan[0].default_waneth_profile_num)
  local brwan_enabled=$(util_execute_uci get qcmap_lan.@global[0].brwan)

  #proto should be dhcp or pppoe
  if [ "$proto" != "dhcp" -a "$proto" != "pppoe" ]; then
    log $(basename "$0") "create_wan_profile" $LINENO "proto should be dhcp or pppoe"
    return
  fi

  #first profile should set to default even is_default is 0
  if [ -z "$default_profile" ]; then
    log $(basename "$0") "create_wan_profile" $LINENO "first profile should set to default"
    is_default=1
  fi

  #checking if profile_id is between 100 and 200
  if [ $profile -le 100 -o $profile -ge 200 ]; then
    log $(basename "$0") "create_wan_profile" $LINENO "profile_id should between 100 and 200"
    return
  fi

  #checking if default profile already exist, then we need create no-default profile
  if [ -n "$default_profile" -a $is_default -eq 1 ]; then
    log $(basename "$0") "create_wan_profile" $LINENO "default profile exist, create no-default profile"
    is_default=0
  fi

  #checking if profile_id exist previously
  profile_id=$(util_execute_uci get qcmap_lan.profile_$profile.profile_id)
  if [ -n "$profile_id" ]; then
   log $(basename "$0") "create_wan_profile" $LINENO "profile already exisit "
   return
  fi

  #checking if vlan_id exist previously
  no_of_wan_profiles=$(util_execute_uci get qcmap_lan.@no_of_configs[0].no_of_profiles)
  for i in $(seq $no_of_wan_profiles)
  do
    temp_vlan_id=$(util_execute_uci get qcmap_lan.@profile[$((i-1))].vlan_id)
    if [ -n "$temp_vlan_id" -a $temp_vlan_id -eq $vlan_id ]; then
      log $(basename "$0") "create_wan_profile" $LINENO "vlan_id already exisit "
      return
    fi
  done

  # check pppoe/dhcp profile number, support max 8 pppoe + 1 dhcp
  idx_dhcp=$(util_execute_uci get qcmap_lan.@no_of_configs[0].no_of_ethwan_dhcp_profiles)
  idx_pppoe=$(util_execute_uci get qcmap_lan.@no_of_configs[0].no_of_ethwan_pppoe_profiles)
  if [ "$proto" == "dhcp" -a $idx_dhcp -lt 1 ]; then
    util_execute_uci set qcmap_lan.@no_of_configs[0].no_of_ethwan_dhcp_profiles=$((idx_dhcp+1))
  elif [ "$proto" == "pppoe" -a $idx_pppoe -lt 8 ]; then
    idx_pppoe=$((idx_pppoe+1))
    util_execute_uci set qcmap_lan.@no_of_configs[0].no_of_ethwan_pppoe_profiles=${idx_pppoe}
  else
    log $(basename "$0") "create_wan_profile" $LINENO "support max 8 pppoe + 1 dhcp"
    return
  fi

  if [ $is_default -eq 1 ]; then
    ethwan_iface="waneth"
    util_execute_uci set qcmap_lan.@lan[0].default_waneth_profile_num=$profile
  else
    ethwan_iface="waneth${profile}"
    ethwan_zone_name="wan${profile}"

    util_execute_uci set dhcp.${ethwan_iface}=dhcp
    util_execute_uci set dhcp.${ethwan_iface}.interface=$ethwan_iface
    util_execute_uci set dhcp.${ethwan_iface}.ignore='1'
    util_execute_uci set dhcp.${ethwan_iface}_v6=dhcp
    util_execute_uci set dhcp.${ethwan_iface}_v6.interface=${ethwan_iface}_v6
  fi
  if [ $vlan_id -gt 0 ]; then
    device="$iface.$vlan_id"
    ifname="$iface.$vlan_id"
  fi

  # add network config
  util_execute_uci set network.$ethwan_iface=interface
  util_execute_uci set network.$ethwan_iface.proto=$proto
  util_execute_uci set network.$ethwan_iface.profile=$profile
  util_execute_uci set network.$ethwan_iface.defaultroute='0'
  util_execute_uci set network.$ethwan_iface.disabled='0'
  util_execute_uci set network.$ethwan_iface.device=$device
  util_execute_uci set network.$ethwan_iface.ipv6='1'
  util_execute_uci set network.${ethwan_iface}_v6=interface
  util_execute_uci set network.${ethwan_iface}_v6.proto='dhcpv6'
  util_execute_uci set network.${ethwan_iface}_v6.profile=$profile
  util_execute_uci set network.${ethwan_iface}_v6.ipv6='1'
  util_execute_uci set network.${ethwan_iface}_v6.type='internet'
  util_execute_uci set network.${ethwan_iface}_v6.defaultroute='0'
  util_execute_uci set network.${ethwan_iface}_v6.lockmtu='1'
  util_execute_uci set network.${ethwan_iface}_v6.disabled='0'

  if [ "$proto" == "dhcp" ]; then
    util_execute_uci del network.${ethwan_iface}_v6.ifname
    util_execute_uci set network.${ethwan_iface}_v6.device=$device
  elif [ "$proto" == "pppoe" ]; then
    util_execute_uci del_list network.waneth_v6.downstream='lan'
    util_execute_uci del network.${ethwan_iface}_v6.device
    util_execute_uci_set network.$ethwan_iface.keepalive '10 2'
    util_execute_uci set network.$ethwan_iface.username=$username
    util_execute_uci set network.$ethwan_iface.password=$password
    util_execute_uci set network.${ethwan_iface}_v6.ifname="@${ethwan_iface}"
    util_execute_uci set network.${ethwan_iface}_v6.reqaddress='try'

    #update to ipa
    if [ $idx_pppoe -eq 1 ]; then
      ipa pppoe category wan $iface enable > /dev/null
      log $(basename "$0") "create_wan_profile" $LINENO "ipa cmd: ipa pppoe category wan $iface enable"
    fi
  fi

  # disable default ipv4/ipv6 if ip_type not ipv4v6(10)
  if [ $is_default -eq 1 -a $ip_type -eq 4 ]; then
    util_execute_uci_set "network.waneth_v6.proto" "none"
  elif [ $is_default -eq 1 -a $ip_type -eq 6 ]; then
    if [ "$proto" == "pppoe" ]; then
      util_execute_uci_set "network.waneth.pppd_options" "noip"
    else
      util_execute_uci_set "network.waneth.proto" "none"
    fi
  fi

  # add profile
  util_execute_uci set qcmap_lan.profile_$profile=profile
  util_execute_uci set qcmap_lan.profile_$profile.profile_id=$profile
  util_execute_uci set qcmap_lan.profile_$profile.device=$iface
  util_execute_uci set qcmap_lan.profile_$profile.vlan_id=$vlan_id
  util_execute_uci set qcmap_lan.profile_$profile.proto=$proto
  util_execute_uci set qcmap_lan.profile_$profile.ip_type=$ip_type
  util_execute_uci set qcmap_lan.profile_$profile.firewall_enabled=0
  util_execute_uci set qcmap_lan.profile_$profile.pkts_allowed=0
  util_execute_uci set qcmap_lan.profile_$profile.no_of_snat_rules=0
  util_execute_uci set qcmap_lan.profile_$profile.no_of_v4_fw_rules=0
  util_execute_uci set qcmap_lan.profile_$profile.no_of_v6_fw_rules=0
  util_execute_uci set qcmap_lan.@no_of_configs[0].no_of_profiles=$((no_of_wan_profiles+1))
  if [ $is_default -eq 1 ]; then
    util_execute_uci del_list qcmap_lan.profile_$profile.vlan_ids="0"
    util_execute_uci add_list qcmap_lan.profile_$profile.vlan_ids="0"
    local default_wwan_profile_index=$(util_get_default_profile_index)
    local enable_fw_flag_wwan_default=$(util_execute_uci get qcmap_lan.@profile[$default_wwan_profile_index].firewall_enabled)
    local pkt_allowed_flag_wwan_default=$(util_execute_uci get qcmap_lan.@profile[$default_wwan_profile_index].pkts_allowed)
    util_execute_uci set qcmap_lan.profile_$profile.firewall_enabled=$enable_fw_flag_wwan_default
    util_execute_uci set qcmap_lan.profile_$profile.pkts_allowed=$pkt_allowed_flag_wwan_default
  fi

  # add firewall zone for on-demand profile
  if [ $is_default -ne 1 ]; then
    util_execute_uci add firewall zone
    util_execute_uci set firewall.@zone[-1].name=$ethwan_zone_name
    util_execute_uci add_list firewall.@zone[-1].network=$ethwan_iface
    util_execute_uci add_list firewall.@zone[-1].network="${ethwan_iface}_v6"
    util_execute_uci set firewall.@zone[-1].input='REJECT'
    util_execute_uci set firewall.@zone[-1].output='ACCEPT'
    util_execute_uci set firewall.@zone[-1].forward='REJECT'
    util_execute_uci set firewall.@zone[-1].masq='1'
    util_execute_uci set firewall.@zone[-1].masq_random='1'

    #whenever a new wan profile is created we will create a corresponding lan zone
    util_execute_uci add firewall zone
    util_execute_uci set firewall.@zone[-1].name=lan_${ethwan_zone_name}
    util_execute_uci set firewall.@zone[-1].input='ACCEPT'
    util_execute_uci set firewall.@zone[-1].output='ACCEPT'
    util_execute_uci set firewall.@zone[-1].forward='ACCEPT'

    #Create forwarding rule for wan profile
    util_execute_uci add firewall forwarding
    util_execute_uci set firewall.@forwarding[-1].src=lan_${ethwan_zone_name}
    util_execute_uci set firewall.@forwarding[-1].dest=${ethwan_zone_name}

    util_execute_uci commit firewall
    /etc/init.d/firewall reload
  fi

  # add br-ethwan iface
  if [ "$brwan_enabled" == "1" ]; then
    add_br_ethwan_iface $ethwan_iface $ifname $profile $proto
  fi

  util_execute_uci commit qcmap_lan
  util_execute_uci commit network
  util_execute_uci commit dhcp
  ubus call network reload
  /etc/init.d/dnsmasq reload
}

# there is no chance user to set default as non-default
# user can set new default on other files, then old-default atomicly be non-default
# <profile-id>, <iface_type>, <vlanid>, <proto>, <ip_type>, <isDefault>, <username>, <password>
function update_wan_profile() {
  local profile=$1
  local iface_type=$2
  local current_vlan_id=$3
  local current_proto=$4
  local current_ip_type=$5
  local is_default=$6
  local current_username=$7
  local current_password=$8
  local iface=$(util_get_interface_name $iface_type)
  local current_device=$iface

  local idx_dhcp idx_pppoe
  local profile_id temp_vlan_id
  local no_of_wan_profiles
  local ethwan_iface="waneth"
  local found_ethwan_iface current_ethwan_iface new_ethwan_iface
  local vlan_id proto
  local default_profile=$(util_execute_uci get qcmap_lan.@lan[0].default_waneth_profile_num)
  local default_vlan_id=$(util_execute_uci get qcmap_lan.profile_$default_profile.vlan_id)
  local eth_type
  local brwan_enabled=$(util_execute_uci get qcmap_lan.@global[0].brwan)

  #current_proto should be dhcp or pppoe
  if [ "$current_proto" != "dhcp" -a "$current_proto" != "pppoe" ]; then
    log $(basename "$0") "update_wan_profile" $LINENO "current_proto should be dhcp or pppoe"
    return
  fi

  #Check if profile_id exist
  profile_id=$(util_execute_uci get qcmap_lan.profile_$profile.profile_id)
  if [ -n "$profile_id" ]; then
    vlan_id=$(util_execute_uci get qcmap_lan.profile_$profile.vlan_id)
    proto=$(util_execute_uci get qcmap_lan.profile_$profile.proto)
    log $(basename "$0") "update_wan_profile" $LINENO "profile old option: vlan$vlan_id, $proto"
  else
    log $(basename "$0") "update_wan_profile" $LINENO "profile not found in qcmap_lan"
    return
  fi

  # different profile can not have same vlan_id
  no_of_wan_profiles=$(util_execute_uci get qcmap_lan.@no_of_configs[0].no_of_profiles)
  for i in $(seq $no_of_wan_profiles)
  do
    temp_vlan_id=$(util_execute_uci get qcmap_lan.@profile[$((i-1))].vlan_id)
    if [ -n "$temp_vlan_id" -a $temp_vlan_id -eq $current_vlan_id ] && [ $current_vlan_id -ne $vlan_id ] ; then
      log $(basename "$0") "update_wan_profile" $LINENO "vlan_id already exisit "
      return
    fi
  done

  #can not set default to non-default
  if [ $is_default -eq 0 -a $profile -eq $default_profile ]; then
    log $(basename "$0") "update_wan_profile" $LINENO "can not set default to non-default"
    return
  fi

  # check pppoe/dhcp profile number, support max 8 pppoe + 1 dhcp
  idx_dhcp=$(util_execute_uci get qcmap_lan.@no_of_configs[0].no_of_ethwan_dhcp_profiles)
  idx_pppoe=$(util_execute_uci get qcmap_lan.@no_of_configs[0].no_of_ethwan_pppoe_profiles)
  if [ "$current_proto" != "$proto" ]; then
    if [ "$current_proto" == "dhcp" -a $idx_dhcp -eq 1 ] || [ "$current_proto" == "pppoe" -a $idx_pppoe -eq 8 ]; then
      log $(basename "$0") "update_wan_profile" $LINENO "support max 8 pppoe + 1 dhcp"
      return
    elif [ "$current_proto" == "dhcp" ]; then
      idx_pppoe=$((idx_pppoe-1))
      util_execute_uci set qcmap_lan.@no_of_configs[0].no_of_ethwan_dhcp_profiles=$((idx_dhcp+1))
      util_execute_uci set qcmap_lan.@no_of_configs[0].no_of_ethwan_pppoe_profiles=${idx_pppoe}
    elif [ "$current_proto" == "pppoe" ]; then
      idx_pppoe=$((idx_pppoe+1))
      util_execute_uci set qcmap_lan.@no_of_configs[0].no_of_ethwan_pppoe_profiles=${idx_pppoe}
      util_execute_uci set qcmap_lan.@no_of_configs[0].no_of_ethwan_dhcp_profiles=$((idx_dhcp-1))
    fi
  fi

  if [ $current_vlan_id -eq 0 ]; then
    current_device="$iface"
  elif [ $current_vlan_id -gt 0 ]; then
    current_device="$iface.$current_vlan_id"
  fi

  # is_default=true, different profile
  # profile=101, vlan=5, waneth, waneth_v6   //default
  # profile=102, vlan=6, waneth102, waneth102_v6
  # now, update profile=102, current_vlan_id=8, then:
  # waneth102(found_ethwan_iface)  --> waneth(current_ethwan_iface), waneth102_v6  --> waneth_v6  //default, vlan8
  # eth1.6 --> eth1.8
  # waneth(ethwan_iface)       --> waneth101(new_ethwan_iface), waneth_v6 --> waneth101_v6
  # keep eth1.5
  if [ $is_default -eq 1 -a $profile -ne $default_profile ]; then
    found_ethwan_iface="waneth${profile_id}"
    current_ethwan_iface="waneth"
    ethwan_iface="waneth"
    new_ethwan_iface="waneth${default_profile}"

    util_execute_uci set network.$found_ethwan_iface.disabled="1"
    util_execute_uci set network.${found_ethwan_iface}_v6.disabled="1"
    util_execute_uci set network.$current_ethwan_iface.disabled="1"
    util_execute_uci set network.${current_ethwan_iface}_v6.disabled="1"


    util_execute_uci commit network
    ubus call network reload


    util_execute_uci set qcmap_lan.@lan[0].default_waneth_profile_num=$profile
    util_execute_uci del_list qcmap_lan.profile_$default_profile.vlan_ids="0"
    util_execute_uci add_list qcmap_lan.profile_$profile.vlan_ids="0"
    util_execute_uci commit qcmap_lan

    #update dhcp
    util_execute_uci rename dhcp.${found_ethwan_iface}=$new_ethwan_iface
    util_execute_uci rename dhcp.${found_ethwan_iface}_v6=${new_ethwan_iface}_v6
    util_execute_uci set dhcp.${new_ethwan_iface}.interface="$new_ethwan_iface"
    util_execute_uci set dhcp.${new_ethwan_iface}_v6.interface="${new_ethwan_iface}_v6"

    #update network_iface
    local waneth_defaultroute=$(util_execute_uci get network.waneth.defaultroute)
    util_execute_uci rename network.$found_ethwan_iface="temp_waneth"  # waneth102 --> temp_waneth
    util_execute_uci rename network.${found_ethwan_iface}_v6="temp_waneth_v6"
    util_execute_uci rename network.$ethwan_iface=$new_ethwan_iface    # waneth --> waneth101
    util_execute_uci rename network.${ethwan_iface}_v6=${new_ethwan_iface}_v6
    util_execute_uci rename network.temp_waneth=$current_ethwan_iface  # temp_waneth --> waneth
    util_execute_uci rename network.temp_waneth_v6=${current_ethwan_iface}_v6
    util_execute_uci set network.${current_ethwan_iface}_v6.ifname="@${current_ethwan_iface}"
    util_execute_uci set network.${new_ethwan_iface}_v6.ifname="@${new_ethwan_iface}"

    if [ "$brwan_enabled" == "1" ]; then
      util_execute_uci set network.ethwan$profile.ifname="$current_device"
      util_execute_uci set network.$new_ethwan_iface.device="br-ethwan$default_profile"
      util_execute_uci set network.$current_ethwan_iface.device="br-ethwan$profile"
    else
      util_execute_uci set network.$current_ethwan_iface.device=$current_device
    fi

    util_execute_uci delete qcmap_lan.profile_${new_ethwan_iface}.bh_present
    util_execute_uci delete qcmap_lan.profile_${new_ethwan_iface}.bh_present_v6
    util_execute_uci set network.${new_ethwan_iface}.defaultroute='0'
    util_execute_uci set network.${new_ethwan_iface}.disabled='0'
    util_execute_uci set network.${new_ethwan_iface}_v6.defaultroute='0'
    util_execute_uci set network.${new_ethwan_iface}_v6.disabled='0'
    util_execute_uci set network.waneth.defaultroute=$waneth_defaultroute
    util_execute_uci set network.waneth.disabled='0'
    util_execute_uci set network.waneth_v6.disabled='0'
    util_execute_uci set network.waneth_v6.defaultroute=$waneth_defaultroute

    #update firewall config, default_profile:101, profile_id:102
    update_lan_waneth_interface $default_profile $profile_id

    #update proto, usename, password
    if [ "$current_proto" == "dhcp"  ]; then
      util_execute_uci set network.${current_ethwan_iface}_v6.device=$current_device
      if [ "$current_proto" != "$proto"]; then
        util_execute_uci set network.$current_ethwan_iface.proto=$current_proto
        util_execute_uci del network.$current_ethwan_iface.keepalive
        util_execute_uci del network.$current_ethwan_iface.username
        util_execute_uci del network.$current_ethwan_iface.password
        util_execute_uci del network.${current_ethwan_iface}_v6.ifname
        util_execute_uci del network.${current_ethwan_iface}_v6.reqaddress
        if [ "$brwan_enabled" == "1" ]; then
          util_execute_uci del network.ethwan$profile.ipaddr
          util_execute_uci set network.ethwan$profile.proto="dhcp"
        fi

        #update to ipa
        if [ $idx_pppoe -eq 0 ]; then
          ipa pppoe category wan $iface disable > /dev/null
          log $(basename "$0") "update_wan_profile" $LINENO "ipa cmd: ipa pppoe category wan $iface disable"
        fi
      fi
    elif [ "$current_proto" == "pppoe" ]; then
      util_execute_uci del network.${current_ethwan_iface}_v6.device
      util_execute_uci set network.$current_ethwan_iface.username=$current_username
      util_execute_uci set network.$current_ethwan_iface.password=$current_password
      if [ "$current_proto" != "$proto" ]; then
        util_execute_uci set network.$current_ethwan_iface.proto=$current_proto
        util_execute_uci_set network.$current_ethwan_iface.keepalive '10 2'
        util_execute_uci set network.${current_ethwan_iface}_v6.ifname="@${current_ethwan_iface}"
        util_execute_uci set network.${current_ethwan_iface}_v6.reqaddress='try'
        if [ "$brwan_enabled" == "1" ]; then
          util_execute_uci set network.ethwan$profile.proto="static"
          util_execute_uci set network.ethwan$profile.ipaddr="169.254.$profile.1"
        fi

        #update to ipa
        if [ $idx_pppoe -eq 1 ]; then
          ipa pppoe category wan $iface enable > /dev/null
          log $(basename "$0") "update_wan_profile" $LINENO "ipa cmd: ipa pppoe category wan $iface enable"
        fi
      fi
    fi
  # is_default=true, current profile is same to default profile, update directly
  # is_default=false, update profile directly
  else
    if [ $is_default -eq 1 ]; then
      ethwan_iface="waneth"
    else
      ethwan_iface="waneth${profile_id}"
    fi

    #update common config
    if [ "$brwan_enabled" == "1" ]; then
      util_execute_uci set network.ethwan$profile.ifname="$current_device"
      util_execute_uci set network.$ethwan_iface.device="br-ethwan$profile"
    else
      util_execute_uci set network.$ethwan_iface.device=$current_device
    fi

    if [ "$current_proto" == "dhcp" ]; then
      util_execute_uci set network.${ethwan_iface}_v6.device=$current_device
      if [ "$current_proto" != "$proto" ]; then
        util_execute_uci set network.$ethwan_iface.proto=$current_proto
        util_execute_uci del network.$ethwan_iface.keepalive
        util_execute_uci del network.$ethwan_iface.username
        util_execute_uci del network.$ethwan_iface.password
        util_execute_uci del network.${ethwan_iface}_v6.ifname
        util_execute_uci del network.${ethwan_iface}_v6.reqaddress
        if [ "$brwan_enabled" == "1" ]; then
          util_execute_uci del network.ethwan$profile.ipaddr
          util_execute_uci set network.ethwan$profile.proto="dhcp"
        fi

        #update to ipa
        if [ $idx_pppoe -eq 0 ]; then
          ipa pppoe category wan $iface disable > /dev/null
          log $(basename "$0") "update_wan_profile" $LINENO "ipa cmd: ipa pppoe category wan $iface disable"
        fi
      fi
    elif [ "$current_proto" == "pppoe" ]; then
      util_execute_uci del network.${ethwan_iface}_v6.device
      util_execute_uci set network.$ethwan_iface.username=$current_username
      util_execute_uci set network.$ethwan_iface.password=$current_password
      if [ "$current_proto" != "$proto" ]; then
        util_execute_uci set network.$ethwan_iface.proto=$current_proto
        util_execute_uci_set network.$ethwan_iface.keepalive '10 2'
        util_execute_uci set network.${ethwan_iface}_v6.ifname="@${ethwan_iface}"
        util_execute_uci set network.${ethwan_iface}_v6.reqaddress='try'
        if [ "$brwan_enabled" == "1" ]; then
          util_execute_uci set network.ethwan$profile.proto="static"
          util_execute_uci set network.ethwan$profile.ipaddr="169.254.$profile.1"
        fi

        #update to ipa
        if [ $idx_pppoe -eq 1 ]; then
          ipa pppoe category wan $iface enable > /dev/null
          log $(basename "$0") "update_wan_profile" $LINENO "ipa cmd: ipa pppoe category wan $iface enable"
        fi
      fi
    fi
  fi

  # disable ipv4/ipv6 if ip_type not ipv4v6(10)
  default_profile=$(util_execute_uci get qcmap_lan.@lan[0].default_waneth_profile_num)
  if [ $profile -eq $default_profile  -a $current_ip_type -eq 4 ]; then
    util_execute_uci_set "network.waneth.proto" "$current_proto"
    util_execute_uci_set "network.waneth_v6.proto" "none"
  elif [ $profile -eq $default_profile  -a $current_ip_type -eq 6 ]; then
    if [ "$current_proto" == "pppoe" ]; then
      util_execute_uci_set "network.waneth.pppd_options" "noip"
    else
      util_execute_uci_set "network.waneth.proto" "none"
    fi
    util_execute_uci_set "network.waneth_v6.proto" "dhcpv6"
  fi

  # update profile
  util_execute_uci set qcmap_lan.profile_$profile.vlan_id=$current_vlan_id
  util_execute_uci set qcmap_lan.profile_$profile.proto=$current_proto
  util_execute_uci set qcmap_lan.profile_$profile.ip_type=$current_ip_type

  util_execute_uci commit qcmap_lan
  util_execute_uci commit firewall
  util_execute_uci commit network
  util_execute_uci commit dhcp
  ubus call network reload
  /etc/init.d/dnsmasq reload
  /etc/init.d/firewall reload
}

# Delete required configs when wan profile is deleted
# $1 profile_handle
function delete_wan_profile() {
  local profile=$1
  local ethwan_iface ethwan_zone_name
  local proto
  local ifname br_ethwan_ifname
  local zone_wan_idx zone_lan_wan_idx idx wan_all_idx
  local default_profile=$(util_execute_uci get qcmap_lan.@lan[0].default_waneth_profile_num)
  local no_of_wan_profiles=$(util_execute_uci get qcmap_lan.@no_of_configs[0].no_of_profiles)
  local idx_dhcp=$(util_execute_uci get qcmap_lan.@no_of_configs[0].no_of_ethwan_dhcp_profiles)
  local idx_pppoe=$(util_execute_uci get qcmap_lan.@no_of_configs[0].no_of_ethwan_pppoe_profiles)
  local no_of_ethwan_profiles=$((idx_dhcp+idx_pppoe))
  local brwan_enabled=$(util_execute_uci get qcmap_lan.@global[0].brwan)
  local iface=$(util_execute_uci get qcmap_lan.profile_$profile.device)
  local eth_type

  #Check if profile exist
  proto=$(util_execute_uci get qcmap_lan.profile_$profile.proto)
  if [ -z "$proto" ]; then
    log $(basename "$0") "delete_wan_profile" $LINENO "profile not found in qcmap_lan"
    return
  fi

  # if no-default wan profile exist, should delete them first, then can delete default wan profile
  if [ $profile -eq $default_profile ] && [ $no_of_ethwan_profiles -gt 1 ] ; then
    log $(basename "$0") "delete_wan_profile" $LINENO "can not delete default profile when no-default profile exist"
    return
  fi

  if [ $profile -eq $default_profile ]; then
    ethwan_iface="waneth"
    util_execute_uci del qcmap_lan.@lan[0].default_waneth_profile_num
  else
    ethwan_iface="waneth${profile}"
    ethwan_zone_name="wan${profile}"
    util_execute_uci del dhcp.$ethwan_iface
    util_execute_uci del dhcp.${ethwan_iface}_v6
  fi

  #Delete br-ethwan iface
  device=$(util_execute_uci get network.$ethwan_iface.device)
  if [ "$brwan_enabled" == "1" -a "$device" == "br-ethwan$profile" ]; then
    br_ethwan_ifname=$(util_execute_uci get network.ethwan$profile.ifname)
    util_execute_uci del network.ethwan$profile
    util_execute_uci set network.$ethwan_iface.device=$br_ethwan_ifname
    device=$br_ethwan_ifname
  fi

  #Delete network config, can not delete waneth/waneth_v6
  if [ $profile -ne $default_profile ]; then
    util_execute_uci del network.$ethwan_iface
    util_execute_uci del network.${ethwan_iface}_v6
  else
    util_execute_uci set network.waneth.device=${device:0:4}
    util_execute_uci set network.waneth_v6.device=${device:0:4}
    if [ "$proto" == "pppoe" ]; then
      device=$(util_execute_uci get network.waneth.device)
      util_execute_uci set network.waneth.proto='dhcp'
      util_execute_uci del_list network.waneth_v6.downstream='lan'
      util_execute_uci add_list network.waneth_v6.downstream='lan'

      util_execute_uci del network.waneth.keepalive
      util_execute_uci del network.waneth.username
      util_execute_uci del network.waneth.password
      util_execute_uci del network.waneth.profile
      util_execute_uci del network.waneth_v6.ifname
      util_execute_uci del network.waneth_v6.reqaddress
      util_execute_uci del network.waneth_v6.profile
    fi
  fi

  #Delete profile
  util_execute_uci del qcmap_lan.profile_$profile

  #Delete firewall on-demand wan config.
  zone_wan_idx=$(util_execute_uci show firewall | grep -i "zone" | grep -i "name" | grep -i -w "$ethwan_zone_name" | awk -F'[][]' '{print $2}')
  if [ -n "$zone_wan_idx" ]; then
    util_execute_uci del firewall.@zone[$zone_wan_idx]
    log $(basename "$0") "delete_wan_profile" $LINENO "wan zone deleted from firewall"
  fi

  #Delete firewall on-demand lan_wanethx config.
  zone_lan_wan_idx=$(util_execute_uci show firewall | grep -i "zone" | grep -i "name" | grep -i -w "lan_$ethwan_zone_name" | awk -F'[][]' '{print $2}')
  if [ "$zone_lan_wan_idx" ]; then
    util_execute_uci del firewall.@zone[$zone_lan_wan_idx]
    log $(basename "$0") "delete_wan_profile" $LINENO "lan_wanethx zone deleted from firewall"
  fi

  #Delete forwarding rule for wan profile
  idx=0
  while true
  do
    src=$(util_execute_uci -q get firewall.@forwarding[$idx].src)
    dest=$(util_execute_uci -q get firewall.@forwarding[$idx].dest)
    if [ -z "$src" ] || [ -z "$dest" ]; then
      break
    fi
    if [ "$src" == "lan_$ethwan_zone_name" ] && [ "$dest" == "$ethwan_zone_name" ]; then
      util_execute_uci delete firewall.@forwarding[$idx]
      log $(basename "$0") "delete_wan_profile" $LINENO "forwarding rule delete from firewall"
      break
    fi
    idx=$((idx+1))
  done

  #Delete wan profile from wan_all zone
  wan_all_idx=`util_execute_uci show firewall | grep -i name | grep -i "wan_all" | awk -F'[][]' '{print $2}'`
  util_execute_uci del_list firewall.@zone[$wan_all_idx].network=$ethwan_iface

  #Delete wan from vlan config
  no_of_vlans=$(util_execute_uci get qcmap_lan.@no_of_configs[0].no_of_vlans)
  for i in $(seq $no_of_vlans)
  do
    waneth_profile=$(util_execute_uci get qcmap_lan.@vlan[$((i-1))].waneth_profile_num)
    if [ $waneth_profile -eq $profile ]; then
      util_execute_uci delete qcmap_lan.@vlan[$((i-1))].waneth_profile_num
    fi
  done

  util_execute_uci set qcmap_lan.@no_of_configs[0].no_of_profiles=$((no_of_wan_profiles-1))
  if [ "$proto" == "dhcp" ]; then
    util_execute_uci set qcmap_lan.@no_of_configs[0].no_of_ethwan_dhcp_profiles=$((idx_dhcp-1))
  elif [ "$proto" == "pppoe" ]; then
    idx_pppoe=$((idx_pppoe-1))
    util_execute_uci set qcmap_lan.@no_of_configs[0].no_of_ethwan_pppoe_profiles=${idx_pppoe}

    if [ $idx_pppoe -eq 0 ]; then
      #update to ipa
      ipa pppoe category wan $iface disable > /dev/null
      log $(basename "$0") "delete_wan_profile" $LINENO "ipa cmd: ipa pppoe category wan $iface disable"
    fi
  fi
  log $(basename "$0") "delete_wan_profile" $LINENO "config deleted from network and qcmap_lan"

  util_execute_uci commit qcmap_lan
  util_execute_uci commit firewall
  util_execute_uci commit network
  util_execute_uci commit dhcp
  ubus call network reload
  /etc/init.d/firewall reload
  /etc/init.d/dnsmasq reload
}

# Checks if vlan is already mapped to a ethwan
function check_vlan_ethwan_map() {
  local vlan_id=$1
  local profile_id
  local temp_vlan_id
  local no_of_vlans

  #Get total number of configured vlan
  no_of_vlans=$(util_execute_uci get qcmap_lan.@no_of_configs[0].no_of_vlans)
  if [ $no_of_vlans -eq 0 ]; then
    log $(basename "$0") "check_vlan_ethwan_map" $LINENO "No VLAN is configured"
    return 1
  fi

  #loop through the number of vlans
  for i in $(seq $no_of_vlans)
  do
    #check if vlan_id matches provided input
    temp_vlan_id=$(util_execute_uci get qcmap_lan.@vlan[$((i-1))].vlan_id)
    if [ $temp_vlan_id -eq $vlan_id ]; then
      #check if waneth_profile_num matches provided input
      profile_id=$(util_execute_uci get qcmap_lan.@vlan[$((i-1))].waneth_profile_num)
      if [ -z "$profile_id" ]; then
        log $(basename "$0") "check_vlan_ethwan_map" $LINENO "profile num not found in vlan"
        return 0
      else
        log $(basename "$0") "check_vlan_ethwan_map" $LINENO "profile num: $profile_id linked to vlan_id: $vlan_id"
        return 1
      fi
    fi
  done
  return 1
}

# Adds eth wan number to mapped vlan
function add_ethwan_num_to_vlan() {
  local vlan_id=$1
  local profile=$2
  local temp_vlan_id
  local no_of_vlans

  #add waneth number to vlan config
  no_of_vlans=$(util_execute_uci get qcmap_lan.@no_of_configs[0].no_of_vlans)
  for i in $(seq $no_of_vlans)
  do
    temp_vlan_id=$(util_execute_uci get qcmap_lan.@vlan[$((i-1))].vlan_id)
    if [ $temp_vlan_id -eq $vlan_id ]; then
      #create a waneth_profile_num entry
      util_execute_uci set qcmap_lan.@vlan[$((i-1))].waneth_profile_num=$profile
      util_execute_uci commit qcmap_lan
      break
    fi
  done
}

# Deletes wan number from un_mapped vlan
function del_ethwan_num_to_vlan() {
  local vlan_id=$1
  local profile=$2
  local temp_vlan_id
  local no_of_vlans
  local waneth_profile_num

  #delete wan number from vlan config
  no_of_vlans=$(util_execute_uci get qcmap_lan.@no_of_configs[0].no_of_vlans)
  for i in $(seq $no_of_vlans)
  do
    temp_vlan_id=$(util_execute_uci get qcmap_lan.@vlan[$((i-1))].vlan_id)
    if [ $temp_vlan_id -eq $vlan_id ]; then
      #check if waneth_profile_num entry is the same as passed
      waneth_profile_num=$(util_execute_uci get qcmap_lan.@vlan[$((i-1))].waneth_profile_num)
      if [ -z "$waneth_profile_num" ]; then
        log $(basename "$0") "del_ethwan_num_to_vlan" $LINENO "No entry of waneth_profile_num found. Exiting!"
        return 0
      elif [ $waneth_profile_num -eq $profile ]; then
        #remove waneth_profile_num
        util_execute_uci del qcmap_lan.@vlan[$((i-1))].waneth_profile_num
        break
      else
        log $(basename "$0") "del_ethwan_num_to_vlan" $LINENO "Invalid combination of vlan to wan map found. Exiting!"
        return 0
      fi
    fi
  done
}

# Adds vlan_id to corresponding profile config in qcmap_lan when mapping is added
function add_vlan_to_ethwan_profile() {
  local vlan_id=$1
  local profile=$2
  local profile_id temp_vlan_id
  local vlan_id_list

  #Add Vlan ID to profile in qcmap_lan
  profile_id=$( util_execute_uci get qcmap_lan.profile_$profile.profile_id)
  if [ -n "$profile_id" ]; then
    vlan_id_list=$(util_execute_uci get qcmap_lan.profile_$profile.vlan_ids)
    for temp_vlan_id in $vlan_id_list
    do
      if [ "$temp_vlan_id" == "$vlan_id" ]; then
        log $(basename "$0") "add_vlan_to_ethwan_profile" $LINENO "vlan_id already exists"
        return 1
      fi
    done

    util_execute_uci add_list qcmap_lan.profile_$profile.vlan_ids=$vlan_id
    util_execute_uci commit qcmap_lan
    return 0
  fi
  return 1
}

# Deletes vlan_id from corresponding profile config in qcmap_lan when mapping is deleted
function del_vlan_from_ethwan_profile() {
  local vlan_id=$1
  local profile=$2
  local profile_id temp_vlan_id
  local vlan_id_list

  #Delete vlan_id from profile in qcmap_lan
  profile_id=$(util_execute_uci get qcmap_lan.profile_$profile.profile_id)
  if [ -n "$profile_id" ]; then
    vlan_id_list=$(util_execute_uci get qcmap_lan.profile_$profile.vlan_ids)
    for temp_vlan_id in $vlan_id_list
    do
      if [ "$temp_vlan_id" == "$vlan_id" ]; then
        util_execute_uci del_list qcmap_lan.profile_$profile.vlan_ids=$vlan_id
        util_execute_uci commit qcmap_lan
        return 0
      fi
    done
  fi
  return 1
}

# Adds lan interface to lan_waneth zone in firewall
function add_lan_waneth_interface() {
  local vlan_id=$1
  local profile=$2
  local idx=""
  local default_profile=$(util_execute_uci get qcmap_lan.@lan[0].default_waneth_profile_num)

  #Add corresponding forwarding rule in firewall config
  if [ $profile -eq $default_profile ]; then
    idx=`util_execute_uci show firewall | grep zone | grep -w "lan_wan" | awk -F'[][]' '{print $2}'`
  else
    idx=`util_execute_uci show firewall | grep zone | grep -w "lan_wan$profile" | awk -F'[][]' '{print $2}'`
  fi
  if [ -z "$idx" ]; then
    log $(basename "$0") "add_lan_waneth_interface" $LINENO "lan_waneth zone for $profile not found"
    return 1
  else
    lan_interface_list=$(util_execute_uci get firewall.@zone[$idx].network)
    for lan_interface in $lan_interface_list
    do
      if [ "$lan_interface" == "lan$vlan_id" ]; then
        log $(basename "$0") "add_lan_waneth_interface" $LINENO "lan_waneth_interface already exists"
        return 1
      fi
    done
    util_execute_uci add_list firewall.@zone[$idx].network=lan$vlan_id
  fi
  return 0
}

# Deletes lan interface from lan_waneth zone from firewall
function del_lan_waneth_interface() {
  local vlan_id=$1
  local profile=$2
  local idx=""
  local is_default=0
  local default_profile=$(util_execute_uci get qcmap_lan.@lan[0].default_waneth_profile_num)
  local downstream downstream_member

  #Delete corresponding forwarding rule from firewall config
  if [ $profile -eq $default_profile ]; then
    idx=`util_execute_uci show firewall | grep zone | grep -w "lan_wan" | awk -F'[][]' '{print $2}'`
    is_default=1
  else
    idx=`util_execute_uci show firewall | grep zone | grep -w "lan_wan$profile" | awk -F'[][]' '{print $2}'`
  fi
  if [ -z "$idx" ]; then
    log $(basename "$0") "del_lan_waneth_interface" $LINENO "lan_waneth zone for $profile not found"
    return 1
  else
    downstream=$(util_execute_uci get network.wan.downstream)
    for downstream_member in $downstream
    do
      if  [ "$is_default" -eq 1 -a "$downstream_member" == "lan$vlan_id" ]; then
        log $(basename "$0") "del_lan_waneth_interface" $LINENO "lan$vlan_id mapped to default wan, don't delete it from lan_wan zone"
        return 0
      fi
    done

    lan_interface_list=$(util_execute_uci get firewall.@zone[$idx].network)
    for lan_interface in $lan_interface_list
    do
      if [ "$lan_interface" == "lan$vlan_id" ]; then
        util_execute_uci del_list firewall.@zone[$idx].network=lan$vlan_id
        return 0
      fi
    done
  fi
  return 1
}

# update lan interface in lan_waneth zone from firewall
function update_lan_waneth_interface() {
  local old_default_profile=$1
  local new_default_profile=$2
  local idx=""
  local default_profile=$(util_execute_uci get qcmap_lan.@lan[0].default_waneth_profile_num)
  local lan_interface lan_interface_list
  local temp_vlan_id vlan_id_list

  if [ $new_default_profile -ne $default_profile ]; then
    log $(basename "$0") "update_lan_waneth_interface" $LINENO "wan$new_default_profile zone not found"
    return
  fi

  #delete network in lan_wan zone
  idx=`util_execute_uci show firewall | grep zone | grep -w "lan_wan$new_default_profile" | awk -F'[][]' '{print $2}'`
  if [ -z "$idx" ]; then
    log $(basename "$0") "update_lan_waneth_interface" $LINENO "lan_waneth$new_default_profile zone not found"
    return
  else
    lan_interface_list=$(util_execute_uci get firewall.@zone[$idx].network)
    for lan_interface in $lan_interface_list
    do
      util_execute_uci del_list firewall.@zone[$idx].network=$lan_interface
    done
  fi

  #add new network in lan_wan zone
  vlan_id_list=$(util_execute_uci get qcmap_lan.profile_$old_default_profile.vlan_ids)
  for temp_vlan_id in $vlan_id_list
  do
    if [ $temp_vlan_id -ne 0 ]; then
      util_execute_uci del_list firewall.@zone[$idx].network=lan$temp_vlan_id
      util_execute_uci add_list firewall.@zone[$idx].network=lan$temp_vlan_id
    fi
  done

  #update zone
  sed -i "s/wan$new_default_profile/wan$old_default_profile/g" /etc/config/firewall
  sed -i "s/waneth$new_default_profile/waneth$old_default_profile/g" /etc/config/firewall
}

# Adds vlan to wan mapping and updates required config
function add_lan_to_wan_map() {
  local vlan_id=$1
  local profile=$2
  local ethwan_iface
  local vlan_iface
  local default_profile=$(util_execute_uci get qcmap_lan.@lan[0].default_waneth_profile_num)

  #checking if profile_id is between 100 and 200
  if [ $profile -le 100 -o $profile -ge 200 ]; then
    log $(basename "$0") "add_lan_to_wan_map" $LINENO "profile_id should between 100 and 200"
    return
  fi

  #Check if given vlan has ethwan mapped already
  if check_vlan_ethwan_map $1; then
    log $(basename "$0") "add_lan_to_wan_map" $LINENO "given vlan_id does not have any profile_num linked"
  else
    log $(basename "$0") "add_lan_to_wan_map" $LINENO "no vlan or vlan_id is already mapped to a ethwan"
    return
  fi

  #get ethwan_iface
  if [ $profile -eq $default_profile ]; then
    ethwan_iface="waneth"
  else
    ethwan_iface="waneth${profile}"
  fi

  #Get associated vlan interfaces
  vlan_iface=$(util_execute_uci get network.lan$vlan_id.ifname)
  if [ -z "$vlan_iface" ]; then
    log $(basename "$0") "add_lan_to_wan_map" $LINENO "no vlan interfaces linked to br-lan$vlan_id, exiting!"
    return
  else
    util_execute_uci add_list network.$ethwan_iface.downstream=lan$vlan_id
    util_execute_uci add_list network.${ethwan_iface}_v6.downstream=lan$vlan_id

    #add_pdn to vlan-config in qcmap_lan
    add_ethwan_num_to_vlan $1 $2

    #Add Vlan ID to profile in qcmap_lan
    if add_vlan_to_ethwan_profile $1 $2; then
      log $(basename "$0") "add_lan_to_wan_map" $LINENO "added vlan_id to profile_section"
    else
      log $(basename "$0") "add_lan_to_wan_map" $LINENO "failed to add vlan_id to profile_section"
      return
    fi

    #Add forwarding rule in firewall
    if add_lan_waneth_interface $1 $2; then
      log $(basename "$0") "add_lan_to_wan_map" $LINENO "add_lan_waneth_interface success"
    else
      log $(basename "$0") "add_lan_to_wan_map" $LINENO "add_lan_waneth_interface fail"
    fi

    util_execute_uci commit qcmap_lan
    util_execute_uci commit firewall
    util_execute_uci commit network
    ubus call network reload
    /etc/init.d/firewall reload

    #add rule if add vlan mapping after pppoe backhaul bring up
    if [ -f /tmp/ipv4config.$ethwan_iface ]; then
      /etc/data/lanUtils.sh util_setup_nonwwan_route_rule_v4 $ethwan_iface
      switch_backhaul $ethwan_iface 1
    fi
    if [ -f /tmp/ipv6config.$ethwan_iface ]; then
      /etc/data/lanUtils.sh util_setup_nonwwan_route_rule_v6 $ethwan_iface
      switch_backhaul_v6 $ethwan_iface 1
    fi
  fi
}

# Deletes vlan to ethwan mapping and updates required config
function del_lan_to_wan_map() {
  local vlan_id=$1
  local profile=$2
  local found=0
  local ethwan_iface
  local no_of_vlans
  local profile_id temp_vlan_id
  local default_profile=$(util_execute_uci get qcmap_lan.@lan[0].default_waneth_profile_num)

  #checking if profile_id is between 100 and 200
  if [ $profile -le 100 -o $profile -ge 200 ]; then
    log $(basename "$0") "del_lan_to_wan_map" $LINENO "profile_id should between 100 and 200"
    return
  fi

  #Get total number of vlans configured
  no_of_vlans=$(util_execute_uci get qcmap_lan.@no_of_configs[0].no_of_vlans)
  if [ "$no_of_vlans" -eq 0 ]; then
    log $(basename "$0") "del_lan_to_wan_map" $LINENO "no vlan is configured"
    return
  fi

  #get ethwan_iface
  if [ $profile -eq $default_profile ]; then
    ethwan_iface="waneth"
  else
    ethwan_iface="waneth${profile}"
  fi

  #loop through the number of vlans
  for i in $(seq $no_of_vlans)
  do
    #check if vlan_id matches provided input
    temp_vlan_id=$(util_execute_uci get qcmap_lan.@vlan[$((i-1))].vlan_id)
    if [ "$temp_vlan_id" == "$vlan_id" ]; then
      #check if waneth_profile_num matches provided input
      profile_id=$(util_execute_uci get qcmap_lan.@vlan[$((i-1))].waneth_profile_num)
      if [ "$profile_id" == "$profile" ]; then
        log $(basename "$0") "del_lan_to_wan_map" $LINENO "entered vlan_id and profile number match found"
        found=1

        #if del vlan mapping before pppoe backhaul bring up
        #update vlan config
        util_execute_uci_set qcmap_lan.@vlan[$((i-1))].bh_present ""
        util_execute_uci_set qcmap_lan.@vlan[$((i-1))].bh_present_v6 ""
        util_execute_uci_set dhcp.lan$vlan_id.dhcpv6 "server"
        util_execute_uci_set dhcp.lan$vlan_id.ra     "server"
        util_execute_uci -q delete dhcp.lan$vlan_id.ndp
        util_execute_uci -q delete dhcp.lan$vlan_id.master_dhcp
        break
      else
        log $(basename "$0") "del_lan_to_wan_map" $LINENO "entered vlan_id and profile number mismatch, please try again!"
        return
      fi
    fi
  done

  if [ $found -eq 1 ]; then
    #if del vlan mapping before pppoe backhaul bring down
    #clean rule first, add rule later
    if [ -f /tmp/ipv4config.$ethwan_iface ]; then
      /etc/data/lanUtils.sh util_delete_nonwwan_route_rule_v4 $ethwan_iface
    fi
    if [ -f /tmp/ipv6config.$ethwan_iface ]; then
      /etc/data/lanUtils.sh util_delete_nonwwan_route_rule_v6 $ethwan_iface
    fi

    #remove downstream linked to ethwan
    util_execute_uci del_list network.$ethwan_iface.downstream=lan$vlan_id
    util_execute_uci del_list network.${ethwan_iface}_v6.downstream=lan$vlan_id

    #Delete wwan backhaul profile number from config vlan
    del_ethwan_num_to_vlan $1 $2

    #Delete vlan_id from profile in qcmap_lan
    if del_vlan_from_ethwan_profile $1 $2 ; then
      log $(basename "$0") "del_lan_to_wan_map" $LINENO "delete vlan_id from profile_section succes!"
    else
      log $(basename "$0") "del_lan_to_wan_map" $LINENO "failed to delete vlan_id from profile_section"
      return
    fi

    #Delete corresponding forwarding rule from firewall config
    if del_lan_waneth_interface $1 $2 ; then
      log $(basename "$0") "del_lan_to_wan_map" $LINENO "del_lan_waneth_interface success"
    else
      log $(basename "$0") "add_lan_to_wan_map" $LINENO "del_lan_waneth_interface fail"
    fi

    util_execute_uci commit qcmap_lan
    util_execute_uci commit firewall
    util_execute_uci commit network
    ubus call network reload
    /etc/init.d/firewall reload

    #if del vlan mapping before pppoe backhaul bring down
    #update rule
    if [ -f /tmp/ipv4config.$ethwan_iface ]; then
      /etc/data/lanUtils.sh util_setup_nonwwan_route_rule_v4 $ethwan_iface
    fi
    if [ -f /tmp/ipv6config.$ethwan_iface ]; then
      /etc/data/lanUtils.sh util_setup_nonwwan_route_rule_v6 $ethwan_iface
    fi

    #delete conntrack after delete waneth vlan to waneth pdn mapping
    util_del_conntrack $profile "ipv4v6" 0
  fi
}

#changing from LAN-LAN to LAN-WAN mode.
#if previously profile is created (over DHCP or PPPoE) and interface is changed
#server can call this function to update interface name
function reverse_ethwan_iface() {
  local profile_id
  local device new_device
  local ethwan_iface
  local vlan_id proto
  local default_profile=$(util_execute_uci get qcmap_lan.@lan[0].default_waneth_profile_num)
  local no_of_wan_profiles=$(util_execute_uci get qcmap_lan.@no_of_configs[0].no_of_profiles)

  if [ -z "$default_profile" ]; then
    log $(basename "$0") "reverse_ethwan_iface" $LINENO "no wan profile not found"
    return
  fi

  device=$(util_execute_uci get qcmap_lan.profile_$default_profile.device)
  if [ "$device" == "eth0" ]; then
    new_device="eth1"
  elif [ "$device" == "eth1" ]; then
    new_device="eth0"
  fi

  for i in $(seq $no_of_wan_profiles)
  do
    profile_id=$(util_execute_uci get qcmap_lan.@profile[$((i-1))].profile_id)
    if [ $profile_id -ge 100 ]; then
      vlan_id=$(util_execute_uci get qcmap_lan.profile_$profile_id.vlan_id)
      proto=$(util_execute_uci get qcmap_lan.profile_$profile_id.proto)
      if [ $profile_id -eq $default_profile ]; then
        ethwan_iface="waneth"
      else
        ethwan_iface="waneth${profile_id}"
      fi

      util_execute_uci set qcmap_lan.profile_$profile_id.device=$new_device
      if [ $vlan_id -eq 0 ]; then
        util_execute_uci set network.$ethwan_iface.device="$new_device"
      else
        util_execute_uci set network.$ethwan_iface.device="$new_device.$vlan_id"
      fi

      if [ "$proto" == "dhcp" ]; then
        util_execute_uci set network.waneth_v6.device=$new_device
      fi
    fi
  done
  log $(basename "$0") "reverse_ethwan_iface" $LINENO "success to reverse wan iface"

  util_execute_uci commit qcmap_lan
  util_execute_uci commit network
  ubus call network reload
}

case $1 in
  set_bh_pref)
    log  $(basename "$0") "set_bh_pref" $LINENO " set $2 $3 $4 $5 $6"
    set_backhaul_pref $2 $3 $4 $5 $6
    ;;

  switch_bh)
    log  $(basename "$0") "run_switch_backhaul_with_lock" $LINENO "Ayush $3 $4 $5 $6 $7 $8 $9 $10"
     run_switch_backhaul_with_lock $2 $3 $4 $5 $6 $7 $8 $9 $10
    ;;

  switch_bh_v6)
    log  $(basename "$0") "switch_bh_v6" $LINENO " $2 $3"
    switch_backhaul_v6 $2 $3
    ;;

  update_present_backhaul)
    log  $(basename "$0") "update_present_backhaul" $LINENO " $2 $3 $4"
    update_present_backhaul $2 $3 $4
    ;;

  read_current_backhaul)
    read_current_backhaul
    log  $(basename "$0") "read_current_backhaul" $LINENO ""
    ;;

  is_wwan_prefer_backhaul)
    is_wwan_prefer_backhaul
    log  $(basename "$0") "is_wwan_prefer_backhaul" $LINENO ""
    ;;

  is_on_demand_wwan_prefer_backhaul)
    is_on_demand_wwan_prefer_backhaul $2
    log  $(basename "$0") "is_on_demand_wwan_prefer_backhaul" $LINENO " $2"
    ;;

  get_backhaul_file)
    get_backhaul_file $2 $3
    echo $file
    log  $(basename "$0") "get_backhaul_file" $LINENO " $2 $3"
    ;;

  disable_ipv4)
    log  $(basename "$0") "case:disable_ipv4" $LINENO "disable_ipv4"
    disable_ipv4
    ;;

  disable_ipv6)
    log  $(basename "$0") "case:disable_ipv6" $LINENO "disable_ipv6"
    disable_ipv6
    ;;

  enable_ipv4)
    log  $(basename "$0") "case:enable_ipv4" $LINENO "enable_ipv4"
    enable_ipv4
    ;;

  enable_ipv6)
    log  $(basename "$0") "case:enable_ipv6" $LINENO "enable_ipv6"
    enable_ipv6
    ;;

  get_bh_load_balance_status)
    log  $(basename "$0") "get_bh_load_balance_status" $LINENO "get_bh_load_balance_status"
    get_bh_load_balance_status
    ;;

  set_ipa_load_balance)
    log  $(basename "$0") "set_ipa_load_balance" $LINENO "set_ipa_load_balance"
    set_ipa_load_balance
    ;;

  set_bh_load_balance)
    log  $(basename "$0") "set_bh_load_balance" $LINENO " set $2 $3 $4"
    set_bh_load_balance $2 $3 $4
    ;;

  update_prefix_info_to_tmp_file)
    log  $(basename "$0") "update_prefix_info_to_tmp_file" $LINENO " update_prefix_info_to_tmp_file"
    update_prefix_info_to_tmp_file
    ;;

   set_ipa_offload_config)
    log  $(basename "$0") "set_ipa_offload_config" $LINENO " set_ipa_offload_config"
    set_ipa_offload_config
    ;;

  create_wan_profile)
    log  $(basename "$0") "create_wan_profile" $LINENO " $2 $3 $4 $5 $6 $7 $8 $9"
    create_wan_profile $2 $3 $4 $5 $6 $7 $8 $9
    ;;

  update_wan_profile)
    log  $(basename "$0") "update_wan_profile" $LINENO " $2 $3 $4 $5 $6 $7 $8 $9"
    update_wan_profile $2 $3 $4 $5 $6 $7 $8 $9
    ;;

  delete_wan_profile)
    log  $(basename "$0") "delete_wan_profile" $LINENO " $2"
    delete_wan_profile $2
    ;;

  add_lan_to_wan_map)
    log  $(basename "$0") "add_lan_to_wan_map" $LINENO " $2 $3"
    add_lan_to_wan_map $2 $3
    ;;

  del_lan_to_wan_map)
    log  $(basename "$0") "del_lan_to_wan_map" $LINENO " $2 $3"
    del_lan_to_wan_map $2 $3
    ;;

  reverse_ethwan_iface)
    log  $(basename "$0") "reverse_ethwan_iface" $LINENO "reverse_ethwan_iface"
    reverse_ethwan_iface
    ;;

  *)
    log $(basename "$0") "" $LINENO ":Invalid option"
    ;;

esac




