#!/bin/sh
#  Copyright (c) 2023 Qualcomm Technologies, Inc.
#  All Rights Reserved.
#  Confidential and Proprietary - Qualcomm Technologies, Inc.

#Usage: For logging
source /etc/data/mbbUtils.sh
source /etc/data/lanUtils.sh


#Function to set ipsec configuration
#$1 - profile_id
#$2 - ike_identifier
#$3 - tunnel_type
#$4 - auth_type
#$5 - remote_ep_addr
#$6 - local_identifier
#$7 - remote_identifier
#$8 - rekey_interval_mins
#$9 - topology
#$10 - child_identifier
#$11 - hw_offload
#$12 - trap_action
#$13 - child_rekey_interval_mins
#$14 - local_port_id
#$15 - local_end_port_id
#$16 - remote_port_id
#$17 - remote_end_port_id
#$18 - protocol_type
#$19 - local_addr
#$20 - remote_addr
function handleSetIpsecTunnel() {
  local profile_id="$1"
  local ike_identifier="$2"
  local tunnel_type="$3"
  local auth_type="$4"
  local remote_ep_addr="$5"
  local local_identifier="$6"
  local remote_identifier="$7"
  local rekey_interval_mins="$8"
  local topology_type="$9"
  local child_identifier="${10}"
  local hw_offload="${11}"
  local trap_action="${12}"
  local child_rekey_interval_mins="${13}"
  local local_port_id="${14}"
  local local_end_port_id="${15}"
  local remote_port_id="${16}"
  local remote_end_port_id="${17}"
  local protocol_type="${18}"
  local local_addr="${19}"
  local remote_addr="${20}"

  local local_ip
  local psk_file_path="/etc/swanctl/private/${ike_identifier}_psk"
  local cert_file_path="/etc/swanctl/x509/${ike_identifier}_cert.pem"
  local ca_cert_file_path="/etc/swanctl/x509ca/${ike_identifier}_ca.pem"
  local cert_key_file_path="/etc/swanctl/private/${ike_identifier}_key.pem"
  local psk
  local cert
  local ca_cert
  local cert_key

  local rmnet_name

  local nft_v6_raw_prerouting_enabled=0
  local nft_v6_raw_prerouting="nft insert rule inet fw4 raw_prerouting "

  #Check if psk file or certificate file exist
  if [ $auth_type -eq 1 ]; then
    if [ ! -f "$psk_file_path" ]; then
      echo "PSK file does not exist"
      return 1
    fi
  fi

  if [ $auth_type -eq 2 ]; then
    if [ ! -f "$cert_file_path" ] || [ ! -f "$ca_cert_file_path" ] || [ ! -f "$cert_key_file_path" ]; then
      echo "Certificate files do not exist"
      return 1
    fi
  fi

  if [ $(util_ip_type $local_addr) == "IPv6" ] && [ $(util_ip_type $remote_addr) == "IPv6" ]; then
    log $(basename "$0") "handleSetIpsecTunnel" $LINENO "Enter IPv6 nft rule setting!"
    nft_v6_raw_prerouting_enabled=1
    nft_v6_raw_prerouting="${nft_v6_raw_prerouting}ip6 daddr $remote_addr "
  fi

  if [ ! -f "/etc/config/ipsec" ]; then
    touch /etc/config/ipsec
  fi

  uci set ipsec.$ike_identifier=remote
  uci set ipsec.$ike_identifier.enabled=0
  uci set ipsec.$ike_identifier.profile_id="$profile_id"
  uci set ipsec.$ike_identifier.tunnel_type="$tunnel_type"
  uci set ipsec.$ike_identifier.topology_type="$topology_type"
  uci set ipsec.$ike_identifier.local_leftip="$local_ip"
  uci set ipsec.$ike_identifier.gateway="$remote_ep_addr"

  if [ $auth_type -eq 1 ]; then
    uci set ipsec.$ike_identifier.authentication_method='psk'
    # Read the value from the file
    read -r psk < "$psk_file_path"
    echo "Value read from file: $psk"
    uci set ipsec.$ike_identifier.pre_shared_key="$psk"
  fi

  if [ $auth_type -eq 2 ]; then
    uci set ipsec.$ike_identifier.authentication_method='pubkey'
    uci set ipsec.$ike_identifier.local_identifier="$local_identifier"
    uci set ipsec.$ike_identifier.remote_identifier="$remote_identifier"
    uci set ipsec.$ike_identifier.local_cert="${ike_identifier}_cert.pem"
    uci set ipsec.$ike_identifier.ca_cert="${ike_identifier}_ca.pem"
    uci set ipsec.$ike_identifier.local_key="${ike_identifier}_key.pem"
  fi

  uci add_list ipsec.$ike_identifier.tunnel="$child_identifier"

  local existing_ikecrypto_list=$(uci get ipsec.$ike_identifier.crypto_proposal)
  if [[ "$existing_ikecrypto_list" != *"ike_proposal"* ]]; then
    # Value does not exist, add it to the list
    uci add_list ipsec.$ike_identifier.crypto_proposal='ike_proposal'
  fi

  uci set ipsec.$child_identifier=tunnel
  uci set ipsec.$child_identifier.enabled=0

  if [ $hw_offload -eq 1 ]; then
    uci set ipsec.$child_identifier.hw_offload='packet'
  else
    uci set ipsec.$child_identifier.hw_offload='no'
  fi

  if [ $trap_action -eq 1 ]; then
    uci set ipsec.$child_identifier.trap_action='trap'
  else
    uci set ipsec.$child_identifier.trap_action='start'
  fi


  uci set ipsec.$ike_identifier.rekeytime="${rekey_interval_mins}m"
  uci set ipsec.$child_identifier.rekeytime="${child_rekey_interval_mins}m"

  local existing_espcrypto_list=$(uci get ipsec.$child_identifier.crypto_proposal)
  if [[ "$existing_espcrypto_list" != *"esp_proposal"* ]]; then
    # Value does not exist, add it to the list
    uci add_list ipsec.$child_identifier.crypto_proposal='esp_proposal'
  fi


  #need more info for topology 2&3: local_addr and remote_addr
  if [ $topology_type -eq 2 ] || [ $topology_type -eq 3 ]; then
    if [ $protocol_type -eq 1 ]; then
      #tcp
      if [ $nft_v6_raw_prerouting_enabled -eq 1 ]; then
        nft_v6_raw_prerouting="${nft_v6_raw_prerouting}ip6 nexthdr tcp "
      fi

      # local subnet
      if [ $local_port_id -eq 0 ]; then
        uci set ipsec.$child_identifier.local_subnet="$local_addr[tcp]"
      else
        if [ $local_end_port_id -eq 0 ]; then
          uci set ipsec.$child_identifier.local_subnet="$local_addr[tcp/$local_port_id]"
          if [ $nft_v6_raw_prerouting_enabled -eq 1 ]; then
            nft_v6_raw_prerouting="${nft_v6_raw_prerouting}tcp sport $local_port_id "
          fi
        else
          subnet_setting_string=""
          util_ipsec_split_port_ranges $local_addr "tcp" $local_port_id $local_end_port_id
          uci set ipsec.$child_identifier.local_subnet="$subnet_setting_string"
          if [ $nft_v6_raw_prerouting_enabled -eq 1 ]; then
            nft_v6_raw_prerouting="${nft_v6_raw_prerouting}tcp sport $local_port_id-$local_end_port_id "
          fi
        fi
      fi

      # remote subnet
      if [ $remote_port_id -eq 0 ]; then
        uci set ipsec.$child_identifier.remote_subnet="$remote_addr[tcp]"
      else
        if [ $remote_end_port_id -eq 0 ]; then
          uci set ipsec.$child_identifier.remote_subnet="$remote_addr[tcp/$remote_port_id]"
          if [ $nft_v6_raw_prerouting_enabled -eq 1 ]; then
            nft_v6_raw_prerouting="${nft_v6_raw_prerouting}tcp dport $remote_port_id "
          fi
        else
          subnet_setting_string=""
          util_ipsec_split_port_ranges $remote_addr "tcp" $remote_port_id $remote_end_port_id
          uci set ipsec.$child_identifier.remote_subnet="$subnet_setting_string"
          if [ $nft_v6_raw_prerouting_enabled -eq 1 ]; then
            nft_v6_raw_prerouting="${nft_v6_raw_prerouting}tcp dport $remote_port_id-$remote_end_port_id "
          fi
        fi
      fi
    elif [ $protocol_type -eq 2 ]; then
      #udp
      if [ $nft_v6_raw_prerouting_enabled -eq 1 ]; then
        nft_v6_raw_prerouting="${nft_v6_raw_prerouting}ip6 nexthdr udp "
      fi

      # local subnet
      if [ $local_port_id -eq 0 ]; then
        uci set ipsec.$child_identifier.local_subnet="$local_addr[udp]"
      else
        if [ $local_end_port_id -eq 0 ]; then
          uci set ipsec.$child_identifier.local_subnet="$local_addr[udp/$local_port_id]"
          if [ $nft_v6_raw_prerouting_enabled -eq 1 ]; then
            nft_v6_raw_prerouting="${nft_v6_raw_prerouting}udp sport $local_port_id "
          fi
        else
          subnet_setting_string=""
          util_ipsec_split_port_ranges $local_addr "udp" $local_port_id $local_end_port_id
          uci set ipsec.$child_identifier.local_subnet="$subnet_setting_string"
          if [ $nft_v6_raw_prerouting_enabled -eq 1 ]; then
            nft_v6_raw_prerouting="${nft_v6_raw_prerouting}udp sport $local_port_id-$local_end_port_id "
          fi
        fi
      fi

      #remote subnet
      if [ $remote_port_id -eq 0 ]; then
        uci set ipsec.$child_identifier.remote_subnet="$remote_addr[udp]"
      else
        if [ $remote_end_port_id -eq 0 ]; then
          uci set ipsec.$child_identifier.remote_subnet="$remote_addr[udp/$remote_port_id]"
          if [ $nft_v6_raw_prerouting_enabled -eq 1 ]; then
            nft_v6_raw_prerouting="${nft_v6_raw_prerouting}udp dport $remote_port_id "
          fi
        else
          subnet_setting_string=""
          util_ipsec_split_port_ranges $remote_addr "udp" $remote_port_id $remote_end_port_id
          uci set ipsec.$child_identifier.remote_subnet="$subnet_setting_string"
          if [ $nft_v6_raw_prerouting_enabled -eq 1 ]; then
            nft_v6_raw_prerouting="${nft_v6_raw_prerouting}udp dport $remote_port_id-$remote_end_port_id "
          fi
        fi
      fi
    else
      #both
      uci set ipsec.$child_identifier.local_subnet="$local_addr"
      uci set ipsec.$child_identifier.remote_subnet="$remote_addr"
    fi

    if [ $nft_v6_raw_prerouting_enabled -eq 1 ]; then
      nft_v6_raw_prerouting="${nft_v6_raw_prerouting}notrack"
      log $(basename "$0") "handleSetIpsecTunnel" $LINENO "Write IPv6 nft rule2file: $nft_v6_raw_prerouting"
      echo $nft_v6_raw_prerouting > /etc/data/nft_v6_raw_prerouting_$ike_identifier\_$profile_id\_$child_identifier
    fi
  fi



  #adding ike crypto list

  uci set ipsec.ike_proposal=crypto_proposal
  uci set ipsec.ike_proposal.encryption_algorithm='aes128'
  uci set ipsec.ike_proposal.hash_algorithm='sha256'
  uci set ipsec.ike_proposal.dh_group='x25519'

  #adding esp list
  uci set ipsec.esp_proposal=crypto_proposal
  uci set ipsec.esp_proposal.encryption_algorithm='aes128gcm128'
  uci set ipsec.esp_proposal.dh_group='x25519-noesn-esn'

  if [ $tunnel_type -eq 2 ]; then
    uci set ipsec.$ike_identifier.encap='no'
  fi

  echo "uci commit ipsec"
  uci commit ipsec


}

#Usage :util_update_iptable_rules
#$1 - operation(1:add, 2: delete)
#$2 - rmnet_name
#$3 - data_ip_type
#$4 - tunnel_ip_type
#$5 - remote_addr
#$6 - event_option(1:delete API, 2:disconnected event, 0:others)
function util_update_iptable_rules(){
  local operation="$1"
  local rmnet_name="$2"
  local data_ip_type="$3"
  local tunnel_ip_type="$4"
  local remote_addr="$5"
  local event_option="$6"
  local iptable_type
  local mss_size

  #Get rmnet MTU size
  local mtu=$(ifconfig "$rmnet_name" | grep -o 'MTU:[0-9]*' | awk -F: '{print $NF}')
  echo "mtu: $mtu"

  #IPv4 traffic go through IPv4 tunnel(Inner IP header and outer IP header are both 20 bytes)
  #v4_v4_ipsec_overhead = 20(Inner IP header) + 60(TCP header and options) + 20(Outer IP Header) + 4(Sequence Number) + 4(SPI) + 16(Initialization Vector AES)
  #+ (0~15)(ESP Padding) + 1(Padding Length) + 1(Next Header) + 32(Authentication Data SHA512) + 8(UDP NAT-T header)= 181
  #+ 19 bytes buffer = 200 bytes
  local v4_v4_ipsec_overhead=200

  #IPv6 traffic go through IPv6 tunnel(Inner IP header and outer IP header are both 40 bytes)
  #v6_v6_ipsec_overhead = 40(Inner IP header) + 60(TCP header + options) + 40(Outer IP Header) + 4(Sequence Number) + 4(SPI) + 16(Initialization Vector AES)
  #+ (0~15)(ESP Padding) + 1(Padding Length) + 1(Next Header) + 32(Authentication Data SHA512) = 213
  #+ 17 bytes buffer = 230 bytes
  local v6_v6_ipsec_overhead=230

  #IPv6 traffic go through IPv4 tunnel(Inner IP header is 20 bytes and outer IP header is 40 bytes)
  #v4_v6_ipsec_overhead = 40(Inner IP header) + 60(TCP header + options) + 20(Outer IP Header) + 4(Sequence Number) + 4(SPI) + 16(Initialization Vector AES)
  #+ (0~15)(ESP Padding) + 1(Padding Length) + 1(Next Header) + 32(Authentication Data SHA512) + 8(UDP NAT-T header)= 201
  # + 19 bytes buffer = 220 bytes
  local v4_v6_ipsec_overhead=220

  #IPv4 traffic go through IPv6 tunnel(Inner IP header is 40 bytes and outer IP header is 20 bytes)
  #v6_v4_ipsec_overhead = 20(Inner IP header) + 60(TCP header + options) + 40(Outer IP Header) + 4(Sequence Number) + 4(SPI) + 16(Initialization Vector AES)
  #+ (0~15)(ESP Padding) + 1(Padding Length) + 1(Next Header) + 32(Authentication Data SHA512) = 193
  #+ 17 bytes buffer = 210 bytes
  local v6_v4_ipsec_overhead=210

  #Calculate TCP MSS size
  if [ "$data_ip_type" == "IPv4" ] && [ "$tunnel_ip_type" == "IPv4" ]; then
    iptables_type='iptables'
    mss_size="$(($mtu-$v4_v4_ipsec_overhead))"
    echo "mss_size: $mss_size"

  elif [ "$data_ip_type" == "IPv6" ] && [ "$tunnel_ip_type" == "IPv6" ]; then
    iptables_type='ip6tables'
    mss_size="$(($mtu-$v6_v6_ipsec_overhead))"
    echo "mss_size: $mss_size"

  elif [ "$data_ip_type" == "IPv4" ] && [ "$tunnel_ip_type" == "IPv6" ]; then
    iptables_type='iptables'
    mss_size="$(($mtu-$v6_v4_ipsec_overhead))"
    echo "mss_size: $mss_size"

  elif [ "$data_ip_type" == "IPv6" ] && [ "$tunnel_ip_type" == "IPv4" ]; then
    iptables_type='ip6tables'
    mss_size="$(($mtu-$v4_v6_ipsec_overhead))"
    echo "mss_size: $mss_size"

  fi

  #IPsec LAN Client forward rule
  if [ ! -f /tmp/data/firewall.user.ipsec ]; then
    touch /tmp/data/firewall.user.ipsec
  fi

  # IPsec traffic forward wan rule
  if [ $event_option -eq 2 ]; then
    sed -i "/nft insert rule inet fw4 forward_wan_all position 0 iifname $rmnet_name meta ipsec exists counter packets 0 bytes 0 accept/d" /tmp/data/firewall.user.ipsec
  fi
  if [ $operation -eq 1 ]; then
    #To allow esp traffic for specific interface
    sed -i "/nft insert rule inet fw4 forward_wan_all position 0 iifname $rmnet_name meta ipsec exists counter packets 0 bytes 0 accept/d" /tmp/data/firewall.user.ipsec
    echo "nft insert rule inet fw4 forward_wan_all position 0 iifname $rmnet_name meta ipsec exists counter packets 0 bytes 0 accept" >> /tmp/data/firewall.user.ipsec
  fi

  #IPsec traffic matching rule for IPv4 over IPv4
  if [ "$data_ip_type" == "IPv4" ] && [ "$tunnel_ip_type" == "IPv4" ]; then
    if [ $event_option -eq 2 ]; then
      sed -i "/nft insert rule inet fw4 srcnat position 0 oifname $rmnet_name ipsec out ip daddr $remote_addr counter packets 0 bytes 0 accept/d" /tmp/data/firewall.user.ipsec
    fi
    if [ $operation -eq 1 ]; then
        #To allow IPsec traffic for specific interface
        sed -i "/nft insert rule inet fw4 srcnat position 0 oifname $rmnet_name ipsec out ip daddr $remote_addr counter packets 0 bytes 0 accept/d" /tmp/data/firewall.user.ipsec
        echo "nft insert rule inet fw4 srcnat position 0 oifname $rmnet_name ipsec out ip daddr $remote_addr counter packets 0 bytes 0 accept" >> /tmp/data/firewall.user.ipsec
    fi
  elif [ "$data_ip_type" == "IPv4" ] && [ "$tunnel_ip_type" == "IPv6" ]; then
    #IPsec traffic matching rule for IPv4 over IPv6
    if [ $event_option -eq 2 ]; then
      sed -i "/nft insert rule inet fw4 srcnat position 0 oifname $rmnet_name ipsec out ip6 daddr $remote_addr counter packets 0 bytes 0 accept/d" /tmp/data/firewall.user.ipsec
    fi
    if [ $operation -eq 1 ]; then
        #To allow IPsec traffic for specific interface
        sed -i "/nft insert rule inet fw4 srcnat position 0 oifname $rmnet_name ipsec out ip6 daddr $remote_addr counter packets 0 bytes 0 accept/d" /tmp/data/firewall.user.ipsec
        echo "nft insert rule inet fw4 srcnat position 0 oifname $rmnet_name ipsec out ip6 daddr $remote_addr counter packets 0 bytes 0 accept" >> /tmp/data/firewall.user.ipsec
    fi
  fi

  # IPsec traffic matching for tunnel IPv6
  if [ "$tunnel_ip_type" == "IPv6" ]; then
    if [ $event_option -eq 2 ]; then
      sed -i "/nft insert rule inet fw4 input_wan_all position 0 meta nfproto ipv6 meta l4proto esp counter accept/d" /tmp/data/firewall.user.ipsec
    fi
    if [ $operation -eq 1 ]; then
      #To allow IPsec v6 traffic forward to wan
      sed -i "/nft insert rule inet fw4 input_wan_all position 0 meta nfproto ipv6 meta l4proto esp counter accept/d" /tmp/data/firewall.user.ipsec
      echo "nft insert rule inet fw4 input_wan_all position 0 meta nfproto ipv6 meta l4proto esp counter accept" >> /tmp/data/firewall.user.ipsec
    fi
  fi

  util_combine_iptable_rules
  util_run_command /etc/init.d/firewall reload


  #TCP MSS Clamping rules
  if [ $operation -eq 1 ]; then
    #Delete before adding to avoid duplicate
    $iptables_type -t mangle -D PREROUTING -m policy --pol ipsec --dir in -p tcp --tcp-flags SYN,RST SYN -j TCPMSS --set-mss $mss_size
    $iptables_type -t mangle -D PREROUTING -m policy --pol ipsec --dir in -p tcp --tcp-flags SYN,ACK SYN,ACK -j TCPMSS --set-mss $mss_size
    $iptables_type -t mangle -D POSTROUTING -m policy --pol ipsec --dir out -p tcp --tcp-flags SYN,RST SYN -j TCPMSS --set-mss $mss_size
    $iptables_type -t mangle -D POSTROUTING -m policy --pol ipsec --dir out -p tcp --tcp-flags SYN,ACK SYN,ACK -j TCPMSS --set-mss $mss_size

    $iptables_type -t mangle -I PREROUTING 1 -m policy --pol ipsec --dir in -p tcp --tcp-flags SYN,RST SYN -j TCPMSS --set-mss $mss_size
    $iptables_type -t mangle -I PREROUTING 1 -m policy --pol ipsec --dir in -p tcp --tcp-flags SYN,ACK SYN,ACK -j TCPMSS --set-mss $mss_size
    $iptables_type -t mangle -I POSTROUTING 1 -m policy --pol ipsec --dir out -p tcp --tcp-flags SYN,RST SYN -j TCPMSS --set-mss $mss_size
    $iptables_type -t mangle -I POSTROUTING 1 -m policy --pol ipsec --dir out -p tcp --tcp-flags SYN,ACK SYN,ACK -j TCPMSS --set-mss $mss_size
  fi

}

#Usage :util_update_ip_type
#$1 - ip
function util_ip_type(){
  local ip="$1"
  local first
  local second


  first=$(echo "$ip" | awk -F'/' '{print $1}')
  second=$(echo "$ip" | awk -F'/' '{print $2}')

  #echo "ip: $ip, first: $first, second: $second"


  if echo "$first" | grep -Eq '^(\d{1,3}\.){3}\d{1,3}$'; then
    echo "IPv4"
    return 1
  elif echo "$first" | grep -Eq '^([0-9a-fA-F]{1,4}::?){1,7}[0-9a-fA-F]{1,4}$'; then
    echo "IPv6"
    return 1
  else
    echo "Invalid"
    return 1
  fi
}

#Function to enable ipsec tunnel
#$1 - ike_identifier
#$2 - child_identifier
function util_ipsec_enable_tunnel(){
  local local_ip
  local rmnet_name
  local profile_id=$(uci get ipsec.$ike_identifier.profile_id)
  local tunnel_type=$(uci get ipsec.$ike_identifier.tunnel_type)
  local ike_identifier="$1"
  local child_identifier="$2"
  local local_subnet=$(uci get ipsec.$child_identifier.local_subnet)
  local remote_subnet=$(uci get ipsec.$child_identifier.remote_subnet)
  local remote_ep_addr=$(uci get ipsec.$ike_identifier.gateway)
  local data_ip_type
  local tunnel_ip_type

  util_ipsec_restart

  #source local ip
  if [ $tunnel_type -eq 1 ] && [ -f /tmp/ipv4config$profile_id ]; then
    . /tmp/ipv4config$profile_id
    local_ip=$PUBLIC_IP
    rmnet_name=$IFNAME
  fi

  if [ $tunnel_type -eq 2 ] && [ -f /tmp/ipv6config$profile_id ]; then
    . /tmp/ipv6config$profile_id
    local_ip=$PUBLIC_IP6
    rmnet_name=$IFNAME
  fi

  if [ -n "$local_ip" ] && [ -n "$rmnet_name" ]; then
    # Enable v6 raw prerouting nft rules
    if [ -f /etc/data/nft_v6_raw_prerouting_$ike_identifier\_$profile_id\_$child_identifier ]; then
      cat /etc/data/nft_v6_raw_prerouting_$ike_identifier\_$profile_id\_$child_identifier | while IFS= read -r nft_rule; do
        sed -i "\@$nft_rule@d" /tmp/data/firewall.user.ipsec
        echo "$nft_rule" >> /tmp/data/firewall.user.ipsec
        sed -i "/nft insert rule inet fw4 raw_prerouting meta ipsec exists notrack/d" /tmp/data/firewall.user.ipsec
        echo "nft insert rule inet fw4 raw_prerouting meta ipsec exists notrack" >> /tmp/data/firewall.user.ipsec
      done
    fi

    if [ -n "$local_subnet" ]; then
      #For S2S (192.168.224.0/23[tcp/5012-6012]) or '192.168.224.0/23[tcp/5014]'
      data_ip_type=$(util_ip_type $remote_subnet)
    else
      #For H2H
      data_ip_type=$(util_ip_type $remote_ep_addr)
    fi
    tunnel_ip_type=$(util_ip_type $local_ip)
    echo "rmnet_name:$rmnet_name,  data_ip_type: $data_ip_type, tunnel_ip_type: $tunnel_ip_type, remote_ep_addr: $remote_ep_addr"
    util_update_iptable_rules 1 $rmnet_name $data_ip_type $tunnel_ip_type $remote_ep_addr 0

    # set local ip
    uci set ipsec.$ike_identifier.local_leftip="$local_ip"
    uci commit ipsec
    uci commit qcmap_lan

    echo "/etc/init.d/swanctl restart"
    util_run_command /etc/init.d/swanctl restart
    echo "swanctl --load-all"
    util_run_command swanctl --load-all
    echo "Initiating"
    swanctl --initiate --child $child_identifier&
    echo "ipsec status"
    ipsec status

  else
    uci commit ipsec
    uci commit qcmap_lan
    echo "Waiting for data call"
  fi
}

#Function to activate ipsec tunnel from QCMAP
#$1 - ike_identifier
#$2 - child_identifier
function handleActivateIpsecTunnel(){
  local ike_identifier="$1"
  local child_identifier="$2"
  local current_active_tunnels=0
  local curr_enable_option

  uci set ipsec.$ike_identifier.enabled=1

  curr_enable_option=$(uci get ipsec.$child_identifier.enabled)

  # Check if the child is enabled or not
  if [ "$curr_enable_option" == "0" ]; then
    uci set ipsec.$child_identifier.enabled=1
    #increment active tunnels count
    current_active_tunnels=$(uci get qcmap_lan.@global[0].ipsec_active_tunnels)
    #Convert string to number
    number=$(echo "$current_active_tunnels" | awk '{print $0+0}')
    uci set qcmap_lan.@global[0].ipsec_active_tunnels=$((number+1))
  fi

  #Q: for later generating /var/swanctl/swanctl.conf
  mkdir -p /var/swanctl

  #call helper function to enable ipsec
  util_ipsec_enable_tunnel $ike_identifier $child_identifier

}

#Function to deactivate ipsec tunnel from QCMAP
#$1 - ike_identifier
#$2 - child_identifier
function handleDeactivateIpsecTunnel(){
  local ike_identifier="$1"
  local child_identifier="$2"
  local current_active_tunnels=0
  local curr_enable_option
  local profile_id=$(uci get ipsec.$ike_identifier.profile_id)

  #Deactivate ipsec tunnel
  if [ -n "$child_identifier" ]; then
    curr_enable_option=$(uci get ipsec.$child_identifier.enabled)

    # Check if the child is disabled or not
    if [ "$curr_enable_option" == "1" ]; then
      uci set ipsec.$child_identifier.enabled=0
      #decrement active tunnels count
      current_active_tunnels=$(uci get qcmap_lan.@global[0].ipsec_active_tunnels)
      #Convert string to number
      number=$(echo "$current_active_tunnels" | awk '{print $0+0}')
      uci set qcmap_lan.@global[0].ipsec_active_tunnels=$((number-1))

      # Delete v6 raw prerouting nft rules & reload firewall
      util_ipsec_delete_v6_raw_prerouting_rules $ike_identifier $profile_id $child_identifier 1
    fi
  else
    uci set ipsec.$ike_identifier.enabled=0

    #Disable all the child SAs under IKE session
    children=$(uci get ipsec.$ike_identifier.tunnel)
    echo "$children" | awk -F" " '{for(i=1; i<=NF; i+=1) print $i}' | while IFS= read -r child_id; do
      curr_enable_option=$(uci get ipsec.$child_id.enabled)
      if [ "$curr_enable_option" == "1" ]; then
        uci set ipsec.$child_id.enabled=0
        current_active_tunnels=$(uci get qcmap_lan.@global[0].ipsec_active_tunnels)
        number=$(echo "$current_active_tunnels" | awk '{print $0+0}')
        uci set qcmap_lan.@global[0].ipsec_active_tunnels=$((number-1))

        # Delete v6 raw prerouting nft rules & reload firewall
        util_ipsec_delete_v6_raw_prerouting_rules $ike_identifier $profile_id $child_id 1
      fi
    done

    uci delete ipsec.$ike_identifier.local_leftip
  fi

  log $(basename "$0") "handleDeactivateIpsecTunnel" $LINENO "Change swanctl settings from uci database."
  uci commit ipsec
  uci commit qcmap_lan
  util_run_command /etc/init.d/swanctl restart
  util_run_command swanctl --load-all

  # Terminate child/ike after uci settings done.
  log $(basename "$0") "handleDeactivateIpsecTunnel" $LINENO "swanctl terminate running..."
  if [ -n "$child_identifier" ]; then
    swanctl --terminate --child $child_identifier&
  else
    swanctl --terminate --ike $ike_identifier&
  fi

}


#Function to enable ipsec tunnel when wwan data call get connected
#$1 - profile_id
#$2 - ip_type(1:IPv4, 2:IPv6)
function handleDataCallConnectedEv() {
  local profile_id="$1"
  local ip_type="$2"
  local ike_identifier
  local child_identifier
  local section
  local option
  local entry
  local value
  local line
  local ike_enable_flag
  local child_enable_flag
  local children
  local child_id
  local tunnel_type
  local feature_mode=$(uci get qcmap_lan.@global[0].ipsec_enable)

  if [ "$feature_mode" == "0" ]; then
    return
  fi

  uci show ipsec | while read -r line; do
    #echo "$line"

    if [[ "$line" == *"profile_id"* ]]; then
      section=$(echo "$line" | awk -F"['.=]" '{print $1}')
      option=$(echo "$line" | awk -F"['.=]" '{print $2}')
      entry=$(echo "$line" | awk -F"['.=]" '{print $3}')
      value=$(echo "$line" | awk -F"['.=]" '{print $5}')
      #echo "Section: $section, Option: $option, Entry: $entry, Value: $value"

      #Check if the datacall is on the right profile
      if [ "$entry" == "profile_id" ] && [ "$value" == "$profile_id" ]; then
        echo "$line"
        echo "Section: $section, Option: $option, Entry: $entry, Value: $value"
        ike_identifier="$option"
        #echo "ike: $ike_identifier"

        tunnel_type=$(uci get ipsec.$ike_identifier.tunnel_type)

        ike_enable_flag=$(uci get ipsec.$ike_identifier.enabled)
        if [ "$ike_enable_flag" == 1 ] && [ "$tunnel_type" == "$ip_type" ]; then
          #Get all the children of ike
          children=$(uci get ipsec.$ike_identifier.tunnel)
          echo "$children" | awk -F" " '{for(i=1; i<=NF; i+=1) print $i}' | while IFS= read -r child_id; do
            #echo "child_id: $child_id"
            child_enable_flag=$(uci get ipsec.$child_id.enabled)
            if [ "$child_enable_flag" == 1 ]; then
              util_ipsec_enable_tunnel $ike_identifier $child_id
            fi
          done
        fi
      fi
    fi
  done

}


#Function to delete ipsec tunnel from QCMAP
#$1 - ike_identifier
#$2 - child_identifier(optional)
function handleDeleteIpsecTunnel() {
  local ike_identifier="$1"
  local child_identifier="$2"
  local children
  local ike
  local current_active_tunnels=0

  local remote_ep_addr=$(uci get ipsec.$ike_identifier.gateway)
  local local_ip=$(uci get ipsec.$ike_identifier.local_leftip)
  local profile_id=$(uci get ipsec.$ike_identifier.profile_id)
  local tunnel_type=$(uci get ipsec.$ike_identifier.tunnel_type)
  local rmnet_name
  local local_addr
  local port_id
  local end_port_id
  local data_ip_type
  local tunnel_ip_type

  #source local ip
  if [ $tunnel_type -eq 1 ] && [ -f /tmp/ipv4config$profile_id ]; then
    . /tmp/ipv4config$profile_id
    rmnet_name=$IFNAME
  fi

  if [ $tunnel_type -eq 2 ] && [ -f /tmp/ipv6config$profile_id ]; then
    . /tmp/ipv6config$profile_id
    rmnet_name=$IFNAME
  fi

  util_ipsec_restart

  if [ -z "$child_identifier" ]; then
    #delete ike tunnel
    swanctl --terminate --ike $ike_identifier&
    #unload routed connection
    ipsec unroute $ike_identifier

    children=$(uci get ipsec.$ike_identifier.tunnel)
    echo "$children" | awk -F" " '{for(i=1; i<=NF; i+=1) print $i}' | while IFS= read -r child_id; do

      #Delete iptable rules based on child_id
      local local_subnet=$(uci get ipsec.$child_id.local_subnet)
      local remote_subnet=$(uci get ipsec.$child_id.remote_subnet)

      # Delete v6 raw prerouting nft rule file & rm nft v6 raw_prerouting DL rule
      util_ipsec_delete_v6_raw_prerouting_rules $ike_identifier $profile_id $child_id 2
      sed -i "/nft insert rule inet fw4 raw_prerouting meta ipsec exists notrack/d" /tmp/data/firewall.user.ipsec

      #Delete iptable rules
      if [ -n "$local_subnet" ]; then
        #For S2S
        data_ip_type=$(util_ip_type $remote_subnet)
      else
        #For H2H
        data_ip_type=$(util_ip_type $remote_ep_addr)
      fi
      tunnel_ip_type=$(util_ip_type $remote_ep_addr)
      echo "rmnet_name:$rmnet_name, data_ip_type: $data_ip_type, tunnel_ip_type: $tunnel_ip_type, remote_ep_addr: $remote_ep_addr"
      util_update_iptable_rules 2 $rmnet_name $data_ip_type $tunnel_ip_type $remote_ep_addr 1

      ipsec unroute $child_id
      uci delete ipsec.$child_id
      current_active_tunnels=$(uci get qcmap_lan.@global[0].ipsec_active_tunnels)
      number=$(echo "$current_active_tunnels" | awk '{print $0+0}')
      uci set qcmap_lan.@global[0].ipsec_active_tunnels=$((number-1))
    done
    uci delete ipsec.$ike_identifier

    # Check if tunnels<0, ike has two children, one is active one is not.
    current_active_tunnels=$(uci get qcmap_lan.@global[0].ipsec_active_tunnels)
    number=$(echo "$current_active_tunnels" | awk '{print $0+0}')
    if [ "$number" -lt 0 ]; then
      uci set qcmap_lan.@global[0].ipsec_active_tunnels='0'
    fi

  else
    #delete child tunnel
    local local_subnet=$(uci get ipsec.$child_identifier.local_subnet)
    local remote_subnet=$(uci get ipsec.$child_identifier.remote_subnet)

    # Delete v6 raw prerouting nft rule file
    util_ipsec_delete_v6_raw_prerouting_rules $ike_identifier $profile_id $child_identifier 2

    #Delete iptable rules
    #For S2S
    data_ip_type=$(util_ip_type $remote_subnet)
    tunnel_ip_type=$(util_ip_type $remote_ep_addr)
    echo "rmnet_name:$rmnet_name, data_ip_type: $data_ip_type, tunnel_ip_type: $tunnel_ip_type, remote_subnet: $remote_subnet"
    util_update_iptable_rules 2 $rmnet_name $data_ip_type $tunnel_ip_type $remote_ep_addr 1

    util_run_command swanctl --terminate --child $child_identifier
    #unload routed connection
    ipsec unroute $child_identifier

    #clean up in UCI DB
    uci delete ipsec.$child_identifier

    #need to del_list
    uci show ipsec | while read -r line; do
      if echo "$line" | grep -q $child_identifier; then
        #echo "$line"
        ike=$(echo "$line" | awk -F"['.=]" '{print $2}')
        uci del_list ipsec.$ike.tunnel=$child_identifier
      fi
    done

    current_active_tunnels=$(uci get qcmap_lan.@global[0].ipsec_active_tunnels)
    number=$(echo "$current_active_tunnels" | awk '{print $0+0}')
    uci set qcmap_lan.@global[0].ipsec_active_tunnels=$((number-1))

    # Check if this is the last child the ike session has, delete the ike session as well
    child_left=$(uci get ipsec.$ike_identifier.tunnel)
    if [ -z "$child_left" ]; then
      swanctl --terminate --ike $ike_identifier&
      ipsec unroute $ike_identifier
      uci delete ipsec.$ike_identifier
      # Delete nft v6 raw_prerouting DL rule when last child has been deleted.
      sed -i "/nft insert rule inet fw4 raw_prerouting meta ipsec exists notrack/d" /tmp/data/firewall.user.ipsec
      util_combine_iptable_rules
      util_run_command /etc/init.d/firewall reload
    fi
  fi

  uci commit ipsec
  uci commit qcmap_lan
  /etc/init.d/swanctl restart
  util_run_command swanctl --load-all

}

#Function to disable ipsec tunnel when wwan datacall disconnected
#$1 - profile_id
#$2 - ip_type
function handledataCallDisconnectedEv() {
  local profile_id="$1"
  local ip_type="$2"
  local local_ip
  local ike_identifier
  local child_identifier
  local section
  local option
  local entry
  local value
  local line
  local local_ip
  local rmnet_name
  local remote_ep_addr
  local children
  local local_subnet
  local remote_subnet
  local local_addr
  local feature_mode=$(uci get qcmap_lan.@global[0].ipsec_enable)

  if [ "$feature_mode" == "0" ]; then
    return
  fi

  uci show ipsec | while read -r line; do
    #echo "$line"

    if [[ "$line" == *"profile_id"* ]]; then
      section=$(echo "$line" | awk -F"['.=]" '{print $1}')
      option=$(echo "$line" | awk -F"['.=]" '{print $2}')
      entry=$(echo "$line" | awk -F"['.=]" '{print $3}')
      value=$(echo "$line" | awk -F"['.=]" '{print $5}')

      #Check if the datacall is on the right profile
      if [ "$entry" == "profile_id" ] && [ "$value" == "$profile_id" ]; then
        ike_identifier="$option"

        remote_ep_addr=$(uci get ipsec.$ike_identifier.gateway)
        local_ip=$(uci get ipsec.$ike_identifier.local_leftip)
        tunnel_type=$(uci get ipsec.$ike_identifier.tunnel_type)

        #source rmnet_name
        if [ $tunnel_type -eq 1 ] && [ -f /tmp/state/network ]; then
          rmnet_name=$(uci_get_state network wan ifname)
        fi

        if [ $tunnel_type -eq 2 ] && [ -f /tmp/state/network ]; then
          rmnet_name=$(uci_get_state network wan_v6 ifname)
        fi

        #If datacall disconnected on the right tunnel type
        if [ "$tunnel_type" == "$ip_type" ]; then

          #Update iptables rules
          children=$(uci get ipsec.$ike_identifier.tunnel)
          echo "$children" | awk -F" " '{for(i=1; i<=NF; i+=1) print $i}' | while IFS= read -r child_id; do
            local local_subnet=$(uci get ipsec.$child_id.local_subnet)
            local remote_subnet=$(uci get ipsec.$child_id.remote_subnet)

            #Delete iptable rules
            if [ -n "$local_subnet" ]; then
              #For S2S
              data_ip_type=$(util_ip_type $remote_subnet)
            else
              #For H2H
              data_ip_type=$(util_ip_type $remote_ep_addr)
            fi
            tunnel_ip_type=$(util_ip_type $remote_ep_addr)
            echo "rmnet_name:$rmnet_name, data_ip_type: $data_ip_type, tunnel_ip_type: $tunnel_ip_type, remote_ep_addr: $remote_ep_addr"
            util_update_iptable_rules 2 $rmnet_name $data_ip_type $tunnel_ip_type $remote_ep_addr 2
          done

          uci delete ipsec.$ike_identifier.local_leftip
          echo "uci commit ipsec"
          uci commit ipsec
          uci commit qcmap_lan
          util_run_command swanctl --terminate --ike $ike_identifier
          /etc/init.d/swanctl restart
          util_run_command swanctl --load-all
        fi
      fi
    fi
  done
}

#Function to get ipsec tunnel configuration
#$1 - ike_identifier
#$2 - child_identifier
function handleGetIpsecTunnelStatus(){
  local ike_identifier="$1"
  local child_identifier="$2"

  output=$(swanctl --list-sas)
  # Prepare env to ensure dir exist
  mkdir -p /tmp/ipsec
  file_path="/tmp/ipsec/ipsec_get_status.txt"

  # Process the output to check for active tunnels
  if echo "$output" | grep -q "$child_identifier:.*INSTALLED"; then
    echo $(basename "$0")":handleGetIpsecTunnelStatus:"$LINENO":child is active" > "$file_path"
    return 1
  else
    if echo "$output" | grep -q "$ike_identifier:.*CONNECTING"; then
      echo $(basename "$0")":handleGetIpsecTunnelStatus:"$LINENO":child is connecting" > "$file_path"
      return 1
    else
      echo $(basename "$0")":handleGetIpsecTunnelStatus:"$LINENO":child is not active" > "$file_path"
      return 0
    fi
  fi

}


#Function to check duplicate remote endpoint address
#$1 remote_ep_addr
function util_ipsec_check_dup_remote_ep_addr(){
  local remote_ep_addr="$1"

  uci show ipsec | while read -r line; do
    if [[ "$line" == *"gateway"* ]]; then
      #ipsec.gwgw.gateway='172.21.150.25'
      section=$(echo "$line" | awk -F"['.=]" '{print $1}')
      option=$(echo "$line" | awk -F"['.=]" '{print $2}')
      entry=$(echo "$line" | awk -F"['.=]" '{print $3}')
      value=$(echo "$line" | awk -F"['=]" '{print $3}')
      #echo "Section: $section, Option: $option, Entry: $entry, Value: $value"

      if [ "$value" == "$remote_ep_addr" ]; then
          echo $(basename "$0")":util_ipsec_check_dup_remote_ep_addr:"$LINENO":duplicate remote_ep_addr"
          return
      fi
    fi
  done

  return 0
}


#Function to check duplicate local subnets
#$1 local_subnet
function util_ipsec_check_dup_local_subnet(){
  local local_subnet="$1"

  uci show ipsec | while read -r line; do
    if [[ "$line" == *"local_subnet"* ]]; then
      #ipsec.gwgw.tunnel='rr'
      section=$(echo "$line" | awk -F"['.=]" '{print $1}')
      option=$(echo "$line" | awk -F"['.=]" '{print $2}')
      entry=$(echo "$line" | awk -F"['.=]" '{print $3}')
      value=$(echo "$line" | awk -F"['=]" '{print $3}')
      #echo "Section: $section, Option: $option, Entry: $entry, Value: $value"

      if [ "$value" == "$local_subnet" ]; then
          echo $(basename "$0")":util_ipsec_check_dup_local_subnet:"$LINENO":duplicate local_subnet"
          return
      fi
    fi
  done
}

#Function to check duplicate remote subnets
#$1 remote_subnet
function util_ipsec_check_dup_remote_subnet(){
  local remote_subnet="$1"

  uci show ipsec | while read -r line; do
    if [[ "$line" == *"remote_subnet"* ]]; then
      #ipsec.gwgw.tunnel='rr'
      section=$(echo "$line" | awk -F"['.=]" '{print $1}')
      option=$(echo "$line" | awk -F"['.=]" '{print $2}')
      entry=$(echo "$line" | awk -F"['.=]" '{print $3}')
      value=$(echo "$line" | awk -F"['=]" '{print $3}')
      #echo "Section: $section, Option: $option, Entry: $entry, Value: $value"

      if [ "$value" == "$remote_subnet" ]; then
          echo $(basename "$0")":util_ipsec_check_dup_remote_subnet:"$LINENO":duplicate remote_subnet"
          return
      fi
    fi
  done
}


#Function to check duplicate child identifier
#$1 child_id
function util_ipsec_check_dup_child_id(){
  local child_id="$1"
  uci show ipsec | while read -r line; do
    if [[ "$line" == *"tunnel"* ]]; then
      #ipsec.gwgw.tunnel='rr'
      section=$(echo "$line" | awk -F"['.=]" '{print $1}')
      option=$(echo "$line" | awk -F"['.=]" '{print $2}')
      entry=$(echo "$line" | awk -F"['.=]" '{print $3}')
      value=$(echo "$line" | awk -F"['.=]" '{print $5}')
      #echo "Section: $section, Option: $option, Entry: $entry, Value: $value"

      if [ "$value" == "$child_id" ]; then
          echo $(basename "$0")":util_ipsec_check_dup_child_id:"$LINENO":duplicate child_identifier"
          return
      fi
    fi
  done
}


# Function to delete v6 raw prerouting nft rules
# $1 ike_identifier
# $2 profile_id
# $3 child_identifier
# $4 operation (1:deactivate, 2: delete)
function util_ipsec_delete_v6_raw_prerouting_rules(){
  local ike_identifier="$1"
  local profile_id="$2"
  local child_identifier="$3"
  local operation="$4"

  # Delete nft v6 raw prerouting rules
  if [ -f /etc/data/nft_v6_raw_prerouting_$ike_identifier\_$profile_id\_$child_identifier ]; then
    cat /etc/data/nft_v6_raw_prerouting_$ike_identifier\_$profile_id\_$child_identifier | while IFS= read -r nft_rule; do
      sed -i "\@$nft_rule@d" /tmp/data/firewall.user.ipsec
    done

    # Reload firewall when deactivate tunnel $child_identifier
    if [ $operation -eq 1 ]; then
      util_combine_iptable_rules
      util_run_command /etc/init.d/firewall reload
    # Remove tunnel rule file when delete tunnel $child_identifier
    elif [ $operation -eq 2 ]; then
      rm /etc/data/nft_v6_raw_prerouting_${ike_identifier}_${profile_id}_${child_identifier}
    fi
  fi
}

# Function to restart /usr/sbin/ipsec
function util_ipsec_restart(){
  # Prepare env to ensure dir exist
  mkdir -p /tmp/ipsec
  if [ ! -f "/tmp/ipsec/ipsec_restart" ]; then
    util_run_command ipsec restart
    # Wait for charon to start and check for 10 seconds
    local counter=0
    while [ ! -f "/var/run/charon.pid" ] && [ $counter -lt 10 ]; do
      sleep 1
      counter=$((counter + 1))
      log $(basename "$0") "util_ipsec_restart" $LINENO "Waiting for charon up...($counter seconds)"
    done
    # Create file when charon successfully starts
    if [ -f "/var/run/charon.pid" ]; then
      log $(basename "$0") "util_ipsec_restart" $LINENO "charon successfully start within $counter seconds!"
      touch /tmp/ipsec/ipsec_restart
    else
      log $(basename "$0") "util_ipsec_restart" $LINENO "charon failed to start within $counter seconds, check /usr/lib/ipsec/charon status!"
    fi
  fi
}

# Variable used by function util_ipsec_split_port_ranges
subnet_setting_string=""

# Function to split port ranges and combine subnet setting string
# This function recursively splits a range of ports into smaller ranges and combines them into a subnet setting string.
# Arguments:
#   $1 - addr: The IP address.
#   $2 - protocol: The protocol (e.g., tcp, udp).
#   $3 - start_port_id: The starting port number.
#   $4 - end_port_id: The ending port number.
# Example:
#   util_ipsec_split_port_ranges 192.168.1.1 tcp 5000 5010
#   Resulting subnet_setting_string: "192.168.1.1[tcp/5000-5007],192.168.1.1[tcp/5008-5009],192.168.1.1[tcp/5010]"
function util_ipsec_split_port_ranges() {
  local addr=$1
  local protocol=$2
  local start_port_id=$3
  local end_port_id=$4

  # Stop recursion if the start port is greater than the end port
  if [ $start_port_id -gt $end_port_id ]; then
    return
  fi

  # If the start port is equal to the end port, add it to the subnet setting string
  if [ $start_port_id -eq $end_port_id ]; then
    if [ "$subnet_setting_string" != "" ]; then
      subnet_setting_string="$subnet_setting_string,$addr[$protocol/$start_port_id]"
    else
      subnet_setting_string="$addr[$protocol/$start_port_id]"
    fi
    return
  fi

  # Fast-check: Only check for even start_port_id.
  if [ $((start_port_id % 2)) -eq 0 ]; then
    # Find stop position (bit) starting from 0x8000 (32768).
    # Example:
    #   start_port_id = 1100 0000
    #   end_port_id   = 1111 1111
    #   stop_pos      = 0100 0000
    local stop_pos=32768
    while [ $((stop_pos & (start_port_id ^ end_port_id))) -eq 0 ]; do
      stop_pos=$((stop_pos >> 1))
    done

    # Find mid port, split to [start_port_id, mid_port_id], and recursively call util_ipsec_split_port_ranges(mid_port_id+1, end_port_id)
    local mid_port_id=$start_port_id
    local bit=1
    while [ $bit -le $stop_pos ]; do
      # Stop at the first 1 that appears at the lowest position of start_port_id
      # Number in after-split port range should be: 1,2,4,8,...2^n
      # Example:
      #   start_port_id = 1100 0000
      #   stop_pos      = 0100 0000
      #   mid_port_id   = 1101 1111
      if [ $((mid_port_id & bit)) -ne 0 ]; then
        break
      else
        mid_port_id=$((mid_port_id + bit))
      fi
      bit=$((bit << 1))
    done

    if [ $mid_port_id -gt $end_port_id ]; then
      mid_port_id=$((mid_port_id - stop_pos))
    fi

    if [ "$subnet_setting_string" != "" ]; then
      subnet_setting_string="$subnet_setting_string,$addr[$protocol/$start_port_id-$mid_port_id]"
    else
      subnet_setting_string="$addr[$protocol/$start_port_id-$mid_port_id]"
    fi
    util_ipsec_split_port_ranges $addr $protocol $((mid_port_id + 1)) $end_port_id
  # Directly setting for odd start_port_id
  else
    if [ "$subnet_setting_string" != "" ]; then
      subnet_setting_string="$subnet_setting_string,$addr[$protocol/$start_port_id]"
    else
      subnet_setting_string="$addr[$protocol/$start_port_id]"
    fi
    util_ipsec_split_port_ranges $addr $protocol $((start_port_id + 1)) $end_port_id
  fi
}

case $1 in
 setIpsecTunnel)
   handleSetIpsecTunnel $2 $3 $4 $5 $6 $7 $8 $9 ${10} ${11} ${12} ${13} ${14} ${15} ${16} ${17} ${18} ${19} ${20} ${21}
   ;;
 activateIpsecTunnel)
   handleActivateIpsecTunnel $2 $3
   ;;
 deactivateIpsecTunnel)
   handleDeactivateIpsecTunnel $2 $3
   ;;
 dataCallConnectedEv)
   handleDataCallConnectedEv $2 $3
   ;;
 deleteIpsecTunnel)
   handleDeleteIpsecTunnel $2 $3
   ;;
 dataCallDisconnectedEv)
   handledataCallDisconnectedEv $2 $3
   ;;
 getIpsecTunnelStatus)
   handleGetIpsecTunnelStatus $2 $3
   ;;
 checkdupremoteepaddr)
   util_ipsec_check_dup_remote_ep_addr $2
   ;;
 checkduplocalsubnet)
   util_ipsec_check_dup_local_subnet $2
   ;;
 checkdupremotesubnet)
   util_ipsec_check_dup_remote_subnet $2
   ;;
 checkdupchildid)
   util_ipsec_check_dup_child_id $2
   ;;
 *)
   echo "Invalid option"
   ;;
esac