#!/bin/sh
#  Copyright (c) 2022 Qualcomm Technologies, Inc.
#  All Rights Reserved.
#  Confidential and Proprietary - Qualcomm Technologies, Inc.
#  Usage: nat_alg_vpn_config.sh <functions> <arg1> <arg2> <...>
#  NAT Type Representation
#  0:SYMMETRIC NAT
#  1:PORT RESTRICTED CONE NAT
#  2:FULL CONE NAT
#  3:ADDRESS RESTRICTED CONE NAT

#define global variabels for Port Trigger module
iptables_port_trigger_ingress_cmd=""
iptables_port_trigger_engress_cmd=""

. /lib/functions.sh
. /lib/functions/network.sh
. /lib/netifd/netifd-proto.sh
. /etc/data/lanUtils.sh
. /etc/data/firewallConfig.sh


if [[ "$#" -le 0 ]]; then
  echo "Check usage nat_alg_vpn_config.sh <functions> <arg1> <arg2> <...>"
  return 0
fi

function NAT_LOG()
{
  logger -t "QCMAP" "$@"
}

function del_port_forward() {
  idx=0
  found=0
  if [ "$1" = "icmp" ]; then
    profile_handle=$3
  else
    profile_handle=$5
  fi

  profile_idx=`uci show qcmap_lan | grep  "profile_id='$profile_handle'" | awk -F'[][]' '{print $2}'`
  snat_count=`uci get qcmap_lan.@profile[$profile_idx].no_of_snat_rules`
  no_of_redirects=$(uci get qcmap_lan.@no_of_configs[0].no_of_redirects)

  if [ $snat_count -eq 0 ]; then
    echo "No port forwarding config present"
    return
  else
    idx=$no_of_redirects
  fi

  #check if whether Static NAT Entry is present
  snat_rules_index=$(uci show firewall | grep -i "StaticNAT-$profile_handle" | awk -F'[][]' '{print $2}')
  for i in $snat_rules_index
  do
    proto=$(uci get firewall.@redirect[$i].proto)
    dest_ip=$(uci get firewall.@redirect[$i].dest_ip)
    if [ "$1" = "icmp" ]; then
      if [ "$1" = "$proto" ] && [ "$2" = "$dest_ip" ]; then
        found=1
        idx=$i
        snat_conntrack_del_command="conntrack -D --proto icmp --reply-src $dest_ip"
        break
      fi
    else
      dest_port=$(uci get firewall.@redirect[$i].dest_port)
      src_dport=$(uci get firewall.@redirect[$i].src_dport)
      if [ "$1" = "$proto" ] && [ "$2" = "$dest_ip" ] && [ "$3" = "$dest_port" ] && [ "$4" = "$src_dport" ]; then
        found=1
        idx=$i
        if [ "$proto" = "tcpudp" ];then
          snat_conntrack_del_command1="conntrack -D --proto tcp --reply-src $dest_ip --dport $src_dport --reply-port-src $dest_port"
          snat_conntrack_del_command2="conntrack -D --proto udp --reply-src $dest_ip --dport $src_dport --reply-port-src $dest_port"
        else
          snat_conntrack_del_command="conntrack -D --proto $proto --reply-src $dest_ip --dport $src_dport --reply-port-src $dest_port"
        fi
        break
      fi
    fi
  done

  if [ $found -eq 0 ]; then
    echo "Port Forward entry not found"
    return
  fi
  uci delete firewall.@redirect[$idx]
  #updated redirect count
  count=$((no_of_redirects-1))
  uci set qcmap_lan.@no_of_configs[0].no_of_redirects="$count"
  new_snat_count=$((snat_count-1))
  uci set qcmap_lan.@profile[$profile_idx].no_of_snat_rules=$new_snat_count
  uci commit
  /etc/init.d/firewall reload
  if [ "$proto" = "tcpudp" ];then
      eval $snat_conntrack_del_command1
      eval $snat_conntrack_del_command2
  else
      eval $snat_conntrack_del_command
  fi
}

function add_port_forward() {
  idx=0
  found=0
  no_of_redirects=$(uci get qcmap_lan.@no_of_configs[0].no_of_redirects)
  idx=$no_of_redirects
  if [ "$1" = "icmp" ]; then
    profile_handle=$3
  else
    profile_handle=$5
  fi

  profile_idx=`uci show qcmap_lan | grep  "profile_id='$profile_handle'" | awk -F'[][]' '{print $2}'`

  snat_count=`uci get qcmap_lan.@profile[$profile_idx].no_of_snat_rules`

  #check if already Static NAT entry is present for the PDN
  snat_rules_index=$(uci show firewall | grep -i "StaticNAT-$profile_handle" | awk -F'[][]' '{print $2}')

  if [ "$snat_rules_index" ];then
    for i in $snat_rules_index
    do
      proto=$(uci get firewall.@redirect[$i].proto)
      dest_ip=$(uci get firewall.@redirect[$i].dest_ip)
      if [ "$1" = "icmp" ]; then
        if [ "$1" = "$proto" ]; then
          found=1
          break
        fi
      else
        dest_port=$(uci get firewall.@redirect[$i].dest_port)
        src_dport=$(uci get firewall.@redirect[$i].src_dport)
        if [ "$1" = "$proto" ] && [ "$3" = "$dest_port" ] && [ "$4" = "$src_dport" ]; then
          found=1
          break
        fi
      fi
    done

    if [ $found -eq 1 ]; then
      echo "Port Forward rule already exist"
      return
    fi
  fi

  #Add rules for ICMP
  if [ "$1" = 'icmp' ]; then
    uci add firewall redirect
    uci set firewall.@redirect[$idx]=redirect
    if [ $3 -eq 1 ]; then
      uci set firewall.@redirect[$idx].src='wan'
    else
      uci set firewall.@redirect[$idx].src=wan$profile_handle
    fi
    uci set firewall.@redirect[$idx].name=StaticNAT-$profile_handle
    uci set firewall.@redirect[$idx].target='DNAT'
    uci set firewall.@redirect[$idx].proto="$1"
    uci set firewall.@redirect[$idx].dest_ip="$2"
    uci set firewall.@redirect[$idx].reflection='0'
  else
    #Add rule for TCP/UDP
    uci add firewall redirect
    uci set firewall.@redirect[$idx]=redirect
    if [ $5 -eq 1 ]; then
      uci set firewall.@redirect[$idx].src='wan'
    else
      uci set firewall.@redirect[$idx].src=wan$profile_handle
    fi
    uci set firewall.@redirect[$idx].name=StaticNAT-$profile_handle
    uci set firewall.@redirect[$idx].target='DNAT'
    uci set firewall.@redirect[$idx].proto="$1"
    uci set firewall.@redirect[$idx].dest_ip="$2"
    uci set firewall.@redirect[$idx].dest_port="$3"
    uci set firewall.@redirect[$idx].src_dport="$4"
    uci set firewall.@redirect[$idx].reflection='0'
  fi

  #Shifting SNAT rule to 0th index
  uci reorder firewall.@redirect[$idx]='0'

  #Getting the previous SNAT Count from qcmap_lan
  prev_snat_count=`uci get qcmap_lan.@profile[$profile_idx].no_of_snat_rules`
  new_snat_count=$((prev_snat_count+1))
  uci set qcmap_lan.@profile[$profile_idx].no_of_snat_rules=$new_snat_count

  #updated redirect count
  count=$((idx+1))
  uci set qcmap_lan.@no_of_configs[0].no_of_redirects="$count"
  uci commit
  /etc/init.d/firewall reload
}

#input:
# $1 file name of ipv4config
# $2 type_of_nat
# $3 zone_wan_indx
function flush_firewall_user_one_backhaul()
{
  ipv4config_file=$1
  type_of_nat=$2
  zone_wan_index=$3

         DEVNAME=`awk -F "=" 'FNR == 1 {print $2}' $ipv4config_file | tr -d '"'`
  PUBLIC_IP_ADDR=`awk -F "=" 'FNR == 2 {print $2}' $ipv4config_file | tr -d '"'`

  checkIP=`ip addr show $DEVNAME | grep $PUBLIC_IP_ADDR`

  if [ ! -n "$checkIP" ]; then
    NAT_LOG "$DEVNAME not up since addr $PUBLIC_IP_ADDR not present in device"
    return
  fi

  NAT_LOG "interface is up with devname=$DEVNAME, addr=$PUBLIC_IP_ADDR"

  case $type_of_nat in
    "SYM")
      uci set firewall.@zone[$zone_wan_index].masq='1'
      NAT_LOG "enable nat for SYM"
      ;;
    "PORT_REST_CONE")
      uci set firewall.@zone[$zone_wan_index].masq='0'
      echo "iptables -t nat -I POSTROUTING -o $DEVNAME -j SNAT --to-source $PUBLIC_IP_ADDR"    >> /etc/firewall.user.nat
      NAT_LOG "enable PORT_REST_CONE nat per file $ipv4config_file"
      ;;
    "FULL_CONE")
      uci set firewall.@zone[$zone_wan_index].masq='0'
      echo "iptables -t nat -I PREROUTING  -i $DEVNAME -j NATTYPE --mode dnat --type 1"        >> /etc/firewall.user.nat
      echo "iptables -t nat -I POSTROUTING -o $DEVNAME -j NATTYPE --mode forward_out --type 1" >> /etc/firewall.user.nat
      echo "iptables -t nat -I POSTROUTING -o $DEVNAME -j MASQUERADE --random"                 >> /etc/firewall.user.nat
      NAT_LOG "enable FULL_CONE nat per file $ipv4config_file"
      ;;
    "ADDR_REST_CONE")
      uci set firewall.@zone[$zone_wan_index].masq='0'
      echo "iptables -t nat -I PREROUTING  -i $DEVNAME -j NATTYPE --mode dnat --type 2"        >> /etc/firewall.user.nat
      echo "iptables -t nat -I POSTROUTING -o $DEVNAME -j NATTYPE --mode forward_out --type 2" >> /etc/firewall.user.nat
      echo "iptables -t nat -I POSTROUTING -o $DEVNAME -j MASQUERADE --random"                 >> /etc/firewall.user.nat
      NAT_LOG "enable ADDR_REST_CONE nat per file $ipv4config_file"
      ;;
    *)
      echo "Not a valid argument"
      break
      ;;
  esac

  util_combine_iptable_rules
}

# enable_nat_type
# $1 the profile that to enable nat
#   value: -1,   means enable nat for default backhaul
#   value: [1,n] means enable nat for special profile,
#                it can be default profile(backhaul) or on-demand profile
# Note:
#   for default backhual, per the backhaul switch setting, loop the backhaul,
#     check the file /tmp/ipv4config.<backhaul-name> to know which backhaul is up,
#     and confirm again through the ip addr command
#     then set the NAT for this active backhaul
# Require:
#   whenever backhaul switch setting change, need run it with profile=-1
#   whenever backhaul connected/disconnected, from wlan/eth/..., need run it with profile=-1
function enable_nat_type() {
  no_of_wan_profiles=$(uci get qcmap_lan.@no_of_configs[0].no_of_profiles)
  local idx defaultProfileId profileId
  local ifacename DEVNAME PUBLIC_IP_ADDR
  ifacename=""
  DEVNAME=""
  PUBLIC_IP_ADDR=""

  defaultProfileId=`uci get qcmap_lan.@no_of_configs[0].default_pdn`
  defaultNATType=""
  defaultZoneWanIndex=""

  #cleanup nat rules before adding new ones
  util_cleanup_iptables_rules /etc/firewall.user.nat I
  cat /dev/null > /etc/firewall.user.nat

  #Set NAT for WWAN
  for i in $(seq $no_of_wan_profiles)
  do
    profile_id=$(uci get qcmap_lan.@profile[$((i-1))].profile_id)
    type_of_nat=$(uci get qcmap_lan.@profile[$((i-1))].nat_type)
    idx=$((i-1))

    if [ $profile_id -eq 1 ]; then
        wan_zone_name="wan"
    else
        wan_zone_name="wan$1"
    fi

    if [ $profile_id -eq $defaultProfileId ]; then
      defaultNATType=$type_of_nat
    fi

    zone_wan_index=`uci show firewall |grep "zone" | grep -w "name='$wan_zone_name'" | awk -F'[][]' '{print $2}'`
    if [ $profile_id -eq $defaultProfileId ]; then
      defaultZoneWanIndex=$zone_wan_index
    fi

    #Check if IPPT is enabled on the PDN
    local is_ippt_active=$(uci -q get qcmap_lan.@profile[$idx].active_ippt)
    if [ -n "$is_ippt_active" ]; then
      if [ $is_ippt_active -eq 1 ]; then
        continue
      fi
    fi

    file=/tmp/ipv4config$profile_id
    if [ -f "$file" ]; then
      flush_firewall_user_one_backhaul $file $type_of_nat $zone_wan_index
    else
      NAT_LOG "Profile profile_id not up"
    fi

  done


  #Set NAT for non-WWAN
  backhaullist=$(uci get mwan3.backhaul_pref.use_member)
  NAT_LOG "loop backhaullist $backhaullist"
  if [ ! -z "$backhaullist" ]; then
    for backhaulmember in $backhaullist
    do
       backhaul=$(uci get mwan3.$backhaulmember.interface)
       if [ $backhaul == "wan" ]; then
          continue
       else
          file=/tmp/ipv4config.$backhaul
       fi

       if [ -f $file ]; then
          flush_firewall_user_one_backhaul $file $defaultNATType $defaultZoneWanIndex
       else
        NAT_LOG "$backhaul profile_id not up since $file not present"
      fi
    done
  else
       NAT_LOG "get backhaul_pref.use_member failed:$backhaullist"
  fi

  # Add rules to iptables
  util_add_iptables_rules /etc/firewall.user.nat

  uci commit firewall

  /etc/init.d/firewall reload
  echo "NAT Type Set Successfully"

  NAT_LOG "enable nat done"

}

function set_nat_type() {
  local profile_index=`mbb_util_get_uci_profile_index $2`
  uci set qcmap_lan.@profile[$profile_index].nat_type=$1
  uci commit

  file=/tmp/ipv4config$2
  if [ -f "$file" ]
  then
    enable_nat_type $2
  else
    echo "Backhaul is not Up"
    echo
  fi
}

function get_nat_type() {
  local profile_index=`mbb_util_get_uci_profile_index $1`
  return $profile_index
}

function add_dmz() {
  idx=$(uci get qcmap_lan.@no_of_configs[0].no_of_redirects)
  uci add firewall redirect
  uci set firewall.@redirect[$idx]=redirect
  if [ $1 -eq 1 ]; then
    uci set firewall.@redirect[$idx].src='wan'
  else
    uci set firewall.@redirect[$idx].src=wan$1
  fi

  uci set firewall.@redirect[$idx].dest_ip="$2"
  uci set firewall.@redirect[$idx].proto='all'
  uci set firewall.@redirect[$idx].target='DNAT'
  uci set firewall.@redirect[$idx].name="$1"
  uci set firewall.@redirect[$idx].reflection='0'
  count=$((idx+1))
  uci set qcmap_lan.@no_of_configs[0].no_of_redirects="$count"
  uci commit
  /etc/init.d/firewall reload
  return
}

function delete_dmz() {
    #Deleting config redirect
    local no_of_redirect=$(uci get qcmap_lan.@no_of_configs[0].no_of_redirects)
    for i in $(seq $no_of_redirect)
    do
      name=$(uci get firewall.@redirect[$((i-1))].name)
      if [ "$name" = "$1" ];then
        idx=$((i-1))
        dest_ip=$(uci get firewall.@redirect[$idx].dest_ip)
        uci del firewall.@redirect[$idx]
        count=$((no_of_redirect-1))
        uci set qcmap_lan.@no_of_configs[0].no_of_redirects="$count"
        uci commit
        /etc/init.d/firewall reload
        #Deleting conntrack based on client ip when dmz rule gets deleted
        conntrack -D --reply-src $dest_ip
        return
      fi
    done
}

function rtsp_alg_config() {
  local action=$1

  #Install RTSP ALG Kernel Module
  curr_kernel_version=$(uname -r | awk '{print $1}' | cut -d '-' -f 1)
  check_curr_mod=`lsmod | grep nf_nat_rtsp`

  if [ $action == "enable" ] && [ "$check_curr_mod" == "" ] ; then
    insmod /lib/modules/rtsp-alg/nf_conntrack_rtsp.ko
    insmod /lib/modules/rtsp-alg/nf_nat_rtsp.ko
    if [ $? -eq 0 ]; then
      NAT_LOG "succesfully insert nf_conntrack_rtsp.ko & nf_nat_rtsp.ko"
    else
      NAT_LOG "Fail to insert nf_conntrack_rtsp.ko & nf_nat_rtsp.ko"
    fi
  elif [ $action == "disable" ] && [ "$check_curr_mod" != "" ] ; then
    # Need to stop firewall first to remove nf_conntrack_rtsp.ko
    /etc/init.d/firewall stop
    rmmod /lib/modules/rtsp-alg/nf_nat_rtsp.ko
    rmmod /lib/modules/rtsp-alg/nf_conntrack_rtsp.ko
    /etc/init.d/firewall start
    /etc/init.d/mwan3 restart
    if [ $? -eq 0 ]; then
      NAT_LOG "succesfully remove nf_conntrack_rtsp.ko & nf_nat_rtsp.ko"
    else
      NAT_LOG "Fail to remove nf_conntrack_rtsp.ko & nf_nat_rtsp.ko"
    fi
  fi
}

function sip_alg_config() {
  local action=$1
  if [ $action == "enable" ] ; then
    echo 0 > /proc/sys/net/netfilter/nf_conntrack_disable_sip_alg
    #Enable media streaming between non signalling endpoints
    echo 0 > /proc/sys/net/netfilter/nf_conntrack_sip_direct_media
    NAT_LOG "successfully enable SIP ALG"
  elif [ $action == "disable" ] ; then
    echo 1 > /proc/sys/net/netfilter/nf_conntrack_disable_sip_alg
    #Disable media streaming between non signalling endpoints
    echo 1 > /proc/sys/net/netfilter/nf_conntrack_sip_direct_media
    NAT_LOG "successfully disable SIP ALG"
  fi
}

function set_sip_server() {
  local sip_serv_type=$1
  local sip_server_info=$2
  QCMAP_CM_SIP_SERVER_ADDR=1
  QCMAP_CM_SIP_SERVER_FQDN=2

  curr_opt_config=`uci get qcmap_lan.@alg_cfg[0].default_sip_server_config`
  dhcp_config="120,"$sip_server_info

#set sip server info as dhcp option 120 starts, if already set, delete first
  if [ "$curr_opt_config" != "" ] ; then
    uci del_list dhcp.lan.dhcp_option_force="120,"$curr_opt_config
    NAT_LOG "already with dhcp 120 config, delete first"
  fi
  uci add_list dhcp.lan.dhcp_option_force=$dhcp_config
  NAT_LOG "dhcp 120 config addlist successfully"

#save sip server config in qcmap_lan, two types IP/FQDN
  if [ $sip_serv_type == $QCMAP_CM_SIP_SERVER_ADDR ] ; then
    uci set qcmap_lan.@alg_cfg[0].default_sip_server_config_type='IP'
    uci set qcmap_lan.@alg_cfg[0].default_sip_server_config=$sip_server_info
  elif [ $sip_serv_type == $QCMAP_CM_SIP_SERVER_FQDN ] ; then
    uci set qcmap_lan.@alg_cfg[0].default_sip_server_config_type='FQDN'
    uci set qcmap_lan.@alg_cfg[0].default_sip_server_config=$sip_server_info
  fi

  uci commit qcmap_lan
  uci commit dhcp
}

function get_sip_server_info() {
  curr_opt_config=`uci get qcmap_lan.@alg_cfg[0].default_sip_server_config`
  NAT_LOG "curr_sip_server_info: $curr_opt_config"
  echo "$curr_opt_config"
}

#Obtain network sip server info from /tmp/data/network_sip_server_info
function update_network_sip_server() {
  local file_path=/tmp/data/network_sip_server_info

  if [ -f $file_path ] && [ -s $file_path ]; then
    total_count=`awk -F'v4_total_count: ' 'BEGIN{ORS=" "}{print $2}' $file_path`
    for i in `seq 1 $total_count`
      do
        single_sip_server_type=`awk -F"network_sip_server_info$i.type= " 'BEGIN{ORS=" "}{print $2}' $file_path`
        if [ $single_sip_server_type == "IP" ]; then
         single_pcscf_address=`awk -F"network_sip_server_info$i.PCSCF_address= " 'BEGIN{ORS=" "}{print $2}' $file_path`
         update_network_sip_server_to_dhcp $single_pcscf_address
        elif [ $single_sip_server_type == "FQDN" ]; then
         single_pcscf_fqdn=`awk -F"network_sip_server_info$i.PCSCF_FQDN= " 'BEGIN{ORS=" "}{print $2}' $file_path`
         update_network_sip_server_to_dhcp $single_pcscf_fqdn
        fi
      done
  fi

  /etc/init.d/dnsmasq reload
}

function update_network_sip_server_to_dhcp() {
  local network_dhcp_config="120,"$1
  uci add_list dhcp.lan.dhcp_option_force=$network_dhcp_config
  uci commit dhcp

  NAT_LOG "successfully update network sip server to dhcp"
}

function generate_iptable_command {
  if [ $1 -eq $2 ] && [ $3 -eq $4 ];then
    iptables_port_trigger_ingress_cmd="iptables -t nat -A PREROUTING -i $8 -j TRIGGER --mode dnat -p $6 --dport $3 --trigger-proto $5 --trigger-port $1 --forward-proto $6 --forward-ports $3"
    iptables_port_trigger_engress_cmd="iptables -t nat -A POSTROUTING -o $8 -j TRIGGER --mode forward_out -p $5 --dport $1 --trigger-proto $5 --trigger-port $1 --forward-proto $6 --forward-ports $3 --timer $7"
  elif [ $1 -eq $2 ] && [ $3 -ne $4 ];then
    iptables_port_trigger_ingress_cmd="iptables -t nat -A PREROUTING -i $8 -j TRIGGER --mode dnat -p $6 --dport $3:$4 --trigger-proto $5 --trigger-port $1 --forward-proto $6 --forward-ports $3-$4"
    iptables_port_trigger_engress_cmd="iptables -t nat -A POSTROUTING -o $8 -j TRIGGER --mode forward_out -p $5 --dport $1 --trigger-proto $5 --trigger-port $1 --forward-proto $6 --forward-ports $3-$4 --timer $7"
  elif [ $1 -ne $2 ] && [ $3 -eq $4 ];then
    iptables_port_trigger_ingress_cmd="iptables -t nat -A PREROUTING -i $8 -j TRIGGER --mode dnat -p $6 --dport $3 --trigger-proto $5 --trigger-port $1-$2 --forward-proto $6 --forward-ports $3"
    iptables_port_trigger_engress_cmd="iptables -t nat -A POSTROUTING -o $8 -j TRIGGER --mode forward_out -p $5 --dport $1:$2 --trigger-proto $5 --trigger-port $1-$2 --forward-proto $6 --forward-ports $3 --timer $7"
  else
    iptables_port_trigger_ingress_cmd="iptables -t nat -A PREROUTING -i $8 -j TRIGGER --mode dnat -p $6 --dport $3:$4 --trigger-proto $5 --trigger-port $1-$2 --forward-proto $6 --forward-ports $3-$4"
    iptables_port_trigger_engress_cmd="iptables -t nat -A POSTROUTING -o $8 -j TRIGGER --mode forward_out -p $5 --dport $1:$2 --trigger-proto $5 --trigger-port $1-$2 --forward-proto $6 --forward-ports $3-$4 --timer $7"
  fi
}

function add_port_trigger {
  idx=0
  #Updating qcmap_lan database
  no_of_port_trigger_info=$(uci get qcmap_lan.@no_of_configs[0].no_of_port_trigger_info)
  idx=$no_of_port_trigger_info
  uci add qcmap_lan port_trigger_info
  uci set qcmap_lan.@port_trigger_info[$idx].trigger_start_port="$1"
  uci set qcmap_lan.@port_trigger_info[$idx].trigger_end_port="$2"
  uci set qcmap_lan.@port_trigger_info[$idx].forward_start_port="$3"
  uci set qcmap_lan.@port_trigger_info[$idx].forward_end_port="$4"
  if [ $5 -eq 6 ];then
    uci set qcmap_lan.@port_trigger_info[$idx].trigger_protocol='tcp'
    tri_proto='tcp'
  fi
  if [ $5 -eq 17 ];then
    uci set qcmap_lan.@port_trigger_info[$idx].trigger_protocol='udp'
    tri_proto='udp'
  fi
  if [ $6 -eq 6 ];then
    uci set qcmap_lan.@port_trigger_info[$idx].forward_protocol='tcp'
    forw_proto='tcp'
  fi
  if [ $6 -eq 17 ];then
    uci set qcmap_lan.@port_trigger_info[$idx].forward_protocol='udp'
    forw_proto='udp'
  fi
  uci set qcmap_lan.@port_trigger_info[$idx].timer="$7"
  uci set qcmap_lan.@port_trigger_info[$idx].handle="$8"
  count=$(( idx + 1 ))
  uci set qcmap_lan.@no_of_configs[0].no_of_port_trigger_info="$count"
  uci commit qcmap_lan
  #Updating iptables rules
  file=/tmp/ipv4config$9
  DEVNAME=`awk -F "=" 'FNR == 1 {print $2}' $file | tr -d '"'`
  generate_iptable_command $1 $2 $3 $4 $tri_proto $forw_proto $7 $DEVNAME
  # Cleanup old rules in iptables
  util_cleanup_iptables_rules /etc/firewall.user.porttrigger A
  echo $iptables_port_trigger_ingress_cmd >> /etc/firewall.user.porttrigger
  echo $iptables_port_trigger_engress_cmd >> /etc/firewall.user.porttrigger
  # Add rules to iptables
  util_add_iptables_rules /etc/firewall.user.porttrigger
  util_combine_iptable_rules
  /etc/init.d/firewall reload
  return
}

function delete_port_trigger {
  found=0
  port_trigger_idx=0
  no_of_port_trigger_info=$(uci get qcmap_lan.@no_of_configs[0].no_of_port_trigger_info)
  for i in $(seq $no_of_port_trigger_info)
    do
          handle=$(uci get qcmap_lan.@port_trigger_info[$((i-1))].handle)
          if [ "$handle" = "$1" ]; then
            port_trigger_idx=$((i-1))
            file=/tmp/ipv4config$2
            #Updating iptables rules
            DEVNAME=`awk -F "=" 'FNR == 1 {print $2}' $file | tr -d '"'`
            trigger_start_port=$(uci get qcmap_lan.@port_trigger_info[$port_trigger_idx].trigger_start_port)
            trigger_end_port=$(uci get qcmap_lan.@port_trigger_info[$port_trigger_idx].trigger_end_port)
            forward_start_port=$(uci get qcmap_lan.@port_trigger_info[$port_trigger_idx].forward_start_port)
            forward_end_port=$(uci get qcmap_lan.@port_trigger_info[$port_trigger_idx].forward_end_port)
            trigger_protocol=$(uci get qcmap_lan.@port_trigger_info[$port_trigger_idx].trigger_protocol)
            forward_protocol=$(uci get qcmap_lan.@port_trigger_info[$port_trigger_idx].forward_protocol)
            timer=$(uci get qcmap_lan.@port_trigger_info[$port_trigger_idx].timer)
            generate_iptable_command $trigger_start_port $trigger_end_port $forward_start_port $forward_end_port $trigger_protocol $forward_protocol $timer $DEVNAME
            sed -i  "/$iptables_port_trigger_ingress_cmd/d" /etc/firewall.user.porttrigger
            $(echo $iptables_port_trigger_ingress_cmd | sed "s/-A/-D/")
            sed -i  "/$iptables_port_trigger_engress_cmd/d" /etc/firewall.user.porttrigger
            $(echo $iptables_port_trigger_engress_cmd | sed "s/-A/-D/")
            #Updating qcmap_lan database
            uci del qcmap_lan.@port_trigger_info[$port_trigger_idx]
            count=$((no_of_port_trigger_info-1))
            uci set qcmap_lan.@no_of_configs[0].no_of_port_trigger_info="$count"
            uci commit qcmap_lan
            util_combine_iptable_rules
          fi
        done
  return
}

# Enable VPN Passthrough
# $1 rule name
# $2 wan section name
# $3 protocol name
# $4 protocol family
# $5 Action-> Enable/Disable
setup_vpnpt() {
    local name=$1
    local public_interface_name=$2
    local proto=$3
    local family=$4
    local action=$5

    if [ "$action" == "enable" ]; then
      if [ "$public_interface_name" ]; then
        if [ "$family" == "ipv4" ];then
          echo "iptables -t filter -D FORWARD -i $public_interface_name -p $proto -j DROP"  >> /etc/firewall.user.nat
          echo "iptables -t filter -D FORWARD -i $public_interface_name -p $proto -j ACCEPT"  >> /etc/firewall.user.nat
          echo "iptables -t filter -I FORWARD -i $public_interface_name -p $proto -j ACCEPT"  >> /etc/firewall.user.nat
        elif [ "$family" == "ipv6" ];then
          echo "ip6tables -t filter -D FORWARD -i $public_interface_name -p $proto -j DROP"  >> /etc/firewall.user.nat
          echo "ip6tables -t filter -D FORWARD -i $public_interface_name -p $proto -j ACCEPT"  >> /etc/firewall.user.nat
          echo "ip6tables -t filter -I FORWARD -i $public_interface_name -p $proto -j ACCEPT"  >> /etc/firewall.user.nat
        fi
      fi
    elif [ "$action" == "disable" ]; then
      if [ "$public_interface_name" ]; then
        if [ "$family" == "ipv4" ];then
          echo "iptables -t filter -D FORWARD -i $public_interface_name -p $proto -j ACCEPT"  >> /etc/firewall.user.nat
          echo "iptables -t filter -D FORWARD -i $public_interface_name -p $proto -j DROP"  >> /etc/firewall.user.nat
          echo "iptables -t filter -I FORWARD -i $public_interface_name -p $proto -j DROP"  >> /etc/firewall.user.nat
        elif [ "$family" == "ipv6" ];then
          echo "ip6tables -t filter -D FORWARD -i $public_interface_name -p $proto -j ACCEPT"  >> /etc/firewall.user.nat
          echo "ip6tables -t filter -D FORWARD -i $public_interface_name -p $proto -j DROP"  >> /etc/firewall.user.nat
          echo "ip6tables -t filter -I FORWARD -i $public_interface_name -p $proto -j DROP"  >> /etc/firewall.user.nat
        fi
      fi
    fi

    util_combine_iptable_rules

    /etc/init.d/firewall restart

    # quectel add to cleanup firewall.user.nat to prevent file to large
    cat /dev/null > /etc/firewall.user.nat
}

# Find wan section name
# $1 interface section name
# $2 profile handle
cfg=
find_cfg() {
    local interface=$1
    local profile proto
    config_get proto $interface "proto"
    config_get profile $interface "profile"
    if [ "$proto" = "rmnet" ] && [ "$profile" = "$2" ]; then
        cfg=$interface
    fi
}

#Function to enable/disable VPN Passthrough
# $1 - profile handle
# $2 - index of profile section in qcmap_lan db
# $3 - enable_state
# $4 - VPN Passthrough type
function set_vpn_passthrough() {
  local profile_handle=$1
  local profile_idx=$2
  local enable_state=$3
  local vpnpt_type=$4
  local downstream
  local firewallrule_name firewallrule_proto proto_family

  config_load network
  config_foreach find_cfg interface $profile_handle

  [ -z "$cfg" ] && {
    echo $(basename "$0")":set_vpn_passthrough:"$LINENO":No cfg found for the profile $profile_handle"
    return
  }

  if [ "$vpnpt_type" = "ipsecptv4" ] || [ "$vpnpt_type" = "l2tp_ipsecptv4" ]; then
    # Set ipsecpt_enable parameter in qcmap_lan db to enable_state value
    uci set qcmap_lan.@profile[$profile_idx].ipsecpt_enable="$enable_state"
    uci commit qcmap_lan
    firewallrule_name=Allow-ESP-V4-Passthrough
    firewallrule_proto=esp
    proto_family=ipv4
  elif [ "$vpnpt_type" = "ipsecptv6" ] || [ "$vpnpt_type" = "l2tp_ipsecptv6" ]; then
    # Set ipsecptv6_enable parameter in qcmap_lan db to enable_state value
    uci set qcmap_lan.@profile[$profile_idx].ipsecptv6_enable="$enable_state"
    uci commit qcmap_lan
    firewallrule_name=Allow-ESP-V6-Passthrough
    firewallrule_proto=esp
    proto_family=ipv6
  elif [ "$vpnpt_type" = "pptpptv4" ]; then
    # Set pptppt_enable parameter in qcmap_lan db to enable_state value
    uci set qcmap_lan.@profile[$profile_idx].pptppt_enable="$enable_state"
    uci commit qcmap_lan
    firewallrule_name=Allow-GRE-V4-Passthrough
    firewallrule_proto=gre
    proto_family=ipv4
  elif [ "$vpnpt_type" = "pptpptv6" ]; then
    # Set pptpptv6_enable parameter in qcmap_lan db to enable_state value
    uci set qcmap_lan.@profile[$profile_idx].pptpptv6_enable="$enable_state"
    uci commit qcmap_lan
    firewallrule_name=Allow-GRE-V6-Passthrough
    firewallrule_proto=gre
    proto_family=ipv6
  else
    echo $(basename "$0")":set_vpn_passthrough:"$LINENO":Invalid VPN Passthrough type."
    return
  fi

  public_interface_name=`get_wan_iface_name_from_wan_profile_handle $profile_handle ipv4`

  if [ $enable_state -eq 1 ]; then
    # Enable VPN Passthrough
    setup_vpnpt $firewallrule_name $public_interface_name $firewallrule_proto $proto_family "enable"
    uci commit firewall
    # reload firewall
    /etc/init.d/firewall reload
  else
    setup_vpnpt $firewallrule_name $public_interface_name $firewallrule_proto $proto_family "disable"
    uci commit firewall
    # reload firewall
    /etc/init.d/firewall reload
  fi
}

#Function to compare kernel version
#$1 is base kernel version_compare
#$2 is the current kernel version
function kernel_version_compare () {
  function sub_ver () {
    local len=${#1}
    temp=${1%%"."*} && indexOf=`echo ${1%%"."*} | echo ${#temp}`
    echo -e "${1:0:indexOf}"
  }
  function cut_dot () {
    local offset=${#1}
    local length=${#2}
    echo -e "${2:((++offset)):length}"
  }
  if [ -z "$1" ] || [ -z "$2" ]; then
    echo "=" && exit 0
  fi
  local v1=`echo -e "${1}" | tr -d '[[:space:]]'`
  local v2=`echo -e "${2}" | tr -d '[[:space:]]'`
  local v1_sub=`sub_ver $v1`
  local v2_sub=`sub_ver $v2`
  if [ $v1_sub -gt $v2_sub ]; then
    retval=0
  elif [ $v1_sub -lt $v2_sub ]; then
    retval=1
  else
    version_compare `cut_dot $v1_sub $v1` `cut_dot $v2_sub $v2`
  fi

  return $retval
}

#Function to set NAT Timeout values on boot-up
function set_nat_timeout() {

  #geting nat timeout values
  nat_timeout_generic_value=`uci get qcmap_lan.@nat_timeout_values[0].nat_timeout_generic`
  nat_timeout_icmp_value=`uci get qcmap_lan.@nat_timeout_values[0].nat_timeout_icmp`
  nat_timeout_icmp6_value=`uci get qcmap_lan.@nat_timeout_values[0].nat_timeout_icmpv6`
  nat_timeout_tcp_value=`uci get qcmap_lan.@nat_timeout_values[0].nat_timeout_tcp_established`
  nat_timeout_udp_value=`uci get qcmap_lan.@nat_timeout_values[0].nat_timeout_udp`

  #Compare Kernal Version
  base_kernel_version=4.9.0
  curr_kernel_version=$(uname -r | awk '{print $1}' | cut -d '-' -f 1)

  kernel_version_compare $base_kernel_version $curr_kernel_version
  isnewer=$?

  if [ $isnewer -eq 1 ]; then
    #Kernel Version is >= 4.9
    echo $nat_timeout_generic_value > /proc/sys/net/netfilter/nf_conntrack_generic_timeout
    echo $nat_timeout_icmp_value > /proc/sys/net/netfilter/nf_conntrack_icmp_timeout
    echo $nat_timeout_icmp6_value > /proc/sys/net/netfilter/nf_conntrack_icmpv6_timeout
    echo $nat_timeout_tcp_value > /proc/sys/net/netfilter/nf_conntrack_tcp_timeout_established
    echo $nat_timeout_udp_value > /proc/sys/net/netfilter/nf_conntrack_udp_timeout
    echo $nat_timeout_udp_value > /proc/sys/net/netfilter/nf_conntrack_udp_timeout_stream
  else
    #Kernel Version is below 4.9
    #echo "FAILURE: $ExpecV is not newer than or equal to $CurrV."
    echo $nat_timeout_generic_value > /proc/sys/net/ipv4/netfilter/ip_conntrack_generic_timeout
    echo $nat_timeout_icmp_value > /proc/sys/net/ipv4/netfilter/ip_conntrack_icmp_timeout
    echo $nat_timeout_tcp_value > /proc/sys/net/ipv4/netfilter/ip_conntrack_tcp_timeout_established
    echo $nat_timeout_udp_value > /proc/sys/net/ipv4/netfilter/ip_conntrack_udp_timeout
    echo $nat_timeout_udp_value > /proc/sys/net/ipv4/netfilter/ip_conntrack_udp_timeout_stream
  fi

}

#input:
# $1 -I or -D
# $2 iana_add
# $3 rmnet_addr
function ps_add_del_nat_rules()
{
  op=$1
  iana_add=$2
  rmnet_addr=$3

  echo "ip6tables -t nat $op POSTROUTING -s $iana_add -j SNAT --to-source $rmnet_addr"  >> /etc/firewall.user.nat
  NAT_LOG "ps_add_del_nat_rules: ip6tables -t nat $op POSTROUTING -s $iana_add -j SNAT --to-source $rmnet_addr"

  util_combine_iptable_rules

  /etc/init.d/firewall restart
}

case $1 in
  "SET_NAT")
    set_nat_type $2 $3
    break
    ;;
  "GET_NAT")
    get_nat_type $2
    echo "$?"
    break
    ;;
  "ENABLE_NAT")
    enable_nat_type $2
    break
    ;;
  add_port_fwd)
    #add_port_forward proto <params>
    #for ICMP: add_port_forward icmp pvt_fwding_ip profile_handle
    #for TCP/UDP: add_port_forward tcp/udp pvt_fwding_ip pvt_port global_port profile_handle
    if [ "$2" = "icmp" ]; then
      add_port_forward $2 $3 $4
    else
      add_port_forward $2 $3 $4 $5 $6
    fi
    break
    ;;
  del_port_fwd)
    echo "Deleting Port forwarding config "
    if [ "$2" = "icmp" ]; then
      del_port_forward $2 $3 $4
    else
      del_port_forward $2 $3 $4 $5 $6
    fi
    break
    ;;
   "ADD_DMZ")
    add_dmz $2 $3
    break
    ;;
  "DELETE_DMZ")
    delete_dmz $2
    break
    ;;
  enable_rtsp_alg)
    rtsp_alg_config enable
    break
    ;;
  disable_rtsp_alg)
    rtsp_alg_config disable
    break
    ;;
  enable_sip_alg)
    sip_alg_config enable
    break
    ;;
  disable_sip_alg)
    sip_alg_config disable
    break
    ;;
  set_sip_server)
    set_sip_server  $2 $3
    break
    ;;
  get_sip_server_info)
    get_sip_server_info
    break
    ;;
  update_network_sip_server)
    update_network_sip_server
    break
    ;;
  update_network_sip_server_to_dhcp)
    update_network_sip_server_to_dhcp $2
    break
    ;;
  "ADD_PORT_TRIGGER")
    add_port_trigger $2 $3 $4 $5 $6 $7 $8 $9 ${10}
    break
    ;;
  "DELETE_PORT_TRIGGER")
    delete_port_trigger $2 $3
    break
    ;;
  set_vpnpt)
    set_vpn_passthrough $2 $3 $4 $5
    break
    ;;
  set_nat_timeout)
    set_nat_timeout
    break
    ;;
  ps_add_del_nat_rules)
    ps_add_del_nat_rules $2 $3 $4
    break
    ;;
  *)
    log  $(basename "$0") "case: Invalid" $LINENO "Invalid option"
    ;;
esac
