#!/bin/sh
#  Copyright (c) 2022 Qualcomm Technologies, Inc.
#  All Rights Reserved.
#  Confidential and Proprietary - Qualcomm Technologies, Inc.

#Usage: For logging
source /etc/data/mbbUtils.sh

#Source lanUtils.sh file
. /etc/data/lanUtils.sh

# Usage: lan_config.sh add_vlan iface_type vlan_id
# Usage: lan_config.sh del_vlan iface_type vlan_id
# Usage: lan_config.sh show
# Usage: lan_config.sh bridge_context bridge_id
# Usage: lan_config.sh setlanconfig gw_ip netmask enable_dhcp dhcp_start dhcp_limit dhcp_lease_time bridge_id(optional)
# Usage: lan_config.sh add_dhcp_reservation device_type mac_addr_or_hostname  reservation_ip_address enable_reservation_flag
# Usage: lan_config.sh edit_dhcp_reservation old_ip_addr new_ip_addr old_enable_flag new_enable_flag Device_type mac_addr_or_hostname
# Usage: lan_config.sh delete_dhcp_reservation ip_addr
# Usage: lan_config.sh SetLANConfigOnLanActivation bridge_id

RANDOM=$$

if [[ "$#" -le 0 ]]; then
  echo "Check usage ./lan_config.sh <add/del> iface_index vlan_id"
  echo "Check usage ./lan_config.sh show"
  echo "Check usage ./lan_config.sh bridge_context bridge_id"
  return 1
elif [[ "$1" == "add" && $# -ne 4 ]]; then
  echo "Check usage ./lan_config.sh <add> iface_index vlan_id"
  return 1
elif [[ "$1" == "del" && $# -ne 3 ]]; then
  echo "Check usage ./lan_config.sh <del> iface_index vlan_id"
  return 1
elif [[ "$1" == "show" && $# -ne 1 ]]; then
  echo "Check usage ./lan_config.sh <show>"
  return 1
elif [[ "$1" == "bridge_context" && $# -ne 2 ]]; then
  echo "Check usage ./lan_config.sh <bridge_context> bridge_id"
  return 1
elif [[ "$1" == "setlanconfig" && $# -ne 8 ]]; then
  echo "Check usage ./lan_config.sh <setlanconfig> gw_ip netmask enable_dhcp dhcp_start dhcp_limit dhcp_lease_time bridge_id(optional)"
  return 1
elif [[ "$1" == "add_dhcp_reservation" && $# -ne 5 ]]; then
  echo "Check usage ./lan_config.sh <add_dhcp_reservation> device_type mac_addr_or_hostname  reservation_ip_address enable_reservation_flag"
  return 1
elif [[ "$1" == "edit_dhcp_reservation" && $# -ne 7 ]]; then
  echo "Check usage ./lan_config.sh <edit_dhcp_reservation> old_ip_addr new_ip_addr old_enable_flag new_enable_flag Device_type mac_addr_or_hostname"
  return 1
elif [[ "$1" == "delete_dhcp_reservation" && $# -ne 2 ]]; then
  echo "Check usage ./lan_config.sh <delete_dhcp_reservation> ip_addr"
  return 1
fi

#function to get vlan_string for given phy type
#$1 - interface type
function get_vlan_phy_str() {
  local iface_type=$1
  if [ $iface_type -eq $QCMAP_INTERFACE_TYPE_ETH ]; then
    echo "$(uci get qcmap_lan.@ipa_vlan_phy_config[0].eth_vlan_str)"
  elif [ $iface_type -eq $QCMAP_INTERFACE_TYPE_RNDIS ]; then
    echo $(uci get qcmap_lan.@ipa_vlan_phy_config[0].rndis_vlan_str)
  elif [ $iface_type -eq $QCMAP_INTERFACE_TYPE_ECM ]; then
    echo "$(uci get qcmap_lan.@ipa_vlan_phy_config[0].ecm_vlan_str)"
  elif [ $iface_type -eq $QCMAP_INTERFACE_TYPE_ETH_NIC2 ]; then
    echo $(uci get qcmap_lan.@ipa_vlan_phy_config[0].eth_nic2_vlan_str)
  else
    echo 0
  fi
}

#function to get vlan count created on given phy interface
#$1 - interface type
function get_phy_vlan_count() {
  local iface_type=$1
  if [ $iface_type -eq $QCMAP_INTERFACE_TYPE_ETH ]; then
    echo $(uci get qcmap_lan.@ipa_vlan_phy_config[0].eth_vlan_count)
  elif [ $iface_type -eq $QCMAP_INTERFACE_TYPE_RNDIS ]; then
    echo $(uci get qcmap_lan.@ipa_vlan_phy_config[0].rndis_vlan_count)
  elif [ $iface_type -eq $QCMAP_INTERFACE_TYPE_ECM ]; then
    echo $(uci get qcmap_lan.@ipa_vlan_phy_config[0].ecm_vlan_count)
  elif [ $iface_type -eq $QCMAP_INTERFACE_TYPE_ETH_NIC2 ]; then
    echo $(uci get qcmap_lan.@ipa_vlan_phy_config[0].eth_nic2_vlan_count)
  else
    echo -1
  fi
}

#function to set vlan count created on given phy interface
#$2 - interface type
#$1 - action i.e., vlan addition (add) or deletion (del)
function set_phy_vlan_count() {
  local iface_type=$2
  local action=$1
  local action_value=""
  if [ $action = "add" ]; then
    action_value="+"
  else
    action_value="-"
  fi
  if [ $iface_type -eq $QCMAP_INTERFACE_TYPE_ETH ]; then
    eth_count=$(uci get qcmap_lan.@ipa_vlan_phy_config[0].eth_vlan_count)
    uci set qcmap_lan.@ipa_vlan_phy_config[0].eth_vlan_count=`expr $eth_count $action_value 1`
    return `expr $eth_count $action_value 1`
  elif [ $iface_type -eq $QCMAP_INTERFACE_TYPE_RNDIS ]; then
    rndis_count=$(uci get qcmap_lan.@ipa_vlan_phy_config[0].rndis_vlan_count)
    uci set qcmap_lan.@ipa_vlan_phy_config[0].rndis_vlan_count=`expr $rndis_count $action_value 1`
    return `expr $rndis_count $action_value 1`
  elif [ $iface_type -eq $QCMAP_INTERFACE_TYPE_ECM ]; then
    ecm_count=$(uci get qcmap_lan.@ipa_vlan_phy_config[0].ecm_vlan_count)
    uci set qcmap_lan.@ipa_vlan_phy_config[0].ecm_vlan_count=`expr $ecm_count $action_value 1`
    return `expr $ecm_count $action_value 1`
  elif [ $iface_type -eq $QCMAP_INTERFACE_TYPE_ETH_NIC2 ]; then
    eth_nic2_count=$(uci get qcmap_lan.@ipa_vlan_phy_config[0].eth_nic2_vlan_count)
    uci set qcmap_lan.@ipa_vlan_phy_config[0].eth_nic2_vlan_count=`expr $eth_nic2_count $action_value 1`
    return `expr $eth_nic2_count $action_value 1`
  else
    echo -1
  fi
}

function create_bridge_context() {
  local vlan_id_param=$1
  local iface=$2
  local ezmesh_vlan_id1 ezmesh_vlan_id2 ezmesh_vlan_id3
  local ipaddr ip_int cidr mask_cidr_int ip_range_start range
  local no_of_vlans=$(uci get qcmap_lan.@no_of_configs[0].no_of_vlans)
  local lan_mac decimal_mac new_lan_mac

  while true
  do
    #Get IP from random IP generator
    ipaddr=$(util_get_random_ip "0xc0a80101")
    log $(basename "$0") "create_bridge_context" $LINENO "IP: $ipaddr"
    #Check if generated IP is valid.
    ip_int=$(printf "%d\n" $ipaddr)
    cidr=$(get_mask_cidr_int "255.255.255.0")
    mask_cidr_int=$((32 - $cidr))
    ip_range_start=$ip_int
    range=$((2**$mask_cidr_int))
    ip_range_end=$(($ip_range_start + $range - 1))
    is_valid_ip=$(validate_ip_range $ip_range_start $ip_range_end)
    if [ $is_valid_ip -eq 1 ]; then
        log $(basename "$0") "create_bridge_context" $LINENO "VALID IP found"
        break
    fi
  done

  res=`printf '%x\n' $ipaddr`
  ip=`printf '%d.%d.%d.%d\n' $(echo $res | sed 's/../0x& /g')`

  #Get br-lan mac
  lan_mac=$(ip link show br-lan 2>/dev/null | awk '/link\/ether/{gsub(/:/,"",$2); print $2}')
  if [ ! -z "$lan_mac" ]; then
    decimal_mac=$(( ($(( 0x$lan_mac )) + $vlan_id_param) & 0xFFFFFFFFFFFF ))
    new_lan_mac=$(printf '%012X' "$decimal_mac" | sed 's/\(..\)/\1:/g;s/:$//')
  fi

  #Set lan configuration
  uci set network.lan$vlan_id_param="interface"
  uci set network.lan$vlan_id_param.type="bridge"
  uci set network.lan$vlan_id_param.proto="static"
  if [ ! -z "$iface" ]; then
    uci set network.lan$vlan_id_param.ifname="$iface.$vlan_id_param "
  fi
  uci set network.lan$vlan_id_param.ipaddr="$ip"
  uci set network.lan$vlan_id_param.netmask="255.255.255.0"
  uci set network.lan$vlan_id_param.ip6assign="60"
  #Fix bridge address is the same
  if [ ! -z "$new_lan_mac" ]; then
    uci set network.lan$vlan_id_param.macaddr="$new_lan_mac"
  fi

  if [ ! -z "$iface" ]; then
    #update ip and netmask in qcmap_lan. Need it for IPPT reset
    uci set qcmap_lan.@vlan[$((no_of_vlans-1))].ip="$ip"
    uci set qcmap_lan.@vlan[$((no_of_vlans-1))].netmask="255.255.255.0"
  else
    ezmesh_vlan_id1=$(uci get qcmap_ezmesh.@r2_config[0].guestapone_vlan_id)
    ezmesh_vlan_id2=$(uci get qcmap_ezmesh.@r2_config[0].guestaptwo_vlan_id)
    ezmesh_vlan_id3=$(uci get qcmap_ezmesh.@r2_config[0].guestapthree_vlan_id)
    if [ $vlan_id_param -eq $ezmesh_vlan_id1 ]; then
      uci set repacd.MAPConfig.BridgeProtoNwOne="static"
      uci set repacd.MAPConfig.BridgeIPAddressNwOne="$ip"
    elif [ $vlan_id_param -eq $ezmesh_vlan_id2 ]; then
      uci set repacd.MAPConfig.BridgeProtoNwTwo="static"
      uci set repacd.MAPConfig.BridgeIPAddressNwTwo="$ip"
    elif [ $vlan_id_param -eq $ezmesh_vlan_id3 ]; then
      uci set repacd.MAPConfig.BridgeProtoNwThree="static"
      uci set repacd.MAPConfig.BridgeIPAddressNwThree="$ip"
    fi
    uci commit repacd
  fi

  #dnsmasq settings
  uci set dhcp.lan${vlan_id_param}_dns=dnsmasq
  uci set dhcp.lan${vlan_id_param}_dns.domainneeded='1'
  uci set dhcp.lan${vlan_id_param}_dns.boguspriv='1'
  uci set dhcp.lan${vlan_id_param}_dns.filterwin2k='0'
  uci set dhcp.lan${vlan_id_param}_dns.localise_queries='1'
  uci set dhcp.lan${vlan_id_param}_dns.rebind_protection='1'
  uci set dhcp.lan${vlan_id_param}_dns.rebind_localhost='1'
  uci set dhcp.lan${vlan_id_param}_dns.local="/lan${vlan_id_param}/"
  uci set dhcp.lan${vlan_id_param}_dns.domain="lan${vlan_id_param}"
  uci set dhcp.lan${vlan_id_param}_dns.expandhosts='1'
  uci set dhcp.lan${vlan_id_param}_dns.nonegcache='0'
  uci set dhcp.lan${vlan_id_param}_dns.authoritative='1'
  uci set dhcp.lan${vlan_id_param}_dns.readethers='1'
  uci set dhcp.lan${vlan_id_param}_dns.leasefile="/tmp/data/dhcp.leases.lan${vlan_id_param}"
  uci set dhcp.lan${vlan_id_param}_dns.resolvfile="/tmp/resolv.conf.d/resolv.conf.lan${vlan_id_param}.auto"
  uci set dhcp.lan${vlan_id_param}_dns.nonwildcard='1'
  uci set dhcp.lan${vlan_id_param}_dns.localservice='1'
  uci set dhcp.lan${vlan_id_param}_dns.ednspacket_max='1232'
  uci add_list dhcp.lan${vlan_id_param}_dns.interface="lan${vlan_id_param}"
  uci set dhcp.lan${vlan_id_param}_dns.dhcpscript='/etc/data/dnsmasq_script.sh'

  #update dhcp
  uci set dhcp.lan$vlan_id_param=dhcp
  uci set dhcp.lan$vlan_id_param.interface=lan$vlan_id_param
  uci set dhcp.lan$vlan_id_param.start='100'
  uci set dhcp.lan$vlan_id_param.limit='150'
  uci set dhcp.lan$vlan_id_param.leasetime='12h'
  uci set dhcp.lan$vlan_id_param.dhcpv4='server'
  uci set dhcp.lan$vlan_id_param.dhcpv6='server'
  uci set dhcp.lan$vlan_id_param.ra='server'
  uci set dhcp.lan$vlan_id_param.ra_slaac='1'
  uci set dhcp.lan$vlan_id_param.ra_flags='managed-config other-config'
  uci set dhcp.lan$vlan_id_param.ignore='0'
  uci set dhcp.lan$vlan_id_param.instance=lan${vlan_id_param}_dns
  uci set dhcp.lan$vlan_id_param.ndp="relay"

  #update lan_config for vlan's on qcmap_current_lan_config
  #This is current lanconfig which is used when SetLANConfig is done
  uci set qcmap_current_lan_config.lan$vlan_id_param=lan_config
  uci set qcmap_current_lan_config.lan$vlan_id_param.ipaddr="$ip"
  uci set qcmap_current_lan_config.lan$vlan_id_param.netmask="255.255.255.0"
  uci set qcmap_current_lan_config.lan$vlan_id_param.ignore='0'
  uci set qcmap_current_lan_config.lan$vlan_id_param.start='100'
  uci set qcmap_current_lan_config.lan$vlan_id_param.limit='150'
  uci set qcmap_current_lan_config.lan$vlan_id_param.leasetime='12h'
  uci set qcmap_current_lan_config.lan$vlan_id_param.bridge_id="$vlan_id_param"

  if [ ! -z "$iface" ]; then
    #zone index
    uci add firewall zone
    uci set firewall.@zone[-1].name=lan$vlan_id_param
    uci set firewall.@zone[-1].network=lan$vlan_id_param
    uci set firewall.@zone[-1].input='ACCEPT'
    uci set firewall.@zone[-1].output='ACCEPT'
    uci set firewall.@zone[-1].forward='ACCEPT'
  else
    #zone index
    uci add firewall zone
    uci set firewall.@zone[-1].name=lan$vlan_id_param
    uci set firewall.@zone[-1].network=lan$vlan_id_param
    uci set firewall.@zone[-1].input='REJECT'
    uci set firewall.@zone[-1].output='ACCEPT'
    uci set firewall.@zone[-1].forward='REJECT'
    uci set firewall.@zone[-1].masq='1'

    uci add firewall forwarding
    uci set firewall.@forwarding[-1].src=lan$vlan_id_param
    uci set firewall.@forwarding[-1].dest=wan

    uci add firewall rule
    uci set firewall.@rule[-1].name="lan${vlan_id_param}_DHCP"
    uci set firewall.@rule[-1].src=lan$vlan_id_param
    uci set firewall.@rule[-1].family=ipv4
    uci set firewall.@rule[-1].src_port='68'
    uci set firewall.@rule[-1].dest_port='67'
    uci set firewall.@rule[-1].target='ACCEPT'
    uci set firewall.@rule[-1].proto='udp'

    uci add firewall rule
    uci set firewall.@rule[-1].name="lan${vlan_id_param}_DNS"
    uci set firewall.@rule[-1].src=lan$vlan_id_param
    uci set firewall.@rule[-1].dest_port='53'
    uci set firewall.@rule[-1].target='ACCEPT'
    uci set firewall.@rule[-1].proto='udp'

    uci add firewall rule
    uci set firewall.@rule[-1].name="Block-lan-lan${vlan_id_param}-forwarding"
    uci set firewall.@rule[-1].src=lan
    uci set firewall.@rule[-1].dest=lan$vlan_id_param
    uci set firewall.@rule[-1].family=any
    uci set firewall.@rule[-1].target='REJECT'
    uci set firewall.@rule[-1].proto='0'

    uci add firewall rule
    uci set firewall.@rule[-1].name="Block-lan${vlan_id_param}-lan-forwarding"
    uci set firewall.@rule[-1].src=lan$vlan_id_param
    uci set firewall.@rule[-1].dest=lan
    uci set firewall.@rule[-1].family='any'
    uci set firewall.@rule[-1].target='REJECT'
    uci set firewall.@rule[-1].proto='0'
  fi

  uci commit
  ubus call network reload
  /etc/init.d/dnsmasq start lan${vlan_id_param}_dns
  /etc/init.d/firewall reload
}

function delete_bridge_context() {
  local vlan_id_param=$1
  local idx=0
  local found=0

  #delete vlan-bridge
  uci delete network."lan$vlan_id_param"
  uci delete dhcp."lan$vlan_id_param"
  uci delete dhcp.lan${vlan_id_param}_dns

  #delete lan zone from firewall
  idx=0
  found=0
  while true
  do
    local zone_name=$(uci get firewall.@zone[$idx].name)
    if [ -z "$zone_name" ]; then
      break
    fi
    if [ "$zone_name" = "lan$vlan_id_param" ]; then
      found=1
      break
    fi
    idx=$((idx+1))
  done
  if [ $found -eq 1 ]; then
    uci delete firewall.@zone[$idx]
  fi

  #delete forwarding firewall config per vlan
  idx=0
  found=0
  while true
  do
    local fwd_rule=$(uci get firewall.@forwarding[$idx].src)
    if [ -z "$fwd_rule" ]; then
      break
    fi
    if [ "$fwd_rule" = "lan$vlan_id_param" ]; then
      found=1
      break
    fi
    idx=$((idx+1))
  done
  if [ $found -eq 1 ]; then
    uci delete firewall.@forwarding[$idx]
  fi

  #delete lan_DHCP firewall config per vlan
  idx=0
  found=0
  while true
  do
    local rule_name=$(uci get firewall.@rule[$idx].name)
    if [ -z "$rule_name" ]; then
      break
    fi
    if [ "$rule_name" = "lan${vlan_id_param}_DHCP" ]; then
      found=1
      break
    fi
    idx=$((idx+1))
  done
  if [ $found -eq 1 ]; then
    uci delete firewall.@rule[$idx]
  fi

  #delete lan_DNS firewall config per vlan
  idx=0
  found=0
  while true
  do
    local rule_name=$(uci get firewall.@rule[$idx].name)
    if [ -z "$rule_name" ]; then
      break
    fi
    if [ "$rule_name" = "lan${vlan_id_param}_DNS" ]; then
      found=1
      break
    fi
    idx=$((idx+1))
  done
  if [ $found -eq 1 ]; then
    uci delete firewall.@rule[$idx]
  fi

  #delete Block-lan-lan${vlan_id_param}-forwarding firewall config per vlan
  idx=0
  found=0
  while true
  do
    local rule_name=$(uci get firewall.@rule[$idx].name)
    if [ -z "$rule_name" ]; then
      break
    fi
    if [ "$rule_name" = "Block-lan-lan${vlan_id_param}-forwarding" ]; then
      found=1
      break
    fi
    idx=$((idx+1))
  done
  if [ $found -eq 1 ]; then
    uci delete firewall.@rule[$idx]
  fi

  #delete Block-lan${vlan_id_param}-lan-forwarding firewall config per vlan
  idx=0
  found=0
  while true
  do
    local rule_name=$(uci get firewall.@rule[$idx].name)
    if [ -z "$rule_name" ]; then
      break
    fi
    if [ "$rule_name" = "Block-lan${vlan_id_param}-lan-forwarding" ]; then
      found=1
      break
    fi
    idx=$((idx+1))
  done
  if [ $found -eq 1 ]; then
    uci delete firewall.@rule[$idx]
  fi

  uci commit
  ubus call network reload
#  if [ -n "$wwan_profile" ]; then
#    /lib/netifd/rmnet_update.sh "up" $wwan_profile
#  fi
  /etc/init.d/dnsmasq stop lan${vlan_id_param}_dns
  /etc/init.d/firewall reload
}

# function to add VLAN
function add_vlan() {
  local iface_type=$1
  local vlan_id_param=$2
  local is_accelerated=$3
  local iface=$(util_get_interface_name $iface_type)
  local max_offload_count
  no_of_vlans=$(uci get qcmap_lan.@no_of_configs[0].no_of_vlans)
  local vlan_id=""
  # flag to check if bridge is already created for specific VLAN
  bridge_created=0

  #check if vlan already exists.
  for i in $(seq $no_of_vlans)
  do
    phy_interface=$(uci get qcmap_lan.@vlan[$((i-1))].phy_interface)
    vlan_id=$(uci get qcmap_lan.@vlan[$((i-1))].vlan_id)
    res=`echo "$phy_interface" | grep -w -c "$iface"`
    if [ $vlan_id -eq $vlan_id_param ]; then
      if [ $res -eq 1 ]; then
        echo "VLAN already exist"
        return 0
      else
        id=$((i-1))
        bridge_created=1
        break
      fi
    fi
  done

  #update lan info.
  if [ $bridge_created -eq 1 ]; then
    #add interface name
    phy_interface=$(uci get qcmap_lan.@vlan[$id].phy_interface)
    phy_interface="${phy_interface}$iface "
    uci set qcmap_lan.@vlan[$id].phy_interface="$phy_interface"
    ifname=$(uci get network.lan$2.ifname)
    ifname="${ifname}$iface.$vlan_id_param "
    uci set network.lan$vlan_id_param.ifname="$ifname"
    vlan_id=$id
    if [ $is_accelerated -eq 1 ]; then
      uci set qcmap_lan.@vlan[$id].ipa_offload="$is_accelerated"
    fi
  else
    #Add VLAN config
    uci add qcmap_lan vlan
    uci set qcmap_lan.@vlan[$no_of_vlans].vlan_id="$vlan_id_param"
    uci set qcmap_lan.@vlan[$no_of_vlans].phy_interface="$iface "
    uci set qcmap_lan.@vlan[$no_of_vlans].ipa_offload="$is_accelerated"
    vlan_id=$no_of_vlans
    no_of_vlans=$((no_of_vlans+1))
    uci set qcmap_lan.@no_of_configs[0].no_of_vlans="$no_of_vlans"
    create_bridge_context $vlan_id_param $iface
  fi
  uci commit

  #Check MAX IPA offload.
  max_offload_count=$(uci get qcmap_lan.@no_of_configs[0].max_offload_count)
  if [ $max_offload_count -eq $QCMAP_MAX_IPA_OFFLOAD ]; then
     log $(basename "$0") "add_vlan" $LINENO "IPA offload reach to max count"
     uci set qcmap_lan.@vlan[$vlan_id].ipa_offload='0'
     #reset IPA offloaded to 0 as it reaches to max limit
     is_accelerated=0
  else
    max_offload_count=$((max_offload_count+1))
    uci set qcmap_lan.@no_of_configs[0].max_offload_count="$max_offload_count"
  fi
  uci commit

  if [ $is_accelerated -eq 1 ]; then
    local reboot=0
    local vlan_str=""
    local offloaded_phy_interfaces=""
    log $(basename "$0") "add_vlan" $LINENO "ipa_offload enable"
    #Update IPA VLAN offload/enabled count for interface
    set_phy_vlan_count "add" $iface_type
    #Update offloaded phy_interface_name
    offloaded_phy_interfaces=$(uci get qcmap_lan.@vlan[$vlan_id].offloaded_phy_interfaces)
    uci set qcmap_lan.@vlan[$vlan_id].offloaded_phy_interfaces="$offloaded_phy_interfaces""$iface "
    for i in $(seq $QCMAP_INTERFACE_TYPE_ETH_NIC2)
    do
      vlan_count=$(get_phy_vlan_count $i)
      if [ $vlan_count != -1 -a $vlan_count != 0 ]; then
        vlan_str="$vlan_str "$(get_vlan_phy_str $i)
        if [ $vlan_count -eq 1 -a $iface_type -eq $i ]; then
          log $(basename "$0") "add_vlan" $LINENO "setting reboot param to 1"
          reboot=1
        fi
      fi
    done
    uci commit
    if [ $reboot -eq 1 ]; then
      ipa vlan $vlan_str
      log $(basename "$0") "add_vlan" $LINENO "rebooting device on first_offloaded_vlan"
      sleep 5
      reboot
    fi
  fi
  ubus call network reload
  return 1
}

function show_vlan() {
  max=$(uci get qcmap_lan.@no_of_configs[0].no_of_vlans)
  if [ $max -eq 0 ]; then
    echo "No VLAN configured"
  fi
  echo "List of VLANs"
  for i in $(seq $max)
  do
    phy_interface=$(uci get qcmap_lan.@vlan[$((i-1))].phy_interface)
    count=`echo $phy_interface | wc -w`
    vlan_id=$(uci get qcmap_lan.@vlan[$((i-1))].vlan_id)
    for j in $(seq $count)
    do
      res=`echo $phy_interface | awk '{print $1}'`
      phy_interface=${phy_interface//"$res "/}
      echo "IFANEM:$res    VLAN ID:$vlan_id"
    done
  done
  return 1
}

function del_vlan() {
  local iface_name=$1
  local iface_type=$(get_interface_type_with_iface_name $iface_name)
  local vlan_id_param=$2
  no_of_vlans=$(uci get qcmap_lan.@no_of_configs[0].no_of_vlans)
  local wwan_profile=""
  local max_offload_count found=0

  if [[ "$no_of_vlans" -eq 0 ]]; then
    echo "NO VLAN configured"
    return 0
  fi

  for i in $(seq $no_of_vlans)
  do
    phy_interface=$(uci get qcmap_lan.@vlan[$((i-1))].phy_interface)
    vlan_id=$(uci get qcmap_lan.@vlan[$((i-1))].vlan_id)
    res=`echo "$phy_interface" | grep -w -c "$1"`
    if [ $vlan_id -eq $vlan_id_param ] && [ $res -eq 1 ]; then
      found=1
      id=$((i-1))
      break
  fi
  done

  if [ $found -eq 0 ]; then
    echo "VLAN not found"
    return 0
  else
    phy_interface=$(uci get qcmap_lan.@vlan[$id].phy_interface)
    count=`echo $phy_interface | wc -w`
    ipa_offload=$(uci get qcmap_lan.@vlan[$id].ipa_offload)
    max_offload_count=$(uci get qcmap_lan.@no_of_configs[0].max_offload_count)
    local reboot=0
    if [ $ipa_offload -eq 1 ]; then
      local vlan_str=""
      local offloaded_phy_ifaces=$(uci get qcmap_lan.@vlan[$id].offloaded_phy_interfaces)
      local is_iface_offloaded=`echo $offloaded_phy_ifaces | grep -c $iface_name`
      if [ $is_iface_offloaded -eq 1 ]; then
        for i in $(seq $QCMAP_INTERFACE_TYPE_ETH_NIC2)
        do
          vlan_count=$(get_phy_vlan_count $i)
          if [ $vlan_count != -1 -a $vlan_count != 0 ]; then
            if [ $vlan_count -eq 1 -a $iface_type -eq $i ]; then
              log $(basename "$0") "del_vlan" $LINENO "setting reboot param to 1"
              reboot=1
            else
              vlan_str="$vlan_str "$(get_vlan_phy_str $i)
            fi
          fi
        done
        #Decrement IPA offloaded count for VLAN
        set_phy_vlan_count "del" $iface_type
        offloaded_phy_ifaces=${offloaded_phy_ifaces//"$iface_name "/}
        offloaded_iface_count=`echo $offloaded_phy_ifaces | wc -w`
        uci set qcmap_lan.@vlan[$id].offloaded_phy_interfaces="$offloaded_phy_ifaces "
        if [ $offloaded_iface_count -eq 0 ]; then
          uci set qcmap_lan.@vlan[$id].ipa_offload="0"
          uci delete qcmap_lan.@vlan[$id].offloaded_phy_interfaces
        fi
      fi
      #Update current max IPA offload count
       max_offload_count=$((max_offload_count-1))
       uci set qcmap_lan.@no_of_configs[0].max_offload_count="$max_offload_count"
    fi
    if [ $count -eq 1 ]; then

      #Delete vlan-pdn mapping
      wwan_profile=$(uci get qcmap_lan.@vlan[$id].wwan_profile_num)
      #delete downstream interface from network file
      if [ -n "$wwan_profile" ]; then
        if [ "$wwan_profile" -eq 1 ]; then
          uci del_list network.wan.downstream=lan$vlan_id_param
          uci del_list network.wan_v6.downstream=lan$vlan_id_param
        else
          uci del_list network.wan$wwan_profile.downstream=lan$vlan_id_param
          uci del_list network.wan${wwan_profile}_v6.downstream=lan$vlan_id_param
        fi
        #delete vlan_id from profile in qcmap_lan as part of del vlan-pdn mapping
        profile_count=$(uci get qcmap_lan.@no_of_configs[0].no_of_profiles)
        for i in $(seq $profile_count)
        do
          profile_id=$(uci get qcmap_lan.@profile[$((i-1))].profile_id)
          if [ "$profile_id" = "$wwan_profile" ]; then
            uci del_list qcmap_lan.@profile[$((i-1))].vlan_ids=$vlan_id_param
            log $(basename "$0") "del_vlan" $LINENO "remove vlan_id from profile"
            break
         fi
        done

        #delete corresponding forwarding rule from firewall config
        idx=""
        if [ "$wwan_profile" -eq 1 ]; then
          idx=`uci show firewall | grep zone | grep -w "lan_wan" | awk -F'[][]' '{print $2}'`
        else
          idx=`uci show firewall | grep zone | grep -w "lan_wan$wwan_profile" | awk -F'[][]' '{print $2}'`
        fi
        if [ -n "$idx" ]; then
          log $(basename "$0") "del_vlan" $LINENO "remove lan from firewall lan_wan_zone"
          uci del_list firewall.@zone[$idx].network=lan$vlan_id_param
          break
        fi
      fi

      #delete bridge
      uci delete qcmap_lan.@vlan[$id]
      uci delete network."lan$vlan_id_param"
      uci delete dhcp."lan$vlan_id_param"
      uci delete dhcp.lan${vlan_id_param}_dns
      uci set qcmap_lan.@no_of_configs[0].no_of_vlans="$((no_of_vlans-1))"
      #delete current_lan_config from qcmap_current_lan_config
      uci delete qcmap_current_lan_config.lan${vlan_id_param}

      #delete lan zone from firewall
      local idx=0
      found=0
      while true
      do
        local zone_name=$(uci get firewall.@zone[$idx].name)
        if [ -z "$zone_name" ]; then
          break
        fi
        if [ "$zone_name" = "lan$vlan_id_param" ]; then
          found=1
          break
        fi
        idx=$((idx+1))
      done
      if [ $found -eq 1 ]; then
        uci delete firewall.@zone[$idx]
      fi
    else
      #remove interface
      phy_interface=$(uci get qcmap_lan.@vlan[$id].phy_interface)
      phy_interface=${phy_interface//"$iface_name "/}
      uci set qcmap_lan.@vlan[$id].phy_interface="$phy_interface"
      ifname=$(uci get network.lan$vlan_id_param.ifname)
      ifname=${ifname//"$iface_name.$vlan_id_param "/}
      uci set network.lan$vlan_id_param.ifname="$ifname"
    fi
  fi

  uci commit
  /etc/init.d/network reload
  if [ -n "$wwan_profile" ]; then
    if [ -f /tmp/ipv4config$wwan_profile ]; then
      /lib/netifd/rmnet_update.sh "up" $wwan_profile 1
    fi
    if [ -f /tmp/ipv6config$wwan_profile ]; then
      /lib/netifd/rmnet_update.sh "up" $wwan_profile 2
    fi
  fi
  /etc/init.d/dnsmasq stop lan${vlan_id_param}_dns
  /etc/init.d/firewall reload
  if [ $reboot -eq 1 ]; then
    ipa vlan $vlan_str
    sleep 5
    log $(basename "$0") "del_vlan" $LINENO "rebooting device on del_last_offloaded_vlan"
    reboot
  fi
  return 1
}

function get_interface_type_with_iface_name() {
  iface=""
  eth_iface=$(uci get qcmap_lan.@phy_iface_names[0].eth)
  ecm_iface=$(uci get qcmap_lan.@phy_iface_names[0].ecm)
  rndis_iface=$(uci get qcmap_lan.@phy_iface_names[0].rndis)
  eth_nic2_iface=$(uci get qcmap_lan.@phy_iface_names[0].eth_nic2)
  if [ $1 = $eth_iface ]; then
     #ETH
     iface=$QCMAP_INTERFACE_TYPE_ETH
  elif [ $1 = $ecm_iface ]; then
     #ECM
     iface=$QCMAP_INTERFACE_TYPE_ECM
  elif [ $1 = $rndis_iface ]; then
     #RNDIS
     iface=$QCMAP_INTERFACE_TYPE_RNDIS
  elif [ $1 = $eth_nic2_iface ]; then
     #ETH_NIC2
     iface=$QCMAP_INTERFACE_TYPE_ETH_NIC2
   fi
  echo $iface
}

# Sets bridge context for current profile_handle in qcmap_lan
# $1 - bridge VLAN ID
function set_bridge_context() {
  local bridge_id=$1

  #Get bridge_context information from qcmap_lan
  local bridge_context=$(uci get qcmap_lan.@no_of_configs[0].bridge_context)
  if [ -z "$bridge_context" ]; then
    echo "lan_config.sh: set_bridge_context: bridge_context entry not found"
    return
  elif [ -n "$bridge_context" ] && [ $bridge_context -eq $bridge_id ]; then
    echo "lan_config.sh: set_bridge_context: bridge_context is already set to $bridge_id"
    return
  else
    echo "lan_config.sh: set_bridge_context: setting bridge_context to $bridge_id"
    uci set qcmap_lan.@no_of_configs[0].bridge_context="$bridge_id"
    uci commit
  fi
}

#Usage : Sets LAN config
#$1 - gw_ip
#$2 - netmask
#$3 - enable_dhcp
#$4 - dhcp_start
#$5 - dhcp_limit
#$6 - dhcp_lease_time
#$7 - bridge_id(optional)
function SetLANConfig() {
  local gw_ip=$(uint_to_ipv4 "$1")
  local net_mask=$(uint_to_ipv4 "$2")
  local enable_dhcp="$3"
  local dhcp_start="$4"
  local dhcp_limit="$5"
  local dhcp_lease_time="$6"
  local bridge_id=$7
  local lan_interface="lan"

  if [ "$bridge_id" -ne 0 ]; then
    lan_interface=lan$bridge_id
  fi

  #Update gw_ip and netmask in network config
  uci set qcmap_current_lan_config.${lan_interface}.ipaddr="$gw_ip"
  uci set qcmap_current_lan_config.${lan_interface}.netmask="$net_mask"
  uci set qcmap_current_lan_config.${lan_interface}.ignore="1"

  #Update dhcp info in dhcp config
  if [ $enable_dhcp -eq 1 ]; then
    uci set qcmap_current_lan_config.${lan_interface}.ignore="0"
    uci set qcmap_current_lan_config.${lan_interface}.start="$dhcp_start"
    uci set qcmap_current_lan_config.${lan_interface}.limit="$dhcp_limit"
    uci set qcmap_current_lan_config.${lan_interface}.leasetime="$dhcp_lease_time"
  fi

  if [ -z "$bridge_id" ] || [ $bridge_id -eq 0 ];then
    uci set qcmap_current_lan_config.${lan_interface}.bridge_id='0'
  else
    uci set qcmap_current_lan_config.${lan_interface}.bridge_id="$bridge_id"
  fi

  uci commit
}

#Usage : SetLANConfigOnLanActivation
#$1 - bridge_id
function SetLANConfigOnLanActivation() {
  local bridge_id=$1
  local lan_interface="lan"
  local dhcp_pd_mode

  if [ "$bridge_id" -ne 0 ]; then
    lan_interface=lan$bridge_id
  fi

  local gw_ip=$(uci get qcmap_current_lan_config.${lan_interface}.ipaddr)
  local net_mask=$(uci get qcmap_current_lan_config.${lan_interface}.netmask)
  local dhcp_ignore=$(uci get qcmap_current_lan_config.${lan_interface}.ignore)

  if [ $dhcp_ignore -eq 0 ]; then
    local dhcp_start=$(uci get qcmap_current_lan_config.${lan_interface}.start)
    local dhcp_limit=$(uci get qcmap_current_lan_config.${lan_interface}.limit)
    local dhcp_lease_time=$(uci get qcmap_current_lan_config.${lan_interface}.leasetime)
  fi

  if [ -z "$bridge_id" ] || [ $bridge_id -eq 0 ];then
    local lan_interface=$(uci get network.lan.device)
    if [ -z "$lan_interface" ]; then
      log $(basename "$0") "SetLANConfig" $LINENO "lan config not found"
      return 1
    fi
    #Update gw_ip and netmask in network config
    uci set network.lan.ipaddr="$gw_ip"
    uci set network.lan.netmask="$net_mask"

    #Update gw_ip and netmask in qcmap_lan
    uci set qcmap_lan.@lan[0].ip="$gw_ip"
    uci set qcmap_lan.@lan[0].netmask="$net_mask"

    dhcp_lan=$(uci get dhcp.lan)
    if [ -z "$dhcp_lan" ]; then
      log $(basename "$0") "SetLANConfig" $LINENO "dhcp_lan config not found"
      return 1
    fi
    #Update dhcp info in dhcp config
    if [ $dhcp_ignore -eq 0 ]; then
      uci set dhcp.lan.start="$dhcp_start"
      uci set dhcp.lan.limit="$dhcp_limit"
      uci set dhcp.lan.leasetime="$dhcp_lease_time"
      uci set dhcp.lan.ignore='0'
    else
      uci set dhcp.lan.ignore='1'
    fi
    cat /dev/null > /tmp/data/dhcp.leases.lan
  fi

  if [ $bridge_id -ne 0 ];then

    #Update gw_ip and netmask in network config
    uci set network.lan$bridge_id.ipaddr="$gw_ip"
    uci set network.lan$bridge_id.netmask="$net_mask"

    dhcp_lan=$(uci get dhcp.lan$bridge_id)
    if [ -z "$dhcp_lan" ]; then
      log $(basename "$0") "SetLANConfig" $LINENO "dhcp_lan config not found"
      return 1
    fi
    #Update dhcp info in dhcp config
    if [ $dhcp_ignore -eq 0 ]; then
      uci set dhcp.lan$bridge_id.start="$dhcp_start"
      uci set dhcp.lan$bridge_id.limit="$dhcp_limit"
      uci set dhcp.lan$bridge_id.leasetime="$dhcp_lease_time"
      uci set dhcp.lan$bridge_id.ignore='0'
    else
      uci set dhcp.lan$bridge_id.ignore='1'
    fi
    cat /dev/null > /tmp/data/dhcp.leases.lan$bridge_id
  fi

  cat /dev/null > /tmp/data/dnsmasq_host.txt
  uci commit
  ubus call network reload
  /etc/init.d/dnsmasq reload

  #this function will change network config, need restart odhcpd
  dhcp_pd_mode=$(util_execute_uci get dhcp.lan.qcmap_prefix_delegation_mode)
  if [ -n "$dhcp_pd_mode" ]; then
    /etc/init.d/odhcpd restart
    log $(basename "$0") "SetLANConfig" $LINENO "For prefix delegation mode, need restart odhcpd when network config changed"
  fi
}

#Fuction to add DHCP reservation
#$1 -> device_type
#$2 -> mac_addr or hostname according to device type
#$3 -> reservation ip address
#$4 -> enable/disable reservation flag

function add_dhcp_reservation() {
  local device_type=$1
  local ip_addr="$3"
  local enable_reserv_flag=$4
  local max_dhcp_reservation=`uci get qcmap_lan.@no_of_configs[0].maximum_no_of_dhcp_reservation`

  log $(basename "$0") "add_dhcp_reservation" $LINENO "Enter add_dhcp_reservation"

  local no_of_dhcp_reserv_entries=`uci get qcmap_lan.@no_of_configs[0].no_of_dhcp_reservation_records`

  #checking for duplicacy of client_reserved_ip
  if [ $no_of_dhcp_reserv_entries -ne 0 ]; then
    for i in $(seq $no_of_dhcp_reserv_entries)
    do
    reserv_ip=$(uci get qcmap_lan.@dhcp_reservation[$((i-1))].ip)
    if [ "$reserv_ip" = "$ip_addr" ];
    then
      log $(basename "$0") "add_dhcp_reservation" $LINENO " IP Address already reserved for another Client!!"
      echo "Add DHCP Reservation Failed"
      exit 1
    fi
    done
  fi

  case $device_type in
    1)
      #device is non-usb
      if [ $no_of_dhcp_reserv_entries -ne 0 ]; then
        mac_reservation_index=$(uci show qcmap_lan | grep -i dhcp_reservation | grep -i "mac" | awk -F'[][]' '{print $2}')
        for idx in $mac_reservation_index
        do
          reserv_mac=$(uci get qcmap_lan.@dhcp_reservation[$idx].mac)
          if [ "$reserv_mac" ]; then
            if [ "$reserv_mac" = "$2" ];then
              log $(basename "$0") "add_dhcp_reservation" $LINENO " IP Address already reserved for same MAC Addr Client!!"
              echo "Add DHCP Reservation Failed"
              exit 1
            fi
          fi
        done
      fi
      uci add qcmap_lan dhcp_reservation
      uci set qcmap_lan.@dhcp_reservation[-1].dns='1'
      uci set qcmap_lan.@dhcp_reservation[-1].mac=$2
      uci set qcmap_lan.@dhcp_reservation[-1].ip=$ip_addr
      uci set qcmap_lan.@dhcp_reservation[-1].device_type="non_usb"
      uci set qcmap_lan.@dhcp_reservation[-1].enable="$enable_reserv_flag"
      uci set qcmap_lan.@no_of_configs[0].no_of_dhcp_reservation_records=$((no_of_dhcp_reserv_entries+1))
      uci commit
      ;;
    0)
      #device is usb
      if [ $no_of_dhcp_reserv_entries -ne 0 ]; then
        hostname_reservation_index=$(uci show qcmap_lan | grep -i dhcp_reservation | grep -i "name" | awk -F'[][]' '{print $2}')
        for idx in $hostname_reservation_index
        do
          reserv_name=$(uci get qcmap_lan.@dhcp_reservation[$idx].name)
          if [ "$reserv_name" ]; then
            if [ "$reserv_name" = "$2" ]; then
              log $(basename "$0") "add_dhcp_reservation" $LINENO " IP Address already reserved for same Host name!!"
              echo "Add DHCP Reservation Failed"
              exit 1
            fi
          fi
        done
      fi
      uci add qcmap_lan dhcp_reservation
      uci set qcmap_lan.@dhcp_reservation[-1].dns='1'
      uci set qcmap_lan.@dhcp_reservation[-1].name=$2
      uci set qcmap_lan.@dhcp_reservation[-1].ip=$ip_addr
      uci set qcmap_lan.@dhcp_reservation[-1].device_type="usb"
      uci set qcmap_lan.@dhcp_reservation[-1].enable="$enable_reserv_flag"
      uci set qcmap_lan.@no_of_configs[0].no_of_dhcp_reservation_records=$((no_of_dhcp_reserv_entries+1))
      uci commit
      ;;
    *)
      echo "wrong device"
    esac

}

#Edit the DHCP Reservation record on the basis of user config
#$1 -> old_ip_addr
#$2 -> new_ip_addr
#$3 -> old_enable_flag
#$4 -> new_enable_flag
#$5 -> Device_type
#$6 -> mac_addr or hostname according to device type
function edit_dhcp_reservation() {
  local old_ip_addr="$1"
  local new_ip_addr="$2"
  local old_enable_flag=$3
  local new_enable_flag=$4
  local uint8_max_val=255

  log $(basename "$0") "edit_dhcp_reservation" $LINENO "Enter edit_dhcp_reservation"

  local no_of_dhcp_reserv_entries=`uci get qcmap_lan.@no_of_configs[0].no_of_dhcp_reservation_records`

  if [ $old_ip_addr = $new_ip_addr ];then
    echo "Old reserv ip and new reserv ip are same"
    exit 1
  fi

  #checking for duplicacy of client_reserved_ip
  if [ $no_of_dhcp_reserv_entries -ne 0 ] && [ $new_ip_addr != "NULL" ]; then
    for i in $(seq $no_of_dhcp_reserv_entries)
    do
    reserv_ip=$(uci get qcmap_lan.@dhcp_reservation[$((i-1))].ip)
    if [ "$reserv_ip" = "$new_ip_addr" ];
    then
      log $(basename "$0") "edit_dhcp_reservation" $LINENO "New Reserv IP Address already reserved for another Client!!"
      exit 1
    fi
    done
  fi

  #case on the basis of Device type.
  case $5 in
    "usb" )
      local hostname=$6

      if [ $no_of_dhcp_reserv_entries -ne 0 ] && [ $hostname != "NULL" ]; then
        hostname_reservation_index=$(uci show qcmap_lan | grep -i dhcp_reservation | grep -i "name" | awk -F'[][]' '{print $2}')
        for idx in $hostname_reservation_index
        do
          reserv_name=$(uci get qcmap_lan.@dhcp_reservation[$idx].name)
          if [ "$reserv_name" ]; then
            if [ "$reserv_name" = "$hostname" ]; then
              log $(basename "$0") "edit_dhcp_reservation" $LINENO " IP Address already reserved for same Host name!!"
              exit 1
            fi
          fi
        done
      fi

      idx1=`uci show qcmap_lan | grep -i dhcp_reservation | grep -w $old_ip_addr | awk -F'[][]' '{print $2}'`
      if [ $new_ip_addr != "NULL" ];then
        uci set qcmap_lan.@dhcp_reservation[$idx1].ip=$new_ip_addr
      fi
      if [ $hostname != "NULL" ];then
        uci set qcmap_lan.@dhcp_reservation[$idx1].name=$hostname
      fi
      if [ $new_enable_flag != $uint8_max_val ];then
        uci set qcmap_lan.@dhcp_reservation[$idx1].enable="$new_enable_flag"
      fi
      uci commit
      ;;
    "non_usb" )
      local mac_addr=$6

      if [ $no_of_dhcp_reserv_entries -ne 0 ] && [ $mac_addr != "NULL" ]; then
        mac_reservation_index=$(uci show qcmap_lan | grep -i dhcp_reservation | grep -i "mac" | awk -F'[][]' '{print $2}')
        for idx in $mac_reservation_index
        do
          reserv_mac=$(uci get qcmap_lan.@dhcp_reservation[$idx].mac)
          if [ "$reserv_mac" ]; then
            if [ "$reserv_mac" = "$mac_addr" ];then
              log $(basename "$0") "edit_dhcp_reservation" $LINENO " IP Address already reserved for same MAC Addr Client!!"
              exit 1
            fi
          fi
        done
      fi

      idx1=`uci show qcmap_lan | grep -i dhcp_reservation | grep -w $old_ip_addr | awk -F'[][]' '{print $2}'`
      if [ $new_ip_addr != "NULL" ];then
        uci set qcmap_lan.@dhcp_reservation[$idx1].ip=$new_ip_addr
      fi
      if [ $mac_addr != "NULL" ];then
        uci set qcmap_lan.@dhcp_reservation[$idx1].mac=$mac_addr
      fi
      if [ $new_enable_flag != $uint8_max_val ];then
        uci set qcmap_lan.@dhcp_reservation[$idx1].enable="$new_enable_flag"
      fi
      uci commit
      ;;
    *)
      echo "wrong device"
      ;;
    esac
}

#Delete the DHCP Reservation record on the basis of IP Address provided
#$1 -> ip_addr
function delete_dhcp_reservation() {
  local addr="$1"

  log $(basename "$0") "delete_dhcp_reservation" $LINENO "Enter delete_dhcp_reservation"

  local no_of_dhcp_reserv_entries=`uci get qcmap_lan.@no_of_configs[0].no_of_dhcp_reservation_records`
  if [ $no_of_dhcp_reserv_entries -eq 0 ]; then
    log $(basename "$0") "delete_dhcp_reservation" $LINENO " No DHCP Reservation Record Entries Found to Delete!!"
    exit 1
  fi

  #index of dhcp reservation in qcmap_lan
  idx1=`uci show qcmap_lan | grep -i dhcp_reservation | grep -w $addr | awk -F'[][]' '{print $2}'`

  if [ -n "$idx1" ]; then
    uci set qcmap_lan.@no_of_configs[0].no_of_dhcp_reservation_records=$((no_of_dhcp_reserv_entries-1))
    uci delete qcmap_lan.@dhcp_reservation[$idx1]
    uci commit
  else
    log $(basename "$0") "delete_dhcp_reservation" $LINENO " No record found for the Given IP"
  fi

}

#Function updates(copies) dhcp records stored in qcmap_lan to dhcp config file
#as part of activate lan
function ActivateDHCPRecordsonLanActivation() {
  local dhcp_reserve_index
  local indicies enable device_type

  log $(basename "$0") "ActivateDHCPRecordsonLanActivation" $LINENO "Enter ActivateDHCPRecordsonLanActivation"

  #Get existing dhcp reservation records from dhcp config and remove
  dhcp_reserve_index=`uci show dhcp | grep -i dhcp_reservation | sort -r | awk -F'[][]' '{print $2}'`
  if [ -n "$dhcp_reserve_index" ]; then
    for idx in $dhcp_reserve_index
    do
      uci delete dhcp.@host[$idx]
    done
    uci commit dhcp
  fi

  #Get enabled dhcp records from qcmap_lan and update in dhcp config file
  indicies=`uci show qcmap_lan | grep -i dhcp_reservation | grep -i ip | awk -F'[][]' '{print $2}'`
  for index in $indicies
  do
    enable=$(uci get qcmap_lan.@dhcp_reservation[$index].enable)
    device_type=$(uci get qcmap_lan.@dhcp_reservation[$index].device_type)
    if [ "$enable" -eq 1 ]; then
       uci add dhcp host
       uci set dhcp.@host[-1].dns='1'
       uci set dhcp.@host[-1].ip=$(uci get qcmap_lan.@dhcp_reservation[$index].ip)
       uci set dhcp.@host[-1].instance='lan_dns'
       uci set dhcp.@host[-1].dhcp_reservation="1"
       if [ "$device_type" = "usb" ]; then
         uci set dhcp.@host[-1].name=$(uci get qcmap_lan.@dhcp_reservation[$index].name)
       else
         uci set dhcp.@host[-1].mac=$(uci get qcmap_lan.@dhcp_reservation[$index].mac)
       fi
    fi
  done

  uci commit dhcp
}

function delete_dhcp_lease_entry() {
  #This is specific to RNDIS client
  #intf mac ip hostname
  local mac=$(cat /tmp/data/rndis.txt | grep rndis0 | awk {'print $2'})
  local ip=$(cat /tmp/data/rndis.txt | grep rndis0 | awk {'print $3'})
  dhcp_release br-lan $ip $mac
  sed -i "/$mac/d" /tmp/data/dhcp.leases.lan
  /etc/init.d/dnsmasq restart
}

#This function will retrun of particular VLAN is offloaded or NOT
#$1 vlan_id
function get_ipa_offload_status() {
 local req_vlan_id=$1
 local vlan_id ipa_offload
 local no_of_vlans=$(uci get qcmap_lan.@no_of_configs[0].no_of_vlans)
 for i in $(seq $no_of_vlans)
 do
   vlan_id=$(uci get qcmap_lan.@vlan[$((i-1))].vlan_id)
   ipa_offload=$(uci get qcmap_lan.@vlan[$((i-1))].ipa_offload)
   if [ $vlan_id -eq $req_vlan_id ] && [ $ipa_offload -eq 1 ]; then
     log $(basename "$0") "get_ipa_offload_status" $LINENO " VLAN:$vlan_id found with IPA offload: $ipa_offload"
     echo 1
     return
   fi
 done

 log $(basename "$0") "get_ipa_offload_status" $LINENO " Requested VLAN:$req_vlan_id not found with IPA offload"
 echo 0
}


case $1 in
 add_vlan)
   echo "Adding vlan"
   add_vlan $2 $3 $4
   ;;
 del_vlan)
   echo "Delete VLAN "
   iface=$(util_get_interface_name $2)
   del_vlan $iface $3
   ;;
 show_vlan)
   echo "Show VLAN"
   show_vlan
   ;;
 bridge_context)
   set_bridge_context $2
   ;;
 setlanconfig)
   SetLANConfig $2 $3 $4 $5 $6 $7 $8
   ;;
 add_dhcp_reservation)
   add_dhcp_reservation $2 $3 $4 $5
   ;;
 edit_dhcp_reservation)
   edit_dhcp_reservation $2 $3 $4 $5 $6 $7
   ;;
 delete_dhcp_reservation)
   delete_dhcp_reservation $2
   ;;
 delete_dhcp_lease_entry)
   delete_dhcp_lease_entry
   ;;
 SetLANConfigOnLanActivation)
   SetLANConfigOnLanActivation $2
   ;;
 ActivateDHCPRecordsonLanActivation)
   ActivateDHCPRecordsonLanActivation
   ;;
 set_ezmesh_create_bridge_context)
   create_bridge_context $2
   ;;
 set_ezmesh_delete_bridge_context)
   delete_bridge_context $2
   ;;
 get_ipa_offload_status)
   get_ipa_offload_status $2
   ;;
*)
   logger "Invalid option"
   ;;
esac


