#ifndef _SSG_CONNSEC_CLIENT_H
#define _SSG_CONNSEC_CLIENT_H

/***************************************************************************************
 ---------------------------------------------------------------------------------------
 Copyright (c) 2023 Qualcomm Technologies, Inc.
 All Rights Reserved.
 Confidential and Proprietary - Qualcomm Technologies, Inc.
 ---------------------------------------------------------------------------------------

 @brief   Provides data structures and API interface for SSG Connection Security Client library. 
 @file    ssg_connsec_client.h
 @version Nonsecure Channel implementation.

 DESCRIPTION:
 This file provides the interface required to write a custom SSG Connection Security client application.
 The simple interface allows C++ clients to receive SSG Connection Security indications.
 The underlying library handles the actual QMI client creation, registration,
 and resource handling.
 
 REQUIREMENTS:
 Compiler:
  C++

 Application Specific Libraries:
  ssg_connsec

 Platform Libraries: 
  qmi_cci, qmi_encdec, diag, qmi_client_qmux, configdb, dsutils, gthread-2.0, 
  glib-2.0, log, time_genoff, xml, pcre, qmi

**************************************************************************************/

#include <string>
#include <vector>
#include <stdint.h>
#include <array>

/** 
 * ===========================================================================
 *                              DATA STRUCTURES
 * ===========================================================================
 */

#define SSG_CONNSEC_PAYLOAD_MAX_LEN 1024
#define WIFI_SSID_SIZE 32

/** 
 *  ssg_connsec_client_error_t: 
 * 
 * @brief 
 *  An enum used for status return codes (isolated to this application).
 *  Refer to this definition to decode.
 */
typedef enum {
  SSG_CONNSEC_CLIENT_NO_ERR = 0x0000,                   /**< No error.\n*/
  SSG_CONNSEC_CLIENT_ERR_QMI_FRAMEWORK = 0x0001,        /**< QMI framework error.\n*/
  SSG_CONNSEC_CLIENT_ERR_INVALID_CCS_LICENSE = 0x0002,  /**< Invalid CCS license error.\n*/
  SSG_CONNSEC_CLIENT_ERR_CONNECT_FAILURE = 0x0003,      /**< Connection failure error.\n*/
  SSG_CONNSEC_CLIENT_ERR_DISCONNECT_FAILURE = 0x0004,   /**< Disconnect failure.\n*/
  SSG_CONNSEC_CLIENT_ERR_STATUS_FAILURE = 0x0005,       /**< Status failure.\n*/
  SSG_CONNSEC_CLIENT_ERR_CLIENT_ARGS = 0x0006,          /**< Client arguments error.\n*/
  SSG_CONNSEC_CLIENT_ERR_INTERNAL = 0x0007,             /**< Internal error.\n*/
  SSG_CONNSEC_CLIENT_ERR_VERSION_MISMATCH = 0x0008,     /**< Version mismatch error.\n*/
  SSG_CONNSEC_CLIENT_ERR_GENERAL = 0x0009,              /**< General error.\n*/
  SSG_CONNSEC_CLIENT_ERR_UNKNOWN = 0x000A               /**< Unknown error.*/
} ssg_connsec_client_error_t;

/** 
 *  ssgccs_client_category_flag_t: 
 * 
 * @brief 
 *  An enum used to classify incoming cellular threats.
 *  Refer to this definition to decode.
 */
typedef enum {
    /*Initial state of basestation node after initializing, no indications have come yet and no scoring has been performed*/
    CATEGORY_FLAG_UNKNOWN = 0,
    /* Basestation sent ID request for IMSI (type 1)
       Basestation sent ATTACH/TAU/REGISTRATION REJECT 9 (UE ID cannot be derived from network)*/
    CATEGORY_FLAG_IMSI_LEAK = 1,
    /* Basestation has no neighbor SIBs scheduled,
       Neighbor lists length is 0,
       inter_freq_reeselection set to not allowed*/
    CATEGORY_FLAG_IMPRISONER = 2,
    /* ATTACH/TAU/REGISTRATION REJECT 9 sent,
       No response for 4+ ATTACH/TAU/REGISTRATION attempts */
    CATEGORY_FLAG_DOS = 4,
    /* ATTACH/TAU/REGISTRATION REJECT 7 sent
      RRC CONNECTION RELEASE with redirected carrier info specifying GERAN*/
    CATEGORY_FLAG_DOWNGRADE = 8,
    /* Repeated ID REQUEST for IMSI followed by REJECT pattern
      Repeated RAND in AUTHENTICATION REQUEST*/
    CATEGORY_FLAG_LOCATION_TRACKER= 16,
    /* SIB parameters configured with many PLMN,
      cell reselection priority 7 (max)*/
    CATEGORY_FLAG_ATTRACTIVE = 32,
    /* UE sent AUTH RESPONSE (auth token was valid)
      ATTACH/TAU/REGISTRATION COMPLETE*/
    CATEGORY_FLAG_AUTH_PASSED = 64,
    /* GERAN cell using null encryption*/
    CATEGORY_FLAG_NO_ENCRYPTION = 128,
    /* GERAN cell using weak encryption*/
    CATEGORY_FLAG_WEAK_ENCRYPTION = 256,
    /* Basestation lists its on PCI/EARFCN in blacklist in LTE*/
    CATEGORY_FLAG_SELF_BLACKLISTING_CELL = 512,
    /* On GERAN (without authentication) a SMS was received*/
    CATEGORY_FLAG_UNAUTH_SMS = 1024,
    /* On GERAN (without authentication) an emergency message was received*/
    CATEGORY_FLAG_UNAUTH_EMERGENCY_MSG = 2048,
    /* Whenever the Circuit Switched call is connected*/
    CATEGORY_FLAG_CS_CALL_CONNECTED = 4096,
    /* Used for location tracking detection in LTE, a basestation sent a REJECT (could be any code)*/
    CATEGORY_FLAG_REJECT_SENT = 8192,
    /* Repeated RAND sent in AUTHENTICATION REQUEST in NR/LTE*/
    CATEGORY_FLAG_LOCATION_TRACKER_AUTH_REQEUST = 16384 
} ssgccs_client_category_flag_t;

/** 
 *  ssgccs_client_policy_action_t: 
 * 
 * @brief 
 *  An enum used to classify the countermeasure policy.
 *  Refer to this definition to decode.
 */
typedef enum {
  /* NO Countermeasure determined*/
  POLICY_NO_ACTION,
  /* Cell DEPRIORITIZATION Countermeasure */
  POLICY_DEPRIORITIZE,
  /* Cell BARRING Countermeasure */
  POLICY_BAR,
  /* Cell UNDEPRIORITIZATION Countermeasure */
  POLICY_UNDEPRIORITIZE,
  /* Cell UNBARRING Countermeasure */
  POLICY_UNBAR,
  /* Abnormal Countermeasure */
  POLICY_ABNORMAL
} ssgccs_client_policy_action_t;

/**
 *  ssgccs_radio_enum_t:
 *
 * @brief
 *  An enum used to classify the Radio Access Technology.
 */
typedef enum radio_enum
{
    /* invalid/unknown radio */
    RADIO_UKNOWN,
    /* 1G RAT */
    RADIO_1,
    /* 2G/GERAN RADIO */
    RADIO_GERAN,
    /* WCDMA/3G RADIO */
    RADIO_WCDMA,
    /* LTE/4G RADIO */
    RADIO_LTE,
    /* NR/5G RADIO */
    RADIO_NR
} ssgccs_radio_enum_t;

/**
 *  ssgccs_environmental_state_t:
 * 
 * @brief
 *  An enum describing SSGCCS environment state. 'env_state' of
.*..ssgccs_indication_payload_t determines this and it indicates
.*..Environment state(UNKNOWN:0, SAFE:1, ALERT:2, HOSTILE:3). Which
.*..is extracted from first 2bits of env_state from LSB in JSON
 *..formatted indication string.
 */
typedef enum environmental_state
{
    UNKNOWN,
    SAFE,
    ALERT,
    HOSTILE
} ssgccs_environmental_state_t;

/**
 *  ssgccs_state_t:
 * 
 * @brief
 *  Enum that specifies the state of SSGCCS.
 */
typedef enum {
   SSGCCS_OFFLINE,
   SSGCCS_ONLINE
} ssgccs_state_t;

/**
 *  ssgccs_session_state_t:
 * 
 * @brief
 *  A struct used to keep track of the current running session.
 *  Provides information of interest to optionally be used by the client.
 */
typedef struct {
  /* No of indications received in current session */
  uint32_t indication_count = 0;
  /* No of indications with score >= SSGCCS_HOSTILE_SCORE(ie., 385) */
  uint32_t hostile_score_count = 0;
  /* Session start timestamp */
  std::string session_start_time = "";
  /* Session average score */
  float average_score = 0;
  /* Various categories detected in the session */
  uint32_t categories_detected = 0;
  /* Policy determined recently on the BS UE connected to */
  std::string most_recent_policy_acted = "";
  /* Indicates if countermeasure was enacted (Cell BAR/DEPRIOR) */
  int8_t was_countermeasure_enacted = 0;
} ssgccs_session_state_t;

/**
 *  ssgccs_indication_payload_t:
 *
 * @brief
 *  Optional struct provided to unpack JSON indication buffer.
 *  Contains all fields from a CCS modem indication, as well as
 *  a 'to_string' convenience method.
 */
typedef struct {
  /* Indication payload type, 
  Eg: ssgccs_modem_cell_report or SSGCCS_ReportParsingError etc.. */
  std::string identifier;
  /* Threat score of the base station */
  int32_t score;
  /* unique cell id of the base station(enodeB) with network */
  uint32_t cid;
  /* Physical cell id of the base station if radio is NR/LTE/WCDMA
     if radio is GERAN then it is reused for LAC(Location Area Code)*/
  uint32_t pid;
  /* Mobile country code */
  uint32_t mcc;
  /* Mobile network code */
  uint32_t mnc;
  /* Threat categories identified */
  uint32_t category;
  /* Determined environment state */
  ssgccs_environmental_state_t env_state;
  /* Radio Access Technology */
  ssgccs_radio_enum_t radio;
  /* Countermeasure policy determined for the threat score */
  uint32_t policy_acted;
  /* SSGCCS version */
  uint64_t ssgccs_version;
  std::string to_string()
  {
    return "Message ID: " + identifier + "\n"
           "Score: " + std::to_string(score) + "\n"
           "Cell ID: " + std::to_string(cid) + "\n"
           "Physical ID: " + std::to_string(pid) + "\n"
           "Mobile Country Code: " + std::to_string(mcc) + "\n"
           "Mobile Network Code: " + std::to_string(mnc) + "\n"
           "Category: " + std::to_string(category) + "\n"
           "Environment State: " + std::to_string(env_state) + "\n"
           "Radio Tech: " + std::to_string(radio) + "\n"
           "SSG Policy Action: " + std::to_string(policy_acted) + "\n"
           "SSG Service Version: " + std::to_string(ssgccs_version) + "\n"
    ;
  }
} ssgccs_indication_payload_t;

/**
 *  ssgccs_indication_cb_t:
 *
 * @brief
 *  A callback type definition used as the main callback for Cellular Connection
 *  Security indications from the modem. Business logic related to each indication
 *  event should be placed here. The function gets indication in JSON formatted String
 *..and indication_cookie pointer passed by client.
 */
typedef int32_t (*ssgccs_indication_cb_t)(std::string, void * indication_cookie);

/**
 *  ssgccs_indication_config_t:
 *
 * @brief
 *  SSGCCS indication callback configuration that is passed to ssg_connsec_ccs_connect().
 *..It contains callback that triggers upon recival of modem SSGCCS indications
 *.., Cookie info that can passed by client  and indication trailing queue size
 */
typedef struct {
    ssgccs_indication_cb_t callback;
    void *cookie;
    uint32_t trailing_indication_queue_size;
}ssgccs_indication_config_t;

/**
 *  ssgccs_ssr_cb_t:
 *
 * @brief
 *  Callback that gets invoked when Modem SSR(Subsystem Recovery) happens and
 *  gets ssgccs state and cookie information.
 */
typedef void(*ssgccs_ssr_cb_t) (ssgccs_state_t ccs_state, void * ssr_cookie);

/** 
 *  ssgccs_ssr_config_t: 
 * 
 * @brief
 *  SSGCCS SSR(Subsystem Recovery) configuration. It contains callback 
 *..that gets invoked when Modem undergoes SSR and Cookie info passed by client
 */
typedef struct {
    ssgccs_ssr_cb_t callback;
    void *cookie;
}ssgccs_ssr_config_t;

/** 
 *  ptr_ssgccs_client_handle_t: 
 * 
 * @brief 
 *  A client handle used internally as a token between the library/application and 
 *  the QMI framework (may be unused).
 */
typedef uint8_t* ptr_ssgccs_client_handle_t;

/** 
 *  QWifiMlClassification: 
 * 
 * @brief 
 *  An enum used for ML classification
 *  Refer to this definition to decode.
 */
typedef enum {
    QkBenign = 0,    /**< No threat detected */
    QkMalicious,     /**< The Machine Learning model flagged the Access Point as malicious */
    QkNewAssociation,/**< The remote service has not connected to this AP before and does not have enough information about it to analyze it. */
    QkNoMlResult     /**< There was no result for this Access Point. This could be because the device is moving or the AP is on the fringes of signal strength */
} QWifiMlClassification;

enum QWifiAttackAnalysis {
    QkNoAttackFound = 0, /**< No threat was found for the attack vector associated with this result */
    QkAttackFound,       /**< The Attack vector associated with this result was detected */
    QkNoAttackResult,    /**< There was no result for the attack vector associated with this result */
};

/** 
 *  QWifiAttackType: 
 * 
 * @brief 
 *  An enum used for Wifi Attack type.
 *  Refer to this definition to decode.
 */
typedef enum {
    QkApSummoning = 0,  /**< AP Summoning Attack */
    // QkDeauthentication, /**< Deauthentication Attack will be handle by real time detection */
    // ArpSpoofing,
    // kVariationInOpenVsSecureNetworkFlagForSameBssid,
    // kVariationInOpenVsSecureNetworkFlagForSameSsid,
    QkWifiAttackTypeSize // Must be last
} QWifiAttackType;

struct QWifiAttack {
    QWifiAttackType type; /**< Type.*/
    QWifiAttackAnalysis analysis; /**< Analysis.*/
};

struct QWifiDeauthInfo {
    int wiphy;      /**< Physical layer hardware corresponding to the wireless interface */
    int ifidx;      /**< Index of the wireless interface */
    int reason;     /**< Reason code for deauth */
    bool by_ap;     /**< True if deauth is initiated by the AP, False otherwise */
    int confidence; /**< Confidence percentage */
};

struct QWifiSsid {
    int length;                  /**< Length.*/
    char ssid[WIFI_SSID_SIZE+1]; /**< SSID. */
};

/** 
 *  TrustedAP: 
 * 
 */
typedef struct {
  std::string ssid;         /**< SSID of Trusted AP. */
  QWifiSsid ssid_struct;    /**< Structure of Trusted AP's SSID. */
  std::string bssid; /**< BSSID of Trusted AP. */
  long entryTimestamp;      /**< Entry Time Stamp of Trusted AP. */
  int clusterId;            /**< Cluster IDog of Trusted AP. */
} TrustedAP;

/** 
 *  ssgwcs_indication_payload_t: 
 * 
 * @brief
 *  Optional struct provided to the indication buffer.
 *  Contains all fields from a wireless communication service (WCS) indication and
 *  a 'to_string' convenience method.
 */
typedef struct {
  std::string identifier;                                 /**< ID. */
  std::string ssid;                                       /**< SSID. */
  std::string bssid;                                      /**< BSSID. */
  bool is_associated;                                     /**< Indicates whether the AP is currently associated. */
  bool is_secure;                                         /**< Indicates whether the AP is in secure connection. */
  QWifiMlClassification classification;                   /**< Classification information of AP. */
  std::array<QWifiAttack, QkWifiAttackTypeSize> attacks;  /**< Array of Attack Analysis information of AP. */
  float confidence;                                       /**< Confidence score of AP based on ML Algorithm. */
  std::string to_string()
  {
    std::string attack_result = "";
    for (auto attack : attacks) {
        attack_result += std::to_string(attack.type) + ":" + std::to_string(attack.analysis);
    }
    return "Message ID: " + identifier + "\n"
           "SSID: " + ssid + "\n"
           "BSSID: " + bssid + "\n"
           "Currently Associated: " + std::to_string(is_associated) + "\n"
           "Secure connection: " + std::to_string(is_secure) + "\n"
           "ML Classification: " + std::to_string(classification) + "\n"
           "Wifi Attack : " + attack_result + "\n"
           "Confidence Value: " + std::to_string(confidence) + "\n"
    ;
  }
} ssgwcs_indication_payload_t;


/** 
 *  ssgwcs_indication_cb_t: 
 * 
 * @brief 
 *  A callback type definition used as the main callback for Wifi Connection
 *  Security indications. Business logic related to each indication
 *  event should be placed here. 
 */

typedef int32_t (*ssgwcs_indication_cb_t)(std::string, void*);

/** 
 *  ssgwcs_deauth_cb_t: 
 * 
 * @brief 
 *  A callback type definition used as one of the main callbacs for Wifi Connection
 *  Security. Business logic related to deauthentication attack detection
 *  event should be placed here. 
 */

typedef int32_t (*ssgwcs_deauth_cb_t)(const QWifiDeauthInfo deauth_info, void *cookie);

/** 
 *  ssgwcs_trustedAP_cb_t: 
 * 
 * @brief 
 *  A callback type definition used as one of the main callbacs for Wifi Connection
 *  Security. Business logic related to user selection for Trusted AP (for ML Detection)
 *  event should be placed here. 
 */

typedef bool (*ssgwcs_trustedAP_cb_t)(std::string ssid, std::string bssid, void *cookie);

/**
 *  ssgwcs_client_config_t:
 *
 * @brief
 *  SSGWCS indication callback configuration that is passed to ssg_connsec_wcs_connect().
 *  It contains callback that triggers upon recival of SSGwCS indications,
 *  deauthentication callback and trustAP callback,
 *  Cookie info that can passed by client and indication trailing queue size
 */
typedef struct {
    ssgwcs_indication_cb_t indication_cb;
    ssgwcs_deauth_cb_t deauth_cb;
    ssgwcs_trustedAP_cb_t trustedAP_cb;
    void *cookie;
}ssgwcs_client_config_t;

#ifdef __cplusplus
extern "C" {
#endif

/** 
 * ===========================================================================
 *                                API CALLS
 * =========================================================================== 
 */

/** 
 * ===========================================================================
 * FUNCTION: ssg_connsec_client_get_status
 * ===========================================================================
 * 
 * @brief
 *	Used to retrieve information about the current session and status of the underlying
 *  QMI client library (e.g. is the service running, how many indications so far, etc.).
 * 
 * @param status  [out]  A pointer to a pointer to the struct type provided above (ssgccs_session_state_t).
 *  The struct that it points to will be modified by the callee, and can be used by the client
 *  for any purpose after the return.
 *
 * @return
 *  Status code indicating success or error (ssg_connsec_client_error_t).
 */
ssg_connsec_client_error_t ssg_connsec_ccs_register_session_state_ptr(ssgccs_session_state_t** status);

/**
 * ===========================================================================
 * FUNCTION: ssg_connsec_ccs_connect
 * ===========================================================================
 *
 * @brief
 *	Allows the client application to easily connect with the underlying CCS QMI client
 *  library. The underlying library handles all QCCI/QMI interface calls and setup.
 *  The client application should disconnect after usage by calling ssg_connsec_ccs_disconnect.
 *
 * @param ssgccs_indication_config_t  [in]  This contains SSGCCS indication callback configuration:
 *.A callback that is invoked upon receival of modem SSGCCS indications, Cookie information that
 *.the client passes and trailing_indication_queue_size. The trailing_indication_queue_size indicates
 *.the size of the Queue that maintains last N indications returned by
 *.ssg_connsec_ccs_get_recent_indications().Valid Range is 1 to 10 and defaults to 5 if invalid.
 *.The callback must conform to the above function type (ssgccs_indication_cb_t).
 *
 * @param ssgccs_ssr_config_t  [in]  SSR(Subsystem Restart) config of ssgccs that includes callback
 * .and cookie passed by the client
 *
 * @return
 *  Status code indicating success or error (ssg_connsec_client_error_t).
 */
ssg_connsec_client_error_t ssg_connsec_ccs_connect(ssgccs_indication_config_t indication_config, ssgccs_ssr_config_t ssr_config);

/**
 * ===========================================================================
 * FUNCTION: ssg_connsec_ccs_disconnect
 * ===========================================================================
 *
 * @brief
 *	Allows the client application to disconnect from the underlying QMI library.
 *  On disconnect, the library will unregister the application's instance with the
 *  QMI framework, releasing resources and handling cleanup. Subsequent calls to
 *  ssg_connsec_ccs_connect will re-register the client application with the 
 *  QMI framework and enable SSG indications again.
 * 
 * @param void
 *
 * @return
 *  Status code indicating success or error (ssg_connsec_client_error_t).
 */
ssg_connsec_client_error_t ssg_connsec_ccs_disconnect(void);

/**
 * ===========================================================================
 * FUNCTION: ssg_connsec_ccs_parse_indication_to_struct
 * ===========================================================================
 *
 * @brief
 *	Convenience function that allows the client application to parse an indication
 *  string into an indication struct (for convenient processing).
 *  Payload identifier will be set to "SSGCCS_ReportParsingError" on parsing error
 *  and to "ssgccs_modem_cell_report" on successful parsing.
 *
 * @param indication  [in]  A string that contains an SSGCCS JSON indication.
 *
 * @return
 *  A struct containing the parsed JSON SSGCCS indication's data.
 */
ssgccs_indication_payload_t ssg_connsec_ccs_parse_indication_to_struct(std::string indication);

/**
 * ===========================================================================
 * FUNCTION: ssg_connsec_ccs_get_recent_indications
 * ===========================================================================
 * 
 * @brief
 *	Convenience function that allows the client application to retrieve the most recent
 *  Connection Security modem indications.
 * 
 * @param void
 *
 * @return
 *  A struct vector containing the most recent SSGCCS indications (size will vary).
 */
std::vector<ssgccs_indication_payload_t> ssg_connsec_ccs_get_recent_indications(void);

/** 
 * ===========================================================================
 * FUNCTION: ssg_connsec_wcs_connect
 * ===========================================================================
 * 
 * @brief
 *	Allows the client application to easily connect with the Wifi detection features in the
 *  library. 
 *  The client application should disconnect after usage by calling ssg_connsec_wcs_disconnect.
 * 
 * @param client_config  [in]  This contains SSGCWCS indication callback configuration:
 *.A callback that is invoked upon receival of SSGWCS indications, a callback that invoked
 *.when the WCS service detects the deauthentication attack, a callback that invoked
 *.when there is a newly associated AP, and the Cookie information that
 *.the client passes. The callback must conform to the above function types
 *.(ssgwcs_indication_cb_t, ssgwcs_deauth_cb_t, and ssgwcs_trustedAP_cb_t).
 *
 * @return
 *  Status code indicating success or error (ssg_connsec_client_error_t).
 */
ssg_connsec_client_error_t ssg_connsec_wcs_connect(ssgwcs_client_config_t client_config);


/** 
 * ===========================================================================
 * FUNCTION: ssg_connsec_wcs_disconnect
 * ===========================================================================
 * 
 * @brief
 *	Allows the client application to disconnect Wifi detection features from the library.
 *  On disconnect, the library will release the WCS-related resources and handling cleanup.
 *  Subsequent calls to ssg_connsec_wcs_connect will re-register the client application 
 *  and enable SSGWCS indications again.
 * 
 * @param void
 *
 * @return
 *  Status code indicating success or error (ssg_connsec_client_error_t).
 */
ssg_connsec_client_error_t ssg_connsec_wcs_disconnect(void);


/** 
 * ===========================================================================
 * FUNCTION: ssg_connsec_wcs_getTrustedAPs
 * ===========================================================================
 * 
 * @brief
 *	Allows the client application to get the list of Trusted APs.
 * 
 * 
 * @param void
 *
 * @return
 *  Vector list of Trusted APs.
 */
std::vector<TrustedAP> ssg_connsec_wcs_getTrustedAPs(void);


/** 
 * ===========================================================================
 * FUNCTION: ssg_connsec_wcs_removeTrustedAP
 * ===========================================================================
 * 
 * @brief
 *	Allows the client application to remove the selected TrustAP from the list of TrustAPs.
 *  
 * 
 * @param trustedAp  [in]  TrustedAp type of one trustedAp
 *
 * @return void
 *  
 */
void ssg_connsec_wcs_removeTrustedAP(TrustedAP trustedAp);

/**
 * ===========================================================================
 * FUNCTION: ssg_connsec_wcs_parse_indication_to_struct
 * ===========================================================================
 *
 * @brief
 *	Convenience function that allows the client application to parse an indication
 *  string into an indication struct (for convenient processing).
 *  Payload identifier will be set to "SSGWCS_ReportParsingError" on parsing error
 *  and to "ssgwcs_report" on successful parsing.
 *
 * @param indication  [in]  A string that contains an SSGWCS JSON indication.
 *
 * @return
 *  A struct containing the parsed JSON SSGWCS indication's data.
 */
ssgwcs_indication_payload_t ssg_connsec_wcs_parse_indication_to_struct(std::string indication);

#ifdef __cplusplus
}
#endif
#endif
