#!/bin/sh
#  Copyright (c) 2022 Qualcomm Technologies, Inc.
#  All Rights Reserved.
#  Confidential and Proprietary - Qualcomm Technologies, Inc.

logEnabled=$(uci get qcmap_lan.@logging[0].enabled)
logFileName=$(uci get qcmap_lan.@logging[0].filename)

QCMAP_INTERFACE_TYPE_WLAN=1
QCMAP_INTERFACE_TYPE_ETH=2
QCMAP_INTERFACE_TYPE_ECM=3
QCMAP_INTERFACE_TYPE_RNDIS=4
QCMAP_INTERFACE_TYPE_MHI=5
QCMAP_INTERFACE_TYPE_ETH_NIC2=6
QCMAP_INTERFACE_TYPE_MACSEC=7
QCMAP_INTERFACE_TYPE_MACSEC_NIC2=8
IP_PASSTHROUGH_MODE_WITH_NAT=1
IP_PASSTHROUGH_MODE_WITHOUT_NAT=2

#Device types
QCMAP_MSGR_DEVICE_TYPE_USB=3
QCMAP_MSGR_DEVICE_TYPE_ETHERNET=4
QCMAP_MSGR_DEVICE_TYPE_ETHERNET_NIC2=16

#readonly QCMAP_INTERFACE_TYPE_NONE QCMAP_INTERFACE_TYPE_WLAN QCMAP_INTERFACE_TYPE_ETH
#readonly QCMAP_INTERFACE_TYPE_ECM QCMAP_INTERFACE_TYPE_RNDIS QCMAP_INTERFACE_TYPE_MHI
#readonly QCMAP_INTERFACE_TYPE_ETH_NIC2

checkLogUsage() {
    local retVal=0
    local argCount="$1"
    #logging will happen when exactly 4 arguments are passed
    if [[ $argCount -lt 4 || $argCount -gt 4 ]]; then
        logger "Usage: file_name function_name line_no log_text"
        retVal=1
    fi
    return "$retVal"
}

log() {
    #"$#" stores total number arguments pass to <log> API
    checkLogUsage $#
    #"$?" holds the return value of <checkLogUsage> API
    local retcheckUsage=$?
    if [ "$retcheckUsage" == "1" ]; then
        return 0
    fi
    local fileName="$1"
    local functionName="$2"
    local lineNo="$3"
    local extraLog="$4"
    local logText="$(date "+%Y-%m-%d %H:%M:%S") [$fileName][$functionName][$lineNo] $extraLog"
    if [ "$logEnabled" == "1" ]; then
        echo "$logText" >> $logFileName
    fi
    logger -t "QCMAP" "$logText"
}

#function to check whether ipv4 address passed is valid or not
function valid_ipv4() {

  ip=${1:-1.2.3.4}

  if expr "$ip" : '[0-9][0-9]*\.[0-9][0-9]*\.[0-9][0-9]*\.[0-9][0-9]*$' >/dev/null; then
    for i in 1 2 3 4; do
      if [ $(echo "$ip" | cut -d. -f$i) -gt 255 ]; then
        echo "fail"
        exit 1
      fi
    done
    echo "success"
    exit 0
  else
    echo "fail"
    exit 1
  fi
}

#function to get profile index in uci
#if no profile-id input, get the index of default profile
#else, get the index of the input profile-id
function mbb_util_get_uci_profile_index()
{
  local profile_id
  local inputProfile
  local profile_idx
  local no_of_profiles=$(uci get qcmap_lan.@no_of_configs[0].no_of_profiles)

  if [ $# != 0 ]; then
    inputProfile=$1
  else
    inputProfile=$(uci get qcmap_lan.@no_of_configs[0].default_pdn)
  fi

  for i in $(seq 0 $((no_of_profiles-1)))
  do
    profile_id=$(uci get qcmap_lan.@profile[$i].profile_id)
    #check if profile_id is same as profile
    if [ $profile_id -eq $inputProfile ]; then
      profile_idx=$i
      break
    fi
  done
  echo $profile_idx
}

function get_vlan_list_from_profile()
{
  local profile_index
  local vlanList


  if [ $# != 0 ]; then
    profile_index=$(mbb_util_get_uci_profile_index $1)
  else
    profile_index=$(mbb_util_get_uci_profile_index)
  fi

  vlanList=$(uci get qcmap_lan.@profile[profile_index].vlan_ids)
  echo $vlanList
  log  $(basename "$0") "get_vlan_list_from_profile" $LINENO " profile $inputProfile have vlan $vlanList "
  return
}

# Convert IPV4 address to unsigned int
# $1 ipv4_address
function ipv4_to_unsigned_int() {
  if [ "$1" ];then
    result=`valid_ipv4 $1`
    if [ $result = "success" ];then
      local ip=$1
      first_part=$(printf '%s\n' "$ip" |  awk -F'.' '{print $1}')
      second_part=$(printf '%s\n' "$ip" |  awk -F'.' '{print $2}')
      third_part=$(printf '%s\n' "$ip" |  awk -F'.' '{print $3}')
      fourth_part=$(printf '%s\n' "$ip" |  awk -F'.' '{print $4}')

      first_multiplier=$((${first_part}*16777216))
      second_multiplier=$((${second_part}*65536))
      third_multiplier=$((${third_part}*256))
      fourth_multiplier=$((${fourth_part}*1))

      converted_int=$((first_multiplier+second_multiplier+third_multiplier+fourth_multiplier))
      echo $converted_int
    fi
  fi
}

function qcmap_reload_mwan3()
{
  /etc/init.d/mwan3 reload

  profile_idx=$(mbb_util_get_uci_profile_index)

  #After mwan3 setting change, re-enable NAT for default backhaul
  ipv4_backhaul=$(uci -q get qcmap_lan.@profile[$profile_idx].bh_present)
  ipv6_backhaul=$(uci -q get qcmap_lan.@profile[$profile_idx].bh_present_v6)

  if [ -n $ipv4_backhaul ] ; then
    /etc/data/backhaulCommonConfig.sh switch_bh $ipv4_backhaul 1
  fi

  if [ -n $ipv6_backhaul ] ; then
    /etc/data/backhaulCommonConfig.sh switch_bh_v6 $ipv6_backhaul 1
  fi
}

#Usage : Converts unit32 to ip addr format
#$1 - ip
function uint_to_ipv4() {
  local ip=$1
  echo "$(( ($ip >> 24) % 256 )).$(( ($ip >> 16) % 256 )).$(( ($ip >> 8) % 256 )).$(( $ip % 256 ))"
}

case $1 in
    get_vlan_list_from_profile)
      get_vlan_list_from_profile $2
      ;;
    ipv4_to_unsigned_int)
      ipv4_to_unsigned_int $2
      ;;
    uint_to_ipv4)
      uint_to_ipv4 $2
      ;;
    qcmap_reload_mwan3)
      qcmap_reload_mwan3
      ;;
    *)
      ;;
esac

