#!/bin/sh
#  Copyright (c) 2023 Qualcomm Technologies, Inc.
#  All Rights Reserved.
#  Confidential and Proprietary - Qualcomm Technologies, Inc.

#source NetIfD related files
. /lib/functions.sh
. /lib/functions/network.sh
. /lib/netifd/netifd-proto.sh


#source mbbutils.sh
. /etc/data/mbbUtils.sh

RANDOM=$$

QCMAP_MAX_IPA_OFFLOAD=15


function util_combine_iptable_rules() {
  local firewall_file=/etc/firewall.user
  local nat_firewall_file=/etc/firewall.user.nat
  local porttrigger_firewall_file=/etc/firewall.user.porttrigger
  local tcpmss_firewall_filev4=/etc/firewall.user.tcpmssv4
  local tcpmss_firewall_filev6=/etc/firewall.user.tcpmssv6

  #cleanup
  cat /dev/null > $firewall_file

  #combine seprate files to /etc/firewall.user
  if [ -f "$nat_firewall_file" ]; then
    cat $nat_firewall_file >> $firewall_file
  fi
  if [ -f "$porttrigger_firewall_file" ]; then
    cat $porttrigger_firewall_file >> $firewall_file
  fi
  if [ -f "$tcpmss_firewall_filev4" ]; then
    cat $tcpmss_firewall_filev4 >> $firewall_file
  fi
  if [ -f "$tcpmss_firewall_filev6" ]; then
    cat $tcpmss_firewall_filev6 >> $firewall_file
  fi
}

# Clean up nat rules in iptables using -D.
# $1 - custom user file(e.g.: /etc/firewall.user.nat)
# $2 - iptables option(I/A)
function util_cleanup_iptables_rules() {
  local firewal_user_file=$1
  if [ -f "$firewal_user_file" ] & [ -s "$firewal_user_file" ]; then
    cat $firewal_user_file | while read line
    do
      $(echo $line | sed "s/-$2/-D/")
    done
  fi
}

# Add rules in iptables using.
# $1 - custom user file(e.g.: /etc/firewall.user.nat)
function util_add_iptables_rules() {
  local firewal_user_file=$1
  if [ -f "$firewal_user_file" ] & [ -s "$firewal_user_file" ]; then
    cat $firewal_user_file | while read line
    do
      $(echo $line)
    done
  fi
}

function util_add_tcpmss_rules() {
  local mtu=$1
  local ifname=$2
  local firewall_user_file=$3
  local ip_type=$4
  local ql_cust_firewall_config=$(uci get ql_cust_config.@qcmap_configs[0].firewall_config)

  log $(basename "$0") "enable_tcpmss" $LINENO " Installing TCP MSS rule ifname:$ifname mtu:$mtu ip_type:$ip_type"
  #Install iptable rules
  #rico-20240826, add for firewall feature preconfiguration.
  if [[ -z $ql_cust_firewall_config ]] || [ $ql_cust_firewall_config -eq 1 ]; then
    if [ $ip_type -eq 4 ]; then
      echo "iptables -t mangle -A FORWARD -p tcp -o $ifname --tcp-flags SYN,RST SYN -j TCPMSS --set-mss $((mtu-40))" >> $firewall_user_file
      echo "iptables -t mangle -A FORWARD -p tcp -i $ifname --tcp-flags SYN,RST SYN -j TCPMSS --set-mss $((mtu-40))" >> $firewall_user_file
    else
      echo "ip6tables -t mangle -A FORWARD -p tcp -o $ifname --tcp-flags SYN,RST SYN -j TCPMSS --set-mss $((mtu-60))" >> $firewall_user_file
      echo "ip6tables -t mangle -A FORWARD -p tcp -i $ifname --tcp-flags SYN,RST SYN -j TCPMSS --set-mss $((mtu-60))" >> $firewall_user_file
    fi
  fi
}


#This function will add MTU option based on new MTU option.
#$1 - ip_type: V4/V6
function util_add_mtu_option_in_firewall() {
  local ip_type=$1
  local ifname=$2
  local mtu=$3
  local firewall_user_file="/etc/firewall.user.tcmpmss${ip_type}"
  util_cleanup_iptables_rules "$firewall_user_file" "A"
  cat /dev/null > $firewall_user_file
  log $(basename "$0") "util_add_mtu_option_in_firewall" $LINENO "add TCP MSS rule to $firewall_user_file"
  util_add_tcpmss_rules "$mtu" "$ifname" "$firewall_user_file" "$ip_type"
  util_add_iptables_rules "$firewall_user_file"
  combine_iptable_rules
  uci commit
  /etc/init.d/firewall reload
}

#Function to get with_nat value saved in qcmap_lan database
function util_get_with_nat() {
	local with_nat=$(uci get qcmap_lan.@no_of_configs[0].with_nat)
	if [ -z "$with_nat" ]; then
		logger $(basename "$0")":util_get_with_nat"$LINENO":NAT config not found"
	else
		echo $with_nat
	fi
}

#Function to get profile index in qcmap_lan corresponding to the profile in context
#$1 - Current profile ID
function util_get_profile_index() {
	local profile="$1"
	local profile_idx=-1
	local profile_id

	#Get profile_idx of qcmap_lan
	local no_of_profiles=$(uci get qcmap_lan.@no_of_configs[0].no_of_profiles)
	#iterate through the profile list and find out the index matching profile
	for i in $(seq 0 $((no_of_profiles-1)))
	do
		profile_id=$(uci get qcmap_lan.@profile[$i].profile_id)
		#check if profile_id is same as profile
		if [ "$profile" = "$profile_id" ]; then
			profile_idx=$i
			break
		fi
	done

	if [ $profile_idx -eq -1 ]; then
		logger $(basename "$0")":util_get_profile_index:"$LINENO":profile index not found in qcmap_lan db. exiting!"
	fi

	echo $profile_idx
}

#Function to get current highest backhaul
function util_get_bh_present() {
  local profile_idx
  local bh

  profile_idx=$(util_get_default_profile_index )

  #Get current BH
  bh=$(uci get qcmap_lan.@profile[$profile_idx].bh_present)

  echo $bh
}
#Function to get default profile index
function util_get_default_profile_index() {
  local def_profile_id
  local profile_idx
  def_profile_id=$(util_get_default_pdn)
  profile_idx=$(util_get_profile_index $def_profile_id)

  echo $profile_idx
}

#Function to get default PDN from qcmap_lan database
function util_get_default_pdn() {
	local default_pdn=-1
	default_pdn=$(uci get qcmap_lan.@no_of_configs[0].default_pdn)

	if [ "$default_pdn" -eq -1 ]; then
		logger $(basename "$0")":util_get_default_pdn:"$LINENO": Default PDN not found."
	fi

	echo $default_pdn
}

#Function to get proxydnsv6 parameter from qcmap_lan database
function util_get_proxy_dnsv6() {
	local proxydnsv6=-1
	proxydnsv6=$(uci get qcmap_lan.@no_of_configs[0].proxydnsv6)

	if [ "$proxydnsv6" -eq -1 ]; then
		logger $(basename "$0")":util_get_proxy_dnsv6:"$LINENO": Proxy DNS v6 not found."
	fi

	echo $proxydnsv6
}

#Function to get bind parameter value
#$1 - Profile ID
function util_get_bind() {
	local profile="$1"
	local bind wan

	if [ $profile -eq 1 ]; then
		wan="wan"
	else
		wan="wan$profile"
	fi

	bind=$(uci get network.$wan.bind)

	[ -n "$bind" ] || bind=0

	echo $bind
}

#Function to create dedicated Routing Table
#$1 - Profile ID
function util_create_dedicated_rt() {
	local profile="$1"
	local bind DEDICATED_RT

	bind=$(util_get_bind $profile)

	[ $bind -eq 1 ] && {
		DEDICATED_RT=custom_bind_$profile
		rtnumber=$((100 + $profile))
		grep $DEDICATED_RT /etc/iproute2/rt_tables || {
			echo "$rtnumber $DEDICATED_RT" >> /etc/iproute2/rt_tables
		}
	}
}

#Getting mac address using ip address from dhcp lease file
#$1 is the lan network iface name
#$2 ip addr
function util_get_mac_from_ip() {
  local downstream=$1
  local ip_addr_passed=$2
  local ip_addr_list ip_addr mac_addr

  ip_addr_list=$(cat /tmp/data/dhcp.leases.$downstream | awk '{print $3}')
  for ip_addr in $ip_addr_list
  do
    if [ "$ip_addr" = "$ip_addr_passed" ]; then
      mac_addr=$(cat /tmp/data/dhcp.leases.$downstream | grep -i $ip_addr | awk '{print $2}')
      echo $mac_addr
      break
    fi
  done
}

#getting ipv4 address using mac from dhcp lease file
#$1 is the lan network iface name
#$2 is the mac address
function util_get_ip_from_mac() {
	local downstream=$1
	local mac_addr_passed=$2
	mac_addr_list=$(cat /tmp/data/dhcp.leases.$downstream | awk '{print $2}')
	for mac_addr in $mac_addr_list;
	do
		if [ "$mac_addr" = "$mac_addr_passed" ]; then
			ip_addr=$(cat /tmp/data/dhcp.leases.$downstream | grep -i $mac_addr | awk '{print $3}')
			echo $ip_addr
			break
		fi
	done
}

# Convert network to host byte order
# $1 network byte order
function util_convert_nbo_to_hbo() {
	local mask=$((0xff))
	local nbo=$1
	hbo="$((nbo>>24&mask)).$((nbo>>16&mask)).$((nbo>>8&mask)).$((nbo&mask))"
	echo "$hbo"
}

#Function to get downstream values
#$1 - Profile ID
function util_get_downstream_value_v4() {
	local profile="$1"
	local downstream wan

	if [ $profile -eq 1 ]; then
		wan="wan"
	else
		wan="wan$profile"
	fi

	downstream=$(uci get network.$wan.downstream)

	echo "$downstream"
}

#Function to get downstream values
#$1 - Profile ID
function util_get_downstream_value_v6() {
	local profile="$1"
	local downstream wan

	if [ $profile -eq 1 ]; then
		wan="wan_v6"
	else
		wan="wan${profile}_v6"
	fi

	downstream=$(uci get network.$wan.downstream)

	echo "$downstream"
}

#Function to Inform NetIfD of DNS information and add DNS information to resolv file
#$1 - Profile ID
#$2 - DNS Server IP address
function util_add_dnsv4() {
	local profile="$1"
	local DNS="$2"
	local downstream isPresent

	#Get downstream values
	downstream=$(util_get_downstream_value_v4 $profile)

	for dns in $DNS; do
		proto_add_dns_server "$dns"
		proto_add_ipv4_route "$dns" 32 "" ""
			for i in $downstream; do
				resolve_file=$(uci get dhcp."$i"_dns.resolvfile)

				#NOTE: check if the DNS entries are already present in resolv file
				isPresent=$(util_check_dns_present "$dns" $resolve_file)
				if [ -n "$isPresent" -a $isPresent -eq 1 ]; then
					#Indicates dnsserver information is already present in file
					logger $(basename "$0")":util_add_dnsv4:"$LINENO": dnsserver information is already present in file"
					continue
				fi

				#Else, add dns entry to resolve file
				echo "nameserver $dns" >> $resolve_file
			done
	done

	#Save dns information in /tmp/state/network
	for i in $downstream; do
		uci_set_state network ${i} dns4 "$DNS"
	done

}

#Function to Inform NetIfD of DNS information
#$1 - Profile ID
#$2 - DNS Server IP address
function util_add_dnsv6() {
	local profile="$1"
	local DNS6="$2"
	local downstream proxydnsv6 resolve_file
	local default_pdn isPresent

	#Get downstream values
	downstream=$(util_get_downstream_value_v6 $profile)

	#Get default PDN
	default_pdn=$(util_get_default_pdn)

	for dns in $DNS6; do
		proto_add_dns_server "$dns"
		proto_add_ipv6_route "$dns" 128 "" "" "" "" ""
		#Update network DNS (disable ProxyDNS)
		[ -n "$downstream" ] && {

		if [ $default_pdn -eq $profile ]; then
			proxydnsv6=$(uci get qcmap_lan.@global[0].proxydnsv6)
		fi

		logger $(basename "$0")":util_add_dnsv6:"$LINENO": proxydnsv6:$proxydnsv6"
		for i in $downstream; do
			#Condition to check for NULL DNS entry
			if [ "$dns" = "::" ]; then
				logger $(basename "$0")":util_add_dnsv6:"$LINENO":NULL dns entry found"
			else
				#if proxydnsv6 is not configured or enabled, dont add n/w DNS to UCI
				if ! [ -n "$proxydnsv6" -a $proxydnsv6 -eq 1 ]; then
					uci add_list dhcp.$i.dns=$dns
				fi
				resolve_file=$(uci get dhcp."$i"_dns.resolvfile)

				#NOTE: check if the DNS entries are already present in resolv file
				isPresent=$(util_check_dns_present "$dns" $resolve_file)
				if [ -n "$isPresent" -a $isPresent -eq 1 ]; then
					#Indicates dnsserver information is already present in file
					logger $(basename "$0")":util_add_dnsv6:"$LINENO": dnsserver information is already present in file"
					continue
				fi

				#Else, add dns entry to resolve file
				echo "nameserver $dns" >> $resolve_file
			fi
		done
		uci commit dhcp
		}
	done

}

#Function to inform NetIfD of clearing DNS information
#$1 - Profile ID
function util_del_dnsv4() {
	local profile="$1"
	local downstream pattern_string DNS

	#Get downstream values
	downstream=$(util_get_downstream_value_v4 $profile)

	#Delete network DNS address
	[ -n "$downstream" ] && {
		for i in $downstream
		do
			pattern_string=''
			DNS=$(uci_get_state network ${i} dns4)
			for j in $DNS; do
				pattern_string="${pattern_string}/${j}/d;"
			done

			resolve_file=$(uci get dhcp."$i"_dns.resolvfile)
			sed -i "$pattern_string" $resolve_file
		done
		uci commit dhcp
	}
}

#Function to inform NetIfD of clearing DNS6 information
#$1 - Profile ID
function util_del_dnsv6() {
	local profile="$1"
	local downstream pattern_string DNS6

	#Get downstream values
	downstream=$(util_get_downstream_value_v6 $profile)

	#Delete network DNS address
	[ -n "$downstream" ] && {
		for i in $downstream
		do
			pattern_string=''
			DNS6=$(uci get dhcp.$i.dns)
			[ -n "$DNS6" ] && uci del dhcp.$i.dns
			for j in $DNS6; do
				pattern_string="${pattern_string}/${j}/d;"
			done

			resolve_file=$(uci get dhcp."$i"_dns.resolvfile)
			sed -i "$pattern_string" $resolve_file
		done
		uci commit dhcp
	}
}

#Function to add DNS options to UCI
#Take profile as input if not act on default profile
#$1 - Profile ID
function util_add_dnsv6_options() {
	local profile_id="$1"
	local DNS6
	local downstream profile

	if [ -n "$profile_id" ]; then
		profile=$profile_id
	else
		profile=$(util_get_default_pdn)
	fi

	if [ -n "$profile" ]; then

		#source the config file
		[ -f /tmp/ipv6config$profile ] && . /tmp/ipv6config$profile
		DNS6=$DNSSERVERS6

		#Get downstream parameters of default PDN
		downstream=$(util_get_downstream_value_v6 $profile)

		if [ -n "$DNS6" ]; then
			#Update DNS and v6 on each LAN mapped
			[ -n "$downstream" ] && {
				logger $(basename "$0")":util_add_dnsv6_options:"$LINENO": adding DNS:$DNS6 for $downstream"
				for i in $downstream
				do
					for dns in $DNS6
					do
						if ! [ "$dns" = "::" ]; then
							uci add_list dhcp.$i.dns=$dns
						fi
					done
				done
				uci commit dhcp
			}
		else
			logger $(basename "$0") "util_add_dnsv6_options" $LINENO " Invalid DNS6:$DNS6!!"
			exit 1
		fi
	else
		logger $(basename "$0") "util_add_dnsv6_options" $LINENO " No profile set: $profile!"
	fi
}

#Function to Delete DNS options from UCI
#Take profile as input if not act on default profile
#$1 - Profile ID
function util_del_dnsv6_options() {
	local profile_id="$1"
	local DNS6
	local downstream profile

	if [ -n "$profile_id" ]; then
		profile=$profile_id
	else
		profile=$(util_get_default_pdn)
	fi

	if [ -n "$profile" ]; then

		#Get downstream parameters of default PDN
		downstream=$(util_get_downstream_value_v6 $profile)

		logger $(basename "$0")":del_dnsv6_options:"$LINENO":remove DNS on downstream:$downstream"
		[ -n "$downstream" ] && {
			for i in $downstream
			do
				#get DNS from UCI
				DNS6=$(uci get dhcp.$i.dns)
				if [ -n "$DNS6" ]; then
					for dns in $DNS6
					do
						uci del_list dhcp.$i.dns=$dns
					done
				else
					logger $(basename "$0") "del_dnsv6_options:"$LINENO":Invalid DNS6:$DNS6!!"
					exit 1
				fi
			done
			uci commit dhcp
		}
	else
		logger $(basename "$0")":del_dnsv6_options:"$LINENO":Invalid Profile: $profile!"
	fi
}

#Function to update MTU options in firewall
# $1 - Profile id
# $2 - IPtype
function util_update_mtu() {
  local profile=$1
  local iptype=$2
  local ifname mtu
  if [ -n "$profile" ] && [ -n "$iptype" ];
  then
    logger $(basename "$0")":util_update_mtu:"$LINENO": $profile $iptype"
  else
    logger $(basename "$0")":util_update_mtu"$LINENO": missing args, exit"
    exit 1
  fi
  # source the config file
  local file="/tmp/ipv${iptype}config${profile}"

  [ -f $file] && . $file

  if [ $iptype -eq "4" ]; then
    mtu=$IPV4MTU
  fi

  if [ $iptype -eq "6" ]; then
    mtu=$IPV6MTU
  fi
  ifname=$IFNAME
  logger $(basename "$0")":util_update_mtu"$LINENO": updating MTU option with arg ifname:$ifname mtu:$mtu iptype:$iptype"
  util_add_mtu_option_in_firewall "$iptype" "$ifname" "$mtu"
}

#Function to add MTU options on LAN side
# $1 - Profile id
# $2 - IPtype
function util_add_mtu_options() {
	local profile=$1
	local iptype=$2
	local wan
	if [ -n "$profile" ] && [ -n "$iptype" ];
	then
		logger $(basename "$0")":util_add_mtu_options:"$LINENO": $profile $iptype"
	else
		logger $(basename "$0")":util_add_mtu_options"$LINENO": missing args, exit"
		exit 1
	fi

	if [ "$iptype" -ne "4" ] && [ "$iptype" -ne "6" ];
	then
		logger $(basename "$0")":util_add_mtu_options:"$LINENO": invalid ip_type:$iptype "
		exit 1
	fi

	logger $(basename "$0")":util_add_mtu_options:"$LINENO": profile :$profile iptype: $iptype"

	if [ $profile -eq 1 ]; then
		wan="wan"
	else
		wan="wan$profile"
	fi

	# source the config file
	[ -f /tmp/ipv4config$profile ] && . /tmp/ipv4config$profile
	[ -f /tmp/ipv6config$profile ] && . /tmp/ipv6config$profile
	ip4mtu=$IPV4MTU
	ip6mtu=$IPV6MTU

	if [ "$iptype" -eq "4" -a -n "$ip4mtu" ] || [ "$iptype" -eq "6" -a -n "$ip6mtu" ]; then

		#Update MTU for v4 and v6 on WAN side
		if [ "$iptype" -eq "4" ]; then
			 uci set network."$wan".mtu=$ip4mtu
		else
			 uci set network."$wan"_v6.mtu=$ip6mtu
		fi

		#Update MTU for v4 and v6 on each LAN mapped
		local downstream
		downstream=$(uci get network.$wan.downstream)
		[ -n "$downstream" ] && {
			logger $(basename "$0")":util_add_mtu_options:"$LINENO": adding MTU:$ip4mtu/$ip6mtu for $downstream"
			for i in $downstream; do
					if [ "$iptype" -eq "4" ]; then
						 uci add_list dhcp.$i.dhcp_option_force='26,'$ip4mtu''
					fi
					if [ "$iptype" -eq "6" ]; then
						 uci set dhcp.$i.ra_mtu=$ip6mtu
					fi
			done
			uci commit dhcp
			}
	else
		logger $(basename "$0")"util_add_mtu_options:"$LINENO": NO MTU found in file"
	fi
}

#Function to delete MTU options on LAN side
# $1 - Profile id
# $2 - IPtype
function util_delete_mtu_options() {
	local profile=$1
	local iptype=$2
	local wan

	if [ -n "$profile" ] && [ -n "$iptype" ];
	then
		 logger $(basename "$0")":util_delete_mtu_options $1 $2 "
	else
		logger $(basename "$0")":util_delete_mtu_options missing args, exit"
		exit 1
	fi

	if [ "$iptype" -ne "4" ] && [ "$iptype" -ne "6" ];
	then
		logger $(basename "$0")":util_delete_mtu_options invalid ip_type:$iptype "
		exit 1
	fi

	if [ $profile -eq 1 ]; then
		wan="wan"
	else
		wan="wan$profile"
	fi

	#load MTU from WAN options set
	ip4mtu=$(uci get network."$wan".mtu)
	ip6mtu=$(uci get network."$wan"_v6.mtu)

	if [ "$iptype" -eq "4"  -a -n "$ip4mtu" ] || [ "$iptype" -eq "6" -a -n "$ip6mtu" ]; then

		#Delete MTU for v4 and v6 on WAN side
		if [ "$iptype" -eq "4" ]; then
			uci del network."$wan".mtu=$ip4mtu
		else
			uci del network."$wan"_v6.mtu=$ip6mtu
		fi

		local downstream
		downstream=$(uci get network.$wan.downstream)

		#Delete MTU for v4 & V6
		[ -n "$downstream" ] && {
		 for i in $downstream; do
			if [ "$iptype" -eq "4" ]; then
				uci del_list dhcp.$i.dhcp_option_force='26,'$ip4mtu''
			fi
			if [ "$iptype" -eq "6" ]; then
				uci del dhcp.$i.ra_mtu=$ip6mtu
			fi
			done
			uci commit dhcp
		}
	else
		logger $(basename "$0")": NO MTU found in file"
	fi
}

#Function to modify dns search into one line
function util_transform_dns_search_in_resolv()
{
  RESOLVCONF=/tmp/resolv.conf.d/resolv.conf.auto
  if [ "$(grep '^[[:space:]]*search[[:space:]]' "$RESOLVCONF" | wc -l)" -gt 1 ]
  then
	tmp="$(mktemp "$RESOLVCONF.XXXXXXX")"
	awk '{ if (/^[[:space:]]*search[[:space:]]+/) {
			 $1 = ""; terms=terms $0;
		   } else {
			 print;
		   }
		 }
		 END {
		   if (terms != "") {
			 print "search" terms;
		   }
		 }' "$RESOLVCONF" > "$tmp"
	 mv "$tmp" "$RESOLVCONF"
  fi
}

#Function to update dns search list
# $1 - dns_search_list string
	function util_update_dns_search_list() {
	local dnssearch="$1"
	proto_add_dns_search "$1"
	# Delete any default list that is set
	if [ -n "$(uci get dhcp.lan.domain)" ]; then
		uci -q delete dhcp.lan.domain
	fi
	# Split DNS Search List String by spaces
	# Process:
	# Sets 'b' equal to the prefix of 'a' ending in "div"
	# Sets positional params to suffix of 'a' ending in "div"
	# Sets 'a' equal to 'b' and repeat process until 'a' = 'b'
	a=$dnssearch; div=' '; set --
	while
		b=${a#*"$div"}
		set -- "$@" "${a%%"$div"*}"
		[ "$a" != "$b" ]
	do
		a=$b
	done
	# Create list option for each DNS name
	for name in "$@"
	do
		uci add_list dhcp.lan.domain="$name"
	done
	uci commit dhcp
	/etc/init.d/odhcpd reload
}

#Function to delete dns search list
function util_delete_dns_search_list() {
	if [ -n "$(uci get dhcp.lan.domain)" ]; then
		uci -q delete dhcp.lan.domain
		uci commit dhcp
	fi

}

#Function to setup routes and rules for Wan side network
#$1 - Wan side section name
#$2 - rmnet_dataX IP address
#$3 - rmnet_dataX interface name
#$4 - Dedicated Routing Table ID
#$5 - MTU
function util_setup_wan_route_rule_v4() {
	local cfg="$1"
	local lladdr="$2"
	local interface="$3"
	local dedicated_rt="$4"
	local mtu="$5"

	#Save the Generated IP address to /tmp/state/network
	uci_set_state network $cfg upip $lladdr

	ip "-4" rule del iif $interface
	ip "-4" rule add iif $interface table $dedicated_rt prio 10
	if [ -n "$mtu" ]; then
		ip "-4" route add default dev $interface table $dedicated_rt mtu lock $mtu
	else
		ip "-4" route add default dev $interface table $dedicated_rt
	fi

	if [ -n "$lladdr" ]; then
		ip "-4" rule del from $lladdr
		ip "-4" rule add from $lladdr table $dedicated_rt prio 10
	fi

}

#Function to setup routes and rules for Wan side network
#$1 - Wan side section name
#$2 - rmnet_dataX interface name
#$3 - Dedicated Routing Table ID
#$4 - Profile ID
#$5 - MTU
function util_setup_wan_route_rule_v6() {
	local cfg="$1"
	local interface="$2"
	local dedicated_rt="$3"
	local profile="$4"
	local mtu="$5"
	local ip6

	#Source config file
	[ -f /tmp/ipv6config$profile ] && . /tmp/ipv6config$profile
	ip6=$PUBLIC_IP6

	ip "-6" rule del iif $interface
	ip "-6" rule add iif $interface table $dedicated_rt prio 10
	ip "-6" rule del oif $interface
	ip "-6" rule add oif $interface table $dedicated_rt prio 10
	ip "-6" rule add from $ip6 table $dedicated_rt prio 10
	if [ -n "$mtu" ]; then
		ip "-6" route add default dev $interface table $dedicated_rt mtu lock $mtu
	else
		ip "-6" route add default dev $interface table $dedicated_rt
	fi

	#Save the Generated IP address to /tmp/state/network
	uci_set_state network $cfg upip6 $ip6

}

#Function to delete routes and rules for Wan side network
#$1 - WAN side section name
#$2 - Profile ID
function util_delete_wan_route_rule_v4() {
	local cfg="$1"
	local profile="$2"
	local bind updevice upip

	#Get bind parameter
	bind=$(util_get_bind $profile)

	if [ -n "$bind" -a $bind -eq 1 ]; then
		ip route flush table custom_bind_${profile}

		network_get_device updevice "$cfg"
		[ -n "$updevice" ] || updevice=$(uci_get_state network $cfg ifname)

		ip rule del iif $updevice

		upip=$(uci_get_state network $cfg upip)
		[ -n "$upip" ] && {
			ip rule del from $upip
		}
	fi

}

#Function to delete routes and rules for Wanv6 side network
#$1 - WAN side section name
#$2 - Profile ID
function util_delete_wan_route_rule_v6() {
	local cfg="$1"
	local profile="$2"
	local bind updevice upip6

	#Get bind parameter
	bind=$(util_get_bind $profile)

	if [ -n "$bind" -a $bind -eq 1 ]; then
		ip -6 route flush table custom_bind_${profile}

		network_get_device updevice "$cfg"
		[ -n "$updevice" ] || updevice=$(uci_get_state network $cfg ifname)

		ip -6 rule del iif $updevice
		ip -6 rule del oif $updevice

		upip6=$(uci_get_state network $cfg upip6)
		[ -n "$upip6" ] && {
			ip -6 rule del from $upip6
		}
	fi

}

# Function to get NetworkAddress
# $1 the IPv4 address
# $2 the netmask
# return the network address
function util_get_Network_Address() {
	i1=$(echo $1 | tr "." " " | awk '{ print $1}')
	i2=$(echo $1 | tr "." " " | awk '{ print $2}')
	i3=$(echo $1 | tr "." " " | awk '{ print $3}')
	i4=$(echo $1 | tr "." " " | awk '{ print $4}')

	m1=$(echo $2 | tr "." " " | awk '{ print $1}')
	m2=$(echo $2 | tr "." " " | awk '{ print $2}')
	m3=$(echo $2 | tr "." " " | awk '{ print $3}')
	m4=$(echo $2 | tr "." " " | awk '{ print $4}')

	network_address="$((i1 & m1))"."$((i2 & m2))"."$((i3 & m3))"."$((i4 & m4))"
	echo $network_address
}

#expand the compress IPv6 address to full address
#  $1: the compressed IPv6 address
# return: the full IPv6 address
function util_expand_IPv6_Address() {
	local addr=$1
	local expand
	local colon dcolon
	local miss

	#replace :: with 0:: if :: at the begin of the string
	addr=$(echo "$addr" | sed "s/^::/0::/")

	#replace compressed symbol :: to repeat :0
	colon=$(echo "$addr" | awk -F":" '{print NF-1}')
	dcolon=$(echo "$addr" | awk -F"::" '{print NF-1}')
	if [ $dcolon -eq 1 ]; then
		miss=$((9 - $colon))
		for i in $(seq 1 1 $miss)
		do
			#Last one expand with: only
			if [ $i -eq $miss ]; then
				expand="$expand:"
			else
				expand="$expand:0"
			fi
		done
		addr=$(echo "$addr" | sed "s/::/$expand/")
	fi
	echo $addr
}

#get the IPv6 prefix
# $1 The IPv6 address
# $2 the prefix length
# return the ipv6 prefix
function util_get_IPv6_Prefix() {
	local ip6addr
	full_groups_end=$(($2 / 16))
	partial_group_begin=$(($full_groups_end + 1))
	partial_group_width=$(($2 % 16))

	ip6addr=$(util_expand_IPv6_Address $1)

	local groups_full
	groups_full=$(echo $ip6addr | cut -d':' -f 1-$full_groups_end)

	if [ "$partial_group_width" -eq "0" ]; then
		printf "${groups_full}"
	else
		local group_partial mask_hex last_group_hex last_group_masked
		group_partial=$(echo $ip6addr | cut -d':' -f $partial_group_begin)
		mask_zero=$(printf "0x%x\n" "$(( $((1 << (16 - $partial_group_width))) - 1))")
		mask_word=$(printf "0x%x\n" "$(( $((1 << 16)) - 1))")
		mask_hex=$(printf "0x%x\n" "$(( $(($mask_word)) ^ $mask_zero))")
		last_group_hex="0x${group_partial}"
		last_group_masked=$(printf "%x\n" "$(( $(($last_group_hex)) & $mask_hex))")
		printf "${groups_full}:${last_group_masked}"
	fi
}

# set IPv6 address of down side device
# $1 down side network section name
# $2 the IPv6 address/prefix_length
function util_rmnet_lan_setup_ip6() {
	local downstream=$1
	local prefix6=$2
	local rtTable=$3
	local addr6="${prefix6%%/*}"
	prefix6="${prefix6#*/}"
	local prefix6len="${prefix6%%/*}"
	local netaddr6=$(util_get_IPv6_Prefix $addr6 $prefix6len)
	local downdevice vlan_id
	network_get_device downdevice "$downstream"
	local dhcp_pd_mode
	local ifname
	local qcmap_lan_pd_activated qcmap_lan_ext_router_mode_enabled dhcp_pd_mode

	qcmap_lan_pd_activated=$(util_execute_uci get qcmap_lan.@lan[0].qcmap_prefix_delegation_activated)
	qcmap_lan_ext_router_mode_enabled=$(util_execute_uci get qcmap_lan.@lan[0].qcmap_ext_router_mode_enabled)
	dhcp_pd_mode=$(util_execute_uci get dhcp.lan.qcmap_prefix_delegation_mode)
	if [ -n "$qcmap_lan_pd_activated" ] && [ -z "$qcmap_lan_ext_router_mode_enabled" ] && [ -z "$dhcp_pd_mode" ]; then
		logger $(basename "$0")"for legacy pd mode, add ipv6 address for br-lan after legacy pd config enabled and odhcpd reload"
		return
	fi

	vlan_id=$(echo $downstream | sed 's/[^0-9]*//g')
	if [ -z "$vlan_id" ]; then
		vlan_id=1
	fi

	uci set network.${downstream}_bind6=interface
	uci set network.${downstream}_bind6.device="$downdevice"
	uci set network.${downstream}_bind6.proto="static"
	uci set network.${downstream}_bind6.ip6addr="$netaddr6::$vlan_id/$prefix6len"
	[ -n "$rtTable" ] && uci set network.${downstream}_bind6.ip6table="$rtTable"

	# prefix sharing mode, create dummy prefix and use it as br-lan's prefix.
	dhcp_pd_mode=$(util_execute_uci get dhcp.lan.qcmap_prefix_delegation_mode)
	if [ -n "$dhcp_pd_mode" ] && [ "$dhcp_pd_mode" == "$IPV6_DHCP_PREFIX_SHARING_MODE" ]; then
		prefix_last_str=${netaddr6:(-1)}  #the last 16bit prefix value
		prefix_last_number=$(printf "%d" "0x${prefix_last_str}")
		prefix_flag=$((prefix_last_number&0x01))

		prefix_last_part=${netaddr6##*:}   #the last 16bit prefix value
		prefix_last_part_length=${#prefix_last_part}
		prefix_length=${#netaddr6}         #other pars
		prefix_others_str=${netaddr6:0:$((prefix_length-$prefix_last_part_length))}
		prefix_last4_number=$(printf "%d" "0x${prefix_last_part}")
		if [ $prefix_flag -eq 1 ]; then
			dummy_prefix_num=$((prefix_last4_number-1))
		else
			dummy_prefix_num=$((prefix_last4_number+1))
		fi
		dummy_prefix_str=$(printf "%x" $dummy_prefix_num)
		util_execute_uci_set network.${downstream}_bind6.ip6addr "$prefix_others_str$dummy_prefix_str::$vlan_id/$prefix6len"

		defaultProfileId=`uci get qcmap_lan.@no_of_configs[0].default_pdn`
		if [ -f /tmp/ipv6config$defaultProfileId ]; then
			. /tmp/ipv6config$defaultProfileId
			ifname=$IFNAME
			logger $(basename "$0")":util_rmnet_lan_setup_ip6, ifname: $ifname"
		fi
		if [ -n $ifname ]; then
			#infor ipa prefix sharing mode
			ipa idu_ipv6_config sharing ipv6_prefix $prefix_others_str$dummy_prefix_str ipv6_mask ffff:ffff:ffff:ffff pdn_name $ifname
			logger $(basename "$0")"cmd: ipa idu_ipv6_config sharing ipv6_prefix $prefix_others_str$dummy_prefix_str ipv6_mask ffff:ffff:ffff:ffff pdn_name $ifname"
		fi
	fi

	uci_set_state network ${downstream} public_gatewayip6 "$netaddr6::$vlan_id/$prefix6len"

	uci commit network

	# quectel fix problem that ipv6 prefix not exist in RA which caused by proto missing
	if [ -z "$(uci get network.lan_bind6.device)" ] || [ -z "$(uci get network.lan_bind6.proto)" ] || [ -z "$(uci get network.lan_bind6.ip6addr)" ]; then
		logger $(basename "$0")":[westring]util_rmnet_lan_setup_ip6:"$LINENO":uci retry to set bind6"
		uci set network.${downstream}_bind6.device="$downdevice"
		uci set network.${downstream}_bind6.proto="static"
		uci set network.${downstream}_bind6.ip6addr="$netaddr6::$vlan_id/$prefix6len"			
		uci commit network
	fi
		
	ubus call network reload

	#After reload network, then add Routes needed for prefix sharing
	dhcp_pd_mode=$(util_execute_uci get dhcp.lan.qcmap_prefix_delegation_mode)
	if [ -n "$dhcp_pd_mode" ] && [ "$dhcp_pd_mode" == "$IPV6_DHCP_PREFIX_SHARING_MODE" ]; then
		ip -6 route add $netaddr6::/64 via $prefix_others_str$dummy_prefix_str::2 dev br-lan metric 10
		logger $(basename "$0")"add route: ip -6 route add $netaddr6::/64 via $prefix_others_str$dummy_prefix_str::2 dev br-lan metric 10"
	fi
}

#Function to setup routes and rules to bind lan interface with wan-v6 interface
#$1 - WAN side section name
#$2 - Profile ID
function util_setup_lan_route_rule_v6() {
	local cfg="$1"
	local profile="$2"
	local downstream downdevice ip6
	local DEDICATED_RT bind downip iptmp downprefix downnetwork

	#Get downstream parameters
	downstream=$(util_get_downstream_value_v6 $profile)

	#Get bind parameter value
	bind=$(util_get_bind $profile)

	#Generate Dedicated routing table number
	DEDICATED_RT=custom_bind_$profile

	#Source config file
	[ -f /tmp/ipv6config$profile ] && . /tmp/ipv6config$profile
	ip6=$PUBLIC_IP6

	for i in $downstream
	do
		[ -n "$i" ] && network_get_device downdevice "$i"

		#If bind is set 1, then downstream must be present.
		if [ $bind -eq 1 -a -n "$downdevice" ]; then
			# Set up the ip rule to use the dedicated route table
			# for downstream device
			ip "-6" rule del iif $downdevice
			ip "-6" rule add iif $downdevice table $DEDICATED_RT prio 10

			iptmp=$(uci_get_state network ${downstream} public_gatewayip6)
			if [ -n "$iptmp" ]; then
				downip="${iptmp%%/*}"
				iptmp="${iptmp#*/}"
				downprefix="${iptmp%%/*}"
			fi
			[ -n "$downip" -a -n "$downprefix" ] && {
				downnetwork=$(util_get_IPv6_Prefix $downip $downprefix)
				[ -n "$downnetwork" ] && downnetwork="$downnetwork::"
			}

			if [ -n "$downip" ]; then
			 ip "-6" rule del from $downip
			 ip "-6" rule add from $downip table $DEDICATED_RT prio 10
			fi

			if [ -n "$downnetwork" -a -n "$downprefix" ]; then
				ip "-6" route add $downnetwork/$downprefix dev $downdevice table $DEDICATED_RT
			fi
		fi
	done

}

#Function to setup routes and rules to bind lan interface with wan interface
#$1 - WAN side section name
#$2 - Profile ID
#$3 - IPPT/Non-IPPT parameter
#$4 - IPPT bridge context (Optional)
function util_setup_lan_route_rule_v4() {
	local cfg="$1"
	local profile="$2"
	local ippt="$3"
	local ippt_bridge_context="$4"
	local downstream downdevice downnetwork downprefix downip with_nat
	local DEDICATED_RT upip bind iptmp

	#Get downstream parameters
	downstream=$(util_get_downstream_value_v4 $profile)

	#Get bind parameter value
	bind=$(util_get_bind $profile)

	#Source config file
	[ -f /tmp/ipv4config$profile ] && . /tmp/ipv4config$profile
	upip=$PUBLIC_IP

	#Generate Dedicated routing table number
	DEDICATED_RT=custom_bind_$profile

	#Get with_nat value
	with_nat=$(util_get_with_nat)

	#Setup routes
	for i in $downstream
	do
		[ -n "$i" ] && network_get_device downdevice "$i"

		#If bind is set 1, then downstream must be present.
		if [ $bind -eq 1 -a -n "$downdevice" ]; then
			# Set up the ip rule to use the dedicated route table
			# for downstream device
			ip rule del iif $downdevice
			ip rule add iif $downdevice table $DEDICATED_RT prio 10

			if [ $ippt -eq 1 ] && [ -n "$ippt_bridge_context" ] && [ "$ippt_bridge_context" = "$i" ]; then
				#Get public gateway IP and public natmask
				downip=$(uci_get_state network ${i} public_gatewayip)
				downnetmask=$(uci_get_state network ${i} public_netmask)
			else
				#Get private gateway IP and private netmask
				downip=$(uci get network.$i.ipaddr)
				downnetmask=$(uci get network.$i.netmask)
			fi

			if [ -n "$downip" -a -n "$downnetmask" ]; then
				downnetwork=$(util_get_Network_Address $downip $downnetmask)
				#IPv4 use netmask directly
				downprefix=$downnetmask
			fi

			if [ -n "$downip" ]; then
				 ip rule del from $downip
				 ip rule add from $downip table $DEDICATED_RT prio 10
			fi

			#IP passthrough set public ip address only to the route table
			#other mode, set the downstream network to the route table
			if [ -n "$downnetwork" -a -n "$downprefix" ]; then
				#ip passthrought ipv4 mode need set only public address to the route table
				#NOTE: Current IPPT implementation is for IPv4 only.
				if [ $ippt -eq 1 ]; then
					if [ -n "$ippt_bridge_context" ] && [ "$ippt_bridge_context" = "$i" ]; then
						if [ "$with_nat" -eq 1 ]; then
							local private_ip=$(uci_get_state network ${i} ipaddr)
							local private_netmask=$(uci_get_state network ${i} netmask)
							local private_downnetwork=$(util_get_Network_Address $private_ip $private_netmask)
							ip rule add from $private_ip table $DEDICATED_RT prio 10
							ip route add $private_downnetwork/$private_netmask dev $downdevice table $DEDICATED_RT
						fi
						ip route add $upip/32 dev $downdevice table $DEDICATED_RT
					else
						#control enters this condition when v4 ippt is set but bridge_context is a different bridge
						#Check if with_nat is set. For WITHOUT_NAT, we will not add any route to custom RT
						if [ "$with_nat" -eq 1 ]; then
							ip route add $downnetwork/$downprefix dev $downdevice table $DEDICATED_RT
						fi
					fi
				else
					#Control enters this condition for v4 and v6 policy based routing (Non-IPPT)
					ip route add $downnetwork/$downprefix dev $downdevice table $DEDICATED_RT
				fi
			fi
		fi
	done

}

#Function to delete routes and rules of lan side
#$1 - Profile ID
#$2 - Active IPPT(optional)
#$3 - IPPT bridge context (optional)
function util_delete_lan_route_rule_v4() {
	local profile="$1"
	local active_ippt="$2"
	local ippt_bridge_context="$3"
	local downstream downdevice bind with_nat
	local default_pdn downip

	#Get downstream values
	downstream=$(util_get_downstream_value_v4 $profile)

	#Get bind paramter
	bind=$(util_get_bind $profile)

	#Get WITH/WITHOUT NAT parameter
	with_nat=$(util_get_with_nat)

	#Get default PDN parameter
	default_pdn=$(util_get_default_pdn)

	for i in $downstream
	do
		network_get_device downdevice "$i"
		[ -n "$downdevice" ] || downdevice=$(uci_get_state network ${i} ifname)

		if [ -n "$bind" -a $bind -eq 1 ]; then
			ip rule del iif $downdevice
		fi

		#If WITH_NAT is enabled, we need to delete private_ip rule
		#Need to delete rule for policy based routing (Non-IPPT) as well
		if [ $with_nat -eq 1 ] || [ $bind -eq 1 ]; then
			downip=$(uci_get_state network $i ipaddr)
			#Add exception to these rules if profile is default_pdn since we do not use dedicated_rt
			if [ $default_pdn -ne $profile ]; then
				[ -n "$downip" ] && ip rule del from $downip
			fi
			#Delete ifname entry from network state
			uci_revert_state network ${i} ifname
		fi

		#Delete the public IP rule if ippt is enabled and bridge_context matches downstream
		if [ -n "$active_ippt" -a $active_ippt -eq 1 ]; then
			if [ -n "$ippt_bridge_context" ] && [ "$ippt_bridge_context" = "$i" ]; then
				downip=$(uci_get_state network ${i} public_gatewayip)
				#Add exception to these rules if profile is default_pdn since we do not use dedicated_rt
				if [ $default_pdn -ne $profile ]; then
					[ -n "$downip" ] && ip rule del from $downip
				fi
			fi
		fi

		#Delete all downstream ipv4 network rules from state
		uci_revert_state network ${i} public_gatewayip
		uci_revert_state network ${i} public_netmask

		if [ $with_nat -eq 2 ]; then
			uci_revert_state network ${i} private_gatewayip
			uci_revert_state network ${i} private_netmask
		fi

		#reset proxy-arp
		echo 0 > /proc/sys/net/ipv4/conf/$downdevice/proxy_arp

	done

}

#Function to delete routes and rules of lan side(V6)
#$1 - Profile ID
function util_delete_lan_route_rule_v6() {
	local profile="$1"
	local downstream downdevice bind
	local downip6 default_pdn

	#Get downstream values
	downstream=$(util_get_downstream_value_v6 $profile)

	#Get bind paramter
	bind=$(util_get_bind $profile)

	#Get default PDN parameter
	default_pdn=$(util_get_default_pdn)

	for i in $downstream
	do
		network_get_device downdevice "$i"
		[ -n "$downdevice" ] || downdevice=$(uci_get_state network ${i} ifname)

		if [ -n "$bind" -a $bind -eq 1 ]; then
			ip -6 rule del iif $downdevice
		fi

		#Iterate through each of downstream to obtain downip and downip6
		downip6=$(uci_get_state network ${i} public_gatewayip6)
		downip6="${downip6%%/*}"
		#Add exception to these rules if profile is default_pdn since we do not use dedicated_rt
		if [ $default_pdn -ne $profile ]; then
			[ -n "$downip6" ] && ip -6 rule del from $downip6
		fi

		#Delete all downstream ipv6 network rules from state
		uci_revert_state network ${i} public_gatewayip6

		uci delete network.${i}_bind6

	done

}

#Function to reset masquerade bit of corresponding firewall zone to zero
#$1 - WAN side section name
function util_reset_masq_bit() {
	local cfg="$1"
	local idx=0
	local found=0

	while true
	do
		#iterate through the firewall zone to find out matching wan zone name
		local zone_name=$(uci get firewall.@zone[$idx].name)
		if [ -z "$zone_name" ]; then
			logger $(basename "$0")":util_reset_masq_bit:"$LINENO":NULL firewall zone found"
			break
		fi
		#condition to check if zone_name is same as cfg
		if [ "$zone_name" = "$cfg" ]; then
			logger $(basename "$0")":util_reset_masq_bit:"$LINENO":cfg-$cfg found at idx-$idx"
			found=1
			break
		fi
		idx=$((idx+1))
	done
	if [ $found -eq 0 ]; then
		logger $(basename "$0")":util_reset_masq_bit:"$LINENO":Firewall zone for cfg: $cfg not found"
		return
	fi
	uci set firewall.@zone[$idx].masq='0'

	uci commit

}

#Function to set masquerade bit of corresponding firewall zone to zero
#$1 - WAN side section name
function util_set_masq_bit() {
	local cfg="$1"
	#Need to set masq bit to 1
	local idx=0
	local found=0
	while true
	do
		#iterate through the firewall zone to find out matching wan zone name
		local zone_name=$(uci get firewall.@zone[$idx].name)
		if [ -z "$zone_name" ]; then
			logger $(basename "$0")":util_set_masq_bit:"$LINENO":NULL firewall zone found"
			break
		fi
		#condition to check if zone_name is same as cfg
		if [ "$zone_name" = "$cfg" ]; then
			logger $(basename "$0")":util_set_masq_bit:"$LINENO":cfg-$cfg found at idx-$idx"
			found=1
			break
		fi
		idx=$((idx+1))
	done
	if [ $found -eq 0 ]; then
		logger $(basename "$0")":util_set_masq_bit:"$LINENO":Firewall zone for cfg: $cfg not found"
		return
	fi
	uci set firewall.@zone[$idx].masq='1'

	uci commit
	/etc/init.d/firewall reload

}

#Function to create SNAT rule
#$1 - WAN side section name
#$2 - Public IP
function util_create_snat_rule() {
	local cfg="$1"
	local ip="$2"

	#Set SNAT with the public address
	proto_add_data

	json_add_array firewall
		json_add_object ""
			json_add_string type nat
			json_add_string target SNAT
			json_add_string family inet
			json_add_string snat_ip $ip
		json_close_object
	json_close_array

	proto_close_data
	proto_send_update "$cfg"

}

#Function to add IPv6 lifetime proc entry for the WAN side interface
#$1 - Profile ID
function util_add_v6_lifetime() {
	local profile=$1
	local valid_time unit hour
	if [ -z $1 ]; then
		profile=$(uci get qcmap_lan.@no_of_configs[0].default_pdn)
	fi

	if [ $profile -eq 1 ]; then
		wan="wan_v6"
	else
		wan="wan"$profile"_v6"
	fi

	#getifacename
	if [ -f /tmp/ipv6config$profile ]; then
		. /tmp/ipv6config$profile
		ifname=$IFNAME
	fi

	if [ -z $ifname ]; then
		logger $(basename "$0")":util_add_v6_lifetime:"$LINENO": invalid $ifname"
		return 0
	fi

	local valid_lft=$(cat /proc/sys/net/ipv6/conf/$ifname/temp_valid_lft)
	if [ -z $valid_lft ]; then
		valid_lft='604800'
	fi
	local pref_lft=$(cat /proc/sys/net/ipv6/conf/$ifname/temp_prefered_lft)
	if [ -z $pref_lft ]; then
		pref_lft='86400'
	fi

	local downstream
	downstream=$(uci get network.$wan.downstream)
	[ -n "$downstream" ] && {
		logger $(basename "$0")":util_add_v6_lifetime:"$LINENO": Add Prefix timers for $downstream on $wan"
		for i in $downstream
		do
			valid_time=$(uci get dhcp.$i.leasetime)
			unit=`echo "${valid_time: -1}"`
			if [ $unit == 'h' ]; then
				hour=`echo "${valid_time::-1}"`
				valid_time=`expr $hour \* 3600`
			fi
			#Update preferred_lifetime based on lease time if ra_useleasetime is Set
			#TODO: if ra_useleasetime is set to 1 the IPV6 valid liftime will be taken from
			# from dhcpv4 leasetime, Need to handle IPV6 valid time separately.
			pref_lft=`expr $valid_time \* 2 \/ 3`
			logger $(basename "$0")":util_add_v6_lifetime:"$LINENO": pref_lft:$pref_lft valid_time:$valid_time"
			uci set dhcp.$i.preferred_lifetime=$pref_lft
			uci set dhcp.$i.ra_useleasetime='1'
		done
		uci commit dhcp
	}
}

#Function to Delete IPv6 lifetime proc entry from the WAN side interface
#$1 - Profile ID
function util_del_v6_lifetime()
{
	local profile=$1
	if [ -z $1 ]; then
		profile=$(uci get qcmap_lan.@no_of_configs[0].default_pdn)
	fi

	if [ $profile -eq 1 ]; then
		wan="wan_v6"
	else
		wan="wan"$profile"_v6"
	fi

	local downstream
	downstream=$(uci get network.$wan.downstream)
	[ -n "$downstream" ] && {
		logger $(basename "$0")":del_v6_lifetime:"$LINENO": del Prefix timers for $downstream on $profile"
		for i in $downstream
		do
			uci del dhcp.$i.preferred_lifetime
			uci set dhcp.$i.ra_useleasetime='0'
		done
		uci commit dhcp
	}
}

#Function to delete Prefix based route
#$1 - Profile ID
#$2 - Prefix V6 information
function util_del_prefix_based_route()
{
	local profile=$1
	local prefix6=$2
	local downstream default_pdn netaddr6 prefix6len addr6

	#Get downstream values
	downstream=$(util_get_downstream_value_v6 $profile)

	#Get default pdn
	default_pdn=$(util_get_default_pdn)

	#Calculate addr6 information
	addr6="${prefix6%%/*}"

	#Get prefix6len and netaddr6 information
	prefix6len="${prefix6#*/}"
	netaddr6=$(util_get_IPv6_Prefix $addr6 $prefix6len)

	if [ $default_pdn = $profile ]; then
		for i in $downstream
		do
			sleep 1
			ip -6 r d "$netaddr6::/$prefix6len" dev "br-$i" proto kernel metric 256 pref medium
			ip -6 r d "$netaddr6::/$prefix6len" dev "br-$i" proto static metric 1024 pref medium
		done
	fi
}

#Function to execute the link toggle commands based on device type parameter passed
#$1 - Device type
#$2 - downdevice
#$3 - Mac address (For Wi-Fi case)
function util_perform_link_toggle() {
	local device_type="$1"
	local downdevice="$2"
	local mac_addr="$3"
	local is_iface_linked cdt_value interfaces if_index increment="1" link_detected
#quectel add for rndis link up/down
	local is_rndis_iface

	if [ "$device_type" = "USB" -o "$device_type" = "Any" ]; then
		is_iface_linked=$(brctl show $downdevice | grep -c -E ecm0\|rndis0)
#quectel add for rndis link up/down
		is_rndis_iface=$(brctl show $downdevice | grep -c rndis0)
		log $(basename "$0") "util_perform_link_toggle" $LINENO "devicetype=$device_type:phy_count=$is_iface_linked"
		#For IPPT_WITHOUT_NAT link count(is_iface_linked) can be greater than 1
#quectel add for rndis link up/down
		if [ -e /etc/data/rndis_link_trigger ] && [ $is_rndis_iface -eq 1 ];then
			echo D > /proc/driver/rndis-000
			sleep 0.5
			echo C > /proc/driver/rndis-000
		elif [ $is_iface_linked -ge 1 ]; then
			util_inc_dec_restart_link_count "$increment" "$QCMAP_MSGR_DEVICE_TYPE_USB"
			/sbin/start_usb stop
			sleep 0.5
			/sbin/start_usb start
		fi
	fi
	if [ "$device_type" = "ETH" -o "$device_type" = "Any" ]; then
		is_iface_linked=$(brctl show $downdevice | grep -c -E eth0)
		link_detected=$(ethtool eth0 | grep -i "Link detected" | awk -F ": " '{print $2}')
		log $(basename "$0") "util_perform_link_toggle" $LINENO "devicetype=$device_type:phy_count=$is_iface_linked:link_detected=$link_detected"
		#For IPPT_WITHOUT_NAT link count(is_iface_linked) can be greater than 1
		if [ $is_iface_linked -ge 1 -a "$link_detected" = "yes" ]; then
			log $(basename "$0") "util_perform_link_toggle" $LINENO "link detected perform link toggle on ETH0"
			util_inc_dec_restart_link_count "$increment" "$QCMAP_MSGR_DEVICE_TYPE_ETHERNET"
			ethtool -r eth0
		fi
	fi
	if [ "$device_type" = "ETH_NIC2" -o "$device_type" = "Any" ]; then
		is_iface_linked=$(brctl show $downdevice | grep -c -E eth1)
		link_detected=$(ethtool eth1 | grep -i "Link detected" | awk -F ": " '{print $2}')
		log $(basename "$0") "util_perform_link_toggle" $LINENO "devicetype=$device_type:phy_count=$is_iface_linked:link_detected=$link_detected"
		#For IPPT_WITHOUT_NAT link count(is_iface_linked) can be greater than 1
		if [ $is_iface_linked -ge 1 -a "$link_detected" = "yes" ]; then
			log $(basename "$0") "util_perform_link_toggle" $LINENO "link detected perform link toggle on ETH_NIC2"
			util_inc_dec_restart_link_count "$increment" "$QCMAP_MSGR_DEVICE_TYPE_ETHERNET_NIC2"
			ethtool -r eth1
		fi
	fi
	if [ "$device_type" = "WiFi" -o "$device_type" = "Any" ]; then
		cdt_value=$(cat /sys/devices/soc0/platform_subtype_id)
		#TODO : Need to update restart_link_count logic to wlan
		#Indicates WKK device
		if [ "$cdt_value" -eq 3 ]; then
			log $(basename "$0") "util_perform_link_toggle" $LINENO "WKK device"
			if [ "$device_type" = "Any" ]; then
				interfaces=$(brctl show | grep -i ath |  tr '\n' '\t')
				wifi_interfaces=""
				for wlan_interface in $interfaces
				do
					if [[ "$wlan_interface" == *"ath"* ]]; then
						if_index=$(echo $wlan_interface | awk '{ print $1 }' | cut -c 4)
						wifi_interfaces="$wifi_interfaces ""wifi$if_index $wlan_interface"
					fi
				done
				log $(basename "$0") "util_perform_link_toggle" $LINENO "eval wifi multi_up $wifi_interfaces"
				if [ -n "$wifi_interfaces" ]; then
					eval wifi multi_up $wifi_interfaces >> /dev/null
					log $(basename "$0") "util_perform_link_toggle" $LINENO "wifi multi_down/up completed if"
				fi
				log $(basename "$0") "util_perform_link_toggle" $LINENO "wifi multi_down/up completed"
				/etc/data/wlanConfig.sh start_hostapd_cli_service_sta
			else
				wlan_interface=$(bridge fdb show | grep -i $mac_addr |awk '{print $3}')
				if_index=$(echo $wlan_interface | awk '{ print $1 }' | cut -c 4)
				if [ -n "$if_index" ]; then
					hostapd_cli -i $wlan_interface -p /var/run/wpad/hostapd-wifi${if_index}/ disassociate $mac_addr
				fi
			fi
		#Indicates HMT device
		elif [ "$cdt_value" -eq 2 ]; then
			log $(basename "$0") "util_perform_link_toggle" $LINENO "HMT device"
			if [ "$device_type" = "Any" ]; then
				/etc/data/wlanConfig.sh restart_tethered_wlan_client
			else
				wlan_interface=$(bridge fdb show | grep -i $mac_addr |awk '{print $3}')
				hostapd_cli -i $wlan_interface disassociate $mac_addr
			fi
		else
			logger $(basename "$0")":util_perform_link_toggle:"$LINENO":unknown CDT"
		fi
	fi
}

#Function to get brige id from downdevice
#$1 - downdevice
function util_get_bridge_id() {
	local downdevice="$1"
	local bridge_id

	bridge_id=$(echo $downdevice | grep -o '[0-9]*')
	if [ -z "$bridge_id" ];then
		bridge_id=1
	fi

	echo "$bridge_id"
}

#Function to enable DHCP on bridge
#$1 - downstream
function util_enableDHCP () {
	local downstream=$1

	if [ -n "$downstream" ]; then
		uci set dhcp.${downstream}.ignore="0"
	fi

}

#Function to disable DHCP on bridge
#$1 - downstream
function util_disableDHCP () {
	local downstream=$1

	if [ -n "$downstream" ]; then
		uci set dhcp.${downstream}.ignore="1"
	fi

}

#Function to delete DNSSERVER information of non-wwan backhaul
#$1 - Interface name
function util_del_non_wwan_dnsv4() {
	local interface="$1"
	local _dns4 _str resolve_file

	logger $(basename "$0")":util_del_non_wwan_dnsv4:"$LINENO": deleting DNS information for interface: $interface"

	#Get saved DNS information from /tmp/state/network
	_dns4=$(uci_get_state network ${interface} dns4)

	[ -n "$_dns4" ] && {
		for i in $_dns4; do
			_str="${_str}/${i}/d;"
		done

		resolve_file="/tmp/resolv.conf.d/resolv.conf.lan.auto"
		sed -i "$_str" $resolve_file
	}

	uci_revert_state network ${interface} dns4
	uci commit
}

#Function to update DNSSERVER information of non-wwan backhaul
#$1 - Interface name
function util_add_non_wwan_dnsv4() {
	local interface="$1"
	local _dns4 resolve_file isPresent

	logger $(basename "$0")":util_add_non_wwan_dnsv4:"$LINENO": updating DNS information for interface: $interface"

	if [ -f /tmp/ipv4config.$interface ]; then
		#get DNSSERVER information
		. /tmp/ipv4config.$interface && _dns4=$DNSSERVERS

		if [ -n "$_dns4" ]; then
			resolve_file="/tmp/resolv.conf.d/resolv.conf.lan.auto"

			#Check if dns is alreaady present in resolve_file
			isPresent=$(util_check_dns_present "$_dns4" $resolve_file)
			if [ -n "$isPresent" -a $isPresent -eq 1 ]; then
				#Indicates dnsserver information is already present in file
				logger $(basename "$0")":util_add_non_wwan_dnsv4:"$LINENO": dnsserver information is already present in file"
				return
			fi

			for dns in $_dns4; do
				echo "nameserver $dns" >> $resolve_file
			done

			#Save the DNS information in /tmp/state/network
			uci_set_state network ${interface} dns4 "$_dns4"
			uci commit
		fi
	else
		logger $(basename "$0")":util_add_non_wwan_dnsv4:"$LINENO": config file for interface: $interface not found"
	fi
}

#Function to delete DNSSERVER6 information of non-wwan backhaul
#$1 - Interface name
function util_del_non_wwan_dnsv6() {
	local interface="$1"
	local _dns6 _str resolve_file

	logger $(basename "$0")":util_del_non_wwan_dnsv6:"$LINENO": deleting DNS information for interface: $interface"

	#Get saved DNS information from /tmp/state/network
	_dns6=$(uci_get_state network ${interface}_v6 dns6)

	[ -n "$_dns6" ] && {
		for i in $_dns6; do
			_str="${_str}/${i}/d;"
		done

		resolve_file="/tmp/resolv.conf.d/resolv.conf.lan.auto"
		sed -i "$_str" $resolve_file
	}

	uci_revert_state network ${interface}_v6 dns6
	uci commit
}

#Function to Update DNSSERVER6 information of non-wwan backhaul
#$1 - Interface name
function util_add_non_wwan_dnsv6() {
	local interface="$1"
	local _dns6 resolve_file isPresent

	if [ -f /tmp/ipv6config.$interface ]; then
		#get DNSSERVER6 information
		. /tmp/ipv6config.$interface && _dns6=$DNSSERVERS6

		if [ -n "$_dns6" ]; then
			resolve_file="/tmp/resolv.conf.d/resolv.conf.lan.auto"

			#Check if dns is alreaady present in resolve_file
			isPresent=$(util_check_dns_present "$_dns6" $resolve_file)
			if [ -n "$isPresent" -a $isPresent -eq 1 ]; then
				#Indicates dnsserver information is already present in file
				logger $(basename "$0")":util_add_non_wwan_dnsv6:"$LINENO": dnsserver information is already present in file"
				return
			fi

			for dns in $_dns6; do
				echo "nameserver $dns" >> $resolve_file
			done

			#Save the DNS information in /tmp/state/network
			uci_set_state network ${interface}_v6 dns6 "$_dns6"
			uci commit
		fi
	else
		logger $(basename "$0")":util_add_non_wwan_dnsv6:"$LINENO": config file for interface: $interface not found"
	fi
}

#Function to check if dnsserver is already present in resolv file
#$1 - DNS
#$2 - resolve file name
function util_check_dns_present() {
	local dns="$1"
	local file="$2"
	local isPresent retVal=0

	for _dns in $dns; do
		isPresent=$(awk /"$_dns"/ "$file")
		if [ -n "$isPresent" ]; then
			retVal=1
			break
		fi
	done

	echo $retVal
}

#Function to execute uci set cmd for string value
#$1 - uci options
#$2 - uci value
function util_execute_uci_set() {
  uci set "$1=$2"
  log $(basename "$0") "util_execute_uci" $LINENO " uci set $1=$2"
}

#Function to execute uci cmd and print to log
#$1 - uci cmd
function util_execute_uci() {
  uci $*
  log  $(basename "$0") "util_execute_uci" $LINENO " uci $*"
}

function util_run_command() {
    local cmd="$1"  # First parameter: the command to run
    log  $(basename "$0") "util_run_command" $LINENO "$*"

    shift           # Shift the parameters to remove the first one (the command)

    # Execute the command with remaining arguments
    "$cmd" "$@"
}

#Function to generate random IP based on IP mask give to argument.
#$1 - ip_mask in form of hex 0xc0a80101
function util_get_random_ip() {
  local ip_mask=$1
  local RANDOM=$(od -An -N4 -l /dev/urandom)
  while [ "$RANDOM" -lt 0 ]
  do
    RANDOM=$(od -An -N4 -l /dev/urandom)
  done
  local ip
  ip=$(($ip_mask|$(($(($RANDOM % 255))<<8))))
  log $(basename "$0")":util_get_random_ip:"$LINENO": Generated IP:$ip"
  echo $ip
}

#Function to get CIDR from subnet mask_hex
#$1 mask in string form with doted notation
function get_mask_cidr_int() {
  local mask=$1
  local cidr=0
  local tmp
  for octet in $(echo $mask | sed 's/\./ /g')
  do
    while [ $octet -gt 0 ]
    do
        tmp=$((octet&1))
        cidr=$(($cidr + $tmp))
        octet=$((octet>>1))
    done
  done
  log $(basename "$0") "get_mask_cidr_int" $LINENO "In get_mask_cidr_int return cidr: $cidr"
  echo "$cidr"
}

#Function to valiated IP range if any conflict with default bridge and VLAN.
#$1 start IP in integer.
#$2 end IP in integer
function validate_ip_range() {
  local start=$1
  local end=$2
  local ip netmask cidr range tmp_ip start_ip_int end_ip_int no_of_vlan

  #Check IP conflict with default bridge
  ip=$(uci get qcmap_lan.@lan[0].ip)
  netmask=$(uci get qcmap_lan.@lan[0].netmask)
  cidr=$(get_mask_cidr_int $netmask)
  cidr=$((32 - $cidr))
  range=$((2**$cidr))
  tmp_ip=0x$(printf '%02x' ${ip//./ })
  start_ip_int=$(printf "%d\n" $tmp_ip)
  end_ip_int=$(($start_ip_int + $range - 1))
  if [ $end -lt $start_ip_int ] || [ $start -gt $end_ip_int ]; then
    log $(basename "$0") "validate_ip_range" $LINENO "No conflict with default bridge"
  else
    echo 0
    log $(basename "$0") "validate_ip_range" $LINENO "conflict with default bridge"
    log $(basename "$0") "validate_ip_range" $LINENO "start:$start end:$end start_ip_int:$start_ip_int end_ip_int:$end_ip_int range:$range cidr:$cidr"
    return
  fi

  no_of_vlan=$(uci get qcmap_lan.@no_of_configs[0].no_of_vlans)
  if [ $no_of_vlan -gt 1 ]; then
      #here number of VLAN is updated in add VLAN check conflict with exisiting VLAN
      for i in `seq 0 $((no_of_vlan-2))`
      do
        #check if any VLAN created and generate IP has any conflict.
        ip=$(uci get qcmap_lan.@vlan[$i].ip)
        netmask=$(uci get qcmap_lan.@vlan[$i].netmask)
        cidr=$(get_mask_cidr_int $netmask)
        cidr=$((32 - $cidr))
        range=$((2**$cidr))
        tmp_ip=0x$(printf '%02x' ${ip//./ })
        start_ip_int=$(printf "%d\n" $tmp_ip)
        end_ip_int=$(($start_ip_int + $range - 1))
        #Check if IP overlap with exisiting VLAN
        if [ $end -lt $start_ip_int ] || [ $start -gt $end_ip_int ]; then
            #ontained IP is in the range
            log $(basename "$0") "validate_ip_range" $LINENO "valid IP found"
        else
            log $(basename "$0") "validate_ip_range" $LINENO "IP conflict found"
            log $(basename "$0") "validate_ip_range" $LINENO "VLAN** start:$start end:$end start_ip_int:$start_ip_int end_ip_int:$end_ip_int range:$range cidr:$cidr"
            echo 0
            return
        fi
      done
  fi
  log $(basename "$0") "validate_ip_range" $LINENO "No conflict found"
  echo 1
}

#Function to generate a random auto-private IP and check for duplicity
#NOTE: currently used as a wrapper function for IPPT and IP Collision
function util_generate_priv_ip() {
	local random_ip ip is_addr_present

	#Get random IP from util function
	while true
	do
		#Use 169.254.50.1 as base
		random_ip=$(util_get_random_ip "0xa9fe3201")
		#Convert network to host byte order
		ip=$(util_convert_nbo_to_hbo "$random_ip")
		#Check if lladdr is already present in network state
		is_addr_present=$( grep -c "$ip" /tmp/state/network)
		if [ $is_addr_present -ne 0 ]; then
			logger $(basename "$0")":util_generate_priv_ip:"$LINENO":address found to be used. creating another random generated address"
		else
			logger $(basename "$0")":util_generate_priv_ip:"$LINENO":No match found for address. Continuing further"
			break
		fi
	done

	echo "$ip"
}

#Function to get the mapping lan and wan
#$1 - bridge_id
function util_get_bridge_id_to_wan_mapping()
{
	local  bridge_id=$1
	local  wwan_profile_num vlan_idx

	vlan_idx=$(util_get_vlan_index $bridge_id)

	if [ $bridge_id -eq 0 -o $bridge_id -eq 1 ]; then
		wwan_profile_num=1
	else
		wwan_profile_num=$(uci get qcmap_lan.@vlan[$vlan_idx].wwan_profile_num)
	fi

	echo $wwan_profile_num
}

#Function to get vlan index in qcmap_lan corresponding to the vlan_id in context
#$1 - vlan ID
function util_get_vlan_index() {
	local vlan="$1"
	#Since -1 is a valid uci index which refers to the last one
	local vlan_idx=-1
	local vlan_id

	#Get vlan_idx of qcmap_lan
	local no_of_vlans=$(uci get qcmap_lan.@no_of_configs[0].no_of_vlans)
	#iterate through the vlan list and find out the index matching vlan
	for i in $(seq 0 $((no_of_vlans-1)))
	do
		vlan_id=$(uci get qcmap_lan.@vlan[$i].vlan_id)
		#check if vlan_id is same as vlan
		if [ "$vlan" = "$vlan_id" ]; then
			vlan_idx=$i
			break
		fi
	done

	if [ $vlan_idx -eq -1 ]; then
		log $(basename "$0") "util_get_vlan_index:" $LINENO "vlan index not found in qcmap_lan db. exiting!"
	else
		# In case of error if we echo this value -1 is still a valid value
		echo $vlan_idx
	fi
}

#Changing what ifname this function retrieves when macsec is enabled
#Right now this fucntion is only used for add/del vlan
#Consider macsec before using this function
function util_get_interface_name() {
  local macsec_active=0
  iface=""
  if [ $1 -eq $QCMAP_INTERFACE_TYPE_ETH ]; then
     #ETH
     macsec_active=$(uci get qcmap_lan.@eth_nic_config[0].macsec_active)
     if [ $macsec_active -eq 0 ]; then
       iface=$(uci get qcmap_lan.@phy_iface_names[0].eth)
     else
       iface=$(uci get qcmap_lan.@eth_nic_config[0].macsec_name)
     fi
  elif [ $1 -eq $QCMAP_INTERFACE_TYPE_ECM ]; then
     #ECM
     iface=$(uci get qcmap_lan.@phy_iface_names[0].ecm)
  elif [ $1 -eq $QCMAP_INTERFACE_TYPE_RNDIS ]; then
     #RNDIS
     iface=$(uci get qcmap_lan.@phy_iface_names[0].rndis)
  elif [ $1 -eq $QCMAP_INTERFACE_TYPE_ETH_NIC2 ]; then
     #ETH_NIC2
     macsec_active=$(uci get qcmap_lan.@eth_nic_config[1].macsec_active)
     if [ $macsec_active -eq 0 ]; then
       iface=$(uci get qcmap_lan.@phy_iface_names[0].eth_nic2)
     else
       iface=$(uci get qcmap_lan.@eth_nic_config[1].macsec_name)
     fi
   fi
  echo $iface
}

#Function returns restart_link_count based on device_type
#$1 - device_type
function util_get_restart_link_count () {
	local device_type=$1
	local restart_link_count="0"

	log $(basename "$0") "util_get_restart_link_count" $LINENO "Enter function"
	if [ "$device_type" = "$QCMAP_MSGR_DEVICE_TYPE_USB" ]; then
		restart_link_count=$(uci get qcmap_lan.@no_of_configs[0].restart_usb_link_count)
	elif [ "$device_type" = "$QCMAP_MSGR_DEVICE_TYPE_ETHERNET" ]; then
		restart_link_count=$(uci get qcmap_lan.@no_of_configs[0].restart_eth_link_count)
	elif [ "$device_type" = "$QCMAP_MSGR_DEVICE_TYPE_ETHERNET_NIC2" ]; then
		restart_link_count=$(uci get qcmap_lan.@no_of_configs[0].restart_eth_nic2_link_count)
	fi

	echo $restart_link_count
}

#Function to increment/decrement restartlink count
#restartlinkcount is incremented before we perform link toggle
#This count is decrement when link comes up
#$1 - increment/decrement
#$1 - Device type
function util_inc_dec_restart_link_count () {
	local increment=$1
	local device_type=$2
	local restart_link_count="0"

	log $(basename "$0") "util_inc_dec_restart_link_count" $LINENO "Enter : increment=$increment :device_type=$device_type "

	#Fetch restart_link_count based on device_type
	restart_link_count=$(util_get_restart_link_count $device_type)
	log $(basename "$0") "util_inc_dec_restart_link_count" $LINENO "restart_link_count=$restart_link_count"
	if [ "$increment" -eq 1 -a "$restart_link_count" -eq 0 ]; then
		restart_link_count=$((restart_link_count+1))
	elif [ "$increment" -eq 0 -a "$restart_link_count" -gt 0 ]; then
		restart_link_count=$((restart_link_count-1))
	else
		log $(basename "$0") "util_inc_dec_restart_link_count" $LINENO "Invalid restart_link_count: restart_link_count=$restart_link_count"
		return
	fi

	#Based on device_type update restart_link_count
	if [ "$device_type" = "$QCMAP_MSGR_DEVICE_TYPE_USB" ]; then
		log $(basename "$0") "util_inc_dec_restart_link_count" $LINENO "USB: restart_link_count=$restart_link_count"
		uci set qcmap_lan.@no_of_configs[0].restart_usb_link_count="$restart_link_count"
	elif [ "$device_type" = "$QCMAP_MSGR_DEVICE_TYPE_ETHERNET" ]; then
		log $(basename "$0") "util_inc_dec_restart_link_count" $LINENO "ETH : restart_link_count=$restart_link_count"
		uci set qcmap_lan.@no_of_configs[0].restart_eth_link_count="$restart_link_count"
	elif [ "$device_type" = "$QCMAP_MSGR_DEVICE_TYPE_ETHERNET_NIC2" ]; then
		log $(basename "$0") "util_inc_dec_restart_link_count" $LINENO "ETH_NIC2 : restart_link_count=$restart_link_count"
		uci set qcmap_lan.@no_of_configs[0].restart_eth_nic2_link_count="$restart_link_count"
	fi

	uci commit
}

#Function to get ipv4_nat_status
#$1 - profile_id
function util_get_ipv4_nat_status() {
	local profile_id=$1
	local ipv4_nat_status profile_idx

	log $(basename "$0") "util_get_ipv4_nat_config" $LINENO "Enter profile_id=$profile_id"

	#Get profile_index.
	#Currently this feature is supported in default PDN only
	#So, we are fetching info from default PDN irrespective of passed profile_id
	default_profile=$(util_get_default_pdn)
	profile_idx=$(util_get_profile_index $default_profile)

	ipv4_nat_status=$(uci get qcmap_lan.@profile[$profile_idx].ipv4_nat_disable)
	if [ ! -n "$ipv4_nat_status" ]; then
		log $(basename "$0") "util_get_ipv4_nat_config" $LINENO "ipv4_nat_status is NULL"
	fi

	log $(basename "$0") "util_get_ipv4_nat_config" $LINENO "ipv4_nat_status=$ipv4_nat_status"

	echo "$ipv4_nat_status"
}

#Function to enable/disable DMZ based on flag passed
#$1 - profile_id
#$2 - flag indicates to enable/disable DMZ
function util_enable_disable_dmz() {
	local profile_id=$1
	local enable_dmz_flag=$2
	local redirect_rule_idxs redirect_index redirect_name redirect_dest_ip

	log $(basename "$0") "util_enable_disable_dmz" $LINENO "Enter profile_id=$profile_id enable_dmz_flag=$enable_dmz_flag"

	redirect_rule_idxs=$(uci show firewall | grep -i "DNAT" | awk -F '[][]' '{print $2}')
	log $(basename "$0") "util_enable_disable_dmz" $LINENO "redirect_rule_idx=$redirect_rule_idxs"

	for redirect_index in $redirect_rule_idxs
	do
		redirect_name=$(uci get firewall.@redirect[$redirect_index].name)
		#For DMZ redirect_name is profile_num
		if [ "$redirect_name" = "$profile_id" ]; then
			uci set firewall.@redirect[$redirect_index].enabled="$enable_dmz_flag"
		fi
	done

}

#Function to enable/disable DMZ based on flag passed
#$1 - profile_id
#$2 - flag indicates to enable/disable DMZ
function util_enable_disable_SNAT() {
	local wan_interface=$1
	local enable_snat_flag=$2
	local redirect_rule_idxs redirect_index redirect_name redirect_dest_ip

	log $(basename "$0") "util_enable_disable_SNAT" $LINENO "Enter profile_id=$profile_id enable_snat_flag=$enable_snat_flag"

	redirect_rule_idxs=$(uci show firewall | grep -i "DNAT" | awk -F '[][]' '{print $2}')
	log $(basename "$0") "util_enable_disable_dmz" $LINENO "redirect_rule_ids=$redirect_rule_idxs"

	for redirect_index in $redirect_rule_idxs
	do
		redirect_name=$(uci get firewall.@redirect[$redirect_index].name)
		#For DMZ redirect_name is profile_num
		if [ "$redirect_name" = "StaticNAT-$profile_id" ]; then
			uci set firewall.@redirect[$redirect_index].enabled="$enable_snat_flag"
		fi
	done

}

#Function to remove lease entry from lease file in case of IPPT_WITH_NAT_FCD
#$1 - interface
#$2 - ip addr
#$3 - mac addr
#$4 -vlan_id
function util_dhcp_release() {
	local interface=$1
	local ip_addr=$2
	local mac_addr=$3
	local vlan_id=$4
	local leasefile="/tmp/data/dhcp.leases.lan"
	local mapped_profile profile_idx="-1" dnsmasq_instance default_profile with_nat
	local is_ippt_active="0" device_type ippt_bridge_context

	log $(basename "$0") "util_dhcp_release" $LINENO "interface=$interface:ip_addr=$ip_addr:mac_addr=$mac_addr"

	#Get withnat state
	with_nat=$(util_get_with_nat)

	#dhcp_release utility is not working only for IPPT with NAT FCD scenario
	if [ "$with_nat" = "$IP_PASSTHROUGH_MODE_WITHOUT_NAT" ]; then
		log $(basename "$0") "util_dhcp_release" $LINENO "without nat is configured, returning"
		return
	fi

	#Check if vlan mapped to PDN on which IPPT_WITH_NAT_FCD is enabled
	if [ "$vlan_id" -eq 0 ]; then
		#If it is default bridge it wil be mapped to default PDN
		#Get default PDN and check IPPT with nat FCD is enabled
		default_profile=$(util_get_default_pdn)
		profile_idx=$(util_get_profile_index $default_profile)
		dnsmasq_instance="lan_dns"
	else
		mapped_profile=$(uci get qcmap_lan.@vlan[$vlan_id].wwan_profile_num)
		if [ -n "mapped_profile" ]; then
			profile_idx=$(util_get_profile_index $mapped_profile)
			leasefile="$leasefile"$vlan_id
			dnsmasq_instance="lan${vlan_id}_dns"
		fi
	fi
	if [ "$profile_idx" -ne -1 ]; then
		is_ippt_active=$(uci get qcmap_lan.@profile[$profile_idx].active_ippt)
	fi

	#If IPPT is active fetch device type and if device type "Any" indicates FCD
	if [ "$is_ippt_active" -eq 1 ]; then
			log $(basename "$0") "util_dhcp_release" $LINENO "IPPT is active on profile_idx=$profile_idx"
			device_type=$(uci get qcmap_lan.@profile[$profile_idx].ippt_device_type)
			ippt_bridge_context=$(uci get qcmap_lan.@profile[$profile_idx].ippt_bridge_context)
			if [ "$device_type" = "Any" -a "$ippt_bridge_context" = "$vlan_id" ]; then
				log $(basename "$0") "util_dhcp_release" $LINENO "sed -i '/$mac_addr/d' $leasefile dnsmasq_instance=$dnsmasq_instance"
				sed -i "/$mac_addr/d" $leasefile
				/etc/init.d/dnsmasq restart "$dnsmasq_instance"
			else
				log $(basename "$0") "util_dhcp_release" $LINENO "IPPTdevtype=$device_type /bridgectx=$ippt_bridge_context vlanid=$vlan_id not matched"
				return
			fi
	else
		log $(basename "$0") "util_dhcp_release" $LINENO "IPPT is not active on profile_idx=$profile_idx, return"
		return
	fi

}

# helper to convert hex to dec
function util_hex2dec(){
	[ "$1" != "" ] && printf "%d" "$(( 0x$1 ))"
}

# expand an ipv6 address
function util_expand_ipv6() {
	ip=$1

	# prepend 0 if we start with :
	echo $ip | grep -qs "^:" && ip="0${ip}"

	# expand ::
	if echo $ip | grep -qs "::"; then
		colons=$(echo $ip | sed 's/[^:]//g')
		missing=$(echo ":::::::::" | sed "s/$colons//")
		expanded=$(echo $missing | sed 's/:/:0/g')
		ip=$(echo $ip | sed "s/::/$expanded/")
	fi

	blocks=$(echo $ip | grep -o "[0-9a-f]\+")
	set $blocks

	printf "%04x:%04x:%04x:%04x:%04x:%04x:%04x:%04x\n" \
		$(util_hex2dec $1) \
		$(util_hex2dec $2) \
		$(util_hex2dec $3) \
		$(util_hex2dec $4) \
		$(util_hex2dec $5) \
		$(util_hex2dec $6) \
		$(util_hex2dec $7) \
		$(util_hex2dec $8)
}

#Function hex2binary: Hex to Binary Conversion.
function util_hex2binary() {

	if [ $# -eq 0 ]
	then
		echo "Argument(s) not supplied "
		echo "Usage: hex2binary.sh hex_number(s)"
	else

		while [ $# -ne 0 ]
		do
			DecNum=`printf "%d" 0x$1`
			Binary=
			Number=$DecNum

			while [ $DecNum -ne 0 ]
			do
				Bit=$(expr $DecNum % 2)
				Binary=$Bit$Binary
				DecNum=$(expr $DecNum / 2)
			done

			printf "%016d\n" \
			${Binary:-0}

			shift
			# Shifts command line arguments one step.Now $1 holds second argument
			unset Binary
		done
	fi
}

#Function to flush v4 conntrack entries
#$1 is ipv4 address
function util_del_v4_conntrack() {
	local ipv4=$1

	log $(basename "$0") "util_del_v4_conntrack" $LINENO "Enter ipv4=$ipv4"

	if [ -n "$ipv4" ]; then
		#Remove conntrack for UL traffic
		conntrack -D --reply-dst $ipv4
		log $(basename "$0") "util_del_v4_conntrack" $LINENO "conntrack -D --reply-dst $ipv4"
		#Remove conntrack for DL traffic
		conntrack -D --orig-dst $ipv4
		log $(basename "$0") "util_del_v4_conntrack" $LINENO "conntrack -D --orig-dst $ipv4"
	fi
}

#Function to flush v6 conntrack entries
#$1 is prefix
function util_del_v6_conntrack() {
	local prefix=$1
	local prefix_len ipv6_prefix
	local conntrack_file="/tmp/data/v6conntrack.txt"
	local prefixWords

	log $(basename "$0") "util_del_v6_conntrack" $LINENO "Enter prefix=$prefix"

	prefix_len=$(prefix | awk -F '/' '{print $2})
	ipv6_prefix=$(prefix | awk -F '/' '{print $1})
	prefixWords=$((prefix_len/16))
	prefixBits=$((prefix_len%16))
	expand_ipv6_prefix=`util_expand_ipv6 $ipv6_prefix`

	touch $conntrack_file
	conntrack -L -f ipv6 | grep -i udp | cut -f9,10,11 -d ' ' > $conntrack_file
	conntrack -L -f ipv6 | grep -i tcp | cut -f10,11,12 -d ' ' >> $conntrack_file

	if [ -f ${conntrack_file} -a -s ${conntrack_file} ]; then
		while read -r line; do
			conntrack_entry=$(echo -e "$line")

			#Fetch ipv6 src and dst address from conntrack file
			src_addr=$(printf '%s\n' "$conntrack_entry" | awk '{print $1}' | cut -c 5-)
			dst_addr=$(printf '%s\n' "$conntrack_entry" | awk '{print $2}' | cut -c 5-)

			#Fetch length of ipv6 addr to compare based on prefix
			cmp_len=$(( $((prefixWords-1)) + ( $prefixWords * 4 )))

			#Expand IPV6 address
			expand_src_ipaddr=`util_expand_ipv6 $src_addr`
			expand_dest_ipaddr=`util_expand_ipv6 $dst_addr`

			#Fetch IPV6 Prefix to compare
			cmp_src_ipaddr=`echo $expand_src_ipaddr | cut -c 1-$cmp_len`
			cmp_dest_ipaddr=`echo $expand_dest_ipaddr | cut -c 1-$cmp_len`
			cmp_prefix_ipaddr=`echo $expand_ipv6_prefix | cut -c 1-$cmp_len`

			#Delete conntracks for UL traffic
			if [ "$cmp_src_ipaddr" = "$cmp_prefix_ipaddr" ]; then
				hex1_after_prefixWords=`echo $expand_src_ipaddr | cut -c $((cmp_len+2))-$((cmp_len+5))`
				hex2_after_prefixWords=`echo $expand_ipv6_prefix | cut -c $((cmp_len+2))-$((cmp_len+5))`
				binary1=`util_hex2binary $hex1_after_prefixWords`
				binary2=`util_hex2binary $hex2_after_prefixWords`
				binary1_prefixBits=`echo $binary1 | cut -c 1-$prefixBits`
				binary2_prefixBits=`echo $binary2 | cut -c 1-$prefixBits`
				if [ "$prefixBits" -eq 0 -o "$binary1_prefixBits" = "$binary2_prefixBits" ];then
					eval "conntrack -D -f ipv6 --orig-src $src_addr"
				fi
			fi
			#Delete conntracks for DL traffic
			if [ "$cmp_dest_ipaddr" = "$cmp_prefix_ipaddr" ]; then
				hex1_after_prefixWords=`echo $expand_dest_ipaddr | cut -c $((cmp_len+2))-$((cmp_len+5))`
				hex2_after_prefixWords=`echo $expand_ipv6_prefix | cut -c $((cmp_len+2))-$((cmp_len+5))`
				binary1=`util_hex2binary $hex1_after_prefixWords`
				binary2=`util_hex2binary $hex2_after_prefixWords`
				binary1_prefixBits=`echo $binary1 | cut -c 1-$prefixBits`
				binary2_prefixBits=`echo $binary2 | cut -c 1-$prefixBits`
				if [ "$prefixBits" -eq 0 -o "$binary1_prefixBits" = "$binary2_prefixBits" ];then
					eval "conntrack -D -f ipv6 --orig-dst $dst_addr"
				fi
			fi
		done <$conntrack_file
		rm -rf $conntrack_file
	fi
}

#Function to flush v4 and v6 conntrack entries
#$1 is profile
#$2 is ip version  - ipv4 / ipv6 /ipv4v6
function util_del_conntrack() {
	local profile=$1
	local ip_version=$2
	local public_ipv4
	local prefix6

	log $(basename "$0") "util_del_conntrack" $LINENO "Enter profile=$profile:ip_version=$ip_version"

	if [ "$ip_version" = "ipv4" -o "$ip_version" = "ipv4v6" ];then
		#Get all ipv4 related values from tmp config file
		[ -f /tmp/ipv4config$profile ] && . /tmp/ipv4config$profile
		public_ipv4=$PUBLIC_IP

		log $(basename "$0") "util_del_conntrack" $LINENO "public_ipv4=$public_ipv4"
		if [ -n "$public_ipv4" ]; then
			util_del_v4_conntrack $public_ipv4
		fi
	fi

	if [ "$ip_version" = "ipv6" -o "$ip_version" = "ipv4v6" ];then
		#Get all ipv6 related values from tmp config file
		[ -f /tmp/ipv6config$profile ] && . /tmp/ipv6config$profile
		prefix6=$PREFIX6

		log $(basename "$0") "util_del_conntrack" $LINENO "prefix6=$prefix6"
		if [ -n "$prefix6" ]; then
			util_del_v6_conntrack $prefix6
		fi
	fi
}

#Function to install iptable rules to drop traffic
#$1 is wan iface name
#$2 is wan_profile_handle
#$3 is ip version  - ipv4 / ipv6 /ipv4v6
function util_install_iptables_rules_to_drop_all_traffic() {
	local wan_iface=$1
	local wan_profile_handle=$2
	local ip_version=$3
	local downstream

	log $(basename "$0") "util_install_iptables_rules_to_drop_all_traffic" $LINENO "Enter:wan_iface=$wan_iface:wan_profile_handle=$wan_profile_handle:ip_version=$ip_version"

	if [ "$ip_version" = "ipv4" -o "$ip_version" = "ipv4v6" ];then
		#iptables -t mangle -I PREROUTING -i $wan_iface -j DROP //quectel prevent commands fail in race condition
		iptables -w -t mangle -I PREROUTING -i $wan_iface -j DROP
	fi
	if [ "$ip_version" = "ipv6" -o "$ip_version" = "ipv4v6" ];then
		#ip6tables -t mangle -I PREROUTING -i $wan_iface -j DROP //quectel prevent commands fail in race condition
		ip6tables -w -t mangle -I PREROUTING -i $wan_iface -j DROP
	fi

	#Get downstream value
	downstream=$(util_get_downstream_value_v4 $wan_profile_handle)

	for lan_iface in $downstream
	do
	if [ "$ip_version" = "ipv4" -o "$ip_version" = "ipv4v6" ];then
		#iptables -t mangle -I PREROUTING -i br-$lan_iface -j DROP //quectel prevent commands fail in race condition
		iptables -w -t mangle -I PREROUTING -i br-$lan_iface -j DROP
	fi
	if [ "$ip_version" = "ipv6" -o "$ip_version" = "ipv4v6" ];then
		#ip6tables -t mangle -I PREROUTING -i br-$lan_iface -j DROP //quectel prevent commands fail in race condition
		ip6tables -w -t mangle -I PREROUTING -i br-$lan_iface -j DROP
	fi
	done

}

#Function to delete iptables drop rules traffic
#$1 is wan iface name
#$2 is wan_profile_handle
#$3 is ip version
function util_delete_iptables_rules_to_drop_all_traffic() {
	local wan_iface=$1
	local wan_profile_handle=$2
	local ip_version=$3
	local downstream

	log $(basename "$0") "util_delete_iptables_rules_to_drop_all_traffic" $LINENO "Enter:wan_iface=$wan_iface:wan_profile_handle=$wan_profile_handle:ip_version=$ip_version"

	if [ "$ip_version" = "ipv4" -o "$ip_version" = "ipv4v6" ];then
		#iptables -t mangle -D PREROUTING -i $wan_iface -j DROP //quectel prevent commands fail in race condition
		iptables -w -t mangle -D PREROUTING -i $wan_iface -j DROP
	fi
	if [ "$ip_version" = "ipv6" -o "$ip_version" = "ipv4v6" ];then
		#ip6tables -t mangle -D PREROUTING -i $wan_iface -j DROP //quectel prevent commands fail in race condition
		ip6tables -w -t mangle -D PREROUTING -i $wan_iface -j DROP
	fi

	#Get downstream value
	downstream=$(util_get_downstream_value_v4 $wan_profile_handle)

	for lan_iface in $downstream
	do
	if [ "$ip_version" = "ipv4" -o "$ip_version" = "ipv4v6" ];then
		#iptables -t mangle -D PREROUTING -i br-$lan_iface -j DROP //quectel prevent commands fail in race condition
		iptables -w -t mangle -D PREROUTING -i br-$lan_iface -j DROP
	fi
	if [ "$ip_version" = "ipv6" -o "$ip_version" = "ipv4v6" ];then
		#ip6tables -t mangle -D PREROUTING -i br-$lan_iface -j DROP //quectel prevent commands fail in race condition
		ip6tables -w -t mangle -D PREROUTING -i br-$lan_iface -j DROP
	fi
	done
}

#quectel add for multi port speed flow control Start 20240930
function util_set_pause() {
	ifaces=$(uci get network.lan.ifname)
	for if in $ifaces
	do
			if [ $1 == "off" ]; then
					ethtool --pause $if rx off
					ethtool --pause $if tx off
			elif [ $1 == "on" ]; then
					ethtool --pause $if rx on
					ethtool --pause $if tx on
			fi
	done
}
#quectel add for multi port speed flow control End

case $1 in
	util_get_with_nat)
		util_get_with_nat
		;;
	util_get_profile_index)
		util_get_profile_index $2
		;;
	util_get_default_pdn)
		util_get_default_pdn
		;;
	util_get_proxy_dnsv6)
		util_get_proxy_dnsv6
		;;
	util_get_bind)
		util_get_bind $2
		;;
	util_create_dedicated_rt)
		util_create_dedicated_rt $2
		;;
	util_get_ip_from_mac)
		util_get_ip_from_mac $2 $3
		;;
	util_get_downstream_value_v4)
		util_get_downstream_value_v4 $2
		;;
	util_get_downstream_value_v6)
		util_get_downstream_value_v6 $2
		;;
	util_add_dnsv4)
		util_add_dnsv4 $2 $3
		;;
	util_add_dnsv6)
		util_add_dnsv6 $2 $3
		;;
	util_del_dnsv4)
		util_del_dnsv4 $2
		;;
	util_del_dnsv6)
		util_del_dnsv6 $2
		;;
	util_add_dnsv6_options)
		util_add_dnsv6_options $2
		;;
	util_del_dnsv6_options)
		util_del_dnsv6_options $2
		;;
	util_add_mtu_options)
		util_add_mtu_options $2 $3
		;;
	util_delete_mtu_options)
		util_delete_mtu_options $2 $3
		;;
	util_update_dns_search_list)
		util_update_dns_search_list $2
		;;
	util_delete_dns_search_list)
		util_delete_dns_search_list
		;;
	util_transform_dns_search_in_resolv)
		util_transform_dns_search_in_resolv
		;;
	util_setup_wan_route_rule_v4)
		util_setup_wan_route_rule_v4 $2 $3 $4 $5 $6
		;;
	util_setup_wan_route_rule_v6)
		util_setup_wan_route_rule_v6 $2 $3 $4 $5 $6
		;;
	util_delete_wan_route_rule_v4)
		util_delete_wan_route_rule_v4 $2 $3
		;;
	util_delete_wan_route_rule_v6)
		util_delete_wan_route_rule_v6 $2 $3
		;;
	util_rmnet_lan_setup_ip6)
		util_rmnet_lan_setup_ip6 $2 $3 $4
		;;
	util_setup_lan_route_rule_v6)
		util_setup_lan_route_rule_v6 $2 $3
		;;
	util_setup_lan_route_rule_v4)
		util_setup_lan_route_rule_v4 $2 $3 $4 $5
		;;
	util_delete_lan_route_rule_v4)
		util_delete_lan_route_rule_v4 $2 $3 $4
		;;
	util_delete_lan_route_rule_v6)
		util_delete_lan_route_rule_v6 $2
		;;
	util_reset_masq_bit)
		util_reset_masq_bit $2
		;;
	util_set_masq_bit)
		util_set_masq_bit $2
		;;
	util_create_snat_rule)
		util_create_snat_rule $2 $3
		;;
	util_add_v6_lifetime)
		util_add_v6_lifetime $2
		;;
	util_del_v6_lifetime)
		util_del_v6_lifetime $2
		;;
	util_del_prefix_based_route)
		util_del_prefix_based_route $2 $3
		;;
	util_perform_link_toggle)
		util_perform_link_toggle $2 $3 $4
		;;
	util_get_mac_from_ip)
		util_get_mac_from_ip $2 $3
		;;
	util_get_bridge_id)
		util_get_bridge_id $2
		;;
	util_enableDHCP)
		util_enableDHCP $2
		;;
	util_disableDHCP)
		util_disableDHCP $2
		;;
	util_del_non_wwan_dnsv4)
		util_del_non_wwan_dnsv4 $2
		;;
	util_add_non_wwan_dnsv4)
		util_add_non_wwan_dnsv4 $2
		;;
	util_del_non_wwan_dnsv6)
		util_del_non_wwan_dnsv6 $2
		;;
	util_add_non_wwan_dnsv6)
		util_add_non_wwan_dnsv6 $2
		;;
	util_get_random_ip)
		util_get_random_ip $2
		;;
	util_convert_nbo_to_hbo)
		util_convert_nbo_to_hbo $2
		;;
	util_generate_priv_ip)
		util_generate_priv_ip
		;;
	util_get_restart_link_count)
		util_get_restart_link_count $2
		;;
	util_inc_dec_restart_link_count)
		util_inc_dec_restart_link_count $2 $3
		;;
	util_get_ipv4_nat_status)
		util_get_ipv4_nat_status $2
		;;
	util_enable_disable_dmz)
		util_enable_disable_dmz $2 $3
		;;
	util_enable_disable_SNAT)
		util_enable_disable_SNAT $2 $3
		;;
	util_dhcp_release)
		util_dhcp_release $2 $3 $4 $5
		;;
	util_get_bh_present)
	 	util_get_bh_present
	 	;;
	util_get_default_profile_index)
	 	util_get_default_profile_index
	 	;;
	util_del_conntrack)
		util_del_conntrack $2 $3
	;;
	util_install_iptables_rules_to_drop_all_traffic)
		util_install_iptables_rules_to_drop_all_traffic $2 $3 $4
	;;
	util_delete_iptables_rules_to_drop_all_traffic)
		util_delete_iptables_rules_to_drop_all_traffic $2 $3 $4
	;;
	util_set_pause)
	util_set_pause $2
	;;
	*)
		logger $(basename "$0")":"$LINENO":Invalid option"
		;;
esac
