#!/bin/sh
#  Copyright (c) 2023 Qualcomm Technologies, Inc.
#  All Rights Reserved.
#  Confidential and Proprietary - Qualcomm Technologies, Inc.

#source common functions
. /etc/data/lanUtils.sh

#source common functions
. /etc/data/mbbUtils.sh

#source NetIfD related files
. /lib/functions.sh
. /lib/functions/network.sh
. /lib/netifd/netifd-proto.sh

#source a function from lan_config.sh file
. /etc/data/lan_config.sh

#Function to check and return ippt_enable parameter from qcmap_lan database
#$1 - Current profile ID
function check_ippt_enable() {
	local profile="$1"
	local ippt present_bh default_pdn isWWANAPrefered

	#Get profile index
	local profile_idx=$(util_get_profile_index $profile)

	#Get default_pdn
	default_pdn=$(util_get_default_pdn)

	if [ $profile_idx -eq -1 ]; then
		log $(basename "$0") "check_ippt_enable" $LINENO "profile index error. exiting!"
		return
	fi

	if [ $default_pdn -ne $profile ]; then
		isWWANAPrefered=1
		log $(basename "$0") "check_ippt_enable" $LINENO " it is on-demand PDP."
	else
		#Check if ippt can be enabled by present backhaul type
		isWWANAPrefered=$(/etc/data/backhaulCommonConfig.sh is_wwan_prefer_backhaul)
		log $(basename "$0") "check_ippt_enable" $LINENO "isWWANAPrefered=$isWWANAPrefered"
	fi

	if [ "$isWWANAPrefered" ==  "0" ]; then
		#wwan is not most prefered active backhaul
		ippt=0
		log $(basename "$0") "check_ippt_enable" $LINENO " wwan is not most prefered active backhaul."
	else
		#Check if ippt_enable is present and if set to 1
		local ippt_enable=$(uci get qcmap_lan.@profile[$profile_idx].ippt_enable)
		if [ -z "$ippt_enable" ] || [ -n "$ippt_enable" -a $ippt_enable -eq 0 ]; then
			#indicates there is no ippt related information.
			#we go ahead with regular BH bringup
			ippt=0
		# Add regex check to validate if wan is with profile 1 or with other than 1.
		elif [ -n "$ippt_enable" ] && [ $ippt_enable -eq 1 ] ; then
			#indicates there is ippt related information
			#we go ahead with IPPT BH bringup
			ippt=1
			log $(basename "$0") "check_ippt_enable" $LINENO " wwan is most prefered active backhaul to be enable IPPT."
		fi
	fi

	echo $ippt
}

#Function to check active ippt
#$1 - Profile ID
function check_active_ippt() {
	local profile="$1"
	local active_ippt profile_idx

	#Get profile index
	profile_idx=$(util_get_profile_index $profile)

	#Get active IPPT value
	active_ippt=$(uci get qcmap_lan.@profile[$profile_idx].active_ippt)
	[ -n "$active_ippt" ] || active_ippt=0

	echo $active_ippt
}

#Function to check and return ippt bridge context
#$1 - Current profile ID
function get_ippt_bridge_context() {
	local profile="$1"
	local bridge_context ippt_bridge_context profile_idx ippt active_ippt

	#Get profile index
	profile_idx=$(util_get_profile_index $profile)

	if [ $profile_idx -eq -1 ]; then
		log $(basename "$0") "get_ippt_bridge_context" $LINENO "profile index error. exiting!"
		return
	fi

	#Check if ippt mode is set to enable
	ippt=$(check_ippt_enable $profile)

	#Get active IPPT parameter(Required during call teardown)
	active_ippt=$(check_active_ippt $profile)

	if [ -n "$ippt" -a $ippt -eq 1 ] || [ -n "$active_ippt" -a $active_ippt -eq 1 ]; then
		#Need to check for bridge-vlan context and match it against downstream list of
		#corresponding profile. If no match found and ippt is enabled, return
		bridge_context=$(uci get qcmap_lan.@profile[$profile_idx].ippt_bridge_context)
		if [ -n "$bridge_context" ] && [ $bridge_context -eq 0 ]; then
			ippt_bridge_context=lan
		elif [ -n "$bridge_context" ] && [ $bridge_context -gt 0 ]; then
			ippt_bridge_context=lan$bridge_context
		else
			#Reason for return: bridge_context is a mandatory parameter in qcmap_lan database
			log $(basename "$0") "get_ippt_bridge_context" $LINENO "Error retrieving valid bridge context!"
			return
		fi
	fi

	echo $ippt_bridge_context
}

#Function to check if ippt bridge in context is enumerated
#$1 - profile ID
#$2 - ippt bridge context
function check_ippt_bridge_found() {
	local profile="$1"
	local ippt_bridge_context="$2"
	local wan downstream default_pdn bridge_found=-1
	local private_ip private_netmask

	#Get downstream values.
	downstream=$(util_get_downstream_value_v4 $profile)
	[ -n "$downstream" ] && {
		for i in $downstream
		do
			#check if bridge_context is present in the downstream list
			if [ -n "$i" ] && [ "$ippt_bridge_context" = "$i" ]; then
				bridge_found=1
				break
			fi
		done
	}

	#Get default pdn
	default_pdn=$(util_get_default_pdn)

	#check, if bridge_found is 1
	if [ $bridge_found -eq 1 ]; then
		private_ip=$(uci_get_state network ${ippt_bridge_context} ipaddr)
		private_netmask=$(uci_get_state network ${ippt_bridge_context} netmask)
		#Check if private_ip and private_netmask is present
		if [ -n "$private_ip" ] && [ -n "$private_netmask" ]; then
			log $(basename "$0") "check_ippt_bridge_found" $LINENO "private_ip-$private_ip private_netmask-$private_netmask"
		else
			#Reason for return: private_ip and private_netmask are mandatory options in network database
			log $(basename "$0") "check_ippt_bridge_found" $LINENO "private_ip or private_netmask values not obtained. Exiting"
			return
		fi
	elif [ $default_pdn = $profile ]; then
		#indicates current PDN context is default PDN. We need to continue with program execution as
		#we need to support embedded traffic for IPPT scenario when there is no downstream present.
		log $(basename "$0") "check_ippt_bridge_found" $LINENO "No downstream found for pdn: $profile which is a default_pdn"
		bridge_found=0
	else
		#Control enters this condition if bridge_found is 0 for on-demand PDN
		log $(basename "$0") "check_ippt_bridge_found" $LINENO "bridge not found while ippt is set. Exiting!"
		return
	fi

	echo $bridge_found
}

# Function to configure DHCP server for First Connected Device
# $1 the network section name
# $2 the address to be distributed
# $3 with_nat config entry
# $4 network subnet
# $5 private_gw of downstream
# $6 DNS server list
function ippt_config_DHCP_Server_For_FCD() {
	local downstream=$1
	local public_ip=$2
	local with_nat=$3
	local subnet=$4
	local private_gw=$5
	local dnsserver=$6
	#Save the current dhcp start and limit values
	local private_dhcpstart=$(uci get dhcp.$downstream.start)
	local private_dhcplimit=$(uci get dhcp.$downstream.limit)
	uci_set_state network ${downstream} start $private_dhcpstart
	uci_set_state network ${downstream} limit $private_dhcplimit

	uci set dhcp.$downstream.start="$public_ip"
	uci set dhcp.$downstream.dhcpv4=server
	if [ $with_nat -eq 2 ]; then
		uci set dhcp.$downstream.interface="$downstream"
		[ -n "$dnsserver" ] && uci add_list dhcp.$downstream.dhcp_option='6,'$dnsserver''
	elif [ $with_nat -eq 1 ]; then
		uci set dhcp.$downstream.interface="${downstream}_bind4"
		#To send private_gw as dns server ip to client
		uci add_list dhcp.$downstream.dhcp_option='6,'$private_gw''
	fi
	uci set dhcp.$downstream.limit='1'
	uci set dhcp.$downstream.netmask="$4"
	#Enable DHCP server on the bridge
	if [ $with_nat -eq 2 ]; then
		if [ "$downstream" = "lan" ]; then
			log $(basename "$0") "ippt_config_DHCP_Server_For_FCD" $LINENO "downstream is default bridge"
		else
			#Set dhcp ignore to true for default bridge
			uci set dhcp.lan.ignore="1"
			uci set dhcp.$downstream.ignore="0"
		fi
	fi
	uci commit
}

# Function to configure DHCP server
# $1 the network section name
# $2 the address to be distributed
# $3 with_nat config entry
# $4 wan section name
# $5 qcmap_lan profile index
# $6 Network subnet
# $7 ip4mtu
# $8 private_gw of downstream
function ippt_config_DHCP_Server() {
	local downstream=$1
	local public_ip=$2
	local with_nat=$3
	local cfg=$4
	local profile_idx=$5
	local subnet=$6
	local ip4mtu=$7
	local private_gw=$8
	local device_type mac_addr host_name ip_addr
	local private_dhcpstart private_dhcplimit
	local host_idx idx found host_instance
	local mac_addr_rndis mac_addr_ecm
	local dnsserver dns_list _dns profile_id adap_ippt_mode
	local dhcp_option_vendorinfo_enable=$(util_execute_uci get qcmap_lan.@dhcp_option[0].dhcp_vendor_info_enable)
	local dhcp_option_tz_enable=$(util_execute_uci get qcmap_lan.@dhcp_option[0].dhcp_tz_enable)

	device_type=$(uci get qcmap_lan.@profile[$profile_idx].ippt_device_type)
	profile_id=$(uci get qcmap_lan.@profile[$profile_idx].profile_id)
	#Quectel: add adaptive IPPT --demon.yang
	adap_ippt_mode=$(uci get qcmap_lan.@profile[$profile_idx].adap_ippt_mode)

	if [ -f /tmp/ipv4config$profile_id ]; then
		dns_list=`cat /tmp/ipv4config$profile_id | grep DNSSERVER`
			_dns=`echo $dns_list | sed 's/"/ /g' | awk '{print $3 "," $4}'`
			#NOTE: the second sed command is required for cases involving a single
			#DNSSERVER information or there are NO DNSSERVER informations.
			#The previous line results in a trailing comma(,)
			#The line below is used to remove the trailing comma
			dnsserver=`echo "$_dns" | sed 's/,$//g'`
	fi

	if [ -n "$device_type" ]; then
		#Check if device type is on First connected mode
		#Quectel: modify for adaptive IPPT --demon.yang
		if [ "$device_type" = "Any" ] && [ "$adap_ippt_mode" == "0" ]; then
			#Pass all information to ippt_config_DHCP_Server_For_FCD()
			ippt_config_DHCP_Server_For_FCD "$downstream" "$public_ip" "$with_nat" "$subnet" "$private_gw" "$dnsserver"
			# Need to remove entire lease file for FCD scenarios
			rm -f /tmp/data/dhcp.leases.$downstream
		else
			#Reason for having this check in the else block:
			#Since we use dynamicdhcp setting for w/o NAT, it expects only static
			#leases under a specific condition which cannot be possible for FCD.
			#So we modify existing dhcp rules for FCD device type
			if [ $with_nat -eq 2 ]; then
				#Ignore all DHCP requests except the ones from known
				#clients configured with static leases
				uci set dhcp.$downstream.dynamicdhcp="0"

				#Save the current dhcp start and limit values
				private_dhcpstart=$(uci get dhcp.$downstream.start)
				private_dhcplimit=$(uci get dhcp.$downstream.limit)
				uci_set_state network ${downstream} start $private_dhcpstart
				uci_set_state network ${downstream} limit $private_dhcplimit

				#1. modify start and limit of downstream parameter
				uci set dhcp.$downstream.start="$public_ip"
				uci set dhcp.$downstream.limit="1"
				uci set dhcp.$downstream.netmask="$subnet"

				# Set DHCP option in case of w/o NAT
				[ -n "$dnsserver" ] && uci add_list dhcp.$downstream.dhcp_option='6,'$dnsserver''

				#2. NOTE: We keep the interface linking to be the downstream itself
				# as we are editing the static uci config file

			elif [ $with_nat -eq 1 ]; then
				#Create a new dhcp configuration for public IP
				uci set dhcp.${downstream}_bind4=dhcp
				uci set dhcp.${downstream}_bind4.leasetime="12h"
				uci set dhcp.${downstream}_bind4.start="$public_ip"
				uci set dhcp.${downstream}_bind4.dhcpv4="server"
				uci set dhcp.${downstream}_bind4.interface="${downstream}_bind4"
				uci set dhcp.${downstream}_bind4.limit="1"
				uci set dhcp.${downstream}_bind4.netmask="$subnet"
				uci set dhcp.${downstream}_bind4.instance="${downstream}_dns"
				uci add_list dhcp.${downstream}_bind4.dhcp_option_force='26,'$ip4mtu''
				uci set dhcp.${downstream}_bind4.ippt_bind4='1'
				#To send private_gw as dns server ip to client
				uci add_list dhcp.${downstream}_bind4.dhcp_option='6,'$private_gw''
				#Handle DHCP 100/101 43/60 oprion
				if [ $dhcp_option_tz_enable -eq 1 ] ; then
					option_100=`uci get qcmap_lan.@dhcp_option[0].dhcp_tz_100`
					option_101=`uci get qcmap_lan.@dhcp_option[0].dhcp_tz_101`
					uci add_list dhcp.${downstream}_bind4.dhcp_option_force=100,$option_100
					uci add_list dhcp.${downstream}_bind4.dhcp_option_force=101,$option_101
				fi
				if [ $dhcp_option_vendorinfo_enable -eq 1 ]; then
					option_vendor=`uci get qcmap_lan.@dhcp_option[0].dhcp_vsi`
					uci add_list dhcp.${downstream}_bind4.dhcp_option_force==$option_vendor
				fi
			fi
			#Check if host entry is already present if yes just update with new values
			host_idx=$(uci show dhcp | grep -i host | grep -i ippt_host | awk -F '[][]' '{print $2}')
			idx="-1"
			found=0
			for host_index in $host_idx
			do
				host_instance=$( uci get dhcp.@host[$host_index].instance)
				if [ "$host_instance" = "${downstream}_dns" ]; then
					idx=$host_index
					found=1
					break
				fi
			done
			#Based on device type, set mac or networkid
			if [ "$device_type" = "USB" ]; then
				#Quectel: add adaptive IPPT for USB --demon.yang
				local mac_addr=$(uci get qcmap_lan.@profile[$profile_idx].ippt_mac_addr)

				host_name=$(uci get qcmap_lan.@profile[$profile_idx].ippt_host_name)
				if [ "$found" -eq 0 ]; then
					uci add dhcp host
				fi
				uci set dhcp.@host[$idx].dns="1"
				uci set dhcp.@host[$idx].ip="$public_ip"
				#Quectel: add adaptive IPPT for USB --demon.yang
				if [ "$adap_ippt_mode" != "0" ]; then
					uci set dhcp.@host[$idx].mac="$mac_addr"
				else
					uci set dhcp.@host[$idx].name="$host_name"
				fi
				uci set dhcp.@host[$idx].instance="${downstream}_dns"
				uci set dhcp.@host[$idx].ippt_host='1'
			elif [ "$device_type" = "ETH" ] || [ "$device_type" = "ETH_NIC2" ] || [ "$device_type" = "WiFi" ]; then
				#add mac_addr to dhcp mac
				#We do not check for default bridge condition for Wifi
				#as the very presence of Wifi IPPT value indicates it is
				#a default bridge. We do not populate values if not default bridge
				local mac_addr=$(uci get qcmap_lan.@profile[$profile_idx].ippt_mac_addr)
				if [ "$found" -eq 0 ]; then
					uci add dhcp host
				fi
				uci set dhcp.@host[$idx].dns="1"
				uci set dhcp.@host[$idx].ip="$public_ip"
				uci set dhcp.@host[$idx].mac="$mac_addr"
				uci set dhcp.@host[$idx].instance="${downstream}_dns"
				uci set dhcp.@host[$idx].ippt_host='1'
			elif [ "$device_type" = "Any" ] && [ "$adap_ippt_mode" != "0" ]; then
				#Quectel: modify for adaptive IPPT --demon.yang
				local mac_addr=$(uci get qcmap_lan.@profile[$profile_idx].ippt_mac_addr)
				if [ "$found" -eq 0 ]; then
					uci add dhcp host
				fi
				uci set dhcp.@host[$idx].dns="1"
				uci set dhcp.@host[$idx].ip="$public_ip"
				uci set dhcp.@host[$idx].mac="$mac_addr"
				uci set dhcp.@host[$idx].instance="${downstream}_dns"
				uci set dhcp.@host[$idx].ippt_host='1'
			else
				log $(basename "$0") "ippt_config_DHCP_Server" $LINENO "Unknown device_type: $device_type found"
				return
			fi

			#Enable DHCP server on the bridge
			#NOTE: For WITHOUT_NAT mode, we would set dhcp ignore option to disable
			#DHCP server. Enabling DHCP server on IPPT bridge
			if [ $with_nat -eq 2 ]; then
				#check if the ippt_bridge_context is default bridge
				if [ "$i" = "lan" ]; then
					log $(basename "$0") "ippt_config_DHCP_Server" $LINENO "downstream is default bridge"
				else
					#if not, set dhcp ignore option to true for default bridge
					uci set dhcp.lan.ignore="1"
					uci set dhcp.$downstream.ignore="0"
				fi
			fi

			uci commit
			#We need to delete dhcp lease file entry for the client in which IPPT is UP.
			if [ $with_nat -eq 2 ]; then
				#For IPPT without NAT, we will delete the whole lease file
				rm -f /tmp/data/dhcp.leases.$downstream
			else
				if [ "$device_type" = "USB" ]; then
					mac_addr_rndis=$(bridge fdb show | grep -i "rndis0 master br-${downstream}" | grep -v "permanent" | awk '{print $1}')
					if [ "$mac_addr_rndis" ];then
						ip_addr=`util_get_ip_from_mac $downstream $mac_addr_rndis`
						dhcp_release br-${downstream} $ip_addr $mac_addr_rndis
					else
						mac_addr_ecm=$(bridge fdb show | grep -i "ecm0 master br-${downstream}" | grep -v "permanent" | awk '{print $1}')
						if [ "$mac_addr_ecm" ];then
							ip_addr=`util_get_ip_from_mac $downstream $mac_addr_ecm`
							dhcp_release br-${downstream} $ip_addr $mac_addr_ecm
						fi
					fi
				elif [ "$device_type" = "ETH" ] || [ "$device_type" = "ETH_NIC2" ] || [ "$device_type" = "WiFi" ]; then
					mac_addr=$(uci get qcmap_lan.@profile[$profile_idx].ippt_mac_addr)
					ip_addr=`util_get_ip_from_mac ${downstream} $mac_addr`
					dhcp_release br-${downstream} $ip_addr $mac_addr
				else
					log $(basename "$0") "ippt_config_DHCP_Server" $LINENO "Unknown device_type: $device_type found"
					return
				fi
			fi
		fi
	else
		log $(basename "$0") "ippt_config_DHCP_Server" $LINENO "device type not found"
		return
	fi
}

# Function to recover DHCP server for First Connected Device type
# $1 the nework section name
function ippt_recover_DHCP_Server_For_FCD() {
	local ippt_bridge_context="$1"
	local with_nat private_gw

	#Get with_nat parameter
	with_nat=$(util_get_with_nat)

	local private_dhcpstart=$(uci_get_state network ${ippt_bridge_context} start)
	local private_dhcplimit=$(uci_get_state network ${ippt_bridge_context} limit)

	uci set dhcp.$ippt_bridge_context.start="$private_dhcpstart"
	uci set dhcp.$ippt_bridge_context.interface="$ippt_bridge_context"
	uci set dhcp.$ippt_bridge_context.limit="$private_dhcplimit"
	uci del dhcp.$ippt_bridge_context.netmask
	uci del_list dhcp.$ippt_bridge_context.dhcp_option
	if [ $with_nat -eq 1 ]; then
		private_gw=$(uci get network.$ippt_bridge_context.ipaddr)
		uci del_list dhcp.$ippt_bridge_context.dhcp_option='6,'$private_gw''
	fi

	if [ $with_nat -eq 2 ]; then
		uci delete dhcp.$ippt_bridge_context.dhcp_option
	fi

	uci_revert_state network ${ippt_bridge_context} start
	uci_revert_state network ${ippt_bridge_context} limit

	uci commit
}

# recover DHCP server
# $1 the nework section name
# $2 wan section name
# $3 with_nat config entry
# $4 qcmap_lan profile index
function ippt_recover_DHCP_Server() {
	local ippt_bridge_context="$1"
	local cfg="$2"
	local with_nat="$3"
	local profile_idx="$4"
	local host_idx host_ip ip_addr
	local private_dhcpstart private_dhcplimit
	local device_type=$(uci get qcmap_lan.@profile[$profile_idx].ippt_device_type)
	if [ -n "$device_type" ]; then
		#Check if device type is on First connected mode
		if [ "$device_type" = "Any" ]; then
			#Pass required information to ippt_recover_DHCP_Server_For_FCD()
			ippt_recover_DHCP_Server_For_FCD "$ippt_bridge_context"
		else
			#Get public ip given to client from network section name
			if [ $with_nat -eq 2 ]; then
				ip_addr=$(uci get dhcp.${ippt_bridge_context}.start)
			elif [ $with_nat -eq 1 ]; then
				ip_addr=$(uci get dhcp.${ippt_bridge_context}_bind4.start)
			fi

			#Reason for having this check in the else block:
			#Since we use dynamicdhcp setting for w/o NAT
			if [ $with_nat -eq 2 ]; then
				#Reset DHCP dynamic request handling
				#Reason for having this under WITHOUT_NAT check:
				#If WITH_NAT is set, we do not even set this value, as the default
				#behavior is WITH_NAT and dynamic dhcp will be enabled.
				#We disable dynamic dhcp only for WITHOUT_NAT and during the teardown process
				#we reset dynamic dhcp to enable state as its default value
				uci set dhcp.$ippt_bridge_context.dynamicdhcp="1"

				#1. Reset the downstream parameters for without_nat mode
				private_dhcpstart=$(uci_get_state network ${ippt_bridge_context} start)
				private_dhcplimit=$(uci_get_state network ${ippt_bridge_context} limit)
				uci set dhcp.$ippt_bridge_context.start="$private_dhcpstart"
				uci set dhcp.$ippt_bridge_context.interface="$ippt_bridge_context"
				uci set dhcp.$ippt_bridge_context.limit="$private_dhcplimit"
				uci del dhcp.$ippt_bridge_context.netmask
				uci delete dhcp.$ippt_bridge_context.dhcp_option

				uci_revert_state network ${ippt_bridge_context} start
				uci_revert_state network ${ippt_bridge_context} limit
			fi

			#Iterate through dhcp hosts and find the matching ip address
			host_idx=$(uci show dhcp | grep -i ippt_host | sort -r | awk -F '[][]' '{print $2}')
			for i in $host_idx
			do
				#get ip
				host_ip=$(uci get dhcp.@host[$i].ip)
				#compare it with public_ip
				if [ "$ip_addr" = "$host_ip" ]; then
					log $(basename "$0") "ippt_recover_DHCP_Server" $LINENO "ip addr match found in dhcp host section"
					uci delete dhcp.@host[$i]
				fi
			done

			#Delete new dhcp configuration created for public_ip
			if [ $with_nat -eq 1 ]; then
				uci delete dhcp.${ippt_bridge_context}_bind4
			fi

			uci commit
		fi
	fi
}

# set ip pass through mode for the lan side
# $1 down side network section name
# $2 down side device name
# $3 IPv4 address to be assigned to lan device
# $4 netmask of the IPv4 address
# $5 the address to be distributed
# $6 with_nat option
# $7 wan section name
# $8 qcmap_lan profile index
# $9 MTU value
rmnet_lan_setup_ippt() {
	local downstream=$1
	local downdevice=$2
	local ip=$3
	local subnet=$4
	local public_ip=$5
	local with_nat=$6
	local cfg=$7
	local profile_idx=$8
	local ip4mtu=$9
	local ipaddr netmask private_ip private_netmask private_gw

	#Check if ippt feature mode is set to with_net/without_nat
	if [ $with_nat -eq 2 ]; then
		#Indicates without_nat mode is enabled
		#1. save the existing uci static ipaddr info to /tmp/state/network
		private_ip=$(uci_get_state network ${downstream} ipaddr)
		uci_set_state network ${downstream} private_gatewayip $private_ip

		#2. save the existing uci static netmask info to /tmp/state/network
		private_netmask=$(uci_get_state network ${downstream} netmask)
		uci_set_state network ${downstream} private_netmask $private_netmask

		#3. reconfigure static uci ipaddr with public_gateway
		uci set network.$downstream.ipaddr="$ip"

		#4. reconfigure static uci netmask with public_gateway
		uci set network.$downstream.netmask="255.255.255.255"

	elif [ $with_nat -eq 1 ]; then
		#1. Create a new downstream entry in network uci
		uci set network.${downstream}_bind4=interface

		#2. Set interface name with downdevice entry
		uci set network.${downstream}_bind4.device="$downdevice"

		#3. Set proto as static
		uci set network.${downstream}_bind4.proto="static"

		#4. Set IP address information
		uci set network.${downstream}_bind4.ipaddr="$ip"

		#5. Set netmask value
		uci set network.${downstream}_bind4.netmask="255.255.255.255"

	else
		log $(basename "$0") "rmnet_lan_setup_ippt"$LINENO":Invalid with_nat value found"
		return
	fi

	uci_set_state network ${downstream} public_gatewayip $ip
	uci_set_state network ${downstream} public_netmask $subnet

	uci commit network

	#Perform network reload
	ubus call network reload

	# Get private_gw need for setting custom dns server
	private_gw=$(uci get network.${downstream}.ipaddr)

	ippt_config_DHCP_Server "$downstream" "$public_ip" "$with_nat" "$cfg" "$profile_idx" "$subnet" "$ip4mtu" "$private_gw"

	# Enable Proxy ARP for downstream device
	echo 1 > /proc/sys/net/ipv4/conf/$downdevice/proxy_arp
}

#Function to inform IPA about IP Passthrough configuration
#$1 - Device type
#$2 - With/Without NAT value
#$3 - default PDN value
#$4 - Profile ID
#$5 - Bridge ID
#$6 - WAN side rmnet_dataX Interface name
function inform_ipa_ippt_setup() {
	local device_type="$1"
	local with_nat="$2"
	local default_pdn="$3"
	local profile="$4"
	local bridge_context="$5"
	local interface="$6"
	local mac_addr

	if [ "$device_type" = "Any" ]; then
		if [ $with_nat -eq 1 ]; then
			if [ $default_pdn = $profile ]; then
				ipa ippass enable dev_type "any" vlan $bridge_context mac 00:00:00:00:00:00 skip_nat 0 default_pdn 1 dev_name $interface
			else
				ipa ippass enable dev_type "any" vlan $bridge_context mac 00:00:00:00:00:00 skip_nat 0 default_pdn 0 dev_name $interface
			fi
		elif [ $with_nat -eq 2 ]; then
			if [ $default_pdn = $profile ]; then
				ipa ippass enable dev_type "any" vlan $bridge_context mac 00:00:00:00:00:00 skip_nat 1 default_pdn 1 dev_name $interface
			else
				ipa ippass enable dev_type "any" vlan $bridge_context mac 00:00:00:00:00:00 skip_nat 1 default_pdn 0 dev_name $interface
			fi
		else
			log $(basename "$0") "inform_ipa" $LINENO "Unknown NAT feature mode found"
		fi
	else
		if [ "$device_type" = "USB" ]; then
			if [ $with_nat -eq 1 ]; then
				if [ $default_pdn = $profile ]; then
					ipa ippass enable dev_type "usb" vlan $bridge_context mac 00:00:00:00:00:00 skip_nat 0 default_pdn 1 dev_name $interface
				else
					ipa ippass enable dev_type "usb" vlan $bridge_context mac 00:00:00:00:00:00 skip_nat 0 default_pdn 0 dev_name $interface
				fi
			elif [ $with_nat -eq 2 ]; then
				if [ $default_pdn = $profile ]; then
					ipa ippass enable dev_type "usb" vlan $bridge_context mac 00:00:00:00:00:00 skip_nat 1 default_pdn 1 dev_name $interface
				else
					ipa ippass enable dev_type "usb" vlan $bridge_context mac 00:00:00:00:00:00 skip_nat 1 default_pdn 0 dev_name $interface
				fi
			else
				log $(basename "$0") "inform_ipa" $LINENO "Unknown NAT feature mode found"
			fi
		elif [ "$device_type" = "ETH" ]; then
			mac_addr=$(uci get qcmap_lan.@profile[$profile_idx].ippt_mac_addr)
			if [ $with_nat -eq 1 ]; then
				if [ $default_pdn = $profile ]; then
					ipa ippass enable dev_type "eth0" vlan $bridge_context mac $mac_addr skip_nat 0 default_pdn 1 dev_name $interface
				else
					ipa ippass enable dev_type "eth0" vlan $bridge_context mac $mac_addr skip_nat 0 default_pdn 0 dev_name $interface
				fi
			elif [ $with_nat -eq 2 ]; then
				if [ $default_pdn = $profile ]; then
					ipa ippass enable dev_type "eth0" vlan $bridge_context mac $mac_addr skip_nat 1 default_pdn 1 dev_name $interface
				else
					ipa ippass enable dev_type "eth0" vlan $bridge_context mac $mac_addr skip_nat 1 default_pdn 0 dev_name $interface
				fi
			else
				log $(basename "$0") "inform_ipa" $LINENO "Unknown NAT feature mode found"
			fi
		elif [ "$device_type" = "ETH_NIC2" ]; then
			mac_addr=$(uci get qcmap_lan.@profile[$profile_idx].ippt_mac_addr)
			if [ $with_nat -eq 1 ]; then
				if [ $default_pdn = $profile ]; then
					ipa ippass enable dev_type "eth1" vlan $bridge_context mac $mac_addr skip_nat 0 default_pdn 1 dev_name $interface
				else
					ipa ippass enable dev_type "eth1" vlan $bridge_context mac $mac_addr skip_nat 0 default_pdn 0 dev_name $interface
				fi
			elif [ $with_nat -eq 2 ]; then
				if [ $default_pdn = $profile ]; then
					ipa ippass enable dev_type "eth1" vlan $bridge_context mac $mac_addr skip_nat 1 default_pdn 1 dev_name $interface
				else
					ipa ippass enable dev_type "eth1" vlan $bridge_context mac $mac_addr skip_nat 1 default_pdn 0 dev_name $interface
				fi
			else
				log $(basename "$0") "inform_ipa" $LINENO "Unknown NAT feature mode found"
			fi
		elif [ "$device_type" = "WiFi" ]; then
			mac_addr=$(uci get qcmap_lan.@profile[$profile_idx].ippt_mac_addr)
			if [ $with_nat -eq 1 ]; then
				if [ $default_pdn = $profile ]; then
					ipa ippass enable dev_type "wifi" vlan $bridge_context mac $mac_addr skip_nat 0 default_pdn 1 dev_name $interface
				else
					ipa ippass enable dev_type "wifi" vlan $bridge_context mac $mac_addr skip_nat 0 default_pdn 0 dev_name $interface
				fi
			elif [ $with_nat -eq 2 ]; then
				if [ $default_pdn = $profile ]; then
					ipa ippass enable dev_type "wifi" vlan $bridge_context mac $mac_addr skip_nat 1 default_pdn 1 dev_name $interface
				else
					ipa ippass enable dev_type "wifi" vlan $bridge_context mac $mac_addr skip_nat 1 default_pdn 0 dev_name $interface
				fi
			else
				log $(basename "$0") "inform_ipa" $LINENO "Unknown NAT feature mode found"
			fi
		else
			log $(basename "$0") "inform_ipa" $LINENO "Unknown device_type: $device_type found"
		fi
	fi
}

#Function to inform IPA about IP Passthrough teardown
#$1 - WAN side section name
#$2 - Profile Index
#$3 - Bridge found parameter
#$4 - Bridge ID
function inform_ipa_ippt_teardown() {
	local cfg="$1"
	local profile_idx="$2"
	local bridge_found="$3"
	local bridge_id="$4"
	local device_type updevice ipa_dev_type

	device_type=$(uci get qcmap_lan.@profile[$profile_idx].ippt_device_type)
	updevice=$(uci_get_state network $cfg ifname)

	if [ -n "$updevice" ] && [ $bridge_found -eq 1 ] && [ -n "$device_type" ]; then
		if [ "$device_type" = "USB" ]; then
			ipa_dev_type="usb"
		elif [ "$device_type" = "ETH" ]; then
			ipa_dev_type="eth0"
		elif [ "$device_type" = "ETH_NIC2" ]; then
			ipa_dev_type="eth1"
		elif [ "$device_type" = "WiFi" ]; then
			ipa_dev_type="wifi"
		elif [ "$device_type" = "Any" ]; then
			ipa_dev_type="any"
		else
			log $(basename "$0") "inform_ipa_ippt_teardown" $LINENO "Unknown device type: $device_type found"
			ipa_dev_type="Unknown"
		fi

		if [ -n "$ipa_dev_type" ] && [ "$ipa_dev_type" != "Unknown" ]; then
			ipa ippass disable dev_type $ipa_dev_type vlan $bridge_id dev_name $updevice
		fi
	else
		log $(basename "$0") "inform_ipa_ippt_teardown" $LINENO "Updevice of $cfg not found"
	fi

}

#Function to mark DHCP packets and add route in custom routing table for IPPT
# $1 down side network section name
# $2 IPPT bridge context
# $3 Gateway address
# $4 Default PDN number
# $5 Current PDN number
# $6 Public IP
# $7 down side device name
ippt_add_dhcp_mark_pkts() {
	local downstream=$1
	local ippt_bridge_context=$2
	local gateway=$3
	local default_pdn=$4
	local profile=$5
	local public_ip=$6
	local downdevice=$7
        local ippt_route_table_id=$((profile + 200))

	#Check if downstream is same as ippt_bridge_context
	if [ "$ippt_bridge_context" = "$downstream" ]; then
		#If yes, indicates IPPT is to be set up on current downstream context
		log $(basename "$0") "ippt_add_dhcp_mark_pkts" $LINENO " Setting up DHCP mark rules on $downstream"
	else
		#if no, return
		log $(basename "$0") "ippt_add_dhcp_mark_pkts" $LINENO " current lan context: $downstream is not the IPPT bridge context: $ippt_bridge_context"
		return
	fi

	#Check if ippastbl already exists
	local is_ippastbl_present=$(grep ippastbl$profile /etc/iproute2/rt_tables)
	if [ -z "$is_ippastbl_present" ]; then
		#create custom routing table
		log $(basename "$0") "ippt_add_dhcp_mark_pkts" $LINENO " Creating custom routing table"
		echo $ippt_route_table_id ippastbl$profile >> /etc/iproute2/rt_tables
	fi

	#Check if ip rule for ippastbl is present
	local is_rule_present=$(ip rule ls | grep ippastbl$profile)
	if [ -z "$is_rule_present" ]; then
		#create a rule
		log $(basename "$0") "ippt_add_dhcp_mark_pkts" $LINENO " Creating rule for ippastbl$profile"
		ip rule add fwmark $ippt_route_table_id table ippastbl$profile
	fi

	#Get the public gateway based route of ippt_bridge
	local gw_route=$(ip route show table local | grep "local $gateway")
	if [ -z "$gw_route" ]; then
		log $(basename "$0") "ippt_add_dhcp_mark_pkts" $LINENO " Gateway route for bridge in main routing table not found"
		return
	fi

	#Mark DHCP packets in PREROUTING chain
	uci add firewall rule
	uci set firewall.@rule[-1].name="IPPT-$downstream-Mangle-PREROUTING"
	uci set firewall.@rule[-1].src="$downstream"
	uci set firewall.@rule[-1].dest="*"
	uci set firewall.@rule[-1].proto="udp"
	uci set firewall.@rule[-1].family="ipv4"
	uci set firewall.@rule[-1].target="MARK"
	uci set firewall.@rule[-1].set_mark="0xc8"
	uci set firewall.@rule[-1].dest_port="67"
	uci set firewall.@rule[-1].enabled="1"

	#Mark DHCP_RELEASE packets in OUTPUT chain
	uci add firewall rule
	uci set firewall.@rule[-1].name="IPPT-$downstream-Mangle-OUTPUT"
	uci set firewall.@rule[-1].dest="$downstream"
	uci set firewall.@rule[-1].proto="udp"
	uci set firewall.@rule[-1].family="ipv4"
	uci set firewall.@rule[-1].target="MARK"
	uci set firewall.@rule[-1].set_mark="0xc8"
	uci set firewall.@rule[-1].dest_port="67"
	uci set firewall.@rule[-1].enabled="1"

	uci commit firewall

	#Increase the rule count by 2
	local idx=$(uci get qcmap_lan.@no_of_configs[0].no_of_rules)
	idx=$((idx+2))
	uci set qcmap_lan.@no_of_configs[0].no_of_rules=$idx
	uci commit qcmap_lan

	#Delete gateway route from main routing table
	ip route del $gw_route table local

	#Add gateway route to ippastbl table
	ip route add $gw_route table ippastbl$profile

	#Check if default_pdn is same as current PDN context
	if [ $default_pdn -eq $profile ]; then
		#Add a public IP route in main routing table
		ip route add $public_ip dev $downdevice
			fi

}

#Delete markings for DHCP packets
# $1: default pdn
# $2: current PDN context
# $3: Public IP assigned to client
# $4: down device name
# $5: Bridge public gateway (hint: pass downip from state/network)
# $6: down side network section name
function ippt_del_dhcp_mark_pkts() {
	local default_pdn=$1
	local profile=$2
	local public_ip=$3
	local downdevice=$4
	local bridge_gw=$5
	local downstream=$6
	local is_ippastbl_present is_rule_present gw_route idx index

	#Check if default_pdn is same as current PDN context
	if [ $default_pdn -eq $profile ]; then
		#delete the public_ip route from main routing table
		ip route del $public_ip dev $downdevice
	fi

	#if not, continue, as the route will be deleted when custom routing table is deleted

	#Check if ippastbl already exists
	is_ippastbl_present=$(grep ippastbl$profile /etc/iproute2/rt_tables)
	if [ -z "$is_ippastbl_present" ]; then
		log $(basename "$0") "ippt_del_dhcp_mark_pkts" $LINENO " table:ippastbl$profile not present. Exiting"
		return
	fi

	#Check if ip rule for ippastbl is present
	is_rule_present=$(ip rule ls | grep ippastbl$profile)
	if [ -z "$is_rule_present" ]; then
		log $(basename "$0") "ippt_del_dhcp_mark_pkts" $LINENO " ippastbl$profile rule not present. Exiting"
		return
	fi

	#check if public_gw route is present in ippastbl
	gw_route=$(ip route show table ippastbl$profile | grep "local $bridge_gw")
	if [ -n "$gw_route" ]; then
		#delete the route
		ip route del $gw_route table ippastbl$profile
	fi

	#Delete firewall PREROUTING and OUTPUT entries

	#Check if PREROUTING rule exists
	idx=$(uci get qcmap_lan.@no_of_configs[0].no_of_rules)
	index=$(uci show firewall | grep -i """IPPT-$downstream-Mangle-PREROUTING""" | sort -r | awk -F'[][]' '{print $2}')
	if [ -n "$index" ]; then
		for rule_idx in $index
		do
		#delete firewall rule
			uci delete firewall.@rule[$rule_idx]
			uci commit firewall
			idx=$((idx-1))
			uci set qcmap_lan.@no_of_configs[0].no_of_rules=$idx
			uci commit qcmap_lan
		done
	fi
	idx=$(uci get qcmap_lan.@no_of_configs[0].no_of_rules)
	#Check if OUTPUT rule exists
	index=$(uci show firewall | grep -i """IPPT-$downstream-Mangle-OUTPUT""" | sort -r |awk -F'[][]' '{print $2}')
	if [ -n "$index" ]; then
		for rule_idx in $index
		do
			#delete firewall rule
			uci delete firewall.@rule[$rule_idx]
			uci commit firewall
			idx=$((idx-1))
			uci set qcmap_lan.@no_of_configs[0].no_of_rules=$idx
			uci commit qcmap_lan
		done

	fi
}

#Function to recover DHCP server during cleanup of IPPT configuration
#$1 - WAN side section name
#$2 - IPPT bridge context
#$3 - WITH/WITHOUT_NAT configuration
#$4 - Profile Index
function ippt_recover_dhcp() {
	local cfg="$1"
	local ippt_bridge_context="$2"
	local with_nat="$3"
	local profile_idx="$4"
	local downdevice default_pdn device_type public_ip
	local downip private_netmask private_gatewayip ippt_enable

	#Get default PDN
	default_pdn=$(util_get_default_pdn)

	#Condition to check if bridge is enumerated and default_pdn is same as profile context
	network_get_device downdevice $ippt_bridge_context
	[ -n "$downdevice" ] || downdevice=$(uci_get_state network ${ippt_bridge_context} ifname)
	if [ -z "$downdevice" ] && [ $default_pdn -eq $profile ]; then
		log $(basename "$0") "ippt_recover_dhcp" $LINENO " Current profile: $profile is the default profile while bind is set to $bind"
	elif [ -n "$downdevice" ]; then
		device_type=$(uci get qcmap_lan.@profile[$profile_idx].ippt_device_type)
		if [ $device_type = "Any" ]; then
			public_ip=$(uci get dhcp.${ippt_bridge_context}.start)
		else
			if [ $with_nat -eq 1 ]; then
				public_ip=$(uci get dhcp.${ippt_bridge_context}_bind4.start)
			elif [ $with_nat -eq 2 ]; then
				public_ip=$(uci get dhcp.${ippt_bridge_context}.start)
			fi
		fi

		#Recover DHCP Server on matching bridge_context
		ippt_recover_DHCP_Server "$ippt_bridge_context" "$cfg" "$with_nat" "$profile_idx"

		downip=$(uci_get_state network ${ippt_bridge_context} public_gatewayip)
		#Delete DHCP Mark rules
		ippt_del_dhcp_mark_pkts $default_pdn $profile $public_ip $downdevice $downip $ippt_bridge_context

		if [ $with_nat -eq 1 ]; then
			ifdown ${ippt_bridge_context}_bind4

			#Delete the new *_bind4 entry created for WITH_NAT scenario
			uci delete network.${ippt_bridge_context}_bind4
			uci commit network
		elif [ $with_nat -eq 2 ]; then
			private_netmask=$(uci_get_state network ${ippt_bridge_context} private_netmask)
			private_gatewayip=$(uci_get_state network ${ippt_bridge_context} private_gatewayip)

			if [ -n "$private_netmask" ] && [ -n "$private_gatewayip" ]; then
				uci set network.${ippt_bridge_context}.ipaddr="$private_gatewayip"
				uci set network.${ippt_bridge_context}.netmask="$private_netmask"
				uci commit network
			fi
		fi
	fi

	#Check if ippt_bridge_context is default bridge
	if [ $ippt_bridge_context = "lan" ]; then
		#We are not resetting ignore option of default bridge since default bridge
		#is the ippt bridge context
		log $(basename "$0") "ippt_recover_dhcp" $LINENO "downstream is default bridge"
	else
		#Set dhcp ignore option of default bridge to false since ippt_bridge_context
		#is not default bridge
		uci set dhcp.lan.ignore="0"
	fi

	uci commit qcmap_lan

}

#Function to disable DMZ when IPPT is active
#$1 - device_type
function ippt_disable_dmz() {
	local redirect_rule_idxs redirect_name with_nat device_type
	device_type=$1

	#Get with_nat value
	with_nat=$(util_get_with_nat)

	if [ $with_nat -eq 1 ]; then
		if [ "$device_type" = "USB" ]; then
			mac_addr_rndis=$(bridge fdb show | grep -i "rndis0 master br-$1" | grep -v "permanent" | awk '{print $1}')
			if [ "$mac_addr_rndis" ];then
				ippt_ip=`util_get_ip_from_mac $ippt_bridge_context $mac_addr_rndis`
			else
				mac_addr_ecm=$(bridge fdb show | grep -i "ecm0 master br-$1" | grep -v "permanent" | awk '{print $1}')
				if [ "$mac_addr_ecm" ];then
					ippt_ip=`util_get_ip_from_mac $ippt_bridge_context $mac_addr_ecm`
				fi
			fi
		elif [ "$device_type" = "ETH" ] || [ "$device_type" = "ETH_NIC2" ] || [ "$device_type" = "WiFi" ]; then
			local mac_addr=$(uci get qcmap_lan.@profile[$profile_idx].ippt_mac_addr)
			ippt_ip=`util_get_ip_from_mac $ippt_bridge_context $mac_addr`
		else
			echo $(basename "$0") "setup_interface" $LINENO "Unknown device_type: $device_type found"
		fi
	fi

	redirect_rule_idxs=$(uci show firewall | grep -i "DNAT" | awk -F '[][]' '{print $2}')
	for redirect_index in $redirect_rule_idxs
	do
		redirect_name=$(uci get firewall.@redirect[$redirect_index].name)
		redirect_dest_ip=$(uci get firewall.@redirect[$redirect_index].dest_ip)
		#For DMZ redirect_name is profile_num
		if [ "$redirect_name" = "$profile" -o "$redirect_name" = "StaticNAT-$profile" ]; then
			if [ $with_nat -eq 1 ]; then
				if [ "$redirect_dest_ip" = "$ippt_ip" ]; then
					uci set firewall.@redirect[$redirect_index].enabled='0'
				fi
			else
				uci set firewall.@redirect[$redirect_index].enabled='0'
			fi
		fi
	done
	uci commit

}

#Function to enable DMZ when IPPT is disabled
function ippt_enable_dmz() {
	local redirect_rule_idxs redirect_name

	redirect_rule_idxs=$(uci show firewall | grep -i "DNAT" | awk -F '[][]' '{print $2}')
	for redirect_index in $redirect_rule_idxs
	do
		redirect_name=$(uci get firewall.@redirect[$redirect_index].name)
		if [ "$redirect_name" = "$profile" -o "$redirect_name" = "StaticNAT-$profile" ]; then
			uci set firewall.@redirect[$redirect_index].enabled='1'
		fi
	done

}

#Function to reset IPPT paramters on bootup
function reset_ippt_lan_params() {

	local pdn_count is_network_reset_done dhcp_reset network_reset
	local no_of_profiles is_ippt_active phy_iface ippt_bridge_context
	local dhcp_start dhcp_limit nat_type ip netmask vlan_idx host_idx
	local ippt_bind4 dhcp_option_list dhcp_option option with_nat vlan_index

        #Get With/Without-NAT parameter
        with_nat=$(util_get_with_nat)

	log $(basename "$0") "reset_ippt_lan_params" $LINENO "Enter reset_ippt_lan_params"
	#If ippt_pdn_count is greater than zero, make it zero
	pdn_count=$(uci get qcmap_lan.@no_of_configs[0].ippt_pdn_count)
	if [ -n "$pdn_count" -a $pdn_count -gt 0 ]; then
		uci set qcmap_lan.@no_of_configs[0].ippt_pdn_count=0
	fi

	#For each profile check if IPPT options are available and reset accordingly
	no_of_profiles=$(uci get qcmap_lan.@no_of_configs[0].no_of_profiles)
	log $(basename "$0") "reset_ippt_lan_params" $LINENO "no_of_profiles=$no_of_profiles"
	for profile_idx in $(seq 0 $((no_of_profiles-1)))
	do
		#Check if IPPT is enabled
		is_ippt_active=$(uci get qcmap_lan.@profile[$profile_idx].active_ippt)
		if [ -n "$is_ippt_active" ]; then

			#If ippt_active param is one set it to zero
			if [ $is_ippt_active -eq 1 ]; then
				uci set qcmap_lan.@profile[$profile_idx].active_ippt='0'
			fi

			#If start and limit values are missing, update start and limit to default values
			ippt_bridge_context=$(uci get qcmap_lan.@profile[$profile_idx].ippt_bridge_context)
			log $(basename "$0") "reset_ippt_lan_params" $LINENO "ippt_bridge_context=$ippt_bridge_context:profile_idx=$profile_idx"
			if [ -n "$ippt_bridge_context" ]; then
				if [ $ippt_bridge_context -eq 0 ]; then
					dhcp_start=$(uci get dhcp.lan.start)
					dhcp_limit=$(uci get dhcp.lan.limit)
					if [ -z "$dhcp_start" ] || [ -z "$dhcp_limit" ] || [ "$dhcp_start" -gt 254 ] || [ "$dhcp_limit" -eq 1 ] ; then
						uci set dhcp.lan.start='100'
						uci set dhcp.lan.limit='280'
					fi
					#Delete netmask from dhcp lan section
					uci delete dhcp.lan.netmask
					dhcp_option_list=$(uci get dhcp.lan.dhcp_option)
					for dhcp_option in $dhcp_option_list
					do
						option=$(echo $dhcp_option | awk -F ',' '{print $1}')
						if [ "$option" -eq 6 ]; then
							uci del_list dhcp.lan.dhcp_option=$dhcp_option
						fi
					done
					dhcp_reset=1
				else
					dhcp_start=$(uci get dhcp.lan${ippt_bridge_context}.start)
					dhcp_limit=$(uci get dhcp.lan${ippt_bridge_context}.limit)
					if [ -z "$dhcp_start" ] || [ -z "$dhcp_limit" ] || [ "$dhcp_start" -gt 254 ] || [ "$dhcp_limit" -eq 1 ] ; then
						uci set dhcp.lan${ippt_bridge_context}.start='100'
						uci set dhcp.lan${ippt_bridge_context}.limit='280'
					fi
					#Delete netmask from dhcp lan section
					uci delete dhcp.lan${ippt_bridge_context}.netmask
					dhcp_option_list=$(uci get dhcp.lan${ippt_bridge_context}.dhcp_option)
					for dhcp_option in $dhcp_option_list
					do
						option=$(echo $dhcp_option | awk -F ',' '{print $1}')
						if [ "$option" -eq 6 ]; then
							uci del_list dhcp.lan${ippt_bridge_context}.dhcp_option=$dhcp_option
						fi
					done
					dhcp_reset=1
				fi
			fi

			#In without_nat gw_ip and netmask are modified. Need to update it to default
			nat_type=$(util_get_with_nat)
			log $(basename "$0") "reset_ippt_lan_params" $LINENO "ippt_bridge_context=$ippt_bridge_context:nat_type=$nat_type"
			if [ -n "$nat_type" -a $nat_type -eq 2 ]; then
				if [ -n "$ippt_bridge_context" -a $ippt_bridge_context -eq 0 ]; then
					#Get saved IP address and Netmask entries from qcmap_lan database
					ip=$(uci get qcmap_lan.@lan[0].ip)
					netmask=$(uci get qcmap_lan.@lan[0].netmask)
					uci set network.lan.ipaddr="$ip"
					uci set network.lan.netmask="$netmask"
					log $(basename "$0") "reset_ippt_lan_params" $LINENO "ippt_bridge_context=$ippt_bridge_context:ip=$ip:netmask=$netmask"
					#Reset dynamic dhcp in without-nat scenario
					uci set dhcp.lan.dynamicdhcp="1"
					dhcp_reset=1
					network_reset=1
				elif [ -n "$ippt_bridge_context" ]; then
					vlan_idx=$(uci show qcmap_lan | grep -i "$ippt_bridge_context" |
										 grep -iw "vlan_id" | awk -F '[][]' '{print $2}' |  tr '\n' '\t')
					log $(basename "$0") "reset_ippt_lan_params" $LINENO "ippt_bridge_context=$ippt_bridge_context:nat_type=$nat_type:vlan_idx=$vlan_idx"
					for vlan_index in $vlan_idx
					do
						vlan_id=$(uci get qcmap_lan.@vlan[$vlan_index].vlan_id)
						if [ "$vlan_id" = "$ippt_bridge_context" ]; then
							ip=$(uci get qcmap_lan.@vlan[$vlan_index].ip)
							netmask=$(uci get qcmap_lan.@vlan[$vlan_index].netmask)
							log $(basename "$0") "reset_ippt_lan_params" $LINENO "ippt_bridge_context=$ippt_bridge_context:vlan_index=$vlan_index:ip=$ip:netmask=$netmask"
							uci set network.lan${ippt_bridge_context}.ipaddr="$ip"
							uci set network.lan${ippt_bridge_context}.netmask="$netmask"
							network_reset=1
							break
						fi
					done
					#reset ignore option for lan
					uci set dhcp.lan.ignore="0"
					#Reset dynamic dhcp in without-nat scenario
					uci set dhcp.lan${ippt_bridge_context}.dynamicdhcp="1"
					dhcp_reset=1
				fi
			fi
		fi
	done

	#If bind4 interfaces are present remove those interfaces
	ippt_bind4=$(uci show dhcp | grep -i ippt_bind4 | awk -F . '{print $2}')
	for interface in $ippt_bind4
	do
		uci delete dhcp.$interface
		dhcp_reset=1
	done
	#If ippt host reservation record is present remove it
	host_idx=$(uci show dhcp | grep -i ippt_host | sort -r | awk -F '[][]' '{print $2}')
	for idx in $host_idx
	do
		uci delete dhcp.@host[$idx]
		dhcp_reset=1
	done

	#Final commit
	uci commit

	#Perform network reload
	if [ $network_reset -eq 1 ]; then
		ubus call network reload
	fi

	#Perform dnsmasq reload
        if [ $dhcp_reset -eq 1 ]; then
          if [ $with_nat -eq 2 ]; then
            /etc/init.d/dnsmasq reload
          else
            /etc/init.d/dnsmasq restart "${ippt_bridge_context}_dns"
          fi
        fi

}

#Function to set enable_ippt parameter to MODE_UP/MODE_DOWN
# $1 - index of profile section in qcmap_lan db
# $2 - enable_state
function set_ippt_state() {
	local profile_idx=$1
	local enable_state=$2

	#Set ippt_enable parameter in qcmap_lan db to enable_state value
	uci set qcmap_lan.@profile[$profile_idx].ippt_enable="$enable_state"
	#Quectel: add adaptive IPPT feature. --demon.yang
	if [ "$enable_state" = "0" ]; then
		uci set qcmap_lan.@profile[$profile_idx].adap_ippt_mode="0"
	fi
	uci commit

}

#Function to set ippt config parameters
# $1 - index of profile section in qcmap_lan db
# $2 - device type
# $3 - Current bridge context
# $4 - Host_name/mac_addr
function set_ippt_config() {
	local profile_idx=$1
	local dev_type=$2
	local bridge_context=$3
	local dev_info=$4
	local host_name mac_addr
	#Quectel: add adaptive IPPT feature. --demon.yang
	local ql_ippt_mode

	#Quectel: add adaptive IPPT feature. --demon.yang -Start
	ql_ippt_mode="0"
	if [ "$dev_info" = "FF:FF:FF:FF:FF:FF" -o "$dev_info" = "ff:ff:ff:ff:ff:ff" ]; then
		ql_ippt_mode="1"
	elif [ "$dev_info" = "00:00:00:00:00:00" ]; then
		ql_ippt_mode="2"
	fi

	uci set qcmap_lan.@profile[$profile_idx].adap_ippt_mode="$ql_ippt_mode"
	log $(basename "$0") "ippt_update_atBHSwitch" $LINENO "[Demon]dev info: $dev_info; adap ippt mode: $ql_ippt_mode"
	#Quectel: add adaptive IPPT feature. --demon.yang -End

	if [ "$dev_type" = "USB" ]; then
		#Quectel: add adaptive IPPT feature. --demon.yang -Start
		if [ "$ql_ippt_mode" != "0" ]; then
			#Add new entries
			uci set qcmap_lan.@profile[$profile_idx].ippt_device_type="$dev_type"
			uci set qcmap_lan.@profile[$profile_idx].ippt_mac_addr="$dev_info"
		else
			#get saved mac_addr
			mac_addr=$(uci get qcmap_lan.@profile[$profile_idx].ippt_mac_addr)
			if [ -n "$mac_addr" ]; then
				#delete mac address entry
				uci delete qcmap_lan.@profile[$profile_idx].ippt_mac_addr
			fi
			#Add new entries
			uci set qcmap_lan.@profile[$profile_idx].ippt_device_type="$dev_type"
			uci set qcmap_lan.@profile[$profile_idx].ippt_host_name="$dev_info"
		fi
	elif [ "$dev_type" = "ETH" ]; then
		#get saved host_name
		host_name=$(uci get qcmap_lan.@profile[$profile_idx].ippt_host_name)
		if [ -n "$host_name" ]; then
			#delete this entry
			uci delete qcmap_lan.@profile[$profile_idx].ippt_host_name
		fi
		#Add new entries
		uci set qcmap_lan.@profile[$profile_idx].ippt_device_type="$dev_type"
		uci set qcmap_lan.@profile[$profile_idx].ippt_mac_addr="$dev_info"
	elif [ "$dev_type" = "ETH_NIC2" ]; then
		#get saved host_name
		host_name=$(uci get qcmap_lan.@profile[$profile_idx].ippt_host_name)
		if [ -n "$host_name" ]; then
			#delete this entry
			uci delete qcmap_lan.@profile[$profile_idx].ippt_host_name
		fi
		#Add new entries
		uci set qcmap_lan.@profile[$profile_idx].ippt_device_type="$dev_type"
		uci set qcmap_lan.@profile[$profile_idx].ippt_mac_addr="$dev_info"
	elif [ "$dev_type" = "WiFi" ]; then
		#get saved host_name
		host_name=$(uci get qcmap_lan.@profile[$profile_idx].ippt_host_name)
		if [ -n "$host_name" ]; then
			#delete this entry
			uci delete qcmap_lan.@profile[$profile_idx].ippt_host_name
		fi
		#Add new entries
		uci set qcmap_lan.@profile[$profile_idx].ippt_device_type="$dev_type"
		uci set qcmap_lan.@profile[$profile_idx].ippt_mac_addr="$dev_info"
	elif [ "$dev_type" = "Any" ]; then
		#Quectel: add adaptive IPPT feature. --demon.yang -Start
		if [ "$ql_ippt_mode" != "0" ]; then
			uci set qcmap_lan.@profile[$profile_idx].ippt_mac_addr="$dev_info"
		else
			#get saved host_name
			host_name=$(uci get qcmap_lan.@profile[$profile_idx].ippt_host_name)
			#get saved mac_addr
			mac_addr=$(uci get qcmap_lan.@profile[$profile_idx].ippt_mac_addr)
			if [ -n "$host_name" ]; then
				#delete this entry
				uci delete qcmap_lan.@profile[$profile_idx].ippt_host_name
			fi
			if [ -n "$mac_addr" ]; then
				#delete mac address entry
				uci delete qcmap_lan.@profile[$profile_idx].ippt_mac_addr
			fi
		fi
		#Add new entries
		uci set qcmap_lan.@profile[$profile_idx].ippt_device_type="$dev_type"
	else
		echo $(basename "$0") "set_ippt_config" $LINENO "dev type-$dev_type unknown"
		exit 1
	fi
	#Set ippt bridge context
	uci set qcmap_lan.@profile[$profile_idx].ippt_bridge_context="$bridge_context"
	#Set active_ippt to default value
	uci set qcmap_lan.@profile[$profile_idx].active_ippt='0'
	uci commit

}

#Function to up/down rmnet interface for IPPT due to backhaul switch
function ippt_update_atBHSwitch() {
	local defaultProfileId profile_idx
	local ippt_enable=0 present_bh

	log $(basename "$0") "ippt_update_atBHSwitch" $LINENO "Enter ippt_update_atBHSwitch"

	#Get default PDN value
	defaultProfileId=$(util_get_default_pdn)

	#Get Profile Index value of default Profile
	profile_idx=$(util_get_profile_index $defaultProfileId)

	#Get IPPT phy interface
	ippt_enable=$(uci get qcmap_lan.@profile[$profile_idx].ippt_enable)

	#Get present Backhaul value
	present_bh=$(uci get qcmap_lan.@profile[$profile_idx].bh_present)

	if [ "$ippt_enable" -eq 1 -a -f /tmp/ipv4config$defaultProfileId ]; then

		log $(basename "$0") "ippt_update_atBHSwitch" $LINENO "Update IPPT"
		/lib/netifd/rmnet_update.sh up $defaultProfileId 1

		# set WWAN address and netmask
		if [ -n "$present_bh" -a $present_bh != "wan" ]; then
			file=/tmp/ipv4config$defaultProfileId
			DEVNAME=`awk -F "=" 'FNR == 1 {print $2}' $file | tr -d '"'`
			PUBLIC_IP_ADDR=`awk -F "=" 'FNR == 2 {print $2}' $file | tr -d '"'`
			NETMASK=`awk -F "=" 'FNR == 3 {print $2}' $file | tr -d '"'`
			ifconfig $DEVNAME $PUBLIC_IP_ADDR netmask $NETMASK
		fi
	fi
}

#Function to reset IPPT firewall parameters on bootup
function reset_ippt_firewall_params(){
	local idx index
	local is_firewall_reset_done=0

	#Delete firewall PREROUTING and OUTPUT entries
	#Check if PREROUTING rule exists
	local idx=$(uci get qcmap_lan.@no_of_configs[0].no_of_rules)
	local index=$(uci show firewall | grep -i "IPPT" | sort -r | awk -F'[][]' '{print $2}')
	if [ -n "$index" ]; then
		#delete firewall rule
		is_firewall_reset_done=1
		for rule_idx in $index
		do
			uci delete firewall.@rule[$rule_idx]
			idx=$((idx-1))
			uci set qcmap_lan.@no_of_configs[0].no_of_rules=$idx
		done
	fi
	uci commit
	if [ "$is_firewall_reset_done" -eq 1 ]; then
		/etc/init.d/firewall reload
	fi
}

#Function to increase IPPT PDN count by 1
function ippt_increase_pdn_count() {
	local ippt_count

	ippt_count=$(uci get qcmap_lan.@no_of_configs[0].ippt_pdn_count)
	ippt_count=$((ippt_count+1))
	uci set qcmap_lan.@no_of_configs[0].ippt_pdn_count=$ippt_count
	uci commit qcmap_lan
}

#Function to deccrease IPPT PDN count by 1
function ippt_decrease_pdn_count() {
	local ippt_count

	ippt_count=$(uci get qcmap_lan.@no_of_configs[0].ippt_pdn_count)
	ippt_count=$((ippt_count-1))
	uci set qcmap_lan.@no_of_configs[0].ippt_pdn_count=$ippt_count
	uci commit qcmap_lan
}

#Function to perform link toggle
#$1 - profile Index
#$2 - Device
#$3 - ippt_bridge_context
function ippt_link_toggle() {
	local profile_idx="$1"
	local device_type="$2"
	local ippt_bridge_context="$3"
	local mac_addr downdevice with_nat

	network_get_device downdevice "$ippt_bridge_context"

	#Updating downdevice as in IPPT_WITHOUT_NAT scenario we need to toggle all links
	#Get with_nat parameter
	with_nat=$(util_get_with_nat)
	if [ "$with_nat" = "$IP_PASSTHROUGH_MODE_WITHOUT_NAT" ]; then
		downdevice=""
	fi

	if [ "$device_type" = "USB" ]; then
		util_perform_link_toggle "USB" $downdevice
	elif [ "$device_type" = "ETH" ]; then
		util_perform_link_toggle "ETH" $downdevice
	elif [ "$device_type" = "ETH_NIC2" ]; then
		util_perform_link_toggle "ETH_NIC2" $downdevice
	elif [ "$device_type" = "WiFi" ]; then
		mac_addr=$(uci get qcmap_lan.@profile[$profile_idx].ippt_mac_addr)
		util_perform_link_toggle "WiFi" $downdevice $mac_addr
	elif [ "$device_type" = "Any" ]; then
		util_perform_link_toggle "Any" $downdevice
	fi
}

#Function to enable DHCP on briges mapped to pdn
#$1 - Flag indicating to start/stop
#$2 - profile
#$3 - ippt_bridge_context(optional parameter)
function StartStopDHCPDOnNonIPPTVLANS () {
	local start_dhcp=$1
	local profile=$2
	local ippt_bridge_context=$3
	local downstream_list perform_link_toggle="0"

	#Get all mapped downstream to pdn
	downstream_list=$(util_get_downstream_value_v4 $profile)

	log $(basename "$0") "StartStopDHCPDOnNonIPPTVLANS" $LINENO "Enter StartStopDHCPDOnNonIPPTVLANS"

	#loop through downstream_list and enable/disable DHCP accordingly
	if [ -n "$downstream_list" ]; then
		for downstream in $downstream_list
		do
			if [ "$downstream" != "lan" -a "$ippt_bridge_context" != "$downstream" ]; then
				perform_link_toggle="1"
				if [ "$start_dhcp" -eq 1 ]; then
					util_enableDHCP "$downstream"
				else
					util_disableDHCP "$downstream"
				fi
			fi
		done
	fi
	uci commit

	log $(basename "$0") "StartStopDHCPDOnNonIPPTVLANS" $LINENO "perform_link_toggle=$perform_link_toggle"

	echo "$perform_link_toggle"

}

#Function to enable DHCP on briges mapped to pdn
#$1 - Flag indicating to start/stop
#$2 - profile
#$3 - cfg
#$4 - setup(1)/teardown(0)
function PerformStartStopDHCPDOnNonIPPTVLANS () {
	local start_dhcp=$1
	local profile=$2
	local cfg=$3
	local setup_ipv4=$4
	local downstream_list perform_toggle="0" bh_status with_nat
	local ipv4_nat_status="0" is_ippt_teardowned profile_idx

	log $(basename "$0") "PerformStartStopDHCPDOnNonIPPTVLANS" $LINENO "start_dhcp=$start_dhcp profile=$profile cfg=$cfg setup_ipv4=$setup_ipv4"

	#Get with_nat parameter
	with_nat=$(util_get_with_nat)

	#Get IPV4NATConfig status
	ipv4_nat_status=$(util_get_ipv4_nat_status $profile)

	#Fecth if cfg is up. This provides if call is valid call from netifd
	bh_status=$(uci_get_state network.$cfg.up)
	log $(basename "$0") "PerformStartStopDHCPDOnNonIPPTVLANS" $LINENO "cfg=$cfg:profile=$profile:bh_status=$bh_status"

	#Get profile index
	profile_idx=$(util_get_profile_index $profile)

	if [ "$setup_ipv4" -eq 1 ]; then
		#If IPPT is not configured and WITHOUT_NAT Feature mode is set
		#Enable dhcp on all mapped vlans and toggle links.
		if [ "$with_nat" = "$IP_PASSTHROUGH_MODE_WITHOUT_NAT" -a "$ipv4_nat_status" -eq 0 ]; then
			perform_toggle=$(StartStopDHCPDOnNonIPPTVLANS $start_dhcp $profile)

			#Check if ippt_enable is present and if set to 1
			is_ippt_teardowned=$(uci_get_state network ${cfg} ippt_teardown_status)
			log $(basename "$0") "PerformStartStopDHCPDOnNonIPPTVLANS" $LINENO "is_ippt_teardowned=$is_ippt_teardowned"
			if [ "$is_ippt_teardowned" -eq 1 ]; then
				#We enter this case if BH is up and IPPT is disabled
				perform_toggle="1"
				uci_revert_state network ${cfg} ippt_teardown_status "1"
			fi
		fi
	else
		#Disable dhcp on all mapped vlans and toggle links.
		#control enters here when IPPT is not enabled and BH is brough down
		if [ "$with_nat" = "$IP_PASSTHROUGH_MODE_WITHOUT_NAT" -a ! -f /tmp/ipv4config$profile -a "$bh_status" -eq 1 -a "$ipv4_nat_status" -eq 0 ]; then
			perform_toggle=$(StartStopDHCPDOnNonIPPTVLANS $start_dhcp $profile)
		fi
	fi

	log $(basename "$0") "PerformStartStopDHCPDOnNonIPPTVLANS" $LINENO "perform_toggle=$perform_toggle"

	echo "$perform_toggle"

}

#Function to perform a series of IPPT checks to ensure configurations are set as required
#$1 - Profile ID
function perform_ippt_check() {
	local profile="$1"
	local ippt profile_idx ippt_bridge_context bridge_found
	local device_type downstream downdevice default_pdn bind

	#Check if ippt is enabled
	ippt=$(check_ippt_enable $profile)
	if [ -n "$ippt" ] && [ $ippt -eq 0 ]; then
		log $(basename "$0") "perform_ippt_check" $LINENO "IPPT is not enabled."
		echo $ippt
		return
	elif [ -n "$ippt" ] && [ $ippt -eq 1 ]; then
		log $(basename "$0") "perform_ippt_check" $LINENO "IPPT is set to enabled state"

		#Get ippt bridge context
		ippt_bridge_context=$(get_ippt_bridge_context $profile)
		if [ -z "$ippt_bridge_context" ]; then
			log $(basename "$0") "perform_ippt_check" $LINENO "Invalid ippt_bridge_context value found"
			return
		fi

		#Check if current ippt bridge context matches the list of downstream bridges
		#and get the bridge_found value
		bridge_found=$(check_ippt_bridge_found $profile $ippt_bridge_context)

		if [ -z "$bridge_found" ] || [ $bridge_found -eq -1 ]; then
			log $(basename "$0") "perform_ippt_check" $LINENO "Invalid bridge_found value found"
			return
		fi

		#Get profile_idx
		profile_idx=$(util_get_profile_index $profile)

		#Check if ippt_device_type is present
		device_type=$(uci get qcmap_lan.@profile[$profile_idx].ippt_device_type)
		if [ -z "$device_type" ]; then
			log $(basename "$0") "perform_ippt_check" $LINENO "Device type not found. Exiting!"
			return
		fi
	else
		log $(basename "$0") "perform_ippt_check" $LINENO "invalid ippt_enable value found. Exiting!"
		return
	fi

	#Get downstream values.
	downstream=$(util_get_downstream_value_v4 $profile)

	#Get default_pdn
	default_pdn=$(util_get_default_pdn)

	#Get bind parameter value
	bind=$(util_get_bind $profile)

	#Check if downdevice is ready
	[ -n "$downstream" ] && {
		for i in $downstream
		do
			[ -n "$i" ] && network_get_device downdevice "$i"
			#NOTE: downstream entry in uci may be present even when bridge is not enumerated
			if [ -z "$downdevice" ] && [ $default_pdn -ne $profile ]; then
				#Control enters this condition when there is no bridge for on-demand PDN
				[ $bind -eq 1 -o $ippt -eq 1 ] && {
					log $(basename "$0") "perform_ippt_check" $LINENO "Down device is not ready yet"
					return
				}
			elif [ -z "$downdevice" ] && [ $default_pdn -eq $profile ]; then
				#Indicates no bridge for default_pdn. Print out a log and continue.
				log $(basename "$0") "perform_ippt_check" $LINENO "downdevice for downstream-$i is not present for profile: $profile which is a default profile"
			fi
		done
	}

	echo $ippt
}

#Function to setup lan side of IP Passthrough
#$1 - Current Profile ID
#$2 - wan side section name
function setup_lan_ippt() {
	local profile="$1"
	local cfg="$2"
	local dedicated_rt downstream downdevice device_type default_pdn
	local ippt_bridge_context with_nat bind profile_idx lladdr
	local bridge_found bridge_id
	local start_dhcp="0"
        local total_sw_path_filters

	#Get all ipv4 related values from tmp config file
	[ -f /tmp/ipv4config$profile ] && . /tmp/ipv4config$profile
	ip=$PUBLIC_IP
	DNS=$DNSSERVERS
	gateway=$GATEWAY
	subnet=$NETMASK
	interface=$IFNAME
	ip4mtu=$IPV4MTU

	#Perform series of NULL checks
	if [ -z "$ip" ]; then
		log $(basename "$0") "setup_lan_ippt" $LINENO "RMNET No IP address information"
		return
	fi
	if [ -z "$interface" ]; then
		log $(basename "$0") "setup_lan_ippt" $LINENO "RMNET Interface not found"
		return
	fi

	#Get profile index
	profile_idx=$(util_get_profile_index $profile)

	#Set active_ippt to 1
	uci set qcmap_lan.@profile[$profile_idx].active_ippt=1

	#Inform NetIfD the interface is to be termed external.
	proto_init_update "$interface" 1 1

	#Get downstream value
	downstream=$(util_get_downstream_value_v4 $profile)

	#Get with_nat value
	with_nat=$(util_get_with_nat)

	#Get bind parameter
	bind=$(util_get_bind $profile)

	#Get ippt bridge context
	ippt_bridge_context=$(get_ippt_bridge_context $profile)

	#Get bridge vlan ID
	bridge_id=$(uci get qcmap_lan.@profile[$profile_idx].ippt_bridge_context)

	#Check if bridge is found
	bridge_found=$(check_ippt_bridge_found $profile $ippt_bridge_context)

	#Get device type
	device_type=$(uci get qcmap_lan.@profile[$profile_idx].ippt_device_type)

	#Get default_pdn
	default_pdn=$(util_get_default_pdn)

	#Disable DMZ when IPPT is active
	ippt_disable_dmz $device_type

	for i in $downstream
	do
		[ -n "$i" ] && network_get_device downdevice "$i"
		#In case of bridge is not enumerated, downdevice will be NULL
		#and control will not enter the below condition
		if [ -n "$downdevice" ]; then
			uci_set_state network ${i} ifname $downdevice

			if [ -n "$ippt_bridge_context" ] && [ "$ippt_bridge_context" = "$i" ]; then
				rmnet_lan_setup_ippt $i $downdevice $gateway $subnet $ip $with_nat $cfg $profile_idx $ip4mtu
			fi
		else
			log $(basename "$0") "setup_lan_ippt" $LINENO "downdevice for downstream-$i is empty"
		fi
	done

	#Create Dedicated Routing Table
	util_create_dedicated_rt $profile
	dedicated_rt=custom_bind_${profile}

	#Generate IP address for rmnet_dataX interface
	lladdr=$(util_generate_priv_ip)

	#Redundancy check if generated ip address is same as public ip address
	if [ "$lladdr" = "$ip" ]; then
		#Indicates the random IP has failed to generate
		return
	fi

	#Inform IPA with details
	if [ $bridge_found -eq 1 ]; then
		inform_ipa_ippt_setup $device_type $with_nat $default_pdn $profile $bridge_id $interface
	fi

	#Install drop rules to drop traffic till IPPT is completely setup
	#This is to avoid traffic generation based on linklocal ip's
	util_install_iptables_rules_to_drop_all_traffic "$interface" "$profile" "ipv4v6"

	#Delete conntrack entries for V4
	util_del_conntrack "$profile" "ipv4"

	#add link local address to rmnet_dataX interface
	ifconfig $interface $lladdr netmask "${mask:-255.255.255.0}"
	proto_add_ipv4_address "$lladdr" "${mask:-255.255.255.0}"

	if [ $bind -ne 1 ]; then
		if [ -n "$ip4mtu" ]; then
			proto_add_ipv4_route 0.0.0.0 0 "" "" "" "$ip4mtu"
		else
			proto_add_ipv4_route 0.0.0.0 0 "" ""
		fi
	fi

	#Inform NetIfD of DNS information and add DNS information to resolv file
	util_add_dnsv4 $profile "$DNS"

	#Add MTU entries
	util_add_mtu_options $profile 4

	#Create SNAT rule
	if [ $with_nat -eq 1 ]; then
		util_create_snat_rule $cfg $ip
	else
		proto_send_update "$cfg"
	fi

	#Setup route for wan side of the network.
	if [ $bind -eq 1 ]; then
		util_setup_wan_route_rule_v4 $cfg $lladdr $interface $dedicated_rt
	fi

	#Setup routes for lan side of the network
	util_setup_lan_route_rule_v4 $cfg $profile 1 $ippt_bridge_context

	#Reset masquerade bit of firewall zone to zero
	util_reset_masq_bit $cfg

	#Add DHCP marking for packets
	for i in $downstream
	do
		[ -n "$i" ] && network_get_device downdevice "$i"
		if [ -n "$downdevice" ]; then
			if [ -n "$ippt_bridge_context" ] && [ $bridge_found -eq 1 ]; then
				ippt_add_dhcp_mark_pkts $i $ippt_bridge_context $gateway $default_pdn $profile $ip $downdevice
			fi
		fi
	done

	#Increase IPPT count by 1
	ippt_increase_pdn_count

	#flush network cache since we have modify network setting.
	network_flush_cache

	#For IPPT_without_NAT disable dhcp on vlans
	if [ "$with_nat" = "$IP_PASSTHROUGH_MODE_WITHOUT_NAT" ]; then
		StartStopDHCPDOnNonIPPTVLANS $start_dhcp $profile $ippt_bridge_context
	fi

	#Perform dnsmasq reload
        if [ $with_nat -eq 2 ]; then
          /etc/init.d/dnsmasq reload
        else
           /etc/init.d/dnsmasq restart "${ippt_bridge_context}_dns"
        fi

	#For WITHOUT_NAT scenario we need to toggle all links
	#So that other clients loose their IP
	if [ $with_nat -eq 2 ]; then
		device_type="Any"
	fi

	#Perform link toggle
	ippt_link_toggle $profile_idx $device_type $ippt_bridge_context

        #Adding Global port sw path filters
        add_global_port_sw_filters $profile

        #Check if sw path filters are added in UCI
        total_sw_path_filters=$(uci get qcmap_lan.@profile[$profile].total_swpath_filters)
        if [ "$total_sw_path_filters" -gt 0 ]; then
          #Adding sw path filters on apps and inform IPA
          add_ippt_sw_path_filters $profile
       fi

	#Delete added drop rules to allow traffic as IPPT is setup completed
	util_delete_iptables_rules_to_drop_all_traffic "$interface" "$profile" "ipv4v6"

	# quectel add to check and add br-lan route
	if [ -e /etc/data/rndis_link_trigger ]; then
		result=$(ip ro show|grep br-lan)
		if [ -z "$result" ]; then
			ip route add $ip dev br-lan
			log $(basename "$0") "setup_lan_ippt" $LINENO "retry add br-lan route"
		fi
	fi

	# quectel add to check and add rmnet_data default route
	if [ -e /etc/data/rndis_link_trigger ]; then
		result=$(ip ro show|grep rmnet_data|grep default)
		if [ -z "$result" ]; then
			if [ -n "$ip4mtu" ]; then
				proto_add_ipv4_route 0.0.0.0 0 "" "" "" "$ip4mtu"
			else
				proto_add_ipv4_route 0.0.0.0 0 "" ""
			fi
			ip ro add default dev $interface proto static scope link mtu lock 1500
			log $(basename "$0") "setup_lan_ippt" $LINENO "retry add default rmnet route"
		fi
	fi
}

#Function to handle cleanup of IP Passthrough configuration
#$1 - Wan side section name
#$2 - Profile ID
function teardown_lan_ippt() {
	local cfg="$1"
	local profile="$2"
	local profile_idx downstream with_nat bind ippt_bridge_context
	local bridge_found active_ippt downdevice bridge_id ippt_count ippt_device_type
	local start_dhcp="0"
	local perform_toggle="1"
        local total_sw_path_filters

        #Delete Global port sw path filters
        delete_global_port_sw_filters $profile

        #Check if sw path filters are added in UCI
        total_sw_path_filters=$(uci get qcmap_lan.@profile[$profile].total_swpath_filters)
        if [ "$total_sw_path_filters" -gt 0 ]; then
          #Delete sw path filters on apps and inform IPA
          delete_ippt_sw_path_filters $profile

          uci set qcmap_lan.@profile[$profile].total_swpath_filters="0"
        fi

	#Get downstream values
	downstream=$(util_get_downstream_value_v4 $profile)

	#Get With/Without-NAT parameter
	with_nat=$(util_get_with_nat)

	#Get bind parameter
	bind=$(util_get_bind $profile)

	#Get profile index
	profile_idx=$(util_get_profile_index $profile)

	for i in $downstream
	do
		#Check if WITH_NAT/WITHOUT_NAT mode is set. if WITHOUT_NAT, disable
		#dhcp server on the bridge. If WITH_NAT, do nothing
		#Check if downstream is default bridge
		if [ -n "$i" ] && [ $with_nat -eq 2 ]; then
			if [ "$i" = "lan" ]; then
				log $(basename "$0") "teardown_lan_ippt" $LINENO "downstream is default bridge"
			else
				uci set dhcp.$i.ignore="1"
				uci commit dhcp
			fi
		fi
	done

	#Get IPPT bridge context
	ippt_bridge_context=$(get_ippt_bridge_context $profile)

	#Check if bridge is found
	bridge_found=$(check_ippt_bridge_found $profile $ippt_bridge_context)

	#Get active IPPT value
	active_ippt=$(check_active_ippt $profile)

	if [ $bridge_found -eq 1 ]; then
		#Recover DHCP server
		ippt_recover_dhcp $cfg $ippt_bridge_context $with_nat $profile_idx
	fi

	#Get bridge vlan ID
	bridge_id=$(uci get qcmap_lan.@profile[$profile_idx].ippt_bridge_context)

	#Inform ipa about the teardown
	inform_ipa_ippt_teardown $cfg $profile_idx $bridge_found $bridge_id

	#Delete DNS entries
	util_del_dnsv4 $cfg $profile

	#Delete MTU options
	util_delete_mtu_options $profile 4

	#Set masquerade bit of firewall to 1 and Reset active_ippt to 0
	util_set_masq_bit $cfg

	#Enable DMZ when IPPT is disabled
	ippt_enable_dmz

	#Set active ippt to zero
	uci set qcmap_lan.@profile[$profile_idx].active_ippt=0
	uci commit

	#Delete wan side routes and rules
	util_delete_wan_route_rule_v4 $cfg $profile

	#Delete routes and rules of lan side
	util_delete_lan_route_rule_v4 $profile $active_ippt $ippt_bridge_context

	#Delete all upstream related network rules from state
	uci_revert_state network $cfg

	#Flag indicates teardown happened for IPPT
	#This flag is required to perform link toggle when BH is up and IPPT is disabled
	uci_set_state network ${cfg} ippt_teardown_status "1"

	#decrease ippt_pdn_count by 1
	ippt_decrease_pdn_count

	#For IPPT_WITHOUT_NAT enable dhcp when IPPT is disabled but BH is UP
	if [ "$with_nat" = "$IP_PASSTHROUGH_MODE_WITHOUT_NAT" ]; then
		#Control enters here if BH is brought down and IPPT is enabled.
		#In this case we need to disable dhcp for vlans mapped to pdn
		if [ ! -f /tmp/ipv4config$profile ]; then
			StartStopDHCPDOnNonIPPTVLANS $start_dhcp $profile $ippt_bridge_context
		else
			#When BH is UP and IPPT disabled toggle and enable dhcp logic
			#will be taken care as part of setup_ipv4_lan
			perform_toggle="0"
		fi
	fi

	#Install drop rules to drop traffic till IPPT is completely setup
	#This is to avoid traffic generation based on linklocal ip's
	util_install_iptables_rules_to_drop_all_traffic "$interface" "$profile" "ipv4v6"

	#Delete conntrack entries for V4
	util_del_conntrack "$profile" "ipv4"

	if [ -f /tmp/ipv4config$profile ]; then
		[ -f /tmp/ipv4config$profile ] && . /tmp/ipv4config$profile
		ip=$PUBLIC_IP
		subnet=$NETMASK
		interface=$IFNAME

		ifconfig $interface $ip netmask "${subnet:-255.255.255.0}"
		log $(basename "$0") "teardown_lan_ippt" $LINENO "ifconfig $interface $ip netmask '${subnet:-255.255.255.0}'"
	fi

	#Perform dnsmasq reload
        if [ $with_nat -eq 2 ]; then
          /etc/init.d/dnsmasq reload
	else
          /etc/init.d/dnsmasq restart "${ippt_bridge_context}_dns"
        fi

	#Perform network reload
	ubus call network reload

	#Get device_type
	device_type=$(uci get qcmap_lan.@profile[$profile_idx].ippt_device_type)
	ippt_device_type="$device_type"

	#For WITHOUT_NAT scenario we need to toggle all links
	#So that other clients loose their IP
	if [ $with_nat -eq 2 ]; then
		device_type="Any"
	fi

	#Delete dhcp lease entry for usb device type
	if [ -n "$active_ippt" -a $active_ippt -eq 1 ]; then
		if [ "$ippt_device_type" = "USB" ]; then
			delete_dhcp_lease_entry
		fi
	fi

	if [ "$perform_toggle" -eq 1 ]; then
		#Perform link toggle
		ippt_link_toggle $profile_idx $device_type $ippt_bridge_context
	fi

	#Final commit
	uci commit

	#Delete added drop rules to allow traffic as IPPT is setup completed
	util_delete_iptables_rules_to_drop_all_traffic "$interface" "$profile" "ipv4v6"

}

#function to inform IPA about SW path filters
function inform_ipa_sw_path_filters() {

       local gw_ip="$1"
       local start_port="$2"
       local end_port=$3
       local ipa_flag="$4"
       local ipv4_enable port_enable

       if [ "$ipa_flag" -eq 1 ] ; then
           ipv4_enable=1
           port_enable=1
       elif [ "$start_port" -eq 53 ] || [ "$start_port" -eq 67 ] ; then
          ipv4_enable=0
          port_enable=1
       else
           ipv4_enable=0
           port_enable=0
       fi

       if [ "$ipv4_enable" -eq 1 ] ; then
           if [ "$end_port" -eq 0 ] ; then
               ipa ippt_sw_flt -E $ipv4_enable --ip "$gw_ip" -e $port_enable --port "$start_port"
           else
               ipa ippt_sw_flt -E $ipv4_enable --ip "$gw_ip" -e $port_enable --port "$start_port $end_port"
           fi
       else
          if [ "$end_port" -eq 0 ] ; then
               ipa ippt_sw_flt -E $ipv4_enable -e $port_enable --port "$start_port"
           else
               ipa ippt_sw_flt -E $ipv4_enable -e $port_enable --port "$start_port $end_port"
           fi
       fi

}

#function to delete sw filetrs on apps
function delete_ippt_sw_path_filters() {

       local profile="$1"
       local bridge_name gw_ip
       local protocol total
       local start_port end_port
       local ippt_route_table_id=$((profile + 200))
       local ipa_flag=0

       local profile_idx=$(util_get_profile_index $profile)
       local bridge_context=$(get_ippt_bridge_context $profile)
       gw_ip=$(uci get network.${bridge_context}_bind4.ipaddr)
       total=$(uci get qcmap_lan.@profile[$profile_idx].total_swpath_filters)

       for i in $(seq 1 $total); do
         protocol=$(uci get qcmap_lan.@profile[$profile_idx].protocol$i)
         start_port=$(uci get qcmap_lan.@profile[$profile_idx].start_port$i)
         end_port=$(uci get qcmap_lan.@profile[$profile_idx].end_port$i)

         if [ "$protocol" -eq 1 ] ; then
           protocol="tcp"
         elif [ "$protocol" -eq 2 ] ; then
           protocol="udp"
         elif [ "$protocol" -eq 4 ] ; then
           protocol="icmp"
         else
           exit 1
         fi

         if [ "$protocol" == "icmp" ]; then
             iptables -t mangle -D PREROUTING -i br-${bridge_context} -p $protocol -d $gw_ip -j MARK --set-mark $ippt_route_table_id
             start_port=0
             end_port=0
         else
             if [ "$start_port" -eq "$end_port" ]; then
                 iptables -t mangle -D PREROUTING -i br-${bridge_context} -p $protocol -d $gw_ip --dport $start_port -j MARK --set-mark $ippt_route_table_id
                 end_port=0
             else
                 iptables -t mangle -D PREROUTING -i br-${bridge_context} -p $protocol -d $gw_ip --dport $start_port:$end_port -j MARK --set-mark $ippt_route_table_id
             fi
         fi

         inform_ipa_sw_path_filters $gw_ip $start_port $end_port $ipa_flag
       done
}

#function to add sw filetrs on apps
function add_ippt_sw_path_filters() {

       local profile="$1"
       local bridge_name gw_ip
       local protocol total
       local start_port end_port
       local dns_port dhcp_port
       local ippt_route_table_id=$((profile + 200))
       local ipa_flag=1

       local profile_idx=$(util_get_profile_index $profile)
       local bridge_context=$(get_ippt_bridge_context $profile)
       gw_ip=$(uci get network.${bridge_context}_bind4.ipaddr)
       total=$(uci get qcmap_lan.@profile[$profile_idx].total_swpath_filters)

       for i in $(seq 1 $total); do
         protocol=$(uci get qcmap_lan.@profile[$profile_idx].protocol$i)
         start_port=$(uci get qcmap_lan.@profile[$profile_idx].start_port$i)
         end_port=$(uci get qcmap_lan.@profile[$profile_idx].end_port$i)

         if [ "$protocol" -eq 1 ] ; then
           protocol="tcp"
         elif [ "$protocol" -eq 2 ] ; then
           protocol="udp"
         elif [ "$protocol" -eq 4 ] ; then
           protocol="icmp"
         else
           exit 1
         fi

         if [ "$protocol" == "icmp" ]; then
             iptables -t mangle -A PREROUTING -i br-${bridge_context} -p $protocol -d $gw_ip -j MARK --set-mark $ippt_route_table_id
             start_port=0
             end_port=0
         else
             if [ "$start_port" -eq "$end_port" ]; then
                 iptables -t mangle -A PREROUTING -i br-${bridge_context} -p $protocol -d $gw_ip --dport $start_port -j MARK --set-mark $ippt_route_table_id
                 end_port=0
             else
                 iptables -t mangle -A PREROUTING -i br-${bridge_context} -p $protocol -d $gw_ip --dport $start_port:$end_port -j MARK --set-mark $ippt_route_table_id
             fi
         fi

         inform_ipa_sw_path_filters $gw_ip $start_port $end_port $ipa_flag
       done
}

function add_global_port_sw_filters() {

       local profile="$1"
       local bridge_name gw_ip
       local protocol="udp"
       local dns_port dhcp_port
       local ippt_route_table_id=$((profile + 200))

       local profile_idx=$(util_get_profile_index $profile)
       local bridge_context=$(get_ippt_bridge_context $profile)
       gw_ip=$(uci get network.${bridge_context}_bind4.ipaddr)
       dhcp_port=$(uci get qcmap_lan.@global[0].dhcp_port)
       dns_port=$(uci get qcmap_lan.@global[0].dns_port)

       iptables -t mangle -A PREROUTING -i br-${bridge_context} -p $protocol -d $gw_ip --dport $dhcp_port -j MARK --set-mark $ippt_route_table_id
       iptables -t mangle -A PREROUTING -i br-${bridge_context} -p $protocol -d $gw_ip --dport $dns_port -j MARK --set-mark $ippt_route_table_id

       inform_ipa_sw_path_filters $gw_ip $dns_port 0 0
       inform_ipa_sw_path_filters $gw_ip $dhcp_port 0 0
}

function delete_global_port_sw_filters() {

       local profile="$1"
       local bridge_name gw_ip
       local protocol="udp"
       local dns_port dhcp_port
       local ippt_route_table_id=$((profile + 200))
       local ipa_flag=0

       local profile_idx=$(util_get_profile_index $profile)
       local bridge_context=$(get_ippt_bridge_context $profile)
       gw_ip=$(uci get network.${bridge_context}_bind4.ipaddr)
       dhcp_port=$(uci get qcmap_lan.@global[0].dhcp_port)
       dns_port=$(uci get qcmap_lan.@global[0].dns_port)

       iptables -t mangle -D PREROUTING -i br-${bridge_context} -p $protocol -d $gw_ip --dport $dhcp_port -j MARK --set-mark $ippt_route_table_id
       iptables -t mangle -D PREROUTING -i br-${bridge_context} -p $protocol -d $gw_ip --dport $dns_port -j MARK --set-mark $ippt_route_table_id

       inform_ipa_sw_path_filters $gw_ip $dns_port 0 $ipa_flag
       inform_ipa_sw_path_filters $gw_ip $dhcp_port 0 $ipa_flag
}

# Quectel: add for update dhcp host
# 1. update qcmap_lan; 2. update dhcp; 3. reload network
# $1=> vlan_id which have netlink evt;
# $2=> device_type ; $3 new mac address;
function ql_update_ippt_dhcp_host() {
	local vlan_id=$1
	local device_type=$2
	local mac_addr=$3
	local profile_idx ippt_bridge_context last_mac_addr public_ip ippt_actived

	log $(basename "$0") "ql_update_ippt_dhcp_host" $LINENO "vlan:${vlan_id}; MAC: ${mac_addr}"
	#get profile_idx to set qcmap_lan ippt_mac_addr
	idx_array=$(uci show qcmap_lan | grep -i profile | grep -i .ippt_bridge_context | awk -F '[][]' '{print $2}')
	for profile_index in $idx_array
	do
		ippt_bridge_ctx=$( uci get qcmap_lan.@profile[$profile_index].ippt_bridge_context)
		if [ "$ippt_bridge_ctx" = "$vlan_id" ]; then
			profile_idx=$profile_index
			ippt_bridge_context=$ippt_bridge_ctx
			break
		fi
	done

	ippt_enabled=$(uci get qcmap_lan.@profile[$profile_idx].ippt_enable)
	adap_ippt_mode=$(uci get qcmap_lan.@profile[$profile_idx].adap_ippt_mode)
	if [ $adap_ippt_mode -eq 0 ] || [ $ippt_enabled -eq 0 ]; then
		return
	fi
	
	saved_dev_type=$(uci get qcmap_lan.@profile[$profile_idx].ippt_device_type)
	if [ "$saved_dev_type" != "$device_type" ] && [ "$saved_dev_type" != "Any" ]; then
		return
	fi
	last_mac_addr=$(uci get qcmap_lan.@profile[$profile_idx].ippt_mac_addr)

	#for the first decice mode
	if [ $adap_ippt_mode -eq 2 ] && [ -n "$last_mac_addr" ] && [ "$last_mac_addr" != "00:00:00:00:00:00" ]; then
		return
	fi
	#Any device mode do not supprt the latest mode
	# if [ $adap_ippt_mode -eq 1 ] && [ $saved_dev_type == "Any" ]; then
	# 	log $(basename "$0") "ql_update_ippt_dhcp_host" $LINENO "Any device mode do not supprt the latest mode"
	# 	return
	# fi

	ippt_actived=$(uci get qcmap_lan.@profile[$profile_idx].active_ippt)
	log $(basename "$0") "ql_update_ippt_dhcp_host" $LINENO "idx:${profile_idx} vlan: ${vlan_id}, last: $last_mac_addr; new: $mac_addr"
	if [ -n "$last_mac_addr" ] && [ -n "$mac_addr" ] && [ "$last_mac_addr" != "$mac_addr" ]; then
		log $(basename "$0") "ql_update_ippt_dhcp_host" $LINENO "need to change MAC addre,stop DHCP"

		if [ $ippt_actived -eq 1 ]; then
			/etc/init.d/dnsmasq stop
		fi
		
		uci set qcmap_lan.@profile[$profile_idx].ippt_mac_addr="$mac_addr"
		log $(basename "$0") "ql_update_ippt_dhcp_host" $LINENO "uci set qcmap_lan.@profile[$profile_idx].ippt_mac_addr='$mac_addr' : $?"

		#update mac in dhcp
		host_idx=$(uci show dhcp | grep -i host | grep -i ippt_host | awk -F '[][]' '{print $2}')
		idx="-1"
		found=0
		for host_index in $host_idx
		do
			host_instance=$( uci get dhcp.@host[$host_index].instance)
			if [ $vlan_id -eq 0 -a "$host_instance" = "lan_dns" ]; then
				idx=$host_index
				log $(basename "$0") "ql_update_ippt_dhcp_host" $LINENO "dhcp.@host[$host_index]; vlan:${vlan_id}"
				found=1
				break
			elif [ $vlan_id -ne 0 -a "$host_instance" = "lan${vlan_id}_dns" ]; then
				idx=$host_index
				log $(basename "$0") "ql_update_ippt_dhcp_host" $LINENO "dhcp.@host[$host_index]; vlan:${vlan_id}"
				found=1
				break
			fi
		done

		if [ "$found" -eq 1 ]; then
			public_ip=$(uci get dhcp.@host[$idx].ip)
			uci set dhcp.@host[$idx].mac="$mac_addr"
		fi
		uci commit

		if [ "${vlan_id}" != "0" ]; then
			# delete old lease of $mac_addr 
			old_ip=$(cat /tmp/data/dhcp.leases.lan${vlan_id} | grep "$mac_addr" | awk {'print $3'})
			dhcp_release br-lan${vlan_id} $old_ip $mac_addr
			sed -i "/$mac_addr/d" /tmp/data/dhcp.leases.lan${vlan_id}
			#delete lease file about public ip
			sed -i "/$public_ip/d" /tmp/data/dhcp.leases.lan${vlan_id}
			log $(basename "$0") "ql_update_ippt_dhcp_host" $LINENO "dhcp_release br-lan${vlan_id} $old_ip $mac_addr"
		else
			# delete old lease of $mac_addr 
			old_ip=$(cat /tmp/data/dhcp.leases.lan | grep "$mac_addr" | awk {'print $3'})
			dhcp_release br-lan $old_ip $mac_addr
			sed -i "/$mac_addr/d" /tmp/data/dhcp.leases.lan
			#delete lease file about public ip
			sed -i "/$public_ip/d" /tmp/data/dhcp.leases.lan
			log $(basename "$0") "ql_update_ippt_dhcp_host" $LINENO "dhcp_release br-lan $old_ip $mac_addr"
		fi

    sync
		#ubus call network reload
		/etc/init.d/dnsmasq reload
		#/lib/netifd/rmnet_update.sh up $profile_idx 1
		if [ $ippt_actived -eq 1 ]; then
			ippt_link_toggle $profile_idx $device_type $ippt_bridge_context
		fi
	fi
}

case $1 in
	set_ippt_state)
		set_ippt_state $2 $3
		;;
	add_ippt)
		set_ippt_config $2 $3 $4 $5
		;;
	check_active_ippt)
		check_active_ippt $2
		;;
	reset_ippt_lan_params)
		reset_ippt_lan_params
		;;
	ippt_update_atBHSwitch)
		ippt_update_atBHSwitch
		;;
	reset_ippt_firewall_params)
		reset_ippt_firewall_params
		;;
	StartStopDHCPDOnNonIPPTVLANS)
		StartStopDHCPDOnNonIPPTVLANS $2 $3 $4
		;;
	PerformStartStopDHCPDOnNonIPPTVLANS)
		PerformStartStopDHCPDOnNonIPPTVLANS $2 $3 $4 $5
		;;
	perform_ippt_check)
		perform_ippt_check $2
		;;
	setup_lan_ippt)
		setup_lan_ippt $2 $3
		;;
	teardown_lan_ippt)
		teardown_lan_ippt $2 $3
                ;;
        delete_ippt_sw_path_filters)
                delete_ippt_sw_path_filters $2
                ;;
        add_ippt_sw_path_filters)
                add_ippt_sw_path_filters $2
                ;;
        add_global_port_sw_filters)
                add_global_port_sw_filters $2
                ;;
        delete_global_port_sw_filters)
                delete_global_port_sw_filters $2
                ;;
        inform_ipa_sw_path_filters)
                inform_ipa_sw_path_filters $2 $3 $4 $5
		;;
	ql_update_ippt_dhcp_host)
		ql_update_ippt_dhcp_host $2 $3 $4
		;;
	*)
		log $(basename "$0")"" $LINENO "Invalid option"
		;;
esac
