#!/bin/sh
#  Copyright (c) 2023 Qualcomm Technologies, Inc.
#  All Rights Reserved.
#  Confidential and Proprietary - Qualcomm Technologies, Inc.

#source common functions
. /etc/data/lanUtils.sh

#source common functions
. /etc/data/mbbUtils.sh

#source NetIfD related files
. /lib/functions.sh
. /lib/functions/network.sh
. /lib/netifd/netifd-proto.sh

#Function to add DHCP reservation record
#$1 - Public IP address
function add_ip_collision_dhcp_reservation_record() {
	local ip="$1"

	util_execute_uci add dhcp host
	util_execute_uci_set dhcp.@host[-1].dns "1"
	util_execute_uci_set dhcp.@host[-1].ip $ip
	util_execute_uci_set dhcp.@host[-1].name "none"

	util_execute_uci commit dhcp
	#We need to delete dhcp lease file
	rm -f /tmp/data/dhcp.leases

}

#Function to delete DHCP reservation record
#$1 - Public IP address
function del_ip_collision_dhcp_reservation_record() {
	local ip="$1"
	local host_ip

	#Iterate through dhcp hosts and find the matching ip address
	for i in $(seq 0 14)
	do
		#get ip
		host_ip=$(util_execute_uci get dhcp.@host[$i].ip)
		#compare it with public_ip
		if [ "$ip" = "$host_ip" ]; then
			log $(basename "$0") "del_ip_collision_dhcp_reservation_record" $LINENO " ip addr match found in dhcp host section"
			break
		fi
	done
	#delete the dhcp host entry
	util_execute_uci delete dhcp.@host[$i]

}

#Function to inform IPA on IP collision setup
#$1 - Profile ID
#$2 - WAN side section name
function inform_ipa_ip_collision_setup() {
	local profile="$1"
	local cfg="$2"
	local default_pdn ip interface downstream downdevice bridge_id ip_collision_active

	#Get default PDN
	default_pdn=$(util_get_default_pdn)

	#Get downstream value
	downstream=$(util_get_downstream_value_v4 $profile)

	for lan_iface in $downstream
	do
		ip_collision_active=$(perform_ip_collision_check_lan $lan_iface)

		if [ -n "$ip_collision_active" -a $ip_collision_active -eq 1 ];
		then
			#Get downdevice
			[ -n "$lan_iface" ] && network_get_device downdevice "$lan_iface"

			bridge_id=$(util_get_bridge_id $downdevice)

			#Get Public IP and Interface name
			ip=$(uci_get_state network $cfg downip)
			interface=$(uci_get_state network $cfg ifname)

			if [ $default_pdn = $profile ]; then
				ipa ip_collision enable vlan 0 default_pdn 1 pdn_name $interface pdn_ip $ip
			else
				ipa ip_collision enable vlan $bridge_id default_pdn 0 pdn_name $interface pdn_ip $ip
			fi

			log $(basename "$0") "inform_ipa_ip_collision_setup" $LINENO " bridge_id $bridge_id, lan_iface $lan_iface,pdn_name $interface, pdn_ip $ip"
		fi

	done
}

#Function to inform IPA on IP collision teardown
#$1 - Profile ID
#$2 - WAN side section name
function inform_ipa_ip_collision_teardown() {
	local profile="$1"
	local cfg="$2"
	local default_pdn ip interface downstream downdevice bridge_id

	#Get default PDN
	default_pdn=$(util_get_default_pdn)

	#Get downstream value
	downstream=$(util_get_downstream_value_v4 $profile)

	for lan_iface in $downstream
	do

		ip_collision_active=$(perform_ip_collision_check_lan $lan_iface)

		if [ -n "$ip_collision_active" -a $ip_collision_active -eq 1 ];
		then
			#Get downdevice
			[ -n "$lan_iface" ] && network_get_device downdevice "$lan_iface"

			bridge_id=$(util_get_bridge_id $downdevice)

			#Get Public IP and Interface name
			ip=$(uci_get_state network $cfg downip)
			interface=$(uci_get_state network $cfg ifname)

			if [ $default_pdn = $profile ]; then
				ipa ip_collision disable vlan 0 default_pdn 1 pdn_name $interface pdn_ip $ip
			else
				ipa ip_collision disable vlan $bridge_id default_pdn 0 pdn_name $interface pdn_ip $ip
			fi

			log $(basename "$0") "inform_ipa_ip_collision_teardown" $LINENO " bridge_id $bridge_id, lan_iface $lan_iface,pdn_name $interface, pdn_ip $ip"
		fi

	done

}

#Function to check if IP Collision flag is set for the lan interface
#$1 - lan interface
function perform_ip_collision_check_lan() {
	local lan_iface=$1
	local vlan_id ip_collision_active vlan_idx

	vlan_id=$(echo $lan_iface | grep -o '[0-9]*')
	if [ -z "$vlan_id" ]; then
		ip_collision_active=$(uci get qcmap_lan.@lan[0].ip_collision_active)
	else
		vlan_idx=$(util_get_vlan_index $vlan_id)
		#Get ip_collision_active flag for the lan interface
		if [ -n "vlan_idx" ];then
			ip_collision_active=$(uci get qcmap_lan.@vlan[$vlan_idx].ip_collision_active)
		fi
	fi

	echo $ip_collision_active
}

#Function to check if IP Collision is active
#$1 - Profile ID
function perform_ip_collision_check() {
	local profile="$1"
	local profile_idx ip_collision_active=-1

	#Get profile index
	local profile_idx=$(util_get_profile_index $profile)

	#Check if ip collision is active (1:1 VLAN , n:1 VLAN)
	#Get ip_collision_active flag for various profiles from qcmap_lan
	ip_collision_active=$(util_execute_uci get qcmap_lan.@profile[$profile_idx].ip_collision_active)

	echo $ip_collision_active
}

# Get the device type corresponding to the ip_collision
# $1 profile_id
# $2 ip addr
function get_ip_collision_device_type(){
  local profile_id=$1
  local ip=$2
  local downstream iface mac_addr device_type

  #Get downstream value
  downstream=$(util_get_downstream_value_v4 $profile_id)

  for iface in $downstream
  do
    mac_addr=$(util_get_mac_from_ip $iface $ip)
    if [ -n "$mac_addr" ]; then
      device_type=$(bridge fdb show | grep -wi "$mac_addr" | grep -wi "master br-$iface" | grep -v "permanent" | awk '{print $3}' | awk -F'.' '{print $1}')
      break
    fi
  done

  log $(basename "$0") $LINENO "get_ip_collision_device_type" "iface $iface, ip $ip, mac $mac_addr, device_type $device_type"
  echo $device_type
}

# Perform link toggle on the ip collision device type
#$1 - Profile ID
#$2 - WAN side section name
function perform_link_toggle_ip_collision(){
  local profile="$1"
  local cfg="$2"
  local downstream lan_iface
  local ip mac_addr ip_collision_active device_type

  #Get downstream value
  downstream=$(util_get_downstream_value_v4 $profile)

  for lan_iface in $downstream
  do
    ip_collision_active=$(perform_ip_collision_check_lan $lan_iface)

    if [ -n "$ip_collision_active" -a $ip_collision_active -eq 1 ]; then

      #Get Public IP
      ip=$(uci_get_state network $cfg downip)
      mac_addr=$(util_get_mac_from_ip $lan_iface $ip)

      # Peform link toggle for the interface having the ip address assigned
      device_type=$(get_ip_collision_device_type $profile $ip)

      if [ -n "$ip_collision_dev_type" -a -n "$downdevice" ]; then
        if [ "$device_type" = "rndis0" -o "$device_type" = "ecm0" ]; then
          util_perform_link_toggle "USB" $downdevice
        elif [ "$device_type" = "eth0" ]; then
          util_perform_link_toggle "ETH" $downdevice
        elif [ "$device_type" = "eth1" ]; then
          util_perform_link_toggle "ETH_NIC2" $downdevice
        elif [ "$device_type" = "wlan0" ]; then
          util_perform_link_toggle "WiFi" $downdevice $mac_addr
        fi
      fi
    fi

  done

  log $(basename "$0") $LINENO "perform_link_toggle_ip_collision" "device_type $device_type,mac_addr $mac_addr,iface $lan_iface"

}

#Function to delete uci option - ip_collision_active
#$1 - Profile ID
function del_ip_collision_active_uci_option()
{
	local profile="$1"
	local profile_idx lan_iface downstream ip_collision_active vlan_idx
	local delete_ip_collision_active_entry=1

	#Get profile Index
	profile_idx=$(util_get_profile_index $profile)

	#Get downstream value
	downstream=$(util_get_downstream_value_v4 $profile)

	# As part of the "lan activation", if the ip collision is detected call setup process is triggered.
	# As part of the call setup, clean up is also triggered.To avoid the clean up of the ip collision
	# parameters,ip_collision_enable_in_progress flag is introduced to distinguish between
	# the regular and ip collision triggered call teardown.
	ip_collision_enable_in_progress=$(uci get qcmap_lan.@profile[$profile_idx].ip_collision_enable_in_progress)
	if [ -n "$ip_collision_enable_in_progress" -a $ip_collision_enable_in_progress -eq 1 ]; then
		delete_ip_collision_active_entry=0
	fi

	if [ $delete_ip_collision_active_entry -eq 1 ]; then
		uci del qcmap_lan.@profile[$profile_idx].ip_collision_active
		log $(basename "$0") "teardown_lan_ip_collision" $LINENO "ip_collision_active deleted for profile $profile,
				profile_idx $profile_idx"


		for lan_iface in $downstream
		do
			ip_collision_active=$(perform_ip_collision_check_lan $lan_iface)

			if [ -n "$ip_collision_active" -a $ip_collision_active -eq 1 ]; then

				vlan_id=$(echo $lan_iface | grep -o '[0-9]*')
				if [ -z "$vlan_id" ];then
					uci del qcmap_lan.@lan[0].ip_collision_active
				else
					vlan_idx=$(util_get_vlan_index $vlan_id)
					if [ -n "vlan_idx" ];then
						uci del qcmap_lan.@vlan[$vlan_idx].ip_collision_active
					fi
				fi
				log $(basename "$0") "teardown_lan_ip_collision" $LINENO "ip_collision_active deleted for vlan_id $vlan_id,
				vlan_idx $vlan_idx"
			fi

		done
	fi
}

#Function to delete uci option on BH disconnect event
#$1 - Profile ID
function del_ip_collision_active_uci_option_on_BH_disconnect()
{
	local profile=$1
	local ip_collision_enable_in_progress

	log $(basename "$0") $LINENO "del_ip_collision_active_uci_option_on_BH_disconnect" "Enter profile=$profile"

	#Get profile Index
	profile_idx=$(util_get_profile_index $profile)

	#In case of BH disconnect event reset IP_Collision related UCI options
	ip_collision_enable_in_progress=$(uci get qcmap_lan.@profile[$profile_idx].ip_collision_enable_in_progress)
	if [ -n "$ip_collision_enable_in_progress" -a $ip_collision_enable_in_progress -eq 1 ]; then
		log $(basename "$0") $LINENO "del_ip_collision_active_uci_option_on_BH_disconnect" "reset IP_collision UCI"
		util_execute_uci_set qcmap_lan.@profile[$profile_idx].ip_collision_enable_in_progress "0"
		del_ip_collision_active_uci_option $profile
		uci commit
	fi

}

#Function to setup backhaul as per IP Collision mode
#$1 - Profile ID
#$2 - WAN side section name
function setup_lan_ip_collision() {
	local profile="$1"
	local cfg="$2"
	local bind dedicated_rt downstream downdevice lladdr
	local default_pdn lladdr
	local profile_idx ip_collision_enable_in_progress

	#Get all ipv4 related values from tmp config file
	[ -f /tmp/ipv4config$profile ] && . /tmp/ipv4config$profile
	ip=$PUBLIC_IP
	DNS=$DNSSERVERS
	gateway=$GATEWAY
	subnet=$NETMASK
	interface=$IFNAME
	ip4mtu=$IPV4MTU

	#Perform series of NULL checks
	if [ -z "$ip" -a -z "$ip6" ]; then
		log $(basename "$0") "setup_lan_ip_collision" $LINENO ":RMNET No IP address information"
		echo $ret_val
		return
	fi
	if [ -z "$interface" ]; then
		log $(basename "$0") "setup_lan_ip_collision" $LINENO ":RMNET Interface not found"
		echo $ret_val
		return
	fi

	uci_set_state network $cfg downip $ip
	uci_set_state network $cfg ifname $interface

	#Inform NetIfD the interface is to be termed external.
	proto_init_update "$interface" 1 1

	#Get bind parameter
	bind=$(util_get_bind $profile)

	#Create Dedicated Routing Table
	util_create_dedicated_rt $profile
	dedicated_rt=custom_bind_${profile}

	#Get downstream value
	downstream=$(util_get_downstream_value_v4 $profile)

	#Get default_pdn
	default_pdn=$(util_get_default_pdn)

	for i in $downstream
	do
		[ -n "$i" ] && network_get_device downdevice "$i"
		#In case of bridge is not enumerated, downdevice will be NULL
		#and control will not enter the below condition
		if [ -n "$downdevice" ]; then
			uci_set_state network ${i} ifname $downdevice
		else
			log $(basename "$0") "setup_lan_ip_collision" $LINENO ":downdevice for downstream-$i is empty"
		fi
	done

	#Generate IP address for rmnet_dataX interface
	lladdr=$(util_generate_priv_ip)

	#Inform IPA
	inform_ipa_ip_collision_setup $profile $cfg

	#add link local address to rmnet_dataX interface
	ifconfig $interface $lladdr netmask "${mask:-255.255.255.0}"
	proto_add_ipv4_address "$lladdr" "${mask:-255.255.255.0}"

	if [ $bind -ne 1 ]; then
		if [ -n "$ip4mtu" ]; then
			proto_add_ipv4_route 0.0.0.0 0 "" "" "" "$ip4mtu"
		else
			proto_add_ipv4_route 0.0.0.0 0 "" ""
		fi
	fi

	#Add dhcp reservation record
	add_ip_collision_dhcp_reservation_record $ip

	#Inform NetIfD of DNS information and add DNS information to resolv file
	util_add_dnsv4 $profile "$DNS"

	#Add MTU entries
	util_add_mtu_options $profile 4

	#Create SNAT rule
	util_create_snat_rule $cfg $ip

	#Setup route for wan side of the network.
	if [ $bind -eq 1 ]; then
		util_setup_wan_route_rule_v4 $cfg $lladdr $interface $dedicated_rt
	fi

	#Setup routes to bind lan/wan interface
	util_setup_lan_route_rule_v4 $cfg $profile 0

	#Reset masquerade bit of firewall zone to zero
	util_reset_masq_bit $cfg

	#Perform dnsmasq reload
	/etc/init.d/dnsmasq reload

	#Perform link toggle
	perform_link_toggle_ip_collision $profile $cfg

	#Get profile Index
	profile_idx=$(util_get_profile_index $profile)

	ip_collision_enable_in_progress=$(uci get qcmap_lan.@profile[$profile_idx].ip_collision_enable_in_progress)

	if [ -n "$ip_collision_enable_in_progress" -a $ip_collision_enable_in_progress -eq 1 ]; then
		uci set qcmap_lan.@profile[$profile_idx].ip_collision_enable_in_progress='0'
		uci commit qcmap_lan
	fi
}

#Function to cleanup IP Collision configuration
#$1 - WAN side section name
#$2 - Profile ID
function teardown_lan_ip_collision() {
	local cfg="$1"
	local profile="$2"
	local bind downstream ip profile_idx ip_collision_enable_in_progress

	#Get bind
	bind=$(util_get_bind $profile)

	#Get downstream values
	downstream=$(util_get_downstream_value_v4 $profile)

	#Get profile Index
	profile_idx=$(util_get_profile_index $profile)

	#Inform IPA of cleanup
	inform_ipa_ip_collision_teardown $profile $cfg

	#Get public IP
	ip=$(uci_get_state network $cfg downip)

	#Delete DHCP reservation record
	del_ip_collision_dhcp_reservation_record $ip

	#Delete DNS entries
	util_del_dnsv4 $profile

	#Delete MTU options
	util_delete_mtu_options $profile 4

	#Set masquerade bit of firewall to 1
	util_set_masq_bit $cfg

	#Delete wan side routes and rules
	util_delete_wan_route_rule_v4 $cfg $profile

	#Delete routes and rules of lan side
	util_delete_lan_route_rule_v4 $profile

	#Delete all upstream related network rules from state
	uci_revert_state network $cfg

	#Delete ip_collision_active uci entries
	del_ip_collision_active_uci_option $profile

	#Final commit
	uci commit

	#Perform dnsmasq reload
	/etc/init.d/dnsmasq reload

}

#Function to set ip_collision_active state
# $1 - index of profile section in qcmap_lan db
# $2 - enable_state
# NOTE: Not used anywhere. can be clubbed to check_Address_conflict or moved to cpp file
function set_ip_collision_state() {
	local profile_idx=$1
	local enable_state=$2

	#Set ip_collision_active parameter in qcmap_lan db to enable_state value
	util_execute_uci_set qcmap_lan.@profile[$profile_idx].ip_collision_active "$enable_state"

	ip_collision_active=`util_execute_uci get qcmap_lan.@profile[$profile_idx].ip_collision_active`
	log $(basename "$0") "set_ip_collision_state" $LINENO "ip_collision_active:$ip_collision_active"
	uci commit

}

# Check address conflict for a given wan_profile_handle against all the lans connected to it.
# wan --> lan (1:n)
# $1 profile_handle
function ip_collision_check_address_conflict(){
	profile_handle=$1
	local ip_addr subnet ip_addr_int vlan_id vlan_idx
	local downstream subnet_int profile_idx iface

	#Get downstream value corresponding to the wan interface
	downstream=$(util_get_downstream_value_v4 $profile_handle)

	# source the config file
	[ -f /tmp/ipv4config$profile_handle ] && . /tmp/ipv4config$profile_handle
	PUBLIC_IP_ADDR=$PUBLIC_IP
	PUBLIC_IP_ADDR_INT=`ipv4_to_unsigned_int $PUBLIC_IP_ADDR`

	for iface in $downstream
	do
		ip_addr=$(util_execute_uci get network.$iface.ipaddr)
		subnet=$(util_execute_uci get network.$iface.netmask)
		if [ -z "$ip_addr" -o -z "$subnet" ] ; then
			log $(basename "$0") "ip_collision_check_address_conflict" $LINENO " addr invalid, return"
			return
		fi

		ip_addr_int=`ipv4_to_unsigned_int $ip_addr`
		subnet_int=`ipv4_to_unsigned_int $subnet`
		profile_idx=$(util_get_profile_index $profile_handle)
		vlan_id=$(echo $iface | grep -o '[0-9]*')
		# Check if the IP address is within the subnet range
		if [ $((ip_addr_int & subnet_int)) -eq $((PUBLIC_IP_ADDR_INT & subnet_int)) ];
		then
			util_execute_uci_set qcmap_lan.@profile[$profile_idx].ip_collision_active '1'
			if [ -z "$vlan_id" ];then
				util_execute_uci_set qcmap_lan.@lan[0].ip_collision_active '1'
			else
				vlan_idx=$(util_get_vlan_index $vlan_id)
				if [ -n "vlan_idx" ];then
					util_execute_uci_set qcmap_lan.@vlan[$vlan_idx].ip_collision_active '1'
				fi
			fi

			log $(basename "$0") "ip_collision_check_address_conflict" $LINENO "IP Collision detected.
			$PUBLIC_IP is within the subnet range of interface $iface: ip_addr $ip_addr, subnet $subnet,
			vlan_id $vlan_id, vlan_idx $vlan_idx"
		else
			util_execute_uci_set qcmap_lan.@profile[$profile_idx].ip_collision_active '0'
			if [ -z "$vlan_id" ];then
				util_execute_uci_set qcmap_lan.@lan[0].ip_collision_active '0'
			else
				vlan_idx=$(util_get_vlan_index $vlan_id)
				if [ -n "vlan_idx" ];then
					util_execute_uci_set qcmap_lan.@vlan[$vlan_idx].ip_collision_active '0'
				fi
			fi
			log $(basename "$0") "ip_collision_check_address_conflict" $LINENO "IP Collision not detected.
			$PUBLIC_IP is not in the subnet range of interface $iface: ip_addr $ip_addr, subnet $subnet,
			vlan_id $vlan_id, vlan_idx $vlan_idx"
		fi
	done

	util_execute_uci commit qcmap_lan
}

# Check address conflict from the bridge_id. bridge_id and vlan_id mean the same.
# lan --> wan (1:1) wan --> lan (1:n)
# $1 bridge_id
function ip_collision_check_address_conflict_from_bridge_id(){
	local bridge_id=$1
	local wan_profile_id

	wan_profile_id=$(util_get_bridge_id_to_wan_mapping $bridge_id)
	if [ -n "wan_profile_id" ];then
		ip_collision_check_address_conflict $wan_profile_id
	else
		log $(basename "$0") "ip_collision_check_address_conflict_from_bridge_id" $LINENO "VLAN $bridge_id is not yet mapped to any PDN "
	fi
}


function reload_ip_collision_as_need()
{
  local profile=$1
  local current_backhaul=$2


  if [ "$current_backhaul" == "wan" ]; then

    #get current ip collision status
    ip_collision_active_old=$(perform_ip_collision_check $profile)
    if [ -z $ip_collision_active_old ]; then
      ip_collision_active_old="0"
    fi

    #get whether need enable ip collision if need
    ip_collision_check_address_conflict $profile

    #get expected ip cllision status
    ip_collision_active_new=$(perform_ip_collision_check $profile)
    if [ -z $ip_collision_active_new ]; then
      ip_collision_active_new="0"
    fi

    #if current != expected, setup it
      # case current=0, expected=1, need setup ip collision
      # case current=0, expected=0, no need action
      # case current=1, expected=1, no need action
      # case current=1, expected=0, would not happen since rmnet not ever down during switch
    if [ "$ip_collision_active_old" != "$ip_collision_active_new" ];then
      /lib/netifd/rmnet_update.sh "up" $profile 1
    fi

  else

    #wan is not high priority BH, need disable ip collision if it is active
    ip_collision_active=$(perform_ip_collision_check $profile)
    if [ -n "$ip_collision_active" ]; then
      if [ $ip_collision_active -eq 1 ]; then
        #set collision active false and trigger rmnet update
        util_execute_uci_set qcmap_lan.@lan[0].ip_collision_active '0'
        /lib/netifd/rmnet_update.sh "up" $profile 1
      fi
    fi
  fi

}


case $1 in
	perform_ip_collision_check)
		perform_ip_collision_check $2
		;;
	setup_lan_ip_collision)
		setup_lan_ip_collision $2 $3
		;;
	teardown_lan_ip_collision)
		teardown_lan_ip_collision $2 $3
		;;
	set_ip_collision_state)
		set_ip_collision_state $2 $3
		;;
	ip_collision_check_address_conflict)
		ip_collision_check_address_conflict $2
		;;
	ip_collision_check_address_conflict_from_bridge_id)
		ip_collision_check_address_conflict_from_bridge_id $2
		;;
	reload_ip_collision_as_need)
		reload_ip_collision_as_need $2 $3
		;;
	del_ip_collision_active_uci_option_on_BH_disconnect)
		del_ip_collision_active_uci_option_on_BH_disconnect $2
		;;
	*)
		log $(basename "$0") "" $LINENO ":Invalid option"
		;;
esac
