#!/bin/sh
#  Copyright (c) 2023 Qualcomm Technologies, Inc.
#  All Rights Reserved.
#  Confidential and Proprietary - Qualcomm Technologies, Inc.

#source log file
# Usage to use log function
# log  $(basename "$0") $LINENO "Log message"
. /etc/data/mbbUtils.sh

if [[ "$#" -le 0 ]]; then
  echo "Check usage gsbConfig.sh <functions> <arg1> <arg2> <...>"
  return 0
fi

function load_gsb() {
  max_retry=50;
  retry=0;


  /usr/bin/data_path_opt -l GSB

  while [[ "$retry" -le "$max_retry" ]]
  do
    result=$(lsmod | grep gsb)
    if [[ "$result" != "" ]]; then
      echo "Load GSB success and enable flag set for GSB"
      uci set qcmap_gsb.@gsb_enable_info[0].gsb_enable_flag="1"
      uci commit qcmap_gsb
      break
    fi
    let retry++
  done

  if [[ "$retry" -ge "$max_retry" ]]; then
    echo "Load GSB fail"
  fi

  return
}

function unload_gsb() {
  max_retry=50;
  retry=0;

  /usr/bin/data_path_opt -u GSB

  while [[ "$retry" -le "$max_retry" ]]
  do
    result=$(lsmod | grep gsb)
    if [[ "$result" = "" ]]; then
      echo "UnLoad GSB success and enable flag reset for GSB"
      uci set qcmap_gsb.@gsb_enable_info[0].gsb_enable_flag="0"
      uci commit qcmap_gsb
      break
    fi
    let retry++
  done

  if [[ "$retry" -ge "$max_retry" ]]; then
    echo "UnLoad GSB fail"
  fi

  return
}

function enable_gsb() {
  if_stopped=0

  num_of_entries=$(uci get qcmap_gsb.@no_of_configs[0].num_of_entries)
  #if no configuration entry found then return
  if [[ "$num_of_entries" = "0" ]]; then
    echo "No configuration found"
    return
  fi

  #if there is even a single config then load GSB
  load_gsb
  usleep 50000

  for i in $(seq $num_of_entries)
  do
    if_type=$(uci get qcmap_gsb.@gsb_info[$((i-1))].if_type)
    if [[ "$if_type" = "1"]] || [[ "$if_type" = "2" ]]; then
      continue
    fi

    if_stopped=0

    #stop all the interfaces who are configured to work with GSB
    #if interfaces are not up then skip
    if_name=$(uci get qcmap_gsb.@gsb_info[$((i-1))].if_name)
    if_state=$(cat /sys/class/net/$if_name/operstate | grep up)
    if [[ "$if_state" != "" ]]; then
      #stop if
      ifconfig $if_name down
      if_stopped=1
      usleep 10000
      echo "[GSB]: if $if_name disabled" > /dev/kmsg
    fi

    if [[ "$if_stopped" = "1" ]]; then
      #restart stopped interfaces, if they were stopped
      ifconfig $if_name up
      usleep 10000
      echo "[GSB]: if $if_name enabled" > /dev/kmsg
    fi
  done

  return
}

function set_gsb_config() {
  idx=0
  max_if_support=3

  num_of_entries=$(uci get qcmap_gsb.@no_of_configs[0].num_of_entries)
  if [[ "$num_of_entries" -ge "$max_if_support" ]]; then
    echo "Can not add $1 to GSB config, Max iface reached"
    return
  fi

  for i in $(seq $num_of_entries)
  do
    if_name=$(uci get qcmap_gsb.@gsb_info[$((i-1))].if_name)
    if [[ "$if_name" = "$1" ]]; then
      echo "Entry already present, if_name: $1"
      return
    fi
  done

  #Updating gsb database
  idx=$num_of_entries
  uci add qcmap_gsb gsb_info
  uci set qcmap_gsb.@gsb_info[$idx].if_name="$1"
  uci set qcmap_gsb.@gsb_info[$idx].bw_reqd_in_mb="$2"
  uci set qcmap_gsb.@gsb_info[$idx].if_high_watermark="$3"
  uci set qcmap_gsb.@gsb_info[$idx].if_low_watermark="$4"
  uci set qcmap_gsb.@gsb_info[$idx].if_type="$5"
  uci set qcmap_gsb.@gsb_info[$idx].ap_ip="$6"
  count=$(( idx + 1 ))
  uci set qcmap_gsb.@no_of_configs[0].num_of_entries="$count"
  uci commit qcmap_gsb

  return
}

function delete_gsb_config() {
  if_stopped=0
  gsb_idx=0
  num_of_entries=$(uci get qcmap_gsb.@no_of_configs[0].num_of_entries)
  for i in $(seq $num_of_entries)
  do
    if_name=$(uci get qcmap_gsb.@gsb_info[$((i-1))].if_name)
    if [[ "$if_name" = "$1" ]]; then
      gsb_idx=$((i-1))
      uci del qcmap_gsb.@gsb_info[$gsb_idx]
      count=$((num_of_entries-1))
      uci set qcmap_gsb.@no_of_configs[0].num_of_entries="$count"
      uci commit qcmap_gsb
    fi
  done

  # if GSB is enabled, it can be assumed that its cache is updated with the configuration.
  # So we need to remove the IF from GSB cache too.
  gsb_enable_flag=$(uci get qcmap_gsb.@gsb_enable_info[0].gsb_enable_flag)
  if [[ "$gsb_enable_flag" = "1" ]]; then
    if_state=$(cat /sys/class/net/$1/operstate | grep up)
    if [[ "$if_state" != "" ]]; then
      #stop if
      ifconfig $1 down
      if_stopped=1
      usleep 10000
    fi

    num_of_entries=$(uci get qcmap_gsb.@no_of_configs[0].num_of_entries)
    if [[ "$num_of_entries" = "0" ]]; then
      #unload GSB module
      unload_gsb

      #no reason to have GSB enabled at bootup either
      uci set qcmap_gsb.@gsb_boot_up_info[0].gsb_boot_up_config="0"
      uci commit qcmap_gsb
    fi

    if [[ "$if_stopped" = "1" ]]; then
      #restart stopped interfaces, if they were stopped
      ifconfig $1 up
    fi
  fi

  return
}

#Function to load SFE module
function load_sfe() {

  log $(basename "$0") "load_sfe" $LINENO "Enter load_sfe"
  /usr/bin/data_path_opt -l SFE
  log $(basename "$0") "load_sfe" $LINENO "Enabling bridge-nf-call-iptable proc entry"
  cmd="net.bridge.bridge-nf-call-iptables=1"
  cmd1="net.bridge.bridge-nf-call-ip6tables=1"
  sed -i '/bridge-nf-call-iptables/d' /etc/sysctl.conf
  sed -i '/bridge-nf-call-ip6tables/d' /etc/sysctl.conf
  echo $cmd >> /etc/sysctl.conf
  echo $cmd1 >> /etc/sysctl.conf
  #Enable proc entry on load_sfe
  echo 1 > /proc/sys/net/bridge/bridge-nf-call-iptables
  echo 1 > /proc/sys/net/bridge/bridge-nf-call-ip6tables

}

#Function to load SFE module
function unload_sfe() {

  log $(basename "$0") "unload_sfe" $LINENO "Enter unload_sfe"
  /usr/bin/data_path_opt -u SFE
  log $(basename "$0") "unload_sfe" $LINENO "Disabling bridge-nf-call-iptable proc entry"
  cmd="net.bridge.bridge-nf-call-iptables=0"
  cmd1="net.bridge.bridge-nf-call-ip6tables=0"
  sed -i '/bridge-nf-call-iptables/d' /etc/sysctl.conf
  sed -i '/bridge-nf-call-ip6tables/d' /etc/sysctl.conf
  echo $cmd >> /etc/sysctl.conf
  echo $cmd1 >> /etc/sysctl.conf
  #Disable proc entry on unload_sfe
  echo 0 > /proc/sys/net/bridge/bridge-nf-call-iptables
  echo 0 > /proc/sys/net/bridge/bridge-nf-call-ip6tables

}

case $1 in
  "LOAD_GSB")
  log  $(basename "$0") "case:LOAD_GSB" $LINENO "LOAD_GSB done"
    load_gsb
    break
    ;;
  "UNLOAD_GSB")
  log  $(basename "$0") "case:UNLOAD_GSB" $LINENO "UNLOAD_GSB done"
    unload_gsb
    break
    ;;
  "ENABLE_GSB")
    log  $(basename "$0") "case:ENABLE_GSB" $LINENO "ENABLE_GSB done"
    enable_gsb
    break
    ;;
  "SET_GSB_CONFIG")
  log  $(basename "$0") "case:SET_GSB_CONFIG" $LINENO "DELETE_GSB_CONFIG done"
    set_gsb_config $2 $3 $4 $5 $6 $7
    break
    ;;
  "DELETE_GSB_CONFIG")
  log  $(basename "$0") "case:DELETE_GSB_CONFIG" $LINENO "DELETE_GSB_CONFIG done"
    delete_gsb_config $2
    break
    ;;
  "LOAD_SFE")
    load_sfe
    ;;
  "UNLOAD_SFE")
    unload_sfe
    ;;
  *)
    echo "Invalid option"
    ;;
esac
