#!/bin/sh
#  Copyright (c) 2022 Qualcomm Technologies, Inc.
#  All Rights Reserved.
#  Confidential and Proprietary - Qualcomm Technologies, Inc.

# Usage: backhaulWWANConfig.sh add_map vlan_id profile_handle is_default_pdn is_pdn_active
# Usage: backhaulWWANConfig.sh del_map vlan_id profile_handle is_default_pdn
# Usage: backhaulWWANConfig.sh create_wwan_profile profile_handle ip_family
# Usage: backhaulWWANConfig.sh ip_collision_state profile_idx enable_state

#source log file.
. /etc/data/mbbUtils.sh

#Source lanUtils file
. /etc/data/lanUtils.sh

#Source firewall file
. /etc/data/firewallConfig.sh

QCMAP_IP_FAMILY_ETH=12
if [ "$#" -le 0 ]; then
  echo "Check usage ./backhaulWWANConfig.sh <add_map> vlan_id profile_handle is_default_pdn is_pdn_active"
  echo "Check usage ./backhaulWWANConfig.sh <del_map> vlan_id profile_handle is_default_pdn"
  echo "Check usage ./backhaulWWANConfig.sh <create_wwan_profile> profile_handle ip_family"
  echo "Check usage ./backhaulWWANConfig.sh <delete_wwan_profile> profile_handle"
  echo "Check usage ./backhaulWWANConfig.sh ip_collision_state profile_idx enable_state"
  return 1
elif [ \( "$1" == "add_map" \) -a \( "$#" -ne 3 \) ]; then
  echo "Check usage ./backhaulWWANConfig.sh <add_map> vlan_id profile_handle is_default_pdn is_pdn_active"
  return 1
elif [ \( "$1" == "del_map" \) -a \( "$#" -ne 3 \) ]; then
  echo "Check usage ./backhaulWWANConfig.sh <del_map> vlan_id profile_handle is_default_pdn"
  return 1
elif [ \( "$1" == "create_wwan_profile" \) -a \( "$#" -ne 3 \) ]; then
  echo "Check usage ./backhaulWWANConfig.sh <create_wwan_profile> profile_handle ip_family"
  return 1
elif [ \( "$1" == "delete_wwan_profile" \) -a \( "$#" -ne 2 \) ]; then
  echo "Check usage ./backhaulWWANConfig.sh <delete_wwan_profile> profile_handle"
  return 1
fi

# $1 profile_handle
#Function will delete all snat rules on profile deletion
function delete_snat_rules_on_profile_deletion(){
  local profile_handle=$1
  local no_of_redirect=$(uci get qcmap_lan.@no_of_configs[0].no_of_redirects)
  local count=$no_of_redirect
  local snat_rules_index=""
  for i in $(seq $no_of_redirect)
  do
    name=$(uci get firewall.@redirect[$((i-1))].name)
    if [ "$name" = "StaticNAT-$profile_handle" ];then
      idx=$((i-1))
      dest_ip=$(uci get firewall.@redirect[$idx].dest_ip)
      snat_rules_index="$idx $snat_rules_index"
      count=$((count-1))
      uci set qcmap_lan.@no_of_configs[0].no_of_redirects="$count"
      uci commit
    fi
  done

  if [ "$snat_rules_index" ];then
    for i in $snat_rules_index
    do
      uci delete firewall.@redirect[$i]
    done
  fi
  uci commit
}

# $1 profile_handle
#Function will delete all dmz rule on profile deletion
function delete_dmz_rule_on_profile_deletion(){
  local profile_handle=$1
  local no_of_redirect=$(uci get qcmap_lan.@no_of_configs[0].no_of_redirects)
  for i in $(seq $no_of_redirect)
  do
    name=$(uci get firewall.@redirect[$((i-1))].name)
    if [ "$name" = "$profile_handle" ];then
      idx=$((i-1))
      dest_ip=$(uci get firewall.@redirect[$idx].dest_ip)
      uci del firewall.@redirect[$idx]
      count=$((no_of_redirect-1))
      uci set qcmap_lan.@no_of_configs[0].no_of_redirects="$count"
      uci commit
      return
    fi
  done
}

# $1 profile_handle
#Function will delete all firewall rules on profile deletion
function delete_firewall_rule_on_profile_deletion(){
  local profile_handle=$1
  local idx=$(uci get qcmap_lan.@no_of_configs[0].no_of_rules)
  rules_index=`uci show firewall | grep -i "name" | grep -i "rule" | grep -i "$1'$" | awk -F'[][]' '{print $2}'`

  if [ "$rules_index" ];then
    for idx in $rules_index
    do
      firewall_handle=`uci get firewall.@rule[$idx].name | awk -F'-' '{print $1}'`
      if [[ $firewall_handle =~ ^[0-9]+$ ]];then
        delete_firewall $firewall_handle $profile_handle
      else
        uci delete firewall.@rule[$idx]
        idx=$((idx-1))  
      fi
    done
  fi

  dl_rule_index=`uci show firewall | grep -i "wan$1_dl_accept" | awk -F'[][]' '{print $2}'`
  if [ "$dl_rule_index" ]; then
    uci delete firewall.@rule[$dl_rule_index]
    idx=$((idx-1))
  fi
  uci set qcmap_lan.@no_of_configs[0].no_of_rules=$idx
  uci commit
}

# Adds pdn number to mapped vlan
# $1 profile_handle
# $2 vlanID
function add_pdn() {
  #add pdn number to vlan config
  no_of_vlans=$(uci get qcmap_lan.@no_of_configs[0].no_of_vlans)
  echo $(basename "$0")":add_pdn:"$LINENO":$no_of_vlans"
  for i in $(seq $no_of_vlans)
  do
    vlan_id=$(uci get qcmap_lan.@vlan[$((i-1))].vlan_id)
    if [ $vlan_id -eq $2 ]; then
      #create a wwan_profile_num entry
      uci set qcmap_lan.@vlan[$((i-1))].wwan_profile_num=$1
      break
    fi
  done
}

# Deletes pdn number from un_mapped vlan
# $1 profile_handle
# $2 vlanID
function del_pdn() {
  #delete pdn number from vlan config
  no_of_vlans=$(uci get qcmap_lan.@no_of_configs[0].no_of_vlans)
  echo $(basename "$0")":del_pdn:"$LINENO":$no_of_vlans"
  for i in $(seq $no_of_vlans)
  do
    vlan_id=$(uci get qcmap_lan.@vlan[$((i-1))].vlan_id)
    if [ $vlan_id -eq $2 ]; then
      #check if wwan_profile_num entry is the same as passed
      wwan_profile_num=$(uci get qcmap_lan.@vlan[$((i-1))].wwan_profile_num)
      if [ -z "$wwan_profile_num" ]; then
        echo $(basename "$0")":del_pdn:"$LINENO":No entry of wwan_profile_num found. Exiting!"
        return 0
      elif [ $wwan_profile_num -eq $1 ]; then
        #remove wwan_profile_num
        uci delete qcmap_lan.@vlan[$((i-1))].wwan_profile_num
        break
      else
        echo $(basename "$0")":del_pdn:"$LINENO":Invalid combination of vlan to pdn map found. Exiting!"
        return 0
      fi
    fi
  done
}

# Checks if vlan is already mapped to a pdn
# $1 vlanID
# $2 profile_handle
function check_vlan_pdn_map() {
  #Get total number of configured vlan
  no_of_vlans=$(uci get qcmap_lan.@no_of_configs[0].no_of_vlans)
  if [ $no_of_vlans -eq 0 ]; then
    echo $(basename "$0")":check_vlan_pdn_map:"$LINENO":No VLAN's configured"
    return 1
  fi
  #loop through the number of vlans
  for i in $(seq $no_of_vlans)
  do
    #check if vlan_id matches provided input
    vlan_id=$(uci get qcmap_lan.@vlan[$((i-1))].vlan_id)
    if [ $vlan_id -eq $1 ]; then
      #check if wwan_profile_num matches provided input
      prof_id=$(uci get qcmap_lan.@vlan[$((i-1))].wwan_profile_num)
      if [ -z "$prof_id" ]; then
        echo $(basename "$0")":check_vlan_pdn_map:"$LINENO":No profile num found"
        return 0
      else
        echo $(basename "$0")":check_vlan_pdn_map:"$LINENO":profile num: $prof_id found linked to vlan_id: $vlan_id"
        return 1
      fi
    fi
  done
  return 1
}

function check_fw_zone_exists() {
  #Check if passed wan zone already exists in firewall entry
  fw_zone=$(uci get firewall.@zone[$1].network)
  for zone in $fw_zone
  do
    if [ "$zone" == "$2" ]; then
    echo $(basename "$0")":check_fw_zone_exists:"$LINENO":zone: $zone already exists"
    return 1
    fi
  done
  return 0
}

# Adds lan interface to lan_wan zone in firewall
# $1 vlanID
# $2 profile_handle
function add_lan_interface() {
  #Add corresponding forwarding rule in firewall config
  idx=""
  if [ $2 -eq 1 ]; then
    idx=`uci show firewall | grep zone | grep -w "lan_wan" | awk -F'[][]' '{print $2}'`
  else
    idx=`uci show firewall | grep zone | grep -w "lan_wan$2" | awk -F'[][]' '{print $2}'`
  fi
  if [ -z "$idx" ]; then
    echo $(basename "$0")":add_lan_interface:"$LINENO":lan_wan_zone for $2 not found"
    return 1
  else
    lan_interface_list=$(uci get firewall.@zone[$idx].network)
    for lan_interface in $lan_interface_list
    do
      if [ "$lan_interface" == "lan$1" ]; then
        echo $(basename "$0")":add_lan_interface:"$LINENO":lan_interface already exists"
        return
      fi
    done
    uci add_list firewall.@zone[$idx].network=lan$1
  fi
}

# Deletes lan interface from lan_wan zone from firewall
# $1 vlanID
# $2 profile_handle
function del_lan_interface() {
  #Delete corresponding forwarding rule from firewall config
  idx=""
  if [ $2 -eq 1 ]; then
    idx=`uci show firewall | grep zone | grep -w "lan_wan" | awk -F'[][]' '{print $2}'`
  else
    idx=`uci show firewall | grep zone | grep -w "lan_wan$2" | awk -F'[][]' '{print $2}'`
  fi
  if [ -z "$idx" ]; then
    echo $(basename "$0")":del_lan_interface:"$LINENO":lan_wan_zone for $2 not found"
    return 1
  else
    lan_interface_list=$(uci get firewall.@zone[$idx].network)
    for lan_interface in $lan_interface_list
    do
      if [ "$lan_interface" == "lan$1" ]; then
        uci del_list firewall.@zone[$idx].network=lan$1
        return 0
      fi
    done
  fi
}

# Adds vlan_id to corresponding profile config in qcmap_lan when mapping is added
# $1 vlanID
# $2 profile_handle
function add_vlan_to_profile() {
  #Add Vlan ID to profile in qcmap_lan
  profile_count=$(uci get qcmap_lan.@no_of_configs[0].no_of_profiles)
  for i in $(seq $profile_count)
  do
    profile_id=$( uci get qcmap_lan.@profile[$((i-1))].profile_id)
    if [ "$profile_id" == "$2" ]; then
      vlan_id_list=$(uci get qcmap_lan.@profile[$((i-1))].vlan_ids)
      for vlan_id in $vlan_id_list
      do
        if [ "$vlan_id" == "$1" ]; then
          echo $(basename "$0")":add_vlan_to_profile:"$LINENO":vlan_id already exists"
          return
        fi
      done
      uci add_list qcmap_lan.@profile[$((i-1))].vlan_ids=$1
      return 0
    fi
  done
  return 1
}

# Deletes vlan_id from corresponding profile config in qcmap_lan when mapping is deleted
# $1 vlanID
# $2 profile_handle
function del_vlan_from_profile() {
  #Delete vlan_id from profile in qcmap_lan
  profile_count=$(uci get qcmap_lan.@no_of_configs[0].no_of_profiles)
  for i in $(seq $profile_count)
  do
    profile_id=$(uci get qcmap_lan.@profile[$((i-1))].profile_id)
    if [ "$profile_id" == "$2" ]; then
      vlan_id_list=$(uci get qcmap_lan.@profile[$((i-1))].vlan_ids)
      for vlan_id in $vlan_id_list
      do
        if [ "$vlan_id" == "$1" ]; then
          uci del_list qcmap_lan.@profile[$((i-1))].vlan_ids=$1
          return 0
        fi
      done
    fi
  done
  return 1
}

# Usage: map_vlan_to_pdn vlan_id profile_handle
# Adds vlan to pdn mapping and updates required config
# $1 vlanID
# $2 profile_handle
function map_vlan_to_pdn() {
  #Check if given vlan has PDN mapped already
  if check_vlan_pdn_map $1 $2; then
    echo $(basename "$0")":map_vlan_to_pdn:"$LINENO":given vlan_ID does not have any profile_num linked"
  else
    echo $(basename "$0")":map_vlan_to_pdn:"$LINENO":Vlan_ID is already mapped to a PDN"
    return
  fi
  #Get associated vlan interfaces
  vlan_iface=$(uci get network.lan$1.ifname)
  if [ -z "$vlan_iface" ]; then
    echo $(basename "$0")":map_vlan_to_pdn:"$LINENO":No vlan interfaces linked to br-lan$1 found. Exiting!"
    return
  else
    #check if profile_handle is 1
    if [ $2 -eq 1 ]; then
      uci add_list network.wan.downstream=lan$1
      uci add_list network.wan_v6.downstream=lan$1
    else
      uci add_list network.wan$2.downstream=lan$1
      uci add_list network.wan$2_v6.downstream=lan$1
      uci set network.wan$2_v6.ip6table=custom_bind_$2
      #lan$1_v6 is added by dynamic in rmnet script
    fi
    #add_pdn to vlan-config in qcmap_lan
    add_pdn $2 $1
    #Add Vlan ID to profile in qcmap_lan
    if add_vlan_to_profile $1 $2 ; then
      echo $(basename "$0")":map_vlan_to_pdn:"$LINENO":Added vlan_id to profile_section"
    else
      echo $(basename "$0")":map_vlan_to_pdn:"$LINENO":Failed to add vlan_id to profile_section"
      return
    fi
    #Add forwarding rule in firewall
    add_lan_interface $1 $2
    uci commit
    ubus call network reload
    if [ -f /tmp/ipv4config$2 ]; then
      /lib/netifd/rmnet_update.sh "up" $2 1
    fi
    if [ -f /tmp/ipv6config$2 ]; then
      /lib/netifd/rmnet_update.sh "up" $2 2
    fi
    /etc/init.d/firewall reload
  fi
}

# Deletes vlan to pdn mapping and updates required config
# $1 vlanID
# $2 profile_handle
function del_vlan_to_pdn_map() {
  found=0
  #Get total number of vlans configured
  no_of_vlans=$(uci get qcmap_lan.@no_of_configs[0].no_of_vlans)
  if [ "$no_of_vlans" -eq 0 ]; then
    echo $(basename "$0")":del_vlan_to_pdn_map:"$LINENO":No VLAN's configured"
    return
  fi
  #loop through the number of vlans
  echo $(basename "$0")":del_vlan_to_pdn_map:"$LINENO":number of vlans = $no_of_vlans"
  for i in $(seq $no_of_vlans)
  do
    #check if vlan_id matches provided input
    vlan_id=$(uci get qcmap_lan.@vlan[$((i-1))].vlan_id)
    if [ "$vlan_id" == "$1" ]; then
      #check if wwan_profile_num matches provided input
      prof_id=$(uci get qcmap_lan.@vlan[$((i-1))].wwan_profile_num)
      if [ "$prof_id" == "$2" ]; then
        echo $(basename "$0")":del_vlan_to_pdn_map:"$LINENO":Entered vlan ID and profile number match found"
        found=1
        break
      else
        echo $(basename "$0")":del_vlan_to_pdn_map:"$LINENO":Entered vlan ID and profile number mismatch. Please try again!"
        return
      fi
    fi
  done

  if [ $found -eq 1 ]; then
    #remove downstream linked to wan<PDN> zone
    if [ $2 -eq 1 ]; then
      uci del_list network.wan.downstream=lan$1
      uci del_list network.wan_v6.downstream=lan$1
    else
      uci del_list network.wan$2.downstream=lan$1
      uci del_list network.wan$2_v6.downstream=lan$1
      uci delete network.wan$2_v6.ip6table
    fi
    #Delete wwan backhaul profile number from config vlan
    del_pdn $2 $1
    fi
    #Delete vlan_id from profile in qcmap_lan
    if del_vlan_from_profile $1 $2 ; then
      echo $(basename "$0")":del_vlan_to_pdn_map:"$LINENO":Delete vlan_id from profile_section success"
    else
      echo $(basename "$0")":del_vlan_to_pdn_map:"$LINENO":Failed to delete vlan_id from profile_section"
      return
    fi
    #Delete corresponding forwarding rule from firewall config
    if del_lan_interface $1 $2 ; then
      echo $(basename "$0")":del_vlan_to_pdn_map:"$LINENO":del_lan_interface success"
    else
      echo $(basename "$0")":del_vlan_to_pdn_map:"$LINENO":Failed to del_lan_interface "
      return
    fi
    uci commit
    #Perform network/firewall reload for changes to take effect
    ubus call network reload
    if [ -f /tmp/ipv4config$2 ]; then
      /lib/netifd/rmnet_update.sh "up" $2 1
    fi
    if [ -f /tmp/ipv6config$2 ]; then
      /lib/netifd/rmnet_update.sh "up" $2 2
    fi
    /etc/init.d/firewall reload
}

# Adds required configs when wwan profile is created
# $1 profile_handle
function create_wwan_profile() {

local profile=$1
local wan_intf
  if [ $profile -eq 1 ]; then
    wan_intf="wan"
  else
    wan_intf="wan${profile}"
    uci set dhcp.${wan_intf}=dhcp
    uci set dhcp.${wan_intf}.interface="$wan_intf"
    uci set dhcp.${wan_intf}.ignore='1'
  fi

  #checking if profile_id exist previously
  no_of_wan_profiles=$(uci get qcmap_lan.@no_of_configs[0].no_of_profiles)
  for i in $(seq $no_of_wan_profiles)
  do
    profile_id=$(uci get qcmap_lan.@profile[$((i-1))].profile_id)
    if [ $profile_id -eq $profile ];
    then
      log $(basename "$0")":create_wwan_profile:"$LINENO":Profile already exisit "
      exit 1
    fi
  done

  if [ $2 -ne $QCMAP_IP_FAMILY_ETH ];
  then
    uci set network.$wan_intf=interface
    uci set network.$wan_intf.proto=rmnet
    uci set network.$wan_intf.profile=$1
    uci set network.$wan_intf.type='internet'
    uci set network.$wan_intf.bind='1'
    uci set network.$wan_intf.ipv6='0'
    uci set network.$wan_intf.defaultroute='0'

    uci set network.${wan_intf}_v6=interface
    uci set network.${wan_intf}_v6.proto=rmnet
    uci set network.${wan_intf}_v6.profile=$1
    uci set network.${wan_intf}_v6.type='internet'
    uci set network.${wan_intf}_v6.bind='1'
    uci set network.${wan_intf}_v6.ipv6='1'
    uci set network.${wan_intf}_v6.defaultroute='0'
  fi
  #support eth pdu

  if [ $2 -eq $QCMAP_IP_FAMILY_ETH  ];
  then
    uci set network.$wan_intf=interface
    uci set network.$wan_intf.proto=rmneteth
    uci set network.$wan_intf.profile=$1
    uci set network.$wan_intf.type='ethwwan'
  fi


  idx=$(uci get qcmap_lan.@no_of_configs[0].no_of_profiles)
  uci add qcmap_lan profile
  uci set qcmap_lan.@profile[$idx].profile_id=$profile
  uci set qcmap_lan.@profile[$idx].nat_type='SYM'
  uci set qcmap_lan.@profile[$idx].firewall_enabled=0
  uci set qcmap_lan.@profile[$idx].pkts_allowed=0
  uci set qcmap_lan.@profile[$idx].no_of_snat_rules=0
  uci set qcmap_lan.@profile[$idx].no_of_v4_fw_rules=0
  uci set qcmap_lan.@profile[$idx].no_of_v6_fw_rules=0
  uci set qcmap_lan.@profile[$idx].ip_type=$2
  uci set qcmap_lan.@profile[$idx].ipv4_nat_disable=0
  uci set qcmap_lan.@profile[$idx].bh_present='wan'
  uci set qcmap_lan.@profile[$idx].bh_present_v6='wan'

  if [ $2 -ne $QCMAP_IP_FAMILY_ETH  ];
  then
    uci set qcmap_lan.@profile[$idx].nat_type='SYM'
    uci set qcmap_lan.@profile[$idx].firewall_enabled=0
    uci set qcmap_lan.@profile[$idx].pkts_allowed=0
    uci set qcmap_lan.@profile[$idx].no_of_snat_rules=0
  fi

  if [ $2 -ne $QCMAP_IP_FAMILY_ETH  ];
  then
    uci add firewall zone
    uci set firewall.@zone[-1].name=$wan_intf
    uci set firewall.@zone[-1].network=$wan_intf
    uci set firewall.@zone[-1].input='REJECT'
    uci set firewall.@zone[-1].output='ACCEPT'
    uci set firewall.@zone[-1].forward='REJECT'
    uci set firewall.@zone[-1].masq='1'

    #whenever a new wan profile is created we will create a corresponding lan zone
    uci add firewall zone
    uci set firewall.@zone[-1].name=lan_${wan_intf}
    uci set firewall.@zone[-1].input='ACCEPT'
    uci set firewall.@zone[-1].output='ACCEPT'
    uci set firewall.@zone[-1].forward='ACCEPT'

    #Create forwarding rule for wan profile
    uci add firewall forwarding
    uci set firewall.@forwarding[-1].src=lan_${wan_intf}
    uci set firewall.@forwarding[-1].dest=${wan_intf}
  fi
  uci set qcmap_lan.@no_of_configs[0].no_of_profiles=$((idx+1))

  uci commit
  ubus call network reload
  /etc/init.d/firewall reload
  /etc/init.d/dnsmasq reload
}

# Delete required configs when wwan profile is deleted
# $1 profile_handle
function delete_wwan_profile() {
  local profile=$1
  local idx=0
  local found=0
  local zone_name lan_zone
  local src dest
  local wan_all_idx
  local no_of_vlans
  local wan_profile
  local default_profile_id=$(uci get qcmap_lan.@no_of_configs[0].default_pdn)
  local wan_intf lan_wan_intf
  local ip_type

  if [ $profile -eq $default_profile_id ]; then
    log $(basename "$0") $LINENO "delete_wwan_profile" "default profile handle passed"
    exit 1
  fi
  #Check if profile exist
  no_of_wan_profiles=$(uci get qcmap_lan.@no_of_configs[0].no_of_profiles)
  for i in $(seq $no_of_wan_profiles)
  do
    profile_id=$(uci get qcmap_lan.@profile[$((i-1))].profile_id)
    ip_type=$(uci get qcmap_lan.@profile[$((i-1))].ip_type)
    if [ $profile_id -eq $profile ]; then
      idx=$((i-1))
      found=1
      break
    fi
  done

  if [ $found -eq 0 ]; then
    log $(basename "$0") $LINENO "delete_wwan_profile" "profile not found in qcmap_lan"
    exit 1
  fi

  if [ $profile -eq 1 ]; then
    wan_intf="wan"
    lan_wan_intf="lan_wan"
  else
    wan_intf="wan${profile}"
    lan_wan_intf="lan_wan${profile}"
    uci delete dhcp.$wan_intf
  fi

  #Delete network config
  uci delete network.$wan_intf
  if [ $ip_type -ne $QCMAP_IP_FAMILY_ETH  ];then
    uci delete network.${wan_intf}_v6
  fi

  uci delete qcmap_lan.@profile[$idx]
  uci set qcmap_lan.@no_of_configs[0].no_of_profiles=$((no_of_wan_profiles-1))
  log $(basename "$0") $LINENO "delete_wwan_profile" "config deleted from network and qcmap_lan"

  #support ETH PDU
  if [ $ip_type -eq $QCMAP_IP_FAMILY_ETH  ];then
    uci commit
    ubus call network reload
    exit 1
  fi

  #Delete firewall wan config.
  idx=0
  found=0
  while true
  do
    zone_name=$(uci -q get firewall.@zone[$idx].name)
    if [ -z "$zone_name" ]; then
      break
    fi
    if [ "$zone_name" = "$wan_intf" ]; then
      found=1
      break
    fi
    idx=$((idx+1))
  done
  if [ $found -eq 1 ]; then
    uci delete firewall.@zone[$idx]
    log $(basename "$0") $LINENO "delete_wwan_profile" "wan zone deleted from firewall"
  fi

  #Delete firewall lan_wanx config.
  idx=0
  found=0
  while true
  do
    lan_zone=$(uci -q get firewall.@zone[$idx].name)
    if [ -z "$zone_name" ]; then
      break
    fi
    if [ "$lan_zone" = "$lan_wan_intf" ]; then
      found=1
      break
    fi
    idx=$((idx+1))
  done
  if [ $found -eq 1 ]; then
    uci delete firewall.@zone[$idx]
    log $(basename "$0") $LINENO  "delete_wwan_profile" "lan_zone delete from firewall"
  fi

  #Delete forwarding rule for wan profile
  idx=0
  found=0
  while true
  do
    src=$(uci -q get firewall.@forwarding[$idx].src)
    dest=$(uci -q get firewall.@forwarding[$idx].dest)
    if [ -z "$src" ] || [ -z "$dest" ]; then
      break
    fi
    if [ "$src" = "$lan_wan_intf" ] && [ "$dest" = "$wan_intf" ]; then
      found=1
      break
    fi
    idx=$((idx+1))
  done
  if [ $found -eq 1 ]; then
    uci delete firewall.@forwarding[$idx]
    log $(basename "$0") $LINENO  "delete_wwan_profile " "forwarding rule delete from firewall"
  fi

  #Delete wan profile from wan_all zone
  wan_all_idx=`uci show firewall | grep -i name | grep -i "wan_all" | awk -F'[][]' '{print $2}'`
  uci del_list firewall.@zone[$wan_all_idx].network=$wan_intf

  #Delete wan from vlan config
  no_of_vlans=$(uci get qcmap_lan.@no_of_configs[0].no_of_vlans)
  for i in $(seq $no_of_vlans)
  do
    wan_profile=$(uci get qcmap_lan.@vlan[$((i-1))].wwan_profile_num)
    if [ $wan_profile -eq $profile ]; then
      #Delete a wwan_profile_num entry from VLAN
      uci delete qcmap_lan.@vlan[$((i-1))].wwan_profile_num
    fi
  done
  uci commit

  delete_snat_rules_on_profile_deletion $profile
  delete_dmz_rule_on_profile_deletion $profile
  delete_firewall_rule_on_profile_deletion $profile
  ubus call network reload
  /etc/init.d/firewall reload
  /etc/init.d/dnsmasq reload
}

#Add wan network iface to wan_all zone when Backhaul is Up in that PDN.
#$1 is WAN Profile Id
function add_wan_iface_on_wan_all_zone(){
  #Getting wan_all zone index
  local wan_profile_id=$1
  local wan_iface wan6_iface
  zone_wan_all_idx=`uci show firewall | grep -i name | grep -i "wan_all" | awk -F'[][]' '{print $2}'`

  if [ $wan_profile_id -eq 1 ]; then
    local profile_index=`mbb_util_get_uci_profile_index $1`
    present_bh=`uci get qcmap_lan.@profile[$profile_index].bh_present`
    if [ "$present_bh" = "wan" ] || [ -z "$present_bh" ];then
      log $(basename "$0") "add_wan_iface_on_wan_all_zone :" $LINENO ": current bh is wwan"
      wan_iface=wan
      wan6_iface=wan_v6
    else
      wan_iface=$present_bh
      wan6_iface=${present_bh}_v6
    fi
  else
    wan_iface=wan$wan_profile_id
    wan6_iface=wan${wan_profile_id}_v6
  fi
  all_iface_wan_listed=`uci get firewall.@zone[$zone_wan_all_idx].network`
  #check if wan V4 interface already there in the list
  found=$(echo $all_iface_wan_listed | grep -w -c $wan_iface)
  if [ $found -eq 0 ]; then
    #IPV4 wan interface not found.
    uci add_list firewall.@zone[$zone_wan_all_idx].network=$wan_iface
  fi
  #check if wan V6 interface already there in the list
  found=$(echo $all_iface_wan_listed | grep -w -c $wan6_iface)
  if [ $found -eq 0 ]; then
    #IPV6 wan interface not found.
    uci add_list firewall.@zone[$zone_wan_all_idx].network=$wan6_iface
  fi

  log $(basename "$0") "add_wan_iface_on_wan_all_zone :" $LINENO ": wan iface is successfully added on list network of wan_all zone"
  uci commit firewall
}

#Deletes a wan network iface from wan_all zone when Backhaul is Down in that PDN.
#$1 is WAN Profile Id
function del_wan_iface_from_wan_all_zone(){
  #Getting wan_all zone index
  local wan_profile_id=$1
  local wan_iface wan6_iface
  zone_wan_all_idx=`uci show firewall | grep -i name | grep -i "wan_all" | awk -F'[][]' '{print $2}'`

  if [ $wan_profile_id -eq 1 ]; then
    local profile_index=`mbb_util_get_uci_profile_index $1`
    present_bh=`uci get qcmap_lan.@profile[$profile_index].bh_present`
    if [ "$present_bh" = "wan" ] || [ -z "$present_bh" ];then
      log $(basename "$0") "del_wan_iface_from_wan_all_zone :" $LINENO ": current bh is wwan"
      wan_iface=wan
      wan6_iface=wan_v6
    else
      wan_iface=${present_bh}
      wan6_iface=${present_bh}_v6
    fi
  else
    wan_iface=wan$wan_profile_id
    wan6_iface=wan${wan_profile_id}_v6
  fi
  all_iface_wan_listed=`uci get firewall.@zone[$zone_wan_all_idx].network`
  #check if wan V4 interface already there in the list
  found=$(echo $all_iface_wan_listed | grep -w -c $wan_iface)
  if [ $found -eq 1 ]; then
    #IPV4 wan interface found.
    uci del_list firewall.@zone[$zone_wan_all_idx].network=$wan_iface
  fi
  #check if wan V6 interface already there in the list
  found=$(echo $all_iface_wan_listed | grep -w -c $wan6_iface)
  if [ $found -eq 1 ]; then
    #IPV6 wan interface found.
    uci del_list firewall.@zone[$zone_wan_all_idx].network=$wan6_iface
  fi

  uci commit firewall
  log $(basename "$0") "del_wan_iface_on_wan_all_zone :" $LINENO ":Deletion of wan iface is Unsuccessful "
}

function update_default_profile() {
  local default_wan_intf=""
  local current_wan_intf=""
  local current_pdn=$1
  local default_pdn=$2
  #set current PDN to default PDN
  if [ $current_pdn -eq 1 ]; then
    default_wan_intf="wan"
  else
    default_wan_intf="wan$1"
  fi

  if [ $default_pdn -eq 1 ]; then
    current_wan_intf="wan"
  else
    current_wan_intf="wan$2"
  fi

  log $(basename "$0") $LINENO "update_default_profile" "$default_wan_intf curr:$current_wan_intf"
  uci set network.${default_wan_intf}.bind=0
  uci set network.${default_wan_intf}_v6.bind=0
  uci set network.${current_wan_intf}.bind=1
  uci set network.${current_wan_intf}_v6.bind=1
  uci set network.${default_wan_intf}.defaultroute=1
  uci set network.${default_wan_intf}_v6.defaultroute=1

  #delete custom bind rule from default PDN.
  uci delete network.${default_wan_intf}_v6.ip6table

  #add custom bind ip6table to new on demand PDN.
  uci set network.${current_wan_intf}_v6.ip6table=custom_bind_$default_pdn

  uci commit network
  #swap the default and current profiles downstream.
  current_downstream_list=$(uci get network.${current_wan_intf}.downstream)
  default_downstream_list=$(uci get network.${default_wan_intf}.downstream)

  #first delete downstream
  uci delete network.${default_wan_intf}.downstream
  uci delete network.${default_wan_intf}_v6.downstream
  uci delete network.${current_wan_intf}.downstream
  uci delete network.${current_wan_intf}_v6.downstream

  if [ -z $current_downstream_list ]; then
    log $(basename "$0") $LINENO "update_default_profile" "No downstream found on previous current profile"
  else
    log $(basename "$0") $LINENO "update_default_profile" "donwstream found on previous current profile"
    #delete downstream before adding it.
    uci commit
    for downstream in $current_downstream_list
    do
        #downstream will attach to default PDN remove custom bind
        uci add_list network.${default_wan_intf}.downstream="$downstream"
        uci add_list network.${default_wan_intf}_v6.downstream="$downstream"
    done
  fi

  if [ -z $default_downstream_list ]; then
    log $(basename "$0") $LINENO "update_default_profile" "No downstream found on previous default profile"
  else
    log $(basename "$0") $LINENO "update_default_profile" "donwstream found on previous default profile"
    uci commit
    for downstream in $default_downstream_list
    do
        uci add_list network.${current_wan_intf}.downstream="$downstream"
        uci add_list network.${current_wan_intf}_v6.downstream="$downstream"
    done
  fi

  uci set qcmap_lan.@no_of_configs[0].default_pdn="$1"
  #Add lan to default PDN.
  lan_zone_idxs=$(uci show firewall | grep -i "zone" | grep -i "lan_wan*" | awk -F'[][]' '{print $2}')
  for idx in $lan_zone_idxs
  do
       lan_zone_name=$(uci get firewall.@zone[$idx].name)
       if [ $lan_zone_name == "lan_${current_wan_intf}" ]; then
            #remove default bridge here as current_pdn is no more default PDN.
            log $(basename "$0") $LINENO "update_default_profile" "removing lan to $lan_zone_name !!!"
            uci del_list firewall.@zone[$idx].network='lan'
       fi
       if [ $lan_zone_name == "lan_${default_wan_intf}" ]; then
            #remove default bridge here as current_pdn is no more default PDN.
            log $(basename "$0") $LINENO "update_default_profile" "adding lan to $lan_zone_name"
            uci add_list firewall.@zone[$idx].network='lan'
       fi
  done
  #Update mwan3.
  uci -q delete mwan3.${current_wan_intf}
  #delete and add default entry
  uci -q delete mwan3.${default_wan_intf}
  uci set mwan3.${default_wan_intf}=interface
  uci set mwan3.${default_wan_intf}.enabled='1'
  uci set mwan3.${default_wan_intf}.family='ipv4'
  uci set mwan3.${default_wan_intf}.reliability='0'
  uci add_list mwan3.${default_wan_intf}.track_ip='8.8.4.4'
  uci add_list mwan3.${default_wan_intf}.track_ip='8.8.8.8'
  uci add_list mwan3.${default_wan_intf}.track_ip='208.67.222.222'
  uci add_list mwan3.${default_wan_intf}.track_ip='208.67.220.220'

  uci -q delete mwan3.${current_wan_intf}_v6
  #delete and add default entry
  uci -q delete mwan3.${default_wan_intf}_v6
  uci set mwan3.${default_wan_intf}_v6=interface
  uci set mwan3.${default_wan_intf}_v6.enabled='1'
  uci set mwan3.${default_wan_intf}_v6.family='ipv6'
  uci set mwan3.${default_wan_intf}_v6.reliability='0'
  uci add_list mwan3.${default_wan_intf}_v6.track_ip='2001:4860:4860::8844'
  uci add_list mwan3.${default_wan_intf}_v6.track_ip='2001:4860:4860::8888'
  uci add_list mwan3.${default_wan_intf}_v6.track_ip='2620:0:ccd::2'
  uci add_list mwan3.${default_wan_intf}_v6.track_ip='2620:0:ccc::2'

  #update member.
  uci set mwan3.m_wan.interface=${default_wan_intf}
  uci set mwan3.m_wan_v6.interface=${default_wan_intf}_v6

  #update qcmap_lan
  default_profile_idx=$(uci show qcmap_lan | grep "profile_id='$default_pdn'" | awk -F'[][]' '{print $2}')
  current_profile_idx=$(uci show qcmap_lan | grep "profile_id='$current_pdn'" | awk -F'[][]' '{print $2}')

  # Swap the profile between current and default in qcmap_lan 
  default_profile_id=$(uci get qcmap_lan.@profile[$default_profile_idx].profile_id)
  current_profile_id=$(uci get qcmap_lan.@profile[$current_profile_idx].profile_id)
  #SWAP the profile idx with default and current porfile in qcmap_lan
  uci set qcmap_lan.@profile[$default_profile_idx].profile_id="$current_profile_id"
  uci set qcmap_lan.@profile[$current_profile_idx].profile_id="$default_profile_id"

  uci commit firewall
  uci commit qcmap_lan
  uci commit network
  uci commit mwan3
  ubus call network reload
  /etc/init.d/firewall reload
  qcmap_reload_mwan3
  log $(basename "$0") $LINENO "update_default_profile" "Updated profile success"
  echo "WWAN profile updated"
}

#Function - To set v4 NAT Conig
#$1 - profile id
#$2 - Enable/Disable
function Setv4NATConfig() {
  local profile_id=$1
  local enable=$2
  local enable_dmz_snat='0'
  local default_pdn profile_idx cfg="wan"

  log $(basename "$0") "Setv4NATConfig" $LINENO "profile_id=$profile_id : enable=$enable"

  if [ -n "$profile_id" -a "$profile_id" -ne 1 ]; then
    cfg=wan$profile_id
  fi

  #Get profile_index
  profile_idx=$(util_get_profile_index $profile_id)

  #Update ipv4_nat_disable option
  uci set qcmap_lan.@profile[$profile_idx].ipv4_nat_disable="$enable"

  if [ "$enable" -eq 1 ]; then
    #Disable masq/DMZ/SNAT when enabling feature
    util_reset_masq_bit $cfg
    util_enable_disable_dmz "$profile_id" "$enable_dmz_snat"
    util_enable_disable_SNAT "$profile_id" "$enable_dmz_snat"
    #Update IPA that feature is eanbled
    ipa xml PublicIpSupport enable
  else
    #Enable masq/DMZ/SNAT while disable feature
    enable_dmz_snat='1'
    util_set_masq_bit $cfg
    util_enable_disable_dmz "$profile_id" "$enable_dmz_snat"
    util_enable_disable_SNAT "$profile_id" "$enable_dmz_snat"
    #Update IPA that feature is disabled
    ipa xml PublicIpSupport disable
  fi

  uci commit qcmap_lan
  uci commit firewall

  reboot
}

case $1 in
    add_map)
      echo $(basename "$0")":"$LINENO":Mapping vlan to pdn"
      map_vlan_to_pdn $2 $3
      ;;
    del_map)
      echo $(basename "$0")":"$LINENO":Deleting vlan to pdn mapping"
      del_vlan_to_pdn_map $2 $3
      ;;
    create_wwan_profile)
      create_wwan_profile $2 $3
      ;;
    delete_wwan_profile)
      delete_wwan_profile $2
      ;;
    add_wan_iface_on_wan_all_zone)
      add_wan_iface_on_wan_all_zone $2
      ;;
    del_wan_iface_from_wan_all_zone)
      del_wan_iface_from_wan_all_zone $2
      ;;
    update_default_profile)
        update_default_profile $2 $3
      ;;
    Setv4NATConfig)
       Setv4NATConfig $2 $3
      ;;
    *)
    echo $(basename "$0")":"$LINENO":Invalid option"
     ;;
esac
