#!/bin/sh
#  Copyright (c) 2023 Qualcomm Technologies, Inc.
#  All Rights Reserved.
#  Confidential and Proprietary - Qualcomm Technologies, Inc.

#  Usage: backhaulCommonConfig.sh set_bh_pref wan1 wan2 wan3 wan4

. /etc/data/mbbUtils.sh

#source common functions
. /etc/data/lanUtils.sh

#source ip collision file
. /etc/data/ip_collision.sh

#source tinyproxy
. /etc/data/tinyproxyConfig.sh

QCMAP_WLAN_MODE_AP_STA=3
QCMAP_WLAN_MODE_AP_AP_STA=5

QCMAP_STA_CONNECTION_DYNAMIC=1
QCMAP_STA_CONNECTION_STATIC=2

IPV6_DHCP_ENABLE_PD_CONFIG=1
IPV6_DHCP_DISABLE_PD_CONFIG=0

if [[ "$#" -le 0 ]]; then
    echo "Usage: backhaulCommonConfig.sh set_bh_pref wan1 wan2 wan3 wan4 wan5"
    echo "       backhaulCommonConfig.sh switch_bh"
    return 1
elif [ \( "$1" == "set_bh_pref" \) -a \( "$#" -ne 6 \) ]; then
    echo "Usage: backhaulCommonConfig.sh set_bh_pref wan1 wan2 wan3 wan4 wan5"
    return 1
elif [ \( "$1" == "switch_bh" \) -a \( "$#" -ne 3 \) ]; then
    echo "Usage: backhaulCommonConfig.sh set_bh_pref"
    return 1
fi

# Get Backhaul WAN type: get_wan_type <arg1>
# $1 : QCMAP backhaul WAN type.
function get_wan_type() {
  local wan_type=$1
  QCMAP_WWAN_BACKHAUL=1
  QCMAP_USB_CRADLE_BACKHAUL=2
  QCMAP_WLAN_BACKHAUL=3
  QCMAP_ETHERNET_BACKHAUL=4
  QCMAP_BT_BACKHAUL=5

  if [ $wan_type -eq $QCMAP_WWAN_BACKHAUL ]; then
    echo "wan"
  elif [ $wan_type -eq $QCMAP_USB_CRADLE_BACKHAUL ]; then
    echo "wanusb"
  elif [ $wan_type -eq $QCMAP_ETHERNET_BACKHAUL ]; then
    echo "waneth"
  elif [ $wan_type -eq $QCMAP_WLAN_BACKHAUL ]; then
    echo "wanwlan"
  elif [ $wan_type -eq $QCMAP_BT_BACKHAUL ]; then
    echo "wanbt"
  fi
}

#Send Signal to QCMAP server
function send_qcmap_signal() {
  local signal=$1
  local qcmap_cm_pid_file=/var/run/data/QCMAP_ConnectionManager.pid

  if [ "$signal" != "" ]; then
    if [ -f $qcmap_cm_pid_file ]; then
      kill $signal `cat $qcmap_cm_pid_file`
    fi
    log $(basename "$0") "send_qcmap_signal" $LINENO "kill $signal `cat $qcmap_cm_pid_file`"
  fi

}

function update_v4_mwan3_default_route() {
  local backhaul_name=$1
  local linkstate=$2
  local defaultroute
  local file=$3


  if [ ! -f "$file" ]; then
    return
  fi

  local default_pdn=$(util_get_default_pdn)
  local default_pdn_index=$(util_get_profile_index $default_pdn)
  local active_ippt=$(uci get qcmap_lan.@profile[$default_pdn_index].active_ippt)
  #can't notify mwan3 if it is IPPT case since mwan3 is stopped
  if [ $active_ippt -eq 1 ] && [ $backhaul_name == "wan" ] ; then
        return
  fi

  . $file

  if [ $linkstate == 0 ]; then

    #delete the ipv4config;
    #if it is rmnet, need check if the IPv4 is still up since IPPT enable case will call rmnet.script down even iface is up
    if [[ $backhaul_name == "wan" ]]; then
      log $(basename "$0") "update_v4_mwan3_default_route" $LINENO " $file will be deleted by lan client"
    else
      log $(basename "$0") "update_v4_mwan3_default_route" $LINENO " delete file $file"
      rm $file
    fi
  fi
}

function update_v6_mwan3_default_route() {
  local backhaul_name=$1
  local linkstate=$2
  local file=$3
  local defaultroute

  if [ ! -f "$file" ]; then
    log $(basename "$0") "update_v6_mwan3_default_route" $LINENO "file $file is not present"
    return
  fi

  local default_pdn=$(util_get_default_pdn)
  local default_pdn_index=$(util_get_profile_index $default_pdn)
  local active_ippt=$(uci get qcmap_lan.@profile[$default_pdn_index].active_ippt)
  #can't notify mwan3 if it is IPPT case since mwan3 is stopped
  if [ -n "$active_ippt" ]; then
    if [ $active_ippt -eq 1 ] && [ $backhaul_name == "wan" ] ; then
        return
    fi
  fi

  . $file

  if [ $linkstate == 0 ]; then
    if [[ $backhaul_name == "wan" ]]; then
      log $(basename "$0") "update_v6_mwan3_default_route" $LINENO " $file will be deleted by lan client"
    else
      log $(basename "$0") "update_v6_mwan3_default_route" $LINENO " delete file $file"
      rm $file
    fi
  fi

}

#Function to check if any BH config file is present
#$1 - IP type
function check_if_config_file_present() {
  local ip_type="$1"
  local isPresent=0
  local backhaullist defaultProfileId file

  backhaullist=$(uci get mwan3.backhaul_pref.use_member)
  defaultProfileId=`uci get qcmap_lan.@no_of_configs[0].default_pdn`

  if [ ! -z "$backhaullist" ]; then
    if [ -n "$ip_type" -a "$ip_type" = "ipv4" ]; then
      #Indicates IPV4 config file
      for backhaulmember in $backhaullist
      do
        backhaul=$(uci get mwan3.$backhaulmember.interface)

        if [ $backhaul == "wan" ]; then
          file=/tmp/ipv4config$defaultProfileId
        else
          file=/tmp/ipv4config.$backhaul
        fi

        if [ -f "$file" ]; then
          log $(basename "$0") ":check_if_config_file_present:" $LINENO ":Config file v4: $file found"
          isPresent=1
          break
        fi
      done
    elif [ -n "$ip_type" -a "$ip_type" = "ipv6" ]; then
      #Indicates IPV6 config file
      for backhaulmember6 in $backhaullist
      do
        backhaul6=$(uci get mwan3.$backhaulmember6.interface)

        if [ $backhaul6 == "wan" ]; then
          file=/tmp/ipv6config$defaultProfileId
        else
          file=/tmp/ipv6config.$backhaul6
        fi

        if [ -f "$file" ]; then
          log $(basename "$0") ":check_if_config_file_present:" $LINENO ":Config file v6: $file found"
          isPresent=1
          break
        fi
      done
    else
      log $(basename "$0") ":check_if_config_file_present:" $LINENO ":Incorrect ip_type passed"
    fi
  fi

  echo $isPresent

}

#Function to delete DNSV4 Information of previous BH and add new DNS Information of current Bh
#$1 - Default Profile ID
function update_dnsv4() {
  local defaultProfileId="$1"
  local linkstate="$2"
  local current_bh previous_bh profile_idx
  local DNS isFilePresent

  log $(basename "$0") ":update_dnsv4:" $LINENO "Updating dnsv4 at Bh Switch"

  #Get current_bh and previous_bh information
  profile_idx=$(util_get_profile_index $defaultProfileId)

  current_bh=$(uci get qcmap_lan.@profile[$profile_idx].bh_present)
  previous_bh=$(uci get qcmap_lan.@profile[$profile_idx].bh_previous)

  #Check if linkstate is 0
  if [ $linkstate -eq 0 ]; then
    #Indicates disconnect BH
    #Check if any config file is present
    #NOTE: if any /tmp/ipv4config file is present, indicates its a part of BHSwitch
    isFilePresent=$(check_if_config_file_present "ipv4")
    if [ -n "$isFilePresent" -a $isFilePresent -eq 0 ]; then
      #Indicates this disconnect BH event is the last of the BH's
      #Delete bh_present and bh_previous entries from qcmap_lan db
      [ -n "$current_bh" ] && uci del qcmap_lan.@profile[$profile_idx].bh_present
      [ -n "$previous_bh" ] && uci del qcmap_lan.@profile[$profile_idx].bh_previous
      uci commit qcmap_lan
      #Check if current_bh is a secondary BH.
      if [ -n "$current_bh" -a "$current_bh" != "wan" -a $linkstate -eq 0 ]; then
        #Indicates secondary backhaul in disconnect stage
        util_del_non_wwan_dnsv4 $current_bh
        #We do not want to execute the next if condition. Hence returning
        return
      fi
    #If isFilePresent is 1, then it indicates the disconnect BH trigger event
    #is part of BH Switch process. We continue with regular processing.
    fi
  fi

  #Delete DNSSERVER information at BH switch
  if [ -n "$previous_bh" ]; then
    if [ "$previous_bh" = "wan" ]; then
      #Indicates rmnet(wwan) backhaul
      util_del_dnsv4 $defaultProfileId
    else
      #indicates non_wwan backhaul
      util_del_non_wwan_dnsv4 $previous_bh
    fi
  else
    #Check if current_bh is not rmnet BH and linkstate is 0(down)
    #NOTE: reason for having this check is to address a corner case where
    #secondary BH is brought up on a fresh boot and is brought down. Since
    #there will be no bh_previous, we rely on bh_present to delete the dnsserver information
    if [ -n "$current_bh" -a "$current_bh" != "wan" -a $linkstate -eq 0 ]; then
      #Indicates secondary backhaul in disconnect stage
      util_del_non_wwan_dnsv4 $current_bh
      #We do not want to execute the next if condition. Hence returning
      return
    fi
  fi

  #Update DNSSERVER information at BH switch
  if [ -n "$current_bh" ]; then
    if [ "$current_bh" = "wan" ]; then
      #Indicates rmnet(wwan) backhaul
      if [ -f /tmp/ipv4config$defaultProfileId ]; then
        #get DNSSERVER information
        . /tmp/ipv4config$defaultProfileId && DNS=$DNSSERVERS

        #Call util function to add dnsv4 information
        [ -n "$DNS" ] && util_add_dnsv4 $defaultProfileId "$DNS"
      fi
    else
      #indicates non_wwan backhaul
      util_add_non_wwan_dnsv4 $current_bh
    fi
  fi

}

#Function to delete DNSV6 Information of previous BH and add new DNS Information of current Bh
#$1 - Default Profile ID
function update_dnsv6() {
  local defaultProfileId="$1"
  local linkstate="$2"
  local current_bh_v6 previous_bh_v6 profile_idx
  local isFilePresent

  log $(basename "$0") ":update_dnsv6:" $LINENO "Updating dnsv6 at Bh Switch"

  #Get current_bh and previous_bh information
  profile_idx=$(util_get_profile_index $defaultProfileId)

  current_bh_v6=$(uci get qcmap_lan.@profile[$profile_idx].bh_present_v6)
  previous_bh_v6=$(uci get qcmap_lan.@profile[$profile_idx].bh_previous_v6)

  #Check if linkstate is 0
  if [ $linkstate -eq 0 ]; then
    #Indicates disconnect BH
    #Check if any config file is present
    #NOTE: if any /tmp/ipv6config file is present, indicates its a part of BHSwitch
    isFilePresent=$(check_if_config_file_present "ipv6")
    if [ -n "$isFilePresent" -a $isFilePresent -eq 0 ]; then
      #Indicates this disconnect BH event is the last of the BH's
      #Delete bh_present and bh_previous entries from qcmap_lan db
      [ -n "$current_bh_v6" ] && uci del qcmap_lan.@profile[$profile_idx].bh_present_v6
      [ -n "$previous_bh_v6" ] && uci del qcmap_lan.@profile[$profile_idx].bh_previous_v6
      uci commit qcmap_lan
      #Check if current_bh is a secondary BH.
      if [ -n "$current_bh_v6" -a "$current_bh_v6" != "wan" -a $linkstate -eq 0 ]; then
        #Indicates secondary backhaul in disconnect stage
        util_del_non_wwan_dnsv6 $current_bh_v6
        #We do not want to execute the next if condition. Hence returning
        return
      fi
    #NOTE:If isFilePresent is 1, then it indicates the disconnect BH trigger event
    #is part of BH Switch process. We continue with regular processing.
    fi
  fi

  #Delete DNSSERVER information at BH switch
  if [ -n "$previous_bh_v6" ]; then
    if [ "$previous_bh_v6" = "wan" ]; then
      #Indicates rmnet(wwan) backhaul
      util_del_dnsv6 $defaultProfileId
    else
      #indicates non_wwan backhaul
      util_del_non_wwan_dnsv6 $previous_bh_v6
    fi
  else
    #Check if current_bh_v6 is not rmnet BH and linkstate is 0(down)
    #NOTE: reason for having this check is to address a corner case where
    #secondary BH is brought up on a fresh boot and is brought down. Since
    #there will be no bh_previous, we rely on bh_present to delete the dnsserver information
    if [ -n "$current_bh_v6" -a "$current_bh_v6" != "wan" -a $linkstate -eq 0 ]; then
      #Indicates secondary backhaul in disconnect stage
      util_del_non_wwan_dnsv6 $current_bh_v6
      #We do not want to execute the next if condition. Hence returning
      return
    fi
  fi

  #Update DNSSERVER information at BH switch
  if [ -n "$current_bh_v6" ]; then
    if [ "$current_bh_v6" = "wan" ]; then
      #Indicates rmnet(wwan) backhaul
      if [ -f /tmp/ipv6config$defaultProfileId ]; then
        #get DNSSERVER6 information
        . /tmp/ipv6config$defaultProfileId && DNS6=$DNSSERVERS6

        #Call util function to add dnsv4 information
        [ -n "$DNS6" ] && util_add_dnsv6 $defaultProfileId "$DNS6"
      fi
    else
      #indicates non_wwan backhaul
      util_add_non_wwan_dnsv6 $current_bh_v6
    fi
  fi

}

#Funtion to cleanup wan_all section from firewall
#config upon Backhaul switch.
#$1 - previous_bh
#$2 - current_bh
function firewall_cleanup() {
  local previous_bh=$1
  local current_bh=$2
  local firewall_change=0
  local wan_all_idx wan_idx found wan_listed wan_all_listed

  log $(basename "$0") "firewall_cleanup" $LINENO "previous_bh:$previous_bh, current_bh:$current_bh"

  #Update wan_all with highest prority BH
  #remove previous BH and add current BH
  wan_all_idx=`uci show firewall | grep -i name | grep -i -w "wan_all" | awk -F'[][]' '{print $2}'`
  wan_idx=`uci show firewall | grep -i name | grep -i -w "wan" | awk -F'[][]' '{print $2}'`

  if [ -n "$previous_bh" ]; then
    util_execute_uci del_list firewall.@zone[$wan_all_idx].network="$previous_bh"
    util_execute_uci del_list firewall.@zone[$wan_all_idx].network="${previous_bh}_v6"
    util_execute_uci del_list firewall.@zone[$wan_idx].network="$previous_bh"
    util_execute_uci del_list firewall.@zone[$wan_idx].network="${previous_bh}_v6"
    util_execute_uci commit firewall
    firewall_change=1
  fi

  if [ -n "$current_bh" ]; then
    wan_all_listed=`util_execute_uci get firewall.@zone[$wan_all_idx].network`
    wan_listed=`util_execute_uci get firewall.@zone[$wan_idx].network`

    found=$(echo $wan_all_listed | grep -w -c $current_bh)
    if [ $found -eq 0 ]; then
      log $(basename "$0") "firewall_cleanup" $LINENO "current_bh:$current_bh not listed to wan_all"
      util_execute_uci add_list firewall.@zone[$wan_all_idx].network="$current_bh"
      util_execute_uci commit firewall
      firewall_change=1
    fi
    found=$(echo $wan_all_listed | grep -w -c ${current_bh}_v6)
    if [ $found -eq 0 ]; then
      log $(basename "$0") "firewall_cleanup" $LINENO "current_bh:${current_bh}_v6 not listed to wan_all"
      util_execute_uci add_list firewall.@zone[$wan_all_idx].network="${current_bh}_v6"
      util_execute_uci commit firewall
      firewall_change=1
    fi

    found=$(echo $wan_listed | grep -w -c $current_bh)
    if [ $found -eq 0 ]; then
      log $(basename "$0") "firewall_cleanup" $LINENO "current_bh:$current_bh not listed to wan"
      util_execute_uci add_list firewall.@zone[$wan_idx].network="$current_bh"
      util_execute_uci commit firewall
      firewall_change=1
    fi
    found=$(echo $wan_listed | grep -w -c ${current_bh}_v6)
    if [ $found -eq 0 ]; then
      log $(basename "$0") "firewall_cleanup" $LINENO "current_bh:${current_bh}_v6 not listed to wan"
      util_execute_uci add_list firewall.@zone[$wan_idx].network="${current_bh}_v6"
      util_execute_uci commit firewall
      firewall_change=1
    fi
  fi

  if [ $firewall_change -eq 1 ]; then
    /etc/init.d/firewall reload
  fi

}

#compare two backhaul priority, echo the one with higher priority
function compare_backhaul_priority()
{
  local bh1=$1
  local bh2=$2
  local backhaullist
  local backhaulmember

  backhaullist=$(util_execute_uci get mwan3.backhaul_pref.use_member)
  for backhaulmember in $backhaullist
  do
    backhaulname=${backhaulmember#*m_}
    if [ "$backhaulname" == "$bh1" ]; then
      echo $bh1
      return
    elif [ "$backhaulname" == "$bh2" ]; then
      echo $bh2
      return
    fi
  done

}

function reload_for_bridge_mode()
{
   local bridge_mode
   local wlan_proto
   local ipv4_current_bh
   local ipv4_higher_bh

   bridge_mode=$(util_execute_uci get qcmap_wlan.@stamodeconfig[0].bridge_mode)
   if [ -z "$bridge_mode" ]; then
     return;
   fi

   if [ $bridge_mode -eq 0 ];then
     return
   fi

   sta_associated_status=$(/etc/data/wlanConfig.sh get_sta_associcated_status)
   if [ "$sta_associated_status" == "0" ]; then
     return
   fi

   log $(basename "$0") "reload_for_bridge_mode" $LINENO "STA is associcated and in bridge mode"

   #use wanwlan.proto to check whether bridge mode is enabled in br-lan
   wlan_proto=$(util_execute_uci get network.wanwlan.proto)

   ipv4_current_bh=$(util_get_bh_present)
   #get the expected high priority backhaul by compared with current and wanwlan
   ipv4_higher_bh=$(compare_backhaul_priority "$ipv4_current_bh" "wanwlan")

   if [ "$wlan_proto" == "static" -a "$ipv4_higher_bh" != "wanwlan" ];then

     #in case wlan_proto is static, it indicate wlan bridge mode is enabled in br-lan
     #but current backhual is not wanwlan, then we need disable ap-sta-brigde mode, to let br-lan have correct setting for backhaul
     log $(basename "$0") "reload_for_bridge_mode" $LINENO "sta-bridge enabled but not higest priority backhaul, disconnect sta-bridge"
     /etc/data/wlanConfig.sh ap_sta_bridge_mode_disconnect

   elif [ "$wlan_proto" == "dhcp" -a "$ipv4_higher_bh" == "wanwlan" ];then

     #in case wlan_proto is dhcp, it indicate wlan bridge mode is not enabled in br-lan
     #and wanwlan should be high priority backhual, then we need enbale ap-sta-brigde mode, to let br-lan have correct setting for backhaul
      log $(basename "$0") "reload_for_bridge_mode" $LINENO "call  ap_sta_bridge_mode_connect since wanwlan is higest priority"
     /etc/data/wlanConfig.sh ap_sta_bridge_mode_connect

   fi
}

function reload_for_tinyproxy()
{
   local tinyproxy_enable_state=$(util_execute_uci get qcmap_lan.@no_of_configs[0].enable_tinyproxy)

   if [ $tinyproxy_enable_state -eq 1 ] ; then
     ipv4_current_bh=$(util_get_bh_present)
     #get the expected high priority backhaul by compared with current and wanwlan
     ipv4_higher_bh=$(compare_backhaul_priority "$ipv4_current_bh" "wan")
     log $(basename "$0") "reload_for_tinyproxy" $LINENO "current bh: $ipv4_current_bh higher BH: $ipv4_higher_bh"

     if [ "$ipv4_higher_bh" == "wan" ] ; then
       stop_tinyproxy
       setup_tinyproxy
       log $(basename "$0") "reload_for_tinyproxy" $LINENO "retart tinyproxy for wan bh"
     elif [ "$ipv4_higher_bh" != "wan" ] ; then
       stop_tinyproxy
       log $(basename "$0") "reload_for_tinyproxy" $LINENO "stop tinyproxy for non-wan bh"
     fi
   fi
}

#Function to do neccessary updates on switch backhaul
#$1 - Backhaul name
#$2 - link_state(setup/teardown) BH
function switch_backhaul() {
  local defaultProfileId
  local current_bh profile_idx previous_bh
  local DNS
  local backhaul_name=$1
  local link_state=$2
  local bridge_mode=$(util_execute_uci get qcmap_wlan.@stamodeconfig[0].bridge_mode)

  log $(basename "$0") ":switch_backhaul:" $LINENO "Enter switch_backhaul"

  defaultProfileId=`util_execute_uci get qcmap_lan.@no_of_configs[0].default_pdn`
  profile_idx=$(util_get_profile_index $defaultProfileId)

  #Get current BH enabled before updating priority BH
  current_bh=$(util_execute_uci get qcmap_lan.@profile[$profile_idx].bh_present)

  update_present_backhaul "$backhaul_name" "$link_state" "ipv4"

  #Get previous BH after updating priority BH
  previous_bh=$(util_execute_uci get qcmap_lan.@profile[$profile_idx].bh_previous)

  #Get present BH after updating priority BH
  new_bh=$(util_execute_uci get qcmap_lan.@profile[$profile_idx].bh_present)
  log $(basename "$0") ":switch_backhaul:" $LINENO "previous_bh=$current_bh:new_bh=$new_bh"

  #Cleanup firewall upon BH switch in case if DL path keep only current BH i.e, highest priority BH
  firewall_cleanup "$current_bh" "$new_bh"

  #Condition to avoid BHSwitch loop during IPPT reconfiguration
  if [ -z "$new_bh" -a "$current_bh" = "wan" -a "$previous_bh" = "wan" ]; then
    #We will enter this condition when:
    #*. WWAN BH is brought up on a non-ippt mode
    #*. During the process of configuring the above rmnet_update system call, user triggers IPPT reconfiguration.
    #*. This leads to a loop where PREVIOUS_BH will be updated to WAN and NEW_BH will be
    #   updated to NULL, while CURRENT_BH was WAN before calling UPDATE_PRESENT_BACKHAUL()
    #*. In this situation, we need to avoid calling IPPT_UPDATE_ATBHSWITCH()
    log $(basename "$0") ":switch_backhaul:" $LINENO "current_bh:$current_bh;previous_bh:$previous_bh;new_bh:$new_bh"

  elif [ -n "$current_bh" -a "$current_bh" != "$new_bh" ]; then

    #IPPT setting need to be updated only when wan(rmnet) BH is up
    #current_bh=wan - Indicates other priority BH is brought up, call rmnet.script to disable IPPT.
    #new_bh=wan - Indicates other priority BH is brought down, call rmnet.script to enable IPPT
    if [ "$current_bh" = "wan" -o "$new_bh" = "wan" ]; then
      log $(basename "$0") ":switch_backhaul:" $LINENO "Update IPPT as wan is up"
      /etc/data/ippt.sh ippt_update_atBHSwitch
    fi

    if [ -n "$new_bh" ]; then
      /etc/data/nat_alg_vpn_config.sh ENABLE_NAT -1

      #Need to stop firewall for bridge mode
      if [ $bridge_mode -eq 1 -a "$new_bh" == "wanwlan" ]; then
        /etc/data/firewallConfig.sh disable_firewall $defaultProfileId 1
        log $(basename "$0") ":switch_backhaul:" $LINENO "stop firewall since bridge mode is enabled"
      else
        /etc/data/firewallConfig.sh enable_firewall $defaultProfileId 1
      fi
    fi
  fi

  #Check ip_collision.sh file is present
  if [ -f /etc/data/ip_collision.sh ]; then
    reload_ip_collision_as_need $defaultProfileId $new_bh
  fi

  reload_for_bridge_mode
  update_dhcp_relay_for_bridge_mode
  reload_for_tinyproxy

  #Update DNSSERVERS information at BH switch
  update_dnsv4 $defaultProfileId $2

}

function update_dhcp_pd_config()
{
  local defaultProfileId
  local profile_idx
  local current_bh_v6

  defaultProfileId=`uci get qcmap_lan.@no_of_configs[0].default_pdn`
  profile_idx=$(util_get_profile_index $defaultProfileId)
  current_bh_v6=$(uci get qcmap_lan.@profile[$profile_idx].bh_present_v6)
  log $(basename "$0") ":update_dhcp_pd_config:" $LINENO "current_bh_v6=$current_bh_v6"

  #If current_bh_v6 is wan and qcmap_lan pd_manager enabled, need add dhcp pd config
  #If current_bh_v6 is not wan or wan backhaul down, need delete dhcp pd config
  if [ -n "$current_bh_v6" -a "$current_bh_v6" == "wan" ]; then
    /etc/data/dhcp_reload_pd_option.sh dhcp_enable_pd_config $IPV6_DHCP_ENABLE_PD_CONFIG
    log $(basename "$0") ":update_dhcp_pd_config:" $LINENO "/etc/data/dhcp_reload_pd_option.sh dhcp_enable_pd_config $IPV6_DHCP_ENABLE_PD_CONFIG"
  elif [ -z "$current_bh_v6" ] || [ "$current_bh_v6" != "wan" ]; then
    /etc/data/dhcp_reload_pd_option.sh dhcp_enable_pd_config $IPV6_DHCP_DISABLE_PD_CONFIG
    log $(basename "$0") ":update_dhcp_pd_config:" $LINENO "/etc/data/dhcp_reload_pd_option.sh dhcp_enable_pd_config $IPV6_DHCP_DISABLE_PD_CONFIG"
  fi
}

#Function to do neccessary updates on switch backhaul
#$1 - Backhaul name
#$2 - link_state(setup/teardown) BH
function switch_backhaul_v6() {
  local defaultProfileId
  local current_bh_v6 profile_idx previous_bh_v6
  local backhaul_name=$1
  local link_state=$2
  local bridge_mode=$(util_execute_uci get qcmap_wlan.@stamodeconfig[0].bridge_mode)

  log $(basename "$0") ":switch_backhaul_v6:" $LINENO "Enter switch_backhaul_v6"

  defaultProfileId=`util_execute_uci get qcmap_lan.@no_of_configs[0].default_pdn`
  profile_idx=$(util_get_profile_index $defaultProfileId)

  #Get previous BH enabled before updating priority BH
  previous_bh_v6=$(util_execute_uci get qcmap_lan.@profile[$profile_idx].bh_present_v6)

  update_present_backhaul_v6 "$backhaul_name" "$link_state" "ipv6"

  current_bh_v6=$(util_execute_uci get qcmap_lan.@profile[$profile_idx].bh_present_v6)

  #Cleanup firewall upon BH switch in case if DL path keep only current BH i.e, highest priority BH
  firewall_cleanup "$previous_bh_v6" "$current_bh_v6"

  if [ -n "$current_bh_v6" ]; then
    #Need to stop firewall for bridge mode
    if [ $bridge_mode -eq 1 -a "$current_bh_v6" == "wanwlan" ]; then
      /etc/data/firewallConfig.sh disable_firewall $defaultProfileId 2
      log $(basename "$0") ":switch_backhaul_v6:" $LINENO "stop firewall since bridge mode is enabled"
    else
      /etc/data/firewallConfig.sh enable_firewall $defaultProfileId 2
    fi
  fi

  reload_for_bridge_mode
  update_dhcp_relay_for_bridge_mode
  reload_for_tinyproxy

  #Update DNSSERVERS6 information at BH switch
  update_dnsv6 $defaultProfileId $2

  #Update pd_manager and qcmap_prefix_delegation_mode in dhcp
  if [ -f /etc/data/dhcp_reload_pd_option.sh ]; then
    update_dhcp_pd_config
  fi
}

function get_backhaul_file()
{
   local iptype=$1
   local backhaul=$2
   defaultProfileId=`uci get qcmap_lan.@no_of_configs[0].default_pdn`

   bridge_mode=$(util_execute_uci -q get qcmap_wlan.@stamodeconfig[0].bridge_mode)

   if [[ $backhaul =~ ^[wan0-9]+$ ]] ; then
     file="/tmp/"$iptype"config"$defaultProfileId
   else
     if [ $bridge_mode -eq 1 -a "$backhaul" == "wanwlan" ];then
       #sta bridge mode, tmp file is /tmp/ipv4config.lan
       file="/tmp/"$iptype"config.lan"
     else
       file="/tmp/"$iptype"config."$backhaul
     fi
   fi
}

#check whether wwan will be prefer backhaul
#  update_present_backhaul: will be called to update prefered backhaul and also default route in UCI(bh_present) when new backhaul active,
#      it has to be called when the inteface is updated done by netifd.
#  is_wwan_prefer_backhaul: can be used when UCI(bh_present) not ready.
function is_wwan_prefer_backhaul()
{
  local iptype="ipv4"
  local isWWANAPrefered="0"

  backhaullist=$(uci get mwan3.backhaul_pref.use_member)

  for backhaulmember in $backhaullist
  do
    backhaul=$(uci get mwan3.$backhaulmember.interface)
    get_backhaul_file "$iptype" "$backhaul"

    if [ ! -f "$file" ]; then
      log $(basename "$0") "is_wwan_prefer_backhaul" $LINENO "Backhaul not up since $file not present"
      continue
    fi
    log $(basename "$0") "is_wwan_prefer_backhaul" $LINENO "file present: $file"

    . $file

    checkIP=`ip addr show $IFNAME | grep inet`

    if [ ! -n "$checkIP" ]; then
      log $(basename "$0") "is_wwan_prefer_backhaul" $LINENO "Backhaul not up since addr $PUBLIC_IP not valid"
      continue
    fi

    log $(basename "$0") "is_wwan_prefer_backhaul" $LINENO "$backhaul is up with devname=$IFNAME, addr=$PUBLIC_IP"
    if [[ $backhaul =~ ^[wan0-9]+$ ]] ; then
      log $(basename "$0") "is_wwan_prefer_backhaul" $LINENO "wwan is prefered active backhaul"
      isWWANAPrefered="1"
      break
    else
      log $(basename "$0") "is_wwan_prefer_backhaul" $LINENO "non-wwan is not prefered active backhaul"
      isWWANAPrefered="0"
      break
    fi

  done

  echo "$isWWANAPrefered"
}

##reconfigure dhcpv6 relay setting
function update_dhcpv6_relay_mode()
{
  local new_backhaul=$1
  local need_reload_dhcp=0
  local need_set_dhcp_server_mode=0
  local ipv6_bh
  ipv6_bh=$(util_execute_uci -q get qcmap_lan.@profile[$profile_idx].bh_present_v6)
  local bridge_mode=$(util_execute_uci get qcmap_wlan.@stamodeconfig[0].bridge_mode)

  if [ -z "$bridge_mode" ]; then
    bridge_mode=0;
  fi

  if [ -z "$ipv6_bh" ];then
    dhcp_lan_ndp=$(util_execute_uci get dhcp.lan.ndp)
    if [ -z "$dhcp_lan_ndp" ]; then
      log $(basename "$0") "update_dhcpv6_relay_mode" $LINENO "dhcpv6 relay was not set and no need set(ipv6 backhual not up), break"
      return
    fi
    #IPv6 not ready on current backhaul but was in other backhaul, handle it as no IPv6 backhaul
    new_backhaul="no"
  fi

  backhaullist=$(uci get mwan3.backhaul_pref.use_member)
  for backhaulmember in $backhaullist
  do
    backhaulname=${backhaulmember#*m_}
    ismaster=$(util_execute_uci -q get dhcp.${backhaulname}_v6.master)
    if [ -z "$ismaster" ];then
      ismaster=0
    fi

    if [ "$ismaster" == "0" ];then
      if [ "$backhaulname" == "$new_backhaul" ]; then
        if [ "$backhaulname" == "wan" ]; then
          log $(basename "$0") "update_dhcpv6_relay_mode" $LINENO "dhcpv6 relay no need set since wwan is default backhaul"
          #set dhcp.lan as server mode later,
          #this is to allow odhcpd know which is slave interface to send deprecated RA.
          need_set_dhcp_server_mode=1
          need_reload_dhcp=1
        else
          if [ $bridge_mode -eq 0 ]; then
            log $(basename "$0") "update_dhcpv6_relay_mode" $LINENO "dhcpv6 master not on default backhaul $backhaulname: set it"
            util_execute_uci_set dhcp.${backhaulname}_v6.dhcpv6 "relay"
            util_execute_uci_set dhcp.${backhaulname}_v6.ra     "relay"
            util_execute_uci_set dhcp.${backhaulname}_v6.ndp    "relay"
            util_execute_uci_set dhcp.${backhaulname}_v6.master "1"
            util_execute_uci_set dhcp.lan.dhcpv6 "relay"
            util_execute_uci_set dhcp.lan.ra     "relay"
            util_execute_uci_set dhcp.lan.ndp    "relay"
            need_reload_dhcp=1
          elif [ $bridge_mode -eq 1 ]; then
            log $(basename "$0") "update_dhcpv6_relay_mode" $LINENO "STA bridge mode: handle in dhcpRelay.sh"
          fi
        fi
      fi
    else
      if [ "$backhaulname" != "$new_backhaul" ]; then
        log $(basename "$0") "update_dhcpv6_relay_mode" $LINENO "dhcpv6 master was on $backhaulname but default backhaul on $new_backhaul: delete it"
        util_execute_uci -q delete dhcp.${backhaulname}_v6.master
        util_execute_uci -q delete dhcp.${backhaulname}_v6.ra
        util_execute_uci -q delete dhcp.${backhaulname}_v6.ndp
        util_execute_uci -q delete dhcp.${backhaulname}_v6.dhcpv6
        #remove the master firstly and trigger reload, then odhcpd able to send deprecated RA for deprecated wan interface.
        util_execute_uci commit dhcp
        /etc/init.d/odhcpd reload
        #wait odhcpd finish send deprecated RA.
        sleep 0.5

        if [ "$new_backhaul" == "no" ]; then
          util_execute_uci_set    dhcp.lan.dhcpv6 "server"
          util_execute_uci_set    dhcp.lan.ra     "server"
        fi
        need_reload_dhcp=1
      else
        log $(basename "$0") "update_dhcpv6_relay_mode" $LINENO "dhcpv6 master already on $backhaulname: no action"
      fi
    fi
  done

  if [ $need_set_dhcp_server_mode -eq 1 ]; then
     util_execute_uci_set    dhcp.lan.dhcpv6 "server"
     util_execute_uci_set    dhcp.lan.ra     "server"
     util_execute_uci delete dhcp.lan.ndp
     util_execute_uci commit dhcp
  fi

  if [ $need_reload_dhcp -eq 1  ];then
     /etc/init.d/odhcpd reload
  fi
}

#we have seperated backhaul with IPv4 IPv6,
#then it is possbile highest backhaul have only IP type active, such as IPv4, but other backhaul may IPv6, or vise versa
#then IPv4 go one backhaul, IPv6 go another,
#to avoid this, we disable the defaultroute of IPv6 to disable the data path, and also set bh_present value
function combine_backhaul_ipv4ipv6()
{
  local profile_idx=$1
  local previous_backhaul
  local new_backhaul=""
  ipv4_bh=$(util_execute_uci get qcmap_lan.@profile[$profile_idx].bh_present)
  ipv6_bh=$(util_execute_uci get qcmap_lan.@profile[$profile_idx].bh_present_v6)

  #if no value, set it to invalid value to easy handle.
  if [ ! -n "$ipv4_bh" ];then
    ipv4_bh="no"
  fi
  if [ ! -n "$ipv6_bh" ];then
    ipv6_bh="no"
  fi

  backhaullist=$(uci get mwan3.backhaul_pref.use_member)
  for backhaulmember in $backhaullist
  do
    backhaulname=${backhaulmember#*m_}
    defaultroute_value=$(util_execute_uci get network.${backhaulname}.defaultroute)
    if [ $defaultroute_value -eq 1 ]; then
      previous_backhaul=$backhaulname
    fi
    util_execute_uci_set network.${backhaulname}_v6.defaultroute "0"
    util_execute_uci_set network.${backhaulname}.defaultroute "0"

    #temp workaround for ipv6: switch_bh_v6 is not called from netifd/odhcp6c when ipv6 interface down;
    #delete the file and don't consider the backhaul as current backhaul
    get_backhaul_file "ipv6" $backhaulname
    if [ "$backhaulname" != "wan" -a -f $file ]; then
      . $file
      checkIP=$(ip -6 addr show $IFNAME | grep $PUBLIC_IP6)
      if [ ! -n "$checkIP" ]; then
        log $(basename "$0") "combine_backhaul_ipv4ipv6" $LINENO "${backhaulname}_v6 not up since IPv6 addr not preset"
        rm $file
        if [ "$backhaulname" == "$ipv6_bh" ];then
           ipv6_bh="no"
        fi
      fi
    fi

  done

  for backhaulmember in $backhaullist
  do
    backhaulname=${backhaulmember#*m_}

    log $(basename "$0") "combine_backhaul_ipv4ipv6" $LINENO "ipv4_bh=$ipv4_bh, ipv6_bh=$ipv6_bh, loop on $backhaulname"

    if [ "$backhaulname" == "$ipv4_bh" -o "$backhaulname" == "$ipv6_bh" ]; then
      #either Ipv4 or IPv6 is up;
      log $(basename "$0") "combine_backhaul_ipv4ipv6" $LINENO "$backhaulname is highest priority"

      #logic to ensure delete route happen before add route
      if [ -n "$previous_backhaul" ]; then
        if [ "$previous_backhaul" != "$backhaulname" ]; then
          #default route change from lowest priority to high priority backhaul
          #trigger network reload for delete-route and wait 200ms, this is to ensure delete route happen firstly;
          util_execute_uci commit network
          ubus call network reload
          log $(basename "$0") "combine_backhaul_ipv4ipv6" $LINENO "wait 200ms to allow netifd delete default route of $previous_backhaul"
          sleep 0.2
        fi
      fi

      util_execute_uci_set network.${backhaulname}_v6.defaultroute "1"
      util_execute_uci_set network.${backhaulname}.defaultroute    "1"
      new_backhaul=$backhaulname

      if [ "$backhaulname" != "$ipv6_bh" ]; then
        get_backhaul_file "ipv6" $backhaulname
        if [ -f $file ]; then
          #IPv6 present
          util_execute_uci_set qcmap_lan.@profile[$profile_idx].bh_present_v6 "$backhaulname"
        else
          #IPv6 is not on this highest prioity backhaul, disable default route for IPv6 on the low priority backhaul
          util_execute_uci_set qcmap_lan.@profile[$profile_idx].bh_present_v6 ""
        fi
      fi

      if [ "$backhaulname" != "$ipv4_bh" ]; then
        get_backhaul_file "ipv4" $backhaulname
        if [ -f $file ]; then
          #IPv4 present
          util_execute_uci_set qcmap_lan.@profile[$profile_idx].bh_present "$backhaulname"
        else
          #IPv4 is not on this highest prioity backhaul, disable  default route for IPv4 on the low priority backhaul
          util_execute_uci_set qcmap_lan.@profile[$profile_idx].bh_present ""
        fi
      fi
      break
    fi

    if [ "$backhaulname" == "$previous_backhaul" ]; then
      #go to here means the previous_backhaul priority is high than new highest active backhaul,
      #then, it indicate the previous_backhaul is down;
      #then, the default route will be deleted automatic when iface down, we don't need add sleep to wait route delete
      log $(basename "$0") "combine_backhaul_ipv4ipv6" $LINENO "$previous_backhaul is down, no need sleep/wait for delete default route for $previous_backhaul"
      previous_backhaul="";
    fi

  done

  util_execute_uci commit qcmap_lan
  util_execute_uci commit network
  ubus call network reload

  update_dhcpv6_relay_mode "$new_backhaul"
}

#update preset backhaul and set flag according to priority
function update_present_backhaul() {
  local checkIP
  local current_backhaul
  local new_backhaul
  local new_backhaul_file
  local backhaulname=$1
  local linkstate=$2
  local iptype=$3

  log $(basename "$0") "update_present_backhaul" $LINENO "Enter update_present_backhaul"

  backhaullist=$(uci get mwan3.backhaul_pref.use_member)
  defaultProfileId=`uci get qcmap_lan.@no_of_configs[0].default_pdn`

  profile_idx=$(util_get_default_profile_index)
  log $(basename "$0") ":update_present_backhaul:" $LINENO ":profile_idx: $profile_idx"

  if [[ "$backhaulname" =~ ^rmnet.* ]]; then
    input_profile=${backhaulname#rmnet}
    if [ $input_profile != $defaultProfileId ]; then
      log $(basename "$0") "update_present_backhaul" $LINENO " no need backhaul-switch for ondemaond PDP $input_profile !"
      return
    else
      backhaulname="wan"
    fi
  fi

  #remove default route when link down
  if [ "$linkstate" == "0" ]; then
    get_backhaul_file $iptype $backhaulname
    update_v4_mwan3_default_route $backhaulname $linkstate $file
  fi

  #initialize the bh_present be empty.
  util_execute_uci_set qcmap_lan.@profile[$profile_idx].bh_present ""

  #run loop to get bh_present
  if [ ! -z "$backhaullist" ]; then
    #IPv4 backhaul
    current_backhaul=$(uci get qcmap_lan.@profile[$profile_idx].bh_present)
    log $(basename "$0") "update_present_backhaul" $LINENO "current_IPv4_backhaul:$current_backhaul"

    for backhaulmember in $backhaullist
    do
      backhaul=$(uci get mwan3.$backhaulmember.interface)
      if [ "$backhaul" == "$backhaulname" ] && [ "$linkstate" == "0" ] ; then
        #the scenario happen when IPPT case, rmnet.script down is called but the ipv4config still need be present
        log $(basename "$0") "update_present_backhaul" $LINENO " ignore $backhaul since already handled"
        continue
      fi
      get_backhaul_file $iptype $backhaul

      if [ ! -f "$file" ]; then
        log $(basename "$0") "update_present_backhaul" $LINENO ":Backhaul not up since $file not present"
        continue
      fi
      log $(basename "$0") "update_present_backhaul" $LINENO ":file present: $file"

      . $file

      #not use PUBLIC_IP check since when IPPT case rmnet don't have publicIP
      checkIP=`ip addr show $IFNAME | grep inet`

      if [ ! -n "$checkIP" ]; then
        log $(basename "$0") "update_present_backhaul" $LINENO ":Backhaul not up since addr $PUBLIC_IP not valid"
        continue
      fi

      log $(basename "$0") "update_present_backhaul" $LINENO ":$backhaul is up with devname=$IFNAME, addr=$PUBLIC_IP"

      if [ -z "$new_backhaul" ] ; then
        new_backhaul=$backhaul
        new_backhaul_file=$file

        if [ -n "$current_backhaul" -a "$current_backhaul" != "$backhaul" ]; then
          util_execute_uci_set qcmap_lan.@profile[$profile_idx].bh_previous "$current_backhaul"
        fi
        util_execute_uci_set qcmap_lan.@profile[$profile_idx].bh_present "$backhaul"
        util_execute_uci commit qcmap_lan
      fi
    done

      if [ -f $new_backhaul_file ]; then
        unset IPV4MTU
        . $new_backhaul_file
        if [ ! -z $IPV4MTU ] && [ -n $IPV4MTU ]; then
          util_add_mtu_option_in_firewall "4" "$IFNAME" "$IPV4MTU"
        fi
      fi

    combine_backhaul_ipv4ipv6 $profile_idx

    if [ -z $current_backhaul ] || [ "$current_backhaul" != "$new_backhaul" ]; then
      #send sig to QCMAP for sending backhaul status indication
      send_qcmap_signal -SIGUSR2
    fi
  else
    log $(basename "$0")":update_present_backhaul:"$LINENO":get backhaul_pref.use_member failed:$backhaullist"
  fi

}

function update_present_backhaul_v6() {
  local checkIP
  local current_backhaul6
  local new_backhaul6
  local new_backhaul6_file
  local backhaulname=$1
  local linkstate=$2
  local iptype=$3

  log $(basename "$0") "update_present_backhaul_v6" $LINENO "Enter update_present_backhaul_v6"

  backhaullist=$(uci get mwan3.backhaul_pref.use_member)
  defaultProfileId=`uci get qcmap_lan.@no_of_configs[0].default_pdn`

  profile_idx=$(util_get_profile_index $defaultProfileId)
  log $(basename "$0") "update_present_backhaul_v6" $LINENO ":profile_idx: $profile_idx"

  if [[ "$backhaulname" =~ ^rmnet.* ]]; then
    input_profile=${backhaul#rmnet}
    if [ $input_profile != $defaultProfileId ]; then
      log $(basename "$0") "update_present_backhaul_v6" $LINENO " no need backhaul-switch for ondemaond PDP $input_profile !"
      return
    else
      backhaulname="wan"
    fi
  fi

  #remove default route when link down
  if [ "$linkstate" == "0" ]; then
    get_backhaul_file $iptype $backhaulname
    update_v6_mwan3_default_route $backhaulname $linkstate $file
  fi

  #initialize the bh_present be empty.
  util_execute_uci_set qcmap_lan.@profile[$profile_idx].bh_present_v6 ""

  #run loop to get bh_present
  if [ ! -z "$backhaullist" ]; then
    #IPv6 backhaul
    current_backhaul6=$(uci get qcmap_lan.@profile[$profile_idx].bh_present_v6)
    log $(basename "$0") "update_present_backhaul_v6" $LINENO "current_IPv6_backhaul:$current_backhaul6"


    for backhaulmember6 in $backhaullist
    do
      backhaul6=$(uci get mwan3.$backhaulmember6.interface)
      get_backhaul_file $iptype $backhaul6
      if [ ! -f "$file" ]; then
        log $(basename "$0") "update_present_backhaul_v6"  $LINENO ":Backhaul not up since $file not present"
        continue
      fi
      log $(basename "$0") "update_present_backhaul_v6" $LINENO ":file present: $file"

      . $file

      checkIP=`ip -6 addr show $IFNAME | grep $PUBLIC_IP6`

      if [ ! -n "$checkIP" ]; then
        log $(basename "$0") "update_present_backhaul_v6" $LINENO ":Backhaul not up since addr $PUBLIC_IP6 not valid"
        continue
      fi

      log $(basename "$0") "update_present_backhaul_v6" $LINENO ":$backhaul6 v6 is up with devname=$IFNAME, addr=$PUBLIC_IP6"

      if [ -z "$new_backhaul6" ] ; then
        new_backhaul6=$backhaul6

        if [ -n "$current_backhaul6" -a "$current_backhaul6" != "$backhaul6" ]; then
          util_execute_uci_set qcmap_lan.@profile[$profile_idx].bh_previous_v6 "$current_backhaul6"
        fi
        util_execute_uci_set qcmap_lan.@profile[$profile_idx].bh_present_v6 "$backhaul6"
        util_execute_uci commit qcmap_lan
      fi
    done

      if [ -f $new_backhaul6_file ]; then
        unset IPV6MTU
        . $new_backhaul6_file
        if [ ! -z $IPV6MTU ] && [ -n $IPV6MTU ]; then
          util_add_mtu_option_in_firewall "6" "$IFNAME" "$IPV6MTU"
        fi
      fi
	combine_backhaul_ipv4ipv6 $profile_idx

    if [ -z $current_backhaul6 ] || [ "$current_backhaul6" != "$new_backhaul6" ]; then
      #send sig to QCMAP for sending backhaul status indication
      send_qcmap_signal -SIGUSR2
    fi

  else
    log $(basename "$0") "update_present_backhaul_v6" $LINENO ":get backhaul_pref.use_member failed:$backhaullist"
  fi
}

#Get preset backhaul and set flag according to priority
function read_current_backhaul() {
  defaultProfileId=$(util_execute_uci get qcmap_lan.@no_of_configs[0].default_pdn)
  no_of_profiles=$(util_execute_uci get qcmap_lan.@no_of_configs[0].no_of_profiles)

  for i in $(seq 0 $((no_of_profiles-1)))
  do
    profile_id=$(util_execute_uci get qcmap_lan.@profile[$i].profile_id)

    #check if profile_id is same as profile
    if [ $profile_id -eq $defaultProfileId ]; then
      profile_idx=$i
      break
    fi
  done

  ipv4_backhaul=$(util_execute_uci get qcmap_lan.@profile[$profile_idx].bh_present)
  ipv6_backhaul=$(util_execute_uci get qcmap_lan.@profile[$profile_idx].bh_present_v6)

  echo "$ipv4_backhaul $ipv6_backhaul"
}


#enable or disable dhcp relay for ap sta bridge mode
function update_dhcp_relay_for_bridge_mode() {

  local wlan_mode=$(util_execute_uci get qcmap_wlan.@wlanconfig[0].mode)
  local bridge_mode=$(util_execute_uci get qcmap_wlan.@stamodeconfig[0].bridge_mode)
  local sta_mode=$(util_execute_uci get qcmap_wlan.@stamodeconfig[0].sta_mode_conn_type)
  local ipv4_bh=$(util_get_bh_present)

  if [ $sta_mode == $QCMAP_STA_CONNECTION_DYNAMIC ] ; then

    file=/tmp/ipv4config.lan
    if [ -f "$file" ] && [ $wlan_mode -eq $QCMAP_WLAN_MODE_AP_STA -o $wlan_mode -eq $QCMAP_WLAN_MODE_AP_AP_STA ] && [ $bridge_mode == 1 ] && [ "$ipv4_bh" == "wanwlan" ]; then
      /etc/data/dhcpRelay.sh enable_dhcp_proxy lan
    else
      /etc/data/dhcpRelay.sh disable_dhcp_proxy
    fi
  fi
}

#Set backhaul prefrence in mwan3 config.
#set_backhaul_pref <wan_pref1> <wan_pref2> ...
#$1 : first BH pref
#$2 : second BH pref
#$3 : third BH pref
#$4 : fourth BH pref
function set_backhaul_pref() {
  local wan1=$1
  local wan2=$2
  local wan3=$3
  local wan4=$4
  local wan5=$5

  w1=$(get_wan_type $wan1)
  w2=$(get_wan_type $wan2)
  w3=$(get_wan_type $wan3)
  w4=$(get_wan_type $wan4)
  w5=$(get_wan_type $wan5)

  uci delete mwan3.backhaul_pref 2>/dev/null
  uci delete mwan3.backhaul_pref6 2> /dev/null

  uci set mwan3.backhaul_pref=policy
  uci set mwan3.backhaul_pref6=policy

  i=1
  for wan_type in $w1 $w2 $w3 $w4 $w5
    do
      uci set mwan3.m_$wan_type.metric="$i"
      uci set mwan3.m_$wan_type.weight="$((i+1))"

      wan6_type=$wan_type"_v6"
      uci set mwan3.m_$wan6_type.metric="$i"
      uci set mwan3.m_$wan6_type.weight="$((i+1))"


      #create policy for backhaul pref
      uci add_list mwan3.backhaul_pref.use_member="m_$wan_type"
      uci add_list mwan3.backhaul_pref6.use_member="m_$wan6_type"

      i=$((i+1))
    done
  log $(basename "$0") "set_backhaul_pref" $LINENO ": set_backhaul_pref success "
  echo "set_backhaul_pref success"

  uci commit mwan3
  qcmap_reload_mwan3

}

# disable ipv4 state for wlan/eth/bt
function disable_ipv4() {
    util_execute_uci_set "network.wanwlan.proto" "none"
    util_execute_uci_set "network.wanbt.proto" "none"
    util_execute_uci_set "network.waneth.proto" "none"
    util_execute_uci commit network
    ubus call network reload

    log  $(basename "$0") "disable_ipv4" $LINENO "disabled IPV4."
}

# disable ipv6 state for wlan/eth/bt
function disable_ipv6() {
    util_execute_uci_set "network.wanwlan_v6.proto" "none"
    util_execute_uci_set "network.wanbt_v6.proto" "none"
    util_execute_uci_set "network.waneth_v6.proto" "none"
    util_execute_uci commit network
    ubus call network reload

    log  $(basename "$0") "disable_ipv6" $LINENO "disabled IPV6."
}

# enable ipv4 state for wlan/eth/bt
function enable_ipv4() {
    util_execute_uci_set "network.wanwlan.proto" "dhcp"
    util_execute_uci_set "network.wanbt.proto" "dhcp"
    util_execute_uci_set "network.waneth.proto" "dhcp"
    util_execute_uci commit network
    ubus call network reload

    log  $(basename "$0") "enable_ipv4" $LINENO "enabled IPV4."
}

# enable ipv6 state for wlan/eth/bt
function enable_ipv6() {
    util_execute_uci_set "network.wanwlan_v6.proto" "dhcpv6"
    util_execute_uci_set "network.wanbt.proto" "dhcpv6"
    util_execute_uci_set "network.waneth.proto" "dhcpv6"
    util_execute_uci commit network
    ubus call network reload

    log  $(basename "$0") "enable_ipv6" $LINENO "enabled IPV6."
}


case $1 in
  set_bh_pref)
    log  $(basename "$0") "set_bh_pref" $LINENO " set $2 $3 $4 $5 $6"
    set_backhaul_pref $2 $3 $4 $5 $6
    ;;

  switch_bh)
    log  $(basename "$0") "switch_bh" $LINENO " $2 $3"
    switch_backhaul $2 $3
    ;;

  switch_bh_v6)
    log  $(basename "$0") "switch_bh_v6" $LINENO " $2 $3"
    switch_backhaul_v6 $2 $3
    ;;

  update_present_backhaul)
    log  $(basename "$0") "update_present_backhaul" $LINENO ""
    update_present_backhaul $2 $3 $4
    ;;

  read_current_backhaul)
    read_current_backhaul
    log  $(basename "$0") "read_current_backhaul" $LINENO ""
    ;;

  is_wwan_prefer_backhaul)
    is_wwan_prefer_backhaul
    log  $(basename "$0") "is_wwan_prefer_backhaul" $LINENO ""
    ;;

  get_backhaul_file)
    get_backhaul_file $2 $3
    echo $file
    log  $(basename "$0") "get_backhaul_file" $LINENO " $2 $3"
    ;;

  disable_ipv4)
    log  $(basename "$0") "case:disable_ipv4" $LINENO "disable_ipv4"
    disable_ipv4
    ;;

  disable_ipv6)
    log  $(basename "$0") "case:disable_ipv6" $LINENO "disable_ipv6"
    disable_ipv6
    ;;

  enable_ipv4)
    log  $(basename "$0") "case:enable_ipv4" $LINENO "enable_ipv4"
    enable_ipv4
    ;;

  enable_ipv6)
    log  $(basename "$0") "case:enable_ipv6" $LINENO "enable_ipv6"
    enable_ipv6
    ;;

  *)
    log $(basename "$0") "" $LINENO ":Invalid option"
    ;;

esac




