#!/bin/sh
#  Copyright (c) 2023 Qualcomm Technologies, Inc.
#  All Rights Reserved.
#  Confidential and Proprietary - Qualcomm Technologies, Inc.

#Usage: For logging
source /etc/data/mbbUtils.sh


#Function to set ipsec configuration
#$1 - profile_id
#$2 - ike_identifier
#$3 - tunnel_type
#$4 - auth_type
#$5 - remote_ep_addr
#$6 - local_identifier
#$7 - remote_identifier
#$8 - rekey_interval_hrs
#$9 - topology
#$10 - child_identifier
#$11 - hw_offload
#$12 - trap_action
#$13 - child_rekey_interval_hrs
#$14 - port_id
#$15 - port_range
#$16 - end_port_id
#$17 - protocol_type
#$18 - local_addr
#$19 - remote_addr
function handleSetIpsecTunnel() {
  local profile_id="$1"
  local ike_identifier="$2"
  local tunnel_type="$3"
  local auth_type="$4"
  local remote_ep_addr="$5"
  local local_identifier="$6"
  local remote_identifier="$7"
  local rekey_interval_hrs="$8"
  local topology_type="$9"
  local child_identifier="${10}"
  local hw_offload="${11}"
  local trap_action="${12}"
  local child_rekey_interval_hrs="${13}"
  local port_id="${14}"
  local port_range="${15}"
  local end_port_id="${16}"
  local protocol_type="${17}"
  local local_addr="${18}"
  local remote_addr="${19}"

  local local_ip
  local psk_file_path="/etc/swanctl/private/${ike_identifier}_psk"
  local cert_file_path="/etc/swanctl/x509/${ike_identifier}_cert.pem"
  local ca_cert_file_path="/etc/swanctl/x509ca/${ike_identifier}_ca.pem"
  local cert_key_file_path="/etc/swanctl/private/${ike_identifier}_key.pem"
  local psk
  local cert
  local ca_cert
  local cert_key

  local rmnet_name

  #Check if psk file or certificate file exist
  if [ $auth_type -eq 1 ]; then
    if [ ! -f "$psk_file_path" ]; then
      echo "PSK file does not exist"
      return 1
    fi
  fi

  if [ $auth_type -eq 2 ]; then
    if [ ! -f "$cert_file_path" ] || [ ! -f "$ca_cert_file_path" ] || [ ! -f "$cert_key_file_path" ]; then
      echo "Certificate files do not exist"
      return 1
    fi
  fi




  if [ ! -f "/etc/config/ipsec" ]; then
    touch /etc/config/ipsec
  fi

  uci set ipsec.$ike_identifier=remote
  uci set ipsec.$ike_identifier.enabled=0
  uci set ipsec.$ike_identifier.profile_id="$profile_id"
  uci set ipsec.$ike_identifier.tunnel_type="$tunnel_type"
  uci set ipsec.$ike_identifier.topology_type="$topology_type"
  uci set ipsec.$ike_identifier.local_leftip="$local_ip"
  uci set ipsec.$ike_identifier.gateway="$remote_ep_addr"

  if [ $auth_type -eq 1 ]; then
    uci set ipsec.$ike_identifier.authentication_method='psk'
    # Read the value from the file
    read -r psk < "$psk_file_path"
    echo "Value read from file: $psk"
    uci set ipsec.$ike_identifier.pre_shared_key="$psk"
  fi

  if [ $auth_type -eq 2 ]; then
    uci set ipsec.$ike_identifier.authentication_method='pubkey'
    uci set ipsec.$ike_identifier.local_identifier="$local_identifier"
    uci set ipsec.$ike_identifier.remote_identifier="$remote_identifier"
    uci set ipsec.$ike_identifier.local_cert="${ike_identifier}_cert.pem"
    uci set ipsec.$ike_identifier.ca_cert="${ike_identifier}_ca.pem"
    uci set ipsec.$ike_identifier.local_key="${ike_identifier}_key.pem"
  fi

  uci add_list ipsec.$ike_identifier.tunnel="$child_identifier"

  local existing_ikecrypto_list=$(uci get ipsec.$ike_identifier.crypto_proposal)
  if [[ "$existing_ikecrypto_list" != *"ike_proposal"* ]]; then
    # Value does not exist, add it to the list
    uci add_list ipsec.$ike_identifier.crypto_proposal='ike_proposal'
  fi

  uci set ipsec.$child_identifier=tunnel
  uci set ipsec.$child_identifier.enabled=0

  if [ $hw_offload -eq 1 ]; then
    uci set ipsec.$child_identifier.hw_offload='packet'
  else
    uci set ipsec.$child_identifier.hw_offload='no'
  fi

  if [ $trap_action -eq 1 ]; then
    uci set ipsec.$child_identifier.trap_action='trap'
  else
    uci set ipsec.$child_identifier.trap_action='start'
  fi


  uci set ipsec.$ike_identifier.rekeytime="${rekey_interval_hrs}h"
  uci set ipsec.$child_identifier.rekeytime="${child_rekey_interval_hrs}h"

  local existing_espcrypto_list=$(uci get ipsec.$child_identifier.crypto_proposal)
  if [[ "$existing_espcrypto_list" != *"esp_proposal"* ]]; then
    # Value does not exist, add it to the list
    uci add_list ipsec.$child_identifier.crypto_proposal='esp_proposal'
  fi


  #need more info for topology 2&3: local_addr and remote_addr
  if [ $topology_type -eq 2 ] || [ $topology_type -eq 3 ]; then
    if [ $port_id -eq 0 ]; then
        uci set ipsec.$child_identifier.local_subnet="$local_addr"
    else
        if [ $protocol_type -eq 1 ]; then
            #tcp
            if [ $port_range -eq 1 ]; then
                uci set ipsec.$child_identifier.local_subnet="$local_addr[tcp/$port_id-$end_port_id]"
            else
                uci set ipsec.$child_identifier.local_subnet="$local_addr[tcp/$port_id]"
            fi
        else
            #udp
            if [ "$port_range" -eq 1 ]; then
                uci set ipsec.$child_identifier.local_subnet="$local_addr[udp/$port_id-$end_port_id]"
            else
                uci set ipsec.$child_identifier.local_subnet="$local_addr[udp/$port_id]"
            fi
        fi
    fi


    uci set ipsec.$child_identifier.remote_subnet="$remote_addr"

  fi



  #adding ike crypto list

  uci set ipsec.ike_proposal=crypto_proposal
  uci set ipsec.ike_proposal.encryption_algorithm='aes128'
  uci set ipsec.ike_proposal.hash_algorithm='sha256'
  uci set ipsec.ike_proposal.dh_group='x25519'

  #adding esp list
  uci set ipsec.esp_proposal=crypto_proposal
  uci set ipsec.esp_proposal.encryption_algorithm='aes128gcm128'
  uci set ipsec.esp_proposal.dh_group='x25519-noesn-esn'


  echo "uci commit ipsec"
  uci commit ipsec


}

#Usage :util_update_iptable_rules
#$1 - operation(1:add, 2: delete)
#$2 - rmnet_name
#$3 - data_ip_type
#$4 - tunnel_ip_type
#$5 - local_addr
#$6 - remote_addr
#$7 - port_id
#$8 - end_port_id
function util_update_iptable_rules(){
  local operation="$1"
  local rmnet_name="$2"
  local data_ip_type="$3"
  local tunnel_ip_type="$4"
  local local_addr="$5"
  local remote_addr="$6"
  local port_id="$7"
  local end_port_id="$8"

  local iptable_type
  local mss_size

  #Get rmnet MTU size
  local mtu=$(ifconfig "$rmnet_name" | grep -o 'MTU:[0-9]*' | awk -F: '{print $NF}')
  echo "mtu: $mtu"

  #IPv4 traffic go through IPv4 tunnel(Inner IP header and outer IP header are both 20 bytes)
  #v4_v4_ipsec_overhead = 20(Inner ip header) + 20(tcp header) + 20(Outer IP Header) + 4(Sequence Number) + 4(SPI) + 16(Initialisation Vector AES)
  #+ (0~15)(ESP Padding) + 1(Padding Length) + 1(Next Header) + 32(Authentication Data SHA512) = 133
  local v4_v4_ipsec_overhead=133

  #IPv6 traffic go through IPv6 tunnel(Inner IP header and outer IP header are both 40 bytes)
  #v6_v6_ipsec_overhead = 40(Inner ip header) + 20(tcp header) + 40(Outer IP Header) + 4(Sequence Number) + 4(SPI) + 16(Initialisation Vector AES)
  #+ (0~15)(ESP Padding) + 1(Padding Length) + 1(Next Header) + 32(Authentication Data SHA512) = 173
  local v6_v6_ipsec_overhead=173

  #IPv6 traffic go through IPv4 tunnel(Inner IP header is 20 bytes and outer IP header is 40 bytes)
  #v6_v6_ipsec_overhead = 40(Inner ip header) + 20(tcp header) + 20(Outer IP Header) + 4(Sequence Number) + 4(SPI) + 16(Initialisation Vector AES)
  #+ (0~15)(ESP Padding) + 1(Padding Length) + 1(Next Header) + 32(Authentication Data SHA512) = 153
  local v4_v6_ipsec_overhead=153

  #Calculate TCP MSS size
  if [ "$data_ip_type" == "IPv4" ] && [ "$tunnel_ip_type" == "IPv4" ]; then
    iptables_type='iptables'
    mss_size="$(($mtu-$v4_v4_ipsec_overhead))"
    echo "mss_size: $mss_size"

  elif [ "$data_ip_type" == "IPv6" ] && [ "$tunnel_ip_type" == "IPv6" ]; then
    iptables_type='ip6tables'
    mss_size="$(($mtu-$v6_v6_ipsec_overhead))"
    echo "mss_size: $mss_size"

  elif [ "$data_ip_type" == "IPv4" ] && [ "$tunnel_ip_type" == "IPv6" ]; then
    iptables_type='iptables'
    mss_size="$(($mtu-$v4_v6_ipsec_overhead))"
    echo "mss_size: $mss_size"

  elif [ "$data_ip_type" == "IPv6" ] && [ "$tunnel_ip_type" == "IPv4" ]; then
    iptables_type='ip6tables'
    mss_size="$(($mtu-$v4_v6_ipsec_overhead))"
    echo "mss_size: $mss_size"

  fi

  #IPsec traffic matching rule for IPv4
  if [ $operation -eq 1 ]; then
      #To allow IPsec traffic for specific interface
      iptables -t nat -D POSTROUTING -o $rmnet_name -m policy --dir out --pol ipsec -j ACCEPT
      iptables -t nat -I POSTROUTING 1 -o $rmnet_name -m policy --dir out --pol ipsec -j ACCEPT
  fi

  #TCP MSS Clamping rules
  if [ -n "$port_id" ] && [ -n "$end_port_id" ]; then
    $iptables_type -t mangle -D POSTROUTING -s $local_addr -d $remote_addr -p tcp --dport $port_id:$end_port_id --tcp-flags SYN,RST SYN -o $rmnet_name -j TCPMSS --set-mss $mss_size

    if [ $operation -eq 1 ]; then
      $iptables_type -t mangle -I POSTROUTING 1 -s $local_addr -d $remote_addr -p tcp --dport $port_id:$end_port_id --tcp-flags SYN,RST SYN -o $rmnet_name -j TCPMSS --set-mss $mss_size
    fi
  elif [ -n "$port_id" ]; then
    $iptables_type -t mangle -D POSTROUTING -s $local_addr -d $remote_addr -p tcp --dport $port_id --tcp-flags SYN,RST SYN -o $rmnet_name -j TCPMSS --set-mss $mss_size

    if [ $operation -eq 1 ]; then
      $iptables_type -t mangle -I POSTROUTING 1 -s $local_addr -d $remote_addr -p tcp --dport $port_id --tcp-flags SYN,RST SYN -o $rmnet_name -j TCPMSS --set-mss $mss_size
    fi
  else
    $iptables_type -t mangle -D POSTROUTING -s $local_addr -d $remote_addr -p tcp --tcp-flags SYN,RST SYN -o $rmnet_name -j TCPMSS --set-mss $mss_size

    if [ $operation -eq 1 ]; then
      $iptables_type -t mangle -I POSTROUTING 1 -s $local_addr -d $remote_addr -p tcp --tcp-flags SYN,RST SYN -o $rmnet_name -j TCPMSS --set-mss $mss_size
    fi
  fi

}

#Usage :util_update_ip_type
#$1 - ip
function util_ip_type(){
  local ip="$1"
  local first
  local second


  first=$(echo "$ip" | awk -F'/' '{print $1}')
  second=$(echo "$ip" | awk -F'/' '{print $2}')

  #echo "ip: $ip, first: $first, second: $second"


  if echo "$first" | grep -Eq '^(\d{1,3}\.){3}\d{1,3}$'; then
    echo "IPv4"
    return 1
  elif echo "$first" | grep -Eq '^([0-9a-fA-F]{1,4}::?){1,7}[0-9a-fA-F]{1,4}$'; then
    echo "IPv6"
    return 1
  else
    echo "Invalid"
    return 1
  fi
}

#Function to enable ipsec tunnel
#$1 - ike_identifier
#$2 - child_identifier
function util_ipsec_enable_tunnel(){
  local local_ip
  local rmnet_name
  local profile_id=$(uci get ipsec.$ike_identifier.profile_id)
  local tunnel_type=$(uci get ipsec.$ike_identifier.tunnel_type)
  local ike_identifier="$1"
  local child_identifier="$2"
  local rule
  local local_subnet=$(uci get ipsec.$child_identifier.local_subnet)
  local remote_subnet=$(uci get ipsec.$child_identifier.remote_subnet)
  local remote_ep_addr=$(uci get ipsec.$ike_identifier.gateway)

  local port_id
  local end_port_id
  local local_addr
  local data_ip_type
  local tunnel_ip_type

  if [ ! -f "/tmp/ipsec/ipsec_restart" ]; then
    ipsec restart
    sleep 1
    touch /tmp/ipsec/ipsec_restart
  fi

  #source local ip
  if [ $tunnel_type -eq 1 ] && [ -f /tmp/ipv4config$profile_id ]; then
    . /tmp/ipv4config$profile_id
    local_ip=$PUBLIC_IP
    rmnet_name=$IFNAME
  fi

  if [ $tunnel_type -eq 2 ] && [ -f /tmp/ipv6config$profile_id ]; then
    . /tmp/ipv6config$profile_id
    local_ip=$PUBLIC_IP6
    rmnet_name=$IFNAME
  fi

  if [ -n "$local_ip" ] && [ -n "$rmnet_name" ]; then

    if [ -n "$local_subnet" ]; then
      #For S2S (192.168.224.0/23[tcp/5012-6012]) or '192.168.224.0/23[tcp/5014]'
      data_ip_type=$(util_ip_type $remote_subnet)
      tunnel_ip_type=$(util_ip_type $local_ip)
      local_addr=$(echo "$local_subnet" | awk -F'[' '{print $1}')
      port_id1=$(echo "$local_subnet" | awk -F'[-/]' '{print $3}')
      port_id=$(echo "$port_id1" | sed 's/]//')
      end_port_id1=$(echo "$local_subnet" | awk -F'[-]' '{print $2}')
      end_port_id=$(echo "$end_port_id1" | sed 's/]//')
      echo "rmnet_name:$rmnet_name, data_ip_type: $data_ip_type, tunnel_ip_type: $tunnel_ip_type, local_addr: $local_addr, remote_subnet: $remote_subnet, port_id: $port_id, end_port_id: $end_port_id"
      util_update_iptable_rules 1 $rmnet_name $data_ip_type $tunnel_ip_type $local_addr $remote_subnet $port_id $end_port_id
    else
      #For H2H
      data_ip_type=$(util_ip_type $remote_ep_addr)
      tunnel_ip_type=$(util_ip_type $local_ip)
      echo "rmnet_name:$rmnet_name,  data_ip_type: $data_ip_type, tunnel_ip_type: $tunnel_ip_type, local_addr: $local_ip, remote_ep_addr: $remote_ep_addr, port_id: $port_id, end_port_id: $end_port_id"
      util_update_iptable_rules 1 $rmnet_name $data_ip_type $tunnel_ip_type $local_ip $remote_ep_addr $port_id $end_port_id
    fi

    #Update iptable rules
    ip_type=$(util_ip_type $remote_ep_addr)


    # set local ip
    uci set ipsec.$ike_identifier.local_leftip="$local_ip"
    uci commit ipsec

    echo "/etc/init.d/swanctl restart"
    /etc/init.d/swanctl restart
    echo "swanctl --load-all"
    swanctl --load-all
    echo "Initiating"
    swanctl --initiate --child $child_identifier&
    echo "ipsec status"
    ipsec status

  else
    echo "Waiting for data call"
  fi



}

#Function to activate ipsec tunnel from QCMAP
#$1 - ike_identifier
#$2 - child_identifier
function handleActivateIpsecTunnel(){
  local ike_identifier="$1"
  local child_identifier="$2"
  local current_active_tunnels=0
  local curr_enable_option

  uci set ipsec.$ike_identifier.enabled=1

  curr_enable_option=$(uci get ipsec.$child_identifier.enabled)

  # Check if the child is enabled or not
  if [ "$curr_enable_option" == "0" ]; then
    uci set ipsec.$child_identifier.enabled=1
    #increment active tunnels count
    current_active_tunnels=$(uci get qcmap_lan.@global[0].ipsec_active_tunnels)
    #Convert string to number
    number=$(echo "$current_active_tunnels" | awk '{print $0+0}')
    uci set qcmap_lan.@global[0].ipsec_active_tunnels=$((number+1))
  fi


  #call helper function to enable ipsec
  util_ipsec_enable_tunnel $ike_identifier $child_identifier

}


#Function to enable ipsec tunnel when wwan data call get connected
#$1 - profile_id
#$2 - ip_type(1:IPv4, 2:IPv6)
function handleDataCallConnectedEv() {
  local profile_id="$1"
  local ip_type="$2"
  local ike_identifier
  local child_identifier
  local section
  local option
  local entry
  local value
  local line
  local ike_enable_flag
  local child_enable_flag
  local children
  local child_id
  local tunnel_type

  uci show ipsec | while read -r line; do
    #echo "$line"

    if [[ "$line" == *"profile_id"* ]]; then
      section=$(echo "$line" | awk -F"['.=]" '{print $1}')
      option=$(echo "$line" | awk -F"['.=]" '{print $2}')
      entry=$(echo "$line" | awk -F"['.=]" '{print $3}')
      value=$(echo "$line" | awk -F"['.=]" '{print $5}')
      #echo "Section: $section, Option: $option, Entry: $entry, Value: $value"

      #Check if the datacall is on the right profile
      if [ "$entry" == "profile_id" ] && [ "$value" == "$profile_id" ]; then
        echo "$line"
        echo "Section: $section, Option: $option, Entry: $entry, Value: $value"
        ike_identifier="$option"
        #echo "ike: $ike_identifier"

        tunnel_type=$(uci get ipsec.$ike_identifier.tunnel_type)

        ike_enable_flag=$(uci get ipsec.$ike_identifier.enabled)
        if [ "$ike_enable_flag" == 1 ] && [ "$tunnel_type" == "$ip_type" ]; then
          #Get all the children of ike
          children=$(uci get ipsec.$ike_identifier.tunnel)
          echo "$children" | awk -F" " '{for(i=1; i<=NF; i+=1) print $i}' | while IFS= read -r child_id; do
            #echo "child_id: $child_id"
            child_enable_flag=$(uci get ipsec.$child_id.enabled)
            if [ "$child_enable_flag" == 1 ]; then
              util_ipsec_enable_tunnel $ike_identifier $child_id
            fi
          done
        fi
      fi
    fi
  done

}





#Function to delete ipsec tunnel from QCMAP
#$1 - ike_identifier
#$2 - child_identifier(optional)
function handleDeleteIpsecTunnel() {
  local ike_identifier="$1"
  local child_identifier="$2"
  local children
  local ike
  local current_active_tunnels=0

  local remote_ep_addr=$(uci get ipsec.$ike_identifier.gateway)
  local local_ip=$(uci get ipsec.$ike_identifier.local_leftip)
  local profile_id=$(uci get ipsec.$ike_identifier.profile_id)
  local tunnel_type=$(uci get ipsec.$ike_identifier.tunnel_type)
  local rmnet_name
  local local_addr
  local port_id
  local end_port_id
  local data_ip_type
  local tunnel_ip_type

  #source local ip
  if [ $tunnel_type -eq 1 ] && [ -f /tmp/ipv4config$profile_id ]; then
    . /tmp/ipv4config$profile_id
    rmnet_name=$IFNAME
  fi

  if [ $tunnel_type -eq 2 ] && [ -f /tmp/ipv6config$profile_id ]; then
    . /tmp/ipv6config$profile_id
    rmnet_name=$IFNAME
  fi

  if [ ! -f "/tmp/ipsec/ipsec_restart" ]; then
    ipsec restart
    sleep 1
    touch /tmp/ipsec/ipsec_restart
  fi

  if [ -z "$child_identifier" ]; then
    #delete ike tunnel
    swanctl --terminate --ike $ike_identifier&
    #unload routed connection
    ipsec unroute $ike_identifier

    children=$(uci get ipsec.$ike_identifier.tunnel)
    echo "$children" | awk -F" " '{for(i=1; i<=NF; i+=1) print $i}' | while IFS= read -r child_id; do

      #Delete iptable rules based on child_id
      local local_subnet=$(uci get ipsec.$child_id.local_subnet)
      local remote_subnet=$(uci get ipsec.$child_id.remote_subnet)

      #Delete iptable rules
      if [ -n "$local_subnet" ]; then
        #For S2S
        data_ip_type=$(util_ip_type $remote_subnet)
        tunnel_ip_type=$(util_ip_type $local_ip)
        local_addr=$(echo "$local_subnet" | awk -F'[' '{print $1}')
        port_id1=$(echo "$local_subnet" | awk -F'[-/]' '{print $3}')
        port_id=$(echo "$port_id1" | sed 's/]//')
        end_port_id1=$(echo "$local_subnet" | awk -F'[-]' '{print $2}')
        end_port_id=$(echo "$end_port_id1" | sed 's/]//')
        echo "rmnet_name:$rmnet_name, data_ip_type: $data_ip_type, tunnel_ip_type: $tunnel_ip_type, local_addr: $local_addr, remote_subnet: $remote_subnet, port_id: $port_id, end_port_id: $end_port_id"
        util_update_iptable_rules 2 $rmnet_name $data_ip_type $tunnel_ip_type $local_addr $remote_subnet $port_id $end_port_id
      else
        #For H2H
        data_ip_type=$(util_ip_type $remote_ep_addr)
        tunnel_ip_type=$(util_ip_type $local_ip)
        echo "rmnet_name:$rmnet_name, data_ip_type: $data_ip_type, tunnel_ip_type: $tunnel_ip_type, local_addr: $local_ip, remote_ep_addr: $remote_ep_addr, port_id: $port_id, end_port_id: $end_port_id"
        util_update_iptable_rules 2 $rmnet_name $data_ip_type $tunnel_ip_type $local_ip $remote_ep_addr $port_id $end_port_id
      fi

      ipsec unroute $child_id
      uci delete ipsec.$child_id
      current_active_tunnels=$(uci get qcmap_lan.@global[0].ipsec_active_tunnels)
      number=$(echo "$current_active_tunnels" | awk '{print $0+0}')
      uci set qcmap_lan.@global[0].ipsec_active_tunnels=$((number-1))
    done
    uci delete ipsec.$ike_identifier

    # Check if tunnels<0, ike has two children, one is active one is not.
    current_active_tunnels=$(uci get qcmap_lan.@global[0].ipsec_active_tunnels)
    number=$(echo "$current_active_tunnels" | awk '{print $0+0}')
    if [ "$number" -lt 0 ]; then
      uci set qcmap_lan.@global[0].ipsec_active_tunnels='0'
    fi

  else
    #delete child tunnel
    local local_subnet=$(uci get ipsec.$child_identifier.local_subnet)
    local remote_subnet=$(uci get ipsec.$child_identifier.remote_subnet)

    #Delete iptable rules
    #For S2S
    data_ip_type=$(util_ip_type $remote_subnet)
    tunnel_ip_type=$(util_ip_type $local_ip)
    local_addr=$(echo "$local_subnet" | awk -F'[' '{print $1}')
    port_id1=$(echo "$local_subnet" | awk -F'[-/]' '{print $3}')
    port_id=$(echo "$port_id1" | sed 's/]//')
    end_port_id1=$(echo "$local_subnet" | awk -F'[-]' '{print $2}')
    end_port_id=$(echo "$end_port_id1" | sed 's/]//')
    echo "rmnet_name:$rmnet_name, data_ip_type: $data_ip_type, tunnel_ip_type: $tunnel_ip_type, local_addr: $local_addr, remote_subnet: $remote_subnet, port_id: $port_id, end_port_id: $end_port_id"
    util_update_iptable_rules 2 $rmnet_name $data_ip_type $tunnel_ip_type $local_addr $remote_subnet $port_id $end_port_id

    swanctl --terminate --child $child_identifier
    #unload routed connection
    ipsec unroute $child_identifier

    #clean up in UCI DB
    uci delete ipsec.$child_identifier

    #need to del_list
    uci show ipsec | while read -r line; do
      if echo "$line" | grep -q $child_identifier; then
        #echo "$line"
        ike=$(echo "$line" | awk -F"['.=]" '{print $2}')
        uci del_list ipsec.$ike.tunnel=$child_identifier
      fi
    done

    current_active_tunnels=$(uci get qcmap_lan.@global[0].ipsec_active_tunnels)
    number=$(echo "$current_active_tunnels" | awk '{print $0+0}')
    uci set qcmap_lan.@global[0].ipsec_active_tunnels=$((number-1))

    # Check if this is the last child the ike session has, delete the ike session as well
    child_left=$(uci get ipsec.$ike_identifier.tunnel)
    if [ -z "$child_left" ]; then
      swanctl --terminate --ike $ike_identifier&
      ipsec unroute $ike_identifier
      uci delete ipsec.$ike_identifier
    fi


  fi

  uci commit
  /etc/init.d/swanctl restart
  swanctl --load-all

}

#Function to disable ipsec tunnel when wwan datacall disconnected
#$1 - profile_id
#$2 - ip_type
function handledataCallDisconnectedEv() {
  local profile_id="$1"
  local ip_type="$2"
  local local_ip
  local ike_identifier
  local child_identifier
  local section
  local option
  local entry
  local value
  local line
  local local_ip
  local rmnet_name
  local remote_ep_addr
  local children
  local local_subnet
  local remote_subnet
  local local_addr

  uci show ipsec | while read -r line; do
    #echo "$line"

    if [[ "$line" == *"profile_id"* ]]; then
      section=$(echo "$line" | awk -F"['.=]" '{print $1}')
      option=$(echo "$line" | awk -F"['.=]" '{print $2}')
      entry=$(echo "$line" | awk -F"['.=]" '{print $3}')
      value=$(echo "$line" | awk -F"['.=]" '{print $5}')

      #Check if the datacall is on the right profile
      if [ "$entry" == "profile_id" ] && [ "$value" == "$profile_id" ]; then
        ike_identifier="$option"

        remote_ep_addr=$(uci get ipsec.$ike_identifier.gateway)
        local_ip=$(uci get ipsec.$ike_identifier.local_leftip)
        tunnel_type=$(uci get ipsec.$ike_identifier.tunnel_type)

        #source rmnet_name
        if [ $tunnel_type -eq 1 ] && [ -f /tmp/ipv4config$profile_id ]; then
          . /tmp/ipv4config$profile_id
          rmnet_name=$IFNAME
        fi

        if [ $tunnel_type -eq 2 ] && [ -f /tmp/ipv6config$profile_id ]; then
          . /tmp/ipv6config$profile_id
          rmnet_name=$IFNAME
        fi

        #If datacall disconnected on the right tunnel type
        if [ "$tunnel_type" == "$ip_type" ]; then

          #Update iptables rules
          children=$(uci get ipsec.$ike_identifier.tunnel)
          echo "$children" | awk -F" " '{for(i=1; i<=NF; i+=1) print $i}' | while IFS= read -r child_id; do
            local local_subnet=$(uci get ipsec.$child_id.local_subnet)
            local remote_subnet=$(uci get ipsec.$child_id.remote_subnet)

            #Delete iptable rules
            if [ -n local_subnet ]; then
              #For S2S
              data_ip_type=$(util_ip_type $remote_subnet)
              tunnel_ip_type=$(util_ip_type $local_ip)
              local_addr=$(echo "$local_subnet" | awk -F'[' '{print $1}')
              port_id1=$(echo "$local_subnet" | awk -F'[-/]' '{print $3}')
              port_id=$(echo "$port_id1" | sed 's/]//')
              end_port_id1=$(echo "$local_subnet" | awk -F'[-]' '{print $2}')
              end_port_id=$(echo "$end_port_id1" | sed 's/]//')
              echo "rmnet_name:$rmnet_name, data_ip_type: $data_ip_type, tunnel_ip_type: $tunnel_ip_type, local_addr: $local_addr, remote_subnet: $remote_subnet, port_id: $port_id, end_port_id: $end_port_id"
              util_update_iptable_rules 2 $rmnet_name $data_ip_type $tunnel_ip_type $local_addr $remote_subnet $port_id $end_port_id
            else
              #For H2H
              data_ip_type=$(util_ip_type $remote_ep_addr)
              tunnel_ip_type=$(util_ip_type $local_ip)
              echo "rmnet_name:$rmnet_name, data_ip_type: $data_ip_type, tunnel_ip_type: $tunnel_ip_type, local_addr: $local_ip, remote_ep_addr: $remote_ep_addr, port_id: $port_id, end_port_id: $end_port_id"
              util_update_iptable_rules 2 $rmnet_name $data_ip_type $tunnel_ip_type $local_ip $remote_ep_addr $port_id $end_port_id
            fi
          done

          uci delete ipsec.$ike_identifier.local_leftip
          echo "uci commit ipsec"
          uci commit
          /etc/init.d/swanctl restart
          swanctl --load-all
        fi
      fi
    fi
  done
}

#Function to get ipsec tunnel configuration
#$1 - ike_identifier
#$2 - child_identifier
function handleGetIpsecTunnelStatus(){
  local ike_identifier="$1"
  local child_identifier="$2"

  output=$(swanctl --list-sas)
  # Process the output to check for active tunnels
  if echo "$output" | grep -q "$child_identifier:.*INSTALLED"; then
    echo $(basename "$0")":handleGetIpsecTunnelStatus:"$LINENO":child is active"
    return 1
  else
    if echo "$output" | grep -q "$ike_identifier:.*CONNECTING"; then
      echo $(basename "$0")":handleGetIpsecTunnelStatus:"$LINENO":child is connecting"
      return 1
    else
      echo $(basename "$0")":handleGetIpsecTunnelStatus:"$LINENO":child is not active"
      return 0
    fi
  fi

}


#Function to check duplicate remote endpoint address
#$1 remote_ep_addr
function util_ipsec_check_dup_remote_ep_addr(){
  local remote_ep_addr="$1"

  uci show ipsec | while read -r line; do
    if [[ "$line" == *"gateway"* ]]; then
      #ipsec.gwgw.gateway='172.21.150.25'
      section=$(echo "$line" | awk -F"['.=]" '{print $1}')
      option=$(echo "$line" | awk -F"['.=]" '{print $2}')
      entry=$(echo "$line" | awk -F"['.=]" '{print $3}')
      value=$(echo "$line" | awk -F"['=]" '{print $3}')
      #echo "Section: $section, Option: $option, Entry: $entry, Value: $value"

      if [ "$value" == "$remote_ep_addr" ]; then
          echo $(basename "$0")":util_ipsec_check_dup_remote_ep_addr:"$LINENO":duplicate remote_ep_addr"
          return
      fi
    fi
  done

  return 0
}


#Function to check duplicate local subnets
#$1 local_subnet
function util_ipsec_check_dup_local_subnet(){
  local local_subnet="$1"

  uci show ipsec | while read -r line; do
    if [[ "$line" == *"local_subnet"* ]]; then
      #ipsec.gwgw.tunnel='rr'
      section=$(echo "$line" | awk -F"['.=]" '{print $1}')
      option=$(echo "$line" | awk -F"['.=]" '{print $2}')
      entry=$(echo "$line" | awk -F"['.=]" '{print $3}')
      value=$(echo "$line" | awk -F"['=]" '{print $3}')
      #echo "Section: $section, Option: $option, Entry: $entry, Value: $value"

      if [ "$value" == "$local_subnet" ]; then
          echo $(basename "$0")":util_ipsec_check_dup_local_subnet:"$LINENO":duplicate local_subnet"
          return
      fi
    fi
  done
}

#Function to check duplicate remote subnets
#$1 remote_subnet
function util_ipsec_check_dup_remote_subnet(){
  local remote_subnet="$1"

  uci show ipsec | while read -r line; do
    if [[ "$line" == *"remote_subnet"* ]]; then
      #ipsec.gwgw.tunnel='rr'
      section=$(echo "$line" | awk -F"['.=]" '{print $1}')
      option=$(echo "$line" | awk -F"['.=]" '{print $2}')
      entry=$(echo "$line" | awk -F"['.=]" '{print $3}')
      value=$(echo "$line" | awk -F"['=]" '{print $3}')
      #echo "Section: $section, Option: $option, Entry: $entry, Value: $value"

      if [ "$value" == "$remote_subnet" ]; then
          echo $(basename "$0")":util_ipsec_check_dup_remote_subnet:"$LINENO":duplicate remote_subnet"
          return
      fi
    fi
  done
}


#Function to check duplicate child identifier
#$1 child_id
function util_ipsec_check_dup_child_id(){
  local child_id="$1"
  uci show ipsec | while read -r line; do
    if [[ "$line" == *"tunnel"* ]]; then
      #ipsec.gwgw.tunnel='rr'
      section=$(echo "$line" | awk -F"['.=]" '{print $1}')
      option=$(echo "$line" | awk -F"['.=]" '{print $2}')
      entry=$(echo "$line" | awk -F"['.=]" '{print $3}')
      value=$(echo "$line" | awk -F"['.=]" '{print $5}')
      #echo "Section: $section, Option: $option, Entry: $entry, Value: $value"

      if [ "$value" == "$child_id" ]; then
          echo $(basename "$0")":util_ipsec_check_dup_child_id:"$LINENO":duplicate child_identifier"
          return
      fi
    fi
  done
}


case $1 in
 setIpsecTunnel)
   handleSetIpsecTunnel $2 $3 $4 $5 $6 $7 $8 $9 ${10} ${11} ${12} ${13} ${14} ${15} ${16} ${17} ${18} ${19} ${20}
   ;;
 activateIpsecTunnel)
   handleActivateIpsecTunnel $2 $3
   ;;
 dataCallConnectedEv)
   handleDataCallConnectedEv $2 $3
   ;;
 deleteIpsecTunnel)
   handleDeleteIpsecTunnel $2 $3
   ;;
 dataCallDisconnectedEv)
   handledataCallDisconnectedEv $2 $3
   ;;
 getIpsecTunnelStatus)
   handleGetIpsecTunnelStatus $2 $3
   ;;
 checkdupremoteepaddr)
   util_ipsec_check_dup_remote_ep_addr $2
   ;;
 checkduplocalsubnet)
   util_ipsec_check_dup_local_subnet $2
   ;;
 checkdupremotesubnet)
   util_ipsec_check_dup_remote_subnet $2
   ;;
 checkdupchildid)
   util_ipsec_check_dup_child_id $2
   ;;
 *)
   echo "Invalid option"
   ;;
esac